/*********************************************************************
*
* Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_diffserv_class_conf.h
*
* @purpose   DiffServ Class Configuration and Status 
*
* @component OpEN
*
* @create    17/05/2023
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_DIFFSERV_CLASS_CONF_H_INCLUDED
#define OPENAPI_DIFFSERV_CLASS_CONF_H_INCLUDED

#include "openapi_common.h"
#include "openapi_routing.h"

/** Values for DiffServ Class type */
typedef enum                     
{   
  OPEN_DIFFSERV_CLASS_TYPE_ALL = 1,              /**< Class Type : Match Any */
  OPEN_DIFFSERV_CLASS_TYPE_ANY                   /**< Class Type : Match All */

} OPEN_DIFFSERV_CLASS_TYPE_t;

/** Values for DiffServ Class Layer3 Protocol type */
typedef enum
{
  OPEN_DIFFSERV_CLASS_L3PROTOCOL_IPV4 = 1,       /**< Class Protocol type : IPV4 */
  OPEN_DIFFSERV_CLASS_L3PROTOCOL_IPV6            /**< Class Protocol type : IPV6 */

} OPEN_DIFFSERV_CLASS_L3PROTOCOL_t;

/** Values for DiffServ Class Rule types supported as match conditions */
typedef enum
{
  OPEN_DIFFSERV_CLASS_RULE_TYPE_COS = 1,           /**< COS : val1 - (0..7) */
  OPEN_DIFFSERV_CLASS_RULE_TYPE_COS2,              /**< Secondary-cos : val1 - (0..7) */
  OPEN_DIFFSERV_CLASS_RULE_TYPE_DSTIP,             /**< Destination IP : val1 - (ip address), val2 - (ip mask) */ 
  OPEN_DIFFSERV_CLASS_RULE_TYPE_DSTIPV6,           /**< Destination IPV6 : val1 - (IPV6 address), val2 - (1..128) prefix length */ 
  OPEN_DIFFSERV_CLASS_RULE_TYPE_DSTL4PORT,         /**< Destination port : val1 - (0-65535) Refer OPEN_DIFFSERV_LAYER_4_PORT_NUM macros
                                                        for user keyword translations */
  OPEN_DIFFSERV_CLASS_RULE_TYPE_DSTMAC,            /**< Destination MAC : val1 - (MAC address), val2 - (MAC address bit mask) */
  OPEN_DIFFSERV_CLASS_RULE_TYPE_ETYPE,             /**< Ether type : val1 - (OPEN_QOS_ETYPE_KEYID_t), 
                                                        val2 - (0x0600..0xffff) valid for OPEN_QOS_ETYPE_KEYID_CUSTOM type */
  OPEN_DIFFSERV_CLASS_RULE_TYPE_EVERY,             /**< Every : Flag to indicate to match on any packet. No input values.  */
  OPEN_DIFFSERV_CLASS_RULE_TYPE_FLOWLBLV6,         /**< Flow Label V6 : val1 - (0..1048575) */
  OPEN_DIFFSERV_CLASS_RULE_TYPE_IPDSCP,            /**< IP DSCP : val1 - (0..63) Refer OPEN_DIFFSERV_IP_DSCP macros 
                                                        for user keyword translations */    
  OPEN_DIFFSERV_CLASS_RULE_TYPE_IPPRECEDENCE,      /**< IP Precedence : val1 - (0..7) */
  OPEN_DIFFSERV_CLASS_RULE_TYPE_IPTOS,             /**< IP TOS : val1 - (00..ff) TOS bits value, val2 - (00..ff) TOS mask */  
  OPEN_DIFFSERV_CLASS_RULE_TYPE_PROTOCOL,          /**< PROTOCOL NUM : val1 - (0..255) 
                                                        Refer OPEN_DIFFSERV_LAYER_4_PROTOCOL & OPEN_DIFFSERV_LAYER_6_PROTOCOL macros
                                                        for user keyword translations */
  OPEN_DIFFSERV_CLASS_RULE_TYPE_REFCLASS,          /**< Reference Class : Val1 - (className) */
  OPEN_DIFFSERV_CLASS_RULE_TYPE_SRCIP,             /**< Source IP : val1 - (ip address), val2 - (ip mask) */
  OPEN_DIFFSERV_CLASS_RULE_TYPE_SRCIPV6,           /**< Source IPV6 : val1 - (IPV6 address), val2 - (1..128) prefix length */
  OPEN_DIFFSERV_CLASS_RULE_TYPE_SRCL4PORT,         /**< Destination port : val1 - (0-65535) Refer OPEN_DIFFSERV_LAYER_4_PORT_NUM macros
                                                        for user keyword translations */
  OPEN_DIFFSERV_CLASS_RULE_TYPE_SRCMAC,            /**< Source MAC : val1 - (MAC address), val2 - (MAC address bit mask) */
  OPEN_DIFFSERV_CLASS_RULE_TYPE_VLANID,            /**< VLAN ID : val1 - (1..4093) */
  OPEN_DIFFSERV_CLASS_RULE_TYPE_VLANID2            /**< Secondary VLAN ID : val1 - (1..4093) */

} OPEN_DIFFSERV_CLASS_RULE_TYPE_t;

/** QOS Ethertype keyword list */                  
typedef enum
{ 
  OPEN_QOS_ETYPE_KEYID_CUSTOM = 1,                          /**< user value     */
  OPEN_QOS_ETYPE_KEYID_APPLETALK,                           /**< Appletalk      */
  OPEN_QOS_ETYPE_KEYID_ARP,                                 /**< ARP            */
  OPEN_QOS_ETYPE_KEYID_IBMSNA,                              /**< IBM SNA        */
  OPEN_QOS_ETYPE_KEYID_IPV4,                                /**< IPv4           */
  OPEN_QOS_ETYPE_KEYID_IPV6,                                /**< IPv6           */
  OPEN_QOS_ETYPE_KEYID_IPX,                                 /**< IPX            */
  OPEN_QOS_ETYPE_KEYID_MPLSMCAST,                           /**< MPLS multicast */
  OPEN_QOS_ETYPE_KEYID_MPLSUCAST,                           /**< MPLS unicast   */
  OPEN_QOS_ETYPE_KEYID_NETBIOS,                             /**< NetBIOS        */
  OPEN_QOS_ETYPE_KEYID_NOVELL,                              /**< Novell         */
  OPEN_QOS_ETYPE_KEYID_PPPOE,                               /**< PPPoE          */
  OPEN_QOS_ETYPE_KEYID_RARP                                 /**< Reverse ARP    */
  
} OPEN_QOS_ETYPE_KEYID_t;

#define OPEN_DIFFSERV_CLASS_NAME_MAX    31
#define OPEN_RULE_VAL_MAX_STRING_LENGTH    129

/* User Keyword Translations for Layer 4 Port Numbers */
#define OPEN_DIFFSERV_LAYER_4_PORT_NUM_DOMAIN     53
#define OPEN_DIFFSERV_LAYER_4_PORT_NUM_ECHO       7
#define OPEN_DIFFSERV_LAYER_4_PORT_NUM_FTP        21
#define OPEN_DIFFSERV_LAYER_4_PORT_NUM_FTPDATA    20
#define OPEN_DIFFSERV_LAYER_4_PORT_NUM_HTTP       80
#define OPEN_DIFFSERV_LAYER_4_PORT_NUM_SMTP       25
#define OPEN_DIFFSERV_LAYER_4_PORT_NUM_SNMP       161
#define OPEN_DIFFSERV_LAYER_4_PORT_NUM_TELNET     23
#define OPEN_DIFFSERV_LAYER_4_PORT_NUM_TFTP       69
#define OPEN_DIFFSERV_LAYER_4_PORT_NUM_WWW        80

/* User Keyword Translations for IP DSCP Values */
#define OPEN_DIFFSERV_IP_DSCP_AF11                10 
#define OPEN_DIFFSERV_IP_DSCP_AF12                12 
#define OPEN_DIFFSERV_IP_DSCP_AF13                14 
#define OPEN_DIFFSERV_IP_DSCP_AF21                18 
#define OPEN_DIFFSERV_IP_DSCP_AF22                20 
#define OPEN_DIFFSERV_IP_DSCP_AF23                22 
#define OPEN_DIFFSERV_IP_DSCP_AF31                26 
#define OPEN_DIFFSERV_IP_DSCP_AF32                28 
#define OPEN_DIFFSERV_IP_DSCP_AF33                30 
#define OPEN_DIFFSERV_IP_DSCP_AF41                34 
#define OPEN_DIFFSERV_IP_DSCP_AF42                36 
#define OPEN_DIFFSERV_IP_DSCP_AF43                38 
#define OPEN_DIFFSERV_IP_DSCP_BE                  0  
#define OPEN_DIFFSERV_IP_DSCP_CS0                 0  
#define OPEN_DIFFSERV_IP_DSCP_CS1                 8  
#define OPEN_DIFFSERV_IP_DSCP_CS2                 16 
#define OPEN_DIFFSERV_IP_DSCP_CS3                 24 
#define OPEN_DIFFSERV_IP_DSCP_CS4                 32 
#define OPEN_DIFFSERV_IP_DSCP_CS5                 40 
#define OPEN_DIFFSERV_IP_DSCP_CS6                 48 
#define OPEN_DIFFSERV_IP_DSCP_CS7                 56 
#define OPEN_DIFFSERV_IP_DSCP_EF                  46 

/* User Keyword Translations for Layer 4 Protocols */
#define OPEN_DIFFSERV_LAYER_4_PROTOCOL_ICMP       1
#define OPEN_DIFFSERV_LAYER_4_PROTOCOL_IGMP       2
#define OPEN_DIFFSERV_LAYER_4_PROTOCOL_IP         255
#define OPEN_DIFFSERV_LAYER_4_PROTOCOL_TCP        6
#define OPEN_DIFFSERV_LAYER_4_PROTOCOL_UDP        17
#define OPEN_DIFFSERV_LAYER_4_PROTOCOL_GRE        47
#define OPEN_DIFFSERV_LAYER_4_PROTOCOL_IPV6       254
#define OPEN_DIFFSERV_LAYER_6_PROTOCOL_ICMP       58

/*********************************************************************
* @purpose  Create the DiffServ class 
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    className       @b{(input)} class name 
* @param    classType       @b{(input)} class type
* @param    classL3Protocol @b{(input)} class L3 protocol
*
* @returns  OPEN_E_NONE      if class is created successfully
* @returns  OPEN_E_FAIL      if failed to create class 
* @returns  OPEN_E_PARAM     if invalid parameter is passed
* @returns  OPEN_E_ERROR     if invalid configuration
* @returns  OPEN_E_EXISTS    if class name already exists
* @returns  OPEN_E_FULL      if class table is full
* @returns  OPEN_E_UNAVAIL   if feature is not supported
*
* @notes   None. 
* 
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServClassCreate(openapiClientHandle_t *client_handle,
                                        open_buffdesc *className,
                                        OPEN_DIFFSERV_CLASS_TYPE_t classType,
                                        OPEN_DIFFSERV_CLASS_L3PROTOCOL_t classL3Protocol);

/*********************************************************************
* @purpose  Delete the DiffServ class 
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    classIndex      @b{(input)} class index 
*
* @returns  OPEN_E_NONE      if class is deleted successfully
* @returns  OPEN_E_FAIL      if failed to create class 
* @returns  OPEN_E_PARAM     if invalid parameter is passed
*
* @notes   None. 
* 
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServClassDelete(openapiClientHandle_t *client_handle,
                                        uint32_t classIndex);

/*********************************************************************
* @purpose  Translate a class name into its associated Class Table index 
*
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    className       @b{(input)}  class name
* @param    classIndex      @b{(output)} class index 
*
* @returns  OPEN_E_NONE      if class name is translated to class index 
* @returns  OPEN_E_FAIL      if failed to translate class name 
* @returns  OPEN_E_PARAM     if invalid parameter is passed
*
* @notes   None. 
* 
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServClassNameToIndex(openapiClientHandle_t *client_handle,
                                             open_buffdesc *className, uint32_t *classIndex);

/*********************************************************************
* @purpose  Determine next sequential row in the Class Table
*
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    classIndex      @b{(input)}  Class index to begin search
* @param    classIndexNext  @b{(output)} Pointer to next sequential class index
*                                        value 
*
* @returns  OPEN_E_NONE      if next class index is retrieved 
* @returns  OPEN_E_FAIL      if failed to get class index 
* @returns  OPEN_E_PARAM     if invalid parameter is passed
*
* @notes   A classIndex value of 0 is used to find the first entry in
*           the table. 
* 
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServClassGetNext(openapiClientHandle_t *client_handle,
                                         uint32_t classIndex, uint32_t *classIndexNext); 

/*********************************************************************
* @purpose Get the class name for the given class index 
*
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    classIndex      @b{(input)}  Class index
* @param    className       @b{(output)} Class name 
*
* @returns  OPEN_E_NONE      if class name is retrieved 
* @returns  OPEN_E_FAIL      if failed to get class name 
* @returns  OPEN_E_PARAM     if invalid parameter is passed
*
* @notes   None 
* 
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServClassNameGet(openapiClientHandle_t *client_handle,
                                         uint32_t classIndex, open_buffdesc *className);

/*********************************************************************
* @purpose Get the class type  
*
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    classIndex      @b{(input)}  Class index
* @param    classType       @b{(output)} Class type 
*
* @returns  OPEN_E_NONE      if class type is retrieved 
* @returns  OPEN_E_FAIL      if failed to get class type 
* @returns  OPEN_E_PARAM     if invalid parameter is passed
*
* @notes   None 
* 
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServClassTypeGet(openapiClientHandle_t *client_handle,
                                         uint32_t classIndex, OPEN_DIFFSERV_CLASS_TYPE_t *classType);

/*********************************************************************
* @purpose Get the class L3 protocol  
*
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    classIndex      @b{(input)}  Class index
* @param    classProtoType  @b{(output)} Class L3 protocol 
*
* @returns  OPEN_E_NONE      if class protocol is retrieved 
* @returns  OPEN_E_FAIL      if failed to get class protocol 
* @returns  OPEN_E_PARAM     if invalid parameter is passed
*
* @notes   None 
* 
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServClassL3ProtoGet(openapiClientHandle_t *client_handle,
                                            uint32_t classIndex, OPEN_DIFFSERV_CLASS_L3PROTOCOL_t *classProtoType);

/*********************************************************************
* @purpose Create a new rule in the Class Rule Table for the specified indexes  
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    classIndex      @b{(input)} Class index
* @param    ruleType        @b{(input)} Class rule type
* @param    ruleVal1        @b{(input)} Rule Value 1
* @param    ruleVal2        @b{(input)} Rule value 2 
*
* @returns  OPEN_E_NONE      if class rule is created successfully
* @returns  OPEN_E_FAIL      if failed to create class rule 
* @returns  OPEN_E_PARAM     if invalid parameter is passed
* @returns  OPEN_E_FULL      if class rule table is full
*
* @notes ruleVal - If rule type supports two input parameters, 
*                  ruleVal1 and ruleVal2 both should be used.
*                  Eg:For OPEN_DIFFSERV_CLASS_RULE_TYPE_DSTIP, 
*                  ruleVal1 should be used for address and 
*                  ruleVal2 should be used for mask. Refer 
*                  OPEN_DIFFSERV_CLASS_RULE_TYPE_t for number of input
*                  parameters supported for the given ruleType.
* 
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServClassRuleCreate(openapiClientHandle_t *client_handle,
                                            uint32_t classIndex, 
                                            OPEN_DIFFSERV_CLASS_RULE_TYPE_t ruleType,
                                            open_buffdesc *ruleVal1,
                                            open_buffdesc *ruleVal2); 

/*********************************************************************
* @purpose  Remove the referenced class from the specified class
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    classIndex      @b{(input)} class index 
* @param    refClassName    @b{(input)} reference class name 
*
* @returns  OPEN_E_NONE      if reference class rule is deleted successfully
* @returns  OPEN_E_FAIL      if failed to create class rule 
* @returns  OPEN_E_PARAM     if invalid parameter is passed
*
* @notes   None. 
* 
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServClassRuleMatchRefClassRemove(openapiClientHandle_t *client_handle,
                                            uint32_t classIndex, open_buffdesc *refClassName);

/*********************************************************************
* @purpose  Determine next sequential row in the Class rule Table
*
* @param    client_handle      @b{(input)}  Client handle from 
*                                           registration API
* @param    classIndex         @b{(input)}  Class index to begin search
* @param    classRuleIndex     @b{(input)}  Class rule index to begin search
* @param    nextClassIndex     @b{(output)} Pointer to next sequential class index
*                                           value
* @param    nextClassRuleIndex @b{(output)} Pointer to next sequential class rule
*                                           index value
* 
* @returns  OPEN_E_NONE      if next class index is retrieved 
* @returns  OPEN_E_FAIL      if failed to get class index 
* @returns  OPEN_E_PARAM     if invalid parameter is passed
*
* @notes    A classIndex of 0 is used to find the first class in the table.
*           A classRuleIndex of 0 is used to find the first rule relative
*           to the specified classIndex.
* 
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServClassRuleGetNext(openapiClientHandle_t *client_handle,
                                             uint32_t classIndex,
                                             uint32_t classRuleIndex,
                                             uint32_t *nextClassIndex,
                                             uint32_t *nextClassRuleIndex);

/*********************************************************************
* @purpose  Get the number of match rules in the Class 
*
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    classIndex      @b{(input)}  class index 
* @param    ruleCount       @b{(output)} class rule count 
*
* @returns  OPEN_E_NONE      if rule count is retrieved successfully
* @returns  OPEN_E_FAIL      if failed to get class rule count 
* @returns  OPEN_E_PARAM     if invalid parameter is passed
*
* @notes   None. 
* 
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServClassRuleCountGet(openapiClientHandle_t *client_handle,
                                             uint32_t classIndex,
                                             uint32_t *ruleCount);

/*********************************************************************
* @purpose   Get the Class Rule match entry type 
*
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    classIndex      @b{(input)}  class index 
* @param    classRuleIndex  @b{(input)}  class rule index
* @param    ruleType        @b{(output)} class rule type 
*
* @returns  OPEN_E_NONE      if class rule match entry type is retrieved 
* @returns  OPEN_E_FAIL      if failed to get rule match entry type 
* @returns  OPEN_E_PARAM     if invalid parameter is passed
*
* @notes   None. 
* 
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServClassRuleMatchEntryTypeGet(openapiClientHandle_t *client_handle,
                                                  uint32_t classIndex,
                                                  uint32_t classRuleIndex,
                                                  OPEN_DIFFSERV_CLASS_RULE_TYPE_t *ruleType);

/*********************************************************************
* @purpose   Get the Class Rule match entry type value 
*
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    classIndex      @b{(input)}  class index 
* @param    classRuleIndex  @b{(input)}  class rule index
* @param    ruleType        @b{(input)}  class rule type 
* @param    ruleVal1        @b{(output)} Rule Value 1
* @param    ruleVal2        @b{(output)} Rule value 2 
*
*
* @returns  OPEN_E_NONE      if class rule match entry type value is retrieved 
* @returns  OPEN_E_FAIL      if failed to get rule match entry type value 
* @returns  OPEN_E_PARAM     if invalid parameter is passed
*
* @notes ruleVal - If rule type supports two parameters, 
*                   ruleVal1 and ruleVal2 both should be used.
*                   Eg:For OPEN_DIFFSERV_CLASS_RULE_TYPE_DSTIP, 
*                   ruleVal1 should be used for address and 
*                   ruleVal2 should be used for mask.
* 
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServClassRuleMatchEntryValueGet(openapiClientHandle_t *client_handle,
                                                   uint32_t classIndex,
                                                   uint32_t classRuleIndex,
                                                   OPEN_DIFFSERV_CLASS_RULE_TYPE_t ruleType,
                                                   open_buffdesc *ruleVal1,
                                                   open_buffdesc *ruleVal2);

#endif
