/*********************************************************************
*
*  Copyright 2016-2023 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_dot1x.h
*
* @purpose   IEEE 802.1x Configuration and Status
*
* @component OpEN
*
* @create    11/16/2016
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_DOT1X_H_INCLUDED
#define OPENAPI_DOT1X_H_INCLUDED

#include "openapi_common.h"
#include "openapi_user_manager.h"

/* Port authorization mode */
typedef enum
{
  OPEN_DOT1X_PORT_FORCE_UNAUTHORIZED = 1,
  OPEN_DOT1X_PORT_AUTO               = 2,
  OPEN_DOT1X_PORT_FORCE_AUTHORIZED   = 3,
  OPEN_DOT1X_PORT_NA                 = 4,
} OPEN_DOT1X_PORT_CONTROL_t;

/* Port authorization status */
typedef enum
{
  OPEN_DOT1X_PORT_STATUS_AUTHORIZED   = 1,
  OPEN_DOT1X_PORT_STATUS_UNAUTHORIZED = 2,
  OPEN_DOT1X_PORT_STATUS_NA           = 3,
} OPEN_DOT1X_PORT_STATUS_t;

/*  PAE State Machine (APM) states */
typedef enum
{
  OPEN_DOT1X_APM_INITIALIZE     = 1,
  OPEN_DOT1X_APM_DISCONNECTED   = 2,
  OPEN_DOT1X_APM_CONNECTING     = 3,
  OPEN_DOT1X_APM_AUTHENTICATING = 4,
  OPEN_DOT1X_APM_AUTHENTICATED  = 5,
  OPEN_DOT1X_APM_ABORTING       = 6,
  OPEN_DOT1X_APM_HELD           = 7,
  OPEN_DOT1X_APM_FORCE_AUTH     = 8,
  OPEN_DOT1X_APM_FORCE_UNAUTH   = 9,
} OPEN_DOT1X_APM_STATES_t;

/* Backend  State Machine (BAM) states */
typedef enum
{
  OPEN_DOT1X_BAM_REQUEST    = 1,
  OPEN_DOT1X_BAM_RESPONSE   = 2,
  OPEN_DOT1X_BAM_SUCCESS    = 3,
  OPEN_DOT1X_BAM_FAIL       = 4,
  OPEN_DOT1X_BAM_TIMEOUT    = 5,
  OPEN_DOT1X_BAM_IDLE       = 6,
  OPEN_DOT1X_BAM_INITIALIZE = 7,
} OPEN_DOT1X_BAM_STATES_t;

/* Vlan Assignment Mode */
typedef enum
{
  OPEN_DOT1X_DEFAULT_ASSIGNED_VLAN = 1,
  OPEN_DOT1X_RADIUS_ASSIGNED_VLAN  = 2,
  OPEN_DOT1X_UNAUTHENTICATED_VLAN  = 3,
  OPEN_DOT1X_GUEST_VLAN            = 4,
  OPEN_DOT1X_VOICE_VLAN            = 5,
  OPEN_DOT1X_MONITOR_MODE_VLAN     = 6,
  OPEN_DOT1X_NOT_ASSIGNED          = 7,
} OPEN_DOT1X_VLAN_ASSIGNED_MODE_t;

/** Port protocol version */
typedef enum
{
  OPEN_DOT1X_PAE_PORT_PROTOCOL_VERSION_1 = 1,   /**< version 1 is the 2001 standard of 802.1x */
  OPEN_DOT1X_PAE_PORT_PROTOCOL_VERSION_2 = 2,   /**< version 2 is the 2004 standard of 802.1x */
  OPEN_DOT1X_PAE_PORT_PROTOCOL_VERSION_3 = 3    /**< version 3 is the 2010 standard of 802.1x */
} OPEN_DOT1X_PAE_PORT_PROTOCOL_VERSION_t;

/** Port PAE capabilities */
typedef enum
{
  OPEN_DOT1X_PAE_PORT_NONE_CAPABLE = 0,         /**< None               */
  OPEN_DOT1X_PAE_PORT_AUTH_CAPABLE              /**< Supplicant         */
} OPEN_DOT1X_PAE_PORT_CAPABILITY;

/*********************************************************************
* @purpose  Gets the dot1x global administrative mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    mode            @b{(output)} Administrative mode
*
* @returns  OPEN_E_NONE        Mode get successful.
* @returns  OPEN_E_FAIL        Mode get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xSystemAuthControlModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Sets the dot1x global administrative mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    mode            @b{(input)}  Administrative mode
*
* @returns  OPEN_E_NONE        Mode set successful.
* @returns  OPEN_E_FAIL        Mode set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xSystemAuthControlModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose  Gets the dot1x default authentication method.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    method          @b{(output)} authentication method
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xDefaultAuthenMethodGet(openapiClientHandle_t *client_handle, OPEN_USER_MGR_AUTH_METHOD_t *method);

/*********************************************************************
* @purpose  Sets the dot1x default authentication method.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    method          @b{(input)}  authentication method
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xDefaultAuthenMethodSet(openapiClientHandle_t *client_handle, OPEN_USER_MGR_AUTH_METHOD_t method);

/*********************************************************************
* @purpose  Gets the dot1x port configured control mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mode            @b{(output)} control mode
*
* @returns  OPEN_E_NONE        Mode get successful.
* @returns  OPEN_E_FAIL        Mode get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
* @This API no longer supported, please use openapiAuthMgrPortControlModeGet instead
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortControlModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_DOT1X_PORT_CONTROL_t *mode);

/*********************************************************************
* @purpose  Sets the dot1x port configured control mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mode            @b{(input)}  control mode
*
* @returns  OPEN_E_NONE        Mode set successful.
* @returns  OPEN_E_FAIL        Mode set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
* @This API no longer supported, please use openapiAuthMgrPortControlModeSet instead
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortControlModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_DOT1X_PORT_CONTROL_t mode);

/*********************************************************************
* @purpose  Gets the dot1x port guest vlan ID.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    vlanId          @b{(output)} VLAN ID
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
* @This API no longer supported, please use openapiAuthMgrPortGuestVlanGet instead
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortGuestVlanGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *vlanId);

/*********************************************************************
* @purpose  Sets the dot1x port guest vlan ID.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    vlanId          @b{(input)}  VLAN ID
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
* @This API no longer supported, please use openapiAuthMgrPortGuestVlanSet instead
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortGuestVlanSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t vlanId);

/*********************************************************************
* @purpose  Sets the dot1x port MAC authentication bypass enable mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mabEnabled      @b{(output)} OPEN_TRUE if MAB enabled, else OPEN_FALSE
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
* @This API no longer supported, please use openapiMabPortMabEnabledGet instead
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortMabEnabledGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_BOOL_t *mabEnabled);

/*********************************************************************
* @purpose  Sets the dot1x port MAC authentication bypass enable mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mabEnabled      @b{(input)}  Set to OPEN_TRUE to enable MAB, OPEN_FALSE to disable
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
* @This API no longer supported, please use openapiMabPortMabEnabledSet instead
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortMabEnabledSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_BOOL_t mabEnabled);

/*********************************************************************
* @purpose  Gets the dot1x port operating control mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mode            @b{(output)} Operating control mode
*
* @returns  OPEN_E_NONE        Mode get successful.
* @returns  OPEN_E_FAIL        Mode get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
* @This API no longer supported, please use openapiAuthMgrPortOperatingControlModeGet instead
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortOperatingControlModeGet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum, OPEN_DOT1X_PORT_CONTROL_t *mode);

/*********************************************************************
* @purpose  Gets the dot1x port status.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    status          @b{(output)} Status
*
* @returns  OPEN_E_NONE        Status get successful.
* @returns  OPEN_E_FAIL        Status get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
* @This API no longer supported, please use openapiAuthMgrPortStatusGet instead
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortStatusGet(openapiClientHandle_t *client_handle,
                                       uint32_t ifNum, OPEN_DOT1X_PORT_STATUS_t *status);

/*********************************************************************
* @purpose  Gets the dot1x port PAE state.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    state           @b{(output)} State
*
* @returns  OPEN_E_NONE        State get successful.
* @returns  OPEN_E_FAIL        State get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortPaeStateGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum, OPEN_DOT1X_APM_STATES_t *state);

/*********************************************************************
* @purpose  Gets the dot1x port back end authorization state.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    state           @b{(output)} State
*
* @returns  OPEN_E_NONE        State get successful.
* @returns  OPEN_E_FAIL        State get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes    This function is not supported from OpEN API version 1.26.
*           port's Backend Authentication state is no longer supported.
*
* @supported-in-version OpEN API Version: 1.14           
* @open-status-deprecated
* @deprecated-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortBackendAuthStateGet(openapiClientHandle_t *client_handle,
                                                 uint32_t ifNum, OPEN_DOT1X_BAM_STATES_t *state);

/*********************************************************************
* @purpose  Gets the dot1x port operational MAB mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mode            @b{(output)} Mode
*
* @returns  OPEN_E_NONE        Mode get successful.
* @returns  OPEN_E_FAIL        Mode get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
* @This API no longer supported, please use openapiMabPortOperationalMabModeGet instead
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortOperationalMabModeGet(openapiClientHandle_t *client_handle,
                                                   uint32_t ifNum, OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Gets the dot1x port assigned VLAN.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    vlan            @b{(output)} VLAN ID
*
* @returns  OPEN_E_NONE        VLAN ID get successful.
* @returns  OPEN_E_FAIL        VLAN ID get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
* @This API no longer supported, please use openapiAuthMgrPortVlanAssignedGet instead
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortVlanAssignedGet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum, uint32_t *vlan);

/*********************************************************************
* @purpose  Gets the dot1x port assigned VLAN reason.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    reason          @b{(output)} Reason
*
* @returns  OPEN_E_NONE        Reason get successful.
* @returns  OPEN_E_FAIL        Reason get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
* @This API no longer supported, please use openapiAuthMgrPortVlanAssignedReasonGet instead
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortVlanAssignedReasonGet(openapiClientHandle_t *client_handle,
                                                   uint32_t ifNum, OPEN_DOT1X_VLAN_ASSIGNED_MODE_t *reason);

/*********************************************************************
* @purpose  Gets the first dot1x logical interface identifier for a given
*           physical port.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID for the physical port
* @param    logicalIfNum    @b{(output)} logical interface identifier
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed or no logical port found.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xLogicalPortFirstGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *logicalIfNum);

/*********************************************************************
* @purpose  Gets the next dot1x logical interface identifier on the
*           same physical port as the given logical interface identifier.
*
* @param    client_handle     @b{(input)}  client handle from registration API
* @param    logicalIfNum      @b{(input)}  logical interface identifier
* @param    nextLogicalIfNum  @b{(output)} next logical interface identifier
*                                          on the same physical port as the
*                                          given logical interface identifier
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed or no next logical port found.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xLogicalPortNextGet(openapiClientHandle_t *client_handle, uint32_t logicalIfNum, uint32_t *nextLogicalIfNum);

/*********************************************************************
* @purpose  Gets the MAC address of the dot1x supplicant present on the
*           specified logical interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    logicalIfNum    @b{(input)}  Interface ID
* @param    macAddr         @b{output)}  supplicant's MAC address, buffdesc must
*                                        provide at least OPEN_MAC_ADDR_LEN bytes
*                                        of space
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xLogicalPortSupplicantMacAddrGet(openapiClientHandle_t *client_handle, uint32_t logicalIfNum, open_buffdesc *macAddr);


/*********************************************************************
* @purpose  Gets the dot1x assigned VLAN ID and reason for the
*           specified logical interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    logicalIfNum    @b{(input)}  Interface ID
* @param    vlan            @b{output)}  VLAN ID
* @param    reason          @b{output)}  reason the VLAN was assigned
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
* @This API no longer supported, please use openapiAuthMgrLogicalPortVlanAssignmentGet instead
*
* @end
*********************************************************************/
open_error_t openapiDot1xLogicalPortVlanAssignmentGet(openapiClientHandle_t *client_handle, uint32_t logicalIfNum, uint32_t *vlan, OPEN_DOT1X_VLAN_ASSIGNED_MODE_t *reason);

/*****************************************************************************
 * @purpose   Set eapol flood mode setting for dot1x
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     adminMode           @b{(input)}  Dot1x eapol flood mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets the eapol flood mode setting for dot1x
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xEapolFloodModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t adminMode);

/*****************************************************************************
 * @purpose   Get eapol flood mode setting for dot1x
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     adminMode           @b{(output)} Dot1x eapol flood mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the eapol flood mode setting for dot1x
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xEapolFloodModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *adminMode);

/*****************************************************************************
 * @purpose   Set the dot1x quiet period on the specified port
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     quietPeriod         @b{(input)}  the uint value for the quiet period
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     The quietPeriod is the initialization value for quietWhile,
 * @notes     which is a timer used by the Authenticator state machine
 * @notes     to define periods of time in which it will not attempt to
 * @notes     acquire a Supplicant.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortQuietPeriodSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t quietPeriod);

/*****************************************************************************
 * @purpose   Get the dot1x quiet period on the specified port
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     quietPeriod         @b{(output)} the uint value for the quiet period
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     The quietPeriod is the initialization value for quietWhile,
 * @notes     which is a timer used by the Authenticator state machine
 * @notes     to define periods of time in which it will not attempt to
 * @notes     acquire a Supplicant.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortQuietPeriodGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *quietPeriod);

/*****************************************************************************
 * @purpose   Set the dot1x transmit period on the specified port
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     txPeriod            @b{(input)}  the uint value for the transmit period
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     The txPeriod is the initialization value for txWhen,
 * @notes     which is a timer used by the Authenticator state machine
 * @notes     to determine when to send an EAPOL EAP Request/Identity
 * @notes     frame to the Supplicant.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortTxPeriodSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t txPeriod);

/*****************************************************************************
 * @purpose   Get the dot1x transmit period on the specified port
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     txPeriod            @b{(output)} the uint value for the transmit period
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     The txPeriod is the initialization value for txWhen,
 * @notes     which is a timer used by the Authenticator state machine
 * @notes     to determine when to send an EAPOL EAP Request/Identity
 * @notes     frame to the Supplicant.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortTxPeriodGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *txPeriod);

/*****************************************************************************
 * @purpose   Get the dot1x port Eapol Frames Received Stats
 *
 * @param     client_handle           @b{(input)}  client handle from registration API
 * @param     intIfNum                @b{(input)}  internal interface number
 * @param     eapolFramesRx           @b{(output)} EAPOL start frames received
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the dot1x port Eapol Frames Received Stats
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortEapolFramesRxGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *eapolFramesRx);

/*****************************************************************************
 * @purpose   Get the dot1x port Eapol Frames Transmitted Stats
 *
 * @param     client_handle           @b{(input)}  client handle from registration API
 * @param     intIfNum                @b{(input)}  internal interface number
 * @param     eapolFramesTx           @b{(output)} EAPOL frames transmitted
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the dot1x port Eapol Frames Transmitted Stats
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortEapolFramesTxGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *eapolFramesTx);

/*****************************************************************************
 * @purpose   Get the dot1x port Eapol Start Frames Received Stats
 *
 * @param     client_handle           @b{(input)}  client handle from registration API
 * @param     intIfNum                @b{(input)}  internal interface number
 * @param     eapolStartFramesRx      @b{(output)} EAPOL start frames received
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the dot1x port Eapol Start Frames Received Stats
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortEapolStartFramesRxGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *eapolStartFramesRx);

/*****************************************************************************
 * @purpose   Get the dot1x port Eapol Logoff Frames Received Stat
 *
 * @param     client_handle           @b{(input)}  client handle from registration API
 * @param     intIfNum                @b{(input)}  internal interface number
 * @param     eapolLogoffFramesRx     @b{(output)} EAPOL logoff frames received
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the dot1x port Eapol Logoff Frames Received Stat
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortEapolLogoffFramesRxGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *eapolLogoffFramesRx);

/*****************************************************************************
 * @purpose   Get number of EAP Response/Identity frames received
 *
 * @param     client_handle           @b{(input)}  client handle from registration API
 * @param     intIfNum                @b{(input)}  internal interface number
 * @param     eapRespIdFramesRx       @b{(output)} EAP Resp/Id frames received
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the number of EAP Response/Identity frames received
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortEapRespIdFramesRxGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *eapRespIdFramesRx);

/*****************************************************************************
 * @purpose   Get the dot1x port Eap Response Frames Received Stat
 *
 * @param     client_handle           @b{(input)}  client handle from registration API
 * @param     intIfNum                @b{(input)}  internal interface number
 * @param     eapResponseFramesRx     @b{(output)} EAP Response frames received
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the dot1x port Eap Response Frames Received Stat
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortEapResponseFramesRxGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *eapResponseFramesRx);

/*****************************************************************************
 * @purpose   Get the dot1x port Eap Request Id Frames Transmitted Stat
 *
 * @param     client_handle           @b{(input)}  client handle from registration API
 * @param     intIfNum                @b{(input)}  internal interface number
 * @param     eapReqIdFramesTx        @b{(output)} EAP Req/Id frames transmitted
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the dot1x port Eap Request Id Frames Transmitted Stat
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortEapReqIdFramesTxGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *eapReqIdFramesTx);

/*****************************************************************************
 * @purpose   Get the dot1x port Eap Request Frames Transmitted Stat
 *
 * @param     client_handle           @b{(input)}  client handle from registration API
 * @param     intIfNum                @b{(input)}  internal interface number
 * @param     eapReqFramesTx          @b{(output)} EAP Request frames transmitted
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the dot1x port Eap Request Frames Transmitted Stat
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortEapReqFramesTxGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *eapReqFramesTx);

/*****************************************************************************
 * @purpose   Get the dot1x port Invalid Eapol Frames Received Stat
 *
 * @param     client_handle           @b{(input)}  client handle from registration API
 * @param     intIfNum                @b{(input)}  internal interface number
 * @param     invalidEapolFramesRx    @b{(output)} Invalid EAPOL frames received
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the dot1x port Invalid Eapol Frames Received Stat
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortInvalidEapolFramesRxGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *invalidEapolFramesRx);

/*****************************************************************************
 * @purpose   Get the dot1x port Eap Length error Frames Received Stat
 *
 * @param     client_handle           @b{(input)}  client handle from registration API
 * @param     intIfNum                @b{(input)}  internal interface number
 * @param     eapLengthErrorFramesRx  @b{(output)} EAP length error frames
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the dot1x port Eap Length error Frames Received Stat
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortEapLengthErrorFramesRxGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *eapLengthErrorFramesRx);

/*****************************************************************************
 * @purpose   Get the dot1x port Last Eapol Frame Version Stat
 *
 * @param     client_handle           @b{(input)}  client handle from registration API
 * @param     intIfNum                @b{(input)}  internal interface number
 * @param     lastEapolFrameVersion   @b{(output)} last EAPOL version received
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the dot1x port Last Eapol Frame Version Stat
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortLastEapolFrameVersionGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *lastEapolFrameVersion);

/*****************************************************************************
 * @purpose   Get the dot1x port Last Eapol Frame Source Stat
 *
 * @param     client_handle           @b{(input)}  client handle from registration API
 * @param     intIfNum                @b{(input)}  internal interface number
 * @param     lastEapolFrameSource    @b{(output)} last EAPOL version received
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the dot1x port Last Eapol Frame Source Stat
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortLastEapolFrameSourceGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, open_buffdesc *lastEapolFrameSource);

/*****************************************************************************
 * @purpose   Clear the dot1x stats on the specified port
 *
 * @param     client_handle           @b{(input)}  client handle from registration API
 * @param     intIfNum                @b{(input)}  internal interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API clears the dot1x stats on the specified port
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortStatsClear(openapiClientHandle_t *client_handle, uint32_t intIfNum);

/*****************************************************************************
 * @purpose   Set the dot1x max reauth request-identity retries count on 
 *            the specified port.
 *
 * @param     client_handle         @b{(input)}  client handle from registration API
 * @param     intIfNum              @b{(input)}  Internal interface number
 * @param     maxReAuthReqIndentity @b{(input)}  Maximum request-Identity value
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets dot1x max reauth request-identity retries count on
 *            the specified port.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiDot1xPortMaxReAuthReqIdentitySet(openapiClientHandle_t *client_handle,
                                        uint32_t intIfNum,
                                        uint32_t maxReAuthReqIndentity);

/*****************************************************************************
 * @purpose   Get the configured dot1x max reauth request-identity retries count
 *            on the specified port.
 *
 * @param     client_handle           @b{(input)}  client handle from registration API
 * @param     intIfNum                @b{(input)}  Internal interface number
 * @param     pMaxReAuthReqIndentity  @b{(output)} Maximum request-Identity count
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches dot1x max reauth request-identity retries count 
 *            on the specified port.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiDot1xPortMaxReAuthReqIdentityGet(openapiClientHandle_t *client_handle, 
                                        uint32_t intIfNum,
                                        uint32_t *pMaxReAuthReqIndentity);

/*****************************************************************************
 * @purpose   Sets the dot1x max requests on the specified port.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API.
 * @param     intIfNum          @b{(input)}  Internal interface number.
 * @param     maxReq            @b{(input)}  Max requests value.
 *
 * @returns   OPEN_E_UNAVAIL            Unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets the dot1x max requests on the specified port.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortMaxReqSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t maxReq);

/*****************************************************************************
 * @purpose   Gets dot1x max requests on the specified port.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API.
 * @param     intIfNum          @b{(input)}  Internal interface number.
 * @param     pMaxReq           @b{(output)} Max requests value.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches dot1x max requests value on the specified port.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortMaxReqGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *pMaxReq);

/*****************************************************************************
 * @purpose   Sets the dot1x supplicant timeout on the specified port.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API.
 * @param     intIfNum          @b{(input)}  Internal interface number.
 * @param     suppTimeout       @b{(input)}  Supplicant timeout value.
 *
 * @returns   OPEN_E_UNAVAIL            Unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets the dot1x supplicant timeout on the specified port.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortSuppTimeoutSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t suppTimeout);

/*****************************************************************************
 * @purpose   Gets dot1x supplicant timeout on the specified port.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API.
 * @param     intIfNum          @b{(input)}  Internal interface number.
 * @param     pSuppTimeout      @b{(output)} Supplicant timeout value.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches dot1x supplicant timeout value on the specified port.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortSuppTimeoutGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *pSuppTimeout);

/*****************************************************************************
 * @purpose   Sets the dot1x Authentiation server timeout on the specified port.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API.
 * @param     intIfNum          @b{(input)}  Internal interface number.
 * @param     serverTimeout     @b{(input)}  Authentication Server timeout value.
 *
 * @returns   OPEN_E_UNAVAIL            Unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets the dot1x server timeout on the specified port.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortServerTimeoutSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t serverTimeout);

/*****************************************************************************
 * @purpose   Gets dot1x Authentiation Server timeout on the specified port.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API.
 * @param     intIfNum          @b{(input)}  Internal interface number.
 * @param     pServerTimeout    @b{(output)} Authentication Server timeout value.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches dot1x Authentiation Server timeout value on the specified port.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortServerTimeoutGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *pServerTimeout);

/*****************************************************************************
 * @purpose   Sets the port's dot1x capabilities (Supplicant or Authenticator).
 *
 * @param     client_handle     @b{(input)}  client handle from registration API.
 * @param     intIfNum          @b{(input)}  Internal interface number.
 * @param     paeCapability     @b{(input)}  Dot1x capabilities.
 *
 * @returns   OPEN_E_UNAVAIL            Unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets port's dot1x capabilities (Supplicant or Authenticator).
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortPaeCapabilitiesSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_DOT1X_PAE_PORT_CAPABILITY paeCapability);

/*****************************************************************************
 * @purpose   Gets dot1x PAE capabilities on the specified port.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API.
 * @param     intIfNum          @b{(input)}  Internal interface number.
 * @param     pPaeCapability    @b{(output)} pae capabilities value.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches dot1x PAE capabilities value on the specified port.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortPaeCapabilitiesGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_DOT1X_PAE_PORT_CAPABILITY *pPaeCapability);

/*****************************************************************************
 * @purpose   Gets dot1x protocol version on the specified port.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API.
 * @param     intIfNum          @b{(input)}  Internal interface number.
 * @param     pProtocolVersion  @b{(output)}  dot1x protocol version of specified port.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches dot1x protocol version value on the specified port.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1xPortProtocolVersionGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_DOT1X_PAE_PORT_PROTOCOL_VERSION_t *pProtocolVersion);



#endif /* OPENAPI_DOT1X_H_INCLUDED */
