/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_dynamic_arp_inspection.h
*
* @purpose   Dynamic ARP table configuration and status.
*
* @component OPEN
*
* @create    12/16/2016
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_DYNAMIC_ARP_INSPECTION_H_INCLUDED
#define OPENAPI_DYNAMIC_ARP_INSPECTION_H_INCLUDED

#include <stdio.h>
#include <stddef.h>
#include <sys/un.h>

#include "openapi_common.h"

#define OPEN_ARP_ACL_NAME_LEN_MIN 1      /**< smallest allowable ACL name */
#define OPEN_ARP_ACL_NAME_LEN_MAX 32     /**< largest allowable ACL name */

/** OpEN uses this enumeration to define ARP ACL Rule Actions */
typedef enum
{
  OPEN_ARP_ACL_RULE_ACTION_PERMIT = 1, /**< PERMIT action */
  OPEN_ARP_ACL_RULE_ACTION_DENY   = 2  /**< DENY action */
} OPEN_ARP_ACL_RULE_ACTION_t;

#define OPEN_DAI_BURST_INTERVAL_MIN  1
#define OPEN_DAI_BURST_INTERVAL_MAX  15

#define OPEN_DAI_RATE_LIMIT_MIN      0 
#define OPEN_DAI_RATE_LIMIT_MAX      300

/** Dynamic ARP Inspection vlan statistics */
typedef struct DAI_VLAN_STATS_S 
{
  uint32_t forwarded;
  uint32_t dropped;
  uint32_t dhcpDrops;
  uint32_t dhcpPermits;
  uint32_t aclDrops;
  uint32_t aclPermits;
  uint32_t aclDenials;
  uint32_t sMacFailures;
  uint32_t dMacFailures;
  uint32_t ipValidFailures;
} OPEN_DAI_VLAN_STATS_t;

/*********************************************************************
* @purpose  API to Check if an ARP ACL exists
*
* @param    client_handle   @b{(input)} client handle from registration API.
* @param    aclName         @b((input)) ACL Name
*
* @returns  OPEN_E_NONE   ACL exists.
* @returns  OPEN_E_PARAM  Bad parameter.
* @returns  OPEN_E_FAIL   ACL does not exist.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiArpAclGet(openapiClientHandle_t *client_handle,
                              open_buffdesc *aclName);

/*********************************************************************
* @purpose  API not supported
*
* @param    client_handle   @b{(input)}   client handle from registration API.
* @param    aclName         @b((input))   ACL Name
* @param    ipAddr          @b((input))   IP address
* @param    macAddr         @b((input))   MAC address
* @param    ipAddrNext      @b((output))  Next IP address
* @param    macAddrNext     @b((output))  Next MAC address
*
* @returns  OPEN_E_UNAVAIL    Functionality unavailable
*
* @notes    This function was deprecated in API version 1.18. Please use
*           @ref openapiArpAclRuleNextGetV2 in new code.
*
* @open-status-deprecated
* @supported-in-version OpEN API Version: 1.14
* @deprecated-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiArpAclRuleNextGet(openapiClientHandle_t *client_handle,
                                        open_buffdesc *aclName, 
                                        open_inet_addr_t ipAddr, 
                                        open_buffdesc *macAddr,
                                        open_inet_addr_t *ipAddrNext, 
                                        open_buffdesc *macAddrNext);

/*********************************************************************
* @purpose  API to get the next ARP ACL Rule in the given ACL
*
* @param    client_handle   @b{(input)}   client handle from registration API.
* @param    aclName         @b((input))   ACL Name
* @param    seqNo           @b((input))   Rule Sequence number
* @param    seqNoNext       @b((output))  Sequence number of next rule
* @param    ipAddrNext      @b((output))  IP address in next rule
* @param    macAddrNext     @b((output))  MAC address in next rule
* @param    actionNext      @b((output))  Action in next rule
*
* @returns  OPEN_E_NONE       Next rule returned.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_FAIL       End of entries, or some other error occurred.
*
* @notes    MAC address string will be in xx:xx:xx:xx:xx:xx format 
* @notes    macAddrNext buffer should have 18 bytes memory.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiArpAclRuleNextGetV2(openapiClientHandle_t *client_handle,
                                        open_buffdesc *aclName, 
                                        uint32_t seqNo, 
                                        uint32_t *seqNoNext, 
                                        open_inet_addr_t *ipAddrNext, 
                                        open_buffdesc *macAddrNext,
                                        OPEN_ARP_ACL_RULE_ACTION_t *actionNext);

/*********************************************************************
* @purpose  API to get the next ARP ACL entry
*
* @param    client_handle   @b{(input)} client handle from registration API.
* @param    aclName         @b((input))  ACL Name
* @param    aclNameNext     @b((output)) Next ACL Name
*
* @returns  OPEN_E_NONE       Next rule returned.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_FAIL       End of entries, or some other error occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiArpAclNextGet(openapiClientHandle_t *client_handle,
                                  open_buffdesc *aclName, 
                                  open_buffdesc *aclNameNext);

/*********************************************************************
* @purpose Set the global dynamic ARP inspection source MAC validate mode.
*
* @param    client_handle     @b{(input)}  Client handle from registration API.
* @param    mode              @b{(input)}  Source MAC validate mode.
*
* @returns  OPEN_E_NONE       Operation succeeded.
* @returns  OPEN_E_FAIL       Unable to set data value.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end                                   
*********************************************************************/
open_error_t openapiDaiSourceMacValidateModeSet(openapiClientHandle_t *client_handle,
                                                OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose Get the global dynamic ARP inspection source MAC validate mode.
*
* @param    client_handle     @b{(input)}  Client handle from registration API.
* @param    mode              @b{(output)} Source MAC validate mode.
*
* @returns  OPEN_E_NONE       Operation succeeded.
* @returns  OPEN_E_FAIL       Unable to get data value.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end                                   
*********************************************************************/
open_error_t openapiDaiSourceMacValidateModeGet(openapiClientHandle_t *client_handle,
                                                OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose Set the global dynamic ARP inspection destination MAC validate mode.
*
* @param    client_handle     @b{(input)}  Client handle from registration API.
* @param    mode              @b{(input)}  Destination MAC validate mode.
*
* @returns  OPEN_E_NONE       Operation succeeded.
* @returns  OPEN_E_FAIL       Unable to set data value.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end                                   
*********************************************************************/
open_error_t openapiDaiDestMacValidateModeSet(openapiClientHandle_t *client_handle,
                                              OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose Get the global dynamic ARP inspection destination MAC validate mode.
*
* @param    client_handle     @b{(input)}  Client handle from registration API.
* @param    mode              @b{(output)} Destination MAC validate mode.
*
* @returns  OPEN_E_NONE       Operation succeeded.
* @returns  OPEN_E_FAIL       Unable to get data value.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end                                   
*********************************************************************/
open_error_t openapiDaiDestMacValidateModeGet(openapiClientHandle_t *client_handle,
                                              OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose Set the global dynamic ARP inspection IP address validate mode.
*
* @param    client_handle     @b{(input)}  Client handle from registration API.
* @param    mode              @b{(input)}  IP address validate mode.
*
* @returns  OPEN_E_NONE       Operation succeeded.
* @returns  OPEN_E_FAIL       Unable to set data value.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end                                   
*********************************************************************/
open_error_t openapiDaiIpAddrValidateModeSet(openapiClientHandle_t *client_handle,
                                             OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose Get the global dynamic ARP inspection IP address validate mode.
*
* @param    client_handle     @b{(input)}  Client handle from registration API.
* @param    mode              @b{(output)} IP address validate mode.
*
* @returns  OPEN_E_NONE       Operation succeeded.
* @returns  OPEN_E_FAIL       Unable to get data value.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end                                   
*********************************************************************/
open_error_t openapiDaiIpAddrValidateModeGet(openapiClientHandle_t *client_handle,
                                             OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Get API for whether DAI considers a port trusted.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  Interface
 * @param     trustMode      @b{(output)} Port trust mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDaiIntfTrustGet(openapiClientHandle_t *client_handle, uint32_t ifNum,
                                    OPEN_BOOL_t *trustMode);

/*****************************************************************************
 * @purpose   Set API for whether DAI considers a port trusted.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  Interface
 * @param     trustMode      @b{(input)}  Port trust mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDaiIntfTrustSet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                                    OPEN_BOOL_t trustMode);

/*****************************************************************************
 * @purpose   Get rate limit value of incoming ARP packets on untrusted interfaces.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  Interface
 * @param     rate           @b{(output)} Upper limit for number of incoming packets 
 *                                        processed per second
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDaiIntfRateLimitGet(openapiClientHandle_t *client_handle, 
                                        uint32_t ifNum, int32_t *rate);

/*****************************************************************************
 * @purpose   Set rate limit value of incoming ARP packets on untrusted interfaces.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  Interface
 * @param     rate           @b{(input)}  Upper limit for number of incoming packets processed
 *                                        per second
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDaiIntfRateLimitSet(openapiClientHandle_t *client_handle, 
                                        uint32_t ifNum, int32_t rate);

/*****************************************************************************
 * @purpose   Get burst interval value of incoming ARP packets on untrusted interfaces.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  Interface
 * @param     burstInterval  @b{(output)} Consecutive interval in seconds, over which the 
 *                                        interface is monitored for a high rate of ARP packets
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDaiIntfBurstIntervalGet(openapiClientHandle_t *client_handle, 
                                            uint32_t ifNum, int32_t *burstInterval);

/*****************************************************************************
 * @purpose   Set burst interval value of incoming ARP packets on untrusted interfaces.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  Interface
 * @param     burstInterval  @b{(input)}  Consecutive interval in seconds, over which the 
 *                                        interface is monitored for a high rate of ARP packets
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDaiIntfBurstIntervalSet(openapiClientHandle_t *client_handle, 
                                            uint32_t ifNum, int32_t burstInterval);

/*****************************************************************************
 * @purpose   Get DAI statistics on a VLAN.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN Id
 * @param     vlanStats      @b{(output)} Statistics for a given VLAN
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDaiVlanStatsGet(openapiClientHandle_t *client_handle, uint32_t vlanId, 
                                    OPEN_DAI_VLAN_STATS_t *vlanStats);

/*****************************************************************************
 * @purpose   Get the next vlan id eligible for DAI configuration, given the 
*             current vlan id.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN Id
 * @param     nextVlan       @b{(output)} next VLAN Id
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDaiVlanNextGet(openapiClientHandle_t *client_handle, 
                                   uint32_t vlanId, uint32_t *nextVlan);

/*****************************************************************************
 * @purpose   Get API for whether Dynamic ARP Inspection is enabled or disabled 
*             on a VLAN.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN Id
 * @param     val            @b{(output)} L7_ENABLE if DAI is enabled on the VLAN
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDaiVlanEnableGet(openapiClientHandle_t *client_handle, 
                                     uint32_t vlanId, uint32_t *val);

/*****************************************************************************
 * @purpose   Set API to enable or disable Dynamic ARP Inspection on a VLAN.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN Id
 * @param     val            @b{(input)}  L7_ENABLE or L7_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDaiVlanEnableSet(openapiClientHandle_t *client_handle, 
                                     uint32_t vlanId, uint32_t val);

/*****************************************************************************
 * @purpose   Get API for whether DAI logging is enabled or disabled on a VLAN.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN Id
 * @param     val            @b{(output)} L7_ENABLE if DAI logging is enabled on 
*                                         the VLAN
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDaiVlanLoggingEnableGet(openapiClientHandle_t *client_handle, 
                                            uint32_t vlanId, uint32_t *val);

/*****************************************************************************
 * @purpose   Set API to enable or disable DAI logging for a VLAN.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN Id
 * @param     val            @b{(input)}  L7_ENABLE or L7_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDaiVlanLoggingEnableSet(openapiClientHandle_t *client_handle, 
                                            uint32_t vlanId, uint32_t val);

/*****************************************************************************
 * @purpose   Get the ARP ACL static flag for a vlan.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN Id
 * @param     staticFlag     @b{(output)} static flag for the VLAN
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDaiVlanArpAclStaticFlagGet(openapiClientHandle_t *client_handle, 
                                               uint32_t vlanId, 
                                               uint32_t *staticFlag);

/*****************************************************************************
 * @purpose   Set the ARP ACL static flag for a vlan.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN Id
 * @param     staticFlag     @b{(input)}  static flag for the VLAN
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDaiVlanArpAclStaticFlagSet(openapiClientHandle_t *client_handle, 
                                               uint32_t vlanId, 
                                               uint32_t staticFlag);

/*****************************************************************************
 * @purpose   Clear DAI statistics on all vlans.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDaiVlanStatsClear(openapiClientHandle_t *client_handle);

/*****************************************************************************
 * @purpose   API to add rule to ARP ACL.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     bufd           @b{(input)}  buffer descriptor containing ACL Name
 * @param     seqNo          @b{(input)}  Rule Sequence Number
 * @param     ipAddr         @b{(input)}  Sender's IP address
 * @param     macAddr        @b{(input)}  Sender's MAC address
 * @param     action         @b{(input)}  PERMIT or DENY action to be performed
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiArpAclRuleAdd(openapiClientHandle_t *client_handle, 
                                  open_buffdesc *bufd, uint32_t seqNo, 
                                  uint32_t ipAddr, open_buffdesc *macAddr, 
                                  OPEN_ARP_ACL_RULE_ACTION_t action);

/*****************************************************************************
 * @purpose   API to remove rule from ARP ACL.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     bufd           @b{(input)}  buffer descriptor containing ACL Name
 * @param     seqNo          @b{(input)}  Rule Sequence Number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiArpAclRuleDelete(openapiClientHandle_t *client_handle, 
                                     open_buffdesc *bufd, uint32_t seqNo);

/*****************************************************************************
 * @purpose   Get the ARP ACL Name for a vlan (Extended Version).
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN Id
 * @param     aclName        @b{(output)} ARP ACL name for this VLAN
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDaiVlanArpAclGet(openapiClientHandle_t *client_handle, 
                                     uint32_t vlanId, open_buffdesc *aclName);


/*****************************************************************************
 * @purpose   Apply the ARP ACL to a range of VLANs.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     aclName        @b{(input)}  ACL name
 * @param     vlanStart      @b{(input)}  Low end of VLAN range
 * @param     vlanEnd        @b{(input)}  High end of VLAN range
 * @param     val            @b{(input)}  L7_ENABLE if ARP ACL is enabled on 
 *                                        all VLANs in the range
 * @param     staticFlag     @b{(input)}  If static is enabled, packets that 
 *                                        do not match any permit statements 
 *                                        of ARP ACL are dropped without 
 *                                        consulting the DHCP snooping bindings.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDaiVlanARPAclRangeSet(openapiClientHandle_t *client_handle, 
                                          open_buffdesc *aclName, 
                                          uint32_t vlanStart, uint32_t vlanEnd, 
                                          uint32_t val, uint32_t staticFlag);

/*****************************************************************************
 * @purpose   Set/reset the ARP ACL Name for a vlan.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN Id
 * @param     aclName        @b{(input)}  ARP ACL name for this VLAN
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     If the aclName is NULL or empty this API clears the ARP ACL 
 *            name for a VLAN
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDaiVlanArpAclSet(openapiClientHandle_t *client_handle, 
                                     uint32_t vlanId, open_buffdesc *aclName);

/*****************************************************************************
 * @purpose   API to create ARP ACL (extensible version).
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     bufd           @b{(input)}  buffer descriptor containing ACL Name
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiArpAclCreate(openapiClientHandle_t *client_handle, 
                                 open_buffdesc *bufd);

/*****************************************************************************
 * @purpose   API to delete ARP ACL (extensible version).
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     bufd           @b{(input)}  buffer descriptor containing ACL Name
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiArpAclDelete(openapiClientHandle_t *client_handle, 
                                 open_buffdesc *bufd);


#endif /* OPENAPI_DYNAMIC_ARP_INSPECTION_H_INCLUDED */
