/**********************************************************************
*
*  Copyright 2022-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_garp.h
*
* @purpose   This code implements the OpEN APIs for GARP
*
* @component OPEN
*
* @create    11/16/2022
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_GARP_H_INCLUDED
#define OPENAPI_GARP_H_INCLUDED

#include "openapi_common.h" 

/* join timer */
#define OPEN_GARP_MIN_JOIN_TIME      10      /* centiseconds */
#define OPEN_GARP_MAX_JOIN_TIME      100     /* centiseconds */
#define OPEN_GARP_DEF_JOIN_TIME      20      /* centiseconds */

/* leave timer */
#define OPEN_GARP_MIN_LEAVE_TIME     20      /* centiseconds */
#define OPEN_GARP_MAX_LEAVE_TIME     600     /* centiseconds */
#define OPEN_GARP_DEF_LEAVE_TIME     60      /* centiseconds */

/* leave all timer */
#define OPEN_GARP_MIN_LEAVE_ALL_TIME 200     /* centiseconds */
#define OPEN_GARP_MAX_LEAVE_ALL_TIME 6000    /* centiseconds */
#define OPEN_GARP_DEF_LEAVE_ALL_TIME 1000    /* centiseconds */


/*****************************************************************************
 * @purpose   Get the GMRP global status.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     globalGmrpStatus  @b{(output)} GMRP global status.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     None.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpGmrpEnableGet(openapiClientHandle_t *client_handle, 
                                      OPEN_CONTROL_t *globalGmrpStatus);

/*****************************************************************************
 * @purpose   Set the GMRP global status.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     globalGmrpStatus  @b{(input)}  GMRP global status.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpGmrpEnableSet(openapiClientHandle_t *client_handle, 
                                      OPEN_CONTROL_t globalGmrpStatus);

/*****************************************************************************
 * @purpose   Get the GMRP interface status.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ifNum             @b{(input)}  interface
 * @param     intfGmrpStatus    @b{(output)} GMRP interface status.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     None.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpGmrpPortEnableGet(openapiClientHandle_t *client_handle,
                                          uint32_t ifNum, 
                                          OPEN_CONTROL_t *intfGmrpStatus);

/*****************************************************************************
 * @purpose   Set the GMRP interface status.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ifNum             @b{(input)}  interface
 * @param     intfGmrpStatus    @b{(input)}  GVRP interface status.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpGmrpPortEnableSet(openapiClientHandle_t *client_handle,
                                          uint32_t ifNum, 
                                          OPEN_CONTROL_t intfGmrpStatus);

/*****************************************************************************
 * @purpose   Get the GVRP global status.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     globalGvrpStatus  @b{(output)} GVRP global status.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     None.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpGvrpEnableGet(openapiClientHandle_t *client_handle,
                                      OPEN_CONTROL_t *globalGvrpStatus);

/*****************************************************************************
 * @purpose   Set the GVRP global status.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     globalGvrpStatus  @b{(input)}  GVRP global status.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpGvrpEnableSet(openapiClientHandle_t *client_handle, 
                                      OPEN_CONTROL_t globalGvrpStatus);

/*****************************************************************************
 * @purpose   Get the GVRP interface status.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ifNum             @b{(input)}  interface
 * @param     intfGvrpStatus    @b{(output)} GVRP interface status.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     None.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpGvrpPortEnableGet(openapiClientHandle_t *client_handle,
                                          uint32_t ifNum, 
                                          OPEN_CONTROL_t *intfGvrpStatus);

/*****************************************************************************
 * @purpose   Set the GVRP interface status.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ifNum             @b{(input)}  interface
 * @param     intfGvrpStatus    @b{(input)}  GVRP interface status.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpGvrpPortEnableSet(openapiClientHandle_t *client_handle,
                                          uint32_t ifNum, 
                                          OPEN_CONTROL_t intfGvrpStatus);

/*****************************************************************************
 * @purpose   Get the GARP Join Time.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ifNum             @b{(input)}  interface
 * @param     joinTime          @b{(output)} GARP join time in centiseconds.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     None.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpJoinTimeGet(openapiClientHandle_t *client_handle, 
                                    uint32_t ifNum, uint32_t *joinTime);

/*****************************************************************************
 * @purpose   Set the GARP Join Time.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ifNum             @b{(input)}  interface
 * @param     joinTime          @b{(input)}  GARP join time in centiseconds.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 * @notes     Refer OPEN_GARP_MIN_JOIN_TIME and OPEN_GARP_MAX_JOIN_TIME 
 *            for minimum and maximum values.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpJoinTimeSet(openapiClientHandle_t *client_handle, 
                                    uint32_t ifNum, uint32_t joinTime);

/*****************************************************************************
 * @purpose   Get the GARP Leave Time.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ifNum             @b{(input)}  interface
 * @param     leaveTime         @b{(output)} GARP leave time in centiseconds.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     None.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpLeaveTimeGet(openapiClientHandle_t *client_handle, 
                                     uint32_t ifNum, uint32_t *leaveTime);

/*****************************************************************************
 * @purpose   Set the GARP Leave Time.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ifNum             @b{(input)}  interface
 * @param     leaveTime         @b{(input)}  GARP leave time in centiseconds.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 * @notes     Refer OPEN_GARP_MIN_LEAVE_TIME and OPEN_GARP_MAX_LEAVE_TIME 
 *            for minimum and maximum values.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpLeaveTimeSet(openapiClientHandle_t *client_handle, 
                                     uint32_t ifNum, uint32_t leaveTime);

/*****************************************************************************
 * @purpose   Get the GARP Leave All Time.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ifNum             @b{(input)}  interface
 * @param     leaveAllTime      @b{(output)} GARP leave All time in centiseconds.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     None.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpLeaveAllTimeGet(openapiClientHandle_t *client_handle,
                                        uint32_t ifNum, uint32_t *leaveAllTime);

/*****************************************************************************
 * @purpose   Set the GARP Leave All Time.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ifNum             @b{(input)}  interface
 * @param     leaveAllTime      @b{(input)}  GARP leave All time in centiseconds.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 * @notes     Refer OPEN_GARP_MIN_LEAVE_ALL_TIME and OPEN_GARP_MAX_LEAVE_ALL_TIME 
 *            for minimum and maximum values.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpLeaveAllTimeSet(openapiClientHandle_t *client_handle,
                                        uint32_t ifNum, uint32_t leaveAllTime);

/*****************************************************************************
 * @purpose  Get the number of failed GMRP registrations of the interface.
 *
 * @param    client_handle           @b{(input)}  Client handle from 
 *                                                registration API.
 * @param    ifNum                   @b{(input)}  Internal interface number.
 * @param    gmrpFailedRegistrations @b{(output)} Number of failed GMRP
 *                                                registrations.
 *
 * @returns  OPEN_E_NONE             On success.
 * @returns  OPEN_E_FAIL             On failure.
 * @returns  OPEN_E_PARAM            Invalid argument.
 * @returns  OPEN_E_NOT_FOUND        If this Counter is not supported.
 * @returns  OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes    None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpGmrpFailedRegistrationsGet(openapiClientHandle_t *client_handle,
                                                   uint32_t ifNum,
                                                   uint32_t *gmrpFailedRegistrations);

/*****************************************************************************
 * @purpose  Get the number of GMRP PDUs received on an interface.
 *
 * @param    client_handle           @b{(input)}  Client handle from 
 *                                                registration API.
 * @param    ifNum                   @b{(input)}  Internal interface number.
 * @param    gmrpPduRxCount          @b{(output)} Number GMRP PDUs received.
 *
 * @returns  OPEN_E_NONE             On success.
 * @returns  OPEN_E_FAIL             On failure.
 * @returns  OPEN_E_PARAM            Invalid argument.
 * @returns  OPEN_E_NOT_FOUND        If this Counter is not supported.
 * @returns  OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes    None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpGmrpReceivePDUCountGet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum,
                                               uint32_t *gmrpPduRxCount);

/*****************************************************************************
 * @purpose  Get the number of GMRP PDUs transmitted from an interface.
 *
 * @param    client_handle           @b{(input)}  Client handle from 
 *                                                registration API.
 * @param    ifNum                   @b{(input)}  Internal interface number.
 * @param    gmrpPduTxCount          @b{(output)} Number GMRP PDUs transmitted.
 *
 * @returns  OPEN_E_NONE             On success.
 * @returns  OPEN_E_FAIL             On failure.
 * @returns  OPEN_E_PARAM            Invalid argument.
 * @returns  OPEN_E_NOT_FOUND        If this Counter is not supported.
 * @returns  OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes    None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpGmrpTransmitPDUCountGet(openapiClientHandle_t *client_handle,
                                                uint32_t ifNum,
                                                uint32_t *gmrpPduTxCount);

/*****************************************************************************
 * @purpose  Get the number of failed GVRP registrations of the interface.
 *
 * @param    client_handle           @b{(input)}  Client handle from 
 *                                                registration API.
 * @param    ifNum                   @b{(input)}  Internal interface number.
 * @param    gvrpFailedRegistrations @b{(output)} Number of failed GVRP
 *                                                registrations.
 *
 * @returns  OPEN_E_NONE             On success.
 * @returns  OPEN_E_FAIL             On failure.
 * @returns  OPEN_E_PARAM            Invalid argument.
 * @returns  OPEN_E_NOT_FOUND        If this Counter is not supported.
 * @returns  OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes    None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpGvrpFailedRegistrationsGet(openapiClientHandle_t *client_handle,
                                                   uint32_t ifNum,
                                                   uint32_t *gvrpFailedRegistrations);

/*****************************************************************************
 * @purpose  Get the number of GVRP PDUs received on an interface.
 *
 * @param    client_handle           @b{(input)}  Client handle from 
 *                                                registration API.
 * @param    ifNum                   @b{(input)}  Internal interface number.
 * @param    gvrpPduRxCount          @b{(output)} Number GVRP PDUs received.
 *
 * @returns  OPEN_E_NONE             On success.
 * @returns  OPEN_E_FAIL             On failure.
 * @returns  OPEN_E_PARAM            Invalid argument.
 * @returns  OPEN_E_NOT_FOUND        If this Counter is not supported.
 * @returns  OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes    None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpGvrpReceivePDUCountGet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum,
                                               uint32_t *gvrpPduRxCount);

/*****************************************************************************
 * @purpose  Get the number of GVRP PDUs transmitted from an interface.
 *
 * @param    client_handle           @b{(input)}  Client handle from 
 *                                                registration API.
 * @param    ifNum                   @b{(input)}  Internal interface number.
 * @param    gvrpPduTxCount          @b{(output)} Number GVRP PDUs transmitted.
 *
 * @returns  OPEN_E_NONE             On success.
 * @returns  OPEN_E_FAIL             On failure.
 * @returns  OPEN_E_PARAM            Invalid argument.
 * @returns  OPEN_E_NOT_FOUND        If this Counter is not supported.
 * @returns  OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes    None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiGarpGvrpTransmitPDUCountGet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum,
                                               uint32_t *gvrpPduTxCount);
#endif /* OPENAPI_GARP_H_INCLUDED */
