/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_instru_nos.h
*
* @purpose   System Information 
*
* @component OPEN
*
* @create    07/31/2014
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_INSTRU_NOS_H_INCLUDED
#define OPENAPI_INSTRU_NOS_H_INCLUDED

#include "openapi_common.h"
#include "openapi_instru_bst.h"
#include "feature.h"

typedef time_t OPEN_TIME_t;
 
/** These data structure are used for OpEN API only */
#define OPEN_INSTRU_MAX_PACKET_SIZE 1588

/* Maximum  length of interface list that can be specified in JSON */
#define OPEN_MAX_PORT_LIST_STR      256

/** Base type for declarations */
#define OPEN_MASK_BASE_UNIT         unsigned int
#define OPEN_MASKWID                32

/** (internal) Number of BVIEW_MASK_BASE_UNITs needed to contain _max bits */
#define OPEN_MASK_SIZE(_max)        (((_max) + OPEN_MASKWID - 1) / OPEN_MASKWID)

#define CONVERT_32_TO_64_INT(dest, src)                                      \
             do{                                                             \
                 int i = 0;                                                  \
                 for (i = 0;i < (sizeof(src)/sizeof(uint32_t)); i++)         \
                 {                                                           \
                   *(((uint64_t*)(&dest)) + i) = *(((uint32_t*)(&src)) + i); \
                 }                                                           \
               }while(0);
    
#define CONVERT_64_TO_32_INT(dest, src)                             \
          do{                                                       \
            int i = 0;                                              \
            for (i=0;i < (sizeof(src)/sizeof(uint64_t)); i++)       \
            {                                                       \
              *(((uint32_t*)&dest) + i) = *(((uint64_t*)&src) + i); \
            }                                                       \
          }while(0); 

typedef struct
{
  /** Number of ports of Asic */
  int numPorts;
  /** Number of unicast queues */
  int numUnicastQueues;
  /** Number of unicast queue groups */
  int numUnicastQueueGroups;
  /** Number of multicast queues */
  int numMulticastQueues;
  /** Number of service pools */
  int numServicePools;
  /** Number of common pools */
  int numCommonPools;
  /** Number of CPU queues */
  int numCpuQueues;
  /** Number of RQE queues */
  int numRqeQueues;
  /** Number of RQE queue pools */
  int numRqeQueuePools;
  /** Number of priority groups */
  int numPriorityGroups;
  /** can this ASIC provide accurate time ? */
  bool support1588;
  /** MMU Cell to BYTE conversion*/
  int cellToByteConv; 
  /** Number of Traffic classes */
  int numTrafficClasses;
  /** support for lossless class */
  bool lossless_class;
  /** support for lossy class */
  bool lossy_class;
  /** support for isolated buffers */
  bool isolated_buffers;
  /**support for dscp to tc mark */
  bool dscp_tc_mark;
  /** support for pbr */
  bool pbr;
  /** support for dscp to pfc*/
  bool dscp_pfc;
  /** support for storage protocol identification */
  bool strg_proto_identification;
  /** number of cos per port*/
  int  num_cos;
} OPEN_ASIC_CAPABILITIES_ALL_t;

/* Interface storage */
typedef struct _open_port_mask_     
{ 
  OPEN_MASK_BASE_UNIT   value[OPEN_MASK_SIZE(OPEN_ASIC_MAX_PORTS)];
  uint8_t               port_list_str[OPEN_MAX_PORT_LIST_STR];

} OPEN_PORT_MASK_t;

#if L7_FEAT_IFA || L7_FEAT_FT
typedef struct _open_instru_ipv4_
{
  uint32_t addr;
  uint32_t mask;
} OPEN_INSTRU_IPV4_t;

typedef struct _open_instru_ipv6_
{
  open_in6_addr_t addr;
  open_in6_addr_t mask;
} OPEN_INSTRU_IPV6_t;

typedef struct _open_instru_ip_prefix_
{
  unsigned short family;
  union
  {
    OPEN_INSTRU_IPV4_t ip;
    OPEN_INSTRU_IPV6_t ipv6;
  } host;
} OPEN_INSTRU_IP_PREFIX_t;

typedef struct _open_instru_ip_addr_
{
  unsigned short family;
  union
  {
    struct in_addr  addr;
    struct in6_addr addr6;
  } ipaddr;
} OPEN_INSTRU_IP_ADDR_t;
#endif

/*********************************************************************
* @purpose  Get System MAC Address
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    mac              @b{(output)} MAC Address
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiMacAddrGet(openapiClientHandle_t *client_handle, open_buffdesc *mac);

/*********************************************************************
* @purpose  Get System Time
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    second           @b{(output)} Time
* @param    nanosecond       @b{(output)} Time
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiTimeGet(openapiClientHandle_t *client_handle, int *second, int *nanosecond);

/*********************************************************************
* @purpose  Get Chip Family name
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    buf              @b{(output)} family name to be fetched
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiChipFamilyNameGet(openapiClientHandle_t *client_handle, open_buffdesc *buf);

/*********************************************************************
* @purpose  Get Chip Max Ports
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    maxPhysPorts     @b{(output)} MAX Physical ports 
* @param    maxHiGigPorts    @b{(output)} MAX Hi Gig Ports
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiHwMaxPortsGet(openapiClientHandle_t *client_handle,
                                  int *maxPhysPorts, int *maxHiGigPorts);

/*********************************************************************
* @purpose  Get LAG string from SDK lag number
*
*
* @param    client_handle   @b{(input)}          Openapi connection handle
* @param    asic            @b{(input)}          The ASIC number 
* @param    lag             @b{(input)}          SDK trunk number 
* @param    dst             @b{(output)}         Lag string 
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.9
*
* @end
**********************************************************************/
open_error_t openapiLagTranslateToNotation(openapiClientHandle_t * client_handle,
                                           int asic, int lag, open_buffdesc *dst);

/*********************************************************************
* @purpose  Get HTTP listening port. 
*
*
* @param    client_handle    @b{(input)}         Openapi connection handle
* @param    asic             @b{(input)}         The UNIT number 
* @param    httpPort         @b{(output)}        HTTP listening port 
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
**********************************************************************/
open_error_t openapiHttpPortGet(openapiClientHandle_t * client_handle,
                                int asic, int *httpPort);

/*************************************************************************
 * @purpose Obtain Complete ASIC Statistics Report of maximum buffer values
 *
 * @param   client_handle  @b{(input)}         OpenAPI connection Handle
 * @param   asic          @b{(input)}         Asic/Unit number 
 * @param   data1         @b{(output)}        Snapshot data
 * @param   data2         @b{(output)}        Eucq snapshot data
 * @param   data3         @b{(output)}        Emcq Snapshot data
 * @param   time          @b{(output)}        Time of retrieving data
 *
 * @returns  OPEN_E_NONE
 * @returns  OPEN_E_ERROR 
 *
 * @notes    none
 *
 * @supported-in-version OpEN API Version: 1.10
 *
 * @end
 ****************************************************************************/
open_error_t openapiBstMaxSnapshotGet(openapiClientHandle_t *client_handle,
                                      int asic,
                                      OPEN_BST_ASIC_SNAPSHOT_DATA_t  *data1,
                                      OPEN_BST_EGRESS_UC_QUEUE_DATA_t *data2,
                                      OPEN_BST_EGRESS_MC_QUEUE_DATA_t  *data3,
                                      time_t *time);

/*************************************************************************
 * @purpose Obtain compressed complete ASIC Statistics Report of maximum buffer values
 *
 * @param   client_handle  @b{(input)}         OpenAPI connection Handle
 * @param   asic          @b{(input)}         Asic/Unit number
 * @param   compSnapshot  @b{(output)}        compressed snapshot data
 * @param   time          @b{(output)}        Time of retrieving data
 *
 * @returns  OPEN_E_NONE
 * @returns  OPEN_E_ERROR
 *
 * @notes    zlib library is used for compression of data.
 *
 * @supported-in-version OpEN API Version: 1.14
 *
 * @end
 ****************************************************************************/
open_error_t
openapiBstMaxSnapshotCompressedGet(openapiClientHandle_t *client_handle,
                                   int asic,
                                   open_buffdesc *compSnapshot,
                                   time_t *time);

/*********************************************************************
* @brief  Register with System for TCAM packets 
*
*
* @param[in]   client_handle                  OpenAPI connection Handle
* @param[in]   asic                          ASIC number
* @param[in]   name                          Client name 
* @param[in]   agentId                       agentId to be Registered
*
* @retval   BVIEW_STATUS_FAILURE      Registration is failure. 
*
* @retval   BVIEW_STATUS_SUCCESS      Registration is successful
*
* @retval   BVIEW_STATUS_UNSUPPORTED  Packet TCAM registration is not supported 
*                                    
* @notes    none
*
* @supported-in-version OpEN API Version: 1.12
*
* @end
*********************************************************************/

open_error_t
openapiSystemPacketTcamRegister(openapiClientHandle_t *client_handle,
                                int asic,
                                open_buffdesc *name,
                                int  agentId);

/*********************************************************************
 * @purpose   Get HW queue map of cosqs in the system
 *
 * @param   client_handle  @b{(input)}         OpenAPI connection Handle
 * @param   asic          @b{(input)}         Asic/Unit number 
 * @param   cosq_hwq_map  @b{(output)}        H/W queue map
 *
 * @returns  OPEN_E_NONE
 * @returns  OPEN_E_ERROR
 *
 * @notes    This function was deprecated in API version 1.19. Please use
 *           @ref openapiSystemCosqHwqFullMapGetV2 in new code.
 *
 * @supported-in-version OpEN API Version: 1.14
 * @deprecated-in-version OpEN API Version: 1.19
 *
 * @end
 *********************************************************************/
open_error_t
openapiSystemCosqHwqFullMapGet(openapiClientHandle_t * client_handle,
                               int asic, OPEN_SYSTEM_COSQ_HWQ_MAP_t *cosq_hwq_map);

/*********************************************************************
 * @purpose   Get the HW queue number of a unicast cosq
 *
 * @param   client_handle  @b{(input)}         OpenAPI connection Handle
 * @param   asic          @b{(input)}         Asic/Unit number 
 * @param   port          @b{(input)}         Port number
 * @param   cosq          @b{(input)}         Cosq number
 * @param   hwQNum        @b{(output)}        H/W queue number
 *
 * @returns  OPEN_E_NONE
 * @returns  OPEN_E_ERROR
 *
 * @notes    none
 *
 * @supported-in-version OpEN API Version: 1.14
 *
 * @end
 *********************************************************************/
open_error_t
openapiSystemUCosqHwqMapGet(openapiClientHandle_t * client_handle,
                            int asic, int port, int cosq, int *hwQNum);

/*********************************************************************
 * @purpose   Get the HW queue number of a multicast cosq
 *
 * @param   client_handle  @b{(input)}         OpenAPI connection Handle
 * @param   asic          @b{(input)}         Asic/Unit number 
 * @param   port          @b{(input)}         Port number
 * @param   cosq          @b{(input)}         Cosq number
 * @param   hwQNum        @b{(output)}        H/W queue number
 *
 * @returns  OPEN_E_NONE
 * @returns  OPEN_E_ERROR
 *
 * @notes    none
 *
 * @supported-in-version OpEN API Version: 1.14
 *
 * @end
 *********************************************************************/
open_error_t
openapiSystemMCosqHwqMapGet(openapiClientHandle_t * client_handle,
                            int asic, int port, int cosq, int *hwQNum);

/*********************************************************************
 * @purpose   Get the port and cos queue number of a unicast HW queue
 *
 * @param   client_handle  @b{(input)}         OpenAPI connection Handle
 * @param   asic          @b{(input)}         Asic/Unit number 
 * @param   hwQNum        @b{(input)}         H/W queue number
 * @param   port          @b{(output)}        Port number
 * @param   cosq          @b{(output)}        Cosq number
 *
 * @returns  OPEN_E_NONE
 * @returns  OPEN_E_ERROR
 *
 * @notes    none
 *
 * @supported-in-version OpEN API Version: 1.14
 *
 * @end
 *********************************************************************/
open_error_t
openapiSystemHwqUCosqMapGet(openapiClientHandle_t * client_handle,
                            int asic, int hwQNum, int *port, int *cosq);

/*********************************************************************
 * @purpose   Get the port and cos queue number of a multicast HW queue
 *
 * @param   client_handle  @b{(input)}         OpenAPI connection Handle
 * @param   asic          @b{(input)}         Asic/Unit number 
 * @param   hwQNum        @b{(input)}         H/W queue number
 * @param   port          @b{(output)}        Port number
 * @param   cosq          @b{(output)}        Cosq number
 *
 * @returns  OPEN_E_NONE
 * @returns  OPEN_E_ERROR
 *
 * @notes    none
 *
 * @supported-in-version OpEN API Version: 1.14
 *
 * @end
 *********************************************************************/
open_error_t
openapiSystemHwqMCosqMapGet(openapiClientHandle_t * client_handle,
                            int asic, int hwQNum, int *port, int *cosq);

/*********************************************************************
* @purpose  Asic capabilities get function
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    asicCap       @b{(output)}        Asic capability
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiAsicCapabilityGet(openapiClientHandle_t *client_handle,
                                      int asic,
                                      OPEN_ASIC_CAPABILITIES_ALL_t *asicCap);

/*********************************************************************
 * @purpose   Get HW queue map of cosqs in the system
 *
 * @param   client_handle  @b{(input)}         OpenAPI connection Handle
 * @param   asic          @b{(input)}         Asic/Unit number 
 * @param   cosq_hwq_map  @b{(output)}        H/W queue map
 *
 * @returns  OPEN_E_NONE
 * @returns  OPEN_E_ERROR
 *
 * @notes    none
 *
 * @supported-in-version OpEN API Version: 1.19
 *
 * @end
 *********************************************************************/
open_error_t
openapiSystemCosqHwqFullMapGetV2(openapiClientHandle_t * client_handle,
                                 int asic, OPEN_SYSTEM_COSQ_HWQ_MAP_V2_t *cosq_hwq_map);
#endif 
