/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_instru_pt.h
*
* @purpose   Instrumentation PT Specific APIs
*
* @component OPEN
*
* @create    03/27/2015
*
* @end
*
**********************************************************************/


#ifndef OPENAPI_PT_H_INCLUDED
#define OPENAPI_PT_H_INCLUDED
#include <string.h>
#include <stdio.h>
#include <stdbool.h>

#include "openapi_common.h"
#include "openapi_instru_nos.h"
#include "openapi_tcam.h"

/** These data structure are used for OpEN API only */
#define     OPEN_PT_MAX_PACKET_SIZE OPEN_INSTRU_MAX_PACKET_SIZE

/** Base type for declarations */
#define     OPEN_MASK_BASE_UNIT     unsigned int
#define     OPEN_MASKWID            32

/** (internal) Number of OPEN_MASK_BASE_UNITs needed to contain _max bits */
#define     OPEN_MASK_SIZE(_max)    (((_max) + OPEN_MASKWID - 1) / OPEN_MASKWID)

/* Instru Packet trace policy match rule number */
#define     OPEN_PT_MATCH_RULE_NUM  61

/* Packet metering related defines */
#define     OPEN_PT_CPU_RATE_LIMIT        3 
#define     OPEN_PT_CPU_RATE_LIMIT_BURST  3

/** Packet Including Checksum*/
typedef struct
{
  unsigned char     data[OPEN_PT_MAX_PACKET_SIZE];
  unsigned int      pkt_len;
} OPEN_PT_PACKET_t;

/* Drop reasons  */
typedef enum
{
  OPEN_PT_NO_DROP         = 1,
  OPEN_PT_DROP_REASON_INTERNAL,
  OPEN_PT_DROP_REASON_MPLS_LABEL_LKUP_MISS,
  OPEN_PT_DROP_REASON_MPLS_INVALID_ACTION,
  OPEN_PT_DROP_REASON_MPLS_INVALID_PAYLOAD,
  OPEN_PT_DROP_REASON_MPLS_TTL_CHECK_FAIL,
  OPEN_PT_DROP_REASON_MPLS_INVALID_CTRL_WORD,
  OPEN_PT_DROP_REASON_L2_GRE_SIP_LKUP_MISS,
  OPEN_PT_DROP_REASON_L2_GRE_VPN_LKUP_MISS,
  OPEN_PT_DROP_REASON_L2_GRE_TUNNEL_ERROR,
  OPEN_PT_DROP_REASON_VXLAN_SIP_LKUP_MISS, 
  OPEN_PT_DROP_REASON_VXLAN_VNID_LKUP_MISS,
  OPEN_PT_DROP_REASON_VXLAN_TUNNEL_ERROR,
  OPEN_PT_DROP_REASON_VLAN_NOT_VALID,
  OPEN_PT_DROP_REASON_ING_PORT_NOT_IN_VLAN_MEMBER,
  OPEN_PT_DROP_REASON_TPID_MISMATCH,
  OPEN_PT_DROP_REASON_IPV4_PROTO_ERROR,
  OPEN_PT_DROP_REASON_HIGIG_LP_BACK,
  OPEN_PT_DROP_REASON_HIGIG_MIRROR_ONLY,
  OPEN_PT_DROP_REASON_HIGIG_UNKNOWN_HDR,
  OPEN_PT_DROP_REASON_HIGIG_UNKNOWN_OP_CODE,
  OPEN_PT_DROP_REASON_LAG_FAIL_LP_BACK,
  OPEN_PT_DROP_REASON_L2_SRC_EQ_L2_DST,
  OPEN_PT_DROP_REASON_IPV6_PROTO_ERROR,
  OPEN_PT_DROP_REASON_NIV_VNTAG_PRESENT,
  OPEN_PT_DROP_REASON_NIV_VNTAG_NOT_PRESENT,
  OPEN_PT_DROP_REASON_NIV_VNTAG_FORMAT,
  OPEN_PT_DROP_REASON_TRILL_ERROR_FRAME,
  OPEN_PT_DROP_REASON_BPDU,
  OPEN_PT_DROP_REASON_BAD_UDP_CHKSUM,
  OPEN_PT_DROP_REASON_TUNNEL_DECAP_ECN_ERROR,
  OPEN_PT_DROP_REASON_IPV4_HDR_ERROR,
  OPEN_PT_DROP_REASON_IPV6_HDR_ERROR,
  OPEN_PT_DROP_REASON_PARITY_ERROR,
  OPEN_PT_DROP_REASON_RPF_CHK_FAIL,
  OPEN_PT_DROP_REASON_STAGE_LKUP,
  OPEN_PT_DROP_REASON_STAGE_INGRESS,
  OPEN_PT_DROP_REASON_MAX = 128
} OPEN_PT_DROP_REASON_t;

/* Drop location enums */
typedef enum _open_pt_drop_loc_s
{
  OPEN_PT_NO_DROP_LOC = 1,
  OPEN_PT_DROP_LOC_INGRESS,
  OPEN_PT_DROP_LOC_PIPELINE,
  OPEN_PT_DROP_LOC_EGRESS,
} OPEN_PT_DROP_LOC_t;

/* This is the size of drop reason bit map interms of 32 bits */  
#define OPEN_PT_DROP_REASON_MAX_SIZE  4

#define OPEN_PT_DROP_REASON_STR_MAX_ELE 32
#define OPEN_PT_DROP_REASON_STR_SIZE  64

/* Structure to hold drop reason information */
typedef struct _open_pt_drop_reason_info_s
{
  OPEN_TIME_t tv;
  uint32_t code[OPEN_PT_DROP_REASON_MAX_SIZE];
  uint32_t num_of_reasons;
  char reason_str[OPEN_PT_DROP_REASON_STR_MAX_ELE][OPEN_PT_DROP_REASON_STR_SIZE];
  OPEN_PT_DROP_LOC_t loc;
} OPEN_PT_DROP_REASON_INFO_t;

/* Interface storage */
typedef struct
{
  OPEN_MASK_BASE_UNIT   value[OPEN_MASK_SIZE(OPEN_PT_DROP_REASON_MAX)];
} OPEN_PT_DROP_REASON_MASK_t;

typedef struct _open_pt_drop_reason_data_s
{
  OPEN_TIME_t tv;
  uint32_t code[OPEN_PT_DROP_REASON_MAX_SIZE];
  OPEN_PT_DROP_LOC_t loc;
  OPEN_PT_DROP_REASON_MASK_t reason_mask;
} OPEN_PT_DROP_REASON_DATA_t;


/* Drop counters */
typedef struct _open_pt_pkt_drop_ctrs_s
{
  uint64_t ingress_drop_cnt;
  uint64_t pipeline_drop_cnt;
  uint64_t egress_drop_cnt;
} OPEN_PT_PKT_DROP_CTRS_t;

typedef struct _open_pt_drop_ctrs_req_info_s
{
   unsigned int             totalRules;
   unsigned int             ruleMatchId;
   OPEN_TCAM_API_POLICYID_t icapPolicyHandle;
   OPEN_TCAM_API_POLICYID_t egressPolicyHandle;
   OPEN_TCAM_API_POLICYID_t cpuPolicyHandle;
} OPEN_PT_DROP_CTRS_REQ_INFO_t;

typedef enum _OPEN_pt_hashing_info_mask_
{
  OPEN_PT_NO_HASHING_RESOLUTION = 0,
  /** level 1 ecmp hashing resolution done */
  OPEN_PT_ECMP_1_RESOLUTION = 1,
  /** level 2 ecmp hashing resolution done */
  OPEN_PT_ECMP_2_RESOLUTION = 2,
  /** trunk hashing resolution done*/
  OPEN_PT_TRUNK_RESOLUTION = 3,
  /** hg trunk hashing resolution done*/
  OPEN_PT_FABRIC_TRUNK_RESOLUTION = 4
} OPEN_PT_HASHING_INFO_MASK_t;

/** packet hashing resolution information */
typedef struct _OPEN_pt_hashing_info_
{
  OPEN_PT_HASHING_INFO_MASK_t flags;
  int ecmp_1_group;           /** multipath egress forwarding object. */
  int ecmp_1_egress;          /** ecmp destination interface */
  int ecmp_2_group;           /** multipath egress forwarding object. */
  int ecmp_2_egress;          /** ecmp destination interface */
  int trunk;                  /** destination trunk group */
  int trunk_member;           /** destination member port which packet egress. */
  int fabric_trunk;           /** destination hg trunk group */
  int fabric_trunk_member;    /** destination member hg port which packet will egress. */
} OPEN_PT_HASHING_INFO_t;


/** packet trace resolution enums */
typedef enum _OPEN_pt_resolution_type_t
{
  OPEN_PT_RESOLUTIONUNKOWN = 0,
  OPEN_PT_RESOLUTIONCONTROLPKT = 1, /** 1 */
  OPEN_PT_RESOLUTIONOAMPKT = 2,
  OPEN_PT_RESOLUTIONBFDPKT = 3,
  OPEN_PT_RESOLUTIONBPDUPKT = 4,
  OPEN_PT_RESOLUTION1588PKT = 6,
  OPEN_PT_RESOLUTIONKNOWNL2UCPKT = 8,
  OPEN_PT_RESOLUTIONUNKNOWNL2UCPKT = 9,
  OPEN_PT_RESOLUTIONKNOWNL2MCPKT = 10,
  OPEN_PT_RESOLUTIONUNKNOWNL2MCPKT = 11,
  OPEN_PT_RESOLUTIONL2BCPKT = 12,
  OPEN_PT_RESOLUTIONKNOWNL3UCPKT = 16,
  OPEN_PT_RESOLUTIONUNKNOWNL3UCPKT = 17,
  OPEN_PT_RESOLUTIONKNOWNIPMCPKT = 18,
  OPEN_PT_RESOLUTIONUNKNOWNIPMCPKT = 19,
  OPEN_PT_RESOLUTIONKNOWNMPLSL2PKT = 24,
  OPEN_PT_RESOLUTIONUNKNOWNMPLSPKT = 25,
  OPEN_PT_RESOLUTIONKNOWNMPLSL3PKT = 26,
  OPEN_PT_RESOLUTIONKNOWNMPLSPKT = 28,
  OPEN_PT_RESOLUTIONKNOWNMIMPKT = 32,
  OPEN_PT_RESOLUTIONUNKNOWNMIMPKT = 33,
  OPEN_PT_RESOLUTIONKNOWNTRILLPKT = 40,
  OPEN_PT_RESOLUTIONUNKNOWNTRILLPKT = 41,
  OPEN_PT_RESOLUTIONKNOWNNIVPKT = 48,
  OPEN_PT_RESOLUTIONUNKNOWNNIVPKT = 49,
  OPEN_PT_RESOLUTIONKNOWNL2GREPKT = 50,
  OPEN_PT_RESOLUTIONKNOWNVXLANPKT = 51,
  OPEN_PT_RESOLUTIONCOUNT = 52
} OPEN_PT_RESOULTION_t;


/** packet trace lookup result enums */
typedef enum _OPEN_pt_lookup_result_type_
{
  OPEN_PT_LOOKUPINVALID = 0,
  OPEN_PT_LOOKUPFIRSTVLANTRANSLATIONHIT = 1,
  OPEN_PT_LOOKUPSECONDVLANTRANSLATIONHIT = 2,
  OPEN_PT_LOOKUPFORWARDINGVLANVALID = 3,
  OPEN_PT_LOOKUPL2SRCHIT = 6,
  OPEN_PT_LOOKUPL2SRCSTATIC = 7,
  OPEN_PT_LOOKUPL2DSTHIT = 8,
  OPEN_PT_LOOKUPL2CACHEHIT = 9,
  OPEN_PT_LOOKUPL3SRCHOSTHIT = 10,
  OPEN_PT_LOOKUPL3DESTHOSTHIT = 11,
  OPEN_PT_LOOKUPL3DESTROUTEHIT = 12,
  OPEN_PT_LOOKUPL2SRCMISS = 13,
  OPEN_PT_LOOKUPDOSATTACK = 14,
  OPEN_PT_LOOKUPIPTUNNELHIT = 15,
  OPEN_PT_LOOKUPMPLSLABEL1HIT = 16,
  OPEN_PT_LOOKUPMPLSLABEL2HIT = 17,
  OPEN_PT_LOOKUPMPLSTERMINATED = 18,
  OPEN_PT_LOOKUPMYSTATIONHIT = 19,
  OPEN_PT_LOOKUPCOUNT = 20
} OPEN_PT_LOOKUP_RESULT_TYPE_t;


/** lookup result set bit map */
typedef struct
{
  OPEN_MASK_BASE_UNIT   value[OPEN_MASK_SIZE(OPEN_PT_LOOKUPCOUNT)]; /** bit map for packet trace lookup result set */
} OPEN_PT_LOOKUP_RESULT_MASK_t;


typedef enum _OPEN_stp_state_
{
  OPEN_DISABLED = 0,
  OPEN_FORWARDING = 1,
  OPEN_DISCRADING = 2,
  OPEN_LEARNING = 3
} OPEN_PT_STP_STATE_t;

/** Trace Profile */
typedef struct  _OPEN_pt_trace_profile_
{
  /** port*/
  int                        port;
  /** Packet Including Checksum*/
  OPEN_PT_PACKET_t          packet;
  /** packet trace lookup status */
  OPEN_PT_LOOKUP_RESULT_MASK_t   lookupResult;
  /** packet trace resolution result */
  OPEN_PT_RESOULTION_t      packetResolution;
  /** packet trace hashing information */
  OPEN_PT_HASHING_INFO_t    hashingInfo;
  /** packet trace ingress stp state */
  OPEN_PT_STP_STATE_t       stpState;
  /** Mask of destination ports */
  OPEN_PORT_MASK_t          destPortMask;

} OPEN_PT_TRACE_PROFILE_t;

typedef struct open_pt_5_tuple_params_s_
{
  unsigned int src_ip;
  unsigned int dst_ip;
  unsigned int protocol;
  unsigned int src_port;
  unsigned int dst_port;
  unsigned int tuple_mask;
}OPEN_PT_5_TUPLE_PARAMS_t;

/* Structure to configure five tuple, port to track droppped packets */  
typedef struct _open_pt_config_pkt_drop_ctrs_s
{
  int id;
  int port;
  OPEN_PT_5_TUPLE_PARAMS_t tuple_params;
  unsigned int duration;
  bool send_samples;
  unsigned int max_samples;
  bool inc_drop_reasons;
} OPEN_PT_CONFIG_PKT_DROP_CTRS_t;

/*********************************************************************
* @brief  Create a 5 tuple profile/rule in HW to capture live traffic.
*
* @param    client_handle         @b{(input)}         OpenAPI connection Handle
* @param    asic                  @b{(input)}         Asic/Unit number 
* @param    requestId             @b{(input)}         Request id of the profile. 
* @param    portList              @b{(input)}         List of ports on which 
*                                                     this rule has to be applied 
* @param    tupleParams           @b{(input)}         5 tuple params 
* @param    policyHandle          @b{(output)}        Handle of the policy
*                                             created.
*
* @retval   BVIEW_STATUS_FAILURE      Failed to create a profile/rule with the requested 
*                                     params. 
*
* @retval   BVIEW_STATUS_SUCCESS      Profile is created in HW
*
* @notes    This API is deprecated.
*
* @open-status-deprecated
* @supported-in-version OpEN API Version: 1.12
* @deprecated-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiPtTrace5TupleProfileCreate(openapiClientHandle_t *client_handle, 
                                               int asic, int requestId,
                                               OPEN_PORT_MASK_t *portList,
                                               OPEN_PT_5_TUPLE_PARAMS_t *tupleParams,
                                               OPEN_TCAM_API_POLICYID_t *policyHandle);

/**************************************************************************************
* @brief  Delete a 5 tuple profile/rule in HW to capture live traffic.
*
* @param    client_handle         @b{(input)}         OpenAPI connection Handle
* @param    asic                  @b{(input)}         Asic/Unit number 
* @param    policyHandle          @b{(input)}         Policy handle
*
* @retval   BVIEW_STATUS_FAILURE      Failed to delete a profile/rule with the requested 
*                                     params. 
*
* @retval   BVIEW_STATUS_SUCCESS      Profile is deleted in HW
*
* @notes    This API is deprecated.
*
* @open-status-deprecated
* @supported-in-version OpEN API Version: 1.12
* @deprecated-in-version OpEN API Version: 1.14
*
* @end
*****************************************************************************************/
open_error_t openapiPtTrace5TupleProfileDelete(openapiClientHandle_t *client_handle, 
                                               int asic, OPEN_TCAM_API_POLICYID_t  *policyHandle);

/*********************************************************************
* @purpose  PT feature trace profile get function
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    port          @b{(input)}         Port number 
* @param    packetData    @b{(input)}         Packet whose trace profile
*                                             is required.
* @param    traceProfile  @b{(output)}        Trace profile of packet.
* @param    time          @b{(output)}        Time.
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiPtTraceProfileGet(openapiClientHandle_t *client_handle,
                                      int asic, int port,
                                      OPEN_PT_PACKET_t *packetData,
                                      OPEN_PT_TRACE_PROFILE_t *traceProfile,
                                      time_t *time);

/*********************************************************************
* @purpose  PT feature trace profile get function
*
* @param    client_handle    @b{(input)}         OpenAPI connection Handle.
* @param    asic            @b{(input)}         Asic/Unit number.
* @param    dropCtrsProfile @b{(input)}         Drop counters profile.
* @param    ruleReqInfo     @b{(output)}        Rule request information.
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @comments From OpEN API Version 1.15 onwards, setting resourceTrackingFlag for
*           a given client handle will ensure that the TCAM policy (resource)
*           created in hardware gets freed when the client ceases to exist.
*           openapiClientResourceTrackingSet OpEN API can be used to
*           enable or disable resourceTrackingFlag. 
*
* @supported-in-version OpEN API Version: 1.16
*
* @end
*********************************************************************/
open_error_t openapiPtPktDropCtrsProfileCreate(openapiClientHandle_t *client_handle,
                                               int asic,
                                               OPEN_PT_CONFIG_PKT_DROP_CTRS_t *dropCtrsProfile,
                                               OPEN_PT_DROP_CTRS_REQ_INFO_t *ruleReqInfo);

/*********************************************************************
* @purpose  PT feature packet drop reason get function
*
* @param    client_handle @b{(input)}         OpenAPI connection Handle
* @param    asic         @b{(input)}         Asic/Unit number 
* @param    port         @b{(input)}         Port number 
* @param    packet       @b{(input)}         Packet whose drop reason info
*                                            is required.
* @param    dropInfo     @b{(output)}        Drop reason info of packet.
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.16
*
* @end
*********************************************************************/
open_error_t openapiPtPktDropReasonInfoGet(openapiClientHandle_t *client_handle,
                                           int asic,
                                           int port,
                                           OPEN_PT_PACKET_t *packet,
                                           OPEN_PT_DROP_REASON_DATA_t *dropInfo);
#endif
