/**********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_ip6map.h
*
* @purpose   This code implements the OpEN IP6MAP API
*
* @component OPEN
*
* @create    01/10/2023
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_IP6MAP_H_INCLUDED
#define OPENAPI_IP6MAP_H_INCLUDED

#include "openapi_common.h"
#include "openapi_routing.h"
#include "openapi_mpls.h"
#include "openapi_routing_config.h"

#define OPEN_RTR6_MAX_INTF_ADDRS         4
#define OPEN_IP6_ND_DAD_ATTEMPTS_MAX     600
#define OPEN_IP6_NDNS_INTERVAL_MIN       1000
#define OPEN_IP6_NDNS_INTERVAL_MAX       4294967295U
#define OPEN_IP6_ND_REACHABLE_MAX        3600000UL

/**< IPv6 ICMP interface statistics */
typedef struct
{
  uint64_t  ipv6IfIcmpInMsgs;
  uint64_t  ipv6IfIcmpInErrors;
  uint64_t  ipv6IfIcmpInDestUnreachs;
  uint64_t  ipv6IfIcmpInAdminProhibs;
  uint64_t  ipv6IfIcmpInTimeExcds;
  uint64_t  ipv6IfIcmpInParmProblems;
  uint64_t  ipv6IfIcmpInPktTooBigs;
  uint64_t  ipv6IfIcmpInEchos;
  uint64_t  ipv6IfIcmpInEchoReplies;
  uint64_t  ipv6IfIcmpInRouterSolicits;
  uint64_t  ipv6IfIcmpInRouterAdvertisements;
  uint64_t  ipv6IfIcmpInNeighborSolicits;
  uint64_t  ipv6IfIcmpInNeighborAdvertisements;
  uint64_t  ipv6IfIcmpInRedirects;
  uint64_t  ipv6IfIcmpInGroupMembQueries;
  uint64_t  ipv6IfIcmpInGroupMembResponses;
  uint64_t  ipv6IfIcmpInGroupMembReductions;

  uint64_t  ipv6IfIcmpOutMsgs;
  uint64_t  ipv6IfIcmpOutErrors;
  uint64_t  ipv6IfIcmpOutDestUnreachs;
  uint64_t  ipv6IfIcmpOutAdminProhibs;
  uint64_t  ipv6IfIcmpOutTimeExcds;
  uint64_t  ipv6IfIcmpOutParmProblems;
  uint64_t  ipv6IfIcmpOutPktTooBigs;
  uint64_t  ipv6IfIcmpOutEchos;
  uint64_t  ipv6IfIcmpOutEchoReplies;
  uint64_t  ipv6IfIcmpOutRouterSolicits;
  uint64_t  ipv6IfIcmpOutRouterAdvertisements;
  uint64_t  ipv6IfIcmpOutNeighborSolicits;
  uint64_t  ipv6IfIcmpOutNeighborAdvertisements;
  uint64_t  ipv6IfIcmpOutRedirects;
  uint64_t  ipv6IfIcmpOutGroupMembQueries;
  uint64_t  ipv6IfIcmpOutGroupMembResponses;
  uint64_t  ipv6IfIcmpOutGroupMembReductions;

  uint64_t  ipv6IfIcmpDupAddrDetects;

  uint64_t  ipv6IfIcmpInCsumErrors;
  uint64_t  ipv6IfIcmpInMLDv2Reports;
  uint64_t  ipv6IfIcmpOutMLDv2Reports;
  uint64_t  ipv6IfIcmpInType134;
  uint64_t  ipv6IfIcmpInType135;
  uint64_t  ipv6IfIcmpInType136;
  uint64_t  ipv6IfIcmpOutType1;
  uint64_t  ipv6IfIcmpOutType133;
  uint64_t  ipv6IfIcmpOutType135;
  uint64_t  ipv6IfIcmpOutType136;
  uint64_t  ipv6IfIcmpOutType143;

} openapiIpstkIpv6IfIcmpStats_t;

/**< Interface IPv6 statistics */
typedef struct
{
  uint64_t  ipv6IfStatsInReceives;
  uint64_t  ipv6IfStatsInMcastPkts;
  uint64_t  ipv6IfStatsInDelivers;
  uint64_t  ipv6IfStatsInUnknownProtos;
  uint64_t  ipv6IfStatsInAddrErrors;
  uint64_t  ipv6IfStatsInHdrErrors;
  uint64_t  ipv6IfStatsInNoRoutes;
  uint64_t  ipv6IfStatsInTooBigErrors;
  uint64_t  ipv6IfStatsInTruncatedPkts;
  uint64_t  ipv6IfStatsReasmReqds;
  uint64_t  ipv6IfStatsReasmOKs;
  uint64_t  ipv6IfStatsReasmFails;
  uint64_t  ipv6IfStatsInDiscards;

  uint64_t  ipv6IfStatsOutRequests;
  uint64_t  ipv6IfStatsOutMcastPkts;
  uint64_t  ipv6IfStatsOutForwDatagrams;
  uint64_t  ipv6IfStatsOutFragCreates;
  uint64_t  ipv6IfStatsOutFragOKs;
  uint64_t  ipv6IfStatsOutFragFails;
  uint64_t  ipv6IfStatsOutDiscards;

  uint64_t  ipv6IfStatsOutNoRoutes;
  uint64_t  ipv6IfStatsReasmTimeout;
  uint64_t  ipv6IfStatsInOctets;
  uint64_t  ipv6IfStatsOutOctets;
  uint64_t  ipv6IfStatsInMcastOctets;
  uint64_t  ipv6IfStatsOutMcastOctets;
  uint64_t  ipv6IfStatsInBcastOctets;
  uint64_t  ipv6IfStatsOutBcastOctets;
  uint64_t  ipv6IfStatsInNoECTPkts;
  uint64_t  ipv6IfStatsInECT1Pkts;
  uint64_t  ipv6IfStatsInECT0Pkts;
  uint64_t  ipv6IfStatsInCEPkts;
} openapiIpv6IfStats_t;


/**< ECMP path structure */
typedef struct openapiEqualCostPath6_s 
{
  open_in6_addr_t     ip6Addr;        /**< Next hop address */
  uint32_t            intIfNum;       /**< Next hop interface */
  uint32_t            flags;          /**< Route entry flags */
  OPEN_MPLS_LABELS_t  mplsLabelStack; /**< The MPLS Label stack for this route */
  uint32_t            updateTime;     /**< Last update timestamp for the specified ECMP route */ 
} openapiEqualCostPath6_t;

typedef struct openapiECMPRoute6_s
{
  openapiEqualCostPath6_t   equalCostPath[OPENR_PLAT_MAX_ECMP_NEXTHOPS]; /**< equalCostPath array */
  uint32_t                  numOfRoutes; /**< number of routes in the equalCostPath array */ 
} openapiECMPRoute6_t;

/**< IPv6 Route entry */
typedef struct openapiRoute6Entry_s
{
  open_in6_addr_t       ip6Addr;      /**< Route prefix address */
  int32_t               ip6PrefixLen; /**< Route prefix lenght */
  uint32_t              vrfId;        /**< VRF identifier */
  uint32_t              protocol;     /**< Protocol via which the route(s) are learned */     
  uint32_t              metric;       /**< cost associated with this route */
  openapiECMPRoute6_t   ecmpRoutes;   /**< ECMP paths */
  uint32_t              flags;        /**< Route entry flags */
  uint8_t               pref;         /**< Route preference (also called administrative distance). [0 to 255] */
  uint32_t              updateTime;   /**< Specifies the last time the route was updated (in hours:minutes:seconds) */
  uint32_t              tableId;      /**< Table ID for the VRF, valid if vrfId != 0 */
  OPEN_BOOL_t           hwFailure;    /**< Hardware failure status */
  uint32_t              seqNum;       /**< Sequence number for successive route addition */
  OPEN_BOOL_t           ip_anycast;   /**< Is route IP Anycast */
} openapiRoute6Entry_t;

/**< Structure for list of IPv6 addresses configured on interface */
typedef struct openapi_ip6_cfg_addr_list_s
{
  open_in6_addr_t ipAddrList[OPEN_RTR6_MAX_INTF_ADDRS];
  uint32_t        pLen[OPEN_RTR6_MAX_INTF_ADDRS];
  uint32_t        pEuiFlag[OPEN_RTR6_MAX_INTF_ADDRS];
} openapi_ip6_cfg_addr_list_t;

/**< Address mode flags */
#define OPEN_RTR6_ADDR_LINK_LOCAL_ENA      0x01  /**< link local addr even if no global addr */
#define OPEN_RTR6_ADDR_EUI64               0x02  /**< use eui64 interface ID */
#define OPEN_RTR6_ADDR_AUTOCONFIG_ENA      0x08  /**< auto config address */
#define OPEN_RTR6_ADDR_DHCP_ENA            0x10  /**< dhcp address */

/**< Interface IPv6 address states */
typedef enum openIp6IntfAddrStates {
  OPEN_IPV6_INTFADDR_INACTIVE = 1,   /**< inactive */
  OPEN_IPV6_INTFADDR_TENTATIVE,      /**< tentative */
  OPEN_IPV6_INTFADDR_ACTIVE,         /**< active */
  OPEN_IPV6_INTFADDR_AUTOCONFIG,     /**< autoconfig */
  OPEN_IPV6_INTFADDR_DHCP            /**< dhcp */
} OPEN_IPV6_INTFADDR_STATES;

/**< Structure of IPv6 address enty */
typedef struct openapiIpv6AddrEntry_s
{
    uint32_t                   ipv6IfIndex;          /**< Internal interface number */
    open_in6_addr_t            ipv6Address;          /**< ipv6 address */
    uint32_t                   ipv6AddrPfxLength;    /**< prefix length of the address */
    uint32_t                   ipv6AddrType;         /**< type of the address */
    uint32_t                   ipv6AddrAnycastFlag;  /**< anycast flag */
    uint32_t                   ipv6AddrEui64Flag;    /**< eui64 flag */
    OPEN_IPV6_INTFADDR_STATES  ipv6AddrStatus;       /**< address status */
} openapiIpv6AddrEntry_t;

/**< Structure for IPv6 prefix enty*/
typedef struct openapiIpv6AddrPrefixEntry_s
{
    uint32_t          ipv6IfIndex;                          /**< Internal interface number */
    open_in6_addr_t   ipv6AddrPrefix;                       /**< ipv6 address prefix */
    uint32_t          ipv6AddrPrefixLength;                 /**< prefix length */
    uint32_t          ipv6AddrPrefixOnLinkFlag;             /**< prefix on link flag */
    uint32_t          ipv6AddrPrefixAutonomousFlag;         /**< prefix autonomous flag */
    uint32_t          ipv6AddrPrefixAdvPreferredLifetime;   /**< preferred life time */
    uint32_t          ipv6AddrPrefixAdvValidLifetime;       /**< valid life time */

} openapiIpv6AddrPrefixEntry_t;

/**< Structure for IPv6 interface address, along with its state */
typedef struct open_ipv6IntfAddr_s
{ 
    open_in6_addr_t             ipv6Addr;
    uint32_t                    ipv6PrefixLen;
    OPEN_IPV6_INTFADDR_STATES   ipv6AddrState;
} open_ipv6IntfAddr_t;

/**< Structure for max list of IPv6 interface address structure */
typedef struct openIpv6IntfAddrList_s
{
    open_ipv6IntfAddr_t   ipv6AddrList[OPEN_RTR6_MAX_INTF_ADDRS + 1]; 
} openIpv6IntfAddrList_t;

/**< Structure for NULL OID */
typedef struct openNullOid_s
{   
    uint32_t         len; 
    unsigned char    oid[2];
} openNullOid_t; 

/**< Structure for IPv6 Route entry */
typedef struct openIpv6RouteEntry_s
{   
    open_in6_addr_t  ipv6RouteDest;
    uint32_t         ipv6RoutePfxLength;
    uint32_t         ipv6RouteIndex;
    uint32_t         ipv6RouteIfIndex;             /**< Internal interface number */
    open_in6_addr_t  ipv6RouteNextHop;
    uint32_t         ipv6RouteType;
    uint32_t         ipv6RouteProtocol;
    uint32_t         ipv6RoutePolicy;
    uint32_t         ipv6RouteAge;
    uint32_t         ipv6RouteNextHopRDI;
    uint32_t         ipv6Metric;
    uint32_t         ipv6RouteWeight;
    openNullOid_t    ipv6RouteInfo;
    uint32_t         ipv6RouteValid;
    uint32_t         ipv6RouteProtocolType;
} openIpv6RouteEntry_t;


/**< Neighbor state definitions as per RFC 2465 */
typedef enum
{
   OPEN_IP6_NBR_STATE_REACHABLE = 1,  /**< Reachable */
   OPEN_IP6_NBR_STATE_STALE,          /**< Stale */
   OPEN_IP6_NBR_STATE_DELAY,          /**< Delay */
   OPEN_IP6_NBR_STATE_PROBE,          /**< Probe */
   OPEN_IP6_NBR_STATE_INCOMPLETE,     /**< Incomplete */
   OPEN_IP6_NBR_STATE_UNKNOWN,        /**< Unknown */
   OPEN_IP6_NBR_STATE_UNREACHABLE,    /**< Unreachable */
   OPEN_IP6_NBR_STATE_PERMANENT       /**< Permanent */
} OPEN_IP6_NBR_STATE_t;

/**< Neighbor type definitions as per RFC 2465 */
typedef enum
{
   OPEN_IP6_NBR_TYPE_OTHER = 1, /**< Other */
   OPEN_IP6_NBR_TYPE_DYNAMIC,   /**< Dynamic */
   OPEN_IP6_NBR_TYPE_STATIC,    /**< Static */
   OPEN_IP6_NBR_TYPE_LOCAL      /**< Local */
} OPEN_IP6_NBR_TYPE_t;

/*****************************************************************************
 * @purpose   Get the IPv6 administrative mode of the router for given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     mode           @b{(output)} OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     This routine returns the user-configured state of the router.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrRtrAdminModeGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the IPv6 administrative mode of the router for given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     mode           @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrRtrAdminModeSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the dynamic renewal mode of IPv6 neighbors in a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     mode           @b{(output)} OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrNbrsDynRenewGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the dynamic renewal mode of IPv6 neighbors in a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     mode           @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrNbrsDynRenewSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the NUD backoff-multiple value set for calculation of timeouts for NS transmissions during NUD for a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     val            @b{(output)} The backoff multiple value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrNudBackoffMultipleGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t *val);

/*****************************************************************************
 * @purpose   Set the NUD backoff-multiple value for calculation of timeouts for NS transmissions during NUD for a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     val            @b{(input)}  The backoff multiple value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrNudBackoffMultipleSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t val);

/*****************************************************************************
 * @purpose   Get the maximum unicast neighbor solicitations sent during NUD for a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     val            @b{(output)} The maximum unicast solicits value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrNudMaxUnicastSolicitsGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t *val);

/*****************************************************************************
 * @purpose   Set the maximum unicast neighbor solicitations sent during NUD for a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     val            @b{(input)}  The maximum unicast solicits value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrNudMaxUnicastSolicitsSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t val);

/*****************************************************************************
 * @purpose   Get the maximum multicast neighbor solicitations sent during NUD for a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     val            @b{(output)} The maximum multicast solicits value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrNudMaxMcastSolicitsGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t *val);

/*****************************************************************************
 * @purpose   Set the maximum multicast neighbor solicitations sent during NUD for a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     val            @b{(input)}  The maximum multicast solicits value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrNudMaxMcastSolicitsSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t val);

/*****************************************************************************
 * @purpose   Get the rate limit value set for the unresolved IPv6 data packets coming to CPU.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     val            @b{(output)} The rate-limt value in pps(packet-per-second)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6UnresolvedRateLimitGet(openapiClientHandle_t *client_handle, uint32_t *val);

/*****************************************************************************
 * @purpose   Set the rate limit value for the unresolved IPv6 data packets coming to CPU.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     val            @b{(input)}  The rate-limt value in pps(packet-per-second)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6UnresolvedRateLimitSet(openapiClientHandle_t *client_handle, uint32_t val);

/*****************************************************************************
 * @purpose   Get the default IPv6 router TTL value for a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     val            @b{(output)} The TTL value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     If 0 is returned for 'val' then it means TTL is not configured.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrRtrDefaultTTLGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t *val);

/*****************************************************************************
 * @purpose   Set the default IPv6 router TTL value for a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     val            @b{(input)}  The TTL value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     This value is sent in router advertisements
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrRtrDefaultTTLSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t val);

/*****************************************************************************
 * @purpose   Get the administrative state of a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     mode           @b{(output)} OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Get the operational state of a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     mode           @b{(output)} OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfOperModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the administrative state of a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     mode           @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the IPv6 AutoState mode of a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     mode           @b{(output)} OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfAutoStateModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the IPv6 AutoState mode of a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     mode           @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfAutoStateModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Determine if IPv6 interface exists.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 *
 * @returns   OPEN_TRUE            If given IPv6 interface exists
 * @returns   OPEN_FALSE           If given IPv6 interface does not exist
 * @returns   OPEN_E_UNAVAIL       Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
OPEN_BOOL_t openapiIp6IntfExists(openapiClientHandle_t *client_handle, uint32_t ifNum);

/*****************************************************************************
 * @purpose   Determine if the interface is valid for participation in the IPv6 component.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 *
 * @returns   OPEN_TRUE            If given interface is valid for IPv6
 * @returns   OPEN_FALSE           If given interface is not valid for IPv6
 * @returns   OPEN_E_UNAVAIL       Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
OPEN_BOOL_t openapiIp6MapIntfIsValid(openapiClientHandle_t *client_handle, uint32_t ifNum);

/*****************************************************************************
 * @purpose   Determine if the interface is valid to be configured with an IPv6 neighbor.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 *
 * @returns   OPEN_TRUE            If given interface is valid for IPv6 neighbor
 * @returns   OPEN_FALSE           If given interface is not valid for IPv6 neighbor
 * @returns   OPEN_E_UNAVAIL       Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
OPEN_BOOL_t openapiIp6MapNbrIntfIsValid(openapiClientHandle_t *client_handle, uint32_t ifNum);

/*****************************************************************************
 * @purpose   Get the IPv6 MTU of a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     val            @b{(output)} The IPv6 MTU value in bytes
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfIpv6MtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *val);

/*****************************************************************************
 * @purpose   Get the IPv6 effective MTU of a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     val            @b{(output)} The IPv6 effective MTU value in bytes
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     Link MTU is returned if IPv6 MTU is not configured.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfIpv6EffectiveMtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *val);

/*****************************************************************************
 * @purpose   Get the maximum IPv6 MTU that may be set on an interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     val            @b{(output)} The IPv6 maximum MTU value in bytes
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     On port based routing interfaces, the IP MTU may be set
 *            as large as the link MTU. On VLAN routing interfaces,
 *            the IP MTU may be set as large as the largest jumbo
 *            frame will allow.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfMaxIpv6MtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *val);

/*****************************************************************************
 * @purpose   Set the IPv6 MTU of a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     val            @b{(input)}  The IPv6 MTU value in bytes
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfIpv6MtuSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t val);

/*****************************************************************************
 * @purpose   Get the interface bandwidth of a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     val            @b{(output)} The bandwidth value in bytes
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfBandwidthGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *val);

/*****************************************************************************
 * @purpose   Get the first valid interface for participation in IPv6 component.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(output)} Internal Interface Number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6MapIntfValidFirstGet(openapiClientHandle_t *client_handle, uint32_t *ifNum);

/*****************************************************************************
 * @purpose   Get the next valid interface for participation in IPv6 component.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     prevIfNum      @b{(input)}  Previous Internal Interface Number
 * @param     ifNum          @b{(output)} Internal Interface Number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6MapIntfValidNextGet(openapiClientHandle_t *client_handle, uint32_t prevIfNum, uint32_t *ifNum);

/*****************************************************************************
 * @purpose   Get IPv6 statistics for the given interface.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API.
 * @param     intIfNum          @b{(input)}  Internal Interface number.
 * @param     ipv6IfStats       @b{(output)} IPv6 Interface Stats.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpv6IfStatsGet (openapiClientHandle_t *client_handle,
                                    uint32_t intIfNum,
                                    openapiIpv6IfStats_t *ipv6IfStats);

/*****************************************************************************
 * @purpose   Get IPv6 ICMP statistics for the given interface.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API.
 * @param     intIfNum          @b{(input)}  Internal Interface number.
 * @param     ipv6IfIcmpStats   @b{(output)} IPv6 ICMP Interface Stats.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpv6IfIcmpStatsGet (openapiClientHandle_t *client_handle,
                                        uint32_t intIfNum,
                                        openapiIpstkIpv6IfIcmpStats_t *ipv6IfIcmpStats);

/*****************************************************************************
 * @purpose   Clear IPv6 statistics for the given interface.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API.
 * @param     intIfNum          @b{(input)}  Internal Interface number.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpv6IfStatsClear (openapiClientHandle_t *client_handle,
                                      uint32_t intIfNum);

/*****************************************************************************
 * @purpose   Get total of each IPv6 statistic on all interfaces.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API.
 * @param     ipv6IfStats       @b{(output)} IPv6 Interface Stats.
 * @param     ipv6IfIcmpStats   @b{(output)} IPv6 ICMP Interface Stats.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpv6TotalStatsGet (openapiClientHandle_t *client_handle,
                                       openapiIpv6IfStats_t *ipv6IfStats,
                                       openapiIpstkIpv6IfIcmpStats_t *ipv6IfIcmpStats);

/*****************************************************************************
 * @purpose   Clear IPv6 statistics on all interfaces.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpv6TotalStatsClear (openapiClientHandle_t *client_handle);

/*****************************************************************************
 * @purpose   Get the next route entry for the specified route destination.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API.
 * @param     vrfName        @b{(input)}  VRF Name.
 * @param     ipv6Route      @b{(inout)}  IPv6 route entry.
 * @param     bestRouteOnly  @b{(input)}  Get best route only.
 *                                         OPEN_TRUE or OPEN_FALSE.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     To get first entry ip6Addr, ip6PrefixLen, protocol and
 *             pref in ipv6Route pointer should be set with 0.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpv6RouteEntryGetNext (openapiClientHandle_t *client_handle,
                                           open_buffdesc *vrfName,
                                           openapiRoute6Entry_t *ipv6Route,
                                           OPEN_BOOL_t bestRouteOnly);

/*****************************************************************************
 * @purpose   Get the number of routes in the routing table.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API.
 * @param     vrfName        @b{(input)}  VRF Name.
 * @param     count          @b{(output)} Route count.
 * @param     bestRouteOnly  @b{(input)}  Get best route only.
 *                                         OPEN_TRUE or OPEN_FALSE.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpv6RouteCountGet (openapiClientHandle_t *client_handle,
                                       open_buffdesc *vrfName,
                                       uint32_t *count,
                                       OPEN_BOOL_t bestRouteOnly);

/*****************************************************************************
 * @purpose   Get the configured IPv6 addresses and modes on the interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API.
 * @param     intIfNum       @b{(input)}  Internal interface number.
 * @param     numAddr        @b{(inout)}  Number of IPv6 addresses to be
 *                                         fetched or configured on interface.
 * @param     ipv6AddrList   @b{(output)} List of IPv6 addresses configured
 *                                         on interface.
 * @param     ipv6Mode       @b{(output)} IPv6(link local) mode on interface.
 * @param     autoConfigMode @b{(output)} Auto config mode on interface.
 * @param     dhcpMode       @b{(output)} DHCP mode on interface.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrCfgIntfAddressesGet (openapiClientHandle_t *client_handle,
                                               uint32_t intIfNum,
                                               uint32_t *numAddr,
                                               openapi_ip6_cfg_addr_list_t *ipv6AddrList,
                                               OPEN_CONTROL_t *ipv6Mode,
                                               OPEN_CONTROL_t *autoConfigMode,
                                               OPEN_CONTROL_t *dhcpMode);

/*****************************************************************************
 * @purpose   Get the number of neighbor solicits configured to be sent for
 *             duplicate address detection (DAD) on an interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API.
 * @param     intIfNum       @b{(input)}  Internal interface number.
 * @param     val            @b{(output)} Number of neighbor solicits to be
 *                                         sent for DAD.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrDadTransmitsGet (openapiClientHandle_t *client_handle,
                                           uint32_t intIfNum, uint32_t *val);

/*****************************************************************************
 * @purpose   Set the number of neighbor solicits to be sent for
 *             duplicate address detection (DAD) on an interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API.
 * @param     intIfNum       @b{(input)}  Internal interface number.
 * @param     val            @b{(input)}  Number of neighbor solicits to be
 *                                         sent for DAD.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrDadTransmitsSet (openapiClientHandle_t *client_handle,
                                           uint32_t intIfNum, uint32_t val);

/*****************************************************************************
 * @purpose   Get the interval between IPv6 neighbor solicitation
 *             retransmissions on an interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API.
 * @param     intIfNum       @b{(input)}  Internal interface number.
 * @param     val            @b{(output)} Interval between IPv6 neighbor
 *                                         solicitation retransmissions
 *                                         in milliseconds.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrNdNsIntervalGet (openapiClientHandle_t *client_handle,
                                           uint32_t intIfNum, uint32_t *val);

/*****************************************************************************
 * @purpose   Set the interval between IPv6 neighbor solicitation
 *             retransmissions on an interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API.
 * @param     intIfNum       @b{(input)}  Internal interface number.
 * @param     val            @b{(input)}  Interval between IPv6 neighbor
 *                                         solicitation retransmissions
 *                                         in milliseconds.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrNdNsIntervalSet (openapiClientHandle_t *client_handle,
                                           uint32_t intIfNum, uint32_t val);

/*****************************************************************************
 * @purpose   Get the amount of time that a remote IPv6 node is reachable.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API.
 * @param     intIfNum       @b{(input)}  Internal interface number.
 * @param     val            @b{(output)} Reachable time in milliseconds.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrNdReachableTimeGet (openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, uint32_t *val);

/*****************************************************************************
 * @purpose   Set the amount of time that a remote IPv6 node is reachable.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API.
 * @param     intIfNum       @b{(input)}  Internal interface number.
 * @param     val            @b{(input)}  Reachable time in milliseconds.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrNdReachableTimeSet (openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, uint32_t val);

/*****************************************************************************
 * @purpose   Get the configured mode for ICMPv6 unreachable messages.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API.
 * @param     intIfNum       @b{(input)}  Internal interface number.
 * @param     mode           @b{(output)} Enable or Disable
 *                                         ICMPv6 unreachable messages.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfICMPUnreachablesModeGet (openapiClientHandle_t *client_handle,
                                                       uint32_t intIfNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the configured mode for ICMPv6 unreachable messages.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API.
 * @param     intIfNum       @b{(input)}  Internal interface number.
 * @param     mode           @b{(input)}  Enable or Disable ICMPv6 unreachable messages.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfICMPUnreachablesModeSet (openapiClientHandle_t *client_handle,
                                                       uint32_t intIfNum, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the IPv6 address details.
 *
 * @param     client_handle        @b{(input)}  client handle from registration API.
 * @param     matchType            @b{(input)}  Match type. (1 - EXACT and 2 - NEXT).
 * @param     ipv6AddrEntry        @b{(inout)}  IPv6 address entry.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpv6AddrEntryGet (openapiClientHandle_t *client_handle,
                                      uint32_t matchType,
                                      openapiIpv6AddrEntry_t *ipv6AddrEntry);

/*****************************************************************************
 * @purpose   Get the IPv6 address details of the next entry.
 *
 * @param     client_handle        @b{(input)}  client handle from registration API.
 * @param     matchType            @b{(input)}  Match type. (1 - EXACT and 2 - NEXT).
 * @param     ipv6AddrEntry        @b{(inout)}  IPv6 address entry.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpv6AddrEntryGetNext (openapiClientHandle_t *client_handle,
                                          uint32_t matchType,
                                          openapiIpv6AddrEntry_t *ipv6AddrEntry);

/*****************************************************************************
 * @purpose   Get the IPv6 address prefix details.
 *
 * @param     client_handle        @b{(input)}  client handle from registration API.
 * @param     matchType            @b{(input)}  Match type. (1 - EXACT and 2 - NEXT).
 * @param     ipv6AddrPrefixEntry  @b{(inout)}  IPv6 address entry.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpv6AddrPrefixEntryGet (openapiClientHandle_t *client_handle,
                                            uint32_t matchType,
                                            openapiIpv6AddrPrefixEntry_t *ipv6AddrPrefixEntry);

/*****************************************************************************
 * @purpose   Get the IPv6 addresses of the interface.
 *
 * @param     client_handle        @b{(input)}  client handle from registration API.
 * @param     intIfNum             @b{(input)}  Internal interface number.
 * @param     numAddr              @b{(inout)}  Number of addresses to be fetched or actual available on interface.
 * @param     ipv6AddrList         @b{(output)} List of IPv6 address on interface.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfAddressesGet (openapiClientHandle_t *client_handle,
                                            uint32_t intIfNum,
                                            uint32_t *numAddr,
                                            openIpv6IntfAddrList_t *ipv6AddrList);

/*****************************************************************************
 * @purpose   Get the IPv6 address of the interface.
 *
 * @param     client_handle        @b{(input)}  client handle from registration API.
 * @param     intIfNum             @b{(input)}  Internal interface number.
 * @param     inetAddr             @b{(output)} Address on the interface.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfAddressGet (openapiClientHandle_t *client_handle,
                                          uint32_t intIfNum,
                                          open_inet_addr_t *inetAddr);

/*****************************************************************************
 * @purpose   Get the IPv6 neighbor entries.
 *
 * @param     client_handle        @b{(input)}  client handle from registration API.
 * @param     vrfName              @b{(input)}  VRF name.
 * @param     ipv6Addr             @b{(inout)}  IPv6 address of the neighbor.
 * @param     macAddr              @b{(output)} Mac address of the neighbor.
 * @param     intIfNum             @b{(inout)}  Internal interface number.
 * @param     ipv6Type             @b{(output)} Type of neighbor.
 * @param     nbrState             @b{(output)} State of neighbor.
 * @param     lastUpdate           @b{(output)} Time since last update to the neighbor state.
 * @param     isRouter             @b{(output)} True is neighbor is router or else False.
 * @param     ipv6ScopeId          @b{(output)} IPv6 Scope id of the neighbor address incase of link local.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Interface should be passed to get neighbor for specific interface,
 *             it can be left 0 otherwise.
 *            Pass the returned ip6Addr and ip6ScopeId to get the next entry.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpv6NbrTblEntryGet (openapiClientHandle_t *client_handle,
                                        open_buffdesc *vrfName,
                                        open_inet_addr_t *ipv6Addr,
                                        open_buffdesc *macAddr,
                                        uint32_t *intIfNum,
                                        OPEN_IP6_NBR_TYPE_t *ipv6Type,
                                        OPEN_IP6_NBR_STATE_t *nbrState,
                                        uint32_t *lastUpdate,
                                        OPEN_BOOL_t *isRouter,
                                        uint32_t *ipv6ScopeId);

/*****************************************************************************
 * @purpose   Flush the IPv6 neighbor entries.
 *
 * @param     client_handle        @b{(input)}  client handle from registration API.
 * @param     vrfName              @b{(input)}  VRF name.
 * @param     intIfNum             @b{(input)}  Internal interface number
 * @param     ipv6Addr             @b{(input)}  Address on the neighbor entry.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Flushes the entire all the neighbors on the interface
 *             if ipv6Addr is NULL pointer.
 *            Flushes the entire table if intIfNum is 0 and
 *             ipv6Addr is a NULL pointer.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpv6NdpFlush (openapiClientHandle_t *client_handle,
                                  open_buffdesc *vrfName,
                                  uint32_t intIfNum,
                                  open_inet_addr_t *ipv6Addr);

/*****************************************************************************
 * @purpose   Get the next static IPv6 route from the routing table.
 *
 * @param     client_handle        @b{(input)}  client handle from registration API.
 * @param     routeEntry           @b{(inout)}  Next route entry.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiSnmpIpv6StaticRouteGetNext (openapiClientHandle_t *client_handle,
                                                openIpv6RouteEntry_t *routeEntry);


#endif /* OPENAPI_IP6MAP_H_INCLUDED */
