/*****************************************************************************
*
*  Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_ip_route_table.h
*
* @purpose   IP Route Table support
*
* @component OpEN
*
* @create    05/26/2016
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_IP_ROUTE_TABLE_H_INCLUDED
#define OPENAPI_IP_ROUTE_TABLE_H_INCLUDED

#include "openapi_common.h"

#define OPEN_API_ROUTE_PREF_MIN               0                /**< Maximum value of the route preference */
#define OPEN_API_ROUTE_PREF_MAX               255              /**< Maximum value of the route preference */
#define OPEN_API_ROUTE_NEXT_HOP_INTF_BUF_MIN  ((16 * 60) + 1)  /**< Minimum size of the buffer for the nex hop interfaces list */
#define OPEN_API_ROUTE_NEXT_HOP_ADDR_BUF_MIN  ((16 * 20) + 1)  /**< Minimum size of the buffer for the nex hop IP addresses list */

typedef enum
{
  OPEN_RTO_RESERVED              = 0,  /**< Not a valid protocol */
  OPEN_RTO_LOCAL                 = 1,  /**< Connected */
  OPEN_RTO_KERNEL                = 2,  /**< Kernel */
  OPEN_RTO_STATIC                = 3,  /**< Static */
  OPEN_RTO_MPLS                  = 4,  /**< MPLS */
  OPEN_RTO_OSPF_INTRA_AREA       = 5,  /**< OSPF Intra Area */
  OPEN_RTO_OSPF_INTER_AREA       = 6,  /**< OSPF Inter Area */
  OPEN_RTO_OSPF_TYPE1_EXT        = 7,  /**< OSPF External Type 1 */
  OPEN_RTO_OSPF_TYPE2_EXT        = 8,  /**< OSPF External Type 2 */
  OPEN_RTO_OSPF_NSSA_TYPE1_EXT   = 9,  /**< OSPF NSSA External Type 1 */
  OPEN_RTO_OSPF_NSSA_TYPE2_EXT   = 10, /**< OSPF NSSA External Type 2 */
  OPEN_RTO_OSPFV3_INTRA_AREA     = 11, /**< OSPFv3 Intra Area */
  OPEN_RTO_OSPFV3_INTER_AREA     = 12, /**< OSPFv3 Inter Area */
  OPEN_RTO_OSPFV3_TYPE1_EXT      = 13, /**< OSPFv3 External Type 1 */
  OPEN_RTO_OSPFV3_TYPE2_EXT      = 14, /**< OSPFv3 External Type 2 */
  OPEN_RTO_OSPFV3_NSSA_TYPE1_EXT = 15, /**< OSPFv3 NSSA External Type 1 */
  OPEN_RTO_OSPFV3_NSSA_TYPE2_EXT = 16, /**< OSPFv3 NSSA External Type 2 */
  OPEN_RTO_RIP                   = 17, /**< RIP */
  OPEN_RTO_DEFAULT               = 18, /**< Default */
  OPEN_RTO_IBGP                  = 19, /**< IBGP */
  OPEN_RTO_EBGP                  = 20, /**< EBGP */
  OPEN_RTO_LBGP                  = 21, /**< Local BGP */
  OPEN_RTO_EXTERNAL              = 22, /**< External */
  OPEN_RTO_UNNUMBERED            = 23, /**< Unnumbered */
  OPEN_RTO_NET_PROTOTYPE         = 24, /**< NetP rototype */
  OPEN_RTO_LAST_PROTOCOL         = 25  /**< Always keep this last */
} OPEN_RTO_PROTOCOL_INDICES_t;

typedef enum
{
  OPEN_RTO6_LOCAL = 1,                  /**< Local route */
  OPEN_RTO6_KERNEL = 2,                 /**< Kernel route */
  OPEN_RTO6_STATIC = 3,                 /**< Static route */
  OPEN_RTO6_MPLS = 4,                   /**< MPLS route */
  OPEN_RTO6_OSPFV3_INTRA_AREA = 5,      /**< OSPFV3 intra area route */
  OPEN_RTO6_OSPFV3_INTER_AREA = 6,      /**< OSPFV3 inter area route */
  OPEN_RTO6_OSPFV3_TYPE1_EXT = 7,       /**< OSPFV3 type 1 ext route */
  OPEN_RTO6_OSPFV3_TYPE2_EXT = 8,       /**< OSPFV3 type 2 ext route */
  OPEN_RTO6_OSPFV3_NSSA_TYPE1_EXT = 9,  /**< OSPFV3 nssa type 1 ext route */
  OPEN_RTO6_OSPFV3_NSSA_TYPE2_EXT = 10, /**< OSPFV3 nssa type 2 ext route */
  OPEN_RTO6_RIP = 11,                   /**< RIP route */
  OPEN_RTO6_DEFAULT = 12,               /**< Default route */
  OPEN_RTO6_IBGP = 13,                  /**< IBGP route */
  OPEN_RTO6_EBGP = 14,                  /**< EBGP route */
  OPEN_RTO6_LBGP = 15,                  /**< local BGP routes, e.g., reject routes for aggregates */
  OPEN_RTO6_EXTERNAL = 16,              /**< Learnt from external third party entity via netlink*/
  OPEN_RTO6_6TO4 = 17,                  /**< 6to4 route */
  OPEN_RTO6_NET_PROTOTYPE = 18          /**< Net prototype route */
} OPEN_RTO6_ROUTE_TYPE_INDICES_t;


/*********************************************************************
* @purpose  Gets the next route entry for the specified route
*           destination.
*
* @param    client_handle     @b{(input)}  client handle from registration API.
* @param    *routeDest        @b{(inout)}  route destination for this and next entry.
* @param    *routeMask        @b{(inout)}  route mask for this and next entry.
* @param    *routeProto       @b{(inout)}  route protocol for this and next entry.
* @param    *routePref        @b{(inout)}  route preference for this and next entry
* @param    -                 Minimum value @ref OPEN_API_ROUTE_PREF_MIN
* @param    -                 Maximum value @ref OPEN_API_ROUTE_PREF_MAX
*
* @returns  OPEN_E_NONE       If the corresponding entry exists.
* @returns  OPEN_E_FAIL       If the corresponding entry does not exist.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_INTERNAL   Internal error.
*
* @comment  To get first entry *routeDest, *routeMask, *routeProto and *routePref
*           should be set with 0.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIpRouteEntryGetNext(openapiClientHandle_t *client_handle,
                                        uint32_t *routeDest,
                                        uint32_t *routeMask,
                                        OPEN_RTO_PROTOCOL_INDICES_t *routeProto,
                                        uint32_t *routePref);

/*********************************************************************
* @purpose  Gets the primary routing metric for this route.
*
* @param    client_handle     @b{(input)}  client handle from registration API.
* @param    *routeDest        @b{(input)}  route destination for this aentry.
* @param    *routeMask        @b{(input)}  route mask for this entry.
* @param    *routeProto       @b{(input)}  route protocol for this entry.
* @param    *routePref        @b{(input)}  route preference for this entry.
* @param    -                 Minimum value @ref OPEN_API_ROUTE_PREF_MIN
* @param    -                 Maximum value @ref OPEN_API_ROUTE_PREF_MAX
* @param    *metric           @b{(output)} route metric for this entry.
*
* @returns  OPEN_E_NONE       If the primary routing metric was retrived successfully.
* @returns  OPEN_E_FAIL       If failed to get the primary routing metric.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_INTERNAL   Internal error.
* 
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIpRouteMetricGet(openapiClientHandle_t *client_handle,
                                     uint32_t *routeDest,
                                     uint32_t *routeMask,
                                     OPEN_RTO_PROTOCOL_INDICES_t *routeProto,
                                     uint32_t *routePref,
                                     uint32_t *metric);

/*********************************************************************
* @purpose  Gets the number of seconds since this route was last
*           updated or otherwise determined to be correct.
*
* @param    client_handle     @b{(input)}  client handle from registration API.
* @param    *routeDest        @b{(input)}  route destination for this aentry.
* @param    *routeMask        @b{(input)}  route mask for this entry.
* @param    *routeProto       @b{(input)}  route protocol for this entry.
* @param    *routePref        @b{(input)}  route preference for this entry.
* @param    -                 Minimum value @ref OPEN_API_ROUTE_PREF_MIN
* @param    -                 Maximum value @ref OPEN_API_ROUTE_PREF_MAX
* @param    *routeAge         @b{(output)} route age for this entry in seconds.
*
* @returns  OPEN_E_NONE       If the routing age was retrived successfully.
* @returns  OPEN_E_FAIL       If failed to get the routing age.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_INTERNAL   Internal error.
* 
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIpRouteAgeGet(openapiClientHandle_t *client_handle,
                                  uint32_t *routeDest,
                                  uint32_t *routeMask,
                                  OPEN_RTO_PROTOCOL_INDICES_t *routeProto,
                                  uint32_t *routePref,
                                  uint32_t *routeAge);

/*********************************************************************
* @purpose  Gets the hardware failure flag for this route.
*
* @param    client_handle     @b{(input)}  client handle from registration API.
* @param    *routeDest        @b{(input)}  route destination for this aentry.
* @param    *routeMask        @b{(input)}  route mask for this entry.
* @param    *routeProto       @b{(input)}  route protocol for this entry.
* @param    *routePref        @b{(input)}  route preference for this entry.
* @param    -                 Minimum value @ref OPEN_API_ROUTE_PREF_MIN
* @param    -                 Maximum value @ref OPEN_API_ROUTE_PREF_MAX
* @param    *hwFailure        @b{(output)} hardware failure flag for this entry.
*
* @returns  OPEN_E_NONE       If the hardware failure flag was retrived successfully.
* @returns  OPEN_E_FAIL       If failed to get the hardware failure flag.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_INTERNAL   Internal error.
* 
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIpRouteHwFailureGet(openapiClientHandle_t *client_handle,
                                        uint32_t *routeDest,
                                        uint32_t *routeMask,
                                        OPEN_RTO_PROTOCOL_INDICES_t *routeProto,
                                        uint32_t *routePref,
                                        OPEN_BOOL_t *hwFailure);

/*********************************************************************
* @purpose  Gets the comma separated list of the local interfaces
*           through which the next hop of this route should be reached.
*
* @param    client_handle     @b{(input)}  client handle from registration API.
* @param    *routeDest        @b{(input)}  route destination for this aentry.
* @param    *routeMask        @b{(input)}  route mask for this entry.
* @param    *routeProto       @b{(input)}  route protocol for this entry.
* @param    *routePref        @b{(input)}  route preference for this entry.
* @param    -                 Minimum value @ref OPEN_API_ROUTE_PREF_MIN
* @param    -                 Maximum value @ref OPEN_API_ROUTE_PREF_MAX
* @param    *nextHopIntf      @b{(output)} comma separated list of the
*                                          next hop interfaces.
* @param    -                 Minimum buffer size @ref OPEN_API_ROUTE_NEXT_HOP_INTF_BUF_MIN
*
* @returns  OPEN_E_NONE       If the hardware failure flag was retrived successfully.
* @returns  OPEN_E_FAIL       If failed to get the hardware failure flag.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_INTERNAL   Internal error.
* 
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIpRouteNextHopIntfGet(openapiClientHandle_t *client_handle,
                                          uint32_t *routeDest,
                                          uint32_t *routeMask,
                                          OPEN_RTO_PROTOCOL_INDICES_t *routeProto,
                                          uint32_t *routePref,
                                          open_buffdesc *nextHopIntf);

/*********************************************************************
* @purpose  Gets the comma separated list of the IP addresses
*           of the next hop of this route.
*
* @param    client_handle     @b{(input)}  client handle from registration API.
* @param    *routeDest        @b{(input)}  route destination for this aentry.
* @param    *routeMask        @b{(input)}  route mask for this entry.
* @param    *routeProto       @b{(input)}  route protocol for this entry.
* @param    *routePref        @b{(input)}  route preference for this entry.
* @param    -                 Minimum value @ref OPEN_API_ROUTE_PREF_MIN
* @param    -                 Maximum value @ref OPEN_API_ROUTE_PREF_MAX
* @param    *nextHopAddr      @b{(output)} comma separated list of the
*                                          next hop IP addresses.
* @param    -                 Minimum buffer size @ref OPEN_API_ROUTE_NEXT_HOP_ADDR_BUF_MIN
* 
* @returns  OPEN_E_NONE       If the hardware failure flag was retrived successfully.
* @returns  OPEN_E_FAIL       If failed to get the hardware failure flag.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_INTERNAL   Internal error.
* 
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIpRouteNextHopAddrGet(openapiClientHandle_t *client_handle,
                                          uint32_t *routeDest,
                                          uint32_t *routeMask,
                                          OPEN_RTO_PROTOCOL_INDICES_t *routeProto,
                                          uint32_t *routePref,
                                          open_buffdesc *nextHopAddr);
#endif /* OPENAPI_IP_ROUTE_TABLE_H_INCLUDED */
