/**********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_iphelper.h
*
* @purpose   This code implements the OpEN IP-HELPER API
*
* @component OPEN
*
* @create    02/28/2023
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_IPHELPER_H_INCLUDED
#define OPENAPI_IPHELPER_H_INCLUDED

#include "openapi_common.h" 


typedef struct OPEN_IPHELPER_STATS_s                    
{
  /**< Number of valid DHCP client messages received */
  uint32_t dhcpClientMsgsReceived;               

  /**< Number of DHCP client packets relayed to a server. If a packet is relayed to
    multiple servers, stat is incremented for each server. */ 
  uint32_t dhcpClientMsgsRelayed;                

  /**< Number of messages received from a DHCP server */
  uint32_t dhcpServerMsgsReceived;               

  /**< Number of DHCP server messages relayed to client */
  uint32_t dhcpServerMsgsRelayed;                

  /**< Number of UDP client messages received. Includes DHCP. */ 
  uint32_t udpClientMsgsReceived;

  /**< Number of UDP client messages relayed. Includes DHCP. */
  uint32_t udpClientMsgsRelayed;                 

  /**< DHCP client message arrived with hops > max allowed */
  uint32_t tooManyHops;                          

  /**< DHCP client message arrived with secs field lower than min allowed */
  uint32_t tooEarly;                             

  /**< Received DHCP client message with giaddr already set to our own address */
  uint32_t spoofedGiaddr;

  /**< Number of packets intercepted whose TTL is <= 1 */
  uint32_t ttlExpired;

  /**< Number of times server lookup matched a discard entry */
  uint32_t matchDiscardEntry;                    

} OPEN_IPHELPER_STATS_t;

/**< Relay interface mode values */
typedef enum
{
  OPEN_RELAY_INTF_DISABLE = 0,                      
  OPEN_RELAY_INTF_ENABLE,
  OPEN_RELAY_INTF_NOT_CONFIGURED
} OPEN_DHCP_RELAY_INTF_VALUE_t;


/*****************************************************************************
 * @purpose   Get the Maximum Hop count configured for DHCP relay agent for the given VRF.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(input)}  VRF name
 * @param     maxHopCount     @b{(output)} The maximum hop count
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayMaxHopCountGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t *maxHopCount);

/*****************************************************************************
 * @purpose   Set the Maximum Hop count for DHCP relay agent for the given VRF.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(input)}  VRF name
 * @param     maxHopCount     @b{(input)}  The maximum hop count
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayMaxHopCountSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t maxHopCount);

/*****************************************************************************
 * @purpose   Get the Minimum Wait time configured for DHCP relay agent for the given VRF.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(input)}  VRF name
 * @param     minWaitTime     @b{(output)} The minimum wait time
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayMinWaitTimeGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t *minWaitTime);

/*****************************************************************************
 * @purpose   Set the Minimum Wait time for DHCP relay agent for the given VRF.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(input)}  VRF name
 * @param     minWaitTime     @b{(input)}  The minimum wait time
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayMinWaitTimeSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t minWaitTime);

/*****************************************************************************
 * @purpose   Get the administrative mode of IP-Helper for the given VRF.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(input)}  VRF name
 * @param     mode            @b{(output)} OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpHelperAdminModeGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the administrative mode of IP-Helper for the given VRF.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(input)}  VRF name
 * @param     mode            @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpHelperAdminModeSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the Relay Agent Circuit ID option mode for the given VRF.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(input)}  VRF name
 * @param     mode            @b{(output)} OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayCircuitIdOptionModeGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the Relay Agent Circuit ID option mode for the given VRF.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(input)}  VRF name
 * @param     mode            @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayCircuitIdOptionModeSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the Relay Agent global option check mode for the given VRF.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(input)}  VRF name
 * @param     mode            @b{(output)} OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayCIDOptionCheckModeGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the Relay Agent global option check mode for the given VRF.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(input)}  VRF name
 * @param     mode            @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayCIDOptionCheckModeSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the IP Helper statistics for the given VRF.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(input)}  VRF name
 * @param     ihStats         @b{(output)} IP helper stats
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpHelperStatisticsGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_IPHELPER_STATS_t *ihStats);

/*****************************************************************************
 * @purpose   Get the Relay Agent global Server Override mode for the given VRF.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(input)}  VRF name
 * @param     mode            @b{(output)} OPEN_TRUE or OPEN_FALSE
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayServerOverrideModeGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_BOOL_t *mode);

/*****************************************************************************
 * @purpose   Set the Relay Agent global Server Override mode for the given VRF.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(input)}  VRF name
 * @param     mode            @b{(input)}  OPEN_TRUE or OPEN_FALSE
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayServerOverrideModeSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_BOOL_t mode);

/*****************************************************************************
 * @purpose   Get the Relay Agent Server Override mode for the given interface.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     ifNum           @b{(input)}  Internal interface number
 * @param     mode            @b{(output)} OPEN_RELAY_INTF_DISABLE or OPEN_RELAY_INTF_ENABLE or OPEN_RELAY_INTF_NOT_CONFIGURED
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayIntfServerOverrideGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_DHCP_RELAY_INTF_VALUE_t *mode);

/*****************************************************************************
 * @purpose   Set the Relay Agent Server Override mode for the given interface.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     ifNum           @b{(input)}  Internal interface number
 * @param     mode            @b{(input)}  OPEN_RELAY_INTF_DISABLE or OPEN_RELAY_INTF_ENABLE or OPEN_RELAY_INTF_NOT_CONFIGURED
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayIntfServerOverrideSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_DHCP_RELAY_INTF_VALUE_t mode);

/*****************************************************************************
 * @purpose   Get the global Source interface of DHCP relay agent for the given VRF.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(input)}  VRF name
 * @param     ifNum           @b{(output)} The source interface
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelaySourceInterfaceGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t *ifNum);

/*****************************************************************************
 * @purpose   Set the global Source interface of DHCP relay agent for the given VRF.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(input)}  VRF name
 * @param     ifNum           @b{(input)}  The source interface
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelaySourceInterfaceSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t ifNum);

/*****************************************************************************
 * @purpose   Clear the global Source interface of DHCP relay agent for the given VRF.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(input)}  VRF name
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelaySourceInterfaceClear(openapiClientHandle_t *client_handle, open_buffdesc *vrfName);

/*****************************************************************************
 * @purpose   Get the interface level Source interface for DHCP relay agent.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     ifNum           @b{(input)}  Internal interface number
 * @param     srcIfNum        @b{(output)} The source interface
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayIntfSourceInterfaceGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *srcIfNum);

/*****************************************************************************
 * @purpose   Set the interface level Source interface for DHCP relay agent.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     ifNum           @b{(input)}  Internal interface number
 * @param     srcIfNum        @b{(input)}  The source interface
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayIntfSourceInterfaceSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t srcIfNum);

/*****************************************************************************
 * @purpose   Clear the interface level Source interface for DHCP relay agent.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     ifNum           @b{(input)}  Internal interface number
 
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayIntfSourceInterfaceClear(openapiClientHandle_t *client_handle, uint32_t ifNum);

/*****************************************************************************
 * @purpose   Remove all IP helper addresses configured on an interface.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     ifNum           @b{(input)}  Internal interface number
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpHelperAddressesRemove(openapiClientHandle_t *client_handle, uint32_t ifNum);

/*****************************************************************************
 * @purpose   Get the Relay Agent Circuit ID Option mode for the given interface.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     ifNum           @b{(input)}  Internal interface number
 * @param     mode            @b{(output)} OPEN_RELAY_INTF_DISABLE or OPEN_RELAY_INTF_ENABLE or OPEN_RELAY_INTF_NOT_CONFIGURED
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayrtrCIDOptionModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_DHCP_RELAY_INTF_VALUE_t *mode);

/*****************************************************************************
 * @purpose   Set the Relay Agent Circuit ID Option mode for the given interface.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     ifNum           @b{(input)}  Internal interface number
 * @param     mode            @b{(input)}  OPEN_RELAY_INTF_DISABLE or OPEN_RELAY_INTF_ENABLE or OPEN_RELAY_INTF_NOT_CONFIGURED
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayrtrCIDOptionModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_DHCP_RELAY_INTF_VALUE_t mode);

/*****************************************************************************
 * @purpose   Get the Relay Agent Circuit ID option check mode for the given interface.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     ifNum           @b{(input)}  Internal interface number
 * @param     mode            @b{(output)} OPEN_RELAY_INTF_DISABLE or OPEN_RELAY_INTF_ENABLE or OPEN_RELAY_INTF_NOT_CONFIGURED
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayrtrCIDOptionCheckModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_DHCP_RELAY_INTF_VALUE_t *mode);

/*****************************************************************************
 * @purpose   Set the Relay Agent Circuit ID option check mode for the given interface.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     ifNum           @b{(input)}  Internal interface number
 * @param     mode            @b{(input)}  OPEN_RELAY_INTF_DISABLE or OPEN_RELAY_INTF_ENABLE or OPEN_RELAY_INTF_NOT_CONFIGURED
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiBootpDhcpRelayrtrCIDOptionCheckModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_DHCP_RELAY_INTF_VALUE_t mode);

/*****************************************************************************
 * @purpose   Clear IP helper statistics for the given VRF.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(input)}  VRF name
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpHelperStatisticsClear(openapiClientHandle_t *client_handle, open_buffdesc *vrfName);

/*****************************************************************************
 * @purpose   Get the first relay entry along with the server's VRF name config
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(inout)}  VRF name
 * @param     intIfNum        @b{(inout)}  internal interface number
 * @param     udpPort         @b{(inout)}  UDP port number
 * @param     serverAddr      @b{(output)} IPv4 address of server
 * @param     discard         @b{(output)} L7_TRUE if a discard entry
 * @param     isServerVrfSet  @b{(output)} L7_TRUE if the server's VRF name us provided
 * @param     serverVrfName   @b{(output)} VRF Name of the server if configured
 * @param     hitCount        @b{(output)} Number of times this entry has 
 *                                         been used to relay a packet
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIhIpHelperAddressFirst(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t *intIfNum,
                                           uint16_t *udpPort, uint32_t *serverAddr, OPEN_BOOL_t *discard,
                                           OPEN_BOOL_t *isServerVrfSet, open_buffdesc *serverVrfName, uint32_t *hitCount);

/*****************************************************************************
 * @purpose   Get the next relay entry along with the server's VRF name config
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API
 * @param     vrfName         @b{(inout)}  VRF name
 * @param     intIfNum        @b{(inout)}  internal interface number
 * @param     udpPort         @b{(inout)}  UDP port number
 * @param     serverAddr      @b{(output)} IPv4 address of server
 * @param     discard         @b{(output)} L7_TRUE if a discard entry
 * @param     isServerVrfSet  @b{(output)} L7_TRUE if the server's VRF name us provided
 * @param     serverVrfName   @b{(output)} VRF Name of the server if configured
 * @param     hitCount        @b{(output)} Number of times this entry has 
 *                                         been used to relay a packet
 *
 * @returns   OPEN_E_NONE     On success.
 * @returns   OPEN_E_FAIL     On failure.
 * @returns   OPEN_E_PARAM    Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiIhIpHelperAddressNext(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t *intIfNum,
                                          uint16_t *udpPort, uint32_t *serverAddr, OPEN_BOOL_t *discard,
                                          OPEN_BOOL_t *isServerVrfSet, open_buffdesc *serverVrfName, uint32_t *hitCount);

#endif /* OPENAPI_IPHELPER_H_INCLUDED */
