/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_lldp.h
*
* @purpose   LLDP Configuration and Status 
*
* @component OpEN
*
* @create    10/10/2012
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_LLDP_H_INCLUDED
#define OPENAPI_LLDP_H_INCLUDED

#include "openapi_common.h"

#define OPEN_LLDP_MED_CAP_CAPABILITIES_BITMASK     0x01
#define OPEN_LLDP_MED_CAP_NETWORKPOLICY_BITMASK    0x02
#define OPEN_LLDP_MED_CAP_LOCATION_BITMASK         0x04
#define OPEN_LLDP_MED_CAP_EXT_PSE_BITMASK          0x08
#define OPEN_LLDP_MED_CAP_EXT_PD_BITMASK           0x10
#define OPEN_LLDP_MED_CAP_INVENTORY_BITMASK        0x20

typedef struct
{
  unsigned char   bitmap[2];
} OPEN_LLDP_MED_CAPABILITIES_t;

typedef enum
{
  OPEN_LLDP_CHASSIS_ID_SUBTYPE_CHASSIS_COMP = 1, /**< Chassis Component */
  OPEN_LLDP_CHASSIS_ID_SUBTYPE_INTF_ALIAS   = 2, /**< Interface Alias */
  OPEN_LLDP_CHASSIS_ID_SUBTYPE_PORT_COMP    = 3, /**< Port Component */
  OPEN_LLDP_CHASSIS_ID_SUBTYPE_MAC_ADDR     = 4, /**< MAC Address */
  OPEN_LLDP_CHASSIS_ID_SUBTYPE_NET_ADDR     = 5, /**< Network Address */
  OPEN_LLDP_CHASSIS_ID_SUBTYPE_INTF_NAME    = 6, /**< Interface Name */
  OPEN_LLDP_CHASSIS_ID_SUBTYPE_LOCAL        = 7  /**< Local */
} OPEN_LLDP_CHASSIS_ID_SUBTYPE_t;

typedef enum {
  OPEN_LLDP_PORT_ID_SUBTYPE_INTF_ALIAS      = 1, /**< Interface Alias */
  OPEN_LLDP_PORT_ID_SUBTYPE_PORT_COMP       = 2, /**< Port Component */
  OPEN_LLDP_PORT_ID_SUBTYPE_MAC_ADDR        = 3, /**< MAC Address */
  OPEN_LLDP_PORT_ID_SUBTYPE_NET_ADDR        = 4, /**< Network Address */
  OPEN_LLDP_PORT_ID_SUBTYPE_INTF_NAME       = 5, /**< Interface Name */
  OPEN_LLDP_PORT_ID_SUBTYPE_AGENT_ID        = 6, /**< Agent Circuit Id */
  OPEN_LLDP_PORT_ID_SUBTYPE_LOCAL           = 7  /**< Local */
} OPEN_LLDP_PORT_ID_SUBTYPE_t;

typedef enum {
  OPEN_LLDP_SYS_DESC_TLV                    = 0, /**< System Description TLV */
  OPEN_LLDP_SYS_NAME_TLV                    = 1, /**< System Name TLV */
  OPEN_LLDP_SYS_CAPABILITIES_TLV            = 2, /**< System Capabilities TLV */
  OPEN_LLDP_PORT_DESC_TLV                   = 3, /**< Port Description TLV */
  OPEN_LLDP_MGMT_ADDRESS_TLV                = 4, /**< Management Address TLV */
  OPEN_LLDP_LAST_TLV                        = 5  /**< Unused */
} OPEN_LLDP_TLV_TYPE_t;

typedef enum {
  OPEN_LLDP_IANA_ADDR_FAMILY_NUMBER_RESERVED = 0, /**< Reserved */
  OPEN_LLDP_IANA_ADDR_FAMILY_NUMBER_IPV4     = 1, /**< IPv4 */
  OPEN_LLDP_IANA_ADDR_FAMILY_NUMBER_IPV6     = 2, /**< IPv6 */
  OPEN_LLDP_IANA_ADDR_FAMILY_NUMBER_802      = 6  /**< 802 */
} OPEN_LLDP_IANA_ADDR_FAMILY_t;

typedef enum
{
  OPEN_UNKNOWN_POLICY_APPTYPE = 0,           /**< Unknwon policy */
  OPEN_VOICE_POLICY_APPTYPE,                 /**< Voice policy */
  OPEN_VOICE_SIGNALING_POLICY_APPTYPE,       /**< Voice signaling polcy*/ 
  OPEN_GUEST_VOICE_POLICY_APPTYPE,           /**< Guest voice policy */ 
  OPEN_GUEST_VOICE_SIGNALING_POLICY_APPTYPE, /**< Guest voice signaling policy */
  OPEN_SOFTPHONE_VOICE_POLICY_APPTYPE,       /**< Softphone voice policy */
  OPEN_VIDEO_CONFERENCING_POLICY_APPTYPE,    /**< Video conferencing policy */ 
  OPEN_STREAMING_VIDEO_POLICY_APPTYPE,       /**< Streaming video policy */ 
  OPEN_VIDEO_SIGNALING_POLICY_APPTYPE        /**< Video signaling policy */
}OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t;

typedef enum
{
  OPEN_UNKNOWN_LOCATION_SUBTYPE = 0,         /**< Unknown location */ 
  OPEN_COORDINATE_BASED_LOCATION_SUBTYPE,    /**< Co ordinate based location */
  OPEN_CIVIC_ADDRESS_LOCATION_SUBTYPE,       /**< Civic address location */
  OPEN_ELIN_LOCATION_SUBTYPE                 /**< Elin location */
}OPEN_LLDPXMED_LOCATION_SUBTYPE_t;

typedef enum {
  OPEN_LLDP_MAN_ADDR_IF_SUBTYPE_UNKNOWN = 1,     /**< Unknwon subtype */
  OPEN_LLDP_MAN_ADDR_IF_SUBTYPE_IF_INDEX = 2,    /**< Interface index subtype */
  OPEN_LLDP_MAN_ADDR_IF_SUBTYPE_PORT_NUMBER = 3  /**< Port number subtype */
} OPEN_LLDP_MAN_ADDR_IF_SUBTYPE_t;

#define OPEN_LLDP_TX_INTERVAL_MIN                  5
#define OPEN_LLDP_TX_INTERVAL_MAX                  32768
#define OPEN_LLDP_TX_HOLD_MULTIPLIER_MIN           2
#define OPEN_LLDP_TX_HOLD_MULTIPLIER_MAX           10
#define OPEN_LLDP_REINIT_DELAY_MIN                 1 
#define OPEN_LLDP_REINIT_DELAY_MAX                 10
#define OPEN_LLDP_NOTIFY_INTERVAL_MIN              5
#define OPEN_LLDP_NOTIFY_INTERVAL_MAX              3600
#define OPEN_LLDP_MED_FASTSTART_REPEAT_COUNT_MIN   1
#define OPEN_LLDP_MED_FASTSTART_REPEAT_COUNT_MAX   10

#define OPEN_LLDP_AUTO_NEG_ADVERTIZED_CAPABILITY_SIZE 3
#define OPEN_LLDP_STRING_SIZE                     32 
#define OPEN_LLDP_LOCATION_INFO_STRING_SIZE       256
#define OPEN_LLDP_UNKNOWN_TLV_INFO_SIZE_MAX       511
#define OPEN_LLDP_CHASSIS_ID_SUBTYPE_STRING_SIZE  64
/*********************************************************************
* @purpose  Gets the LLDP Management string length. This length is used to
*           allocate and initialize the open_buffdesc type argument buffers
*           used by various LLDP OpEN APIs.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    strLen          @b{(output)}  string length
*
* @returns  OPEN_E_NONE        String length get successful.
* @returns  OPEN_E_PARAM       Error in parameter passed. 
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpMgmtStrLenGet(openapiClientHandle_t *client_handle,
                                      uint32_t *strLen);

/*********************************************************************
* @purpose  Sets LLDP PDU transmit mode on the specified interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ifNum           @b{(input)}   ID of the interface
* @param    lldpTxMode      @b{(input)}   LLDP transmit mode
*
* @returns  OPEN_E_NONE        Transmit mode set successful.
* @returns  OPEN_E_FAIL        Transmit mode set failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.  Invalid lldpTxMode can
*                              cause this return code.
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfTransmitModeSet(openapiClientHandle_t *client_handle,
                                            uint32_t ifNum, 
                                            OPEN_CONTROL_t lldpTxMode);

/*********************************************************************
* @purpose  Gets the LLDP PDU transmit mode for the specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  ID of the interface
* @param    lldpTxMode      @b{(output)} LLDP transmit mode
*
* @returns  OPEN_E_NONE        Transmit mode get successful .
* @returns  OPEN_E_FAIL        Transmit mode get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.  NULL lldpTxMode can
*                              cause this return code.
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfTransmitModeGet(openapiClientHandle_t *client_handle,
                                            uint32_t ifNum, 
                                            OPEN_CONTROL_t *lldpTxMode);

/*********************************************************************
* @purpose  Sets LLDP PDU reception mode on the specified interface.
*
* @param    client_handle   @b{(input)} client handle from registration API
* @param    ifNum           @b{(input)} ID of the interface
* @param    lldpRxMode      @b{(input)} LLDP receive mode
*
* @returns  OPEN_E_NONE        Receive mode set successful.
* @returns  OPEN_E_FAIL        Receive mode set failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.  Invalid lldpRxMode can
*                             cause this return code.
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfReceiveModeSet(openapiClientHandle_t *client_handle,
                                           uint32_t ifNum, 
                                           OPEN_CONTROL_t lldpRxMode);

/*********************************************************************
* @purpose  Gets the LLDP PDU reception mode for the specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  ID of the interface
* @param    lldpRxMode      @b{(output)} LLDP receive mode
*
* @returns  OPEN_E_NONE        Receive mode get successful .
* @returns  OPEN_E_FAIL        Receive mode get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.  NULL lldpRxMode can
*                              cause this return code.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfReceiveModeGet(openapiClientHandle_t *client_handle,
                                           uint32_t ifNum, 
                                           OPEN_CONTROL_t *lldpRxMode);

/*********************************************************************
* @purpose  Sets the LLDP optional TLVs transmit mode for the specified interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    ifNum            @b{(input)}   ID of the interface
* @param    tlvType          @b{(input)}   LLDP TLV of type OPEN_LLDP_TLV_TYPE_t
* @param    tlvTransmitMode  @b{(input)}   TLV transmit mode
*
* @returns  OPEN_E_NONE        TLVs transmit mode set successful.
* @returns  OPEN_E_FAIL        TLVs transmit mode set failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfTLVTransmitModeSet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum, 
                                               OPEN_LLDP_TLV_TYPE_t tlvType,
                                               OPEN_CONTROL_t tlvTransmitMode);

/*********************************************************************
* @purpose  Gets the LLDP optional TLVs transmit mode for the specified interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    ifNum            @b{(input)}   ID of the interface
* @param    tlvType          @b{(input)}   LLDP TLV of type OPEN_LLDP_TLV_TYPE_t
* @param    tlvTransmitMode  @b{(output)}   TLV transmit mode
*
* @returns  OPEN_E_NONE        TLVs transmit mode get successful.
* @returns  OPEN_E_FAIL        TLVs transmit mode get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfTLVTransmitModeGet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum, 
                                               OPEN_LLDP_TLV_TYPE_t tlvType,
                                               OPEN_CONTROL_t *tlvTransmitMode);

/*********************************************************************
* @purpose  Sets the LLDP Notification mode for the specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  ID of the interface
* @param    lldpNotMode     @b{(input)}  LLDP Notification mode
*
* @returns  OPEN_E_NONE        Notification mode set successful.
* @returns  OPEN_E_FAIL        Notification mode set failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.  Invalid lldpNotMode can
*                              cause this return code.
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfNotificationModeSet(openapiClientHandle_t *client_handle,
                                                uint32_t ifNum, 
                                                OPEN_CONTROL_t lldpNotMode);

/*********************************************************************
* @purpose  Gets the LLDP Notification mode for the specified interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ifNum           @b{(input)}   ID of the interface
* @param    lldpNotMode     @b{(output)}  LLDP Notification mode
*
* @returns  OPEN_E_NONE        Notification mode get successful.
* @returns  OPEN_E_FAIL        Notification mode get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfNotificationModeGet(openapiClientHandle_t *client_handle, 
                                                uint32_t ifNum,
                                                OPEN_CONTROL_t *lldpNotMode);

/*********************************************************************
* @purpose  Gets the next valid LLDP interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ifNum           @b{(input)}   ID of the interface
* @param    nextIfNum       @b{(output)}  ID of the next interface
*
* @returns  OPEN_E_NONE        Next valid LLDP interface get successful.
* @returns  OPEN_E_UNAVAIL     No next valid LLDP interface found.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfNextGet(openapiClientHandle_t *client_handle, 
                                    uint32_t ifNum,
                                    uint32_t *nextIfNum);

/*********************************************************************
* @purpose  Gets the next LLDP Remote Device entry for the specified interface.
*           Passing 0 for the remIndex and timeStamp returns the first Neighbor entry.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  ID of the interface
* @param    remIndex        @b{(inout)}  Remote Device Index
* @param    timeStamp       @b{(inout)} Timestamp when the entry was added
*
* @returns  OPEN_E_NONE        Neighbor entry get successful.
* @returns  OPEN_E_FAIL        Next Neighbor entry is not present.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfRemoteDeviceNextGet(openapiClientHandle_t *client_handle,
                                                uint32_t ifNum, 
                                                uint32_t *remIndex,
                                                uint32_t *timeStamp);

/*********************************************************************
* @purpose  Gets the LLDP Remote Device Chassis Id and Chassis Id Subtype 
*           for the specified interface.
*           
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    remIndex         @b{(input)}  Remote Device Index.
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    timeStamp        @b{(input)}  Remote Device entry timestamp.
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    chassisIdSubtype @b{(output)} Chassis Id Subtype of type OPEN_LLDP_CHASSIS_ID_SUBTYPE_t
* @param    chassisId        @b{(output)} Chassis Id.
*
* @notes   Maximum size for chassisId is determined by calling
*          openapiLldpMgmtStrLenGet().
*
* @returns  OPEN_E_NONE        Chassis Id and Subtype get successful.
* @returns  OPEN_E_FAIL        Chassis Id and Subtype get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other internal error.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfRemoteDeviceChassisIdGet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum, 
                                                     uint32_t remIndex,
                                                     uint32_t timeStamp,
                                                     OPEN_LLDP_CHASSIS_ID_SUBTYPE_t *chassisIdSubtype, 
                                                     open_buffdesc *chassisId);

/*********************************************************************
* @purpose  Gets the LLDP Remote Device Port Id and Port Id Subtype 
*           for the specified interface.
*           
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    remIndex         @b{(input)}  Remote Device Index
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    timeStamp        @b{(input)}  Remote Device entry timestamp
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    portIdSubtype    @b{(output)} Port Id Subtype of type OPEN_LLDP_PORT_ID_SUBTYPE_t
* @param    portId           @b{(output)} Port Id
* 
* @notes   Maximum size for portId is determined by calling
*          openapiLldpMgmtStrLenGet().
*
* @returns  OPEN_E_NONE        Port Id and Subtype get successful.
* @returns  OPEN_E_FAIL        Port Id and Subtype get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfRemoteDevicePortIdGet(openapiClientHandle_t *client_handle,
                                                  uint32_t ifNum, 
                                                  uint32_t remIndex,
                                                  uint32_t timeStamp,
                                                  OPEN_LLDP_PORT_ID_SUBTYPE_t *portIdSubtype, 
                                                  open_buffdesc *portId);

/*********************************************************************
* @purpose  Gets the LLDP Remote Device System Name 
*           for the specified interface.
*           
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    remIndex         @b{(input)}  Remote Device Index
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    timeStamp        @b{(input)}  Remote Device entry timestamp
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    systemName       @b{(output)} Remote Device System Name
*
* @notes   Maximum size for systemName is determined by calling
*          openapiLldpMgmtStrLenGet(). 
*
* @returns  OPEN_E_NONE        System Name get successful.
* @returns  OPEN_E_FAIL        System Name get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfRemoteDeviceSystemNameGet(openapiClientHandle_t *client_handle,
                                                      uint32_t ifNum, 
                                                      uint32_t remIndex,
                                                      uint32_t timeStamp,
                                                      open_buffdesc *systemName);

/*********************************************************************
* @purpose  Gets the LLDP Remote Device System Description 
*           for the specified interface.
*           
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    remIndex         @b{(input)}  Remote Device Index
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    timeStamp        @b{(input)}  Remote Device entry timestamp
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    systemDesc       @b{(output)} Remote Device System Description
*
* @notes   Maximum size for systemDesc is determined by calling
*          openapiLldpMgmtStrLenGet(). 
*
* @returns  OPEN_E_NONE        System Description get successful.
* @returns  OPEN_E_FAIL        System Description get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfRemoteDeviceSystemDescGet(openapiClientHandle_t *client_handle,
                                                      uint32_t ifNum, 
                                                      uint32_t remIndex,
                                                      uint32_t timeStamp,
                                                      open_buffdesc *systemDesc);

/*********************************************************************
* @purpose  Gets the LLDP Remote Device Port Description 
*           for the specified interface.
*           
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    remIndex         @b{(input)}  Remote Device Index
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    timeStamp        @b{(input)}  Remote Device entry timestamp
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    portDesc         @b{(output)} Remote Device Port Description
*
* @notes   Maximum size for portDesc is determined by calling
*          openapiLldpMgmtStrLenGet(). 
*
* @returns  OPEN_E_NONE        Port Description get successful.
* @returns  OPEN_E_FAIL        Port Description get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfRemoteDevicePortDescGet(openapiClientHandle_t *client_handle,
                                                    uint32_t ifNum, 
                                                    uint32_t remIndex,
                                                    uint32_t timeStamp,
                                                    open_buffdesc *portDesc);

/*********************************************************************
* @purpose  Gets the LLDP Remote Device System Capabilities Supported 
*           for the specified interface.
*           
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    remIndex         @b{(input)}  Remote Device Index
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    timeStamp        @b{(input)}  Remote Device entry timestamp
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    sysCapSupp       @b{(output)} Remote Device System Capabilities Supported
*
* @notes   Maximum size for sysCapSupp is determined by calling
*          openapiLldpMgmtStrLenGet(). 
*
* @returns  OPEN_E_NONE        Supported Capabilities get successful.
* @returns  OPEN_E_FAIL        Supported Capabilities get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfRemoteDeviceSysCapSuppGet(openapiClientHandle_t *client_handle,
                                                      uint32_t ifNum, 
                                                      uint32_t remIndex,
                                                      uint32_t timeStamp,
                                                      open_buffdesc *sysCapSupp);

/*********************************************************************
* @purpose  Gets the LLDP Remote Device System Capabilities Enabled 
*           for the specified interface.
*           
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    remIndex         @b{(input)}  Remote Device Index
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    timeStamp        @b{(input)}  Remote Device entry timestamp
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    sysCapEnabled    @b{(output)} Remote Device System Capabilities Enabled
*
* @notes   Maximum size for sysCapEnabled is determined by calling
*          openapiLldpMgmtStrLenGet().
*
* @returns  OPEN_E_NONE        Enabled Capabilities get successful.
* @returns  OPEN_E_FAIL        Supported Capabilities get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfRemoteDeviceSysCapEnabledGet(openapiClientHandle_t *client_handle,
                                                         uint32_t ifNum, 
                                                         uint32_t remIndex,
                                                         uint32_t timeStamp,
                                                         open_buffdesc *sysCapEnabled);

/*********************************************************************
* @purpose  Gets the LLDP Remote Device next Management Address 
*           for the specified interface.
*           
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    remIndex         @b{(input)}  Remote Device Index
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    timeStamp        @b{(input)}  Remote Device entry timestamp
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    addressFamily    @b{(inout)}  Address Family. Pass 0 for the first time.
* @param    address          @b{(inout)}  Address
*                                         Initialize portDesc:pstart and size with a char buffer of length
*                                         returned by openapiLldpMgmtStrLenGet()
* @param    addrLen          @b{(inout)}  Address Length. Passing 0 will return
*                                         the first address
*
* @notes   Maximum size for address is determined by calling
*          openapiLldpMgmtStrLenGet().
* 
* @returns  OPEN_E_NONE        Management Address get successful.
* @returns  OPEN_E_FAIL        Management Address get failed.
* @returns  OPEN_E_UNAVAIL     Next Management Address not available.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfRemoteDeviceMgmtAddressNextGet(openapiClientHandle_t *client_handle,
                                                           uint32_t ifNum, 
                                                           uint32_t remIndex,
                                                           uint32_t timeStamp,
                                                           OPEN_LLDP_IANA_ADDR_FAMILY_t *addressFamily, 
                                                           open_buffdesc *address,
                                                           uint8_t *addrLen);

/*********************************************************************
* @purpose  Gets the LLDP Remote Device TTL 
*           for the specified interface.
*           
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    remIndex         @b{(input)}  Remote Device Index
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    timeStamp        @b{(input)}  Remote Device entry timestamp
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    ttl              @b{(output)} Remote Device TTL
*
* @returns  OPEN_E_NONE        TTL get successful.
* @returns  OPEN_E_FAIL        TTL get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfRemoteDeviceTTLGet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum, 
                                               uint32_t remIndex,
                                               uint32_t timeStamp,
                                               uint32_t *ttl);

/*********************************************************************
* @purpose  Gets the LLDP Local Device Chassis Id and Chassis Id Subtype 
*           for the specified interface.
*           
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    chassisIdSubtype @b{(output)} Chassis Id Subtype of type OPEN_LLDP_CHASSIS_ID_SUBTYPE_t
* @param    chassisId        @b{(output)} Chassis Id
*
* @notes   Maximum size for chassisId is determined by calling
*          openapiLldpMgmtStrLenGet().
*
* @returns  OPEN_E_NONE        Chassis Id and Subtype get successful.
* @returns  OPEN_E_FAIL        Chassis Id and Subtype get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface or
*                              system capabilities are not available as transmit
*                              mode is disabled.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfLocalDeviceChassisIdGet(openapiClientHandle_t *client_handle,
                                                    uint32_t ifNum, 
                                                    OPEN_LLDP_CHASSIS_ID_SUBTYPE_t *chassisIdSubtype, 
                                                    open_buffdesc *chassisId);

/*********************************************************************
* @purpose  Gets the LLDP Local Device Port Id and Port Id Subtype 
*           for the specified interface.
*           
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    portIdSubtype    @b{(output)} Port Id Subtype of type OPEN_LLDP_PORT_ID_SUBTYPE_t
* @param    portId           @b{(output)} Port Id
*
* @notes   Maximum size for portId is determined by calling
*          openapiLldpMgmtStrLenGet().
*
* @returns  OPEN_E_NONE        Port Id and Subtype get successful.
* @returns  OPEN_E_FAIL        Port Id and Subtype get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface or
*                              system capabilities are not available as transmit
*                              mode is disabled.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfLocalDevicePortIdGet(openapiClientHandle_t *client_handle,
                                                 uint32_t ifNum, 
                                                 OPEN_LLDP_PORT_ID_SUBTYPE_t *portIdSubtype, 
                                                 open_buffdesc *portId);

/*********************************************************************
* @purpose  Gets the LLDP Local Device System Name 
*           for the specified interface.
*           
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    systemName       @b{(output)} Remote Device System Name
*
* @notes   Maximum size for systemName is determined by calling
*          openapiLldpMgmtStrLenGet().
*
* @returns  OPEN_E_NONE        System Name get successful.
* @returns  OPEN_E_FAIL        System Name get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface or
*                              system capabilities are not available as transmit
*                              mode is disabled.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfLocalDeviceSystemNameGet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum, 
                                                     open_buffdesc *systemName);

/*********************************************************************
* @purpose  Gets the LLDP Local Device System Description 
*           for the specified interface.
*           
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    systemDesc       @b{(output)} Local Device System Description
*
* @notes   Maximum size for systemDesc is determined by calling
*          openapiLldpMgmtStrLenGet().
*
* @returns  OPEN_E_NONE        System Description get successful.
* @returns  OPEN_E_FAIL        System Description get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface or
*                              system capabilities are not available as transmit
*                              mode is disabled.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfLocalDeviceSystemDescGet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum, 
                                                     open_buffdesc *systemDesc);

/*********************************************************************
* @purpose  Gets the LLDP Local Device Port Description 
*           for the specified interface.
*           
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    portDesc         @b{(output)} Local Device Port Description
*
* @notes   Maximum size for portDesc is determined by calling
*          openapiLldpMgmtStrLenGet().
*
* @returns  OPEN_E_NONE        Port Description get successful.
* @returns  OPEN_E_FAIL        Port Description get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface or
*                              system capabilities are not available as transmit
*                              mode is disabled.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfLocalDevicePortDescGet(openapiClientHandle_t *client_handle,
                                                   uint32_t ifNum, 
                                                   open_buffdesc *portDesc);

/*********************************************************************
* @purpose  Gets the LLDP Local Device System Capabilities Supported 
*           for the specified interface.
*           
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    sysCapSupp       @b{(output)} Local Device System Capabilities Supported
*
* @notes   Maximum size for sysCapSupp is determined by calling
*          openapiLldpMgmtStrLenGet().
*
* @returns  OPEN_E_NONE        Supported Capabilities get successful.
* @returns  OPEN_E_FAIL        Supported Capabilities get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface or
*                              system capabilities are not available as transmit
*                              mode is disabled.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfLocalDeviceSysCapSuppGet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum, 
                                                     open_buffdesc *sysCapSupp);

/*********************************************************************
* @purpose  Gets the LLDP Local Device System Capabilities Enabled 
*           for the specified interface.
*           
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    sysCapEnabled    @b{(output)} Local Device System Capabilities Enabled
*
* @notes   Maximum size for sysCapEnabled is determined by calling
*          openapiLldpMgmtStrLenGet().
*
* @returns  OPEN_E_NONE        Enabled Capabilities get successful.
* @returns  OPEN_E_FAIL        Supported Capabilities get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface or
*                              system capabilities are not available as transmit
*                              mode is disabled.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfLocalDeviceSysCapEnabledGet(openapiClientHandle_t *client_handle,
                                                        uint32_t ifNum, 
                                                        open_buffdesc *sysCapEnabled);

/*********************************************************************
* @purpose  Gets the LLDP Local Device Management Address 
*           for the specified interface.
*           
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    addressFamily    @b{(inout)}  Address Family
* @param    address          @b{(inout)}  Address
*
* @notes   Maximum size for address is determined by calling
*          openapiLldpMgmtStrLenGet().
*
* @returns  OPEN_E_NONE        Management Address get successful.
* @returns  OPEN_E_FAIL        Management Address get failed.
* @returns  OPEN_E_UNAVAIL     LLDP not supported on this interface or
*                              system capabilities are not available as transmit
*                              mode is disabled.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfLocalDeviceMgmtAddressGet(openapiClientHandle_t *client_handle,
                                                      uint32_t ifNum, 
                                                      OPEN_LLDP_IANA_ADDR_FAMILY_t *addressFamily,
                                                      open_buffdesc *address);

/*********************************************************************
* @purpose  Get the interface LLDP-MED administrative mode.
* 
* @param    client_handle    @b{(input)}  client handle from registration API 
* @param    intf             @b{(input)}  ID of the interface
* @param    mode             @b{(output)} lldp mode 
*                            Possible values @ref OPEN_CONTROL_t
* 
* @returns  OPEN_E_NONE      Get successful.
* @returns  OPEN_E_FAIL      Get failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.14
* 
* @end
*********************************************************************/
open_error_t openapiLldpMedIntfModeGet(openapiClientHandle_t *client_handle,
                                       uint32_t intf,
                                       OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Set LLDP-MED administrative mode on valid LLDP interfaces.
* 
* @param    client_handle    @b{(input)}  client handle from registration API 
* @param    intf             @b{(input)}  ID of the interface or
*                            @ref OPEN_ALL_INTERFACES for all interfaces
* @param    mode             @b{(input)}  lldp mode 
*                            Possible values @ref OPEN_CONTROL_t
* 
* @returns  OPEN_E_NONE      Set successful.
* @returns  OPEN_E_FAIL      Set failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Administrative mode is set on all valid LLDP interfaces, if 
*           @ref OPEN_ALL_INTERFACES is passed in for the 'intf' parameter.
* 
* @supported-in-version OpEN API Version: 1.14
* 
* @end
*********************************************************************/
open_error_t openapiLldpMedIntfModeSet(openapiClientHandle_t *client_handle,
                                       uint32_t intf,
                                       OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose  Get the interface LLDP-MED configuration notification mode.
* 
* @param    client_handle    @b{(input)}  client handle from registration API 
* @param    intf             @b{(input)}  ID of the interface
* @param    mode             @b{(output)} notification mode 
*                            Possible values @ref OPEN_BOOL_t
* 
* @returns  OPEN_E_NONE      Get successful.
* @returns  OPEN_E_FAIL      Get failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.14
* 
* @end
*********************************************************************/
open_error_t openapiLldpMedIntfConfigNotificationModeGet(openapiClientHandle_t *client_handle,
                                                         uint32_t intf,
                                                         OPEN_BOOL_t *mode);

/*********************************************************************
* @purpose  Set LLDP-MED configuration notification mode on valid LLDP interfaces.
* 
* @param    client_handle    @b{(input)}  client handle from registration API 
* @param    intf             @b{(input)}  ID of the interface or
*                            @ref OPEN_ALL_INTERFACES for all interfaces
* @param    mode             @b{(input)}  notification mode 
*                            Possible values @ref OPEN_BOOL_t
* 
* @returns  OPEN_E_NONE      Set successful.
* @returns  OPEN_E_FAIL      Set failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Configuration notification mode is set on all valid LLDP interfaces, 
*           if @ref OPEN_ALL_INTERFACES is passed in for the 'intf' parameter.
* 
* @supported-in-version OpEN API Version: 1.14
* 
* @end
*********************************************************************/
open_error_t openapiLldpMedIntfConfigNotificationModeSet(openapiClientHandle_t *client_handle,
                                                         uint32_t intf,
                                                         OPEN_BOOL_t mode);

/*********************************************************************
* @purpose  Get the interface LLDP-MED TLVs enabled for transmision.
* 
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    intf             @b{(input)}  ID of the interface
* @param    cap              @b{(output)} bitmask of TLVs enabled
* 
* @returns  OPEN_E_NONE      Get successful.
* @returns  OPEN_E_FAIL      Get failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @notes    The following bitmask values are provided for convenience. 
*           - Bitmask value @ref OPEN_LLDP_MED_CAP_CAPABILITIES_BITMASK
*           - Bitmask value @ref OPEN_LLDP_MED_CAP_NETWORKPOLICY_BITMASK
*           - Bitmask value @ref OPEN_LLDP_MED_CAP_LOCATION_BITMASK
*           - Bitmask value @ref OPEN_LLDP_MED_CAP_EXT_PSE_BITMASK
*           - Bitmask value @ref OPEN_LLDP_MED_CAP_EXT_PD_BITMASK
*           - Bitmask value @ref OPEN_LLDP_MED_CAP_INVENTORY_BITMASK
* 
* @supported-in-version OpEN API Version: 1.14
* 
* @end
*********************************************************************/
open_error_t openapiLldpMedIntfTransmitTlvGet(openapiClientHandle_t *client_handle,
                                              uint32_t intf,
                                              OPEN_LLDP_MED_CAPABILITIES_t *cap);

/*********************************************************************
* @purpose  Set LLDP-MED TLVs for transmision on valid LLDP interfaces.
* 
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    intf             @b{(input)}  ID of the interface or
*                            @ref OPEN_ALL_INTERFACES for all interfaces
* @param    cap              @b{(input)}  bitmask of TLVs to be enabled
* 
* @returns  OPEN_E_NONE      Set successful.
* @returns  OPEN_E_FAIL      Set failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @notes    Calling this API will change the running configuration of the switch.
* @notes    TLV bitmask for transmission is set on all valid LLDP interfaces, 
*           if @ref OPEN_ALL_INTERFACES is passed in for the 'intf' parameter.
* @notes    The following bitmask values are provided for convenience. 
*           - Bitmask value @ref OPEN_LLDP_MED_CAP_CAPABILITIES_BITMASK
*           - Bitmask value @ref OPEN_LLDP_MED_CAP_NETWORKPOLICY_BITMASK
*           - Bitmask value @ref OPEN_LLDP_MED_CAP_LOCATION_BITMASK
*           - Bitmask value @ref OPEN_LLDP_MED_CAP_EXT_PSE_BITMASK
*           - Bitmask value @ref OPEN_LLDP_MED_CAP_EXT_PD_BITMASK
*           - Bitmask value @ref OPEN_LLDP_MED_CAP_INVENTORY_BITMASK
* 
* @supported-in-version OpEN API Version: 1.14
* 
* @end
*********************************************************************/
open_error_t openapiLldpMedIntfTransmitTlvSet(openapiClientHandle_t *client_handle,
                                              uint32_t intf,
                                              OPEN_LLDP_MED_CAPABILITIES_t *cap);

/*********************************************************************
* @purpose  Get the LLDP-MED capabilities received from the remote device.
*
* @param    client_handle          @b{(input)}  client handle from registration API
* @param    intf                   @b{(input)}  ID of the interface
* @param    remIndex               @b{(input)}  Remote Device Index.
*                                               Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    timeStamp              @b{(input)}  Remote Device entry timestamp.
*                                               Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    supportedCapabilities  @b{(output)} bitmask of TLVs supported
* @param    enabledCapabilities    @b{(output)} bitmask of TLVs enabled
*
* @returns  OPEN_E_NONE      Get successful.
* @returns  OPEN_E_FAIL      Get failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @notes    The following bitmask values are provided for convenience.
*           - Bitmask value @ref OPEN_LLDP_MED_CAP_CAPABILITIES_BITMASK
*           - Bitmask value @ref OPEN_LLDP_MED_CAP_NETWORKPOLICY_BITMASK
*           - Bitmask value @ref OPEN_LLDP_MED_CAP_LOCATION_BITMASK
*           - Bitmask value @ref OPEN_LLDP_MED_CAP_EXT_PSE_BITMASK
*           - Bitmask value @ref OPEN_LLDP_MED_CAP_EXT_PD_BITMASK
*           - Bitmask value @ref OPEN_LLDP_MED_CAP_INVENTORY_BITMASK
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiLldpMedRemoteCapabilitiesGet(openapiClientHandle_t *client_handle,
                                                 uint32_t intf, uint32_t remIndex, uint32_t timeStamp,
                                                 OPEN_LLDP_MED_CAPABILITIES_t *supportedCapabilities,
                                                 OPEN_LLDP_MED_CAPABILITIES_t *enabledCapabilities);

/*********************************************************************
* @purpose  Get the LLDP-MED device class of the remote device.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    intf             @b{(input)}  ID of the interface
* @param    remIndex         @b{(input)}  Remote Device Index.
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    timeStamp        @b{(input)}  Remote Device entry timestamp.
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    deviceClass      @b{(output)} class of the remote device
*
* @returns  OPEN_E_NONE      Get successful.
* @returns  OPEN_E_FAIL      Get failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiLldpMedRemoteDeviceClassGet(openapiClientHandle_t *client_handle,
                                                uint32_t intf, uint32_t remIndex, uint32_t timeStamp,
                                                uint32_t *deviceClass);

/*********************************************************************
* @purpose  Get the size in bytes for the open_buffdesc used to call
*           @ref openapiLldpMedRemoteModelNameGet().
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    size             @b{(output)} number of bytes to allocate in buffer
*
* @returns  OPEN_E_NONE      Get successful.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiLldpMedRemoteModelNameSizeGet(openapiClientHandle_t *client_handle,
                                                  uint32_t *size);

/*********************************************************************
* @purpose  Get the LLDP-MED model name of the remote device.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    intf             @b{(input)}  ID of the interface
* @param    remIndex         @b{(input)}  Remote Device Index.
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    timeStamp        @b{(input)}  Remote Device entry timestamp.
*                                         Use openapiLldpIntfRemoteDeviceNextGet() to get the value
* @param    modelName        @b{(output)} model name of the remote device
*
* @returns  OPEN_E_NONE      Get successful.
* @returns  OPEN_E_FAIL      Get failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @note Size of buffer is returned by @ref openapiLldpMedRemoteModelNameSizeGet(). Using
*       a smaller buffer may result in truncated data being returned.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiLldpMedRemoteModelNameGet(openapiClientHandle_t *client_handle,
                                              uint32_t intf, uint32_t remIndex, uint32_t timeStamp,
                                              open_buffdesc *modelName);

/*****************************************************************************
 * @purpose   Gets the 802.1AB global transmit interval.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     interval       @b{(output)} Interval in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpTxIntervalGet(openapiClientHandle_t *client_handle, uint32_t *interval);

/*****************************************************************************
 * @purpose   Sets the 802.1AB global transmit interval.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     interval       @b{(input)}  Interval in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpTxIntervalSet(openapiClientHandle_t *client_handle, uint32_t interval);

/*****************************************************************************
 * @purpose   Gets the 802.1AB global transmit hold muliplier.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     holdCount      @b{(output)} Hold multiplier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpTxHoldGet(openapiClientHandle_t *client_handle, uint32_t *holdCount);

/*****************************************************************************
 * @purpose   Sets the 802.1AB global transmit hold muliplier.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     holdCount      @b{(input)}  Hold multiplier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpTxHoldSet(openapiClientHandle_t *client_handle, uint32_t holdCount);

/*****************************************************************************
 * @purpose   Gets the 802.1AB global transmit reinit delay.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     delay          @b{(output)} Delay in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpTxReinitDelayGet(openapiClientHandle_t *client_handle, uint32_t *delay);

/*****************************************************************************
 * @purpose   Sets the 802.1AB global transmit reinit delay.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     delay          @b{(input)}  Delay in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpTxReinitDelaySet(openapiClientHandle_t *client_handle, uint32_t delay);

/*****************************************************************************
 * @purpose   Gets the 802.1AB global notification interval limit.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     interval       @b{(output)} Interval in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpNotificationIntervalGet(openapiClientHandle_t *client_handle, 
                                                uint32_t *interval);

/*****************************************************************************
 * @purpose   Sets the 802.1AB global notification interval limit.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     interval       @b{(input)}  Interval in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpNotificationIntervalSet(openapiClientHandle_t *client_handle, 
                                                uint32_t interval);

/*****************************************************************************
 * @purpose   Gets the fast start repeat count.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     count          @b{(output)} Fast start repeat count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpXMedFastStartRepeatCountGet(openapiClientHandle_t *client_handle, 
                                                    uint32_t *count);

/*****************************************************************************
 * @purpose   Sets the fast start repeat count.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     count          @b{(input)}  Fast start repeat count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpXMedFastStartRepeatCountSet(openapiClientHandle_t *client_handle, 
                                                    uint32_t count);

/*****************************************************************************
 * @purpose   Gets the 802.1AB mgmt addr transmit mode for the specified interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     mode           @b{(output)} Transmit mgmt addr TLV mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpIntfTxMgmtAddrModeGet(openapiClientHandle_t *client_handle, 
                                              uint32_t ifNum, OPEN_BOOL_t *mode);

/*****************************************************************************
 * @purpose   Sets the 802.1AB mgmt addr transmit mode for the specified interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     mode           @b{(input)}  Transmit mgmt addr TLV mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpIntfTxMgmtAddrModeSet(openapiClientHandle_t *client_handle, 
                                              uint32_t ifNum, OPEN_BOOL_t mode);

/*****************************************************************************
 * @purpose   Gets the vlan id associated with a particular policy type.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     policyAppType  @b{(input)}  Policy application type
 * @param     vlanId         @b{(output)} Vlan Id
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpXMedLocMediaPolicyVlanIDGet(openapiClientHandle_t *client_handle, 
                                                    uint32_t ifNum, 
                                                    OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType, 
                                                    uint32_t *vlanId);

/*****************************************************************************
 * @purpose   Gets the tagged bit associated with a particular policy type.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     policyAppType  @b{(input)}  Policy application type
 * @param     taggedStatus   @b{(output)} Tagged Status
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpXMedLocMediaPolicyTaggedGet(openapiClientHandle_t *client_handle, 
                                                    uint32_t ifNum, 
                                                    OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType, 
                                                    OPEN_BOOL_t *taggedStatus);

/*****************************************************************************
 * @purpose   Gets the priority associated with a particular policy type.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     policyAppType  @b{(input)}  Policy application type
 * @param     priority       @b{(output)} Priority
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpXMedLocMediaPolicyPriorityGet(openapiClientHandle_t *client_handle, 
                                                      uint32_t ifNum, 
                                                      OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType, 
                                                      uint32_t *priority);

/*****************************************************************************
 * @purpose   Gets the DSCP associated with a particular policy type.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     policyAppType  @b{(input)}  Policy application type
 * @param     dscp           @b{(output)} Dscp vlaue
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpXMedLocMediaPolicyDscpGet(openapiClientHandle_t *client_handle, 
                                                  uint32_t ifNum, 
                                                  OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType, 
                                                  uint32_t *dscp);

/*****************************************************************************
 * @purpose   Get the first interface enable for LLDP receives.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(output)} First valid interface 
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpRemoteIntfFirstGet (openapiClientHandle_t *client_handle, 
                                            uint32_t *ifNum); 
 
/*****************************************************************************
 * @purpose   Get the next valid interface enabled for LLDP receives.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  First valid interface number
 * @param     nextIfNum      @b{(output)} Next valid interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpRemoteIntfNextGet (openapiClientHandle_t *client_handle, 
                                           uint32_t ifNum, uint32_t *nextIfNum); 

/*****************************************************************************
 * @purpose   Get remote port's auto negotiation capability.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     isCapable      @b{(output)} Remote port auto negotiation capability 
 *                                        status
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpdot3RemPortAutoNegSupportedGet(openapiClientHandle_t *client_handle, 
                                                       uint32_t ifNum, 
                                                       uint32_t remIndex, 
                                                       uint32_t timestamp, 
                                                       OPEN_BOOL_t *isCapable);

/*****************************************************************************
 * @purpose   Get remote port's auto negotiation enabled status.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     isEnabled      @b{(output)} Remote port auto negotiation enabled 
 *                                        status
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpdot3RemPortAutoNegEnabledGet(openapiClientHandle_t *client_handle, 
                                                     uint32_t ifNum, 
                                                     uint32_t remIndex, 
                                                     uint32_t timestamp, 
                                                     OPEN_BOOL_t *isEnabled);

/*****************************************************************************
 * @purpose   Get remote port's auto negotiation advertized capability as a string.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     advrtCap       @b{(output)} Remote port auto negotiation advertized 
 *                                        capability
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpdot3RemPortAutoNegAdvertizedCapGet(openapiClientHandle_t *client_handle, 
                                                           uint32_t ifNum, 
                                                           uint32_t remIndex, 
                                                           uint32_t timestamp, 
                                                           open_buffdesc *advrtCap);

/*****************************************************************************
 * @purpose   Get remote port's auto negotiation operational MAU Type.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     type           @b{(output)} MAU type
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpdot3RemPortAutoNegOperMauTypeGet(openapiClientHandle_t *client_handle, 
                                                         uint32_t ifNum, 
                                                         uint32_t remIndex, 
                                                         uint32_t timestamp, 
                                                         uint32_t *type);

/*****************************************************************************
 * @purpose   Get remote port's auto negotiation operational MAU in string format.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     mauType        @b{(input)}  MAU Type
 * @param     mauString      @b{(output)} MAU type in string
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpdot3RemPortAutoNegOperMauStringGet(openapiClientHandle_t *client_handle, 
                                                           uint32_t mauType, 
                                                           open_buffdesc *mauString);

/*****************************************************************************
 * @purpose   Get remote device's hardware revision.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     hardwareRev    @b{(output)} Remote device hardware revision
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemHardwareRevGet(openapiClientHandle_t *client_handle, 
                                             uint32_t ifNum, uint32_t remIndex, 
                                             uint32_t timestamp, 
                                             open_buffdesc *hardwareRev);

/*****************************************************************************
 * @purpose   Get remote device's firmware revision.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     firmwareRev    @b{(output)} Remote device firmware revision
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemFirmwareRevGet(openapiClientHandle_t *client_handle, 
                                             uint32_t ifNum, uint32_t remIndex, 
                                             uint32_t timestamp, 
                                             open_buffdesc *firmwareRev);

/*****************************************************************************
 * @purpose   Get remote device's software revision.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     softwareRev    @b{(output)} Remote device software revision
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemSoftwareRevGet(openapiClientHandle_t *client_handle, 
                                             uint32_t ifNum, uint32_t remIndex, 
                                             uint32_t timestamp, 
                                             open_buffdesc *softwareRev);

/*****************************************************************************
 * @purpose   Get remote device's serial number.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     serialNum      @b{(output)} Remote device serial number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemSerialNumGet(openapiClientHandle_t *client_handle, 
                                           uint32_t ifNum, uint32_t remIndex, 
                                           uint32_t timestamp, 
                                           open_buffdesc *serialNum);

/*****************************************************************************
 * @purpose   Get remote device's asset ID.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     assetId        @b{(output)} Remote device asset ID
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemAssetIdGet(openapiClientHandle_t *client_handle, 
                                         uint32_t ifNum, uint32_t remIndex, 
                                         uint32_t timestamp, 
                                         open_buffdesc *assetId);

/*****************************************************************************
 * @purpose   Get remote device's PoE device type connected to this port.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     deviceType     @b{(output)} Remote device POE device type
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemPoeDeviceTypeGet(openapiClientHandle_t *client_handle, 
                                               uint32_t ifNum, uint32_t remIndex, 
                                               uint32_t timestamp, 
                                               uint32_t *deviceType);

/*****************************************************************************
 * @purpose   Get remote ports PSE power value in tenths of watts.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     powerAv        @b{(output)} Available power
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemPoePSEPowerAvGet(openapiClientHandle_t *client_handle, 
                                               uint32_t ifNum, uint32_t remIndex, 
                                               uint32_t timestamp, 
                                               uint32_t *powerAv);

/*****************************************************************************
 * @purpose   Get remote ports PSE power source.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     powerSrcType   @b{(output)} Power soruce type
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemPoePSEPowerSrcGet(openapiClientHandle_t *client_handle,
                                                uint32_t ifNum, uint32_t remIndex, 
                                                uint32_t timestamp, 
                                                uint32_t *powerSrcType);

/*****************************************************************************
 * @purpose   Get remote ports PSE power priority.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote
 *                                        entry
 * @param     powerPri       @b{(output)} Power priority
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemPoePSEPowerPriGet(openapiClientHandle_t *client_handle, 
                                                uint32_t ifNum, uint32_t remIndex, 
                                                uint32_t timestamp, 
                                                uint32_t *powerPri);

/*****************************************************************************
 * @purpose   Get the location info received on this port for the given location 
 *            subtype.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     subType        @b{(input)}  Loaction subtype
 * @param     locInfo        @b{(output)} Location information
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemLocationInfoGet(openapiClientHandle_t *client_handle,
                                              uint32_t ifNum, uint32_t remIndex, 
                                              uint32_t timestamp, 
                                              OPEN_LLDPXMED_LOCATION_SUBTYPE_t subType, 
                                              open_buffdesc *locInfo);

/*****************************************************************************
 * @purpose   Get the vlan id associated with a particular policy type.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     policyAppType  @b{(input)}  Type of application policy
 * @param     vlanId         @b{(output)} Location information
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemMediaPolicyVlanIdGet(openapiClientHandle_t *client_handle, 
                                                   uint32_t ifNum, 
                                                   uint32_t remIndex, 
                                                   uint32_t timestamp, 
                                                   OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType, 
                                                   uint32_t *vlanId);

/*****************************************************************************
 * @purpose   Get the priority associated with a particular policy type.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     policyAppType  @b{(input)}  Type of application policy
 * @param     priority       @b{(output)} Priority
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemMediaPolicyPriorityGet(openapiClientHandle_t *client_handle, 
                                                     uint32_t ifNum, 
                                                     uint32_t remIndex, 
                                                     uint32_t timestamp, 
                                                     OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType, 
                                                     uint32_t *priority);

/*****************************************************************************
 * @purpose   Get the DSCP associated with a particular policy type.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     policyAppType  @b{(input)}  Type of application policy
 * @param     dscp           @b{(output)} DSCP value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemMediaPolicyDscpGet(openapiClientHandle_t *client_handle, 
                                                 uint32_t ifNum, uint32_t remIndex,
                                                 uint32_t timestamp, 
                                                 OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType, 
                                                 uint32_t *dscp);

/*****************************************************************************
 * @purpose   Get the unknown bit associated with a particular policy type.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     policyAppType  @b{(input)}  Type of application policy
 * @param     unknown        @b{(output)} Unknown bit status
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemMediaPolicyUnknownGet(openapiClientHandle_t *client_handle, 
                                                    uint32_t ifNum, 
                                                    uint32_t remIndex, 
                                                    uint32_t timestamp, 
                                                    OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType, 
                                                    OPEN_BOOL_t *unknown);

/*****************************************************************************
 * @purpose   Get the tagged bit associated with a particular policy type.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     policyAppType  @b{(input)}  Type of application policy
 * @param     tagged         @b{(output)} Tagged bit status
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemMediaPolicyTaggedGet(openapiClientHandle_t *client_handle, 
                                                   uint32_t ifNum, 
                                                   uint32_t remIndex, 
                                                   uint32_t timestamp, 
                                                   OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType, 
                                                   OPEN_BOOL_t *tagged);

/*****************************************************************************
 * @purpose   Get the next unknown TLV entry for the indicated remote data entry.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     tlvType        @b{(input)}  TLV type
 * @param     nextTlvType    @b{(output)} Next TLV type
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     if tlvType is 0, this API will return first entry. 
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpRemUnknownTLVEntryNextGet(openapiClientHandle_t *client_handle, 
                                                  uint32_t ifNum, uint32_t remIndex, 
                                                  uint32_t timestamp, 
                                                  uint32_t tlvType, 
                                                  uint32_t *nextTlvType);

/*****************************************************************************
 * @purpose   Determine if unknown TLV entry exists.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     tlvType        @b{(input)}  TLV type
 * @param     isExists       @b{(output)} Entry existtance status
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpRemUnknownTLVEntryIsExists(openapiClientHandle_t *client_handle, 
                                                   uint32_t ifNum, 
                                                   uint32_t remIndex, 
                                                   uint32_t timestamp, 
                                                   uint32_t tlvType, 
                                                   OPEN_BOOL_t *isExists);

/*****************************************************************************
 * @purpose   Get the TLV info for the specified remote entry.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     tlvType        @b{(input)}  TLV type
 * @param     info           @b{(output)} Organization defined info
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpRemUnknownTLVInfoGet(openapiClientHandle_t *client_handle, 
                                             uint32_t ifNum, 
                                             uint32_t remIndex, 
                                             uint32_t timestamp, 
                                             uint32_t tlvType, 
                                             open_buffdesc *info);

/*****************************************************************************
 * @purpose   Get the remote device's manufacturer's name.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     mfgName        @b{(output)} Manufacturer's name as a string
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemMfgNameGet(openapiClientHandle_t *client_handle, 
                                         uint32_t ifNum, uint32_t remIndex, 
                                         uint32_t timestamp, 
                                         open_buffdesc *mfgName);

/*****************************************************************************
 * @purpose   Get the remote device's model name.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     modelName      @b{(output)} Model name as a string
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemModelNameGet(openapiClientHandle_t *client_handle, 
                                           uint32_t ifNum, uint32_t remIndex, 
                                           uint32_t timestamp, 
                                           open_buffdesc *modelName);

/*****************************************************************************
 * @purpose   Get the remote port's PD power requirement.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     powerReq       @b{(output)} Power requirement
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemPoePDPowerReqGet(openapiClientHandle_t *client_handle, 
                                               uint32_t ifNum, uint32_t remIndex, 
                                               uint32_t timestamp, 
                                               uint32_t *powerReq);

/*****************************************************************************
 * @purpose   Get the remote port's PD power source.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     powerSrc       @b{(output)} Power source
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemPoePDPowerSrcGet(openapiClientHandle_t *client_handle, 
                                               uint32_t ifNum, uint32_t remIndex, 
                                               uint32_t timestamp, 
                                               uint32_t *powerSrc);

/*****************************************************************************
 * @purpose   Get the remote port's PD power priority.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     powerPri       @b{(output)} Power priority
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedRemPoePDPowerPriGet(openapiClientHandle_t *client_handle, 
                                               uint32_t ifNum, uint32_t remIndex, 
                                               uint32_t timestamp, 
                                               uint32_t *powerPri);

/*****************************************************************************
 * @purpose   Get the organizationally defined OUI for specified entry.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     infoIndex      @b{(input)}  Info index
 * @param     oui            @b{(output)} Organization unique identifier in 
 *                                        XX:XX:XX format
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpRemOrgDefOUIGet(openapiClientHandle_t *client_handle, 
                                        uint32_t ifNum, uint32_t remIndex, 
                                        uint32_t timestamp, uint32_t infoIndex, 
                                        open_buffdesc *oui);

/*****************************************************************************
 * @purpose   Get the organizationally defined subtype for specified entry.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     remIndex       @b{(input)}  Index associated with this remote entry
 * @param     timestamp      @b{(input)}  Time Stamp associated with this remote 
 *                                        entry
 * @param     infoIndex      @b{(input)}  Info index
 * @param     subtype        @b{(output)} Organization defined subtype
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpRemOrgDefSubtypeGet(openapiClientHandle_t *client_handle, 
                                            uint32_t ifNum, uint32_t remIndex, 
                                            uint32_t timestamp, uint32_t infoIndex, 
                                            uint32_t *subtype);

/*****************************************************************************
 * @purpose   Get a string representation for Chassis ID Subtype.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     subtype        @b{(input)}  Chassis ID subtype
 * @param     subtypeStr     @b{(output)} Subtype string
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpChassisIdSubtypeStringGet(openapiClientHandle_t *client_handle, 
                                                  OPEN_LLDP_CHASSIS_ID_SUBTYPE_t subtype, 
                                                  open_buffdesc *subtypeStr);

/*****************************************************************************
 * @purpose   Get a string representation for port ID Subtype.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     subtype        @b{(input)}  Port ID subtype
 * @param     subtypeStr     @b{(output)} Subtype string
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpPortIdSubtypeStringGet(openapiClientHandle_t *client_handle, 
                                               uint32_t subtype, 
                                               open_buffdesc *subtypeStr);

/*********************************************************************
* @purpose  Gets the LLDP Remote Device management address entry details 
*           for the specified interface.
*           
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    remIndex         @b{(input)}  Remote Device Index
*                                         Use openapiLldpIntfRemoteDeviceNextGet() 
*                                         to get the value
* @param    timeStamp        @b{(input)}  Remote Device entry timestamp
*                                         Use openapiLldpIntfRemoteDeviceNextGet() 
*                                         to get the value
* @param    addressFamily    @b{(inout)}  Address Family. Pass 0 for the first time.
* @param    address          @b{(inout)}  Address
*                                         Initialize portDesc:pstart and size with a 
*                                         char buffer of length returned by 
*                                         openapiLldpMgmtStrLenGet()
* @param    addrLen          @b{(inout)}  Address Length. Passing 0 will return
*                                         the first address
* @param    subType          @b{(output)} Interface subtype 
* @param    intfId           @b{(output)} Interface ID 
*
* @notes   Maximum size for address is determined by calling
*          openapiLldpMgmtStrLenGet().
* 
* @returns  OPEN_E_NONE        Management Address get successful.
* @returns  OPEN_E_FAIL        Management Address get failed.
* @returns  OPEN_E_UNAVAIL     Next Management Address not available.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiLldpIntfRemoteDeviceEntryNextGet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum,
                                                     uint32_t remIndex, 
                                                     uint32_t timeStamp,
                                                     OPEN_LLDP_IANA_ADDR_FAMILY_t  *addressFamily,
                                                     open_buffdesc *address, 
                                                     uint8_t *addrLen,
                                                     OPEN_LLDP_MAN_ADDR_IF_SUBTYPE_t *subType,
                                                     uint32_t *intfId);

/*****************************************************************************
 * @purpose   Get the port's auto negotiation capability.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     isCapable      @b{(output)} Interface auto negotiation capability 
 *                                        status
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpdot3LocPortAutoNegSupportedGet(openapiClientHandle_t *client_handle, 
                                                       uint32_t ifNum, 
                                                       OPEN_BOOL_t *isCapable);

/*****************************************************************************
 * @purpose   Get port's auto negotiation enabled status.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     isEnabled      @b{(output)} Interface auto negotiation enabled status
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpdot3LocPortAutoNegEnabledGet(openapiClientHandle_t *client_handle, 
                                                     uint32_t ifNum, 
                                                     OPEN_BOOL_t *isEnabled);

/*****************************************************************************
 * @purpose   Get port's auto negotiation advertized capability as a string.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     advrtCap       @b{(output)} Interface auto negotiation advertized 
 *                                        capability
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpdot3LocPortAutoNegAdvertizedCapGet(openapiClientHandle_t *client_handle, 
                                                           uint32_t ifNum, 
                                                           open_buffdesc *advrtCap);

/*****************************************************************************
 * @purpose   Get port's auto negotiation operational MAU Type.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     type           @b{(output)} MAU type
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpdot3LocPortAutoNegOperMauTypeGet(openapiClientHandle_t *client_handle, 
                                                         uint32_t ifNum, 
                                                         uint32_t *type);

/*****************************************************************************
 * @purpose   Get local device's MED Classification
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     deviceClass    @b{(output)} Device class
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLldpMedLocDeviceClassGet(openapiClientHandle_t *client_handle, 
                                             uint32_t *deviceClass);

#endif /* OPENAPI_LLDP_H_INCLUDED */
