/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_logging.h
*
* @purpose   Logging Configuration and Status 
*
* @component OpEN
*
* @create    28/01/2013
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_LOGGING_H_INCLUDED
#define OPENAPI_LOGGING_H_INCLUDED

#include "openapi_common.h"

/** OpEN uses this enumeration to define Logging Severity Levels*/
typedef enum
{ 
  OPEN_LOG_SEVERITY_UNDEFINED = -1, /**< Severity level undefined */
  OPEN_LOG_SEVERITY_EMERGENCY = 0, /**< Emergency conditions */
  OPEN_LOG_SEVERITY_ALERT     = 1, /**< Alert conditions */
  OPEN_LOG_SEVERITY_CRITICAL  = 2, /**< Critical conditions */
  OPEN_LOG_SEVERITY_ERROR     = 3, /**< Error conditions */
  OPEN_LOG_SEVERITY_WARNING   = 4, /**< Warning conditions */
  OPEN_LOG_SEVERITY_NOTICE    = 5, /**< Normal but significant conditions */
  OPEN_LOG_SEVERITY_INFO      = 6, /**< Informational messages */
  OPEN_LOG_SEVERITY_DEBUG     = 7  /**< Debug-level messages */

} OPEN_LOG_SEVERITY_t;

typedef enum
{
  OPEN_USB_LOGGING_OVERWRITE_ON_FULL = 1, /**< Log into USB drive, stop when no space left */
  OPEN_USB_LOGGING_STOP_ON_FULL,          /**< Log into USB drive, overwrite when no space left */
  OPEN_USB_LOGGING_DISABLE,               /**< Disable USB logging */ 
} OPEN_USB_LOGGING_MODE_t;

typedef enum
{
  OPEN_USB_LOGGING_STATUS_ACTIVE = 1, /**< USB logging operational status Active */
  OPEN_USB_LOGGING_STATUS_INACTIVE,   /**< USB logging operational status Inactive */ 
} OPEN_USB_LOGGING_OPER_STATUS_t;

typedef enum
{
  OPEN_MEMORY_WRAP_BEHAVIOR,         /**< Buffered logging behavior Wrap on full */
  OPEN_MEMORY_STOP_ON_FULL_BEHAVIOR  /**< Buffered logging behavior Stop on full */
} OPEN_MEMORY_BEHAVIOR_t;

#define OPENAPI_LOG_FORMAT_BUF_SIZE 512

#define OPENAPI_LOG_ENTRY_BUF_SIZE (OPENAPI_LOG_FORMAT_BUF_SIZE + 80)

#define OPENAPI_TRAP_ENTRY_SIZE (300)

#define OPENAPI_LOG_MAX_PERSISTENT_LOGS (3)

#define OPENAPI_LOG_PERSISTENT_FILE_INDEX_MIN (0)
#define OPENAPI_LOG_PERSISTENT_FILE_INDEX_MAX (2)

typedef struct
{
  char trapEntry[OPENAPI_TRAP_ENTRY_SIZE];
} openapiTrapLogEntry_t;

#define OPENAPI_TRAP_LOG_BUF_SIZE (256 * sizeof(openapiTrapLogEntry_t))


/*********************************************************************
* @purpose  Enable or disable CLI command logging.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    cmdLoggingMode  @b{(input)}  CLI command logging mode
*
* @returns  OPEN_E_NONE     if CLI command logging mode is set successfully.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/ 
open_error_t openapiCliCmdLoggingModeSet(openapiClientHandle_t *client_handle,
                                         OPEN_CONTROL_t cmdLoggingMode);

/*********************************************************************
* @purpose  Get CLI command logging mode.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    cmdLoggingMode  @b{(output)} CLI command logging mode
*
* @returns  OPEN_E_NONE     if CLI command logging mode is returned successfully.
* @returns  OPEN_E_FAIL     if invalid mode is returned.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiCliCmdLoggingModeGet(openapiClientHandle_t *client_handle,
                                         OPEN_CONTROL_t *cmdLoggingMode);

/*********************************************************************
* @purpose  Set Console Logging mode.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    loggingConsoleMode  @b{(input)}  Console Logging mode
*
* @returns  OPEN_E_NONE     if logging console mode is set successfully.
* @returns  OPEN_E_EXISTS   if the same mode is already configured.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiConsoleLoggingModeSet(openapiClientHandle_t *client_handle,
                                          OPEN_CONTROL_t loggingConsoleMode);

/*********************************************************************
* @purpose  Get Console Logging mode.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    loggingConsoleMode  @b{(output)} Console Logging mode
*
* @returns  OPEN_E_NONE     if logging console mode is returned successfully.
* @returns  OPEN_E_FAIL     if failed to get logging console mode.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiConsoleLoggingModeGet(openapiClientHandle_t *client_handle,
                                          OPEN_CONTROL_t *loggingConsoleMode);

/*********************************************************************
* @purpose  Set Console Logging severity level.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    severityLevel   @b{(input)}  Console Logging severity level
*
* @returns  OPEN_E_NONE     if logging console severity level is set successfully.
* @returns  OPEN_E_FAIL     if failed to set logging console severity level.
* @returns  OPEN_E_INTERNAL if internal error.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* @notes  To set console logging severity level to default pass 0.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiConsoleLoggingSeveritySet(openapiClientHandle_t *client_handle,
                                              OPEN_LOG_SEVERITY_t severityLevel);

/*********************************************************************
* @purpose  Get Console Logging severity level.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    severityLevel   @b{(output)} Console Logging severity level
*
* @returns  OPEN_E_NONE     if logging console severity level is returned.
* @returns  OPEN_E_FAIL     if failed to get logging console severity level.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiConsoleLoggingSeverityGet(openapiClientHandle_t *client_handle,
                                              OPEN_LOG_SEVERITY_t *severityLevel);

/*********************************************************************
* @purpose  Set the persistent logging mode.
*
* @param    client_handle           @b{(input)}  Client handle from registration API
* @param    persistentLoggingMode   @b{(input)}  Persistent logging mode
*
* @returns  OPEN_E_NONE     if persistent logging mode is set.
* @returns  OPEN_E_EXISTS   if the mode is already enabled. 
* @returns  OPEN_E_PARAM    if invalid parameter is passed. 
* @returns  OPEN_E_UNAVAIL  if feature is not supported.
*
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiPersistentLoggingModeSet(openapiClientHandle_t *client_handle,
                                             OPEN_CONTROL_t persistentLoggingMode);

/*********************************************************************
* @purpose  Set persistent logging severity level.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    severityLevel   @b{(input)}  Persistent logging severity level
*
* @returns  OPEN_E_NONE     if persistent logging severity level is set.
* @returns  OPEN_E_FAIL     if failed to set severity level.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL  if feature is not supported.
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  To set the severity level to default value pass @ref OPEN_LOG_SEVERITY_UNDEFINED.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiPersistentLoggingSeverityLevelSet(openapiClientHandle_t *client_handle,
                                                      OPEN_LOG_SEVERITY_t severityLevel);

/*********************************************************************
* @purpose  Get the persistent logging mode.
* 
* @param    client_handle            @b{(input)}  Client handle from registration API
* @param    loggingPersistentMode    @b{(output)} Persistent Logging admin mode
* 
* @returns  OPEN_E_NONE     if persistent logging admin mode is returned.
* @returns  OPEN_E_FAIL     if failed to get persistent logging mode.
* @returns  OPEN_E_PARAM    if invalid parameter is passed. 
* @returns  OPEN_E_UNAVAIL  if feature is not supported.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiPersistentLoggingModeGet(openapiClientHandle_t *client_handle,
                                             OPEN_CONTROL_t *loggingPersistentMode);

/*********************************************************************
* @purpose  Get the persistent logging severity level.
* 
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    severityLevel       @b{(output)} Persistent logging severity level
* 
* @returns  OPEN_E_NONE     if persistent logging severity level is returned.
* @returns  OPEN_E_FAIL     if failed to get persistent logging severity level.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL  if feature is not supported.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiPersistentLoggingSeverityLevelGet(openapiClientHandle_t *client_handle,
                                                      OPEN_LOG_SEVERITY_t *severityLevel);

/*********************************************************************
* @purpose  Get the persistent logging Current Number of Messages in the file.
* 
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    persistentLoggingNumMessages  @b{(output)} Persistent logging Number of Messages
* 
* @returns  OPEN_E_NONE     if persistent logging Current Number of Messages is returned.
* @returns  OPEN_E_FAIL     if failed to get persistent logging Number of Messages.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL  if feature is not supported.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiPersistentLoggingNumberOfCurrentMessagesGet(openapiClientHandle_t *client_handle,
                                                                uint32_t *persistentLoggingNumMessages);

/*********************************************************************
* @purpose  Get the persistent logging Number of Messages in the file.
* 
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    fileIndex           @b{(input)}  Persistent file index. Use fileIndex 0 for Current messages
* @param    persistentLoggingNumMessages  @b{(output)} Persistent logging Number of Messages
* 
* @returns  OPEN_E_NONE     if persistent logging Number of Messages is returned.
* @returns  OPEN_E_FAIL     if failed to get persistent logging Number of Messages.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL  if feature is not supported.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiPersistentLoggingNumberOfPreviousMessagesGet(openapiClientHandle_t *client_handle,
                                                                 uint32_t fileIndex,
                                                                 uint32_t *persistentLoggingNumMessages);

/*********************************************************************
* @purpose  Get the persistent logging next entry in the file.
* 
* @param    client_handle      @b{(input)}  Client handle from registration API
* @param    fileIndex          @b{(input)}  Persistent file index
*                                           (valid indexes are 0, 1 and 2).
* @param    logIndex           @b{(inout)}  Log file index
* @param    bufIndex           @b{(inout)}  Buffer index
* @param    logEntry           @b{(output)} Persistent logging entry
* 
* @returns  OPEN_E_NONE     if persistent logging entry is returned.
* @returns  OPEN_E_FAIL     if failed to get persistent logging entry.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL  if feature is not supported.
*
* @notes    use along with openapiPersistentLoggingNumberOfMessagesGet() to 
*           loop through the messages
*
* @notes    fileIndex 0 is for current logs and 1,2 are for previous logs.
* 
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiPersistentLoggingEntryNextGet(openapiClientHandle_t *client_handle, uint32_t fileIndex, 
                                                  uint32_t *logIndex, uint32_t *bufIndex, open_buffdesc *logEntry);

/*********************************************************************
* @purpose  Get the Number of Trap Logs since last reset.
* 
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    numTrapsSinceLastReset  @b{(output)} Number of Trap Logs since last reset
* 
* @returns  OPEN_E_NONE     if Number of Trap Logs since last reset is returned.
* @returns  OPEN_E_FAIL     if failed to get Number of Trap Logs since last reset.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiTrapLogNumberOfTrapsSinceLastResetGet(openapiClientHandle_t *client_handle,
                                                          uint32_t *numTrapsSinceLastReset);

/*********************************************************************
* @purpose  Get the Number of Trap Logs since last viewed.
* 
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    numTrapsSinceLastViewed  @b{(output)} Number of Trap Logs since last viewed
* 
* @returns  OPEN_E_NONE     if Number of Trap Logs since last viewed is returned.
* @returns  OPEN_E_FAIL     if failed to get Number of Trap Logs since last viewed.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiTrapLogNumberOfTrapsSinceLastViewedGet(openapiClientHandle_t *client_handle,
                                                           uint32_t *numTrapsSinceLastViewed);

/*********************************************************************
* @purpose  Get the Trap Logs Capacity.
* 
* @param    client_handle      @b{(input)}  Client handle from registration API
* @param    trapsLogsCapacity  @b{(output)} Number of Trap Logs Capacity
* 
* @returns  OPEN_E_NONE     if Trap Logs Capacity is returned.
* @returns  OPEN_E_FAIL     if failed to get Trap Logs Capacity.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiTrapLogCapacityGet(openapiClientHandle_t *client_handle,
                                       uint32_t *trapsLogsCapacity);

/*********************************************************************
* @purpose  Set the syslog mode.
* 
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    sysLogMode      @b{(input)}  Syslog mode: enable/disable
*
* @returns  OPEN_E_NONE     if syslog status is set.
* @returns  OPEN_E_EXISTS   if the admin mode is already enabled. 
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSysLogModeSet(openapiClientHandle_t *client_handle,
                                  OPEN_CONTROL_t sysLogMode);

/*********************************************************************
* @purpose  Get syslog mode.
* 
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    sysLogMode     @b{(output)}  Syslog mode: enable/disable
*
* @returns  OPEN_E_NONE     if syslog status is returned.
* @returns  OPEN_E_FAIL     if failed to get syslog status.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes 
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSysLogModeGet(openapiClientHandle_t *client_handle,
                                  OPEN_CONTROL_t *sysLogMode);

/*********************************************************************
* @purpose  Set syslog port.
* 
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    port            @b{(input)}  Syslog port ID
* @param    -               Minimum value @ref OPENAPI_SYSLOG_MIN_PORT
* @param    -               Maximum value @ref OPENAPI_SYSLOG_MAX_PORT
*
* @returns  OPEN_E_NONE     if syslog port is set.
* @returns  OPEN_E_EXISTS   if the port number is already configured.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  To set the syslog port value to default pass 0.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSysLogPortSet(openapiClientHandle_t *client_handle,
                                  uint32_t port);

/*********************************************************************
* @purpose  Get syslog port.
* 
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    port            @b{(output)} Syslog port ID
*
* @returns  OPEN_E_NONE     if syslog port is returned.
* @returns  OPEN_E_FAIL     if failed to get syslog port.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSysLogPortGet(openapiClientHandle_t *client_handle,
                                  uint16_t *port);

/*********************************************************************
* @purpose  Add a logging host.
* 
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    loggingHost     @b{(input)}  DNS Name/IP address of Logging host
*
* @returns  OPEN_E_NONE     if logging host is added.
* @returns  OPEN_E_EXISTS   if host already exists.
* @returns  OPEN_E_FULL     if host table is full.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* @returns  OPEN_E_INTERNAL if internal error has occurred.
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  Maximum size for loggingHost is determined by calling
*         openapiLoggingHostMaxAddrLenGet(). 
* @notes   A valid domain name;
* \n     (i) Must not contain spaces/dots at the end and beginning.
* \n    (ii) Each substring must start with an alphanumeric character.
* \n   (iii) Must not contain subsequent dots/spaces.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLoggingHostAdd(openapiClientHandle_t *client_handle,
                                   open_buffdesc *loggingHost);

/*********************************************************************
* @purpose Set port number to the logging host.
* 
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    loggingHost     @b{(input)}  DNS Name/IP address of Logging host
* @param    port            @b{(input)}  Logging host port ID
* @param    -               Minimum value @ref OPENAPI_TCP_UDP_MIN_PORT
* @param    -               Maximum value @ref OPENAPI_TCP_UDP_MAX_PORT
*
* @returns  OPEN_E_NONE      if port number is set to the logging host.
* @returns  OPEN_E_FAIL      if failed to set port to the logging host.
* @returns  OPEN_E_NOT_FOUND if logging host does not exist.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  Maximum size for loggingHost is determined by calling
*         openapiLoggingHostMaxAddrLenGet(). 
* @notes  To set the logging host port value to default pass 0.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLoggingHostPortSet(openapiClientHandle_t *client_handle,
                                       open_buffdesc *loggingHost, uint32_t port);

/*********************************************************************
* @purpose Set severity level to the logging host.
* 
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    loggingHost     @b{(input)}  DNS Name/IP address of Logging host
* @param    severityLevel   @b{(input)}  Severity Level
*
* @returns  OPEN_E_NONE      if severity level is set to the logging host.
* @returns  OPEN_E_FAIL      if failed to set severity level to the logging host.
* @returns  OPEN_E_NOT_FOUND if logging host does not exist.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  Maximum size for loggingHost is determined by calling
*         openapiLoggingHostMaxAddrLenGet(). 
* @notes  To set the severity level to default value pass the 
*         enum @ref OPEN_LOG_SEVERITY_UNDEFINED.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLoggingHostSeveritySet(openapiClientHandle_t *client_handle,
                                           open_buffdesc *loggingHost, 
                                           OPEN_LOG_SEVERITY_t severityLevel);

/*********************************************************************
* @purpose  Get logging host IP address.
* 
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    loggingHost     @b{(inout)}  DNS Name/IP address of Logging host
*
* @returns  OPEN_E_NONE       if logging host IP address is returned.
* @returns  OPEN_E_FAIL       if failed to get logging host IP address.
* @returns  OPEN_E_NOT_FOUND  if no more entries exist.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes  Maximum size for loggingHost is determined by calling
*         openapiLoggingHostMaxAddrLenGet(). 
* @notes  To get the first host address pass empty string as host address.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLoggingHostIpAddrNextGet(openapiClientHandle_t *client_handle,
                                             open_buffdesc *loggingHost);

/*********************************************************************
* @purpose  Get logging host severity level.
* 
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    loggingHost     @b{(input)}  DNS Name/IP address of Logging host
* @param    severityLevel   @b{(output)} Logging host severity
*
* @returns  OPEN_E_NONE       if logging host priority is returned.
* @returns  OPEN_E_NOT_FOUND  if the logging host does not exist.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes  Maximum size for loggingHost is determined by calling
*         openapiLoggingHostMaxAddrLenGet(). 
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLoggingHostSeverityGet(openapiClientHandle_t *client_handle,
                                           open_buffdesc *loggingHost, 
                                           OPEN_LOG_SEVERITY_t *severityLevel);

/*********************************************************************
* @purpose  Get logging host port.
* 
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    loggingHost     @b{(input)}  DNS Name/IP address of Logging host
* @param    port            @b{(output)} Logging host port
*
* @returns  OPEN_E_NONE       if logging host port is returned.
* @returns  OPEN_E_NOT_FOUND  if the logging host does not exist.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes  Maximum size for loggingHost is determined by calling
*         openapiLoggingHostMaxAddrLenGet(). 
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLoggingHostPortGet(openapiClientHandle_t *client_handle,
                                       open_buffdesc *loggingHost, 
                                       uint16_t *port);

/*********************************************************************
* @purpose  Remove a logging host entry.
* 
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    loggingHost     @b{(input)}  DNS Name/IP address of Logging host
* 
* @returns  OPEN_E_NONE       if logging host entry is removed.
* @returns  OPEN_E_FAIL       if unable to remove the host entry.
* @returns  OPEN_E_NOT_FOUND  if host name does not exist.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* @notes  Maximum size for loggingHost is determined by calling
*         openapiLoggingHostMaxAddrLenGet(). 
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLoggingHostRemove(openapiClientHandle_t *client_handle, 
                                      open_buffdesc *loggingHost);

/*********************************************************************
* @purpose  Get the maximum logging hosts supported.
* 
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    maxLoggingHosts @b{(output)} Maximum logging hosts supported
* 
* @returns  OPEN_E_NONE     if maximum logging hosts are returned.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiMaxLoggingHostsSupportedGet(openapiClientHandle_t *client_handle,
                                                uint32_t *maxLoggingHosts);

/*********************************************************************
* @purpose  Get the maximum length of logging host address.
* 
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    logHostMaxAddrLen   @b{(output)} Maximum length of logging host address
* 
* @returns  OPEN_E_NONE     if maximum length of logging host address is returned.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* 
* @notes    Host name includes ipv4, ipv6 addresses and DNS name
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLoggingHostMaxAddrLenGet(openapiClientHandle_t *client_handle,
                                             uint32_t *logHostMaxAddrLen);

/*********************************************************************
* @purpose  Write system log
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    logMessage          @b{(input)}  Log message
* @param    -                                Maximum length @ref OPENAPI_LOG_FORMAT_BUF_SIZE
* @param    severityLevel       @b{(input)}  Logging severity level
*
* @returns  OPEN_E_NONE     if operation is successful
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* @returns  OPEN_E_ERROR    if input log message contains single symbol '%'.
*                           To print percent symbol in message - double percent must be used.
*                           For ex. message "%%s" will be printed as "%s".
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiSysLogWrite(openapiClientHandle_t *client_handle,
                                open_buffdesc         *logMessage,
                                OPEN_LOG_SEVERITY_t    severityLevel);

/*********************************************************************
* @purpose  Get the next entry in the in-memory (buffered) message log.
*           Allows for iterating over the content of the message log.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    logIndex            @b{(in/out)} Index to last retrieved entry. Set to
*                                            0 to retrieve latest log entry.
* @param    logEntry            @b{(output)} Buffer to hold log message. Size
*                                            must be @ref OPENAPI_LOG_ENTRY_BUF_SIZE.
*
* @returns  OPEN_E_NONE      if operation is successful
* @returns  OPEN_E_PARAM     if invalid parameter is passed
* @returns  OPEN_E_NOT_FOUND if no more log entries are found
* @returns  OPEN_E_FAIL      if failure retrieving next log message
*
* @notes    The indices are continuously updated as messages are logged.
*           Therefore, a slow reader and a fast writer (lot's of logging)
*           may continuously retrieve the oldest message (which may not be
*           the same).
*           Use a logIndex parameter of 0 to retrieve the latest log message.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiBufferedLogEntryNextGet(openapiClientHandle_t *client_handle, uint32_t *logIndex,
                                            open_buffdesc *logEntry);

/*********************************************************************
* @purpose  Clear the content of the buffered messages log.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
*
* @returns  OPEN_E_NONE     if operation is successful
* @returns  OPEN_E_PARAM    if invalid parameter is passed
* @returns  OPEN_E_FAIL     if operation failed
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiBufferedLogClear(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Enable or disable Buffered logging.
*
* @param    client_handle        @b{(input)}  Client handle from registration API
* @param    bufferedLoggingMode  @b{(input)}  Buffered logging mode
*
* @returns  OPEN_E_NONE     if Buffered logging mode is set successfully.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/ 
open_error_t openapiBufferedLogModeSet(openapiClientHandle_t *client_handle,
                                       OPEN_CONTROL_t bufferedLoggingMode);

/*********************************************************************
* @purpose  Get Buffered logging mode.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    bufferedLoggingMode  @b{(output)} Buffered logging mode
*
* @returns  OPEN_E_NONE     if Buffered logging mode is returned successfully.
* @returns  OPEN_E_FAIL     if invalid mode is returned.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiBufferedLogModeGet(openapiClientHandle_t *client_handle,
                                       OPEN_CONTROL_t *bufferedLoggingMode);

/*********************************************************************
* @purpose  Set Buffered Behavior.
*
* @param    client_handle        @b{(input)}  Client handle from registration API
* @param    bufferedLoggingBehavior  @b{(input)}  Buffered logging behavior
*
* @returns  OPEN_E_NONE     if Buffered logging behavior is set successfully.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/ 
open_error_t openapiBufferedLogBehaviorSet(openapiClientHandle_t *client_handle,
                                           OPEN_MEMORY_BEHAVIOR_t bufferedLoggingBehavior);

/*********************************************************************
* @purpose  Get Buffered Behavior.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    bufferedLoggingBehavior  @b{(output)} Buffered logging Behavior
*
* @returns  OPEN_E_NONE     if Buffered logging mode is returned successfully.
* @returns  OPEN_E_FAIL     if invalid mode is returned.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiBufferedLogBehaviorGet(openapiClientHandle_t *client_handle,
                                           OPEN_MEMORY_BEHAVIOR_t *bufferedLoggingBehavior);

/*********************************************************************
* @purpose  Set Buffered Severity Filter.
*
* @param    client_handle        @b{(input)}  Client handle from registration API
* @param    bufferedLoggingSeverityFilter  @b{(input)}  Buffered logging Severity Filter
*
* @returns  OPEN_E_NONE     if Buffered logging Severity Filter is set successfully.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/ 
open_error_t openapiBufferedLogSeverityFilterSet(openapiClientHandle_t *client_handle,
                                                 OPEN_LOG_SEVERITY_t bufferedLoggingSeverityFilter);

/*********************************************************************
* @purpose  Get Buffered Severity Filter.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    bufferedLoggingSeverityFilter  @b{(output)} Buffered logging Severity Filter
*
* @returns  OPEN_E_NONE     if Buffered logging Severity Filter is returned successfully.
* @returns  OPEN_E_FAIL     if invalid mode is returned.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiBufferedLogSeverityFilterGet(openapiClientHandle_t *client_handle,
                                                 OPEN_LOG_SEVERITY_t *bufferedLoggingSeverityFilter);

/*********************************************************************
* @purpose  Get Buffered Number of Messages.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    bufferedLoggingNumMessages  @b{(output)} Buffered logging Number of Messages
*
* @returns  OPEN_E_NONE     if Buffered logging Number of Messages is returned successfully.
* @returns  OPEN_E_FAIL     if invalid mode is returned.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiBufferedLogNumberOfMessagesGet(openapiClientHandle_t *client_handle,
                                                   uint32_t *bufferedLoggingNumMessages);

/*********************************************************************
* @purpose  Retrieve the content of the trap log. Returns all entries
*           present as an array of @ref openapiTrapLogEntry_t.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    trapLogData         @b{(output)} Buffer to hold log message. Size
*                                            must be @ref OPENAPI_TRAP_LOG_BUF_SIZE.
* @param    numberOfEntries     @b{(output)} The number of trap log entries returned,
*
* @returns  OPEN_E_NONE     if operation is successful
* @returns  OPEN_E_PARAM    if invalid parameter is passed
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiTrapLogGet(openapiClientHandle_t *client_handle, open_buffdesc *trapLogData,
                               uint32_t *numberOfEntries);

/*********************************************************************
* @purpose  Clear the content of the trap log.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
*
* @returns  OPEN_E_NONE     if operation is successful
* @returns  OPEN_E_PARAM    if invalid parameter is passed
* @returns  OPEN_E_FAIL     if operation failed
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiTrapLogClear(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Set USB logging mode to enable USB logging
*
* @param    client_handle  @b{(input)} Client handle from registration API
* @param    mode           @b{(input)} USB logging mode
*
*
* @returns  OPEN_E_NONE    if operation is successful
* @returns  OPEN_E_PARAM   if invalid parameter is passed
* @returns  OPEN_E_FAIL    if operation failed
* @returns  OPEN_E_UNAVAIL if feature is unavailable
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiUsbLoggingModeSet(openapiClientHandle_t *client_handle,
                                      OPEN_USB_LOGGING_MODE_t mode);

/*********************************************************************
* @purpose  Get USB logging mode
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    mode           @b{(output)} USB logging mode
*
*
* @returns  OPEN_E_NONE    if operation is successful
* @returns  OPEN_E_PARAM   if invalid parameter is passed
* @returns  OPEN_E_FAIL    if operation failed
* @returns  OPEN_E_UNAVAIL if feature is unavailable
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiUsbLoggingModeGet(openapiClientHandle_t *client_handle,
                                      OPEN_USB_LOGGING_MODE_t *mode);

/*********************************************************************
* @purpose  Set USB logging severity
*
* @param    client_handle  @b{(input)} Client handle from registration API
* @param    severity       @b{(input)} USB logging mode
*
*
* @returns  OPEN_E_NONE    if operation is successful
* @returns  OPEN_E_PARAM   if invalid parameter is passed
* @returns  OPEN_E_FAIL    if operation failed
* @returns  OPEN_E_UNAVAIL if feature is unavailable
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiUsbLoggingSeveritySet(openapiClientHandle_t *client_handle,
                                         OPEN_LOG_SEVERITY_t severity);

/*********************************************************************
* @purpose  Get USB logging severity 
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    severity       @b{(output)} USB logging mode
*
*
* @returns  OPEN_E_NONE    if operation is successful
* @returns  OPEN_E_PARAM   if invalid parameter is passed
* @returns  OPEN_E_FAIL    if operation failed
* @returns  OPEN_E_UNAVAIL if feature is unavailable
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiUsbLoggingSeverityGet(openapiClientHandle_t *client_handle,
                                          OPEN_LOG_SEVERITY_t *severity);

/*********************************************************************
* @purpose  Get USB logging operational status
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    mode           @b{(output)} USB logging operational status
*
*
* @returns  OPEN_E_NONE    if operation is successful
* @returns  OPEN_E_PARAM   if invalid parameter is passed
* @returns  OPEN_E_FAIL    if operation failed
* @returns  OPEN_E_UNAVAIL if feature is unavailable
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiUsbLoggingOperStatusGet(openapiClientHandle_t *client_handle,
                                            OPEN_USB_LOGGING_OPER_STATUS_t *mode);

/*****************************************************************************
 * @purpose   Get duplicate log suppression repeat interval.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     *interval      @b{(output)} Repeat interval value in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLogSuppressDupLogRepeatIntvlGet(openapiClientHandle_t *client_handle, uint32_t *interval);

/*****************************************************************************
 * @purpose   Set duplicate log suppression repeat interval.
 *
 * @param     client_handle  @b{(input)} Client handle from registration API
 * @param     interval       @b{(input)} Repeat interval value in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLogSuppressDupLogRepeatIntvlSet(openapiClientHandle_t *client_handle, uint32_t interval);

/*****************************************************************************
 * @purpose   Get duplicate log suppression mode.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     *mode          @b{(output)} Suppression admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLogSuppressDupLogModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set duplicate log suppression mode.
 *
 * @param     client_handle  @b{(input)} Client handle from registration API
 * @param     mode           @b{(input)} Suppression admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLogSuppressDupLogModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose  Set USB logging device number 
*
* @param    client_handle  @b{(input)} Client handle from registration API
* @param    devNumber      @b{(input)} USB logging device number
*
*
* @returns  OPEN_E_NONE    if operation is successful
* @returns  OPEN_E_PARAM   if invalid parameter is passed
* @returns  OPEN_E_FAIL    if operation failed
* @returns  OPEN_E_UNAVAIL if feature is unavailable
*
* @notes    OPEN_E_UNAVAIL is returned on platforms that does not support
*           multiple USB devices.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiLogUsbDeviceNumberSet(openapiClientHandle_t *client_handle, uint32_t devNumber);

/*********************************************************************
* @purpose  Get USB logging device number 
*
* @param    client_handle  @b{(input)} Client handle from registration API
* @param    devNumber      @b{(output)} USB logging device number
*
*
* @returns  OPEN_E_NONE    if operation is successful
* @returns  OPEN_E_PARAM   if invalid parameter is passed
* @returns  OPEN_E_FAIL    if operation failed
* @returns  OPEN_E_UNAVAIL if feature is unavailable
*
* @notes    OPEN_E_UNAVAIL is returned on platforms that does not support
*           multiple USB devices.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiLogUsbDeviceNumberGet(openapiClientHandle_t *client_handle, uint32_t *devNumber);
#endif
