/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_authmgr.h
*
* @purpose   MAB Configuration and Status
*
* @component OPEN
*
* @create    02/28/2023
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_MAB_H_INCLUDED
#define OPENAPI_MAB_H_INCLUDED

#include "openapi_common.h"

/* MAB Auth Type */
typedef enum
{
  OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_INVALID = 0,
  OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_EAP_MD5 = 1,
  OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_PAP = 2,
  OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_CHAP = 3,
} OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_t;


/*********************************************************************
* @purpose  Sets the port MAC authentication bypass enable mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mabEnabled      @b{(output)} OPEN_TRUE if MAB enabled, else OPEN_FALSE
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiMabPortMabEnabledGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_BOOL_t *mabEnabled);

/*********************************************************************
* @purpose  Sets the port MAC authentication bypass enable mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mabEnabled      @b{(input)}  Set to OPEN_TRUE to enable MAB, OPEN_FALSE to disable
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiMabPortMabEnabledSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_BOOL_t mabEnabled);

/*********************************************************************
* @purpose  Sets the authentication type to be used by MAB for the 
*           specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    val             @b{(input)}  Authentication type
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiMabPortMABAuthTypeSet(openapiClientHandle_t *client_handle,
                                          uint32_t ifNum,
                                          OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_t val);

/*********************************************************************
* @purpose  Gets the authentication host mode for the specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    val             @b{(output)} Authentication type
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiMabPortMABAuthTypeGet(openapiClientHandle_t *client_handle,
                                          uint32_t ifNum,
                                          OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_t *val);

/*********************************************************************
* @purpose  Gets the port operational MAB mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mode            @b{(output)} Mode
*
* @returns  OPEN_E_NONE        Mode get successful.
* @returns  OPEN_E_FAIL        Mode get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiMabPortOperationalMabModeGet(openapiClientHandle_t *client_handle,
                                                   uint32_t ifNum, OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Set password for MAB request attribute 2 
*
* @param    client_handle   @b{(input)} client handle from registration API
* @param    passwd          @b{(input)} Attribute 2 password 
* @param    encrypted       @b{(input)} Password is in encrypted format or not 
*
* @returns  OPEN_E_NONE        Password set is successful.
* @returns  OPEN_E_FAIL        Unable to set password.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiMabRequestAttribute2paramSet(openapiClientHandle_t *client_handle,
                                                 open_buffdesc *passwd, OPEN_BOOL_t encrypted);

/*********************************************************************
* @purpose  Get password for MAB request attribute 2 
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    password        @b{(output)} Attribute 2 password 
*
* @returns  OPEN_E_NONE        Password get is successful.
* @returns  OPEN_E_FAIL        Unable to get password.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiMabRequestAttribute2paramGet(openapiClientHandle_t *client_handle,
                                                 open_buffdesc *password);

#endif /* OPENAPI_MAB_H_INCLUDED */
