/********************************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.?0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
********************************************************************************
*
* @filename  openapi_mirror.h
*
* @purpose   Provide interface to Port Mirroring APIs
*
* @component OpEN
*
* @create    06/08/2015
*
* @end
*
********************************************************************************/
#ifndef OPENAPI_MIRROR_H_INCLUDED
#define OPENAPI_MIRROR_H_INCLUDED

#include "openapi_common.h"

/** OpEN uses this enumeration to define ports
 *  traffic direction */
typedef enum
{
  OPENAPI_MIRROR_UNCONFIGURED  = 0, /**< Traffic direction is unconfigured */
  OPENAPI_MIRROR_BIDIRECTIONAL = 1, /**< Traffic direction is bidirectional */
  OPENAPI_MIRROR_INGRESS       = 2, /**< Traffic direction is ingress */
  OPENAPI_MIRROR_EGRESS        = 3, /**< Traffic direction is egress */
  OPENAPI_MIRROR_SFLOW         = 4  /**< Traffic direction is swlow */
} OPENAPI_MIRROR_DIRECTION_t;

/*******************************************************************************
* @purpose  Returns list of source ports for session "sessionNum".
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    sessionNum          @b{(input)}   session number
* @param    bufd                @b{(output)}  buffer for outputting comma separated list of source ports. 
*                                             Must be at least OPENAPI_CLI_MAX_STRING_LENGTH bytes in length. 
*                                             Comma separated list of source ports.
*
* @returns  OPEN_E_NONE         List of source ports returned.
* @returns  OPEN_E_FAIL         Failed to get list of source ports.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
********************************************************************************/
open_error_t openapiSwPortMonitorSourcePortsListGet(openapiClientHandle_t * client_handle,
                                                    uint32_t sessionNum,
                                                    open_buffdesc * bufd);

/*********************************************************************
* @purpose  Returns the direction to be mirrored for a source port.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    sessionNum          @b{(input)}   session number
* @param    intIfNum            @b{(input)}   internal interface number
* @param    probeType           @b{(output)}  direction of traffic to be mirrored
* 
* @returns  OPEN_E_NONE         Direction for a source port returned.
* @returns  OPEN_E_FAIL         Failed to get direction for a source port.
* @returns  OPEN_E_INTERNAL     Internal error
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
********************************************************************************/
open_error_t openapiMirrorSourcePortDirectionGet(openapiClientHandle_t * client_handle,
                                                 uint32_t sessionNum,
                                                 uint32_t intIfNum,
                                                 OPENAPI_MIRROR_DIRECTION_t * probeType);

/********************************************************************************
* @purpose  Adds interface to the mirrored ports of session "sessionNum".
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    sessionNum          @b{(input)}   session number
* @param    intIfNum            @b{(input)}   internal interface number
* @param    probeType           @b{(input)}   direction of traffic to be mirrored
* 
* @returns  OPEN_E_NONE         Interface added to the mirrored ports successfully.
* @returns  OPEN_E_FAIL         Failed to add interface to the mirrored ports.
* @returns  OPEN_E_INTERNAL     Internal error
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorSourcePortAdd(openapiClientHandle_t * client_handle,
                                               uint32_t sessionNum,
                                               uint32_t intIfNum,
                                               OPENAPI_MIRROR_DIRECTION_t probeType);

/********************************************************************************
* @purpose  Remove interface from mirrored ports of session "sessionNum".
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    sessionNum          @b{(input)}   session number
* @param    intIfNum            @b{(input)}   internal interface number to be
*                                             removed
* 
* @returns  OPEN_E_NONE         Interface removed from source interface list.
* @returns  OPEN_E_FAIL         Failed to remove interface from source interface list.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes Calling this API will change the running configuration of the switch   
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
********************************************************************************/
open_error_t openapiSwPortMonitorSourcePortRemove(openapiClientHandle_t * client_handle, 
                                                  uint32_t sessionNum,
                                                  uint32_t intIfNum);

/*******************************************************************************
* @purpose  Returns Destination Port for session "sessionNum".
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    sessionNum          @b{(input)}   session number
* @param    intIfNum            @b{(output)}  internal destination interface number
* 
* @returns  OPEN_E_NONE         Destination Port returned.
* @returns  OPEN_E_FAIL         Failed to get Destination Port.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
********************************************************************************/
open_error_t openapiSwPortMonitorDestPortGet(openapiClientHandle_t * client_handle,
                                             uint32_t sessionNum,
                                             uint32_t *intIfNum);

/********************************************************************************
* @purpose  Sets Destination Port for session "sessionNum".
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    sessionNum          @b{(input)}   session number
* @param    intIfNum            @b{(input)}   the internal destibation interface
*                                             number to set
* 
* @returns  OPEN_E_NONE        Destination Port for session set successfully.
* @returns  OPEN_E_FAIL        Failed to set Destination Port for session.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorDestPortSet(openapiClientHandle_t * client_handle, 
                                             uint32_t sessionNum,
                                             uint32_t intIfNum);

/********************************************************************************
* @purpose  Remove Destination Port of session "sessionNum".
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    sessionNum          @b{(input)}   session number
* 
* @returns  OPEN_E_NONE         Destination port is cleared.
* @returns  OPEN_E_FAIL         Failed to clear destination port.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorDestPortRemove(openapiClientHandle_t * client_handle,
                                                uint32_t sessionNum);

/*********************************************************************************
* @purpose  Returns the Port Monitoring Mode for session "sessionNum".
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    sessionNum          @b{(input)}   session number
* @param    mode                @b{(output)}  monitoring mode (enable/disable)
* 
* @returns  OPEN_E_NONE         Port Monitoring Mode returned.
* @returns  OPEN_E_FAIL         Failed to get Port Monitoring Mode.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorModeGet(openapiClientHandle_t * client_handle,
                                         uint32_t sessionNum, 
                                         OPEN_CONTROL_t * mode);

/********************************************************************************
* @purpose  Sets the Port Monitoring Mode for session "sessionNum".
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    sessionNum          @b{(input)}   session number
* @param    mode                @b{(input)}   monitoring mode (enable/disable)
* 
* @returns  OPEN_E_NONE         Port Monitoring Mode set successfully.
* @returns  OPEN_E_FAIL         Failed to set Port Monitoring Mode.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorModeSet(openapiClientHandle_t * client_handle,
                                         uint32_t sessionNum, 
                                         OPEN_CONTROL_t mode);

/*********************************************************************************
* @purpose  Returns count of the sessiosns of Port Monitoring sessions.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    sessionCount        @b{(output)}  count of the sessions
* 
* @returns  OPEN_E_NONE         Count of Port Monitoring sessions returned.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes  It returns maximum number of sessions that can be configured.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorSessionCount(openapiClientHandle_t * client_handle, 
                                              uint32_t * sessionCount);

/********************************************************************************
* @purpose  Removes port monitoring configuration for all sessions.
*
* @param    client_handle       @b{(input)}   client handle from registration API
* 
* @returns  OPEN_E_NONE         Mirroring config data is removed successfully.
* @returns  OPEN_E_FAIL         Failed to remove destination mirroring config data.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorConfigRemove(openapiClientHandle_t * client_handle);

/********************************************************************************
* @purpose  Removes port monitoring configuration for session.
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    sessionNum          @b{(input)}   session number
* 
* @returns  OPEN_E_NONE         Session removed successfully.
* @returns  OPEN_E_FAIL         Failed to remove session.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorSessionRemove(openapiClientHandle_t * client_handle,
                                               uint32_t sessionNum);

/*********************************************************************************
* @purpose  Determines status of internal interface number which is configured
*           as destination port.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    intIfNum            @b{(input)}   the internal interface number
* @param    sessionNum          @b{(output)}  session number
* @param    status              @b{(output)}  interface status (true or false)
* 
* @returns  OPEN_E_NONE         Status of interface determined successfully.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorIsDestConfigured(openapiClientHandle_t * client_handle,
                                                  uint32_t intIfNum,
                                                  uint32_t * sessionNum,
                                                  OPEN_BOOL_t * status);

/*********************************************************************************
* @purpose  Determines status of internal interface number which is configured
*           as source port.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    intIfNum            @b{(input)}   the internal interface number
* @param    status              @b{(output)}  interface status (true or false)
* 
* @returns  OPEN_E_NONE         Status of interface determined successfully.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorIsSrcConfigured(openapiClientHandle_t * client_handle,
                                                 uint32_t intIfNum,
                                                 OPEN_BOOL_t * status);

/*********************************************************************************
* @purpose  Returns the session in which internal interface number is configured
*           as a source port.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    intIfNum            @b{(input)}   the internal interface number
* @param    sessionNum          @b{(output)}  session number
* 
* @returns  OPEN_E_NONE         Session number returned.
* @returns  OPEN_E_FAIL         Failed to get session number.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorSrcConfiguredSessionGet(openapiClientHandle_t * client_handle,
                                                         uint32_t intIfNum,
                                                         uint32_t * sessionNum);

/*********************************************************************************
* @purpose  Checks validity of interface for a source port.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    intIfNum            @b{(input)}   the internal interface number
* @param    valid               @b{(output)}  validation status
* 
* @returns  OPEN_E_NONE         Validity of interface checked.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorIsValidSrcPort(openapiClientHandle_t * client_handle,
                                                uint32_t intIfNum,
                                                OPEN_BOOL_t * valid);

/*********************************************************************************
* @purpose  Checks validity of interface for a destination port.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    intIfNum            @b{(input)}   the internal interface number
* @param    valid               @b{(output)}  validation status
* 
* @returns  OPEN_E_NONE         Validity of interface checked.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorIsValidDestPort(openapiClientHandle_t * client_handle,
                                                 uint32_t intIfNum,
                                                 OPEN_BOOL_t * valid);

/*********************************************************************************
* @purpose  Returns first valid interface that can be configured as source port.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    intIfNum            @b{(output)}  internal interface number of first
*                                             valid source interface
* 
* @returns  OPEN_E_NONE         First valid interface returned.
* @returns  OPEN_E_FAIL         Failed to get first valid interface.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorValidSrcIntfFirstGet(openapiClientHandle_t * client_handle,
                                                      uint32_t * intIfNum);

/*********************************************************************************
* @purpose  Returns next valid interface that can be configured as source port.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    intIfNum            @b{(input)}   internal interface number of 
*                                             a valid source port
* @param    nextIntIfNum        @b{(output)}  internal interface number of 
*                                             next valid source port
* 
* @returns  OPEN_E_NONE         Next valid interface returned.
* @returns  OPEN_E_FAIL         Failed to get next valid interface.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorValidSrcIntfNextGet(openapiClientHandle_t * client_handle,
                                                     uint32_t intIfNum,
                                                     uint32_t * nextIntIfNum);

/*********************************************************************************
* @purpose  Returns list of valid interfaces that can be configured as source port.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    bufd                @b{(output)}  buffer for outputting comma separated list of source ports
* 
* @returns  OPEN_E_NONE         List of valid interfaces returned.
* @returns  OPEN_E_FAIL         Failed to get list of valid interfaces.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorValidSrcIntfListGet(openapiClientHandle_t * client_handle,
                                                     open_buffdesc * bufd);

/*********************************************************************************
* @purpose  Returns first valid interface that can be configured as destination port.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    intIfNum            @b{(output)}  internal interface number of first
*                                             valid destination interface
* 
* @returns  OPEN_E_NONE         First valid interface returned.
* @returns  OPEN_E_FAIL         Failed to get first valid interface.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorValidDestIntfFirstGet(openapiClientHandle_t * client_handle,
                                                       uint32_t * intIfNum);

/*********************************************************************************
* @purpose  Returns next valid interface that can be configured as destination port.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    intIfNum            @b{(input)}   internal interface number of 
*                                             a valid destination port
* @param    nextIntIfNum        @b{(output)}  internal interface number of 
*                                             next valid destination port
* 
* @returns  OPEN_E_NONE         Next valid interface returned.
* @returns  OPEN_E_FAIL         Failed to get next valid interface.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorValidDestIntfNextGet(openapiClientHandle_t * client_handle,
                                                      uint32_t intIfNum,
                                                      uint32_t *nextIntIfNum);

/*********************************************************************************
* @purpose  Returns list of valid interfaces that can be configured as destination port.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    bufd                @b{(output)}  buffer for outputting comma separated list of destination ports
* 
* @returns  OPEN_E_NONE         List of valid interfaces returned.
* @returns  OPEN_E_FAIL         Failed to get list of valid interfaces.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwPortMonitorValidDestIntfListGet(openapiClientHandle_t * client_handle,
                                                     open_buffdesc * bufd);

/*********************************************************************************
* @purpose  Returns the VLAN IDs for session "sessionNum".
* 
* @param    client_handle       @b{(input)}   client handle from registration API 
* @param    sessionNum          @b{(input)}   session number 
* @param    bufd                @b{(output)}  buffer for outputting comma separated list of
*                                             attached VLANs
*
* @returns  OPEN_E_NONE         List of valid VLANs returned.
* @returns  OPEN_E_FAIL         Failed to get list of valid VLANs.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwMonitorVlanListGet(openapiClientHandle_t * client_handle, 
                                         uint32_t sessionNum,
                                         open_buffdesc * bufd);

/********************************************************************************
* @purpose  Sets the VLAN IDs for session "sessionNum".
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    sessionNum         @b{(input)}   session number
* @param    bufd               @b{(input)}   string buffer of comma separated list of vlans to set
*                                            attached VLANs
* @param    updateAction       @b{(input)}   action to specify add/remove
* 
* @returns  OPEN_E_NONE        List of valid VLANs set successful.
* @returns  OPEN_E_FAIL        Failed to set list of valid VLANs for the session.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwMonitorVlanListSet(openapiClientHandle_t * client_handle,
                                         uint32_t sessionNum,
                                         open_buffdesc * bufd,
                                         OPEN_CONTROL_t updateAction);

/*********************************************************************************
* @purpose  Returns the source mirror type(Port based or VLAN based) for
*           session "sessionNum".
* 
* @param    client_handle       @b{(input)}   client handle from registration API 
* @param    sessionNum          @b{(intput)}  session number 
* @param    mirrorSrcType       @b{(output)}  type of mirroring
* 
* @returns  OPEN_E_NONE         Source mirror type returned.
* @returns  OPEN_E_FAIL         Failed to get source mirror type for the session.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwMonitorSrcTypeGet(openapiClientHandle_t * client_handle,
                                        uint32_t sessionNum,
                                        uint32_t * mirrorSrcType);

/*********************************************************************************
* @purpose  Returns the destination mirror type (Port based or VLAN based) for
*           session "sessionNum".
* 
* @param    client_handle       @b{(input)}    client handle from registration API 
* @param    sessionNum          @b{(input)}    session number 
* @param    mirrorDstType       @b{(output)}   type of mirroring
* 
* @returns  OPEN_E_NONE         Destination mirror type returned.
* @returns  OPEN_E_FAIL         Failed to get destination mirror type for the session.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwMonitorDstTypeGet(openapiClientHandle_t * client_handle,
                                        uint32_t sessionNum,
                                        uint32_t * mirrorDstType);

/*********************************************************************************
* @purpose  Returns the remote source VLAN for session "sessionNum".
* 
* @param    client_handle       @b{(input)}   client handle from registration API 
* @param    sessionNum          @b{(input)}   session number 
* @param    remoteSrcVlan       @b{(output)}  remote source VLAN
* 
* @returns  OPEN_E_NONE         Remote source VLAN returned.
* @returns  OPEN_E_FAIL         Failed to get remote source VLAN.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_EMPTY        Source VLAN isn't configured.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwMonitorRemoteSrcVlanGet(openapiClientHandle_t * client_handle,
                                              uint32_t sessionNum,
                                              uint32_t * remoteSrcVlan);

/********************************************************************************
* @purpose  Sets the remote source VLAN for session "sessionNum".
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    sessionNum         @b{(input)}   session number
* @param    vlanId             @b{(input)}   set of Valid VLAN IDs to be added
*                                            to the session
* @param    updateAction       @b{(input)}   action to specify add/remove
* 
* @returns  OPEN_E_NONE        Remote source VLAN set successful.
* @returns  OPEN_E_FAIL        Failed to set remote source VLAN.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwRemoteMonitorSrcVlanSet(openapiClientHandle_t * client_handle,
                                              uint32_t sessionNum,
                                              uint16_t vlanId,
                                              OPEN_CONTROL_t updateAction);

/*********************************************************************************
* @purpose  Returns the remote destination VLAN for session "sessionNum".
* 
* @param    client_handle       @b{(input)}   client handle from registration API 
* @param    sessionNum          @b{(input)}   session number 
* @param    remoteDstVlan       @b{(output)}  remote destination VLAN
* 
* @returns  OPEN_E_NONE         Remote destination VLAN returned.
* @returns  OPEN_E_FAIL         Failed to get remote destination VLAN.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_EMPTY        Destination VLAN isn't configured.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwMonitorRemoteDstVlanGet(openapiClientHandle_t * client_handle,
                                              uint32_t sessionNum,
                                              uint32_t * remoteDstVlan);

/********************************************************************************
* @purpose  Sets the remote destination VLAN for session "sessionNum".
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    sessionNum         @b{(input)}   session number
* @param    vlanId             @b{(input)}   remote destination VLAN to set
* @param    updateAction       @b{(input)}   action to specify add/remove
* 
* @returns  OPEN_E_NONE        Remote destination VLAN set successful.
* @returns  OPEN_E_FAIL        Failed to set remote destination VLAN.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwRemoteMonitorDstVlanSet(openapiClientHandle_t * client_handle,
                                              uint32_t sessionNum,
                                              uint16_t vlanId,
                                              OPEN_CONTROL_t updateAction);

/*********************************************************************************
* @purpose  Returns Monitoring Reflector Port for session "sessionNum".
* 
* @param    client_handle       @b{(input)}   client handle from registration API 
* @param    sessionNum          @b{(input)}   session number 
* @param    intIfNum            @b{(output)}  interface number of reflector  port
* 
* @returns  OPEN_E_NONE         Port Monitoring Reflector Port returned.
* @returns  OPEN_E_FAIL         Failed to get Port Monitoring Reflector Port.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwMonitorReflectorPortGet(openapiClientHandle_t * client_handle,
                                              uint32_t sessionNum,
                                              uint32_t * intIfNum);

/********************************************************************************
* @purpose  Sets Monitoring Reflector Port for session "sessionNum".
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    sessionNum         @b{(input)}   session number
* @param    intIfNum           @b{(input)}   internal interface number to set
* @param    updateAction       @b{(input)}   action to specify add/remove
* 
* @returns  OPEN_E_NONE        Port Monitoring Reflector Port set successful.
* @returns  OPEN_E_FAIL        Failed to set Port Monitoring Reflector Port.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwMonitorReflectorPortSet(openapiClientHandle_t * client_handle,
                                              uint32_t sessionNum,
                                              uint32_t intIfNum,
                                              OPEN_CONTROL_t updateAction);

/*********************************************************************************
* @purpose  Returns the IP ACL attached to the mirroring session "sessionNum".
* 
* @param    client_handle       @b{(input)}   client handle from registration API 
* @param    sessionNum          @b{(input)}   session number 
* @param    ipAccessListNum     @b{(output)}  IP access list number
* 
* @returns  OPEN_E_NONE         IP ACL attached to the mirroring session returned.
* @returns  OPEN_E_FAIL         Failed to get IP ACL attached to the mirroring
*                               session.
* @returns  OPEN_E_UNAVAIL      Feature is not supported.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwMonitorIpAclGet(openapiClientHandle_t * client_handle,
                                      uint32_t sessionNum,
                                      uint32_t * ipAccessListNum);

/********************************************************************************
* @purpose  Attaches IP ACL to the mirroring session "sessionNum".
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    sessionNum         @b{(input)}   session number
* @param    ipAccessListNum    @b{(input)}   IP access list number to set
* @param    updateAction       @b{(input)}   action to specify add/remove
* 
* @returns  OPEN_E_NONE        IP ACL to the mirroring session attached successful.
* @returns  OPEN_E_FAIL        Failed to attach IP ACL to the mirroring session.
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwMonitorIpAclSet(openapiClientHandle_t * client_handle,
                                      uint32_t sessionNum,
                                      uint32_t ipAccessListNum,
                                      OPEN_CONTROL_t updateAction);

/*********************************************************************************
* @purpose  Returns the MAC ACL attached to the mirroring session "sessionNum".
* 
* @param    client_handle       @b{(input)}   client handle from registration API 
* @param    sessionNum          @b{(input)}   session number 
* @param    macAccessListName   @b{(output)}  MAC access list name
* @param    -                                 Minimum length @ref OPENAPI_ACL_NAME_LEN_MIN
* @param    -                                 Maximum length @ref OPENAPI_ACL_NAME_LEN_MAX
* 
* @returns  OPEN_E_NONE         MAC ACL attached to the mirroring session returned.
* @returns  OPEN_E_FAIL         Failed to get MAC ACL attached to the mirroring
*                               session.
* @returns  OPEN_E_UNAVAIL      Feature is not supported.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwMonitorMacAclGet(openapiClientHandle_t * client_handle,
                                       uint32_t sessionNum,
                                       open_buffdesc * macAccessListName);

/********************************************************************************
* @purpose  Attaches MAC ACL to the mirroring session "sessionNum".
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    sessionNum          @b{(input)}   session number
* @param    macAccessListName   @b{(input)}   MAC access list name to attach
* @param    -                                 Minimum length @ref OPENAPI_ACL_NAME_LEN_MIN
* @param    -                                 Maximum length @ref OPENAPI_ACL_NAME_LEN_MAX
* @param    updateAction        @b{(input)}   action to specify add/remove
* 
* @returns  OPEN_E_NONE         MAC ACL to the mirroring session attached successful.
* @returns  OPEN_E_FAIL         Failed to attach IP ACL to the mirroring session.
* @returns  OPEN_E_INTERNAL     IP ACL is has already been attached to this session.
* @returns  OPEN_E_ERROR        MAC ACL does not exist.
* @returns  OPEN_E_UNAVAIL      Feature is not supported.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSwMonitorMacAclSet(openapiClientHandle_t * client_handle,
                                       uint32_t sessionNum,
                                       open_buffdesc * macAccessListName,
                                       OPEN_CONTROL_t updateAction);

/*********************************************************************************
* @purpose  Returns the RSPAN VLAN.
* 
* @param    client_handle       @b{(input)}   client handle from registration API 
* @param    rspanVlan           @b{(output)}  VLAN ID
* 
* @returns  OPEN_E_NONE         RSPAN VLAN returned.
* @returns  OPEN_E_FAIL         Failed to get RSPAN VLAN
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiRemoteSpanVlanGet(openapiClientHandle_t * client_handle,
                                      uint32_t * rspanVlan);

/*********************************************************************************
* @purpose  Returns next configured RSPAN VLAN.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    rspanVlan           @b{(input)}   VLAN ID
* @param    rspanVlanNext       @b{(output)}  next VLAN ID
* 
* @returns  OPEN_E_NONE         Next configured RSPAN VLAN returned.
* @returns  OPEN_E_FAIL         Failed to get next configured RSPAN VLAN.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiRemoteSpanVlanNextGet(openapiClientHandle_t * client_handle,
                                          uint32_t rspanVlan,
                                          uint32_t * rspanVlanNext);

/*********************************************************************************
* @purpose  Returns list of RSPAN VLAN.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    bufd                @b{(output)}  buffer for outputting comma separated list of source ports
* 
* @returns  OPEN_E_NONE         Lst of RSPAN VLAN returned.
* @returns  OPEN_E_FAIL         Failed to get list of RSPAN VLAN.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_EMPTY        List is empty.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiRemoteSpanVlanListGet(openapiClientHandle_t * client_handle,
                                          open_buffdesc * bufd);

/********************************************************************************
* @purpose  Sets the VLAN as RSPAN VLAN.
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    rspanVlan          @b{(input)}   VLAN ID
* @param    updateAction       @b{(input)}   action to specify add/remove
* 
* @returns  OPEN_E_NONE        VLAN as RSPAN VLAN set successful.
* @returns  OPEN_E_FAIL        Failed to set VLAN as RSPAN VLAN.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiRemoteSpanVlanSet(openapiClientHandle_t * client_handle,
                                      uint32_t rspanVlan,
                                      OPEN_CONTROL_t updateAction);

/*********************************************************************************
* @purpose  Checks if the VLAN is a RSPAN VLAN.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    vlanId              @b{(input)}   VLAN ID
* @param    status              @b{(output)}  VLAN RSPAN status (true or false)
* 
* @returns  OPEN_E_NONE         VLAN RSPAN status checked.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiMirrorIsRspanVlan(openapiClientHandle_t * client_handle,
                                      uint32_t vlanId,
                                      OPEN_BOOL_t * status);

/*********************************************************************************
* @purpose  Checks if the VLAN is source to any of the mirroring sessions.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    vlanId              @b{(input)}   VLAN ID
* @param    status              @b{(output)}  VLAN source status (true or false)
* 
* @returns  OPEN_E_NONE         VLAN source status checked.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiMirrorIsSrcVlan(openapiClientHandle_t * client_handle,
                                    uint32_t vlanId,
                                    OPEN_BOOL_t * status);

/*********************************************************************************
* @purpose  Returns the Port Monitoring session "sessionNum" reserved state.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    sessionNum          @b{(input)}   session number
* @param    state               @b{(output)}  monitoring reserved state (enable or disable)
* 
* @returns  OPEN_E_NONE         Monitoring reserved state returned.
* @returns  OPEN_E_FAIL         Failed to get monitoring reserved state
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiMirrorSessionReserveStateGet(openapiClientHandle_t * client_handle,
                                                 uint32_t sessionNum,
                                                 OPEN_CONTROL_t * state);

/********************************************************************************
* @purpose  Sets the Port Monitoring session "sessionNum" reserved state.
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    sessionNum         @b{(input)}   session number
* @param    state              @b{(input)}   monitoring reserved state (enable or disable)
* 
* @returns  OPEN_E_NONE        Port Monitoring session reserved state set successful.
* @returns  OPEN_E_FAIL        Failed to set Port Monitoring session reserved state.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiMirrorSessionReserveStateSet(openapiClientHandle_t * client_handle,
                                                 uint32_t sessionNum,
                                                 OPEN_CONTROL_t state);
#endif /* OPENAPI_MIRROR_H_INCLUDED */

