/**********************************************************************
*
*  Copyright 2022-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_mvr.h
*
* @purpose   This code implements the OpEN MVR API
*
* @component OPEN
*
* @create    11/30/2022
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_MVR_H_INCLUDED
#define OPENAPI_MVR_H_INCLUDED

#include "openapi_common.h" 

typedef enum
{
  OPEN_MVR_NONE_MODE = 0,
  OPEN_MVR_COMPATIBLE_MODE,
  OPEN_MVR_DYNAMIC_MODE
} OPEN_MVR_MODE_TYPE_t;

typedef enum
{
  OPEN_MVR_NONE_TYPE = 0,
  OPEN_MVR_STATIC_TYPE,
  OPEN_MVR_DYNAMIC_TYPE
} OPEN_MVR_ENTRY_TYPE_t;

typedef enum
{
  OPEN_MVR_NONE_PORT_TYPE = 0,
  OPEN_MVR_SOURCE_PORT_TYPE,
  OPEN_MVR_RECEIVER_PORT_TYPE,
  OPEN_MVR_LAST_TYPE
} OPEN_MVR_INTF_TYPE_t;

typedef enum
{  
  OPEN_MVR_PKT_IGMP_QUERY_RX_COUNTER = 0,
  OPEN_MVR_PKT_IGMP_REPORT_V1_RX_COUNTER,
  OPEN_MVR_PKT_IGMP_REPORT_V2_RX_COUNTER,
  OPEN_MVR_PKT_IGMP_LEAVE_RX_COUNTER,
  OPEN_MVR_PKT_IGMP_QUERY_TX_COUNTER,
  OPEN_MVR_PKT_IGMP_REPORT_V1_TX_COUNTER,
  OPEN_MVR_PKT_IGMP_REPORT_V2_TX_COUNTER,
  OPEN_MVR_PKT_IGMP_LEAVE_TX_COUNTER,
  OPEN_MVR_PKT_RX_FAILURE_COUNTER,
  OPEN_MVR_PKT_TX_FAILURE_COUNTER,
  OPEN_MVR_PKT_COUNTER_LAST
} OPEN_MVR_PDU_COUNTER_t;

#define OPEN_MVR_QUERY_TIME_MIN     1
#define OPEN_MVR_QUERY_TIME_MAX     100
#define OPEN_MVR_MULT_IP_ADDR_MIN       0xe0000100
#define OPEN_MVR_MULT_IP_ADDR_MAX       0xefffffff
/*****************************************************************************
 * @purpose   Gets the MVR admin mode.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     adminMode          @b{(output)} MVR admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrAdminModeGet(openapiClientHandle_t *client_handle, 
                                    OPEN_CONTROL_t *adminMode);

/*****************************************************************************
 * @purpose   Sets the MVR admin mode.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     adminMode          @b{(input)}  MVR admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrAdminModeSet(openapiClientHandle_t *client_handle,
                                    OPEN_CONTROL_t adminMode);

/*****************************************************************************
 * @purpose   Gets the MVR query time.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     queryTime           @b{(output)} MVR query time
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrQueryTimeGet(openapiClientHandle_t *client_handle,
                                    uint32_t *queryTime);

/*****************************************************************************
 * @purpose   Sets the MVR query time.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     queryTime        @b{(input)}  MVR query time
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrQueryTimeSet(openapiClientHandle_t *client_handle,
                                    uint32_t queryTime);

/*****************************************************************************
 * @purpose   Gets the MVR mode type.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     modeType           @b{(output)} MVR mode type
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrModeTypeGet(openapiClientHandle_t *client_handle, 
                                   OPEN_MVR_MODE_TYPE_t *modeType);

/*****************************************************************************
 * @purpose   Sets the MVR mode type.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     modeType       @b{(input)}  MVR mode type
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrModeTypeSet(openapiClientHandle_t *client_handle, 
                                   OPEN_MVR_MODE_TYPE_t modeType);

/*****************************************************************************
 * @purpose   Gets the MVR multicast vlan.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     multicastVlan     @b{(output)} MVR multicast vlan
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrMulticastVlanGet(openapiClientHandle_t *client_handle, 
                                        uint32_t *multicastVlan);

/*****************************************************************************
 * @purpose   Sets the MVR multicast vlan.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     multicastVlan     @b{(input)}  MVR multicast vlan
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrMulticastVlanSet(openapiClientHandle_t *client_handle,
                                        uint32_t multicastVlan);

/*****************************************************************************
 * @purpose   Check if receivers participate in specific vlan.
 *
 * @param     client_handle             @b{(input)}  client handle from 
 *                                                   registration API
 * @param     vlanId                    @b{(input)}  MVR multicast vlan
 * @param     rxVlanParticipationStatus @b{(output)} Receiver's vlan 
 *                                                   participation status
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrIsReceiversInVlan(openapiClientHandle_t *client_handle, 
                                         uint32_t vlanId, 
                                         OPEN_BOOL_t *rxVlanParticipationStatus);

/*****************************************************************************
 * @purpose   Gets the MVR interface mode.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ifNum               @b{(input)}  interface
 * @param     intfMode            @b{(output)} MVR interface mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrIntfModeGet(openapiClientHandle_t *client_handle, 
                                   uint32_t ifNum,    
                                   OPEN_CONTROL_t *intfMode);

/*****************************************************************************
 * @purpose   Sets the MVR interface mode.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     ifNum              @b{(input)}  interface
 * @param     intfMode           @b{(input)}  MVR interface mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrIntfModeSet(openapiClientHandle_t *client_handle, 
                                   uint32_t ifNum,    
                                   OPEN_CONTROL_t intfMode);

/*****************************************************************************
 * @purpose   Gets the interface MVR type of a specified interface.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ifNum             @b{(input)}  interface
 * @param     intfMvrType       @b{(output)} Interface MVR type
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrIntfMvrTypeGet(openapiClientHandle_t *client_handle, 
                                      uint32_t ifNum, 
                                      OPEN_MVR_INTF_TYPE_t *intfMvrType);

/*****************************************************************************
 * @purpose   Sets the interface MVR type of a specified interface.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ifNum             @b{(input)}  interface
 * @param     intfMvrType       @b{(input)}  Interface MVR type
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrIntfMvrTypeSet(openapiClientHandle_t *client_handle, 
                                      uint32_t ifNum, 
                                      OPEN_MVR_INTF_TYPE_t intfMvrType);

/*****************************************************************************
 * @purpose   Gets the immediate leave mode of a specified interface.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     ifNum              @b{(input)}  interface
 * @param     immediateMode      @b{(output)} Immediate leave mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrIntfImmediateModeGet(openapiClientHandle_t *client_handle, 
                                            uint32_t ifNum, 
                                            OPEN_CONTROL_t *immediateMode);

/*****************************************************************************
 * @purpose   Sets the immediate leave mode of a specified interface.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ifNum               @b{(input)}  interface
 * @param     immediateMode       @b{(input)}  Immediate leave mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrIntfImmediateModeSet(openapiClientHandle_t *client_handle,
                                            uint32_t ifNum, 
                                            OPEN_CONTROL_t immediateMode);

/*****************************************************************************
 * @purpose   Add static group IP address.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     groupIp          @b{(input)}  Group IP address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrGroupIpAdd(openapiClientHandle_t *client_handle, 
                                  uint32_t groupIp);

/*****************************************************************************
 * @purpose   Delete static group IP address.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     groupIp         @b{(input)}  Group IP address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrGroupIpDelete(openapiClientHandle_t *client_handle, 
                                     uint32_t groupIp);

/*****************************************************************************
 * @purpose   Add specified interface to the group.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     ifNum            @b{(input)}  interface
 * @param     groupIp          @b{(input)}  Group IP address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrIntfGroupIpAdd(openapiClientHandle_t *client_handle, 
                                      uint32_t ifNum, uint32_t groupIp);

/*****************************************************************************
 * @purpose   Delete specified interface from the group.
 *
 * @param     client_handle         @b{(input)}  client handle from registration API
 * @param     ifNum                 @b{(input)}  interface
 * @param     groupIp               @b{(input)}  Group IP address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrIntfGroupIpDelete(openapiClientHandle_t *client_handle, 
                                         uint32_t ifNum, uint32_t groupIp);

/*****************************************************************************
 * @purpose   Get next group ID.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     groupId          @b{(input)}  Group ID
 * @param     nextGroupId      @b{(output)} Next group Id
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrGroupEntryGetNext(openapiClientHandle_t *client_handle, 
                                         uint32_t groupId, uint32_t *nextGroupId);

/*****************************************************************************
 * @purpose   Get next group IP address.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     groupIp           @b{(input)}  Group IP
 * @param     nextGroupIp       @b{(output)} Next group IP
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrEntryGetNext(openapiClientHandle_t *client_handle, 
                                    uint32_t groupIp, uint32_t *nextGroupIp);

/*****************************************************************************
 * @purpose   Get the ip address of the group with specified ID.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     groupId            @b{(input)}  Group Id
 * @param     groupIp            @b{(output)} Group IP address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrGroupIpGet(openapiClientHandle_t *client_handle, 
                                  uint32_t groupId, uint32_t *groupIp);

/*****************************************************************************
 * @purpose   Get the interface type static or dynamic.
 *
 * @param     client_handle @b{(input)}  client handle from registration API
 * @param     ifNum         @b{(input)}  interface
 * @param     groupId       @b{(input)}  Group Id
 * @param     type          @b{(output)} Entry type
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrIntfEntryTypeGet(openapiClientHandle_t *client_handle, 
                                        uint32_t ifNum, uint32_t groupId, 
                                        OPEN_MVR_ENTRY_TYPE_t *type);

/*****************************************************************************
 * @purpose   Get next interface from the group with specified ID.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     groupId         @b{(input)}  Group Id
 * @param     ifNum           @b{(input)}  interface
 * @param     nextIfNum       @b{(output)} Next MVR interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrIntfGroupGetNext(openapiClientHandle_t *client_handle, 
                                        uint32_t groupId, uint32_t ifNum, 
                                        uint32_t *nextIfNum);

/*****************************************************************************
 * @purpose   Get next MVR interface.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     ifNum           @b{(input)}  interface
 * @param     nextIfNum       @b{(output)} Next MVR interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrIntfGetNext(openapiClientHandle_t *client_handle, 
                                   uint32_t ifNum, uint32_t *nextIfNum);

/*****************************************************************************
 * @purpose   Get the number of the group entries.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     groupEntries     @b{(output)} Number of the group entries
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrGroupNumGet(openapiClientHandle_t *client_handle, 
                                   uint32_t *groupEntries);

/*****************************************************************************
 * @purpose   Get the group status whether active or not.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     groupId          @b{(input)}  Group ID
 * @param     status           @b{(output)} Group status
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrGroupStatusGet(openapiClientHandle_t *client_handle, 
                                      uint32_t groupId, OPEN_BOOL_t *status);

/*****************************************************************************
 * @purpose   Get the group ID of the given IP address.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     groupIp          @b{(input)}  Group IP address
 * @param     groupId          @b{(output)} Group ID
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrGroupIdGet(openapiClientHandle_t *client_handle, 
                                  uint32_t groupIp, uint32_t *groupId);

/*****************************************************************************
 * @purpose   Get next group registered on specified interface.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     ifNum           @b{(input)}  interface
 * @param     groupId         @b{(input)}  Group Id
 * @param     nextGroupId     @b{(output)} Next registered group ID on the 
 *                                         interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrIntfGroupIdGetNext(openapiClientHandle_t *client_handle, 
                                          uint32_t ifNum, uint32_t groupId,
                                          uint32_t *nextGroupId);

/*****************************************************************************
 * @purpose   Get next group IP statically registered on specified interface.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     ifNum            @b{(input)}  interface
 * @param     groupIp          @b{(input)}  Group IP
 * @param     nextGroupIp      @b{(output)} Next statically registered group 
 *                                          IP on the interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrIntfStaticGroupIpGetNext(openapiClientHandle_t *client_handle,
                                                uint32_t ifNum, uint32_t groupIp,
                                                uint32_t *nextGroupIp);

/*****************************************************************************
 * @purpose   Get the global pdu counter values.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     counterType       @b{(input)}  Type of the counter
 * @param     *counterValue     @b{(output)} Counter value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrPduCounterGet(openapiClientHandle_t *client_handle, 
                                     OPEN_MVR_PDU_COUNTER_t counterType, 
                                     uint32_t *counterValue);


/*****************************************************************************
 * @purpose   Get the maximum number of multicast group entries.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration
 *                                          API.
 * @param     maxGroupCount    @b{(output)} Maximum number of multicast group
 *                                          entries.
 *
 * @returns   OPEN_E_NONE      On success.
 * @returns   OPEN_E_PARAM     Invalid argument.
 * @returns   OPEN_E_UNAVAIL   Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrMaxGroupCountGet(openapiClientHandle_t *client_handle,
                                        uint32_t *maxGroupCount);
#endif /* OPENAPI_MVR_H_INCLUDED */


