/**********************************************************************
*
*  Copyright 2020 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_mvrp.h
*
* @purpose   MVRP Open API(s)
*
* @component OPEN
*
* @create    06/18/2020
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_MVRP_H_INCLUDED
#define OPENAPI_MVRP_H_INCLUDED

#include "openapi_common.h" 
#include "openapi_vlan.h"

typedef enum
{
  OPEN_MVRP_VLAN_REGISTERED = 0,  /**< Registered VLAN  */
  OPEN_MVRP_VLAN_DECLARED         /**< Declared VLAN */
} OPEN_MVRP_VLAN_TYPE_t;

typedef enum 
{
  OPEN_MVRP_STATS_RX = 0,         /**< all received mvrp PDUs */
  OPEN_MVRP_STATS_RX_BAD_HEADER,  /**< received mvrp PDUs with bad header*/
  OPEN_MVRP_STATS_RX_BAD_FORMAT,  /**< received mvrp PDUs with bad data field*/
  OPEN_MVRP_STATS_TX,             /**< all transmitted mvrp PDUs*/
  OPEN_MVRP_STATS_TX_FAILURE,     /**< mvrp PDUs which failed*/
  OPEN_MVRP_STATS_MSG,            /**< message queue send failed*/
  OPEN_MVRP_STATS_REG_FAIL        /**< registration failed*/
} OPEN_MVRP_STATS_t;


/*****************************************************************************
 * @purpose   Get the global MVRP mode.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     mode           @b{(output)} Global admin mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MVRP feature is not supported.
 *
 * @notes     MVRP Can be enabled/disabled at global or interface level. This API gets the global admin mode status.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrpModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   set the global MVRP mode.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     mode           @b{(input)}  Global admin mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MVRP feature is not supported.
 *
 * @notes     MVRP Can be enabled/disabled at global or interface level. This API sets the global admin mode status.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrpModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Gets the MVRP periodic state machine mode.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     mode           @b{(output)} Periodic state machine mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MVRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrpPeriodicStateMachineModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Sets the MVRP periodic state machine mode.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     mode           @b{(input)}  Periodic state machine mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MVRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrpPeriodicStateMachineModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t  mode);

/*****************************************************************************
 * @purpose   Get the MVRP mode for the specified interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     intIfNum       @b{(input)}  Interface Number.
 * @param     mode           @b{(output)} Interface MVRP admin mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MVRP feature is not supported.
 *
 * @notes     MVRP Can be enabled/disabled at global or interface level. This API gets the interface admin mode status.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrpIntfModeGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the MVRP mode for the given interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     intIfNum       @b{(input)}  Interface Number.
 * @param     mode           @b{(input)}  Interface MVRP admin mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MVRP feature is not supported.
 *
 * @notes     MVRP Can be enabled/disabled at global or interface level. This API sets the interface admin mode status.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrpIntfModeSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Check if the current interface is valid for MVRP.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     intIfNum       @b{(input)}  Interface Number.
 * @param     isValid        @b{(output)} Flag to tell if Interface is valid MVRP interface.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MVRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrpIsValidIntfGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_BOOL_t *isValid);

/*****************************************************************************
 * @purpose   Given the current interface, get the next valid MVRP interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     intIfNum       @b{(input)}  Interface Number.
 * @param     nextIfNum      @b{(output)} Next valid MVRP Interface Number.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MVRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrpValidIntfNextGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *nextIfNum);

/*****************************************************************************
 * @purpose   Get the value of specified MVRP counter.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     type           @b{(input)}  Type of counter.
 * @param     counter        @b{(output)} Counter value.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MVRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrpTrafficPduCounterGet(openapiClientHandle_t *client_handle, OPEN_MVRP_STATS_t type, uint32_t *counter);

/*****************************************************************************
 * @purpose   Get the value of specified MVRP counter for an interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     intIfNum       @b{(input)}  Interface Number.
 * @param     type           @b{(input)}  Type of counter.
 * @param     counter        @b{(output)} Counter value.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MVRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrpIntfTrafficPduCounterGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_MVRP_STATS_t type, uint32_t *counter);

/*****************************************************************************
 * @purpose   Get MVRP VLAN array in the system.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanType       @b{(input)}  Flag to tell if vlan is Declared/Registered.
 * @param     vlanList       @b{(output)} MVRP VLAN List in the system.
 * @param     count          @b{(output)} Number of MVRP Vlans.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MVRP feature is not supported.
 * @returns   OPEN_E_INTERNAL         More than max allowed VLANs(4096) are returned by application/
 *                                    Something is wrong internally.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrpVlanArrayGet(openapiClientHandle_t *client_handle, OPEN_MVRP_VLAN_TYPE_t vlanType, OPEN_VLAN_LIST_t *vlanList, uint32_t *count);

/*****************************************************************************
 * @purpose   Get MVRP VLAN array on an interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     intIfNum       @b{(input)}  Interface Number.
 * @param     vlanType       @b{(input)}  Flag to tell if vlan is Declared/Registered.
 * @param     vlanList       @b{(output)} MVRP VLAN List on the interface given.
 * @param     count          @b{(output)} Number of MVRP Vlans.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MVRP feature is not supported.
 * @returns   OPEN_E_INTERNAL         More than max allowed VLANs(4096) are returned by application/
 *                                    Something is wrong internally.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrpIntfVlanArrayGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_MVRP_VLAN_TYPE_t vlanType, OPEN_VLAN_LIST_t *vlanList, uint32_t *count);

/*****************************************************************************
 * @purpose   Clear MVRP traffic counters on specified interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     intIfNum       @b{(input)}  Interface Number.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MVRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrpTrafficCountersPerIfClear(openapiClientHandle_t *client_handle, uint32_t intIfNum);

/*****************************************************************************
 * @purpose   Clear global MVRP traffic counters.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MVRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrpTrafficGlobalCountersClear(openapiClientHandle_t *client_handle);

/*****************************************************************************
 * @purpose   Clear all MVRP traffic counters.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MVRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMvrpTrafficCountersClear(openapiClientHandle_t *client_handle);

#endif /* OPENAPI_MVRP_H_INCLUDED */

