/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_ospf_config.h
*
* @purpose   Open Shortest Path First Configuration and Status 
*
* @component OPEN
*
* @create    06/25/2013
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_OSPF_CONFIG_H_INCLUDED
#define OPENAPI_OSPF_CONFIG_H_INCLUDED

#include <stdbool.h>
#include "openapi_common.h"

#define OPEN_OSPF_INTF_SECONDARIES_NONE 0            /**< secondaries are not advertised */
#define OPEN_OSPF_INTF_SECONDARIES_ALL  0xFFFFFFFFUL /**< all secondaries are advertised */

#define OPEN_OSPF_INTF_METRIC_NOT_CONFIGURED 0       /**< Metric computed from port speed */

//! Areas are identified by a 32-bit Area ID.
/// The Area ID may be expressed either as a decimal number or in dotted decimal
typedef enum
{
  OPEN_OSPF_AREA_ID_INTEGER_FORMAT        = 0, /**< Integer format */
  OPEN_OSPF_AREA_ID_DOTTED_DECIMAL_FORMAT = 1  /**< Dotted decimal format */
} OPEN_OSPF_AREA_ID_FORMAT_t;

typedef enum
{
  OPEN_OSPF_AREA_IMPORT_EXT     = 1, /**< Import and propagate T5 LSAs */
  OPEN_OSPF_AREA_IMPORT_NO_EXT  = 2, /**< Do not import and propagate ext LSAs- make stub area*/
  OPEN_OSPF_AREA_IMPORT_NSSA    = 3  /**< Import and propagate T7 LSAs */
} OPEN_OSPF_AREA_EXT_ROUTING_t;

typedef enum
{
  OPEN_OSPF_AREA_NO_SUMMARY     = 1, /**< Do not originate and propagate summary LSAs */
  OPEN_OSPF_AREA_SEND_SUMMARY   = 2  /**< Originate and propagate summary LSAs */
} OPEN_OSPF_AREA_SUMMARY_t;

typedef enum
{
  OPEN_OSPF_INTF_DEFAULT        = 0, /**< Default interface */
  OPEN_OSPF_INTF_BROADCAST      = 1, /**< Broadcast */
  OPEN_OSPF_INTF_PTP            = 3  /**< Point to point */
} OPEN_OSPF_INTF_TYPES_t;

//! The cost, or external metric in the route.
typedef enum
{
  OPEN_OSPF_METRIC_EXT_TYPE1    = 1, /**< External Type 1 */
  OPEN_OSPF_METRIC_EXT_TYPE2    = 2  /**< External Type 2 */
} OPEN_OSPF_EXT_METRIC_TYPES_t;

typedef enum
{
  OPEN_OSPF_STUB_RTR_CFG_DISABLED = 0, /**< Not configured as a stub router. */
  OPEN_OSPF_STUB_RTR_CFG_ALWAYS   = 1, /**< Configured as a stub router. */
  OPEN_OSPF_STUB_RTR_CFG_STARTUP  = 2  /**< Is a stub router for a limited time after reboot. */
} OPEN_OSPF_STUB_ROUTER_CFG_t;

//! Type of Service
typedef enum
{
  OPEN_OSPF_TOS_NORMAL_SERVICE        = 0, /**< normal service */
  OPEN_OSPF_TOS_MINIMIZE_COST         = 2, /**< minimize cost */
  OPEN_OSPF_TOS_MAXIMIZE_RELIABILITY  = 4, /**< maximize reliability */
  OPEN_OSPF_TOS_MAXIMIZE_THROUGHPUT   = 8, /**< maximize throughput */
  OPEN_OSPF_TOS_MINIMIZE_DELAY        = 16  /**< minimize delay */
} OPEN_OSPF_TOS_METRIC_TYPES_t;

//! OSPFv2 MIB trap bit masks
typedef enum
{
  OPEN_OSPF_TRAP_NONE                       = 0x0,     /**< no traps */
  OPEN_OSPF_TRAP_VIRT_IF_STATE_CHANGE       = 0x1,     /**< virtual interface, state change */
  OPEN_OSPF_TRAP_NBR_STATE_CHANGE           = 0x2,     /**< non-virtual interface, OSPF neighbor state change */
  OPEN_OSPF_TRAP_VIRT_NBR_STATE_CHANGE      = 0x4,     /**< virtual interface, OSPF neighbor state change */
  OPEN_OSPF_TRAP_IF_CONFIG_ERROR            = 0x8,     /**< non-virtual interface, parameter conflict */
  OPEN_OSPF_TRAP_VIRT_IF_CONFIG_ERROR       = 0x10,    /**< virtual interface, parameter conflict */
  OPEN_OSPF_TRAP_IF_AUTH_FAILURE            = 0x20,    /**< non-virtual interface, auth mismatch*/
  OPEN_OSPF_TRAP_VIRT_IF_AUTH_FAILURE       = 0x40,    /**< virtual interface, auth mismatch */
  OPEN_OSPF_TRAP_RX_BAD_PACKET              = 0x80,    /**< non-virtual interface, can't be parsed */
  OPEN_OSPF_TRAP_VIRT_IF_RX_BAD_PACKET      = 0x100,   /**< virtual interface, can't be parsed */
  OPEN_OSPF_TRAP_TX_RETRANSMIT              = 0x200,   /**< virtual interface, retransmitted */
  OPEN_OSPF_TRAP_VIRT_IF_TX_RETRANSMIT      = 0x400,   /**< non-virtual interface, retransmitted */
  OPEN_OSPF_TRAP_ORIGINATE_LSA              = 0x800,   /**< new link-state has originated */
  OPEN_OSPF_TRAP_MAX_AGE_LSA                = 0x1000,  /**< link-state database has aged */
  OPEN_OSPF_TRAP_LS_DB_OVERFLOW             = 0x2000,  /**< LSDB exceeded limits */
  OPEN_OSPF_TRAP_LS_DB_APPROACHING_OVERFLOW = 0x4000,  /**< LSDB exceeded 90% of limits */
  OPEN_OSPF_TRAP_IF_STATE_CHANGE            = 0x8000,  /**< non-virtual interface, state change */
  OPEN_OSPF_TRAP_IF_RX_PACKET               = 0x10000, /**< non-virtual interface, can't be parsed */
  OPEN_OSPF_TRAP_RTB_ENTRY_INFO             = 0x20000  /**< table entry */
} OPEN_OSPF_TRAP_TYPES_t;

/*********************************************************************
* @purpose  Get the area summary.
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    areaId              @b{(input)}  Stub or NSSA Id, can be decimal or IP address
* @param    summary             @b{(output)} summary (see notes)
*  
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes    If summary is OPEN_OSPF_AREA_SEND_SUMMARY; 
*           The router will both summarize and propagate summary LSAs. 
* @notes    If summary is OPEN_OSPF_AREA_NO_SUMMARY; 
*           The router will neither originate nor propagate summary LSAs into the 
*           stub area.  It will rely entirely on its default route.
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapAreaSummaryGet(openapiClientHandle_t *client_handle,
                                          uint32_t areaId,
                                          OPEN_OSPF_AREA_SUMMARY_t *summary);

/*********************************************************************
* @purpose  Set the area summary.
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    areaId              @b{(input)}  Stub or NSSA Id, can be decimal or IP address
* @param    summary             @b{(input)}  summary (see notes)
*  
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      if area is the backbone
* 
* @notes    If summary is OPEN_OSPF_AREA_SEND_SUMMARY; 
*           The router will both summarize and propagate summary LSAs. 
* @notes    If summary is OPEN_OSPF_AREA_NO_SUMMARY; 
*           The router will neither originate nor propagate summary LSAs into the 
*           stub area.  It will rely entirely on its default route. 
* @notes    Calling this API will change the running configuration of the switch. 
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapAreaSummarySet(openapiClientHandle_t *client_handle,
                                          uint32_t areaId,
                                          OPEN_OSPF_AREA_SUMMARY_t summary);

/*********************************************************************
* @purpose  Get the autocost reference bandwidth for OSPF links
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    autoCost            @b{(output)} autoCost reference bandwidth allowed
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapAutoCostRefBwGet(openapiClientHandle_t *client_handle,
                                            uint32_t *autoCost);

/*********************************************************************
* @purpose  Set the autocost reference bandwidth for OSPF links
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    autoCost            @b{(input)}  autoCost reference bandwidth allowed
* @param    -                   Minimum value @ref OPENAPI_OSPF_MIN_REF_BW
* @param    -                   Maximum value @ref OPENAPI_OSPF_MAX_REF_BW
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    Calling this API will change the running configuration of the switch. 
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapAutoCostRefBwSet(openapiClientHandle_t *client_handle,
                                            uint32_t autoCost);

/********************************************************************* 
* @purpose  Get the area's support for importing AS external 
*           link-state advertisements.
* 
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    areaId              @b{(input)}  Stub or NSSA Id, can be decimal or IP address
* @param    import              @b{(output)} import/propagate T5, T7, or no import/stub
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes    if import is OPEN_OSPF_AREA_IMPORT_EXT
*           Import and propagate T5 LSAs
* @notes    if import is OPEN_OSPF_AREA_IMPORT_NO_EXT
*           Do not import and propagate ext LSAs- make stub area
* @notes    if import is OPEN_OSPF_AREA_IMPORT_NSSA
*           Import and propagate T7 LSAs
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapImportAsExternGet(openapiClientHandle_t *client_handle,
                                             uint32_t areaId, 
                                             OPEN_OSPF_AREA_EXT_ROUTING_t *import);

/********************************************************************* 
* @purpose  Set the area's support for importing AS external 
*           link-state advertisements.
* 
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    areaId              @b{(input)}  Stub or NSSA Id, can be decimal or IP address
* @param    import              @b{(input)}  import/propagate T5, T7, or no import/stub
*  
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      if attempt to make backbone a stub or NSSA 
* @returns  OPEN_E_EXISTS       if area id is a virtual link
* 
* @notes    if import is OPEN_OSPF_AREA_IMPORT_EXT
*           Import and propagate T5 LSAs
* 
* @notes    if import is OPEN_OSPF_AREA_IMPORT_NO_EXT
*           Do not import and propagate ext LSAs- make stub area
* @notes    if import is OPEN_OSPF_AREA_IMPORT_NSSA
*           Import and propagate T7 LSAs 
* @notes    Calling this API will change the running configuration of the switch. 
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end 
*********************************************************************/
open_error_t openapiOspfMapImportAsExternSet(openapiClientHandle_t *client_handle,
                                             uint32_t areaId, 
                                             OPEN_OSPF_AREA_EXT_ROUTING_t import);

/*********************************************************************
* @purpose  Get the OSPF admin mode for the specified interface 
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    intf                @b{(input)}  internal interface number
* @param    mode                @b{(output)} OPEN_ENABLE or OPEN_DISABLE
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      if interface does not exist or
*                               if interface has not been enabled for routing
*
* @notes
*       
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapIntfAdminModeGet(openapiClientHandle_t *client_handle,
                                            uint32_t intf,
                                            OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Set the OSPF admin mode for the specified interface 
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    intf                @b{(input)}  internal interface number
* @param    mode                @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      if interface does not exist or
*                               if interface has not been enabled for routing
*
* @notes    Called when OSPF is explicitly enabled on an interface,
*           such as when "ip ospf area" command is given. Not called for
*           interfaces matching a "network area" command.
* @notes    Calling this API will change the running configuration of the switch. 
*       
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapIntfAdminModeSet(openapiClientHandle_t *client_handle,
                                            uint32_t intf,
                                            OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose  Gets the OSPF area id for the specified interface 
* 
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    intf                @b{(input)}  internal interface number
* @param    areaId              @b{(output)} Stub or NSSA Id, can be decimal or IP address
* @param    mode                @b{(output)} admin mode for the specified interface
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed. 
*
* @notes
*       
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapIntfAreaIdGet(openapiClientHandle_t *client_handle,
                                         uint32_t intf,
                                         uint32_t *areaId,
                                         OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Sets the OSPF area id for the specified interface 
* 
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    intf                @b{(input)}  internal interface number
* @param    areaId              @b{(input)}  Stub or NSSA Id, can be decimal or IP address
* @param    mode                @b{(input)}  admin mode for the specified interface
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed. 
*
* @notes    After successfully setting the area id, this OpEN_API sets the OSPF 
*           admin mode. Typically, OSPF should be disabled on the target interface 
*           when the area id is not defined.
* @notes    Calling this API will change the running configuration of the switch. 
*       
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapIntfAreaIdSet(openapiClientHandle_t *client_handle,
                                         uint32_t intf,
                                         uint32_t areaId,
                                         OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose  Gets the Ospf Dead Interval for the specified interface 
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    intf                @b{(input)}  internal interface number
* @param    seconds             @b{(output)} see notes
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      if interface does not exist or
*                               if interface has not been enabled for routing
*
* @notes    The number of seconds that a router's Hello packets have not 
*           been seen before it's neighbors declare the router down. 
*           This should be some multiple of the Hello interval. 
*           This value must be the same for all routers attached to a common network.
*       
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapIntfDeadIntervalGet(openapiClientHandle_t *client_handle,
                                               uint32_t intf,
                                               uint32_t *seconds);

/*********************************************************************
* @purpose  Sets the Ospf Dead Interval for the specified interface 
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    intf                @b{(input)}  internal interface number
* @param    seconds             @b{(input)}  see notes
* @param    -                   Minimum value @ref OPENAPI_OSPF_INTF_DEAD_MIN
* @param    -                   Maximum value @ref OPENAPI_OSPF_INTF_DEAD_MAX
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      if interface does not exist or
*                               if interface has not been enabled for routing
*
* @notes    The number of seconds that a router's Hello packets have not 
*           been seen before it's neighbors declare the router down. 
*           This should be some multiple of the Hello interval. 
*           This value must be the same for all routers attached to a common network.
* @notes    Calling this API will change the running configuration of the switch. 
*       
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapIntfDeadIntervalSet(openapiClientHandle_t *client_handle,
                                               uint32_t intf,
                                               uint32_t seconds);

/*********************************************************************
* @purpose  Get the Ospf Hello Interval for the specified interface 
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    intf                @b{(input)}  internal interface number
* @param    seconds             @b{(output)} see notes
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      if interface does not exist or
*                               if interface has not been enabled for routing
*
* @notes    The length of time, in seconds between the Hello packets that the 
*           router sends on the interface. This value must be the same for all 
*           router attached to a common network.
*       
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapIntfHelloIntervalGet(openapiClientHandle_t *client_handle,
                                                uint32_t intf,
                                                uint32_t *seconds);

/*********************************************************************
* @purpose  Sets the Ospf Hello Interval for the specified interface 
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    intf                @b{(input)}  internal interface number
* @param    seconds             @b{(input)}  see notes
* @param    -                   Minimum value @ref OPENAPI_OSPF_INTF_SECONDS_MIN
* @param    -                   Maximum value @ref OPENAPI_OSPF_INTF_SECONDS_MAX
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      if interface does not exist or
*                               if interface has not been enabled for routing
*
* @notes    The length of time, in seconds between the Hello packets that the 
*           router sends on the interface. This value must be the same for all 
*           router attached to a common network.
* @notes    Calling this API will change the running configuration of the switch. 
*       
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapIntfHelloIntervalSet(openapiClientHandle_t *client_handle,
                                                uint32_t intf,
                                                uint32_t seconds);

/*********************************************************************
* @purpose  Get the interface metric value.
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    intf                @b{(input)}  internal interface number
* @param    tos                 @b{(input)}  TOS metric
* @param    metric              @b{(output)} metric value
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed. 
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapIntfMetricValueGet(openapiClientHandle_t *client_handle,
                                              uint32_t intf,
                                              OPEN_OSPF_TOS_METRIC_TYPES_t tos,
                                              uint32_t *metric);

/*********************************************************************
* @purpose  Set the interface metric value.
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    intf                @b{(input)}  internal interface number
* @param    ipAddr              @b{(input)}  ip address
* @param    tos                 @b{(input)}  type of service metric
* @param    metric              @b{(input)}  metric value
* @param    -                   Minimum value @ref OPENAPI_OSPF_INTF_COST_MIN
* @param    -                   Maximum value @ref OPENAPI_OSPF_INTF_COST_MAX
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed. 
* @returns  OPEN_E_UNAVAIL      Not supported for non-default TOS values.
*
* @notes    If metric is @ref OPEN_OSPF_INTF_METRIC_NOT_CONFIGURED, the metric 
*           will be computed from port speed otherwise only the value 
*           configured by user will be used.
* @notes    Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapIntfMetricValueSet(openapiClientHandle_t *client_handle,
                                              uint32_t intf,
                                              uint32_t ipAddr,
                                              OPEN_OSPF_TOS_METRIC_TYPES_t tos,
                                              uint32_t metric);

/*********************************************************************
* @purpose  gets the passive mode of the interface
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    intf                @b{(input)}  internal interface number
* @param    mode                @b{(output)} passive mode of interface
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      if interface does not exist or
*                               if interface has not been enabled for routing
*
* @notes
*       
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapIntfPassiveModeGet(openapiClientHandle_t *client_handle,
                                              uint32_t intf,
                                              bool *mode);

/*********************************************************************
* @purpose  Sets the passive mode of the interface
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    intf                @b{(input)}  internal interface number
* @param    mode                @b{(input)}  passive mode of interface
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      if interface does not exist or
*                               if interface has not been enabled for routing
*
* @notes    Calling this API will change the running configuration of the switch. 
*       
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapIntfPassiveModeSet(openapiClientHandle_t *client_handle,
                                              uint32_t intf,
                                              bool mode);

/*********************************************************************
* @purpose  Get the SecondariesFlag for the specified interface
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    intf                @b{(input)}  internal interface number
* @param    secondaries         @b{(output)} see notes
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes    if secondaries @ref OPEN_OSPF_INTF_SECONDARIES_ALL, all are advertised
* @notes    if secondaries @ref OPEN_OSPF_INTF_SECONDARIES_NONE, none are advertised
*       
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapIntfSecondariesFlagGet(openapiClientHandle_t *client_handle,
                                                  uint32_t intf,
                                                  uint32_t *secondaries);

/*********************************************************************
* @purpose  Set the SecondariesFlag for the specified interface
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    intf                @b{(input)}  Internal Interface Number
* @param    secondaries         @b{(input)}  see notes
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes    if secondaries @ref OPEN_OSPF_INTF_SECONDARIES_ALL, all are advertised
* @notes    if secondaries @ref OPEN_OSPF_INTF_SECONDARIES_NONE, none are advertised
* @notes    Calling this API will change the running configuration of the switch. 
*       
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapIntfSecondariesFlagSet(openapiClientHandle_t *client_handle,
                                                  uint32_t intf,
                                                  uint32_t secondaries);

/*********************************************************************
* @purpose  Get the OSPF interface type.
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    intf                @b{(input)}  internal interface number
* @param    type                @b{(output)} configure network type
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapIntfTypeGet(openapiClientHandle_t *client_handle,
                                       uint32_t intf,
                                       OPEN_OSPF_INTF_TYPES_t *type);

/*********************************************************************
* @purpose  Set the OSPF interface type.
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    intf                @b{(input)}  internal interface number
* @param    type                @b{(input)}  allowable network types
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes    Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapIntfTypeSet(openapiClientHandle_t *client_handle,
                                       uint32_t intf,
                                       OPEN_OSPF_INTF_TYPES_t type);

/*********************************************************************
* @purpose  Get logging information of neighbor state changes.
* 
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    log                 @b{(output)} logging of adjacency changes 
* @param    detail              @b{(output)} detailed logging
*  
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes    if log is OPEN_ENABLE, then enable logging of adjacency changes
* @notes    if log is OPEN_DISABLE, then disable logging of adjacency changes
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapLogAdjChngGet(openapiClientHandle_t *client_handle,
                                         OPEN_CONTROL_t *log,
                                         bool *detail);

/*********************************************************************
* @purpose  Configure logging of neighbor state changes.
* 
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    log                 @b{(input)}  logging of adjacency changes 
* @param    detail              @b{(input)}  detailed logging
*  
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes    if log is OPEN_ENABLE, then enable logging of adjacency changes
* @notes    if log is OPEN_DISABLE, then disable logging of adjacency  changes
* @notes    Calling this API will change the running configuration of the switch. 
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapLogAdjChngSet(openapiClientHandle_t *client_handle,
                                         OPEN_CONTROL_t log,
                                         bool detail);

/*********************************************************************
* @purpose  Get the maximum number of paths that OSPF can report
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    maxPaths            @b{(output)} the maximum number of paths allowed
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed. 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapMaxPathsGet(openapiClientHandle_t *client_handle,
                                       uint32_t *maxPaths);

/*********************************************************************
* @purpose  Set the maximum number of paths that OSPF can report
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    maxPaths            @b{(input)}  see notes
* @param    -                   Minimum value 1
* @param    -                   Maximum value openapiOspfEqualCostRoutesMax()
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed. 
* 
* @notes    The maxPaths value is the maximum number of paths supported for
*           the given platform. The maximum paths is essential the maximum
*           number of ECMP next hops supported for the active template. This
*           value can be obtained by using openapiOspfEqualCostRoutesMax().
* @notes    Calling this API will change the running configuration of the switch. 
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapMaxPathsSet(openapiClientHandle_t *client_handle,
                                       uint32_t maxPaths);

/*********************************************************************
* @purpose  Delete a network area entry.
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    ipAddr              @b{(input)}  ip address
* @param    wildcardMask        @b{(input)}  wild card mask
* @param    areaId              @b{(input)}  Stub or NSSA Id, can be decimal or IP address
*
* @returns  OPEN_E_NONE         Delete successful.
* @returns  OPEN_E_FAIL         Delete failed.
* @returns  OPEN_E_PARAM        Error in parameter passed. 
*
* @notes    Calling this API will change the running configuration of the switch.  
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapNetworkAreaEntryDelete(openapiClientHandle_t *client_handle,
                                                  uint32_t ipAddr,
                                                  uint32_t wildcardMask,
                                                  uint32_t areaId);

/*********************************************************************
* @purpose  Find a specific network area object.
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    ipAddr              @b{(input)}  ip address
* @param    wildcardMask        @b{(input)}  wild card mask
* @param    areaId              @b{(input)}  Stub or NSSA Id, can be decimal or IP address
* @param    found               @b{(output)} network object exists or not
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed. 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapNetworkAreaEntryGet(openapiClientHandle_t *client_handle,
                                               uint32_t ipAddr,
                                               uint32_t wildcardMask,
                                               uint32_t areaId,
                                               bool *found);

/*********************************************************************
* @purpose  Create/Update a network area entry. This command defines the 
*           interfaces covered by the network ipAddr on which OSPF runs 
*           and also defines the area ID for those interfaces.
* 
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    ipAddr              @b{(input)}  ip address
* @param    wildcardMask        @b{(input)}  wild card mask
* @param    areaId              @b{(input)}  Stub or NSSA Id
* @param    areaIdFmt           @b{(input)}  areaId format: integer/dotted-decimal
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed. 
*
* @notes    areaIdFmt can be either OPEN_OSPF_AREA_ID_INTEGER_FORMAT or
*           OPEN_OSPF_AREA_ID_DOTTED_DECIMAL_FORMAT 
* @notes    Calling this API will change the running configuration of the switch.  
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapNetworkAreaEntrySet(openapiClientHandle_t *client_handle,
                                               uint32_t ipAddr,
                                               uint32_t wildcardMask,
                                               uint32_t areaId,
                                               OPEN_OSPF_AREA_ID_FORMAT_t areaIdFmt);

/*********************************************************************
* @purpose  Get the import summary configuration for the specified NSSA
* 
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    areaId              @b{(input)}  Stub or NSSA Id, can be decimal or IP address
* @param    summary             @b{(output)} import summary (see notes)
*  
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes    If summary is OPEN_OSPF_AREA_SEND_SUMMARY; 
*           OSPF's summary routes are imported into the NSSA as Type-3 summary-LSAs. 
* @notes    If summary is OPEN_OSPF_AREA_NO_SUMMARY; 
*           Summary routes are not imported into the NSSA. 
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
OPEN_RC_t ospfMapNSSAImportSummariesGet (OPEN_uint32 areaId, OPEN_uint32 *impSum)
*********************************************************************/
open_error_t openapiOspfMapNSSAImportSummariesGet(openapiClientHandle_t *client_handle,
                                                  uint32_t areaId, 
                                                  OPEN_OSPF_AREA_SUMMARY_t *summary);

/*********************************************************************
* @purpose  Set the import summary configuration for the specified NSSA
* 
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    areaId              @b{(input)}  Stub or NSSA Id, can be decimal or IP address
* @param    summary             @b{(input)}  import summary (see notes)
*  
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      if area is the backbone
*
* @notes    If summary is OPEN_OSPF_AREA_SEND_SUMMARY; 
*           OSPF's summary routes are imported into the NSSA as Type-3 summary-LSAs. 
* @notes    If summary is OPEN_OSPF_AREA_NO_SUMMARY; 
*           Summary routes are not imported into the NSSA. 
* @notes    Calling this API will change the running configuration of the switch. 
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
OPEN_RC_t ospfMapNSSAImportSummariesSet (OPEN_uint32 areaId, OPEN_uint32 impSum)
*********************************************************************/
open_error_t openapiOspfMapNSSAImportSummariesSet(openapiClientHandle_t *client_handle,
                                                  uint32_t areaId, 
                                                  OPEN_OSPF_AREA_SUMMARY_t summary);

/********************************************************************* 
* @purpose  Set configuration parameters for OSPF's redistribution 
*           of routes from other sources. 
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    redistribute        @b{(input)}  Whether OSPF redistributes from sourceProto
* @param    sourceProto         @b{(input)}  Other protocol from which OSPF will redistribute.
* @param    -                   Maximum length openapiRouteProtoNameLenMax()
* @param    distList            @b{(input)}  See notes
* @param    -                   Minimum value @ref OPENAPI_OSPF_DIST_LIST_MIN
* @param    -                   Maximum value @ref OPENAPI_OSPF_DIST_LIST_MAX
* @param    metric              @b{(input)}  See notes
* @param    -                   Minimum value @ref OPENAPI_OSPF_REDIST_METRIC_MIN
* @param    -                   Maximum value @ref OPENAPI_OSPF_REDIST_METRIC_MAX
* @param    type                @b{(input)}  Metric type OSPF advertises for routes from sourceProto
* @param    tag                 @b{(input)}  See notes
* @param    -                   Minimum value 0
* @param    -                   Maximum value unsigned integer (4294967295)
* @param    subnets             @b{(input)}  If true, OSPF will redistribute subnetted routes from this 
*                                            protocol.
* 
* @returns  OPEN_E_NONE         set successful.
* @returns  OPEN_E_FAIL         set failed.
* @returns  OPEN_E_PARAM        error in parameter passed.  
* @returns  OPEN_E_FULL         if route-map couldn't be created 
* @returns  OPEN_E_NOT_FOUND    if the route-map name doesn't match (no form)
* @returns  OPEN_E_UNAVAIL      either the sourceProto does not identify a protocol from which OSPF may redistribute,
*                               or redistribute metric is out of range. 
* 
* @notes    distList - access list number identifying the distribute list 
*           used to filter routes OSPF redistributes from other protocols. 
*           @ref OPENAPI_OSPF_DIST_LIST_NA indicates no distribute list is configured.
* @notes    metric - routes OSPF redistributes from this protocol will be
*           advertised with this metric. A value of @ref OPENAPI_OSPF_REDIST_METRIC_NA
*           indicates that no metric is configured. 
* @notes    tag - routes OSPF redistributes from this protocol will be advertised
*           with this tag. 
* @notes    Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.2
*
* @end 
*********************************************************************/
open_error_t openapiOspfMapRedistributionSet(openapiClientHandle_t *client_handle,
                                             bool redistribute,
                                             open_buffdesc *sourceProto,
                                             uint32_t distList,
                                             uint32_t metric,
                                             OPEN_OSPF_EXT_METRIC_TYPES_t type,
                                             int32_t tag,
                                             bool subnets);

/*********************************************************************
* @purpose  Get the OSPF router ID
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    routerId            @b{(output)} router id
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapRouterIdGet(openapiClientHandle_t *client_handle,
                                       uint32_t *routerId);

/*********************************************************************
* @purpose  Set the OSPF router ID
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    routerId            @b{(input)}  router id, can be decimal or IP address
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes    To set a non-zero router Id, either OSPF must be disabled or
*           the router Id must be 0.
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapRouterIdSet(openapiClientHandle_t *client_handle,
                                       uint32_t routerId);

/*********************************************************************
* @purpose  Get OSPF mode for RFC 1583 compatibility
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    mode                @b{(output)} see notes
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes    If mode is OPEN_ENABLE, preference rules remain those as defined 
*           by RFC 1583. 
* @notes    If mode is OPEN_DISABLE, preference rules are as stated in Section 
*           16.4.1 of the OSPFv2 standard. When disabled, greater protection
*           is provided against routing loops.  
*       
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapRfc1583CompatibilityGet(openapiClientHandle_t *client_handle,
                                                   OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Configure if OSPF is compatible with RFC 1583
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    mode                @b{(input)}  see notes
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes    If mode is OPEN_ENABLE, preference rules remain those as defined 
*           by RFC 1583. 
* @notes    If mode is OPEN_DISABLE, preference rules are as stated in Section 
*           16.4.1 of the OSPFv2 standard. When disabled, greater protection
*           is provided against routing loops. 
* @notes    Calling this API will change the running configuration of the switch. 
* @notes    Source can also be a loopback interface.  
*       
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapRfc1583CompatibilitySet(openapiClientHandle_t *client_handle,
                                                   OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose  Gets the delay time between when OSPF receives a topology
*           change and when it starts an SPF calculation
* 
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    time                @b{(output)} SPF Delay Time
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapSpfDelayTimeGet(openapiClientHandle_t *client_handle,
                                           uint32_t *time);

/*********************************************************************
* @purpose  Sets the delay time between when OSPF receives a topology
*           change and when it starts an SPF calculation
* 
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    time                @b{(input)}  SPF Delay Time
* @param    -                   Minimum value @ref OPENAPI_OSPF_SPF_DELAY_TIME_MIN
* @param    -                   Maximum value @ref OPENAPI_OSPF_SPF_DELAY_TIME_MAX
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    Calling this API will change the running configuration of the switch. 
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapSpfDelayTimeSet(openapiClientHandle_t *client_handle,
                                           uint32_t time);

/*********************************************************************
* @purpose  Gets the minimum time (in seconds) between two consecutive
*           SPF calculations
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    time                @b{(output)} SPF Hold Time
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapSpfHoldTimeGet(openapiClientHandle_t *client_handle,
                                          uint32_t *time);

/*********************************************************************
* @purpose  Sets the minimum time (in seconds) between two consecutive
*           SPF calculations
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    time                @b{(input)}  SPF Hold Time
* @param    -                   Minimum value @ref OPENAPI_OSPF_SPF_HOLD_TIME_MIN
* @param    -                   Maximum value @ref OPENAPI_OSPF_SPF_HOLD_TIME_MAX
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    Calling this API will change the running configuration of the switch. 
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapSpfHoldTimeSet(openapiClientHandle_t *client_handle,
                                          uint32_t time);

/*********************************************************************
* @purpose  Get the configured stub router advertisement mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    mode                @b{(output)} the configured stub router mode
* @param    duration            @b{(output)} If the configured stub router mode is
*                                            stub router at startup, the duration 
*                                            is the startup time in seconds. May be NULL
*                                            if caller doesn't care about startup time.
* @param    override            @b{(output)} override the metric in summary LSAs
*                                            when in stub router mode.
* @param    metric              @b{(output)} Metric to set in summary LSAs in stub 
*                                            router mode, if overriding summary metric
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    if mode is OPEN_OSPF_STUB_RTR_CFG_DISABLED
*           not configured as a stub router.
* @notes    if mode is OPEN_OSPF_STUB_RTR_CFG_ALWAYS
*           configured as a stub router.
* @notes    if mode is OPEN_OSPF_STUB_RTR_CFG_STARTUP
*           is a stub router for a limited time after reboot.
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapStubRtrModeGet(openapiClientHandle_t *client_handle,
                                          OPEN_OSPF_STUB_ROUTER_CFG_t *mode, 
                                          uint32_t *duration, 
                                          bool *override, 
                                          uint32_t *metric);

/*********************************************************************
* @purpose  Set the configured stub router advertisement mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    mode                @b{(input)}  the configured stub router mode
* @param    duration            @b{(input)}  If the configured stub router mode is
*                                            stub router at startup, the duration 
*                                            is the startup time in seconds. May be NULL
*                                            if caller doesn't care about startup time.
* @param    -                   Minimum value @ref OPENAPI_OSPF_STUB_ROUTER_DURATION_MIN
* @param    -                   Maximum value @ref OPENAPI_OSPF_STUB_ROUTER_DURATION_MAX
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed. 
*
* @notes    Calling this API will change the running configuration of the switch. 
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfMapStubRtrModeSet(openapiClientHandle_t *client_handle,
                                          OPEN_OSPF_STUB_ROUTER_CFG_t mode, 
                                          uint32_t duration);

/*********************************************************************
* @purpose  Set the configured stub router advertisement mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    override            @b{(input)}  override the metric in summary LSAs
*                                            when in stub router mode. 
* @param    metric              @b{(input)}  Metric to set in summary LSAs in stub 
*                                            router mode, if overriding summary metric
* @param    -                   Minimum value @ref OPENAPI_OSPF_STUB_ROUTER_SUMLSA_MET_MIN
* @param    -                   Maximum value @ref OPENAPI_OSPF_STUB_ROUTER_SUMLSA_MET_MAX 
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed. 
*
* @notes    Calling this API will change the running configuration of the switch. 
*
* @supported-in-version OpEN API Version: 1.2
*
* @end 
*********************************************************************/
open_error_t openapiOspfMapStubRtrSumLsaMetricOverride(openapiClientHandle_t *client_handle,
                                                       bool override,
                                                       uint32_t metric);

/*********************************************************************
*
* @purpose  Get the specified trap type mode
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    trapType            @b{(input)}  ospf trap type (see notes)
* @param    mode                @b{(output)} trap type (on or off)
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed. 
*
* @notes    Any combination of trapType is valid. 
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiOspfTrapModeGet(openapiClientHandle_t *client_handle,
                                    OPEN_OSPF_TRAP_TYPES_t trapType,
                                    bool *mode);

/*********************************************************************
*
* @purpose  Allows a user to enable or disable traps based on the
*           specified mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API 
* @param    trapType            @b{(input)}  ospf trap type (see notes)
* @param    mode                @b{(input)}  Turn trap type (on or off)
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed. 
*
* @notes    Any combination of trapType is valid. 
* @notes    Calling this API will change the running configuration of the switch. 
*
* @supported-in-version OpEN API Version: 1.2
*
* @end 
*********************************************************************/
open_error_t openapiOspfTrapModeSet(openapiClientHandle_t *client_handle,
                                    OPEN_OSPF_TRAP_TYPES_t trapType,
                                    bool mode);

#endif /* OPENAPI_OSPF_CONFIG_H_INCLUDED */

