/*********************************************************************
*
*  Copyright 2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_ping.h
*
* @purpose   APIs for issuing ping to a host or IPv4/IPv6 address
*
* @component OpEN
*
* @comments  This code must not include any ICOS header files.
*
* @create    03/29/2018
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_PING_H_INCLUDED
#define OPENAPI_PING_H_INCLUDED

#define OPEN_PING_STRING_SIZE           2048
#define OPEN_PING_ADDRESS_MAX_LEN       256
#define OPEN_PING_DEFAULT_PROBE_SIZE    0
#define OPEN_PING_OUT_INTF_STRING_SIZE  32
#define OPEN_PING_CALLBACK_STR_MAX_LEN  256

typedef enum
{
  OPEN_PING_ADDR_TYPE_IPV4 = 0,  /**< Ping IPv4 Address Type */
  OPEN_PING_ADDR_TYPE_IPV6 = 1,  /**< Ping Ipv6 Address Type */
  OPEN_PING_ADDR_TYPE_NONE = 2   /**< Ping Unspecified Address Type */
}OPEN_PING_ADDR_TYPE_t;

typedef enum
{
 OPEN_PING_SOURCE_INTF_SERVICE_PORT = 0,  /**< Ping IP Address/Hostname/IPv6 Address with source interface as service port */
 OPEN_PING_SOURCE_INTF_NETWORK_PORT = 1   /**< Ping IP Address/Hostname/IPv6 Address with source interface as network port */
}OPEN_PING_SOURCE_INTF_TYPE_t;

typedef enum
{
 OPEN_PING_SOURCE_SERVICE_PORT = 0,  /**< Ping IP Address/Hostname/IPv6 Address with source interface as service port */
 OPEN_PING_SOURCE_NETWORK_PORT = 1,  /**< Ping IP Address/Hostname/IPv6 Address with source interface as network port */
 OPEN_PING_SOURCE_IP_PORT      = 2,  /**< Ping IP Address/Hostname/IPv6 Address with source interface as routing interface */
 OPEN_PING_SOURCE_ADDRESS      = 3,  /**< Ping IP Address/Hostname/IPv6 Address with source address */
 OPEN_PING_SOURCE_NONE         = 4   /**< Ping IP Address/Hostname/IPv6 Address without source */
}OPEN_PING_SOURCE_TYPE_t;

/*********************************************************************
* @purpose  Ping host or IPv4/IPv6 Address 
* 
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    address         @b{(input)}  host or IPv4/IPv6 Address
* @param    output          @b{(output)} ping output
*
* @returns  OPEN_E_NONE      if ping to host or IPv4/IPv6 address is successful.
* @returns  OPEN_E_FAIL      if failed to ping host or IPv4/IPv6 address.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Buffer size for address should be @ref OPEN_PING_ADDRESS_MAX_LEN.
* @notes    Buffer size for output should be @ref OPEN_PING_STRING_SIZE.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiPingAddress(openapiClientHandle_t *client_handle,
                                open_buffdesc *address,
                                open_buffdesc *output);

/*********************************************************************
* @purpose  Ping host or IPv4/IPv6 Address with specific count
* 
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    address         @b{(input)}  host or IPv4/IPv6 Address
* @param    pingCount       @b{(input)}  number of times to ping
* @param    output          @b{(output)} ping output
*
* @returns  OPEN_E_NONE      if ping to host or IPv4/IPv6 address is successful.
* @returns  OPEN_E_FAIL      if failed to ping host or IPv4/IPv6 address.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Buffer size for address should be @ref OPEN_PING_ADDRESS_MAX_LEN.
* @notes    Buffer size for output should be @ref OPEN_PING_STRING_SIZE.
* @notes    Min and Max values for pingCount are 
*           @ref OPEN_PING_MIN_PROBES and @ref OPEN_PING_MAX_PROBES.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiPingAddressWithCount (openapiClientHandle_t *client_handle,
                                          open_buffdesc *address,
                                          uint32_t pingCount,
                                          open_buffdesc *output);

/*********************************************************************
* @purpose  Ping host or IPv4/IPv6 Address with specific interval
* 
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    address         @b{(input)}  host or IPv4/IPv6 Address
* @param    pingInterval    @b{(input)}  interval between probes in seconds
* @param    output          @b{(output)} ping output
*
* @returns  OPEN_E_NONE      if ping to host or IPv4/IPv6 address is successful.
* @returns  OPEN_E_FAIL      if failed to ping host or IPv4/IPv6 address.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Buffer size for address should be @ref OPEN_PING_ADDRESS_MAX_LEN.
* @notes    Buffer size for output should be @ref OPEN_PING_STRING_SIZE.
* @notes    Min and Max values for pingInterval are
*           @ref OPEN_PING_MIN_INTERVAL and @ref OPEN_PING_MAX_INTERVAL.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiPingAddressWithInterval(openapiClientHandle_t *client_handle,
                                            open_buffdesc *address,
                                            uint32_t pingInterval,
                                            open_buffdesc *output);

/*********************************************************************
* @purpose  Ping host or IPv4/IPv6 Address with specific probe size
* 
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    address         @b{(input)}  host or IPv4/IPv6 Address
* @param    pingSize        @b{(input)}  ping probe PDU size
* @param    output          @b{(output)} ping output
*
* @returns  OPEN_E_NONE      if ping to host or IPv4/IPv6 address is successful.
* @returns  OPEN_E_FAIL      if failed to ping host or IPv4/IPv6 address.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Buffer size for address should be @ref OPEN_PING_ADDRESS_MAX_LEN.
* @notes    Buffer size for output should be @ref OPEN_PING_STRING_SIZE.
* @notes    Min and Max values for pingSize are 
*           @ref OPEN_PING_MIN_SIZE and @ref OPEN_PING_MAX_SIZE.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiPingAddressWithPDUSize(openapiClientHandle_t *client_handle,
                                           open_buffdesc *address,
                                           uint32_t pingSize,
                                           open_buffdesc *output);

/*********************************************************************
* @purpose  Ping host or IPv4/IPv6 Address with source interface
* 
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    address         @b{(input)}  host or IPv4/IPv6 Address
* @param    srcIntf         @b{(input)}  network port/service port 
* @param    output          @b{(output)} ping output
*
* @returns  OPEN_E_NONE      if ping to host or IPv4/IPv6 address is successful.
* @returns  OPEN_E_FAIL      if failed to ping host or IPv4/IPv6 address.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL   if service port/network port is not supported for 
*                            source interface 
*
* @notes    Buffer size for address should be @ref OPEN_PING_ADDRESS_MAX_LEN.
* @notes    Buffer size for output should be @ref OPEN_PING_STRING_SIZE.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiPingAddressWithSourceInterface(openapiClientHandle_t *client_handle,
                                                   open_buffdesc *address,
                                                   OPEN_PING_SOURCE_INTF_TYPE_t srcIntf,
                                                   open_buffdesc *output);

/*********************************************************************
* @purpose  Ping host or IPv4/IPv6 Address with all parameters 
*           given at a time
* 
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    address         @b{(input)}  host or IPv4/IPv6 Address
* @param    pingCount       @b{(input)}  number of times to ping
* @param    pingInterval    @b{(input)}  interval between probes in seconds
* @param    pingSize        @b{(input)}  ping probe PDU size
* @param    srcIntf         @b{(input)}  network port/service port 
* @param    output          @b{(output)} ping output
*
* @returns  OPEN_E_NONE      if ping to host or IPv4/IPv6 address is successful.
* @returns  OPEN_E_FAIL      if failed to ping host or IPv4/IPv6 address.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL   if service port/network port is not supported for 
*                            source interface 
*
* @notes    Buffer size for address should be @ref OPEN_PING_ADDRESS_MAX_LEN.
* @notes    Buffer size for output should be @ref OPEN_PING_STRING_SIZE.
* @notes    Min and Max values for pingCount are 
*           @ref OPEN_PING_MIN_PROBES and @ref OPEN_PING_MAX_PROBES.
* @notes    Min and Max values for pingInterval are
*           @ref OPEN_PING_MIN_INTERVAL and @ref OPEN_PING_MAX_INTERVAL.
* @notes    Min and Max values for pingSize are 
*           @ref OPEN_PING_MIN_SIZE and @ref OPEN_PING_MAX_SIZE.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiPingAddressExplicit(openapiClientHandle_t *client_handle,
                                        open_buffdesc *address,
                                        uint32_t pingCount,
                                        uint32_t pingInterval,
                                        uint32_t pingSize,
                                        OPEN_PING_SOURCE_INTF_TYPE_t srcIntf,
                                        open_buffdesc *output);

/*********************************************************************
* @purpose  Ping host or IPv4/IPv6 Address with given source.
*
* @param    client_handle   @b{(input)}  Client handle from
*                                        registration API
* @param    vrfName         @b{(input)}  VRF name
* @param    family          @b{(input)}  Address family
* @param    address         @b{(input)}  host or IPv4/IPv6 Address
* @param    srcType         @b{(input)}  network port/service port/ip port/address
* @param    srcIntf         @b{(input)}  internal interface number incase of ip port
* @param    srcAddr         @b{(input)}  source address incase of source type is address
* @param    output          @b{(output)} ping output
*
* @returns  OPEN_E_NONE      if ping to host or IPv4/IPv6 address is successful.
* @returns  OPEN_E_FAIL      if failed to ping host or IPv4/IPv6 address.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL   if service port/network port/ip port/address
*                             is not supported for source type.
*
* @notes    Buffer size for address should be @ref OPEN_PING_ADDRESS_MAX_LEN.
* @notes    Buffer size for output should be @ref OPEN_PING_STRING_SIZE.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiPingWithSource (openapiClientHandle_t *client_handle,
                                    open_buffdesc *vrfName,
                                    OPEN_AF_t family,
                                    open_buffdesc *address,
                                    OPEN_PING_SOURCE_TYPE_t srcType,
                                    uint32_t srcIntf,
                                    open_buffdesc *srcAddr,
                                    open_buffdesc *output);

/*********************************************************************
* @purpose  Ping host or IPv4/IPv6 Address with all parameters.
* 
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    vrfName         @b{(input)}  VRF name
* @param    family          @b{(input)}  Address family
* @param    address         @b{(input)}  host or IPv4/IPv6 Address
* @param    pingCount       @b{(input)}  number of times to ping
* @param    pingInterval    @b{(input)}  interval between probes in seconds
* @param    pingSize        @b{(input)}  ping probe PDU size
* @param    srcType         @b{(input)}  network port/service port/ip port/address
* @param    intIfNum        @b{(input)}  internal interface number incase of ip port
* @param    srcAddr         @b{(input)}  source address incase of source type is address
* @param    output          @b{(output)} ping output
*
* @returns  OPEN_E_NONE      if ping to host or IPv4/IPv6 address is successful.
* @returns  OPEN_E_FAIL      if failed to ping host or IPv4/IPv6 address.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL   if service port/network port/ip port/address
*                             is not supported for source type.
*
* @notes    Buffer size for address should be @ref OPEN_PING_ADDRESS_MAX_LEN.
* @notes    Buffer size for output should be @ref OPEN_PING_STRING_SIZE.
* @notes    Min and Max values for pingCount are 
*           @ref OPEN_PING_MIN_PROBES and @ref OPEN_PING_MAX_PROBES.
* @notes    Min and Max values for pingInterval are
*           @ref OPEN_PING_MIN_INTERVAL and @ref OPEN_PING_MAX_INTERVAL.
* @notes    Min and Max values for pingSize are 
*           @ref OPEN_PING_MIN_SIZE and @ref OPEN_PING_MAX_SIZE.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiPing (openapiClientHandle_t *client_handle,
                          open_buffdesc *vrfName,
                          OPEN_AF_t family,
                          open_buffdesc *address,
                          uint32_t pingCount,
                          uint32_t pingInterval,
                          uint32_t pingSize,
                          OPEN_PING_SOURCE_TYPE_t srcType,
                          uint32_t intIfNum,
                          open_buffdesc *srcAddr,
                          open_buffdesc *output);

#endif
