/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_routing.h
*
* @purpose   Routing Protocol Process Interface 
*
* @component OPEN
*
* @create    07/10/2012
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_ROUTING_H_INCLUDED
#define OPENAPI_ROUTING_H_INCLUDED

#include <stdio.h>
#include <stddef.h>
#include <sys/un.h>

#include "openapi_common.h"
#include "openapi_mpls.h"

/* Event registration variables */
#define OPEN_RPPI_CLIENT_NAME_MAX_LEN (32)

/** The "address" of the client end of the UNIX datagram
 *  socket opened to receive forwarding table change
 *  notifications. */
#define RPPI_BRC_ADDR          "/tmp/rppi_brc.%04u"   

#define ROUTER_EVENT_CLIENT_ADDR         "/tmp/rppi_rec.%04u"

/* UNIX "address" on client side for policy change notifications */
#define RPPI_POLICY_CLIENT_ADDR          "/tmp/rppi_pc.%04u"   

/* The maximum length of an RPPI event message, in bytes. */
#define RPPI_MSG_SIZE_MAX   256

/* The maximum length of the name of a best route client */
#define OPENR_BRC_NAME_LEN  16

/* The maximum length of the name of a routing policy client */
#define OPENR_POLICY_CLIENT_NAME_LEN 16

#define OPENR_RTR_GBL_MSG_LEN (sizeof(openrRouterEventMsg_t))
#define OPENR_RTR_INTF_MSG_LEN (sizeof(openrRouterEventMsg_t))

/* The maximum length of the name of a routing protocol */
#define OPENR_ROUTING_PROTOCOL_NAME_LEN  10

/* The maximum number of route changes that we can fetch in one call */
#define OPENR_NUM_BEST_ROUTES_PER_GET  50
/* 
*  Use the maximum here no matter how many next hops a route has. This
*  way we can pre-allocate buffers for route gets making them faster.
* The number of next hops are indicated in a field in the routes
*/  
#define OPENR_PLAT_MAX_ECMP_NEXTHOPS 48

/* Number of routes queried to get the policy results */
#define OPENR_NUM_ROUTES_CHECK_POLICY_ACTION 50

/** IP address configuration method.  */
typedef enum
{
  OPEN_INTF_IP_ADDR_METHOD_NONE       = 0, /**< No method. */
  OPEN_INTF_IP_ADDR_METHOD_CONFIG     = 1, /**< Manual configuration. */
  OPEN_INTF_IP_ADDR_METHOD_DHCP       = 2  /**< Leased through DHCP. */
} OPEN_INTF_IP_ADDR_METHOD_t;

/** OPEN uses these 3 enumerators to indicate the addition,
 *  deletion, or change to the best route for a given
 *  destination. */
typedef enum
{
  OPEN_ADD_ROUTE        = 0,
  OPEN_DELETE_ROUTE     = 1,
  OPEN_CHANGE_ROUTE     = 2

} OPEN_ROUTE_EVENT_t;

/** These values must be powers of 2. They are used as indexes to a bit
 * map when registering for routing policy changes. */
typedef enum
{
  OPEN_POLICY_ROUTE_MAP         = 1, /* (1 << 0) */
  OPEN_POLICY_PREFIX_LIST       = 2, /* (1 << 1) */
  OPEN_POLICY_AS_PATH_LIST      = 4, /* (1 << 2) */
  OPEN_POLICY_ACL_ROUTE_FILTER  = 8  /* (1 << 3) */

} OPEN_ROUTING_POLICY_TYPE_t;

/** Type of a policy change notification.  */
typedef enum
{
  OPEN_POLICY_CHANGE = 1, /**< Policy has been added or modified. */
  OPEN_POLICY_DELETE = 2  /**< Policy has been deleted. */
} OPEN_ROUTING_POLICY_CHANGE_t;

/** Values that go in the Message Type field of event messages sent to applications. */
typedef enum
{
  OPENR_KEEPALIVE                 = 1,

  /* Best route change service messages */
  OPENR_BEST_ROUTE_CHANGE         = 10,

  /* router event message types */
  OPENR_RTR_ENABLE                = 101,
  OPENR_RTR_DISABLE               = 102,
  OPENR_RTR_STARTUP_DONE          = 103,
  OPENR_RTR_INTF_CREATE           = 104,
  OPENR_RTR_INTF_DELETE           = 105,
  OPENR_RTR_INTF_ENABLE           = 106,
  OPENR_RTR_INTF_ENABLE_PENDING   = 107,
  OPENR_RTR_INTF_DISABLE          = 108,
  OPENR_RTR_INTF_DISABLE_PENDING  = 109,
  OPENR_RTR_INTF_ADDR_CHANGE      = 110,
  OPENR_RTR_INTF_MTU              = 111,
  OPENR_RTR_INTF_BW_CHANGE        = 112,
  OPENR_RTR_INTF_HOST_MODE        = 113,
  OPENR_RTR_INTF_CFG_CHANGE       = 114,
  OPENR_RTR_VRF_CREATE            = 115,
  OPENR_RTR_VRF_DELETE            = 116,

  OPENR_RTR6_ENABLE                = 201,
  OPENR_RTR6_DISABLE               = 202,
  OPENR_RTR6_STARTUP_DONE          = 203,
  OPENR_RTR6_INTF_CREATE           = 204,
  OPENR_RTR6_INTF_DELETE           = 205,
  OPENR_RTR6_INTF_ENABLE           = 206,
  OPENR_RTR6_INTF_DISABLE          = 207,
  OPENR_RTR6_INTF_ADDR_CHANGE      = 208,
  OPENR_RTR6_INTF_MTU              = 209,
  OPENR_RTR6_INTF_BW_CHANGE        = 210,
  OPENR_RTR6_INTF_HOST_MODE        = 211,

  OPENR_POLICY_CHANGE              = 300

} OPENR_MESSAGE_TYPE_t;

/* NSF applications that may send IPv4 routes to RTO. Values are powers
 * of two used to index nsfRoutesPending. RIP and BGP don't support NSF
 * now but adding values for them anyway. */
typedef enum
{
  OPEN_NSF_PROTOCOL_OSPF = 0x1,
  OPEN_NSF_PROTOCOL_RIP  = 0x2,
  OPEN_NSF_PROTOCOL_BGP  = 0x4,
  OPEN_NSF_PROTOCOL_LAST = 0x8
} OPEN_NSF_PROTOCOL_t;

typedef enum
{
  OPEN_FILTER_ACTION_PERMIT = 0,
  OPEN_FILTER_ACTION_DENY = 1
} OPEN_FILTER_ACTION_t;

/** Message structure for forwarding table change
 *  notifications. */
typedef struct openrBestRouteChangeMsg_s
{
  /** One of OPENR_MESSAGE_TYPE_t */
  uint16_t msgType;

  /** Number of bytes in payload. Includes the Message Type and Message Length fields. */
  uint16_t msgLen;

  /** Client ID given to client at registration time. */
  uint32_t clientId;

  /** VRF Id provided by the client at registration time. */
  uint32_t vrfId;

} openrBestRouteChangeMsg_t;

typedef struct openrRouterEventMsg_s
{
  /* One of OPENR_MESSAGE_TYPE_t */
  uint16_t msgType;

  /* Number of bytes in payload. Includes the Message Type and Message Length fields. */
  uint16_t msgLen;

  uint32_t eventSeqNumber;

  /* For events associated with an interface, this field contains the interface identifier.
  ** Only sent if needed.
  */
  uint32_t ifNum;

  /* VRF Identifier */
  uint32_t vrfId;

  /* Start of the Event specific data memory, which is allocated  based on msgType value. */
  uint8_t  data[0]; 

} openrRouterEventMsg_t;

typedef struct openrPolicyChangeMsg_s
{
  /* One of OPENR_MESSAGE_TYPE_t */
  uint16_t msgType;

  /* Number of bytes in payload. Includes the Message Type and Message Length fields. */
  uint16_t msgLen;

  /* Change sequence number */
  uint32_t seqNo;

  /* The type of routing policy that changed. One of OPEN_ROUTING_POLICY_TYPE_t */
  uint32_t policyType;

  /* The type of change. One of OPEN_ROUTING_POLICY_CHANGE_t. */
  uint32_t changeType;

  /* Name or number of changed policy follows the changeType. */

} openrPolicyChangeMsg_t;

typedef struct openNextHop_s 
{
  /*! IP address of the gateway */
  open_inet_addr_t  nhAddr;      

  /*! Outgoing interface. 
   * This is the unique, 32-bit identifier that RPPI associates with the interface. */
  uint32_t  ifNum;    

  /*! MPLS labels.
   * The first label is the outermost or top label. */
  uint32_t label[OPEN_MAX_MPLS_IMPOSE_LABELS];

} openNextHop_t;


typedef struct openRoute_s
{
  /** Destination prefix and prefix length. Prefix length range is determined
   * by the address family. */
  open_inet_addr_t      destPfx;
  uint8_t              pfxLen;

  /** Type of route. Specifies both the protocol and its internal type, such as
   * "OSPF intra-area route" or "external BGP route." */
  uint8_t         routeType;   

  /** Protocol ID assigned by FASTPATH to the routing protocol that added this route.
   * The protoId uniquely identifies a routing protocol. Different route types may be
   * originated by the same routing protocol
   */  
  uint32_t        protoId;

  /** VRF Identifier
   */  
  uint32_t        vrfId;

  /** if not 0, this is a reject route. Matching packets are dropped and not forwarded by software.
   * ICMP Error Message will be generated for the such rejected packets. */
  uint8_t         rejectRoute;   

  /** if not 0, this is a discard route. Matching packets are dropped and not forwarded by hardware. 
   * No ICMP Error Message will be generated for the such rejected packets. */
  uint8_t         discardRoute;   

  /** Route preference (also called administrative distance). [0 to 255].
   * 0 is reserved for local (connected) routes. A route with a preference of
   * 255 is considered unreachable and is not installed in the forwarding table. */
  uint8_t         pref;

  /** Route metric. */
  uint32_t        metric;        

  /** Number of nexthops in the route */
  uint32_t        numNextHops;

  /** Flag to indicate if this route is leaked from another VR. **/
  uint8_t        leakedRoute;

  /* Flag to indicate if this is an RFC 5549 route */
  uint8_t        rfc5549Route;

  /* Flag to indicate if this route is unnumbered. */
  uint8_t        unnumberedRoute;

  /* Flag to indicate if this route is MPLS. */
  uint8_t        mplsRoute;

  /* Flag to indicate if this route is Syncdb entry. */
  uint8_t        syncdbRoute;
} openRoute_t;

/*
 * This structure defines a route change event. It contains the best route data,
 * the operation being performed and the set of next hops. It also contains a flag
 * that indicates whether more route changes are pending based on which a 
 * subsequent call can be made
 */
typedef struct openRouteChange_s
{
  /* The route itself */
  openRoute_t route;

  /* operation being performed, change type */
  uint8_t chType;

  /* The set of next hops, will assume the maximum value of all supported platforms here */ 
  openNextHop_t   nextHops[OPENR_PLAT_MAX_ECMP_NEXTHOPS];

  /* Is it valid, needed because an ip and mask of 0 are valid and cannot be checked against */
  uint8_t valid;

} openRouteChange_t;

/* Needed by openapi to be a structure as arrays cannot be handled over RPC */
typedef struct openRouteChangeList_s
{
  openRouteChange_t routeChange[OPENR_NUM_BEST_ROUTES_PER_GET];
} openRouteChangeList_t;

/* Bit mask to indicate which match or set terms
 * are set in a route-map statement. Note that not all of these are supported. */
typedef enum 
{
  OPENR_RMAP_MATCH_AS_PATH               = 0x1,
  OPENR_RMAP_MATCH_COMMUNITY_LIST        = 0x2,
  OPENR_RMAP_MATCH_INTERFACE             = 0x4,
  OPENR_RMAP_MATCH_IP_ADDRESS            = 0x8,
  OPENR_RMAP_MATCH_IP_NEXT_HOP           = 0x10,
  OPENR_RMAP_MATCH_IP_ROUTE_SOURCE       = 0x20,
  OPENR_RMAP_MATCH_LENGTH                = 0x40,
  OPENR_RMAP_MATCH_METRIC                = 0x80,
  OPENR_RMAP_MATCH_METRIC_TYPE           = 0x100,
  OPENR_RMAP_MATCH_TAG                   = 0x200,
#ifdef L7_QOS_PACKAGE
  OPENR_RMAP_MATCH_MAC_ACL               = 0x400, 
#endif /* L7_QOS_PACKAGE */
  OPENR_RMAP_MATCH_EXTCOMMUNITY_LIST     = 0x800,
  OPENR_RMAP_MATCH_IPV6_ADDRESS          = 0x1000,

  OPENR_RMAP_SET_AS_PATH                 = 0x1,
  OPENR_RMAP_SET_AUTOMATIC_TAG           = 0x2,
  OPENR_RMAP_SET_COMMUNITY               = 0x4,
  OPENR_RMAP_SET_COMM_LIST_DELETE        = 0x8,
  OPENR_RMAP_SET_DEFAULT_INTF            = 0x10,
  OPENR_RMAP_SET_INTF                    = 0x20,
  OPENR_RMAP_SET_IP_DEFAULT_NEXT_HOP     = 0x40,
  OPENR_RMAP_SET_IPV6_NEXT_HOP           = 0x80,
  OPENR_RMAP_SET_LEVEL                   = 0x100,
  OPENR_RMAP_SET_LOCAL_PREF              = 0x200,
  OPENR_RMAP_SET_METRIC                  = 0x400,
  OPENR_RMAP_SET_METRIC_TYPE             = 0x800,
  OPENR_RMAP_SET_NEXT_HOP                = 0x1000,
  OPENR_RMAP_SET_TAG                     = 0x2000,
  OPENR_RMAP_SET_IP_TOS                  = 0x4000,
  OPENR_RMAP_SET_IP_PREC                 = 0x8000,
  OPENR_RMAP_SET_RT_EXTCOMMUNITY         = 0x10000,
  OPENR_RMAP_SET_SOO_EXTCOMMUNITY        = 0x20000,
  OPENR_RMAP_SET_EXTCOMM_LIST_DELETE     = 0x40000,
  OPENR_RMAP_SET_IPV6_DEFAULT_NEXT_HOP   = 0x80000,
  OPENR_RMAP_SET_IPV6_PREC               = 0x100000,
  OPENR_RMAP_SET_IP_UNCHANGED_NEXT_HOP   = 0x200000,
} OPENR_RMAP_PARAMS_MASK_t;

/* Route map match parameters. */
typedef struct openRmMatchParams_s
{
  /* Bit-map to indicate what match params are set */
  OPENR_RMAP_PARAMS_MASK_t  matchParamsMask;

  /* To match any prefix, set the prefix and prefixLen to 0. */
  open_inet_addr_t      prefix;
  uint32_t              prefixLen;

  /* To match on the metric, metric-type (ospf type-1/type-2)
   * and route tag */
  uint32_t              metric;
  uint32_t              metricType;
  uint32_t              tag;
} openRmMatchParams_t;

typedef struct openRmSetParams_s
{
  /* Bit-map to indicate what set params are set */
  OPENR_RMAP_PARAMS_MASK_t  setParamsMask;

  /* metric value, if set */
  uint32_t     metric;

  /* metric type value, if set */
  uint32_t     metricType;

  /* tag value, if set */
  uint32_t     tag;
} openRmSetParams_t;

/* Needed by openapi to get the results of the route-map policies for
 * a given set of routes with match params */
typedef struct openRoutePolicyResults_s
{
  openRoute_t          route[OPENR_NUM_ROUTES_CHECK_POLICY_ACTION];
  openRmMatchParams_t  matchParams[OPENR_NUM_ROUTES_CHECK_POLICY_ACTION];
  openRmSetParams_t    setParams[OPENR_NUM_ROUTES_CHECK_POLICY_ACTION];
} openRoutePolicyResults_t;

/**< Message structure for fetching IPv4 statistics */
typedef struct openrIpv4Stats_s
{
  uint64_t ipInReceives;
  uint64_t ipInHdrErrors;
  uint64_t ipInAddrErrors;
  uint64_t ipForwDatagrams;
  uint64_t ipInUnknownProtos;
  uint64_t ipInDiscards;
  uint64_t ipInDelivers;
  uint64_t ipOutRequests;
  uint64_t ipOutDiscards;
  uint64_t ipNoRoutes;
  uint64_t ipReasmTimeout;
  uint64_t ipReasmReqds;
  uint64_t ipReasmOKs;
  uint64_t ipReasmFails;
  uint64_t ipFragOKs;
  uint64_t ipFragFails;
  uint64_t ipFragCreates;
  uint64_t ipRoutingDiscards;
  uint64_t icmpInMsgs;
  uint64_t icmpInErrors;
  uint64_t icmpInDestUnreachs;
  uint64_t icmpInTimeExcds;
  uint64_t icmpInParmProbs;
  uint64_t icmpInSrcQuenchs;
  uint64_t icmpInRedirects;
  uint64_t icmpInEchos;
  uint64_t icmpInEchoReps;
  uint64_t icmpInTimestamps;
  uint64_t icmpInTimestampReps;
  uint64_t icmpInAddrMasks;
  uint64_t icmpInAddrMaskReps;
  uint64_t icmpOutMsgs;
  uint64_t icmpOutErrors;
  uint64_t icmpOutDestUnreachs;
  uint64_t icmpOutTimeExcds;
  uint64_t icmpOutParmProbs;
  uint64_t icmpOutSrcQuenchs;
  uint64_t icmpOutRedirects;
  uint64_t icmpOutEchos;
  uint64_t icmpOutEchoReps;
  uint64_t icmpOutTimestamps;
  uint64_t icmpOutTimestampReps;
  uint64_t icmpOutAddrMasks;
  uint64_t icmpOutAddrMaskReps;

} openrIpv4Stats_t;

/* BFD Admin modes */
typedef enum
{
  OPEN_BFD_DISABLE = 0,
  OPEN_BFD_ENABLE = 1
} OPEN_BFD_ADMIN_MODE_t;

/* BFD endpoint parameters used to initiate a session */
typedef struct openBfdEndpoint_s {
  uint32_t           compId;               /* Id of component creating the session  */
  uint32_t           vrfId;                /* VRID associated with this BFD session */
  uint32_t           intIfNum;             /* Interface associated with this session */
  open_inet_addr_t   srcIpAddr;            /* Source IPv4 or Ipv6 address */
  open_inet_addr_t   dstIpAddr;            /* Destination IPv4 or Ipv6 address */
  open_inet_addr_t   nextHopIpAddr;        /* NextHop IPv4 or Ipv6 address for Destination */
  OPEN_MPLS_LABELS_t mpls_label;           /* MPLS Label Stack */
  uint32_t           type;                 /* Type of BFD encapsulation. */

  /* The BFD application can take the echo mode and BFD session timers from the
  ** interface and global config or from the session setting. In order to use the session 
  ** settings set the override_config to 1. 
  ** The timer values are in milliseconds. 
  */ 
  uint32_t           override_config;
  uint32_t           bfdMinTx; 
  uint32_t           bfdMinRx;
  uint32_t           bfdMultiplier;
  uint32_t           bfdEchoMode;
  uint32_t           bfdSlowTimer;
} openBfdEndpoint_t;


/* Maximum protocols allowed to Register with BFD */
#define OPEN_BFD_MAX_REG_PROT    2 /* BGP and OSPF */

/* BFD session information */
#define OPEN_BFD_SESSION_ID_INVALID  (~0)

/* BFD session states */
typedef enum
{
  OPEN_BFD_SESSION_STATE_ADMIN_DOWN = 0,
  OPEN_BFD_SESSION_STATE_DOWN = 1,
  OPEN_BFD_SESSION_STATE_INIT = 2,
  OPEN_BFD_SESSION_STATE_UP = 3,
} OPEN_BFD_SESSION_STATE_t;

/* BFD Diagnostic codes */
typedef enum {
  OPEN_BFD_DIAG_CODE_NONE,
  OPEN_BFD_DIAG_CODE_CTRL_DETECT_TIME_EXPIRED,
  OPEN_BFD_DIAG_CODE_ECHO_FAILED,
  OPEN_BFD_DIAG_CODE_NEIGHBOR_SIGNALED_SESSION_DOWN,
  OPEN_BFD_DIAG_CODE_FORWARDING_PLANE_RESET,
  OPEN_BFD_DIAG_CODE_PATH_DOWN,
  OPEN_BFD_DIAG_CODE_CONCATENATED_PATH_DOWN,
  OPEN_BFD_DIAG_CODE_ADMIN_DOWN,
  OPEN_BFD_DIAG_CODE_REVERSE_CONCATENATED_PATH_DOWN,
  OPEN_BFD_DIAG_CODE_MIS_CONNECTIVITY_DEFECT
} OPEN_BFD_DIAG_CODE_t;

typedef enum {
  OPEN_BFD_TUNNEL_TYPE_RAW,                /* RAW UDP */
  OPEN_BFD_TUNNEL_TYPE_UDP,                /* BFD over IPv4/v6. */
  OPEN_BFD_TUNNEL_TYPE_COUNT
} openBfdTunnelType_t;

#define SYNCDB_OPENAPI_BFD "OpEN API BFD"
#define SYNCDB_OPENAPI_BFD_VERSION 1

/** The key into the BFD database. The Syncdb evaluates this key
  using the memcmp() function, so the BFD entries don't appear 
  in any particular order on the "GetNext" operation. 
*/ 
typedef struct openBfdSessionKey_s {
  uint32_t            intIfNum;           /* Interface associated with this session */
  uint32_t            vrfId;              /* VRID associated with this BFD session */
  open_inet_addr_t    srcIpAddr;          /* Source IPv4 or Ipv6 address */
  open_inet_addr_t    dstIpAddr;          /* Destination IPv4 or Ipv6 address */
  OPEN_MPLS_LABELS_t  mpls_label;         /* MPLS Label Stack */
} openBfdSessionKey_t;

/** The BFD Session data.
*/
typedef struct openBfdSessionInfo_s {
  openBfdSessionKey_t key;                /**< Unique BFD session identifier */
  uint32_t            session_id;         /**< BFD Session ID */
  uint8_t               version;          /**< BFD version */
  OPEN_BFD_SESSION_STATE_t state;         /**< Local session state. */
  uint8_t              flags;             /**< BFD flags */
  OPEN_BFD_DIAG_CODE_t diag;              /**< Local diagnostic code. */
  uint32_t            authType;           /**< Authentication type - not supported. */
  uint32_t            localDiscr;         /**< Local discriminator */
  uint32_t            remoteDiscr;        /**< Remote discriminator */
  uint32_t            localMinTx;         /**< Desired local min tx interval. */
  uint32_t            localMinRx;         /**< Required local rx interval. */
  uint8_t             localDetectMult;    /**< Local detection interval multiplier. */
  uint32_t            remoteMinTx;        /**< Desired remote min tx interval. */
  uint32_t            remoteMinRx;        /**< Required remote rx interval. */
  uint8_t             remoteDetectMult;   /**< Remote detection interval multiplier. */
  uint32_t            actualTx;           /**< Negotiated tx value. */
  uint32_t            echoEnable;         /**< Echo mode (0-Disabled, 1-Enabled) */
  uint32_t            actualTxEcho;       /**< Negotiated tx echo value. */
  uint32_t            id[OPEN_BFD_MAX_REG_PROT]; /**< Id of component creating the session  */
  open_inet_addr_t    nextHopIpAddr;      /**< NextHop IPv4 or Ipv6 address for Destination */
  openBfdTunnelType_t type;               /**< Type of BFD encapsulation RAW, UDP. */
  uint32_t            upTime;             /**< Session UP time */
} openBfdSessionInfo_t;

/* BFD session statistics */
typedef struct openBfdSessionStats_s {
  uint32_t inPkts;          /* BFD packets received. */
  uint32_t outPkts;         /* BFD packets transmitted. */
  uint32_t dropPkts;        /* Total BFD packets drop. */
  uint32_t echoInPkts;      /* BFD echo packets received. */
  uint32_t echoOutPkts;     /* BFD echo packets transmitted. */
} openBfdSessionStats_t;


/*********************************************************************
* @purpose  Register a new routing protocol. If a protocol already exists with the
*           given protocol name, the existing protocol ID is returned. 
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    vrfId            @b{(input)}  VRF Identifier
* @param    addressFamily    @b{(input)}  Address family (IPv4 or IPv6)
* @param    protoName        @b{(input)}  Name of the routing protocol. This name is
*                                         used in the UI to refer to this protocol.
*                                         (e.g., "redistribute <protoName>).
*                                         openapiRouteProtoNameLenMax() gives the max length.
* 
* @param    protoCode        @b{(input)}  One character routing protocol code. Used in
*                                         "show ip route" to identify the routing protocol
*                                         that provided a route.
* 
* @param    protoId          @b{(output)}  Unique identifier for the routing protocol.
*
* @returns  OPEN_E_NONE        If a protocol is added and a valid protocol ID is returned.
* @returns  OPEN_E_EXISTS      If the protocol name matches an existing protocol
*                             type. In this case, protoId is set to the value for
*                             the existing entry.
* @returns  OPEN_E_UNAVAIL     If the address family is not supported for this operation.
* @returns  OPEN_E_FULL        If the maximum number of protocols are already registered.
* @returns  OPEN_E_PARAM       If the output parameter is NULL.
* @returns  OPEN_E_INTERNAL    If any of the input names are too long or otherwise invalid.
* @returns  OPEN_E_FAIL        Other failure.
* @returns  OPEN_E_NOT_FOUND   If the VRF instance does not exist.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRoutingProtocolRegister(openapiClientHandle_t *client_handle,
                                            uint32_t vrfId,
                                            OPEN_AF_t  addressFamily, 
                                            open_buffdesc  *protoName,
                                            open_buffdesc  *protoCode,
                                            uint32_t  *protoId);

/*********************************************************************
* @purpose  Register a new route type. If a route type already exists with the
*           given route type name, the existing route type ID is returned. 
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    addressFamily    @b{(input)}  Address family (IPv4 or IPv6)
* @param    protoId          @b{(input)}  ID of routing protocol already registered with RPPI.
* 
* @param    routeTypeName    @b{(input)}  Name of an individual route type within a protocol.
*                                         For example, "OSPF External Type 1".
*                                         openapiRouteTypeNameLenMax() gives the max length.
* 
* @param    routeTypeCode    @b{(input)}  One or two character code that identifies a route
*                                         type in the UI. Used in "show ip route" to identify
*                                         the route type for each route. If a routing protocol
*                                         has a single route type, this parameter can be set to
*                                         the empty string.
* 
* @param    routeType        @b{(output)}  Unique identifier for a route type. The routeType field
*                                          in openRoute_t must be set to this value
*                                          when adding, modifying, or removing routes. 
*
* @returns  OPEN_E_NONE        If a new route type is added and a valid route type is returned.
* @returns  OPEN_E_EXISTS      If the route type name matches an existing route
*                             type. In this case, routeType is set to the value for
*                             the existing entry.
* @returns  OPEN_E_UNAVAIL     If the address family is not supported for this operation.
* @returns  OPEN_E_FULL        If the maximum number of route types are already registered.
* @returns  OPEN_E_PARAM       If the output parameters is NULL.
* @returns  OPEN_E_INTERNAL    If any of the input names are too long or otherwise invalid.
* @returns  OPEN_E_FAIL        Other failure.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRouteTypeRegister(openapiClientHandle_t *client_handle,
                                      OPEN_AF_t  addressFamily,
                                      uint32_t  protoId, 
                                      open_buffdesc  *routeTypeName,
                                      open_buffdesc  *routeTypeCode,
                                      uint32_t  *routeType);

/*********************************************************************
* @purpose  Deregister a routing protocol. 
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    vrfId            @b{(input)}  VRF Identifier
* @param    addressFamily    @b{(input)}  Address family (IPv4 or IPv6)
* @param    protoId          @b{(input)}  Unique protocol identifier that RPPI provided at
*                                         registration time.
*
* @returns  OPEN_E_NONE        If a new route type is removed and a valid route type is returned.
* @returns  OPEN_E_NOT_FOUND   If the route type does not match a registered route type.
* @returns  OPEN_E_FAIL        Other failure.
*
* @notes    Only route types that were registered through the register API above can
*           be deregistered. 
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRoutingProtocolDeregister(openapiClientHandle_t *client_handle,
                                              uint32_t vrfId,
                                              uint32_t addressFamily, 
                                              uint32_t protoId);

/*********************************************************************
* @purpose  Deregister a route type. 
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    addressFamily    @b{(input)}  Address family (IPv4 or IPv6)
* @param    routeType        @b{(input)}  Unique identifier that RPPI provided at
*                                         registration time.
*
* @returns  OPEN_E_NONE        If a route type is removed and a valid route type is returned.
* @returns  OPEN_E_NOT_FOUND   If the route type does not match a registered route type.
* @returns  OPEN_E_FAIL        Other failure.
*
* @notes    Only route types that were registered through the register API above can
*           be deregistered. If the route type is the last route type for its protocol,
*           the protocol itself is also removed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRouteTypeDeregister(openapiClientHandle_t *client_handle,
                                        uint32_t addressFamily, 
                                        uint32_t routeType);

/*********************************************************************
* @purpose  Register with RTO to save External routes to SyncDB
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    addressFamily    @b{(input)}  Address family (IPv4 or IPv6)
* @param    vrfId            @b{(input)}  Virtual router identifier
* @param    protoId          @b{(input)}  Unique protocol identifier that RPPI provided at
*                                         registration time.
*
* @returns  OPEN_E_NONE      If the registration is successful
* @returns  OPEN_E_ERROR     Other failure.
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t
openapiExternalRoutesSaveRegister (openapiClientHandle_t *client_handle,
                                   OPEN_AF_t addressFamily,
                                   uint32_t vrfId,
                                   uint32_t protoId);

/*********************************************************************
* @purpose  De-Register with RTO to save External routes to SyncDB
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    addressFamily    @b{(input)}  Address family (IPv4 or IPv6)
* @param    vrfId            @b{(input)}  Virtual router identifier
* @param    protoId          @b{(input)}  Unique protocol identifier that RPPI provided at
*                                         registration time.
*
* @returns  OPEN_E_NONE      If the de-registration is successful
* @returns  OPEN_E_ERROR     Other failure.
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t
openapiExternalRoutesSaveDeRegister (openapiClientHandle_t *client_handle,
                                     OPEN_AF_t addressFamily,
                                     uint32_t vrfId,
                                     uint32_t protoId);

/*********************************************************************
* @purpose  For a given route type, get the protocol and route type names. 
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    routeType      @b{(input)}  Route type ID
* @param    protoName      @b{(output)} Name of the protocol that uses this route type.
*                                       Up to OPENR_ROUTE_PROTO_NAME_LEN.
* @param    routeTypeName  @b{(output)} Descriptive name of the route type
*                                       Up to OPENR_ROUTE_TYPE_NAME_LEN.
*
* @returns  OPEN_E_NONE        If the route type is recognized and the names are returned
* @returns  OPEN_E_PARAM       If a parameter is invalid
* @returns  OPEN_E_NOT_FOUND   If route type is not recognized
* @returns  OPEN_E_FAIL        Otherwise
*
* @notes    On input, the size field in the open_buffdesc for each output variable
*           must be set to the buffer length allocated.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRouteTypeInfoGet(openapiClientHandle_t *client_handle,
                                     uint32_t routeType,
                                     open_buffdesc *protoName,
                                     open_buffdesc *routeTypeName);

/*********************************************************************
* @purpose  Get the maximum length of a protocol name. 
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    protoNameLen   @b{(output)}  The maximum length of a protocol name.
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_PARAM if protoNameLen is NULL
*
* @notes    The name length does not include space for a NULL terminator.
* 
*           If protoName has more than this number of characters in a call to
*           openapiRoutingProtocolRegister(), the registration fails.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRouteProtoNameLenMax(openapiClientHandle_t *client_handle,
                                         uint32_t *protoNameLen);

/*********************************************************************
* @purpose  Get the maximum length of a protocol name. 
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    routeTypeNameLen   @b{(output)}  The maximum length of a route type name.
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_PARAM if routeTypeNameLen is NULL
*
* @notes    The name length does not include space for a NULL terminator.
* 
*           If routeTypeName has more than this number of characters in a call to
*           openapiRoutingProtocolRegister(), the registration fails.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRouteTypeNameLenMax(openapiClientHandle_t *client_handle,
                                        uint32_t *routeTypeNameLen);

/*********************************************************************
* @purpose  Add a route to the common routing table.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    routeEntry    @b{(input)}  Route to be added.
* @param    nextHopListBuff   @b{(input)}  List of nextHops associated with route.
*
* @returns  OPEN_E_NONE        If the route was added successfully.
* @returns  OPEN_E_NOT_FOUND   If none of the route's next hops is in a local subnet.
* @returns  OPEN_E_FULL        If the route was not added because the routing table is full.
* @returns  OPEN_E_EXISTS      If a route with the same route type and destination is already
*                              in the routing table.
* @returns  OPEN_E_FAIL        Any other failure.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRouteAdd(openapiClientHandle_t *client_handle,
                             openRoute_t *routeEntry, 
                             open_buffdesc *nextHopListBuff);

/*********************************************************************
* @purpose  Modify an existing route to the common routing table.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    routeEntry    @b{(input)}  Route to be modified.
* @param    nextHopListBuff   @b{(input)}  List of nextHops associated with route.
*
* @returns  OPEN_E_NONE        If the route was successfully modified.
* @returns  OPEN_E_NOT_FOUND   If the route table does not have a route to this destination
*                             with the given route type.
* @returns  OPEN_E_FAIL        Any other failure.
*
* @notes    This API can only be called when replacing one route with
*           another route with the same route type. It cannot be used to
*           replace a route with a different route type, even if the two
*           route types are for the same routing protocol.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRouteMod(openapiClientHandle_t *client_handle,
                             openRoute_t *routeEntry, 
                             open_buffdesc *nextHopListBuff);

/*********************************************************************
* @purpose  Delete a route from the common routing table.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    routeEntry    @b{(input)}  Route to be deleted.
*
* @returns  OPEN_E_NONE        If the route was successfully deleted.
* @returns  OPEN_E_NOT_FOUND   If the route table does not have a route to this destination
*                             with the given route type.
* @returns  OPEN_E_FAIL        Any other failure.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRouteDel(openapiClientHandle_t *client_handle,
                             openRoute_t *routeEntry);


/*********************************************************************
* @purpose  Return next best route in the routing table. 
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    family            @b{(input)}         Address family (IPv4 or IPv6)
* @param    routeEntry        @b{(input/output)} Next best route in the routing table.
* @param    nextHopListBuff   @b{(output)}       List of nextHops associated with route.
*
* @returns  OPEN_E_NONE        If a next route is returned in routeEntry.
* @returns  OPEN_E_NOT_FOUND   If there are no more best routes.
* @returns  OPEN_E_PARAM       If an input parameter is invalid.
* @returns  OPEN_E_FAIL        Other failure.
*
* @notes    In order to look up the first route all fields of the routeEntry 
*           structure must be set to zero. In order to look up subsequent 
*           routes use the previously reported route as input.
*
*           The route returned includes all active next hops.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBestRouteNextGet(openapiClientHandle_t *client_handle,
                                     OPEN_AF_t family, 
                                     openRoute_t *routeEntry, 
                                     open_buffdesc *nextHopListBuff);

/*********************************************************************
* @purpose  Return next best route in the specified VRF routing table. 
*
* @param    client_handle     @b{(input)} client handle from registration API
* @param    family            @b{(input)} Address family (IPv4 or IPv6)
* @param    vrfId             @b{(input)} VRF Identifier
* @param    openRoute         @b{(input/output)} Next best route in the routing table.
* @param    nextHopListBuff   @b{(output)}       List of nextHops associated with route.
*
* @returns  OPEN_E_NONE        If a next route is returned in routeEntry.
* @returns  OPEN_E_NOT_FOUND   If there are no more best routes.
* @returns  OPEN_E_PARAM       If an input parameter is invalid.
* @returns  OPEN_E_FAIL        Other failure.
*
* @notes    In order to look up the first route all fields of the routeEntry 
*           structure must be set to zero. In order to look up subsequent 
*           routes use the previously reported route as input.
*
*           The route returned includes all active next hops.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiBestRouteNextGetVr(openapiClientHandle_t *client_handle,
                                       OPEN_AF_t family,
                                       uint32_t vrfId,
                                       openRoute_t *openRoute,
                                       open_buffdesc *nextHopListBuff);

/*********************************************************************
* @purpose  Return next route in the routing table. 
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    family            @b{(input)}        Address family (IPv4 or IPv6)
* @param    route             @b{(input/output)} Next route in the routing table.
* @param    nextHopListBuff   @b{(output)}  List of nextHops associated with route.
*
* @returns  OPEN_E_NONE        If a next route is returned in routeEntry.
* @returns  OPEN_E_NOT_FOUND   If there are no more best routes.
* @returns  OPEN_E_PARAM       If an input parameter is invalid.
* @returns  OPEN_E_FAIL        Other failure.
*
* @notes    In order to look up the first route all fields of the routeEntry 
*           structure must be set to zero. In order to look up subsequent 
*           routes use the previously reported route as input.
*
*           The route returned includes all active next hops.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRouteNextGet(openapiClientHandle_t *client_handle,
                                 OPEN_AF_t family, 
                                 openRoute_t *route, 
                                 open_buffdesc *nextHopListBuff);

/*********************************************************************
* @purpose  Find the longest prefix match for a given destination IP address. 
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    destAddr      @b{(input)}  destination IP address
* @param    route         @b{(output)} best route to destination
* @param    nextHopListBuff   @b{(output)}  List of nextHops associated with route.
*
* @returns  OPEN_E_NONE        If a best route is found
* @returns  OPEN_E_NOT_FOUND   If there is no route to the destination
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBestRouteLookup(openapiClientHandle_t *client_handle,
                                    open_inet_addr_t destAddr, 
                                    openRoute_t *route, 
                                    open_buffdesc *nextHopListBuff);

/*********************************************************************
* @purpose  Find the best route with a specific prefix.  
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    destPrefix   @b{(input)}  destination IP prefix
* @param    pfxLen       @b{(input)}  destination prefix length
* @param    route        @b{(output)} best route to this prefix
* @param    nextHopListBuff  @b{(output)}  List of nextHops associated with route.
* 
* @returns  OPEN_E_NONE        If a best route is found
* @returns  OPEN_E_NOT_FOUND   If there is no route to the destination
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiPrefixFind(openapiClientHandle_t *client_handle,
                               open_inet_addr_t destPrefix, 
                               uint32_t pfxLen, 
                               openRoute_t *route, 
                               open_buffdesc *nextHopListBuff);
/*********************************************************************
* @purpose  Find the best route with a specific prefix.  
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    vrfId            @b{(input)}  VRF Identifier
* @param    destPrefix       @b{(input)}  destination IP prefix
* @param    pfxLen           @b{(input)}  destination prefix length
* @param    route            @b{(output)} best route to this prefix
* @param    nextHopListBuff  @b{(output)} List of nextHops associated with route.
* 
* @returns  OPEN_E_NONE        If a best route is found
* @returns  OPEN_E_NOT_FOUND   If there is no route to the destination
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiPrefixFindVr(openapiClientHandle_t *client_handle,
                                 uint32_t vrfId,
                                 open_inet_addr_t destPrefix, 
                                 uint32_t pfxLen, 
                                 openRoute_t *route, 
                                 open_buffdesc *nextHopListBuff);

/*********************************************************************
* @purpose  Resolve ARP for the given IP Address
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    vrfId            @b{(input)}  VRF Identifier
* @param    nh_addr          @b{(input)}  IP Address
* @param    mac              @b{(output)} Mac address
* @param    cbParm1          @b{(input)}  place holder parameter
* @param    cbParm2          @b{(input)}  place holder parameter.
*                                         stackIfIndex in case of unnumbered.
* @param    cbParm3          @b{(input)}  place holder parameter
*
* @returns  OPEN_E_NONE        If a ARP is resolved.
* @returns  OPEN_E_UNAVAIL     If failed to get MAC address for an interface.
* @returns  OPEN_E_PARAM       If parameters passed are not valid.
* @returns  OPEN_E_NOT_FOUND   If route lookup for address failed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiArpResolve(openapiClientHandle_t *client_handle,
                               uint32_t vrfId,
                               open_inet_addr_t *nh_addr,
                               open_buffdesc *mac,
                               uint32_t cbParm1,
                               uint32_t cbParm2, 
                               uint32_t cbParm3);

/*********************************************************************
* @purpose  Register to be notified when there are changes to one or more
*           routes used for IP forwarding. 
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    family      @b{(input)}  OPEN_AF_INET to be notified of changes to IPv4 routes
*                                    OPEN_AF_INET6 to be notified of changes to IPv6 routes
* @param    pid         @b{(input)}  Linux process ID of the process registering
* @param    vrfId       @b{(input)}  VRF Identifier
* @param    clientName  @b{(input)}  Name of the client. Must be unique among all clients
*                                    registering for best route change notification. Up to
*                                    OPENR_BRC_NAME_LEN characters long.
* @param    clientId    @b{(output)} Unique integer ID assigned to the client for this
*                                    registration. Used to identify the client when retrieving
*                                    changes.
*
* @returns  OPEN_E_NONE        If the registration was successful
*           OPEN_E_UNAVAIL     If RPPI is not ready to accept registrations yet. This may occur when
*                             switchdrvr is first starting and has not yet reached a state where it
*                             can accept best route client registrations. The caller should
*                             try again later.
* @returns  OPEN_E_FULL        If the maximum number of best route clients are already registered.
* @returns  OPEN_E_FAIL        Any other failure
*
* @notes    RPC API.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiBestRouteEventRegister(openapiClientHandle_t *client_handle,
                                           OPEN_AF_t family,
                                           uint32_t pid,
                                           uint32_t vrfId,
                                           open_buffdesc *clientName, 
                                           uint32_t   *clientId);

/*********************************************************************
* @purpose  Deregister for notifications when there are changes to one or more
*           routes used for IP forwarding. 
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    clientId    @b{(input)} Unique integer ID assigned to the client for this
*                                   registration. Used to identify the client when retrieving
*                                   changes.
*
* @returns  OPEN_E_NONE        If the registration was successful
* @returns  OPEN_E_UNAVAIL     If RPPI is not ready to accept registrations yet. This may occur when
*                             switchdrvr is first starting and has not yet reached a state where it
*                             can accept best route client registrations. The caller should
*                             try again later.
* @returns  OPEN_E_NOT_FOUND   If the client ID does not match a registered client.
* @returns  OPEN_E_FAIL        Any other failure
*
* @notes    RPC API.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBestRouteEventDeregister(openapiClientHandle_t *client_handle,
                                             uint32_t clientId);

/*********************************************************************
* @purpose  Get the next best route change pending for a specific client.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    clientId    @b{(input)} Unique integer ID assigned to the client for this
*                                   registration. Used to identify the client when retrieving
*                                   changes.
* @param    chType      @b{(output)}  Type of change (add, modify, delete)
* @param    route       @b{(output)}  The changed route. On a delete route event, the only
*                                     valid fields are the prefix and prefix length.
* @param    nextHopListBuff @b{(output)}  List of nextHops associated with route.
*
* @returns  OPEN_E_NONE        If a route change is returned
* @returns  OPEN_E_PARAM       If any output parameter is NULL
* @returns  OPEN_E_EMPTY       If no more route changes are pending
* @returns  OPEN_E_NOT_FOUND   No client is registered with the given client ID
* @returns  OPEN_E_FAIL        Any other failure
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBestRouteChangeNextGet(openapiClientHandle_t *client_handle,
                                           uint32_t clientId, 
                                           OPEN_ROUTE_EVENT_t *chType, 
                                           openRoute_t *route, 
                                           open_buffdesc *nextHopListBuff);

/*********************************************************************
* @purpose  Get the next best route change pending for a specific client.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    clientId    @b{(input)} Unique integer ID assigned to the client for this
*                                   registration. Used to identify the client when retrieving
*                                   changes.
* @param    numChanges  @b{(output)}  The number of changes to retrieve
* @param    routeListBuff @b{(output)}  The list that holds the set of best routes.
*
* @returns  OPEN_E_NONE        If a route change is returned
* @returns  OPEN_E_PARAM       If any output parameter is NULL
* @returns  OPEN_E_EMPTY       If no more route changes are pending
* @returns  OPEN_E_NOT_FOUND   No client is registered with the given client ID
* @returns  OPEN_E_FAIL        Any other failure
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBestRouteChangeNextGetMult(openapiClientHandle_t *client_handle,
                                              uint32_t clientId, 
                                              uint32_t numChanges, 
                                              open_buffdesc *routeListBuff);

/*********************************************************************
* @purpose  Get the next set of best-routes on an interface for a specific client.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    clientId    @b{(input)} Unique integer ID assigned to the client for this
*                                   registration. Used to identify the client when retrieving
*                                   changes.
* @param    intIfNum    @b{(input)}   interface number
* @param    numChanges  @b{(output)}  The number of changes to retrieve
* @param    routeListBuff @b{(output)}  The list that holds the set of best routes.
*
* @returns  OPEN_E_NONE        If a route change is returned
* @returns  OPEN_E_PARAM       If any output parameter is NULL
* @returns  OPEN_E_EMPTY       If no more route changes are pending
* @returns  OPEN_E_NOT_FOUND   No client is registered with the given client ID
* @returns  OPEN_E_FAIL        Any other failure
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIntfBestRoutesNextGet(openapiClientHandle_t *client_handle,
                                          uint32_t clientId, uint32_t intIfNum, 
                                          uint32_t numChanges, 
                                          open_buffdesc *routeListBuff);

/*********************************************************************
* @purpose  Get the maximum number of next hops in an ECMP route. 
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    maxNextHops   @b{(output)} Maximum number of next hops
*
* @returns  OPEN_E_NONE        If the maximum is successfully reported
* @returns  OPEN_E_PARAM       If any output parameter is NULL
* @returns  OPEN_E_FAIL        Any other failure
*
* @notes    If an application attempts to add a route with more next hops than
*           this limit, OPEN truncates the number of next hops at this limit.
*           OPEN will not report a route with more next hops than the maximum
*           reported by this API.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiMaxNextHopsGet(openapiClientHandle_t *client_handle,
                                   uint32_t *maxNextHops);

/*********************************************************************
* @purpose  Register for routing events.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    family      @b{(input)}  IPv4 or IPv6 events
* @param    pid         @b{(input)}  Linux process ID
* @param    vrfId       @b{(input)}  VRF Identifier
* @param    clientName  @b{(input)}  Name of the client. If a client is already
*                                    registered with the same name, that registration
*                                    is deleted and replaced. Up to
*                                    (OPEN_CLIENT_NAME_MAX_LEN - 1) characters.
* @param    clientId    @b{(output)} Unique client ID RPPI assigns.
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on invalid input parameters
* @returns  OPEN_E_FULL if the maximum number of router event clients are already registered
* @returns  OPEN_E_INTERNAL for an internal error
* @returns  OPEN_E_FAIL on other failure
*
* @notes     
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRouterEventRegister(openapiClientHandle_t *client_handle,
                                        OPEN_AF_t family, 
                                        uint32_t pid, 
                                        uint32_t vrfId,
                                        open_buffdesc *clientName, 
                                        uint32_t *clientId);

/*********************************************************************
* @purpose  Deregister for routing events.
*
* @param    vrfId       @b{(input)} VRF Identifier
* @param    client_handle    @b{(input)} client handle from registration API
* @param    clientId    @b{(input)} Unique client ID received from RPPI at registration.
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_NOT_FOUND if client ID doesn't match a registered client
* @returns  OPEN_E_INTERNAL for an internal error
* @returns  OPEN_E_FAIL on other failure
*
* @notes     
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRouterEventDeregister(openapiClientHandle_t *client_handle,
                                          uint32_t vrfId,
                                          uint32_t clientId);

/*********************************************************************
* @purpose  Acknowledge a router event.
*
* @param    vrfId       @b{(input)} VRF Identifier
* @param    client_handle    @b{(input)} client handle from registration API
* @param    clientId    @b{(input)} Unique client ID received from RPPI at registration.
* @param    eventId     @b{(input)} ID of the event to be acknowledged. This is the
*                                   value of the eventSeqNumber field in the router
*                                   event message.
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_NOT_FOUND if client ID doesn't match a registered client
* @returns  OPEN_E_INTERNAL for an internal error
* @returns  OPEN_E_FAIL on other failure
*
* @notes     
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRouterEventAcknowledge(openapiClientHandle_t *client_handle,
                                           uint32_t vrfId,
                                           uint32_t clientId, 
                                           uint32_t eventId);

/*********************************************************************
* @purpose  Get the maximum number of OPEN routing interfaces.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    routingIntfMax      @b{(output)} The maximum number of routing interfaces.
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on invalid input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes    This is the maximum number of interfaces that can be simultaneously
*           configured as routing interfaces. 
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiMaxRoutingInterfacesGet(openapiClientHandle_t *client_handle,
                                            uint32_t *routingIntfMax);

/*********************************************************************
* @purpose  Get the next configured router interface
*
*
* @param    client_handle @b{(input)} client handle from registration API
* @param    vrfId         @b{(input)}  VRF Identifier
* @param    intf          @b{(input)}  The Configured Router Interface or 0 to get the first
*                                      router interface
* @param    nextIntf      @b{(output)} The next configured router interface
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on invalid input parameters
* @returns  OPEN_E_NOT_FOUND when no next interface found
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfNextGetVrf(openapiClientHandle_t *client_handle,
                                   uint32_t vrfId, 
                                   uint32_t intf, 
                                   uint32_t *nextIntf);

/*********************************************************************
* @purpose  Get the next configured router interface
*
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    intf          @b{(input)}  The Configured Router Interface or 0 to get the first
*                                      router interface
* @param    nextIntf      @b{(output)} The next configured router interface
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on invalid input parameters
* @returns  OPEN_E_NOT_FOUND when no next interface found
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfNextGet(openapiClientHandle_t *client_handle,
                                   uint32_t intf, 
                                   uint32_t *nextIntf);

/*********************************************************************
* @purpose  Get the type of the interface for the given router interface
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    intf        @b{(input)}    The Configured Router Interface
* @param    intfType      @b{(output)} OPEN_INTF_TYPE_t structure to hold the type of the interface
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfTypeGet(openapiClientHandle_t *client_handle,
                                   uint32_t intf, 
                                   OPEN_INTF_TYPE_t *intfType);

/*********************************************************************
* @purpose  Check if a given interface is unnumbered
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    intIfNum          @b{(input)}  Internal Interface Number
* @param    isUnnumbered      @b{(output)} status
*
* @returns  OPEN_E_NONE       If the unnumbered status retrieval is
                              successful
* @returns  OPEN_E_NOT_FOUND  If the interface is not configured
* @returns  OPEN_E_PARAM      If any of the input params are invalid
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t
openapiIntfIsUnnumbered (openapiClientHandle_t *client_handle,
                         uint32_t intIfNum,
                         OPEN_BOOL_t *isUnnumbered);

/*********************************************************************
* @purpose  Get the borrowed address and mask for an unnnumbered interface
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    intIfNum        @b{(input)} internal interface number of unnumbered interface
* @param    addr            @b{(input)} IP address
* @param    mask            @b{(input)} Address mask
* 
* @returns  borrowed IPv4 address and mask.
*
* @returns  OPEN_E_NONE       If the unnumbered status retrieval is
                              successful
* @returns  OPEN_E_NOT_FOUND  If the interface is not configured
* @returns  OPEN_E_PARAM      If any of the input params are invalid
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t
openapiIntfBorrowedAddrAndMaskGet (openapiClientHandle_t *client_handle,
                                   uint32_t intIfNum, uint32_t *addr,
                                   uint32_t *mask);

/*********************************************************************
* @purpose  Get the VRF name for a given VRF ID
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    vrfId            @b{(input)}  VRF Identifier
* @param    vrfName          @b{(output)} VRF name
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiVrfNameGet(openapiClientHandle_t *client_handle,
                               uint32_t vrfId, open_buffdesc *vrfName);

/*********************************************************************
* @purpose  Get the next VRF name
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    vrfName          @b{(input)}  VRF name (set to "") for initial.
* @param    vrfNameNext      @b{(output)} Next VRF name
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL if no more VRFs to report
*
* @notes	This API returns "" if VRF is not supported.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiVrfNameNextGet(openapiClientHandle_t *client_handle,
                                    open_buffdesc *vrfName,
                                    open_buffdesc *vrfNameNext);

/*********************************************************************
 * @purpose  Get the VRF Identifier from VRF name
 *
 * @param    client_handle    @b{(input)}  client handle from registration API
 * @param    vrfName          @b{(input)}  VRF name (set to "") for initial.
 * @param    vrfId            @b{(output)} VRF Id
 *
 * @returns  OPEN_E_NONE on success
 * @returns  OPEN_E_PARAM on improper input parameters
 *
 * @notes
 *
 * @supported-in-version OpEN API Version: 1.15
 *
 * @end
 *********************************************************************/
open_error_t
openapiVrfNameToIdGet (openapiClientHandle_t *client_handle,
                       open_buffdesc *vrfName,
                       uint32_t *vrfId);

/*********************************************************************
* @purpose  Get the vlan id on the configured router interface
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    intf        @b{(input)}   The Configured Router Interface
* @param    vlanId      @b{(output)}  Vlan id of the interface
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfVlanIdGet(openapiClientHandle_t *client_handle,
                                     uint32_t intf, 
                                     uint32_t *vlanId);

/*********************************************************************
* @purpose  Get the loopback id for the configured router interface
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    intf            @b{(input)}    The Configured Router Interface
* @param    loopbackId      @b{(output)}   Loopback id of the interface
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfLoopbackIdGet(openapiClientHandle_t *client_handle,
                                         uint32_t intf, 
                                         uint32_t *loopbackId);

/*********************************************************************
* @purpose  Get the internal interface number of the loopback interface
*
* @param    client_handle   @b{(input)} client handle from registration API
* @param    loopbackId      @b{(input)} Loopback id of the interface
* @param    intf            @b{(output)} Internal interface number
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiRtrLoopbackIntIfNumGet(openapiClientHandle_t *client_handle,
                                           uint32_t loopbackId, 
                                           uint32_t *intf);

/*********************************************************************
* @purpose  Get the maximum length of an OPEN interface name.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    ifNameSize    @b{(output)}  Maximum number of characters in an
*                                       OPEN interface name (not including
*                                       NULL terminator).
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIntfNameSizeGet(openapiClientHandle_t *client_handle,
                                    uint32_t *ifNameSize);

/*********************************************************************
* @purpose  Get the interface name of the configured router interface
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    intf          @b{(input)}   The Configured Router Interface
* @param    intfName      @b{(output)}  Name of the interface. This buffer
*                                       must be at least the length indicated
*                                       by openapiIntfNameSizeGet().
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfNameGet(openapiClientHandle_t *client_handle,
                                   uint32_t intf, 
                                   open_buffdesc *intfName);

/*********************************************************************
* @purpose  Get the interface number corresponding to the given ipstack ifIndex.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    ifIndex      @b{(input)} ipstack ifIndex of the interface
* @param    intIfNum     @b{(output)}  The Configured Router Interface
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiStackIfIndexIntIfNumGet(openapiClientHandle_t *client_handle,
    uint32_t ifIndex, uint32_t *intIfNum);

/*********************************************************************
* @purpose  Get the ipstack ifIndex on the configured router interface.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    intf         @b{(input)}  The Configured Router Interface
* @param    ifIndex      @b{(output)} ipstack ifIndex of the interface
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpStackIfIndexGet(openapiClientHandle_t *client_handle,
                                             uint32_t intf, 
                                             uint32_t *ifIndex);
 
/*********************************************************************
* @purpose  Get the name of the configured router interface.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    intf          @b{(input)}  The Configured Router Interface
* @param    intfName      @b{(output)} Name of the interface
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpStackIfNameGet(openapiClientHandle_t *client_handle,
                                   uint32_t intf, 
                                   open_buffdesc *intfName);

/*********************************************************************
* @purpose  Get the stk name of the configured router interface.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    vrfId         @b{(input)}  The Configured VRF interface
* @param    intf          @b{(input)}  The Configured Router Interface
* @param    intfName      @b{(output)} Name of the interface
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpStackStkNameGet(openapiClientHandle_t *client_handle,
                                             uint32_t vrfId, uint32_t intf, open_buffdesc *intfName);

/*********************************************************************
* @purpose  Get the drv name of the configured router interface.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    vrfId         @b{(input)}  The Configured VRF interface
* @param    intf          @b{(input)}  The Configured Router Interface
* @param    intfName      @b{(output)} Name of the interface
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpStackDrvNameGet(openapiClientHandle_t *client_handle,
                                             uint32_t vrfId, uint32_t intf, open_buffdesc *intfName);

/*********************************************************************
* @purpose  Get the name of the configured VRF interface.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    vrfId         @b{(input)}  The Configured VRF interface
* @param    intfName      @b{(output)} Name of the interface
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiVrfIntfNameGet(openapiClientHandle_t *client_handle,
                                   uint32_t vrfId, open_buffdesc *intfName);

/*********************************************************************
* @purpose  Get the unit, slot and port of the router interface.
*
* @param    client_handle @b{(input)}   client handle from registration API
* @param    intf          @b{(input)}   The Configured Router Interface
* @param    usp           @b{(output)}  unit/slot/port of the interface
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfUnitSlotPortGet(openapiClientHandle_t *client_handle,
                                           uint32_t intf,
                                           open_USP_t *usp);

/*********************************************************************
* @purpose  Get the L3 MAC address of the router interface.
*
* @param    client_handle @b{(input)}   client handle from registration API
* @param    intf          @b{(input)}   The Configured Router Interface
* @param    mac           @b{(output)}  L3 MAC address of the interface
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfL3MacAddressGet(openapiClientHandle_t *client_handle,
                                           uint32_t intf,
                                           open_buffdesc *mac);

/*********************************************************************
* @purpose Get the up/down status of the router interface given
*          the address family.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    af           @b{(input)}    Address Family
* @param    intf         @b{(input)}    The Configured Router Interface
* @param    ifState      @b{(output)}   OPEN_CONTROL_t structure returning the Operating mode
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfOperModeGet(openapiClientHandle_t *client_handle,
                                       OPEN_AF_t af, 
                                       uint32_t intf, 
                                       OPEN_CONTROL_t *ifState);

/*********************************************************************
* @purpose  Get an IP address on a given router interface for a given address family.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    af       @b{(input)}    Address Family (IPv4 or IPv6)
* @param    intf     @b{(input)}    A routing interface ID
* @param    ipAddr   @b{(in/out)}   IP address and prefix length. On input, this can
*                                   be set to an IP address on this interface, and
*                                   the call returns the next IP address on the
*                                   interface. Set to all zeros to retrieve the first
*                                   IP address on the interface.  
*
* @returns  OPEN_E_NONE      if an address is returned
* @returns  OPEN_E_EMPTY     if there are no more IP addresses to report on this interface
* @returns  OPEN_E_PARAM     if an input parameter is invalid
* @returns  OPEN_E_NOT_FOUND if a previous address is specified, but that address is not
*                           on this interface
* @returns  OPEN_E_FAIL      for any other failure
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpAddrGet(openapiClientHandle_t *client_handle,
                                     OPEN_AF_t af, 
                                     uint32_t intf, 
                                     open_inet_pfx_t *ipAddr);

/*********************************************************************
* @purpose  Get a Secondary IP address on a given router interface for a given address family.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    af       @b{(input)}    Address Family (IPv4)
* @param    intf     @b{(input)}    A routing interface ID
* @param    ipAddr   @b{(in/out)}   IP address and prefix length. On input, this can
*                                   be set to an IP address on this interface, and
*                                   the call returns the next IP address on the
*                                   interface. Set to all zeros to retrieve the first
*                                   IP address on the interface.  
* @param    index     @b{(output)}  Index of the Secondary IP
*
* @returns  OPEN_E_NONE      if an address is returned
* @returns  OPEN_E_EMPTY     if there are no more IP addresses to report on this interface
* @returns  OPEN_E_PARAM     if an input parameter is invalid
* @returns  OPEN_E_NOT_FOUND if a previous address is specified, but that address is not
*                           on this interface
* @returns  OPEN_E_FAIL      for any other failure
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfSecIpAddrGet(openapiClientHandle_t *client_handle,
                                     OPEN_AF_t af, 
                                     uint32_t intf, 
                                     open_inet_pfx_t *ipAddr,
                                     uint32_t *index);

/*********************************************************************
* @purpose  Get the method used to assign the IP address to a Router interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    addrMethod      @b{(output)}  IP address method
* @param    clientIdEnable  @b{(output)}  DHCP Client ID option enable flag
*
* @returns  OPEN_E_NONE        get successful
* @returns  OPEN_E_NOT_FOUND   not a configured routing interface
* @returns  OPEN_E_PARAM       error in parameter passed
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes    The clientIdEnable value is meaningful only when the
*           addrMethod is OPEN_INTF_IP_ADDR_METHOD_DHCP, otherwise
*           it is reported as OPEN_DISABLE.
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpAddrMethodGet(openapiClientHandle_t *client_handle,
                                           uint32_t intf,
                                           OPEN_INTF_IP_ADDR_METHOD_t *addrMethod,
                                           OPEN_CONTROL_t *clientIdEnable);

/*********************************************************************
* @purpose Get the configured ipmtu value for the router interface given
*          an address family
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    af               @b{(input)}    Address Family
* @param    intf             @b{(input)}    The Configured Router Interface
* @param    ipMtu            @b{(output)}   IP MTU configured on the interface
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIntfIpMtuGet(openapiClientHandle_t *client_handle,
                                 OPEN_AF_t af, 
                                 uint32_t intf, 
                                 uint32_t *ipMtu);

/*********************************************************************
* @purpose Get the bandwidth value configure on the device ( ** to change **)
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    af               @b{(input)}    Address Family
* @param    intf             @b{(input)}    The Configured Router Interface
* @param    bandwidth        @b{(output)}   Bandwidth for the interface
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIntfBandwidthGet(openapiClientHandle_t *client_handle,
                                     OPEN_AF_t af, 
                                     uint32_t intf, 
                                     uint32_t *bandwidth);

/*********************************************************************
* @purpose Get the ICMP Unreachables mode of the interface
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    intf             @b{(input)}    The Configured Router Interface
* @param    mode             @b{(output)}   ICMP Unreachables mode of the interface
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRtrICMPUnreachablesModeGet(openapiClientHandle_t *client_handle,
                                     uint32_t intf, 
                                     uint32_t *mode);

/*********************************************************************
* @purpose Get the ICMP Redirects mode of the interface
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    intf             @b{(input)}    The Configured Router Interface
* @param    mode             @b{(output)}   ICMP Redirects mode of the interface
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRtrICMPRedirectsModeGet(openapiClientHandle_t *client_handle,
                                     uint32_t intf, 
                                     uint32_t *mode);

/*********************************************************************
* @purpose Get the ICMP Ratelimit parameters
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    vrfName          @b{(input)}  VRF name
* @param    burstSize        @b{(output)} burst-size for ICMP rate-limit
* @param    interval         @b{(output)} interval for ICMP rate-limit
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRtrICMPRatelimitGet(openapiClientHandle_t *client_handle,
                                        open_buffdesc *vrfName, 
                                        uint32_t *burstSize,
                                        uint32_t *interval);

/*********************************************************************
* @purpose Get the ICMP ignore echo-req mode
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    vrfName          @b{(input)}  VRF name
* @param    mode             @b{(output)} ICMP ignore echo-request mode
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_FAIL on other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRtrICMPEchoReplyModeGet(openapiClientHandle_t *client_handle,
                                        open_buffdesc *vrfName, 
                                        uint32_t *mode);

/*********************************************************************
* @purpose  Get the current router admin mode given an address family
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    af          @b{(input)}   Address Family
* @param    enable      @b{(output)}  OPEN_CONTROL_t structure returning the admin mode
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on improper input parameters
* @returns  OPEN_E_UNAVAIL on failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrAdminModeGet(openapiClientHandle_t *client_handle,
                                    OPEN_AF_t af, 
                                    OPEN_CONTROL_t *enable);

/*********************************************************************
* @purpose  Report whether IP routing has finished initialization for a given
*           address family. 
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    af          @b{(input)}   Address Family
* @param    complete    @b{(output)}  1 if OPEN routing initialization is complete
*                                     0 if OPEN routing initialization is still underway
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on if the address family is unsupported
* @returns  OPEN_E_FAIL for any other failure
*
* @notes    When OPEN starts, the system configuration is applied and the
*           up/down status of all routing interfaces is determined. Prior to
*           completion of this initialization, the system does not push
*           state changes. When initialization is completed, RPPI router event clients 
*           are expected to proactively retrieve the routing configuration and state,
*           including the configuration and state of all routing interfaces.
* 
*           When an application starts, it can use this API to determine whether
*           OPEN is still initializing (startup is not complete). If so, the
*           application should wait until it receives a startup complete event
*           (OPENR_RTR_STARTUP_DONE for IPv4, OPENR_RTR6_STARTUP_DONE for IPv6)
*           through the router event service and then retrieve the complete
*           routing state. If this API indicates that OPEN initialization is already
*           complete, an application may immediately retrieve the routing state and
*           expect to receive a notification through the RPPI router event service
*           if any state changes.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRoutingStartupStatus(openapiClientHandle_t *client_handle,
                                         OPEN_AF_t af, 
                                         uint32_t *complete);

/*********************************************************************
* @purpose  Report whether IP routing has finished initialization in the
*           requested VRF for a given address family. 
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    af          @b{(input)}   Address Family
* @param    vrfId       @b{(input)}   VRF Identifier
* @param    complete    @b{(output)}  1 if OPEN routing initialization is complete
*                                     0 if OPEN routing initialization is still underway
*
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on if the address family is unsupported
* @returns  OPEN_E_FAIL for any other failure
*
* @notes    When OPEN starts, the system configuration is applied and the
*           up/down status of all routing interfaces is determined. Prior to
*           completion of this initialization, the system does not push
*           state changes. When initialization is completed, RPPI router event clients 
*           are expected to proactively retrieve the routing configuration and state,
*           including the configuration and state of all routing interfaces.
* 
*           When an application starts, it can use this API to determine whether
*           OPEN is still initializing (startup is not complete). If so, the
*           application should wait until it receives a startup complete event
*           (OPENR_RTR_STARTUP_DONE for IPv4, OPENR_RTR6_STARTUP_DONE for IPv6)
*           through the router event service and then retrieve the complete
*           routing state. If this API indicates that OPEN initialization is already
*           complete, an application may immediately retrieve the routing state and
*           expect to receive a notification through the RPPI router event service
*           if any state changes.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRoutingStartupStatusVr (openapiClientHandle_t *client_handle,
                                            OPEN_AF_t af,
                                            uint32_t vrfId,
                                            uint32_t *complete);

/*********************************************************************
* @purpose  Apply a route map with a given name to a route whose attributes are
*           set in the matchParams argument.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    routeMapName    @b{(input)}  Name of the route map to apply. Up to 32 characters.
* @param    matchParams     @b{(input)}  Route attributes to be matched
* @param    setParams       @b{(output)} If the route map changes any of the route's parameter
*                                        (through the route map's set statements), the new
*                                        values are set in this output structure. 
*
* @returns  0                 If the route map denies the route
* @returns  1                 If the route map permits the route
* @returns  OPEN_E_NOT_FOUND   If there is no route map with the given name.
* @returns  OPEN_E_PARAM       If an input parameter is invalid
* @returns  OPEN_E_FAIL        Any other failure
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
int openapiRouteMapApply(openapiClientHandle_t *client_handle,
                         open_buffdesc *routeMapName,
                         openRmMatchParams_t *matchParams,
                         openRmSetParams_t *setParams);

/*********************************************************************
* @purpose  Apply a route map with a given name to a route whose attributes are
*           set in the matchParams (metric,metricType and Tag) argument.
*
* @param    client_handle   @b{(input)} client handle from registration API
* @param    routeMapName    @b{(input)}  Name of the route map to apply. Up to 32 characters.
* @param    matchParams     @b{(input)}  Route attributes to be matched
* @param    setParams       @b{(output)} If the route map changes any of the route's parameter
*                                        (through the route map's set statements), the new
*                                        values are set in this output structure. 
* @param    action          @b{(output)} permit (1) or deny (0)
*
* @returns  OPEN_E_NOT_FOUND   If there is no route map with the given name.
* @returns  OPEN_E_PARAM       If an input parameter is invalid
* @returns  OPEN_E_FAIL        Any other failure
* @returns  OPEN_E_NONE        If succeeded
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiRouteMapFilter( openapiClientHandle_t *client_handle,
                                    open_buffdesc *routeMapName,
                                    openRmMatchParams_t *matchParams,
                                    openRmSetParams_t *setParams,
                                    OPEN_FILTER_ACTION_t *action);

/*********************************************************************
* @purpose  Apply a prefix list with a given name to a given prefix.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    pfxListName     @b{(input)}  Name of the prefix list to apply. Up to 32 characters.
* @param    prefix          @b{(input)}  Prefix to be matched
* @param    pfxLen          @b{(input)}  Length, in bits, of prefix to be matched.
*
* @returns  0                 If the prefix list denies the prefix
* @returns  1                 If the prefix list permits the prefix
* @returns  OPEN_E_NOT_FOUND   If there is no prefix list with the given name.
* @returns  OPEN_E_PARAM       If an input parameter is invalid
* @returns  OPEN_E_UNAVAIL     If the prefix's address family is not supported
* @returns  OPEN_E_FAIL        Any other failure
*
* @notes    Only IPv4 prefix lists are supported at this time.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
int openapiPrefixListApply(openapiClientHandle_t *client_handle,
                           open_buffdesc *pfxListName, 
                           open_inet_addr_t prefix, 
                           uint32_t pfxLen);

/*********************************************************************
* @purpose  Register to be notified when there are routing policy changes.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    clientName      @b{(input)}  Name of the client. Must be unique among all clients
*                                        registering for routing policy change notification. Up to
*                                        OPENR_POLICY_CLIENT_NAME_LEN characters long.
* @param    pid             @b{(input)}  Linux process ID of the process registering
* @param    policyTypeMask  @b{(input)}  Indicates which types of policies the application wants
*                                        to receive change notifications for. A 1 bit indicates
*                                        the application wants to receive change notification. The
*                                        mask is indexed by OPEN_ROUTING_POLICY_TYPE_t. To deregister
*                                        for all policy changes, set policyTypeMask to 0.
* @param    clientId        @b{(output)} Unique integer ID assigned to the client for this
*                                        registration. Used to create the address for the application's
*                                        UNIX socket. Must be non-zero if policyTypeMask is non-zero.
*
* @returns  OPEN_E_NONE        If the registration was successful
* @returns  OPEN_E_UNAVAIL     If RPPI is not ready to accept registrations yet. This may occur when
*                             switchdrvr is first starting and has not yet reached a state where it
*                             can accept best route client registrations. The caller should
*                             try again later.
* @returns  OPEN_E_PARAM       If an input parameter is invalid
* @returns  OPEN_E_FULL        If the maximum number of policy clients are already registered.
* @returns  OPEN_E_NOT_FOUND   If no client with the given name is in the current list of clients
*                             when a call is made to deregister (policyTypeMask is 0).
* @returns  OPEN_E_FAIL        Any other failure
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRoutingPolicyRegister(openapiClientHandle_t *client_handle,
                                          open_buffdesc *clientName, 
                                          uint32_t  pid,
                                          uint32_t policyTypeMask,
                                          uint32_t *clientId);

/*********************************************************************
* @purpose  Acknowledge receipt of a policy change notification.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    clientId  @b{(input)}  Policy client ID assigned by RPPI at registration time.
* @param    seqNo     @b{(input)}  Sequence number from policy change notification message.
*                                  Acknowledges the change with this and all lower sequence
*                                  numbers.
*
* @returns  OPEN_E_NONE        Acknowledgement accepted.
* @returns  OPEN_E_NOT_FOUND   If the client ID does not match a registered client
* @returns  OPEN_E_FAIL        If the sequence number does not match a change
*                             previously reported.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiPolicyChangeAck(openapiClientHandle_t *client_handle,
                                    uint32_t clientId, 
                                    uint32_t seqNo);

/*********************************************************************
* @purpose  Get router preference associated with the specified protocol
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    index               @b{(input)}  router protocol type
* @param    pref                @b{(output)} router preference value
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIpRouterPreferenceGet(openapiClientHandle_t *client_handle,
                                          OPEN_PREF_INDICES_t index,
                                          uint32_t *pref);

/*********************************************************************
* @purpose  Set router preference associated with the specified protocol
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    index               @b{(input)}  router protocol type
* @param    pref                @b{(input)}  router preference value
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIpRouterPreferenceSet(openapiClientHandle_t *client_handle,
                                          OPEN_PREF_INDICES_t index,
                                          uint32_t pref);

/*********************************************************************
* @purpose  Get the protocol name assigned to the associated protocol id. 
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    protoId          @b{(input)}  unique identifier for the routing protocol.
* @param    protoName        @b{(input)}  Name of the routing protocol. This name is
*                                         used in the UI to refer to this protocol.
*                                         (e.g., "redistribute <protoName>).
*                                         openapiRouteProtoNameLenMax() gives the max length.
* @param    nextProtoId      @b{(output)} next unique identifier for the routing protocol.
* 
* @returns  OPEN_E_PARAM       If the output parameter is NULL.
* @returns  OPEN_E_INTERNAL    If any of the input names are too long or otherwise invalid.
* @returns  OPEN_E_FAIL        Other failure.
*
* @notes    Initial value of protoId should be 0 if caller wants to retrieve first available.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIpRouterProtoNameNextGet(openapiClientHandle_t *client_handle,
                                             uint32_t protoId,
                                             open_buffdesc *protoName,
                                             uint32_t *nextProtoId);

/*********************************************************************
* @purpose  Get IPv6 router preference associated with the specified protocol
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    index               @b{(input)}  router protocol type
* @param    pref                @b{(output)} router preference value
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIp6RouterPreferenceGet(openapiClientHandle_t *client_handle,
                                           OPEN_PREF_INDICES_t index,
                                           uint32_t *pref);

/*********************************************************************
* @purpose  Set IPV6 router preference associated with the specified protocol
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    index               @b{(input)}  router protocol type
* @param    pref                @b{(input)}  router preference value
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIp6RouterPreferenceSet(openapiClientHandle_t *client_handle,
                                           OPEN_PREF_INDICES_t index,
                                           uint32_t pref);

/******* NEW OpENAPIs Added for virtual router support *************/

/*********************************************************************
* @purpose  Register to be notified when there are changes to one or more
*           routes used for IP forwarding.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    vrfId            @b{(input)} VRF instance
* @param    family      @b{(input)}  OPEN_AF_INET to be notified of changes to IPv4 routes
*                                    OPEN_AF_INET6 to be notified of changes to IPv6 routes
* @param    pid         @b{(input)}  Linux process ID of the process registering
* @param    clientName  @b{(input)}  Name of the client. Must be unique among all clients
*                                    registering for best route change notification. Up to
*                                    OPENR_BRC_NAME_LEN characters long.
* @param    clientId    @b{(output)} Unique integer ID assigned to the client for this
*                                    registration. Used to identify the client when retrieving
*                                    changes.
*
* @returns  OPEN_E_NONE        If the registration was successful
*           OPEN_E_UNAVAIL     If RPPI is not ready to accept registrations yet. This may occur when
*                             switchdrvr is first starting and has not yet reached a state where it
*                             can accept best route client registrations. The caller should
*                             try again later.
* @returns  OPEN_E_FULL        If the maximum number of best route clients are already registered.
* @returns  OPEN_E_FAIL        Any other failure
*
* @notes    RPC API.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiBestRouteEventRegisterVr(openapiClientHandle_t *client_handle,
                                           uint32_t vrfId,
                                           OPEN_AF_t family,
                                           uint32_t pid,
                                           open_buffdesc *clientName,
                                           uint32_t   *clientId);

/*********************************************************************
* @purpose  Add a route to the common routing table.
*
* @param    client_handle     @b{(input)} client handle from registration API
* @param    vrfId             @b{(input)} VRF Identifier
* @param    routeEntry        @b{(input)}  Route to be added.
* @param    nextHopListBuff   @b{(input)}  List of nextHops associated with route.
*
* @returns  OPEN_E_NONE        If the route was added successfully.
* @returns  OPEN_E_NOT_FOUND   If none of the route's next hops is in a local subnet.
* @returns  OPEN_E_FULL        If the route was not added because the routing table is full.
* @returns  OPEN_E_EXISTS      If a route with the same route type and destination is already
*                              in the routing table.
* @returns  OPEN_E_FAIL        Any other failure.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRouteAddVr(openapiClientHandle_t *client_handle,
                               uint32_t vrfId,
                               openRoute_t *routeEntry,
                               open_buffdesc *nextHopListBuff);


/******************************************************************************
* @purpose  Set the BFD Global Admin Mode
*
* @param    mode @b{(input)} the new value of the BFD mode
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_FAIL
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*******************************************************************************/
open_error_t openapiBfdModeSet(OPEN_BFD_ADMIN_MODE_t mode);

/*********************************************************************
* @purpose  Create a BFD Session.
* 
* @param    client_handle     @b{(input)}  client handle from registration API
* @param    ep                @b{(input)}  BFD endpoint parameters
* @param    id                @b{(output)} Session identifier 
* 
* @returns  OPEN_E_NONE        If the session was created successfully.
* @returns  OPEN_E_FAIL        If the session was not created.
*   
* @notes
* 
* @supported-in-version OpEN API Version: 1.3
*   
* @end
*********************************************************************/
open_error_t openapiBfdSessionCreate(openapiClientHandle_t *client_handle,
                                     openBfdEndpoint_t *ep, 
                                     uint32_t *id);

/*********************************************************************
* @purpose  Delete a BFD Session.
* 
* @param    client_handle     @b{(input)}  client handle from registration API
* @param    ep                @b{(input)}  BFD endpoint parameters
* @param    id                @b{(input)}  Session identifier 
* 
* @returns  OPEN_E_NONE        If the session was deleted successfully.
* @returns  OPEN_E_FAIL        If the session was not deleted.
*   
* @notes
* 
* @supported-in-version OpEN API Version: 1.3
*   
* @end 
*********************************************************************/
open_error_t openapiBfdSessionDelete(openapiClientHandle_t *client_handle,
                                     openBfdEndpoint_t *ep, 
                                     uint32_t id);

/*********************************************************************
* @purpose  Get the BFD Session Identifier for a given peer by IP address.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    peer          @b{(input)}  peer IP address
* @param    id            @b{(output)} session identifier
*
* @returns  OPEN_E_NONE           On success.
* @returns  OPEN_E_PARAM          Invalid parameter.
* @returns  OPEN_E_FAIL           Failure occurred.
* @returns  OPEN_E_INTERNAL       Internal error occurred. 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiBfdSessionFind(openapiClientHandle_t *client_handle,
                                   open_inet_addr_t peer,
                                   uint32_t *id);

/*********************************************************************
* @purpose  Get the BFD session information
*
* @param    client_handle     @b{(input)}  client handle from registration API
* @param    id                @b{(input)}  Session identifier
* @param    info              @b{(output)} Session information
*
* @returns  OPEN_E_NONE        If the session was created successfully.
* @returns  OPEN_E_FAIL        If the session was not created.
*
* @notes    The session identifier is returned by openapiBfdSessionCreate() API.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiBfdSessionInfoGet(openapiClientHandle_t *client_handle,
                                      uint32_t id,
                                      openBfdSessionInfo_t *info);

/*********************************************************************
* @purpose  Get the BFD statistics associated with the given session
*
* @param    client_handle     @b{(input)}  client handle from registration API
* @param    id        @b{(input)} session identifier
* @param    stats     @b{(output)} session stats
*
* @returns  OPEN_E_NONE  on successful operation
* @returns  OPEN_E_FAIL  if the operation failed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiBfdSessionStatsGet(openapiClientHandle_t *client_handle,
                                       uint32_t id, openBfdSessionStats_t *stats);


/*********************************************************************
* @purpose  Add a kernel route to the common routing table.
*
* @param    client_handle     @b{(input)} client handle from registration API
* @param    vrfId             @b{(input)} VRF Identifier
* @param    routeEntry        @b{(input)}  Route to be added.
* @param    nextHopListBuff   @b{(input)}  List of nextHops associated with route.
*
* @returns  OPEN_E_NONE        If the route was added successfully.
* @returns  OPEN_E_NOT_FOUND   If none of the route's next hops is in a local subnet.
* @returns  OPEN_E_FULL        If the route was not added because the routing table is full.
* @returns  OPEN_E_EXISTS      If a route with the same route type and destination is already
*                              in the routing table.
* @returns  OPEN_E_FAIL        Any other failure.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiKernelRouteAddVr(openapiClientHandle_t *client_handle,
                               uint32_t vrfId,
                               openRoute_t *routeEntry,
                               open_buffdesc *nextHopListBuff);

/*********************************************************************
* @purpose  Modify an existing route to the common routing table.
*
* @param    client_handle     @b{(input)} client handle from registration API
* @param    vrfId             @b{(input)} VRF Identifier
* @param    routeEntry        @b{(input)}  Route to be modified.
* @param    nextHopListBuff   @b{(input)}  List of nextHops associated with route.
*
* @returns  OPEN_E_NONE        If the route was successfully modified.
* @returns  OPEN_E_NOT_FOUND   If the route table does not have a route to this destination
*                             with the given route type.
* @returns  OPEN_E_FAIL        Any other failure.
*
* @notes    This API can only be called when replacing one route with
*           another route with the same route type. It cannot be used to
*           replace a route with a different route type, even if the two
*           route types are for the same routing protocol.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRouteModVr(openapiClientHandle_t *client_handle,
                               uint32_t vrfId,
                               openRoute_t *routeEntry,
                               open_buffdesc *nextHopListBuff);

/*********************************************************************
* @purpose  Delete a route from the common routing table.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    vrfId            @b{(input)} VRF Identifier
* @param    routeEntry       @b{(input)} Route to be deleted.
* @param    nextHopListBuff  @b{(input)} Next hop list.
*
* @returns  OPEN_E_NONE        If the route was successfully deleted.
* @returns  OPEN_E_NOT_FOUND   If the route table does not have a route to this destination
*                             with the given route type.
* @returns  OPEN_E_FAIL        Any other failure.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiRouteDelVr(openapiClientHandle_t *client_handle,
                               uint32_t vrfId,
                               openRoute_t *routeEntry,
                               open_buffdesc *nextHopListBuff);

/*********************************************************************
* @purpose  Delete a kernel route from the common routing table.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    vrfId            @b{(input)} VRF Identifier
* @param    routeEntry       @b{(input)} Route to be deleted.
* @param    nextHopListBuff  @b{(input)} Next hop list.
*
* @returns  OPEN_E_NONE        If the route was successfully deleted.
* @returns  OPEN_E_NOT_FOUND   If the route table does not have a route to this destination
*                             with the given route type.
* @returns  OPEN_E_FAIL        Any other failure.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiKernelRouteDelVr(openapiClientHandle_t *client_handle,
                               uint32_t vrfId,
                               openRoute_t *routeEntry,
                               open_buffdesc *nextHopListBuff);

/*********************************************************************
* @purpose  Get the number of available routes in RTO for this VRF Id
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    vrfId             @b{(input)}  VRF Identifier
* @param    rtoFreeRouteCount @b{(output)} Free route count in RTO
*
* @returns  OPEN_E_NONE        If the RTO call is successful
* @returns  OPEN_E_ERROR       If the RTO call is unsuccessful
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t
openapiRtoFreeRouteCountGet (openapiClientHandle_t *client_handle,
                             uint32_t vrfId,
                             uint32_t *rtoFreeRouteCount);

/*********************************************************************
* @purpose  Get the router admin mode of the VRF
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    af                @b{(input)}  Address family
* @param    vrfId             @b{(input)}  VRF Identifier
* @param    enable            @b{(output)} Enable mode
*
* @returns  OPEN_E_NONE       If the Admin mode get call is successful
* @returns  OPEN_E_FAIL       If the Admin mode get call is unsuccessful
* @returns  OPEN_E_PARAM      If the passed params are invalid
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t
openapiRtrAdminModeGetVrf (openapiClientHandle_t *client_handle,
                           OPEN_AF_t af,
                           uint32_t vrfId,
                           OPEN_CONTROL_t *enable);

/*********************************************************************
* @purpose  Get the number of ECMP MAX PATHS FROM PLATFORM params
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    pval              @b{(output)} Max ECMP paths
*
* @returns  OPEN_E_NONE        If the OPEN call is successful
* @returns  OPEN_E_ERROR       If the OPEN call is unsuccessful
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t
openapiPlatRtrRouteMaxEqualCostEntriesGet (openapiClientHandle_t *client_handle,
                                           uint32_t *pval);

/*********************************************************************
* @purpose  Get the number of IPv4 routes for this platform
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    pval              @b{(output)} Max IPv4 routes
*
* @returns  OPEN_E_NONE        If the OPEN call is successful
* @returns  OPEN_E_ERROR       If the OPEN call is unsuccessful
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t
openapiPlatRtrRouteMaxEntriesGet (openapiClientHandle_t *client_handle,
                                  uint32_t *pval);

/*********************************************************************
* @purpose  Indicate that a protocol intends to add routes to RTO following
*           a warm restart.
*
* @param    client_handle     @b{(input)} Client handle from registration API
* @param    vrfId             @b{(input)} VRF Id
* @param    protocol          @b{(input)} Protocol to add NSF routes
* 
* @returns  OPEN_E_PARAM      If the passed params are invalid
* @returns  OPEN_E_NONE       If the OPEN call is successful
* @returns  OPEN_E_FAIL       If the OPEN call is unsuccessful
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t
openapiIpMapNsfRouteSource(openapiClientHandle_t *client_handle,
                           uint32_t vrfId,
                           OPEN_NSF_PROTOCOL_t protocol);

/*********************************************************************
* @purpose  A protocol that controls NSF routes calls this function to
*           indicate it has updated RTO with all its initial routes after
*           a warm restart.
*
* @param    client_handle     @b{(input)} Client handle from registration API
* @param    vrfId             @b{(input)} VRF Id
* @param    protocol          @b{(input)} Protocol to add NSF routes
*
* @returns  OPEN_E_PARAM      If the passed params are invalid
* @returns  OPEN_E_NONE       If the OPEN call is successful
* @returns  OPEN_E_FAIL       If the OPEN call is unsuccessful
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t
openapiIpMapInitialRoutesDone(openapiClientHandle_t *client_handle,
                              uint32_t vrfId,
                              OPEN_NSF_PROTOCOL_t protocol);

/*********************************************************************
* @purpose  Check if backup is elected.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    isBackupElected   @b{(output)} Backup elected status
* 
* @returns  OPEN_E_PARAM      If the passed params are invalid
* @returns  OPEN_E_NONE       If the OPEN call is successful
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiIpMapIsBackupElectedGet(openapiClientHandle_t *client_handle,
                                            OPEN_BOOL_t *isBackupElected);

/*********************************************************************
* @purpose  Set the Unnumbered peer info on an interface
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    intIfNum           @b{(input)}  interface number
* @param    peerIp            @b{(input)}  Unnumbered peer IP address
* @param    isAdd             @b{(input)}  add/delete
* @param    flag              @b{(input)}  flag to determine the source
*
* @returns  OPEN_E_NONE        If the OPEN call is successful
* @returns  OPEN_E_ERROR       otherwise
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t
openapiUnnumberedPeerInfoSet (openapiClientHandle_t *client_handle,
                              uint32_t intIfNum, uint32_t peerIp, OPEN_BOOL_t isAdd, uint32_t flag);

/*********************************************************************
* @purpose  Add the ARP entry 
*
* @param    vrfId         @b{(input)}  VRF Identifier
* @param    client_handle @b{(input)}  client handle from registration API
* @param    dest_addr     @b{(input)}  destination IP address
* @param    mac           @b{(input)}  destination MAC address
* @param    stkIfIndex    @b{(input)}  stack if index of the interface
* @param    state         @b{(input)}  state of the arp entry
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_PARAM        
* @returns  OPEN_E_UNAVAIL
* @returns  OPEN_E_ERROR
* @returns  OPEN_E_EXISTS   
* @returns  OPEN_E_INTERNAL
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiKernelArpAdd (openapiClientHandle_t *client_handle,
                                  uint32_t vrfId, 
                                  open_inet_addr_t *dest_addr,
                                  open_buffdesc *mac,
                                  int32_t stkIfIndex,
                                  uint16_t state);

/*********************************************************************
* @purpose  Delete the ARP entry 
*
* @param    vrfId         @b{(input)}  VRF Identifier
* @param    client_handle @b{(input)}  client handle from registration API
* @param    dest_addr     @b{(input)}  destination IP address
* @param    stkIfIndex    @b{(input)}  stack if index of the interface
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_PARAM        
* @returns  OPEN_E_UNAVAIL
* @returns  OPEN_E_ERROR   
* @returns  OPEN_E_NOT_FOUND
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiKernelArpDelete (openapiClientHandle_t *client_handle,
                                     uint32_t vrfId,
                                     open_inet_addr_t *dest_addr,
                                     int32_t stkIfIndex);

/*********************************************************************
* @purpose  Get router preference associated with the specified protocol in a VRF
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vrfId               @b{(input)}  VRF Identifier
* @param    index               @b{(input)}  router protocol type
* @param    pref                @b{(output)} router preference value
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.3
* 
* @end
*********************************************************************/
open_error_t openapiIpRouterPreferenceGetVr (openapiClientHandle_t *client_handle,
                                             uint32_t vrfId,
                                             OPEN_PREF_INDICES_t index,
                                             uint32_t *pref);

/*********************************************************************
* @purpose  Get the IPv4 statistics for the system
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ipv4Stats           @b{(output)} IPv4 system statistics
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_FAIL         Upon any failure in fetching the stats.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t
openapiRtrIpv4StatsGet (openapiClientHandle_t *client_handle,
                        openrIpv4Stats_t *ipv4Stats);

/*********************************************************************
* @purpose  Get the IP default TTL of the system
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    defTTL           @b{(output)} default TTL value
*
* @returns  OPEN_E_NONE      on success
* @returns  OPEN_E_PARAM     on error in the parameters passed
* @returns  OPEN_E_FAIL      on any other failure
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
 open_error_t
 openapiIpDefaultTTLGet (openapiClientHandle_t *client_handle,
                         uint32_t *defTTL);

#endif /* OPENAPI_ROUTING_H_INCLUDED */
