/**********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_sslt.h
*
* @purpose   This code implements the OpEN SSLT API
*
* @component OPEN
*
* @create    01/30/2023
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_SSLT_H_INCLUDED
#define OPENAPI_SSLT_H_INCLUDED

#include "openapi_common.h"

typedef enum
{
  OPEN_SSLT_PROTOCOL_SSL30 = 0,
  OPEN_SSLT_PROTOCOL_TLS10,
  OPEN_SSLT_PROTOCOL_TLS11,
  OPEN_SSLT_PROTOCOL_TLS12,
  OPEN_SSLT_PROTOCOL_TLS13
} OPEN_SSLT_PROTOCOL_t;

/*****************************************************************************
 * @purpose   Sets admin Mode of the SSL Tunnel server.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     mode           @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API sets SSL Admin Mode. And remove all existing HTTPS sessions 
 *            if required
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltAdminModeSet(openapiClientHandle_t *client_handle,
                                     OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Gets admin mode of SSL Tunnel server.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pMode          @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API fetches SSL Admin Mode.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltAdminModeGet(openapiClientHandle_t *client_handle,
                                     OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Sets secure port number to listen on for SSLT connections.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     port           @b{(input)}  port-number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API sets secure port number to listen on for SSLT connections
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltSecurePortSet(openapiClientHandle_t *client_handle,
                                      uint32_t port);

/*****************************************************************************
 * @purpose   Gets secure port number that the SSLT connection is using.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pPort          @b{(output)} port number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API fetches secure port information used for SSLT connection
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltSecurePortGet(openapiClientHandle_t *client_handle,
                                      uint32_t *pPort);

/*****************************************************************************
 * @purpose   Sets SSL Protocol Level to be used by SSL Tunnel codes.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     protocolId     @b{(input)}  ID of SSLT supported protocol
 * @param     mode           @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API sets SSL Protocol Level to be used by SSL Tunnel connections
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltProtocolLevelSet(openapiClientHandle_t *client_handle,
                                         OPEN_SSLT_PROTOCOL_t protocolId,
                                         OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Gets SSL Protocol mode status in use by the SSL Tunnel code.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     protocolId     @b{(input)}  ID of SSLT supported protocol
 * @param     pMode          @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API fetches SSL Protocol mode information used for SSLT connection
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltProtocolLevelGet(openapiClientHandle_t *client_handle,
                                         OPEN_SSLT_PROTOCOL_t protocolId,
                                         OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Sets Soft Timeout (in minutes) for SSLT sessions.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     timeout        @b{(input)}  SSLT session soft timeout
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API sets SSLT sessions soft timeout
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltSessionSoftTimeOutSet(openapiClientHandle_t *client_handle,
                                              uint32_t timeout);

/*****************************************************************************
 * @purpose   Gets SSLT session soft timeout information.(in minutes)
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pTimeout       @b{(output)} SSLT session soft timeout
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API fetches information of SSLT session soft timeout
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltSessionSoftTimeOutGet(openapiClientHandle_t *client_handle,
                                              uint32_t *pTimeout);

/*****************************************************************************
 * @purpose   Sets SSLT session hard timeout (in hours).
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     timeout        @b{(input)}  SSLT session hard timeout
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API sets SSLT session hard timeout
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltSessionHardTimeOutSet(openapiClientHandle_t *client_handle,
                                              uint32_t timeout);

/*****************************************************************************
 * @purpose   Gets SSLT session hard timeout information (in hours).
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pTimeout       @b{(output)} HTTP session hard timeout
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API fetches information of SSLT session hard timeout
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltSessionHardTimeOutGet(openapiClientHandle_t *client_handle,
                                              uint32_t *pTimeout);

/*****************************************************************************
 * @purpose   Sets maximum allowable SSLT sessions.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     maxSession     @b{(input)}  maximum allowable number of SSLT sessions
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API sets maximum number of SSLT sessions
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltNumSessionsSet(openapiClientHandle_t *client_handle,
                                       uint32_t maxSession);

/*****************************************************************************
 * @purpose   Gets  maximum number of SSLT sessions supported.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pSession       @b{(output)} maximum allowable number of web sessions configured
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API fetches maximum number of SSLT sessions configured
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltNumSessionsGet(openapiClientHandle_t *client_handle,
                                       uint32_t *pSession);

/*****************************************************************************
 * @purpose   Gets operation mode of SSL Tunnel server.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pMode          @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API fetches SSL oper Mode.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltOperModeGet(openapiClientHandle_t *client_handle,
                                    OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Determine if self-signed server certificate exists.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     certNum        @b{(input)}  certificate number (1 or 2)
 *
 * @returns   OPEN_E_NOT_FOUND        If cert not found/present.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API determine server certificate existence.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltCertificateExists(openapiClientHandle_t *client_handle,
                                          uint32_t certNum);

/*****************************************************************************
 * @purpose   Determine if self-signed server and rootcert certificates exist.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     certNum        @b{(input)}  certificate number (1 or 2)
 *
 * @returns   OPEN_E_NOT_FOUND        If cert not found/present.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API determine server and rootcert certificate existence.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltCertificateAndServerKeyExists(openapiClientHandle_t *client_handle,
                                                    uint32_t certNum);

/*****************************************************************************
 * @purpose   Generate a self-signed server certificate for SSL tunnel.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     certNum        @b{(input)}  certificate number
 * @param     keyLength      @b{(input)}  RSA key length, 0 will use existing key.
 * @param     pCommonName    @b{(input)}  subject DN common name field.
 * @param     pOrgName       @b{(input)}  subject DN organization name field.
 * @param     pOrgUnit       @b{(input)}  subject DN organization unit field.
 * @param     pLocation      @b{(input)}  subject DN location field.
 * @param     pState         @b{(input)}  subject DN state field.
 * @param     pCountry       @b{(input)}  subject DN country field.
 * @param     pEmail         @b{(input)}  subject DN email field.
 * @param     days           @b{(input)}  days certificate will be valid.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     If certificate does not exist, a keyLength must be specified
 * @notes     to generate an RSA key pair on the device.
 * @notes     keyLength value range is determined by calling 
 * @notes     @ref openapiSsltCertificateKeyLengthGet()
 * @notes     DN field length range is determined by calling
 * @notes     @ref openapiSsltCertificateSubjectNameSizeGet()
 * @notes     country length is determined by calling
 *            @ref openapiSsltCertificateCountryNameSizeGet().
 * @notes     days value range is determined by calling
 *            @ref openapiSsltCertificateValidDaysGet().
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltCertificateGenerate(openapiClientHandle_t *client_handle,
                                            uint32_t certNum, uint32_t keyLength,
                                            open_buffdesc *pCommonName,
                                            open_buffdesc *pOrgName,
                                            open_buffdesc *pOrgUnit,
                                            open_buffdesc *pLocation,
                                            open_buffdesc *pState,
                                            open_buffdesc *pCountry,
                                            open_buffdesc *pEmail,
                                            uint32_t days);

/*****************************************************************************
 * @purpose   Remove SSLT certificate.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     certNum        @b{(input)}  certificate number (1 or 2)
 *
 * @returns   OPEN_E_DEPENDENCY_NOT_MET  If HTTPS is operationally enabled.
 * @returns   OPEN_E_INTERNAL            If input certificate is user selected 
 *                                       certificate.
 * @returns   OPEN_E_NONE                On success.
 * @returns   OPEN_E_FAIL                On failure.
 * @returns   OPEN_E_PARAM               Invalid argument.
 * @returns   OPEN_E_UNAVAIL             Feature not available.
 *
 * @notes     This API removes SSLT certificate.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltCertRemove(openapiClientHandle_t *client_handle,
                                   uint32_t certNum);

/*****************************************************************************
 * @purpose   Gets the active certificate for the SSL tunnel.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pCertNum       @b{(output)} certificate number (1 or 2)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API fetches active SSL tunnel certificate number.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltCertificateActiveGet(openapiClientHandle_t *client_handle, 
                                             uint32_t *pCertNum);

/*****************************************************************************
 * @purpose   Gets operational active SSL certificate.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pCertNum       @b{(output)} certificate number (1 or 2)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API fetches SSL tunnel operational active certificate number.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltCertificateOperActiveGet(openapiClientHandle_t *client_handle,
                                                 uint32_t *pCertNum);

/*****************************************************************************
 * @purpose   Gets status of SSL certificate expiry.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     CertNum        @b{(input)}  certificate number (1 or 2)
 * @param     pIsExpired     @b{(output)} TRUE  - Certificate is not valid, 
 *                                        FALSE - Certificate is valid
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API fetches status of SSL certificate expiry.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltCertificateExpiryStatusGet(openapiClientHandle_t *client_handle,
                                                   uint32_t CertNum,
                                                   OPEN_BOOL_t *pIsExpired);

/*****************************************************************************
 * @purpose   Set the active certificate for the SSL tunnel.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     certNum        @b{(input)}  certificate number (1 or 2)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API Sets the active certificate for the SSL tunnel.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltCertificateActiveSet(openapiClientHandle_t *client_handle,
                                             uint32_t certNum);

/*****************************************************************************
 * @purpose   Check if HTTPS mode is enabled and the certificate is active.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     certNum             @b{(input)}  certificate number (1 or 2)
 * @param     pIsHttpModeEnabled  @b{(output)} certificate number (1 or 2)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     This API check if HTTPS mode is enabled and the certificate is active.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltCheckHTTPSEnabledAndOperational(openapiClientHandle_t *client_handle,
                                                        uint32_t certNum,
                                                        OPEN_BOOL_t *pIsHttpModeEnabled);

/*****************************************************************************
 * @purpose   Get the minimum and maximum supported sizes of certificate 
 *            keylength.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API.
 * @param     pMinKeyLength       @b{(output)} Minimum certificate keylength.
 * @param     pMaxKeyLength       @b{(output)} Maximum certificate keylength.
 *
 * @returns   OPEN_E_NONE         Maximum and minimum certificate keylength
 *                                are returned successfully.
 * @returns   OPEN_E_PARAM        Invalid argument.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltCertificateKeyLengthGet(openapiClientHandle_t *client_handle,
                                                    uint32_t *pMinKeyLength,
                                                    uint32_t *pMaxKeyLength);

/*****************************************************************************
 * @purpose   Get the minimum and maximum supported sizes of certificate 
 *            Subject domain name.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API.
 * @param     pMinSubjectName     @b{(output)} Minimum size of certificate Subject domain name.
 * @param     pMaxSubjectName     @b{(output)} Maximum size of certificate Subject domain name.
 *
 * @returns   OPEN_E_NONE         Maximum and minimum sizes of the certificate Subject domain name
 *                                are returned successfully.
 * @returns   OPEN_E_PARAM        Invalid argument.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltCertificateSubjectNameSizeGet(openapiClientHandle_t *client_handle,
                                                      uint32_t *pMinSubjectName,
                                                      uint32_t *pMaxSubjectName);

/*****************************************************************************
 * @purpose   Get the minimum and maximum supported duration of certificate validity.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API.
 * @param     pMinValidDays       @b{(output)} Minimum certificate valid days.
 * @param     pMaxValidDays       @b{(output)} Maximum certificate valid days.
 *
 * @returns   OPEN_E_NONE         Maximum and minimum certificate validity
 *                                are returned successfully.
 * @returns   OPEN_E_PARAM        Invalid argument.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltCertificateValidDaysGet(openapiClientHandle_t *client_handle,
                                                uint32_t *pMinValidDays,
                                                uint32_t *pMaxValidDays);

/*****************************************************************************
 * @purpose   Get certificate country name size.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API.
 * @param     pCountryName        @b{(output)} certificate country name size.
 *
 * @returns   OPEN_E_NONE         certificate country name size is returned successfully.
 * @returns   OPEN_E_PARAM        Invalid argument.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiSsltCertificateCountryNameSizeGet(openapiClientHandle_t *client_handle,
                                                      uint32_t *pCountryName);
#endif /* OPENAPI_SSLT_H_INCLUDED */
