/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_stacking.h
*
* @purpose   Stacking Configuration and Status 
*
* @component OpEN
*
* @create    10/22/2015
*
* @end
*                                    
**********************************************************************/
#ifndef OPENAPI_STACKING_H_INCLUDED
#define OPENAPI_STACKING_H_INCLUDED

#include "openapi_common.h"

/** Stacking Unit Status */
typedef enum
{
  OPEN_USMDB_UNITMGR_UNIT_OK          = 1,  /**< Unit Ok */
  OPEN_USMDB_UNITMGR_UNIT_UNSUPPORTED = 2,  /**< Unit Unsupported */
  OPEN_USMDB_UNITMGR_CODE_MISMATCH    = 3,  /**< Code Mismatch */
  OPEN_USMDB_UNITMGR_CFG_MISMATCH     = 4,  /**< CFG Mismatch */
  OPEN_USMDB_UNITMGR_SDM_MISMATCH     = 5,  /**< SDM Mismatch */
  OPEN_USMDB_UNITMGR_UNIT_NOT_PRESENT = 6,  /**< Unit Not Present */
  OPEN_USMDB_UNITMGR_CODE_UPDATE      = 7,  /**< Code Update */
  OPEN_USMDB_UNITMGR_STM_MISMATCH     = 8   /**< STM Mismatch*/
} OPEN_USMDB_UNITMGR_STATUS_t;

/** Management Flag */
typedef enum
{
  OPEN_USMDB_UNITMGR_MGMT_ENABLED     = 1,  /**< Enabled */
  OPEN_USMDB_UNITMGR_MGMT_DISABLED    = 2,  /**< Disabled */
  OPEN_USMDB_UNITMGR_MGMT_UNASSIGNED  = 3   /**< Unassigned */
} OPEN_USMDB_UNITMGR_MGMT_FLAG_t;

/** Unit's Management Preference */
typedef enum
{
  OPEN_USMDB_UNITMGR_MGMTPREF_DISABLED    = 0,  /**< Disabled */
  OPEN_USMDB_UNITMGR_MGMTPREF_1           = 1,  /**< Preference 1 */
  OPEN_USMDB_UNITMGR_MGMTPREF_2           = 2,  /**< Preference 2 */
  OPEN_USMDB_UNITMGR_MGMTPREF_3           = 3,  /**< Preference 3 */
  OPEN_USMDB_UNITMGR_MGMTPREF_4           = 4,  /**< Preference 4 */
  OPEN_USMDB_UNITMGR_MGMTPREF_5           = 5,  /**< Preference 5 */
  OPEN_USMDB_UNITMGR_MGMTPREF_6           = 6,  /**< Preference 6 */
  OPEN_USMDB_UNITMGR_MGMTPREF_7           = 7,  /**< Preference 7 */
  OPEN_USMDB_UNITMGR_MGMTPREF_8           = 8,  /**< Preference 8 */
  OPEN_USMDB_UNITMGR_MGMTPREF_9           = 9,  /**< Preference 9 */
  OPEN_USMDB_UNITMGR_MGMTPREF_10          = 10, /**< Preference 10 */
  OPEN_USMDB_UNITMGR_MGMTPREF_11          = 11, /**< Preference 11 */
  OPEN_USMDB_UNITMGR_MGMTPREF_12          = 12, /**< Preference 12 */
  OPEN_USMDB_UNITMGR_MGMTPREF_13          = 13, /**< Preference 13 */
  OPEN_USMDB_UNITMGR_MGMTPREF_14          = 14, /**< Preference 14 */
  OPEN_USMDB_UNITMGR_MGMTPREF_15          = 15, /**< Preference 15 */
  OPEN_USMDB_UNITMGR_MGMTFUNC_UNASSIGNED  = 16  /**< Preference 16 */
} OPEN_USMDB_UNITMGR_MGMTPREF_t;

/** Stack Firmware Synchronization Status */
typedef enum
{
  OPEN_SFS_SWITCH_STATUS_NO_ACTION    = 1,  /**< No Action */
  OPEN_SFS_SWITCH_STATUS_IN_PROGRESS  = 2   /**< In Progress */
} OPEN_UNIT_MGR_SFS_SWITCH_STATUS_t;

/** Stack Firmware Synchronization Last Attempt Status */
typedef enum
{
  OPEN_SFS_NONE                             = 1,  /**< None */
  OPEN_SFS_SUCCESS                          = 2,  /**< Success */
  OPEN_SFS_FAIL                             = 3,  /**< Fail */
  OPEN_SFS_MIN_BOOTCODE_VERSION_NOT_PRESENT = 4   /**< Min Bootcode Version Not Present */
} OPEN_SFS_LAST_ATTEMPT_STATUS_t;

/**< stack speed config */
typedef enum
{
  OPEN_STACK_SPEED_10G = 0, /**< 10G stack speed */
  OPEN_STACK_SPEED_25G,     /**< 25G stack speed */
  OPEN_STACK_SPEED_40G,     /**< 40G stack speed */
  OPEN_STACK_SPEED_50G,     /**< 50G stack speed */
  OPEN_STACK_SPEED_100G,    /**< 100G stack speed */
}OPEN_STACK_SPEED_t;

typedef enum
{
  OPEN_UNITMGR_ROOTFS_SYNCED = 1,  /**< Rootfs synced */
  OPEN_UNITMGR_ROOTFS_OUT_OF_SYNC, /**< Rootfs out of sync */
  OPEN_UNITMGR_ROOTFS_COPIED       /**< Rootfs copied */
} OPEN_UNIT_MGR_ROOTFS_STATUS_t;

typedef enum
{
  OPEN_UNIT_STATUS_SAMPLEMODE_NONE,              /**< Samplemode none */
  OPEN_UNIT_STATUS_SAMPLEMODE_CUMULATIVESUMMING, /**< Samplemode cumulative summing */
  OPEN_UNIT_STATUS_SAMPLEMODE_HISTORY,           /**< Samplemode history */
  OPEN_UNIT_STATUS_SAMPLEMODE_MAX,               /**< Samplemode max */
} openUnitStatusSampleMode_t;

typedef enum
{
  /**< Running configuration has not yet been copied to the backup unit. */
  OPEN_RUN_CFG_NONE = 0,
  /**< Running configuration has been copied, but it is out of date */
  OPEN_RUN_CFG_STALE,
  /**< Copy of running configuration is in progress */
  OPEN_RUN_CFG_IN_PROGRESS,
  /**< Running configuration on backup is the same as on mgmt unit */
  OPEN_RUN_CFG_CURRENT,
  /**< No backup unit has been elected */
  OPEN_RUN_CFG_NO_BACKUP
} OPEN_RUN_CFG_STATUS_t;

typedef enum
{
  /**< The switch re-booted. This could have been caused by a power cycle
      or an administrative "Reload" command. */
  OPEN_STARTUP_POWER_ON = 0,
  /**< The administrator issued the "move-management" command */
  OPEN_STARTUP_COLD_ADMIN_MOVE,
  /**< The administrator issued the "initiate failover" command for the
      stand-by manager to take over. */
  OPEN_STARTUP_WARM_ADMIN_MOVE,
  /**< The primary management unit restarted because of a failure, and the
      system executed a nonstop forwarding failover. */
  OPEN_STARTUP_AUTO_WARM,
  /**< The system switched from the active manager to the backup manager and
      was unable to maintain nonstop forwarding. This is usually caused by
      multiple failures occurring close together or because NSF is operationally
      disabled. */
  OPEN_STARTUP_AUTO_COLD
} OPEN_LAST_STARTUP_REASON_t;

#define OPEN_UNITMGR_ROOTFS_VER_SIZE    33
#define OPEN_UNITMGR_ROOTFS_VER_LENGTH  32
#define OPEN_UNITMGR_ROOTFS_SNAPSHOT_FILE  "/mnt/fastpath/upper1.tgz"
#define OPEN_UNITMGR_ROOTFS_SNAPSHOT_VER_FILE  "/mnt/fastpath/upper1.md5"
#define OPEN_CDA_FILE_NAME_LENGTH  128
#define OPEN_UNITMGR_ROOTFS_COMPAT_FILE "/etc/bootstrap-distro-compat"
#define OPEN_UNITMGR_ROOTFS_SNAPSHOT_FILE_NAME_LENGTH 64
#define OPEN_UNITMGR_ROOTFS_SNAPSHOT_VERSION_FILE_NAME_LENGTH 64
#define OPEN_UNITMGR_LEGACY_ROOTFS_COMPAT_STRING "1"
#define OPEN_UNITMGR_ROOTFS_COMPAT_STRING_LENGTH 32
#define OPEN_MAX_UNITS_PER_STACK                 12
#define OPEN_ALL_UNITS                           (OPEN_MAX_UNITS_PER_STACK + 1)

typedef struct openUnitStatusInfo_s
{
  uint32_t unit;
  OPEN_BOOL_t isValid;
  uint32_t avgTimeOffset;
  uint32_t minTimeOffset;
  uint32_t maxTimeOffset;
  uint32_t curTimeOffset;
  uint32_t lostMsgs;
} openUnitStatusInfo_t;

typedef struct openUnitStatusUnitInfo_s
{
  uint32_t unit;
  openUnitStatusSampleMode_t sampleMode;
  uint32_t sampleSize;
  openUnitStatusInfo_t status_info[OPEN_MAX_UNITS_PER_STACK];
} openUnitStatusUnitInfo_t;

/*********************************************************************
* @purpose  Given a unit, gets the next member of the current stack
*
* @param    client_handle    @b{(input)} Client handle from
*                                        registration API
* @param    startUnit       @b{(input)}  Initial unit number
* @param    nextUnit        @b{(output)} Pointer to return next unit in stack
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStackMemberGetNext(openapiClientHandle_t *client_handle,
                                              uint32_t              startUnit,
                                              uint32_t              *nextUnit);

/*********************************************************************
* @purpose  Get the manager's unit number
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(output)} Pointer to manager's unit number
*
* @returns  OPEN_E_NONE      if issued successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrNumberGet(openapiClientHandle_t *client_handle,
                                     uint32_t              *unit);

/*********************************************************************
* @purpose  Given a unit, gets the unit's serial number
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  unit number
* @param    serialNumber     @b{(output)} Pointer to unit's serial number
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_UNITMGR_SERIAL_NUMBER_SIZE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrSerialNumberGet(openapiClientHandle_t *client_handle,
                                           uint32_t              unit,
                                           open_buffdesc         *serialNumber);

/*********************************************************************
* @purpose  Given a unit, gets the service tag
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    serviceTag       @b{(output)} Pointer to unit's service tag.
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_SYSMGMT_SERVICE_TAG_MAX_LENGTH
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrServiceTagGet(openapiClientHandle_t *client_handle,
                                         uint32_t              unit,
                                         open_buffdesc         *serviceTag);

/*********************************************************************
* @purpose  Given a unit, gets the unit's model identifier string
*           assigned by HPC(Hardware Platform Control)
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    modelId          @b{(output)} Pointer to model id string storage
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_SYSAPI_HPC_UNIT_MODEL_SIZE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrPreconfigModelIdGet(openapiClientHandle_t *client_handle,
                                               uint32_t              unit,
                                               open_buffdesc         *modelId);

/*********************************************************************
* @purpose  Given a unit, verify that it's within the valid range.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit number
* @param    -                             Minimum value 1  
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    isValid          @b{(output)} Returns @ref OPEN_TRUE if unit number
*                                         is valid, else @ref OPEN_FALSE if not
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitIsValid(openapiClientHandle_t *client_handle,
                                       uint32_t              unit,
                                       OPEN_BOOL_t           *isValid);

/*********************************************************************
* @purpose  Gets the maximum stacking unit number (largest possible
*           number of units in the stack).
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(output)} Unit number
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes    The default unit number is returned for non-stacking builds.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitMaxGet(openapiClientHandle_t *client_handle,
                                      uint32_t              *unit);

/*********************************************************************
* @purpose  Given a unit, gets the unit's status.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    status           @b{(output)} Unit status
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitStatusGet(openapiClientHandle_t       *client_handle,
                                         uint32_t                    unit,
                                         OPEN_USMDB_UNITMGR_STATUS_t *status);

/*********************************************************************
* @purpose  Given a unit, gets the unit's management status.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    mgmtStatus       @b{(output)} Management status
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrMgmtUnitStatusGet(openapiClientHandle_t          *client_handle,
                                             uint32_t                       unit,
                                             OPEN_USMDB_UNITMGR_MGMT_FLAG_t *mgmtStatus);

/*********************************************************************
* @purpose  Given a unit, gets the unit's model identifier string
*           assigned by HPC and pre-configured model identifier string.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    modelId          @b{(output)} Model id
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_SYSAPI_HPC_UNIT_MODEL_SIZE
* @param    preconfigModelId @b{(output)} Pre-configured model id
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_SYSAPI_HPC_UNIT_MODEL_SIZE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrModelIdentifierGet(openapiClientHandle_t *client_handle,
                                              uint32_t              unit,
                                              open_buffdesc         *modelId,
                                              open_buffdesc         *preconfigModelId);

/*********************************************************************
* @purpose  Given a unit, gets the unit's version of code in flash.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    codeVersion      @b{(output)} Code version
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrDetectCodeVerFlashGet(openapiClientHandle_t *client_handle,
                                                 uint32_t              unit,
                                                 open_revision_data_t  *codeVersion);

/*********************************************************************
* @purpose  Given a unit, gets the unit's running version of code.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    codeVersion      @b{(output)} Code version
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrDetectCodeVerRunningGet(openapiClientHandle_t *client_handle,
                                                   uint32_t              unit,
                                                   open_revision_data_t  *codeVersion);

/*********************************************************************
* @purpose  Given a unit, gets the unit's hardware and admin management preference.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    hwPref           @b{(output)} Hardware preference
* @param    adminPref        @b{(output)} Admin preference
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrMgmtPrefGet(openapiClientHandle_t         *client_handle,
                                       uint32_t                      unit,
                                       OPEN_USMDB_UNITMGR_MGMTPREF_t *hwPref,
                                       OPEN_USMDB_UNITMGR_MGMTPREF_t *adminPref);

/*********************************************************************
* @purpose  Given a unit, gets the unit's type and index.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    unitType         @b{(output)} Unit type of hardware 
*                                         used to determine stacking compatibility
* @param    unitTypeIndex    @b{(output)} Unit type index
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitTypeGet(openapiClientHandle_t *client_handle,
                                       uint32_t              unit,
                                       uint32_t              *unitType,
                                       uint32_t              *unitTypeIndex);

/*********************************************************************
* @purpose  Get unit description that corresponds to the index provided.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unitTypeIndex    @b{(input)}  Unit index
* @param    unitDescr        @b{(output)} Unit description
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_SYSAPI_HPC_UNIT_DESCRIPTION_SIZE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrSupportedDescriptionGet(openapiClientHandle_t *client_handle,
                                                   uint32_t              unitTypeIndex,
                                                   open_buffdesc         *unitDescr);

/*********************************************************************
* @purpose  Get stack firmware synchronization status of member unit.
*
* @param    client_handle       @b{(input)}  Client handle from
*                                            registration API
* @param    unit                @b{(input)}  Unit number
* @param    firmwareSyncStatus  @b{(output)} Firmware synchronization status
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStackFirmwareSyncMemberStatusGet(openapiClientHandle_t             *client_handle,
                                                            uint32_t                          unit,
                                                            OPEN_UNIT_MGR_SFS_SWITCH_STATUS_t *firmwareSyncStatus);

/*********************************************************************
* @purpose  Get stack firmware synchronization status.
*
* @param    client_handle       @b{(input)}  Client handle from
*                                            registration API
* @param    firmwareSyncStatus  @b{(output)} Firmware synchronization status
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
* @returns  OPEN_E_UNAVAIL   if feature is not supported 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStackFirmwareSyncStatusGet(openapiClientHandle_t             *client_handle,
                                                      OPEN_UNIT_MGR_SFS_SWITCH_STATUS_t *firmwareSyncStatus);

/*********************************************************************
* @purpose  Get stack firmware synchronization last attempt status.
*
* @param    client_handle     @b{(input)}  Client handle from
*                                          registration API
* @param    unit              @b{(input)}  Unit number
* @param    lastAttemptStatus @b{(output)} Last attempt status
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
* @returns  OPEN_E_UNAVAIL   if feature is not supported 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStackFirmwareSyncMemberLastAttemptStatusGet(openapiClientHandle_t *client_handle,
                                                                       uint32_t              unit,
                                                                       uint32_t              *lastAttemptStatus);

/*********************************************************************
* @purpose  Gets active template id of the stack unit.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    templateId       @b{(output)} Template id
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
* @returns  OPEN_E_UNAVAIL   if feature is not supported 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiStmUnitActiveTemplateIdGet(openapiClientHandle_t *client_handle,
                                               uint32_t              unit,
                                               uint32_t              *templateId);

/*********************************************************************
* @purpose  Gets description of specified template id.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    templateId       @b{(input)}  Template id
* @param    templateDescr    @b{(output)} Template description
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
* @returns  OPEN_E_UNAVAIL   if feature is not supported 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiStmTemplateDescriptionGet(openapiClientHandle_t *client_handle,
                                              uint32_t              templateId,
                                              open_buffdesc         *templateDescr);

/*********************************************************************
* @purpose  Given a unit, gets the unit's up time in seconds.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    upTime           @b{(output)} Up time
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitUpTimeGet(openapiClientHandle_t *client_handle,
                                         uint32_t              unit,
                                         uint32_t              *upTime);

/*********************************************************************
* @purpose  Get unit type identifier that corresponds to
*           the index provided
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unitTypeIndex    @b{(input)}  Unit type index
* @param    unitType         @b{(output)} Unit type of hardware 
*                                         used to determine stacking compatibility
*
* @returns  OPEN_E_NONE      if issued successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrSupportedUnitIdGet(openapiClientHandle_t *client_handle,
                                              uint32_t              unitTypeIndex,
                                              uint32_t              *unitType);

/*********************************************************************
* @purpose  Get unit type identifier that corresponds to
*           the next index
*
* @param    client_handle     @b{(input)}  Client handle from
*                                          registration API
* @param    unitTypeIndex     @b{(input)}  Unit type index
* @param    nextUnitTypeIndex @b{(output)} Unit type index
* @param    unitType         @b{(output)} Unit type of hardware 
*                                         used to determine stacking compatibility
*
* @returns  OPEN_E_NONE      if issued successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrSupportedUnitIdNextGet(openapiClientHandle_t *client_handle,
                                                  uint32_t              unitTypeIndex,
                                                  uint32_t              *nextUnitTypeIndex,
                                                  uint32_t              *unitType);

/*********************************************************************
* @purpose  Create a new unit record for unit
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    unitTypeIndex    @b{(input)}  Unit type index
* @param    synchMode        @b{(input)}  Synchronous mode
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if create operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes    When it is set to 1, registrants within the switch 
*           are notified of card insertion(s) during the execution 
*           of the API call, otherwise they are notified afterwards (asynchronously).
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitCreate(openapiClientHandle_t *client_handle,
                                      uint32_t              unit,
                                      uint32_t              unitTypeIndex,
                                      OPEN_BOOL_t           synchMode);

/*********************************************************************
* @purpose  Remove a unit only if it is not an active member of
*           the stack
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if delete operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_ERROR     if unit is an active member of the stack
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitDelete(openapiClientHandle_t *client_handle,
                                      uint32_t              unit);

/*********************************************************************
* @purpose  Transfers the management function from the current unit
*           to another
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    targetUnit       @b{(input)}  Target unit
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_UNAVAIL   if the operation is not supported/unavailable 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitTransfer(openapiClientHandle_t *client_handle,
                                        uint32_t              targetUnit);

/*********************************************************************
* @purpose  Get QOS configuration for all front-panel stacking ports
*           stack port
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    fpsQosMode       @b{(output)} Returns @ref OPEN_ENABLE if QOS mode
*                                         is enabled, else @ref OPEN_DISABLE if not
*
* @returns  OPEN_E_NONE      if issued successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_UNAVAIL   if feature is not supported 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiSpmFpsConfigQosModeGet(openapiClientHandle_t *client_handle,
                                           OPEN_CONTROL_t        *fpsQosMode);

/*********************************************************************
* @purpose  Set QOS configuration for all front-panel stacking ports
*           stack port
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    fpsQosMode       @b{(input)}  Front panel stacking QOS mode
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if set operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_UNAVAIL   if feature is not supported 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiSpmFpsConfigQosModeSet(openapiClientHandle_t *client_handle,
                                           OPEN_CONTROL_t        fpsQosMode);

/*********************************************************************
* @purpose  Gets the configured and standby unit number and admin mode in the stack.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    configUnit       @b{(output)} Configured unit
* @param    standbyUnit      @b{(output)} Standby unit
* @param    standbyAdminMode @b{(output)} Returns @ref OPEN_TRUE if admin mode
*                                         is enabled, else @ref OPEN_FALSE if not
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStandbyGet(openapiClientHandle_t *client_handle,
                                      uint32_t              *configUnit,
                                      uint32_t              *standbyUnit,
                                      OPEN_BOOL_t           *standbyAdminMode);

/*********************************************************************
* @purpose  Sets the unit as standby in the stack
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    standbyUnit      @b{(input)}  Standby unit
* @param    standbyAdminMode @b{(input)}  Standby admin mode
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if set operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStandbySet(openapiClientHandle_t *client_handle,
                                      uint32_t              standbyUnit,
                                      OPEN_BOOL_t           standbyAdminMode);

/*********************************************************************
* @purpose  Get stacking mode configuration for a front-panel
*           stack port
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    slot             @b{(input)}  Slot
* @param    port             @b{(input)}  Port
* @param    stackPortMode    @b{(output)} Returns @ref OPEN_TRUE if admin mode
*                                         is enabled, else @ref OPEN_FALSE if not
*
* @returns  OPEN_E_NONE      if issued successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_UNAVAIL   if feature is not supported 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiSpmFpsConfigStackingModeGet(openapiClientHandle_t *client_handle,
                                                uint32_t              unit,
                                                uint32_t              slot,
                                                uint32_t              port,
                                                OPEN_BOOL_t           *stackPortMode);

/*********************************************************************
* @purpose  Set stacking mode configuration for a front-panel
*           stack port
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    slot             @b{(input)}  Slot
* @param    port             @b{(input)}  Port
* @param    stackPortMode    @b{(input)}  Enable front-panel stacking mode
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if set operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_UNAVAIL   if feature is not supported 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiSpmFpsConfigStackingModeSet(openapiClientHandle_t *client_handle,
                                                uint32_t              unit,
                                                uint32_t              slot,
                                                uint32_t              port,
                                                OPEN_BOOL_t           stackPortMode);



/*********************************************************************
* @purpose  Given a unit and slot number, verify whether the slot exists in the unit.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit number
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    slot             @b{(input)}  Slot ID of the slot port
* @param    -                             Maximum value @ref OPENAPI_L7_MAX_SLOTS_PER_UNIT
* @param    isValid          @b{(output)} Returns @ref OPEN_TRUE if slot number
*                                         exists in the unit, else @ref OPEN_FALSE if not
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiSlotGet(openapiClientHandle_t *client_handle,
                                       uint32_t unit,
                                       uint32_t slot,
                                       OPEN_BOOL_t *isValid);

/**************************************************************************
* @purpose  Given a unit number, it retrieves all slot information from the unit descriptor database.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b((input))  unit type identifier
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    count            @b((output)) The number of slots in array
* @param    bufd             @b((output)) The slots array. The size of each slot is uint32_t.
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_L7_MAX_PHYSICAL_SLOTS_PER_UNIT
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_FAIL      if get operation has failed
* 
* @supported-in-version OpEN API Version: 1.10
*
* @end
*************************************************************************/
open_error_t openapiUnitDbEntrySlotsGet(openapiClientHandle_t *client_handle,
                                        uint32_t unit,
                                        uint32_t *count,
                                        open_buffdesc *bufd);

/*********************************************************************
* @purpose  Given a unit and slot number, verify whether the slot exists in the unit.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit ID of the unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    slot             @b{(input)}  Slot ID of the slot port
* @param    -                             Maximum value @ref OPENAPI_L7_MAX_SLOTS_PER_UNIT
* @param    isFull           @b{(output)} Returns @ref OPEN_TRUE if slot is full
*                                         else @ref OPEN_FALSE if not
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiSlotIsFullGet(openapiClientHandle_t *client_handle,
                                  uint32_t unit,
                                  uint32_t slot,
                                  OPEN_BOOL_t *isFull);

/*********************************************************************
* @purpose  Given a unit and slot number, gets admin mode of slot.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit ID of the unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    slot             @b{(input)}  Slot ID of the slot port
* @param    -                             Maximum value @ref OPENAPI_L7_MAX_SLOTS_PER_UNIT
* @param    adminMode        @b{(output)} Configured Administrative mode
* @param    -                             Returns @ref OPEN_ENABLE if adminMode is enabled
*                                         else @ref OPEN_DISABLE if not
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiSlotAdminModeGet(openapiClientHandle_t *client_handle,
                                     uint32_t unit,
                                     uint32_t slot,
                                     OPEN_CONTROL_t *adminMode);

/*********************************************************************
* @purpose  Given a unit,  slot number and admin mode, configures it on device.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit ID of the unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    slot             @b{(input)}  Slot ID of the slot port
* @param    -                             Maximum value @ref OPENAPI_L7_MAX_SLOTS_PER_UNIT
* @param    adminMode        @b{(input)}  Configured Administrative mode
* @param    -                             Configures enabled if adminMode is @ref OPEN_ENABLE
* @param    -                             Configures disabled if adminMode is @ref OPEN_DISABLE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiSlotAdminModeSet(openapiClientHandle_t *client_handle,
                                     uint32_t unit,
                                     uint32_t slot,
                                     OPEN_CONTROL_t adminMode);

/*********************************************************************
* @purpose  Given a unit and slot number, gets power mode of slot.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit ID of the unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    slot             @b{(input)}  Slot ID of the slot port
* @param    -                             Maximum value @ref OPENAPI_L7_MAX_SLOTS_PER_UNIT
* @param    powerMode        @b{(output)} Configured Power mode
* @param    -                             Returns @ref OPEN_ENABLE if powerMode is enabled
*                                         else @ref OPEN_DISABLE if not
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiSlotPowerModeGet(openapiClientHandle_t *client_handle,
                                     uint32_t unit,
                                     uint32_t slot,
                                     OPEN_CONTROL_t *powerMode);

/*********************************************************************
* @purpose  Given a unit,  slot number and power mode, configures it on device.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit ID of the unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    slot             @b{(input)}  Slot ID of the slot port
* @param    -                             Maximum value @ref OPENAPI_L7_MAX_SLOTS_PER_UNIT
* @param    powerMode        @b{(input)}  Configured power mode
* @param    -                             Configures enabled if powerMode is @ref OPEN_ENABLE
* @param    -                             Configures disabled if powerMode is @ref OPEN_DISABLE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiSlotPowerModeSet(openapiClientHandle_t *client_handle,
                                     uint32_t unit,
                                     uint32_t slot,
                                     OPEN_CONTROL_t powerMode);

/*********************************************************************
* @purpose  Given a unit and slot number, gets configured card type and inserted card type.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit ID of the unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    slot             @b{(input)}  Slot ID of the slot port
* @param    -                             Maximum value @ref OPENAPI_L7_MAX_SLOTS_PER_UNIT
* @param    cardIdConfig     @b{(output)} Configured card type
* @param    cardIdInsert     @b{(output)} Inserted card type
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiCardTypeGet(openapiClientHandle_t *client_handle,
                                uint32_t unit,
                                uint32_t slot,
                                uint32_t *cardIdConfig,
                                uint32_t *cardIdInsert);

/*********************************************************************
* @purpose  Given card index, gets model Identifier.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    cardId           @b{(input)}  Card Index.
* @param    bufd             @b{(output)} Buffer to store Card Model Id
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_SYSAPI_HPC_CARD_MODEL_SIZE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiCardSupportedCardModelIdGet(openapiClientHandle_t *client_handle,
                                                uint32_t cardId,
                                                open_buffdesc *bufd);

/*********************************************************************
* @purpose  Given card index, gets card description.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    cardId           @b{(input)}  Card Index.
* @param    bufd             @b{(output)} Buffer to store Card description
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_SYSAPI_HPC_CARD_DESCRIPTION_SIZE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiCardSupportedCardDescriptionGet(openapiClientHandle_t *client_handle,
                                                    uint32_t cardId,
                                                    open_buffdesc *bufd);

/*********************************************************************
* @purpose  Given a unit and slot number, verify whether it has pluggable indicator.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit ID of the unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    slot             @b{(input)}  Slot ID of the slot port
* @param    -                             Maximum value @ref OPENAPI_L7_MAX_SLOTS_PER_UNIT
* @param    isPluggable      @b{(output)} Returns @ref OPEN_TRUE if card can be plugged into slot
*                                         else @ref OPEN_FALSE if not
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiSlotSupportedPluggableGet(openapiClientHandle_t *client_handle,
                                              uint32_t unit,
                                              uint32_t slot,
                                              OPEN_BOOL_t *isPluggable);

/*********************************************************************
* @purpose  Given a unit and slot number, verify whether it powerdown indicator.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit ID of the unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    slot             @b{(input)}  Slot ID of the slot port
* @param    -                             Maximum value @ref OPENAPI_L7_MAX_SLOTS_PER_UNIT
* @param    isPowerdown      @b{(output)} Returns @ref OPEN_TRUE if power down on slot
*                                         else @ref OPEN_FALSE if not
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiSlotSupportedPowerdownGet(openapiClientHandle_t *client_handle,
                                              uint32_t unit,
                                              uint32_t slot,
                                              OPEN_BOOL_t *isPowerdown);

/*********************************************************************
* @purpose  Given a unit and filename, transfer file from current unit
*           to the specified unit
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  The Unit ID of destination unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
*                                         Use @ref OPENAPI_CDA_ALL_UNITS 
*                                         to transfer to all member units
* @param    srcFileName      @b{(input)}  Filename to be transferred to destination unit
* *           -                           Buffer allocated size must be at max
* *           -                           @ref OPENAPI_CDA_FILE_NAME_LENGTH
* @param    dstFileName      @b{(input)}  Filename of transferred file on destination unit
*           -                             Buffer allocated size must be at max
*           -                             @ref OPENAPI_CDA_FILE_NAME_LENGTH
*
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiCdaTransferFile(openapiClientHandle_t *client_handle,
                                    uint32_t unit,
                                    open_buffdesc *srcFileName,
                                    open_buffdesc *dstFileName);

/*********************************************************************
* @purpose  Given a unit, get the Debian Rootfs changes snapshot version
*           stored
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  The Unit ID of destination unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    version          @b{(output)} MD5 Checksum of changes snapshot
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_UNITMGR_ROOTFS_VER_SIZE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrDebianRootfsVerGet(openapiClientHandle_t *client_handle,
                                              uint32_t unit,
                                              open_buffdesc *version);

/*********************************************************************
* @purpose  Given a unit, set the Debian Rootfs changes snapshot version
*           stored
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  The Unit ID of destination unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    version          @b{(input)} MD5 Checksum of changes snapshot
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_UNITMGR_ROOTFS_VER_SIZE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrDebianRootfsVerSet(openapiClientHandle_t *client_handle,
                                              uint32_t unit,
                                              open_buffdesc *version);

/*********************************************************************
* @purpose  Given a unit, get the Debian Rootfs changes snapshot version
*           currently in use
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  The Unit ID of destination unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    version          @b{(output)} MD5 Checksum of changes snapshot
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_UNITMGR_ROOTFS_VER_SIZE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrDebianRootfsOperVerGet(openapiClientHandle_t *client_handle,
                                                  uint32_t unit,
                                                  open_buffdesc *version);

/*********************************************************************
* @purpose  Given a unit, set the Debian Rootfs changes snapshot version
*           currently in use
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  The Unit ID of destination unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    version          @b{(input)} MD5 Checksum of changes snapshot
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_UNITMGR_ROOTFS_VER_SIZE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrDebianRootfsOperVerSet(openapiClientHandle_t *client_handle,
                                                  uint32_t unit,
                                                  open_buffdesc *version);

/*********************************************************************
* @purpose  Given a unit, reboot the corresponding switch 
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  The Unit ID of destination unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrResetUnit(openapiClientHandle_t *client_handle,
                                     uint32_t unit);

/*********************************************************************
* @purpose  Get the Debian Rootfs changes snapshot file name
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  The Unit ID of destination unit
*           -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    filename         @b{(output)}  File name of the rootfs changes snapshot
*           -                             Buffer allocated size must be at least
*           -                             @ref OPEN_UNITMGR_ROOTFS_SNAPSHOT_FILE_NAME_LENGTH
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrDebianRootfsFileNameGet(openapiClientHandle_t *client_handle,
                                                  uint32_t unit,
                                                  open_buffdesc *filename);

/*********************************************************************
* @purpose  Get the Debian Rootfs changes snapshot version file name
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  The Unit ID of destination unit
*           -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    filename         @b{(output)}  File name of the rootfs changes snapshot version
*           -                             Buffer allocated size must be at least
*           -                             @ref OPEN_UNITMGR_ROOTFS_SNAPSHOT_FILE_VERSION_NAME_LENGTH
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrDebianRootfsVersionFileNameGet(openapiClientHandle_t *client_handle,
                                                  uint32_t unit,
                                                  open_buffdesc *filename);

/*********************************************************************
* @purpose  Get stacking running speed configuration for a front-panel
*           stack port.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API.
* @param    unit             @b{(input)}  Unit number.
* @param    slot             @b{(input)}  Slot.
* @param    port             @b{(input)}  Port.
* @param    stackPortSpeed   @b{(output)} Operational stack port speed.
*
* @returns  OPEN_E_NONE      if issued successful.
* @returns  OPEN_E_FAIL      if get operation has failed.
* @returns  OPEN_E_PARAM     if invalid parameter is supplied.
* @returns  OPEN_E_UNAVAIL   if feature is not supported.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26 
*
* @end
*********************************************************************/
open_error_t openapiSpmFpsStackingRunningSpeedGet(openapiClientHandle_t *client_handle,
                                                  uint32_t              unit,
                                                  uint32_t              slot,
                                                  uint32_t              port,
                                                  OPEN_STACK_SPEED_t    *stackPortSpeed);

/*********************************************************************
* @purpose  Get stacking speed configuration for a front-panel
*           stack port.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API.
* @param    unit             @b{(input)}  Unit number.
* @param    slot             @b{(input)}  Slot.
* @param    port             @b{(input)}  Port.
* @param    stackPortSpeed   @b{(output)} Stack port speed.
*
* @returns  OPEN_E_NONE      if issued successful.
* @returns  OPEN_E_FAIL      if get operation has failed.
* @returns  OPEN_E_PARAM     if invalid parameter is supplied.
* @returns  OPEN_E_UNAVAIL   if feature is not supported.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26 
*
* @end
*********************************************************************/
open_error_t openapiSpmFpsStackingConfigSpeedGet(openapiClientHandle_t *client_handle,
                                                 uint32_t              unit,
                                                 uint32_t              slot,
                                                 uint32_t              port,
                                                 OPEN_STACK_SPEED_t    *stackPortSpeed);

/*********************************************************************
* @purpose  Set stacking speed configuration for a front-panel
*           stack port.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API.
* @param    unit             @b{(input)}  Unit number.
* @param    slot             @b{(input)}  Slot.
* @param    port             @b{(input)}  Port.
* @param    stackPortSpeed   @b{(input)}  Stack port speed.
*
* @returns  OPEN_E_NONE      if issued successful.
* @returns  OPEN_E_FAIL      if get operation has failed.
* @returns  OPEN_E_PARAM     if invalid parameter is supplied.
* @returns  OPEN_E_UNAVAIL   if feature is not supported.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26 
*
* @end
*********************************************************************/
open_error_t openapiSpmFpsStackingConfigSpeedSet(openapiClientHandle_t *client_handle,
                                                 uint32_t              unit,
                                                 uint32_t              slot,
                                                 uint32_t              port,
                                                 OPEN_STACK_SPEED_t    stackPortSpeed);

/*********************************************************************
* @purpose  Get the unit index of the local unit
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    unitTypeIdIndex  @b{(output)} index into the unit descriptor table
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrSupportedLocalUnitIndexGet(openapiClientHandle_t *client_handle,
                                                      uint32_t *unitTypeIdIndex);

/*********************************************************************
* @purpose  Get the supported unit model identifier for unit index
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    unitTypeIdIndex  @b{(input)}  index into the unit descriptor table
* @param    unitModel        @b{(output)} unit model
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrSupportedModelGet(openapiClientHandle_t *client_handle,
                                             uint32_t unitTypeIdIndex,
                                             open_buffdesc *unitModel);

/*********************************************************************
* @purpose  Get default template id for selected unit type index
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    unitTypeIdIndex    @b{(input)}  index into the unit descriptor table
* @param    defaultTemplateId  @b{(output)} default template
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Unsupported.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrSupportedStmDefaultTemplateIdGet(openapiClientHandle_t *client_handle,
                                                            uint32_t unitTypeIdIndex,
                                                            uint32_t *defaultTemplateId);

/*********************************************************************
* @purpose  Get the management preference of the unit index
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    unitTypeIdIndex  @b{(input)}  index into the unit descriptor table
* @param    unitMgmtPref     @b{(output)} preference
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrSupportedMgmtPrefGet(openapiClientHandle_t *client_handle,
                                                uint32_t unitTypeIdIndex,
                                                uint32_t *unitMgmtPref);

/*********************************************************************
* @purpose  Get the first member of the current stack
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    firstUnit       @b{(output)} first unit number
*
* @returns  OPEN_E_NONE      On success.
* @returns  OPEN_E_FAIL      On failure.
* @returns  OPEN_E_PARAM     Invalid argument.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStackMemberGetFirst(openapiClientHandle_t *client_handle,
                                               uint32_t *firstUnit);

/*********************************************************************
* @purpose  Get the unit system identifier key
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    unit           @b{(input)}  unit number
* @param    key            @b{(output)} mac address key
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitIdKeyGet(openapiClientHandle_t *client_handle,
                                        uint32_t unit, open_buffdesc *key);

/*********************************************************************
* @purpose  Set the admin management preference
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    unit           @b{(input)}  unit number
* @param    adminPref      @b{(input)}  admin preference
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrAdminMgmtPrefSet(openapiClientHandle_t *client_handle,
                                            uint32_t unit,
					    OPEN_USMDB_UNITMGR_MGMTPREF_t adminPref);

/*********************************************************************
* @purpose  Initiate a warm failover to the backup unit 
*
* @param    client_handle  @b{(input)}  client handle from registration API
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiFailoverInitiate(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Administratively enable or disable nonstop forwarding 
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    nsf            @b{(input)} enable or disable
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Unsupported.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrNsfAdminSet(openapiClientHandle_t *client_handle,
                                       OPEN_CONTROL_t nsf);

/*********************************************************************
* @purpose  Get the administrative status of nonstop forwarding 
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    nsf            @b{(output)} enabled or disabled
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Unsupported.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrNsfAdminGet(openapiClientHandle_t *client_handle,
                                       OPEN_CONTROL_t *nsf);

/*********************************************************************
* @purpose  Get the reason for the last restart
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    reason         @b{(output)} reason
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Unsupported.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStartupReasonGet(openapiClientHandle_t *client_handle,
                                            OPEN_LAST_STARTUP_REASON_t *reason);

/*********************************************************************
* @purpose  Convert a startup reason to a string
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    reason         @b{(input)}  reason
* @param    reasonStr      @b{(output)} reason string
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Unsupported.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStartupReasonStrGet(openapiClientHandle_t *client_handle,
                                               OPEN_LAST_STARTUP_REASON_t reason,
                                               open_buffdesc *reasonStr);

/*********************************************************************
* @purpose  Convert a run config status to a string
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    runCfgStatus   @b{(input)}  status
* @param    statusStr      @b{(output)} status string
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Unsupported.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrRunCfgStatusString(openapiClientHandle_t *client_handle,
                                              OPEN_RUN_CFG_STATUS_t runCfgStatus,
                                              open_buffdesc *statusStr);

/*********************************************************************
* @purpose  Get the age of the configuration on the backup unit
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    age            @b{(output)} age
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Unsupported.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrConfigAgeOnBackup(openapiClientHandle_t *client_handle,
                                             uint32_t *age);

/*********************************************************************
* @purpose  Get the status of copying the running config to the backup unit
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    status         @b{(output)} status
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Unsupported.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapixUnitMgrRunCfgStatusGet(openapiClientHandle_t *client_handle,
                                            OPEN_RUN_CFG_STATUS_t *status);

/*********************************************************************
* @purpose  Get how long before the running config will next be
*           copied to the backup unit, if a copy is scheduled.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    wait           @b{(output)} Number of seconds until config will be copied
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Unsupported.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrNextConfigCopyTime(openapiClientHandle_t *client_handle,
                                              uint32_t *wait);

/*********************************************************************
* @purpose  Get this unit's nonstop forwarding support
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    unit           @b{(input)}  unit number
* @param    nsfSupport     @b{(output)} nsf support
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Unsupported.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrNsfSupportGet(openapiClientHandle_t *client_handle,
                                         uint32_t unit,
                                         OPEN_BOOL_t *nsfSupport);

/*********************************************************************
* @purpose  Changes a unit's number
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    unit           @b{(input)}  unit number
* @param    newUnitNumber  @b{(input)}  new unit number
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments Reset will be initiated after changing the number
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitNumberChange(openapiClientHandle_t *client_handle,
                                            uint32_t unit,
                                            uint32_t newUnitNumber);

/*********************************************************************
* @purpose  Erase the startup-config file on other connected units
*
* @param    client_handle  @b{(input)}  client handle from registration API
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrEraseStartupConfig(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose   Erase the factory-defaults file on other connected units
*
* @param    client_handle  @b{(input)}  client handle from registration API
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrEraseFactoryDefaults(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Set unit user description
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    unit           @b{(input)}  unit number
* @param    descr          @b{(output)}  description
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUserDescriptionSet(openapiClientHandle_t *client_handle,
                                              uint32_t unit,
                                              open_buffdesc *descr);

/*********************************************************************
* @purpose  Get unit user description
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    unit           @b{(input)}  unit number
* @param    descr          @b{(output)} description
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUserDescriptionGet(openapiClientHandle_t *client_handle,
                                              uint32_t unit,
                                              open_buffdesc *descr);

/*********************************************************************
* @purpose  Get stack firmware synchronication mode
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    mode           @b{(output)} mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStackFirmwareSyncModeGet(openapiClientHandle_t *client_handle,
                                                    OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Set stack firmware synchronication mode
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    mode           @b{(input)}  mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStackFirmwareSyncModeSet(openapiClientHandle_t *client_handle,
                                                    OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose  Get stack firmware synchronication trap mode
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    mode           @b{(output)} mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStackFirmwareSyncTrapModeGet(openapiClientHandle_t *client_handle,
                                                        OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Set stack firmware synchronication trap mode
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    mode           @b{(input)}  mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStackFirmwareSyncTrapModeSet(openapiClientHandle_t *client_handle,
                                                        OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose  Get stack firmware synchronization downgrade mode
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    mode           @b{(output)} mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStackFirmwareSyncAllowDowngradeModeGet(openapiClientHandle_t *client_handle,
                                                                  OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Set stack firmware synchronization downgrade mode
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    mode           @b{(input)}  mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStackFirmwareSyncAllowDowngradeModeSet(openapiClientHandle_t *client_handle,
                                                                  OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose  Set the mode of sample collection, size of sample
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    sampleMode     @b{(input)}  Mode of sampling
* @param    sampleSize     @b{(input)}  Number of samples
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrSampleModeSet(openapiClientHandle_t *client_handle,
                                         openUnitStatusSampleMode_t sampleMode,
                                         uint32_t sampleSize);

/*********************************************************************
* @purpose  Clear the sampling information of unit status
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    unit           @b{(input)}  unit number
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitStatusClear(openapiClientHandle_t *client_handle,
                                           uint32_t unit);

/*********************************************************************
* @purpose  Get the status information of specified unit
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    unit           @b{(input)}  unit number
* @param    unit_info      @b{(output)} unit status information
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitStatusInfoGet(openapiClientHandle_t *client_handle,
                                            uint32_t unit,
                                            openUnitStatusUnitInfo_t *unit_info);

/*********************************************************************
* @purpose  Reset stack config on a unit/stack
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    unit           @b{(input)}  unit number
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Unsupported.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStackConfigReset(openapiClientHandle_t *client_handle,
                                            uint32_t unit);

/*********************************************************************
* @purpose  Get the unit's stored Debian rootfs sync status
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    unit           @b{(input)}  unit number
* @param    status         @b{(output)} rootfs sync status

* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Unsupported.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrDebianRootfsStatusGet(openapiClientHandle_t *client_handle,
                                                uint32_t unit,
                                                OPEN_UNIT_MGR_ROOTFS_STATUS_t *status);

/*********************************************************************
* @purpose  Erase the DHCP client configuration file on the stand-by unit
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    all            @b{(input)}  all files or specific file
* @param    fileName       @b{(input)}  file to be deleted. When all files
*                                       needs to be deleted then pass the
*                                       directory path as fileName
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrEraseDhcpClientConfigFile(openapiClientHandle_t *client_handle,
                                                     OPEN_BOOL_t all,
                                                     open_buffdesc *fileName);

/*********************************************************************
* @purpose  Clear stack counters for all stack ports.
*
* @param    client_handle  @b{(input)}  client handle from registration API
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiSpmStackPortCountersClear(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  To determine if a given port is stack capable.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    capable        @b{(output)}  capable or not
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_UNAVAIL          Unsupported.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiSpmIsPortStackCapable(openapiClientHandle_t *client_handle,
                                       uint32_t intIfNum, OPEN_BOOL_t *capable);

/*********************************************************************
* @purpose  Set hash mode configuration for all HG trunks.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    hashMode       @b{(input)}  hashing mode(1 to 10)
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_PARAM            Invalid argument.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiStackTrunkHashModeSet(openapiClientHandle_t *client_handle,
                                          uint32_t hashMode);

/*********************************************************************
* @purpose  Get the hash mode configuration for all HG trunks.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    hashMode       @b{(output)} hashing mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiStackTrunkHashModeGet(openapiClientHandle_t *client_handle,
                                          uint32_t *hashMode);

/*********************************************************************
* @purpose  To check if the hash mode is configurable.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    hashMode       @b{(input)}  hashing mode(1 to 10)
* @param    valid          @b{(output)} valid or not
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_PARAM            Invalid argument.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiStackIsTrunkHashModeConfigurable(openapiClientHandle_t *client_handle,
                                                     uint32_t hashMode, OPEN_BOOL_t *valid);

#endif /* OPENAPI_STACKING_H_INCLUDED */
