/**********************************************************************
*
*  Copyright 2019-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_static_filter.h
*
* @purpose   This code implements the OpEN Static Filter API
*
* @component OPEN
*
* @create    06/08/2019
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_STATIC_FILTER_H_INCLUDED
#define OPENAPI_STATIC_FILTER_H_INCLUDED

#include "openapi_common.h" 

/**< Filtering VLAN modes */
typedef enum
{
  OPEN_FILTER_FORWARD_ALL,             /**< Forward all */
  OPEN_FILTER_FORWARD_UNREGISTERED,    /**< Forward unregistered */
  OPEN_FILTER_FILTER_UNREGISTERED      /**< Filter unregistered */
} OPEN_FILTER_VLAN_FILTER_MODE_t;

/*****************************************************************************
 * @purpose   Create a MAC Static Filter.
 *
 * @param     client_handle  @b{(input)} client handle from registration API
 * @param     macAddr        @b{(input)} static MAC address for the filter
 * @param     vlanID         @b{(input)} VLAN ID for the filter
 * 
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 * @returns   OPEN_E_EXISTS           Already exists.
 * @returns   OPEN_E_FULL             Maximum count reached or table full.
 *
 * @notes     Refer OPENAPI_DOT1Q_MIN_VLAN_ID, OPENAPI_DOT1Q_MAX_VLAN_ID for minimum
 *            and maximum values of VLAN ID
 * @notes     MAC address string should be in one of the below formats:
 *            "xx:xx:xx:xx:xx:xx", "xx-xx-xx-xx-xx-xx", "xxxx.xxxx.xxxx" 
 * @notes     macAddr buffer should have 18 bytes memory.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterCreate(openapiClientHandle_t *client_handle,
                                 open_buffdesc *macAddr, uint32_t vlanID);

/*****************************************************************************
 * @purpose   Remove a configured MAC Static Filter.
 *
 * @param     client_handle  @b{(input)} client handle from registration API
 * @param     macAddr        @b{(input)} static MAC address for the filter
 * @param     vlanID         @b{(input)} VLAN ID for the filter
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes     Refer OPENAPI_DOT1Q_MIN_VLAN_ID, OPENAPI_DOT1Q_MAX_VLAN_ID for minimum
 *            and maximum values of VLAN ID
 * @notes     MAC address string should be in one of the below formats:
 *            "xx:xx:xx:xx:xx:xx", "xx-xx-xx-xx-xx-xx", "xxxx.xxxx.xxxx" 
 * @notes     macAddr buffer should have 18 bytes memory.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterRemove(openapiClientHandle_t *client_handle,
                                 open_buffdesc *macAddr, uint32_t vlanID);

/*****************************************************************************
 * @purpose   Sets the static MAC filter admin mode.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     adminMode           @b{(input)}  Static MAC filter admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterAdminModeSet(openapiClientHandle_t *client_handle, 
                                       OPEN_CONTROL_t adminMode);

/*****************************************************************************
 * @purpose   Gets the static MAC filter admin mode.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     adminMode           @b{(output)} Static MAC filter admin mode 
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterAdminModeGet(openapiClientHandle_t *client_handle, 
                                       OPEN_CONTROL_t *adminMode);

/*****************************************************************************
 * @purpose   Apply a source interface list to the static MAC filter.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     macAddr             @b{(input)}  MAC address
 * @param     vlanId              @b{(input)}  VLAN ID
 * @param     srcIntfList         @b{(input)}  List of interfaces specifying source ports
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterSrcIntfListModify(openapiClientHandle_t *client_handle, 
                                            open_buffdesc *macAddr, uint32_t vlanId,
                                            open_buffdesc *srcIntfList);

/*****************************************************************************
 * @purpose   Adds a source interface to the static MAC filter.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     macAddr             @b{(input)}  MAC address
 * @param     vlanId              @b{(input)}  VLAN ID
 * @param     ifNum               @b{(input)}  Interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterSrcIntfAdd(openapiClientHandle_t *client_handle, 
                                     open_buffdesc *macAddr, uint32_t vlanId, 
                                     uint32_t ifNum);

/*****************************************************************************
 * @purpose   Deletes a source interface from the source interface list of the static MAC filter.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     macAddr             @b{(input)}  MAC address
 * @param     vlanId              @b{(input)}  VLAN ID
 * @param     ifNum               @b{(input)}  Interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterSrcIntfDelete(openapiClientHandle_t *client_handle, 
                                        open_buffdesc *macAddr, uint32_t vlanId, 
                                        uint32_t ifNum);

/*****************************************************************************
 * @purpose   Adds a destination interface to the static MAC filter.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     macAddr             @b{(input)}  MAC address
 * @param     vlanId              @b{(input)}  VLAN ID
 * @param     ifNum               @b{(input)}  Interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterDstIntfAdd(openapiClientHandle_t *client_handle, 
                                     open_buffdesc *macAddr, uint32_t vlanId, 
                                     uint32_t ifNum);

/*****************************************************************************
 * @purpose   Deletes a destination interface from the destination interface list
 *            of the static MAC filter.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     macAddr             @b{(input)}  MAC address
 * @param     vlanId              @b{(input)}  VLAN ID
 * @param     ifNum               @b{(input)}  Interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterDstIntfDelete(openapiClientHandle_t *client_handle, 
                                        open_buffdesc *macAddr, uint32_t vlanId, 
                                        uint32_t ifNum);

/*****************************************************************************
 * @purpose   Adds destination interfaces to the static MAC filter.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     macAddr             @b{(input)}  MAC address
 * @param     vlanId              @b{(input)}  VLAN ID
 * @param     ifNum               @b{(input)}  Interface number
 * @param     fwdIntfList         @b{(input)}  List of ports to forward
 * @param     filterIntfList      @b{(input)}  List of ports to forbid
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterDstIntfListAdd(openapiClientHandle_t *client_handle, 
                                         open_buffdesc *macAddr, uint32_t vlanId, 
                                         uint32_t ifNum, open_buffdesc *fwdIntfList, 
                                         open_buffdesc *filterIntfList);

/*****************************************************************************
 * @purpose   Deletes destination interfaces from the static MAC filter.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     macAddr             @b{(input)}  MAC address
 * @param     vlanId              @b{(input)}  VLAN ID
 * @param     ifNum               @b{(input)}  Interface number
 * @param     fwdIntfList         @b{(input)}  List of ports to forward
 * @param     filterIntfList      @b{(input)}  List of ports to forbid
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterDstIntfListDelete(openapiClientHandle_t *client_handle, 
                                            open_buffdesc *macAddr, uint32_t vlanId, 
                                            uint32_t ifNum, 
                                            open_buffdesc *fwdIntfList, 
                                            open_buffdesc *filterIntfList);

/*****************************************************************************
 * @purpose   Gets the first configured MAC static filter.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     macAddr             @b{(output)} MAC address
 * @param     vlanId              @b{(output)} VLAN ID
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterFirstGet(openapiClientHandle_t *client_handle, 
                                   open_buffdesc *macAddr, uint32_t *vlanId);

/*****************************************************************************
 * @purpose   Gets the next configured MAC static filter.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     macAddr             @b{(input)}  MAC address
 * @param     vlanId              @b{(input)}  VLAN ID
 * @param     nextMacAddr         @b{(output)} MAC address of the next filter
 * @param     nextVlanId          @b{(output)} VLAN ID of the next filter
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterNextGet(openapiClientHandle_t *client_handle, 
                                  open_buffdesc *macAddr, uint32_t vlanId, 
                                  open_buffdesc *nextMacAddr, uint32_t *nextVlanId);

/*****************************************************************************
 * @purpose   Gets list of source interfaces for a MAC static filter.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     macAddr             @b{(input)}  MAC address
 * @param     vlanId              @b{(input)}  VLAN ID
 * @param     numOfSrcIntf        @b{(output)} Number of source interfaces
 * @param     srcIntfList         @b{(output)} List of source interfaces
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     srcIntfList buffer should have ((4 * 1024)+1) bytes memory.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterSrcIntfListGet(openapiClientHandle_t *client_handle, 
                                         open_buffdesc *macAddr, uint32_t vlanId, 
                                         uint32_t *numOfSrcIntf, open_buffdesc *srcIntfList);

/*****************************************************************************
 * @purpose   Gets list of destination interfaces for a MAC static filter.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     macAddr             @b{(input)}  MAC address
 * @param     vlanId              @b{(input)}  VLAN ID
 * @param     numOfDstIntf        @b{(output)} Number of destination interfaces
 * @param     dstIntfList         @b{(output)} List of destination interfaces
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     dstIntfList buffer should have ((4 * 1024)+1) bytes memory.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterDstIntfListGet(openapiClientHandle_t *client_handle, 
                                         open_buffdesc *macAddr, uint32_t vlanId, 
                                         uint32_t *numOfDstIntf, 
                                         open_buffdesc *dstIntfList);

/*****************************************************************************
 * @purpose   Gets list of destination filter interfaces for a MAC static filter.
 *
 * @param     client_handle       @b{(input)}  Client handle from registration API
 * @param     macAddr             @b{(input)}  MAC address
 * @param     vlanId              @b{(input)}  VLAN ID
 * @param     numOfDstIntf        @b{(output)} Number of destination filter interfaces
 * @param     dstFiltIntfList     @b{(output)} List of destination filter interfaces
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     dstFiltIntfList buffer should have ((4 * 1024)+1) bytes memory.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterDstFiltIntfListGet(openapiClientHandle_t *client_handle, 
                                             open_buffdesc *macAddr, uint32_t vlanId, 
                                             uint32_t *numOfDstIntf, 
                                             open_buffdesc *dstFiltIntfList);

/*****************************************************************************
 * @purpose   Gets the number of MAC static filters configured.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     count               @b{(output)} Number of MAC static filters configured
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterCountGet(openapiClientHandle_t *client_handle, 
                                   uint32_t *count);

/*****************************************************************************
 * @purpose   Gets the number of total, multicast and unicast MAC static filters configured.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     totalCount          @b{(output)} Total number of MAC static filters configured
 * @param     multicastFiltCount  @b{(output)} Number of multicast MAC static filters configured
 * @param     unicastFiltCount    @b{(output)} Number of unicast MAC static filters configured
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterCountersGet(openapiClientHandle_t *client_handle, 
                                      uint32_t *totalCount, uint32_t *multicastFiltCount, 
                                      uint32_t *unicastFiltCount);

/*****************************************************************************
 * @purpose   Checks to see if a MAC filter is configured or not.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     macAddr             @b{(input)}  MAC address
 * @param     vlanId              @b{(input)}  VLAN ID
 * @param     status              @b{(output)} Status of MAC filter
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterIsConfigured(openapiClientHandle_t *client_handle, 
                                       open_buffdesc *macAddr, uint32_t vlanId, 
                                       OPEN_BOOL_t *status);

/*****************************************************************************
 * @purpose   Gets the buffer size used by filtering component for the bit mask.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     size                @b{(output)} Buffer size for the bit mask
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterSizeGet(openapiClientHandle_t *client_handle, 
                                  uint32_t *size);

/*****************************************************************************
 * @purpose   Checks if a particular interface is present in any MAC filter.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ifNum               @b{(input)}  Interface number
 * @param     status              @b{(output)} Presence of MAC filter in the specified 
 *                                             interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterIsIntfInAnyFilter(openapiClientHandle_t *client_handle, 
                                            uint32_t ifNum, OPEN_BOOL_t *status);

/*****************************************************************************
 * @purpose   Check to see if the port is the appropriate type on which the param can be set.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ifNum               @b{(input)}  Interface number
 * @param     status              @b{(output)} Status of the param can be set on this port type
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterPortParmCanSet(openapiClientHandle_t *client_handle, 
                                         uint32_t ifNum, OPEN_BOOL_t *status);

/*****************************************************************************
 * @purpose   Checks to see the mac address type, unicast or multicast.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     macAddr             @b{(input)}  MAC address
 * @param     status              @b{(output)} OPEN_TRUE  MAC address is of type 
 *                                             multicast OPEN_FALSE  MAC address is 
 *                                             of type unicast
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterCheckMacAddrTypeMulticast(openapiClientHandle_t *client_handle, 
                                                    open_buffdesc *macAddr, OPEN_BOOL_t *status);

/*****************************************************************************
 * @purpose   Set multicast address filtering configuration for all vlans.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     mode                @b{(input)}  Filtering mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterFilteringModeSet(openapiClientHandle_t *client_handle, 
                                           OPEN_FILTER_VLAN_FILTER_MODE_t mode);

/*****************************************************************************
 * @purpose   Sets multicast address filtering configuration for the specified vlan.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     vlanId              @b{(input)}  VLAN ID
 * @param     mode                @b{(input)}  Filtering mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterVlanFilteringModeSet(openapiClientHandle_t *client_handle, 
                                               uint32_t vlanId, 
                                               OPEN_FILTER_VLAN_FILTER_MODE_t mode);

/*****************************************************************************
 * @purpose   Gets multicast address filtering configuration for the specified vlan.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     vlanId              @b{(input)}  VLAN ID
 * @param     mode                @b{(output)} Filtering mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterVlanFilteringModeGet(openapiClientHandle_t *client_handle, 
                                               uint32_t vlanId, 
                                               OPEN_FILTER_VLAN_FILTER_MODE_t *mode);

/*****************************************************************************
 * @purpose   Checks if the interface is valid for filtering.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ifNum               @b{(input)}  Interface number
 * @param     status              @b{(output)} Validity of interface for MAC filtering
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterIsValidIntf(openapiClientHandle_t *client_handle, 
                                      uint32_t ifNum, OPEN_BOOL_t *status);

/*****************************************************************************
 * @purpose   Sets the stream id of an existing macfilter with given MAC address 
 *            and vlan id.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     macAddr             @b{(input)}  MAC address
 * @param     vlanId              @b{(input)}  VLAN ID
 * @param     streamId            @b{(input)}  802.1qci stream-id
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterStreamSet(openapiClientHandle_t *client_handle, 
                                    open_buffdesc *macAddr, uint32_t vlanId, 
                                    uint32_t streamId);

/*****************************************************************************
 * @purpose   Gets the configured stream id.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     macAddr             @b{(input)}  MAC address
 * @param     vlanId              @b{(input)}  VLAN ID
 * @param     streamId            @b{(output)} 802.1qci stream-id
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterStreamGet(openapiClientHandle_t *client_handle, 
                                    open_buffdesc *macAddr, uint32_t vlanId, 
                                    uint32_t *streamId);

#endif /* OPENAPI_STATIC_FILTER_H_INCLUDED */
