/**********************************************************************
*
*  Copyright 2022-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_stm template.h
*
* @purpose   This code implements the OpEN APIs for STM Templates
*
* @component OPEN
*
* @create    11/24/2022
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_STM_TEMPLATE_H_INCLUDED
#define OPENAPI_STM_TEMPLATE_H_INCLUDED

#include "openapi_common.h" 

#define OPEN_STM_TEMPLATE_DESCRIPTION_SIZE 80
#define OPEN_STM_TEMPLATE_NAME_SIZE        16

/*****************************************************************************
 * @purpose   Gets current active Template Id.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     templateId         @b{(output)} STM template ID
 *
 * @returns   OPEN_E_NONE        On success.
 * @returns   OPEN_E_FAIL        On failure.
 * @returns   OPEN_E_PARAM       Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiStmActiveTemplateIdGet(openapiClientHandle_t *client_handle, 
                                           uint32_t *templateId);

/*****************************************************************************
 * @purpose   Sets active template id.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     templateId         @b{(input)}  STM template ID
 *
 * @returns   OPEN_E_NONE        On success.
 * @returns   OPEN_E_FAIL        On failure.
 * @returns   OPEN_E_UNAVAIL     If STM template is not supported.
 * @returns   OPEN_E_EXISTS      If STM template is already configured.
 * @returns   OPEN_E_PARAM       Invalid argument.
* @returns    OPEN_E_INTERNAL    Internal error.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiStmActiveTemplateIdSet(openapiClientHandle_t *client_handle, 
                                           uint32_t templateId);

/*****************************************************************************
 * @purpose   Get the first valid template ID.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     firstTemplateId    @b{(output)} First valid STM template ID
 *
 * @returns   OPEN_E_NONE        On success.
 * @returns   OPEN_E_FAIL        On failure.
 * @returns   OPEN_E_PARAM       Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiStmFirstTemplateIdGet(openapiClientHandle_t *client_handle, 
                                          uint32_t *firstTemplateId);

/*****************************************************************************
 * @purpose   Get the next valid template ID.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     currentTemplateId  @b{(input)}  Current STM template ID
 * @param     nextTemplateId     @b{(output)} Next valid STM template ID
 *
 * @returns   OPEN_E_NONE        On success.
 * @returns   OPEN_E_FAIL        On failure.
 * @returns   OPEN_E_PARAM       Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiStmNextTemplateIdGet(openapiClientHandle_t *client_handle, 
                                         uint32_t currentTemplateId, uint32_t *nextTemplateId);

/*****************************************************************************
 * @purpose   Gets name and description of specific template id
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     templateId         @b{(input)}  Template ID
 * @param     templateName       @b{(output)} Template name
 * @param     templateDesc       @b{(output)} Template Description
 *
 * @returns   OPEN_E_NONE        On success.
 * @returns   OPEN_E_FAIL        On failure.
 * @returns   OPEN_E_PARAM       Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiStmTemplateNameDescriptionGet(openapiClientHandle_t *client_handle, 
                                                  uint32_t templateId, 
                                                  open_buffdesc *templateName, 
                                                  open_buffdesc *templateDesc);

/*****************************************************************************
 * @purpose   Sets(Revert to) Default template.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 *
 * @returns   OPEN_E_NONE        On success.
 * @returns   OPEN_E_FAIL        On failure.
 * @returns   OPEN_E_PARAM       Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiStmTemplateRevertToDefault(openapiClientHandle_t *client_handle);

#endif /* OPENAPI_STM TEMPLATE_H_INCLUDED */
