/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename    syncdb_api.h
*
* @purpose     API for the Synchronization Database
*
* @component 
*
* @comments    
*
* @create      7/5/2012
*
* @end
*
**********************************************************************/
#ifndef SYNCDB_API_H_INCLUDED
#define SYNCDB_API_H_INCLUDED

#include <pthread.h>

/** Maximum size of the message buffer between the SyncDB process and the
 SyncDB Agent. The actual message size may be smaller, and is limited by 
 the transport message size. 
*/ 
#define SYNCDB_AGENT_MAX_MSG_SIZE 8000

/** Maximum size of the text string that uniquely identifies 
 a database table. The size includes the 0 terminator. 
 This value must be divisible by 2. 
*/
#define SYNCDB_TABLE_NAME_SIZE  32

/** Maximum length for string fields used by syncdb. This includes
 client names, file names and various descriptions.
*/
#define SYNCDB_MAX_STR_LEN  256

/** Maximum size of a single data record. In case of AVL trees and Queues
 this is the maximum size of each AVL or Queue node.
*/
#define SYNCDB_RECORD_MAX_SIZE (1024*64)

/* Supported Data Table Types.
*/
#define SYNCDB_TABLE_TYPE_RECORD 0
#define SYNCDB_TABLE_TYPE_STORABLE_RECORD 1
#define SYNCDB_TABLE_TYPE_AVL_TREE 2
#define SYNCDB_TABLE_TYPE_QUEUE 3

/* Return Codes from the syncdb API functions.
*/
#define SYNCDB_OK   (0)    /**< Operation is Successful */
#define SYNCDB_ERROR (-1)        /**< General Error */
#define SYNCDB_MAX_CLIENTS (-2)  /**< Reached maximum number of clients */
#define SYNCDB_DUPNAME (-3) /**< Table exists with duplicate name */
#define SYNCDB_NO_TABLE (-4) /**< Specified table does not exist */
#define SYNCDB_FULL (-5) /**< Specified table is full */
#define SYNCDB_SIZE (-6) /**< Specified record size is invalid for this table. */
#define SYNCDB_NOT_FOUND (-7) /**< Specified record is not in the table. */
#define SYNCDB_SCHEMA_ERROR (-8) /**< Schema validation error */
#define SYNCDB_ENTRY_EXISTS (-9) /**< AVL entry with the specified key already exists */
#define SYNCDB_TIMEOUT (-10) /**< Operation timed out */

/* Data Table Status.
** The status is a bit mask returned by the syncdbTableStatusGet() function. 
** The status is maintained per client.  
*/

/** Specified Table Exist
 If this bit is not set to 1 then all other bits are meaningless.
*/
#define SYNCDB_TABLE_STAT_EXISTS (1 << 0) 

/** Changes pending for this client. (Cleared on Read)
 This bit is set whenever something changes in the table. The bit is cleared 
 when the syncdbTableStatusGet() function is invoked by the client for the table. 
 When a client gets a table change notification, this bit can be used to determine 
 which table has changed. However the client does not need to be registered for 
 change notifications. The bit is set for all clients when the table changes. 
*/ 
#define SYNCDB_TABLE_STAT_CHANGED (1 << 1) 

/** AVL Tree Purged (Cleared on Read)
 This bit indicates that all delete-pending entries have been removed from the 
 AVL tree since the last call to the syncdbTableStatusGet() function. 
 This means that some record delete notifications may have been 
 missed by this client. The client needs to completely 
 resynchronize with the AVL tree if it needs to know the exact 
 content. If the table is not an AVL tree then this bit is 
 always set to 0. 
*/
#define SYNCDB_TABLE_STAT_AVL_TREE_PURGED (1 << 2) 

/** New Table (Cleared on Read)
 This bit indicates that the syncdbTableStatusGet() function is called for 
 the first time since the table has been created. 
 This flag can be used to detect that a table has been deleted and recreated. 
 Normally applications do not delete tables, even when they are restarted. If 
 a table needs to be deleted then all applications using this table should be 
 restarted.  
*/
#define SYNCDB_TABLE_STAT_NEW_TABLE (1 << 3) 

/********  Data Table Creation Flags
** The flags specified on the table create call.
** The flags is a bit mask returned by the syncdbTableStatusGet() function. 
********/
/**
 This table can be stored in a file. 
 When this flag is set the table creation function must be given the schema and 
 the schema size.  
*/ 
#define SYNCDB_TABLE_FLAG_STORABLE (1 << 0)

/** This table is copied to NVRAM when the table-store command is received.
 The flag is allowed only if the SYNCDB_TABLE_FLAGS_STORABLE is
 also set. 
*/
#define SYNCDB_TABLE_FLAG_NVRAM (1 << 1)

/** If this flag is enabled then the syncdb tries to populate the table
 from a file when creating the table. 
 The flag is allowed only if the SYNCDB_TABLE_FLAG_STORABLE is also set. 
*/
#define SYNCDB_TABLE_FLAG_FILE_LOAD (1 << 2)


/** When this flag is set, the syncdbInsert() function returns an error if an
 entry with the specified key already exists. When this flag is not set the 
 old entry is overwriten with the new content. 
*/
#define SYNCDB_TABLE_FLAG_EXISTS (1 << 3)

/** When this flag is set, the content of the table is synchronized with the backup
 manager. On non-stacking platform the flag has no effect. 
 Applications that use SyncDB for the Non Stop Forwarding feature should store 
 the data in the table using the big-endian format.  
*/ 
#define SYNCDB_TABLE_FLAG_NSF (1 << 4)

/** syncdb Client Handle. */
typedef struct 
{
    int cmd_socket;       /**< Socket for sending commands to Syncdb process. */
    int notify_socket;  /**< Socket for waiting on events  from Syncdb process. */
    int client_id;  /**< Unique Syncdb client identifier. */
    unsigned long long seqNum; /**< Sequence number to track the correct response from server. */
    pthread_mutex_t client_lock; /**< Per-client semaphore. */
    unsigned int rxTimeout;  /**< Command socket receive timeout. */
} syncdbClientHandle_t;

/** Data Table Status
*/
typedef struct
{
    char table_name [SYNCDB_TABLE_NAME_SIZE]; /**< Table Name. */

    /** The version of the table. This information can be used by processes
     to decide whether the table is compatible with the process.
    */
    unsigned int table_version;

    /** Flags specified on table creation. SYCNDB_TABLE_FLAG_STORABLE and others...
    */
    unsigned int table_flags;

    /** Bit Mask. See description for SYNCDB_TABLE_STAT_EXISTS and others...
    */
    unsigned int table_status; 

    /** If the table exists then this specifies the table type.
     The types are SYNCDB_TABLE_TYPE_RECORD,... 
    */
    int table_type; 

    /** Number of records in the table. 
     Valid only for Queues and AVL Trees 
    */ 
    int num_elements; 

    /** Number of records that are not in delete-pending state.
     This parameter is valid only for AVL trees.
    */
    int num_non_deleted_elements;
} syncdbDataTableStatus_t;

/** Client Table Status.
*/
typedef struct 
{
    int client_id; /**< syncdb client ID of the client for which to retrieve the status */
    char client_description [SYNCDB_MAX_STR_LEN]; /**< Brief client description. */
    int  client_pid;  /**< Linux Process ID for this client */
    int  num_commands; /**< Total number of commands issued by this client */ 
    int  num_table_change_events; /**< Total number of table-change notifications sent to the client */
    int  num_table_purges; /**< Number of AVL tree purge events sent to the client */
} syncdbClientStatus_t;

/******************************** 
** Storable Record Schema 
********************************/ 

/* The AVL trees and records can be stored in a file system using JSON notation.
** In order to store information in the file system the table creator must tell syncdb how 
** the data is formatted. Also to facilitate data format changes the table creator must 
** tell syncdb the default values of fields that are not present in the file. 
*/

/** The schema for storable records is also defined using JSON format.
 This compiler constant defines the maximum length of the schema. 
 The constant may be changed to be smaller than the maximum record size, but 
 never bigger. 
*/
#define SYNCDB_JSON_MAX_SCHEMA_SIZE  (SYNCDB_RECORD_MAX_SIZE)

/** The syncdb supports the following JSON data types:
 SYNCDB_JSON_NUMBER - A numeric value. Syncdb supports only unsigned integers.
 SYNCDB_JSON_STRING - Zero-terminated array of characters.
 SYNCDB_JSON_ARRAY - Syncdb supports only arrays of bytes. The default value is 0.
  
 SYNCDB Data Type Limitations: 
 The syncdb supports only a subset of JSON data types. 
 The most important limitation is the limited support for 
 arrays. Arrays are difficult to handle during data format 
 migration, so should be avoided when defining storable records. 
 Syncdb does support arrays of bytes. The table creator cannot 
 provide default values for the byte arrays. The default for 
 array content is always zero. 
  
 The syncdb supports only unsigned integer values for SYNCDB_JSON_NUMBER type. 
 The supported integer sizes are 1, 2, 4, and 8 bytes. 
 Floating point numbers and exponential notation is not supported. 
 Signed integers are also not supported. If a signed integer is used in the node 
 definition then the data migration may not work properly. 
 
 The syncdb also does not support "true", "false", and "null" values. These can be represented 
 with an integer. 
*/
typedef enum
{
  SYNCDB_JSON_NUMBER = 1,
  SYNCDB_JSON_STRING = 2,
  SYNCDB_JSON_ARRAY = 3
} syncdbJsonDataType_e;

/** The JSON node definition.
*/
typedef struct syncdbJsonNode_s
{
    syncdbJsonDataType_e data_type;

    /** The unique identifier for this variable. The data_name is
     used to match fields when performing version migration. All 
     names within the element must be unique. A good policy is to 
     use the fully qualified C field name, such as "element.key.k1" 
     There is no limit on the size of the name field, other than the 
     overall schema size limit. 
    */
    char *data_name;

    /** Offset of this element from the beginning of the data buffer.
     Note that if an element is a member of a structure within another 
     structure then the offset of the nested structure needs to be added 
     to the offset of the variable. 
    */
    unsigned int data_offset; 

    /** The size of this data element in bytes.
    */
    unsigned int  data_size; 

    union {
        unsigned long long default_number; /**< Default value for SYNCDB_JSON_NUMBER object. */
        char *default_string; /**< Default value for SYNCDB_JSON_STRING object */
    } val;

} syncdbJsonNode_t;

/** The following errors may be reported during schema validation.
*/
typedef enum 
{
    SYNCDB_SCHEMA_OK = 0,   /**< No error */
    SYNCDB_SCHEMA_TOO_BIG = 1, /**< Schema node offset plus size is larger than data element */
    SYNCDB_SCHEMA_OVERLAP = 2, /**< Schema nodes overlap each other. */
    SYNCDB_SCHEMA_TOO_SHORT = 3, /**< Data element has 8 or more bytes beyond last schema node */
    SYNCDB_SCHEMA_GAP = 4, /**< Two schema nodes have an 8 or more byte gap between them */
    SYNCDB_SCHEMA_INT_OVERFLOW = 5, /**< Default value for an integer does not fit into size */
    SYNCDB_SCHEMA_STRING_OVERFLOW = 6, /**< Default value for a string does not fit into size */
    SYNCDB_SCHEMA_INT_SIZE = 7, /**< Integer size is not 1, 2, 4, or 8 */
    SYNCDB_SCHEMA_TYPE = 8, /**< Data type is not Number, String or Array. */
    SYNCDB_SCHEMA_NO_ZERO_OFFSET = 9, /**< No node with offset set to 0. */
    SYNCDB_SCHEMA_ZERO_SIZE = 10, /**< Found an element with size equal to 0 */
    SYNCDB_SCHEMA_DUP_NAME = 11 /**< The element has the same name as another element in the schema */
} syncdbSchemaError_e;

/******************************** 
** syncdb API Functions 
*********************************/ 

/*********************************************************************
* @purpose  Generate a storable record schema.
*           
* @param    element_node      @b{(input)} Element descriptor nodes.
* @param    node_schema_size  @b{(input)} Size of the element node array.
* @param    schema_buf        @b{(input)} Buffer in which the JSON schema is returned.
*                   The caller must allocate space for this buffer.
* @param    buf_size          @b{(input)} Number of bytes in the schema_buf.
*                   The maximum buffer size is: @ref SYNCDB_JSON_MAX_SCHEMA_SIZE
* @param    schema_size       @b{(output)} Actual schema length.
* @param    data_element_size @b{(input)} Data record size. Used for error checking.
* @param    schema_error      @b{(output)} Detailed schema error code.
* 
* 
* 
* @returns  SYNCDB_OK       Entry is found.
* @returns  SYNCDB_ERROR    Error in input parameters or element nodes.
* @returns  SYNCDB_SIZE     JSON schema does not fit into schema_buf buffer.
* @returns  SYNCDB_SCHEMA_ERROR  Error in schema.
*
* @notes    If an error is detected while generating the schema,
*           the caller can print out the content of the schema_buf
*           to see approximately where the error is found. The
*           schema_buf contains generated schema up to the point
*           where the error was found.
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbUtilSchemaCreate (syncdbJsonNode_t *element_node,
                            unsigned int node_schema_size, 
                            char *schema_buf,
                            unsigned int buf_size,
                            unsigned int *schema_size,
                            unsigned int data_element_size,
                            syncdbSchemaError_e *schema_error
                            );

/*********************************************************************
* @purpose  Register a new client with the syncdb.
*           
* @param    client_name @b{(Input)} Descriptive name for this client.
* @param    client_id @b{(Output)}  ID used on subsequent transactions.
*           The client_id contains the command socket, the notification
*           socket and unique syncdb client ID.
*
* @returns  SYNCDB_OK - Client is registered.
* @returns  SYNCDB_ERROR - Can't create socket.
* @returns  SYNCDB_MAX_CLIENTS - Too many clients already registered.
*
* @notes   The client may deregister using the syncdbClientDeregister(). 
*          The syncdb automatically deregisters clients whose process
*          is terminated.
* 
* @notes         The client_id.client_id returned parameter is a unique
*          identifier for the client within the system. The client ID
*          is an integer between 0 and 65535. The client IDs are
*          assigned in the incrementing order. When the maximum value
*          is reached the client ID is wrapped to 0. This means that
*          client IDs are not reused immediately.
* 
* @notes         The list of active clients 
*          is maintained by syncdb in the internal AVL
*          table "syncdb-client-table". The applications can register
*          to be notified about changes to this table and perform
*          Get/GetNext/GetNextChanged operations on that table.
*          The client table in conjunction with the client ID re-use
*          policy allows applications to detect when any other application
*          fails.
*       
* @notes         Multiple clients may register with the same client_name.
* 
* @notes         The function may be called from a process or a pthread.
*          Multiple registrations by the same process or by the same
*          pthread are allowed.
* 
* @notes         The return client handle includes socket descriptors.
*          The returned client handle can only be used by the process 
*          that registered with the syncdb. A pthread can pass the
*          handle to another pthread, but only one pthread can invoke
*          a syncdb API call on the same client handle at any one time.
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbClientRegister (char *client_name,
                          syncdbClientHandle_t  *client_id);

/*********************************************************************
* @purpose  Deregister a client from the syncdb.
*           
* @param    client_id @b{(Input)} Client ID to deregister.
*
* @returns  SYNCDB_OK - Client is registered.
* @returns  SYNCDB_ERROR - Can't create socket.
*
* @notes  The syncdb automatically deregisters clients whose process
*         is terminated. 
*          
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbClientDeregister (syncdbClientHandle_t  *client_id);

/*********************************************************************
* @purpose  Get client information for the specified client ID.
*           
* @param    client_id @b{(input)} Handle returned by syncdbClientRegister()
* @param    client_status @b{(input/output)} Information about this client.
*
* @returns  SYNCDB_OK - Status is returned.
* @returns  SYNCDB_ERROR - The status request failed.
* @returns  SYNCDB_NOT_FOUND - The client does not exist.
*
* @notes    The caller must set the client_status->client_id to the
*           syncdb client ID for which to retrieve status.
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbClientStatusGet (syncdbClientHandle_t  *client_id,
                          syncdbClientStatus_t *client_status);

/*********************************************************************
* @purpose  Store specified table or all tables into the file system.
*           
* @param    client_id  @b{(input)} Handle returned by syncdbClientRegister()
* @param    table_name @b{(input)} Name of the table to be written to the file.
*                         NULL - Store all tables to the file system.
* @param    nvram @b{(input)} Non-Zero - Save tables in NVRAM.
*                             0 - Save tables only in the RAM disk.
*
* @returns  SYNCDB_OK - Specified table(s) is written to file.
* @returns  SYNCDB_ERROR - Error.
* @returns  SYNCDB_NO_TABLE - Specified table does not exist.
*
* @notes   Only storable tables are written to the file system.
*          The storable tables are created with the SYNCDB_TABLE_FLAG_STORABLE
*          property.
* @notes         If the "nvram" flag is set to 1, then tables with the
*          SYNCDB_TABLE_FLAG_NVRAM are saved in NVRAM.
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbTableStore (syncdbClientHandle_t  *client_id,
                          char *table_name,
                          unsigned int nvram);

/*********************************************************************
* @purpose  Check whether any change notifications are pending
*           for this client.
*           
* @param    client_id @b{(input)} Handle returned by syncdbClientRegister()
* @param    timeout_secs @b{(input)} How many seconds to wait.
*                       0 - Do not wait. Return immediately.
*                       < 0 - Wait forever.
*                       > 0 - Wait specified number of seconds.
*
* @returns  SYNCDB_OK - Change notifications are pending.
* @returns  SYNCDB_ERROR - No pending notifications.
*
* @notes    This function checks the event socket and returns
*           immediately or waits for an event.

*  @notes         If the user application needs to handle other socket
*           events then it can explicitly add the
*           client_id->notify_socket to the select() statement which
*           it uses to wait for other sockets.
*           The syncdbTableChaneCheck() can then be called with
*           0 timeout to check for events and drain the event
*           socket.
* 
* @notes          If the process receives a signal, and the wait is
*           with a timeout then the function exits.
* 
* @notes          If the wait is "forever" then signals do not
*           cause the function to exit.
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbTableChangeCheck (syncdbClientHandle_t  *client_id,
                          int  timeout_secs);

/*********************************************************************
* @purpose  Notify this client about changes to the specified table.
*           
* @param    client_id @b{(input)} Handle returned by syncdbClientRegister()
* @param    table_name @b{(input)} Unique name for the table.
*
* @returns  SYNCDB_OK - Table notification is enabled.
* @returns  SYNCDB_ERROR - Table notification is not enabled.
* @returns  SYNCDB_NO_NAME - Specified table does not exist.
*
* @notes    The function works for every table type.
*           Note that there is no API to disable table notifications.
*           When a process that created the client ID dies, the
*           syncdb automatically disables table change notifications
*           for that client.
* 
* @notes          The table changes made after the registration cause events to
*           be generated to the client. To make sure that the client
*           does not miss any events, the client should read the
*           content of whole table after registering for
*           notifications.
* 
* @notes          The change notifications are NOT sent to the client which
*           is making the table change.
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbTableChangeNotify (syncdbClientHandle_t  *client_id,
                          char  *table_name);

/*********************************************************************
* @purpose  Get Table Status as it pertains to the specified client.
*           
* @param    client_id @b{(input)} Handle returned by syncdbClientRegister()
* @param    num_tables @b{(input)} Number of tables in the list.
* @param    table_list @b{(input/output)} List of tables. 
*
* @returns  SYNCDB_OK - Status is returned.
* @returns  SYNCDB_ERROR - The status request failed.
* @returns  SYNCDB_SIZE - Too many tables in the table_list.
*
* @notes    The SYNCDB_OK means that the status command returned
*           information. The return code should not be used as an
*           indication that specified table or tables exist. The caller
*           needs to check the status for each table to see if
*           it is present.
* 
* @notes          The maximum size of the table_list array must be less
*           or equal to SYNCDB_RECORD_MAX_SIZE. Otherwise the
*           function returns the SYNCDB_SIZE error.
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbTableStatusGet (syncdbClientHandle_t  *client_id,
                          int num_tables,
                          syncdbDataTableStatus_t *table_list);

/*********************************************************************
* @purpose  Create AVL Table.
*           
* @param    client_id @{(input)} Handle returned by syncdbClientRegister()
* @param    table_name  @{(input)} Unique name for this table.
* @param    table_version @{(input)} Version of this table. 
* @param    max_elements  @{(input)} Maximum number of nodes in the AVL tree.
* @param    max_live_elements  @{(input)} Maximum number of nodes in the AVL tree
*                         that are not pending for removal. This value must
*                         be greater than 0 and less or equal to max_elements.
* @param    node_size  @{(input)} User data number of bytes in each table element.
*                        The size includes the key field.
*                   The maximum node size is: @ref SYNCDB_RECORD_MAX_SIZE
* @param    key_size  @{(input)} Number of bytes in the key element. The first
*                       'key_size' bytes are treated as the key.
*                   The maximum key size is: @ref SYNCDB_RECORD_MAX_SIZE
* @param    flags @{(input)} Table Creation Flags
*                    SYNCDB_TABLE_FLAG_STORABLE - Schema is present.
*                    SYNCDB_TABLE_FLAG_FILE_LOAD - Load the table from file if file exists.
*                    SYNCDB_TABLE_FLAG_NVRAM - Copy to NVRAM when storing the table.
* @param    schema  @{(input)} JSON schema create using the syncdbUtilSchemaCreate().
*                     Must be set to 0 for non-storable tables. Must be non-zero
*                     for storable tables.
* @param    schema_size  @{(input)} JSON schema size returned by the syncdbUtilSchemaCreate().
*                      Must be 0 for non-storable tables.
*                   The maximum schema size is: @ref SYNCDB_JSON_MAX_SCHEMA_SIZE
*
* @returns  SYNCDB_OK - The table is created.
* @returns  SYNCDB_ERROR - Can't create the table.
* @returns  SYNCDB_DUPNAME - The table is not created because another
*                            table already exists with the same name.
*
* @notes   
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbAvlTableCreate (syncdbClientHandle_t  *client_id,
                          char  *table_name,
                          unsigned int   table_version,
                          unsigned int   max_elements,
                          unsigned int   max_live_elements,
                          unsigned int   node_size,
                          unsigned int   key_size,
                          unsigned int   flags,
                          char * schema,
                          unsigned int   schema_size);

/*********************************************************************
* @purpose  Create a Record Table.
*           
* @param    client_id @b{(input)}  Handle returned by syncdbClientRegister()
* @param    table_name  @b{(input)} Unique name for this table.
* @param    table_version @b{(input)} Version of this table. 
* @param    node_size  @b{(input)} User data number of bytes in the table element.
*                   The maximum node size is: @ref SYNCDB_RECORD_MAX_SIZE
* @param    flags @b{(input)} Table Creation Flags
*                    SYNCDB_TABLE_FLAG_STORABLE - Schema is present.
*                    SYNCDB_TABLE_FLAG_FILE_LOAD - Load the table from file if file exists.
*                    SYNCDB_TABLE_FLAG_NVRAM - Copy to NVRAM when storing the table.
* @param    schema  @b{(input)} JSON schema create using the syncdbUtilSchemaCreate().
*                     Must be set to 0 for non-storable tables. Must be non-zero
*                     for storable tables.
* @param    schema_size  @b{(input)} JSON schema size returned by the syncdbUtilSchemaCreate().
*                      Must be 0 for non-storable tables.
*                   The maximum schema size is: @ref SYNCDB_JSON_MAX_SCHEMA_SIZE
*
* @returns  SYNCDB_OK - The table is created.
* @returns  SYNCDB_ERROR - Can't create the table.
* @returns  SYNCDB_DUPNAME - The table is not created because another
*                            table already exists with the same name.
*
* @notes   
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbRecordTableCreate (syncdbClientHandle_t  *client_id,
                          char  *table_name,
                          unsigned int   table_version,
                          unsigned int   node_size,
                          unsigned int   flags,
                          char * schema,
                          unsigned int   schema_size);

/*********************************************************************
* @purpose  Delete specified table or all tables.
*           
* @param    client_id @b{(input)} Handle returned by syncdbClientRegister()
* @param    table_name  @b{(input)} Name of the table to be deleted.
*
* @returns  SYNCDB_OK - Specified table is deleted.
* @returns  SYNCDB_ERROR - Error.
* @returns  SYNCDB_NO_TABLE - Specified table does not exist.
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbTableDelete (syncdbClientHandle_t  *client_id,
                          char *table_name);

/*********************************************************************
* @purpose  Insert a new entry into the AVL table.
*           
* @param    client_id @b{(input)}  Handle returned by syncdbClientRegister()
* @param    table_name  @b{(input)} Unique name for this table.
* @param    element  @b{(input)} Pointer to the data element to be inserted.
* @param    size  @b{(input)} Number of bytes in the element.
*
* @returns  SYNCDB_OK - Entry is inserted.
* @returns  SYNCDB_ERROR - Unspecified error.
* @returns  SYNCDB_NO_TABLE - Specified table does not exist.
* @returns  SYNCDB_FULL - Specified table is already full.
* @returns  SYNCDB_SIZE - The specified size is invalid.
* @returns  SYNCDB_ENTRY_EXISTS - The specified entry already exists and
*                           the SYNCDB_TABLE_FLAG_EXISTS is enabled for
*                           the table.
*
* @notes   The Insert operation is supported only on AVL trees.
*          If the specified table is not an AVL then
*          the SYCNDB_ERROR error code is returned.
* 
* @notes         The 'size' must be exactly the same as the
*          record size specified on table creation.
*          If the size is invalid then SYNCDB_SIZE error code
*          is returned.
* 
* @notes         If the element with the same key already exists in 
*          the tree then the existing element is updated with the new content.
*          If the table is created with the SYNCDB_TABLE_FLAG_EXISTS then
*          the syncdbInsert() fails for an entry with the duplicate
*          key and returns the SYNCDB_ENTRY_EXISTS error.
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbInsert (syncdbClientHandle_t  *client_id,
                          char  *table_name,
                          void *element,
                          unsigned int   size);

/*********************************************************************
* @purpose  Delete an entry from the AVL table.
*           
* @param    client_id @b{(input)} Handle returned by syncdbClientRegister()
* @param    table_name  @b{(input)} Unique name for this table.
* @param    element  @b{(input)} Pointer to the data element to be deleted.
* @param    size  @b{(input)} Number of bytes in the element.
*
* @returns  SYNCDB_OK - Entry is deleted.
* @returns  SYNCDB_ERROR - Unspecified error.
* @returns  SYNCDB_NO_TABLE - Specified table does not exist.
* @returns  SYNCDB_SIZE - The specified size is invalid.
* @returns  SYNCDB_NOT_FOUND - Entry is not found in the table.
*
* @notes   The Delete operation is supported only on AVL trees.
*          If the specified table is not an AVL tree then
*          the SYCNDB_ERROR error code is returned.
* 
* @notes         The 'size' must be exactly the same as the
*          record size specified on table creation.
*          If the size is invalid then SYNCDB_SIZE error code
*          is returned.
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbDelete (syncdbClientHandle_t  *client_id,
                          char  *table_name,
                          void *element,
                          unsigned int   size);

/*********************************************************************
* @purpose  Set an entry in the table.
*           
* @param    client_id @b{(input)}  Handle returned by syncdbClientRegister()
* @param    table_name  @b{(input)} Unique name for this table.
* @param    element  @b{(input)} Pointer to the data buffer to be set in the table. 
* @param    size  @b{(input)} Number of bytes in the element.
*
* @returns  SYNCDB_OK - Entry is modified.
* @returns  SYNCDB_ERROR - Unspecified error.
* @returns  SYNCDB_NO_TABLE - Specified table does not exist.
* @returns  SYNCDB_SIZE - The specified size is invalid.
* @returns  SYNCDB_NOT_FOUND - Entry is not found in the table.
*
* @notes   This command works for Records and AVL Trees.
*          For AVL Trees the command modifies the element which
*          matches the key.
*          For Records the command modifies the whole record.
* 
* @notes         For AVL Trees and Records the 'size' must be exactly the
*          same as the record size specified on table creation.
* 
* @notes         If the size is invalid then SYNCDB_SIZE error code
*          is returned.
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbSet (syncdbClientHandle_t  *client_id,
                          char  *table_name,
                          void *element,
                          unsigned int   size);

/*********************************************************************
* @purpose  Set the specified field in an entry in the table.
*           The remaining fields in the entry are unchanged.
*           
* @param    client_id @b{(input)} Handle returned by syncdbClientRegister()
* @param    table_name  @b{(input)} Unique name for this table.
* @param    element @b{(Input)} Caller must pass a buffer where
*                   the data is located.
* @param    size  @b{(input)} Number of bytes in the element.
* @param    field_offset  @b{(input)} Start of the field to set in the table.
*                           The value represents the number of bytes from the
*                           start of the element.
* @param    field_size  @b{(input)} Size of the field to set.
*
* @returns  SYNCDB_OK - Entry is modified.
* @returns  SYNCDB_ERROR - Unspecified error.
* @returns  SYNCDB_NO_TABLE - Specified table does not exist.
* @returns  SYNCDB_SIZE - The specified size is invalid.
* @returns  SYNCDB_NOT_FOUND - Entry is not found in the table.
*
* @notes   This command works for Records and AVL Trees.
*          For AVL Trees the command modifies the element which
*          matches the key.
* 
* @notes         The command is useful when multiple processes modify
*          different fields in the same record.
* 
* @notes         The 'size' must be exactly the
*          same as the record size specified on the table creation.
*          If the size is invalid then SYNCDB_SIZE error code
*          is returned.
* 
* @notes         The field offset plus the field size must be less or
*          equal to the "size".
*          Note that multiple adjacent fields can be set
*          at the same time. The whole record can be set
*          by setting the "field_offset" to 0 and "field_size" to
*          "size".
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbFieldSet (syncdbClientHandle_t  *client_id,
                          char  *table_name,
                          void *element,
                          unsigned int   size,
                          unsigned int field_offset,
                          unsigned int field_size);

/*********************************************************************
* @purpose  Get an entry from the table.
*           
* @param    client_id @b{(input)} Handle returned by syncdbClientRegister()
* @param    table_name  @b{(input)} Unique name for this table.
* @param    element @b{(input/output)} Caller must pass a buffer where
*                   the data is written. The buffer must be big
*                   enough to contain 'size' bytes.
* @param    size  @b{(input)} Number of bytes in the element.
* @param    delete_pending @b{(output)} For AVL Trees this indicates
*                           whether the record is pending for
*                           deletion. The field is unused for
*                           records and queues.
*
* @returns  SYNCDB_OK - Entry is found. Need to check delete_pending
*                       flag for AVL trees.
* @returns  SYNCDB_ERROR - Unspecified error.
* @returns  SYNCDB_NO_TABLE - Specified table does not exist.
* @returns  SYNCDB_SIZE - The specified size is invalid.
* @returns  SYNCDB_NOT_FOUND - Entry is not found in the table.
*
* @notes   This command works for Records and AVL Trees.
*          For AVL Trees the command retrieves the element which
*          matches the key.
*          For Records the command retrieves the whole record.
* 
* @notes         The 'size' must be exactly the
*          same as the record size specified on the table creation.
*          If the size is invalid then SYNCDB_SIZE error code
*          is returned.
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbGet (syncdbClientHandle_t  *client_id,
                          char  *table_name,
                          void *element,
                          unsigned int   size,
                          int *delete_pending);

/*********************************************************************
* @purpose  Get the specified field in an entry from the table.
*           
* @param    client_id @b{(input)} Handle returned by syncdbClientRegister()
* @param    table_name  @b{(input)} Unique name for this table.
* @param    element @b{(input/output)} Caller must pass a buffer where
*                   the data is written. The buffer must be big
*                   enough to contain 'size' bytes.
* @param    size  @b{(input)} Number of bytes in the element.
* @param    field_offset  @b{(input)} Start of the field to retrieve from the table.
*                           The value represents the number of bytes from the
*                           start of the table.
* @param    field_size  @b{(input)} Size of the field to get.
* @param    flags_unchanged @b{(input)} Do not clear the notification-pending and the
*                             delete-pending flags.
* @param    delete_pending @b{(output)} For AVL Trees this indicates
*                           whether the record is pending for
*                           deletion. The field is unused for
*                           records and queues.
*
* @returns  SYNCDB_OK - Entry is found. Need to check delete_pending
*                       flag for AVL trees.
* @returns  SYNCDB_ERROR - Unspecified error.
* @returns  SYNCDB_NO_TABLE - Specified table does not exist.
* @returns  SYNCDB_SIZE - The specified size is invalid.
* @returns  SYNCDB_NOT_FOUND - Entry is not found in the table.
*
* @notes   This command works for Records and AVL Trees.
*          For AVL Trees the command retrieves the element which
*          matches the key.
* 
* @notes         The 'size' must be exactly the
*          same as the record size specified on the table creation.
*          If the size is invalid then SYNCDB_SIZE error code
*          is returned.
* 
* @notes         The field offset plus the field size must be less or
*          equal to the "size".
*          Note that multiple adjacent fields can be retrieved
*          at the same time. The whole record can be retrieved
*          by setting the "field_offset" to 0 and "field_size" to
*          "size".
* 
* @notes         The function may be used to retrieve data without
*          clearing the change notification and delete-pending flags
*          by setting the "flags_unchanged" parameter to 1.
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbFieldGet (syncdbClientHandle_t  *client_id,
                          char  *table_name,
                          void *element,
                          unsigned int   size,
                          unsigned int field_offset,
                          unsigned int field_size,
                          int  flags_unchanged,
                          int *delete_pending);

/*********************************************************************
* @purpose  Get the Next entry from the table.
*           
* @param    client_id @{(input)} Handle returned by syncdbClientRegister()
* @param    table_name  @b{(input)} Unique name for this table.
* @param    element @b{(input/output)} Caller must pass a buffer where
*                   the data is written. The buffer must be big
*                   enough to contain 'size' bytes.
* @param    size  @b{(input)} Number of bytes in the element.
* @param    delete_pending @b{(output)} For AVL Trees this indicates
*                           whether the record is pending for
*                           deletion. The field is unused for
*                           records and queues.
*
* @returns  SYNCDB_OK - Entry is found.
* @returns  SYNCDB_ERROR - Unspecified error.
* @returns  SYNCDB_NO_TABLE - Specified table does not exist.
* @returns  SYNCDB_SIZE - The specified size is invalid.
* @returns  SYNCDB_NOT_FOUND - Entry is not found in the table.
*
* @notes   This command works only for AVL Trees.
*          The command does not work for other table types.
*          For AVL Trees the command retrieves the element which
*          matches the next higher key.
* 
* @notes         The 'size' must be exactly the
*          same as the record size specified on the table creation.
*          If the size is invalid then SYNCDB_SIZE error code
*          is returned.
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbGetNext (syncdbClientHandle_t  *client_id,
                          char  *table_name,
                          void *element,
                          unsigned int   size,
                          int *delete_pending);

/*********************************************************************
* @purpose  Get the Next Changed entry from the table.
*           
* @param    client_id @b{(input)} Handle returned by syncdbClientRegister()
* @param    table_name  @b{(input)} Unique name for this table.
* @param    element @b{(input/output)}  Caller must pass a buffer where
*                   the data is written. The buffer must be big
*                   enough to contain 'size' bytes.
* @param    size  @b{(input)} Number of bytes in the element.
* @param    delete_pending @b{(output)} For AVL Trees this indicates
*                           whether the record is pending for
*                           deletion. The field is unused for
*                           records and queues.
*
* @returns  SYNCDB_OK - Entry is found.
* @returns  SYNCDB_ERROR - Unspecified error.
* @returns  SYNCDB_NO_TABLE - Specified table does not exist.
* @returns  SYNCDB_SIZE - The specified size is invalid.
* @returns  SYNCDB_NOT_FOUND - Entry is not found in the table.
*
* @notes   This command works only for AVL Trees.
*          The command does not work for other table types.
*          For AVL Trees the command retrieves the element which
*          matches the next higher key and which has changed since
*          this client performed the last Get/GetNext/GetNextChanged
*          operation for this record.
* 
* @notes         The 'size' must be exactly the
*          same as the record size specified on the table creation.
*          If the size is invalid then SYNCDB_SIZE error code
*          is returned.
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbGetNextChanged (syncdbClientHandle_t  *client_id,
                          char  *table_name,
                          void *element,
                          unsigned int   size,
                          int *delete_pending);

/*********************************************************************
* @purpose  Tell SyncDB that it should start or stop data sync
*           with the backup manager.
*           
* @param    client_id @b{(input)} Handle returned by syncdbClientRegister()
* @param    sync_mode @b{(input)} 1 - Enable Sync. 0 - Disable Sync.
* @param    max_msg_size @b{(input)} Maximum number of bytes in the sync message.
*
* @returns  SYNCDB_OK - Command is successful.
* @returns  SYNCDB_ERROR - Error.
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
int syncdbNsfModeSet (syncdbClientHandle_t  *client_id,
                          unsigned int sync_mode,
                          unsigned int max_msg_size);

#endif /* SYNCDB_API_H_INCLUDED */

