/**********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_timezone.h
*
* @purpose   This code implements the Open Api for the timesZone API
*
* @component OPEN
*
* @create    01/03/2023
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_TIMEZONE_H_INCLUDED
#define OPENAPI_TIMEZONE_H_INCLUDED

#include "openapi_common.h" 

/** OpEN uses this enumeration to define summer time mode */
typedef enum
{
  OPEN_NO_SUMMER_TIME_MODE = 0,         /**< No summer time */
  OPEN_SUMMER_TIME_MODE_RECURRING,      /**< recurring */
  OPEN_SUMMER_TIME_MODE_RECURRING_EU,   /**< recurring EU */
  OPEN_SUMMER_TIME_MODE_RECURRING_USA,  /**< recurring USA */
  OPEN_SUMMER_TIME_MODE_NON_RECURRING   /**< non recurring */
}OPEN_SUMMER_TIME_MODE_t;

/** OpEN uses this enumeration to define the summer time recurring parameters */
typedef enum
{
  OPEN_ST_RECURRING_PARAM_START_DAY = 1,       /**< start day      : none(0),sun(1),mon(2),tue(3),wed(4),thu(5),fri(6),sat(7) */
  OPEN_ST_RECURRING_PARAM_START_MONTH,         /**< start month    : none(0),jan(1),feb(2),mar(3),apr(4),may(5),jun(6),
                                                                     jul(7),aug(8),sep(9),oct(10),nov(11),dec(12) */
  OPEN_ST_RECURRING_PARAM_START_WEEK,          /**< start week     : none(0),first(1),second(2),third(3),fourth(4),last(5) */
  OPEN_ST_RECURRING_PARAM_START_HOUR,          /**< start hour     : (0..23) */
  OPEN_ST_RECURRING_PARAM_START_MINUTE,        /**< start minute   : (0..59) */
  OPEN_ST_RECURRING_PARAM_END_DAY,             /**< end day        : none(0),sun(1),mon(2),tue(3),wed(4),thu(5),fri(6),sat(7) */
  OPEN_ST_RECURRING_PARAM_END_MONTH,           /**< end month      : none(0),jan(1),feb(2),mar(3),apr(4),may(5),jun(6),
                                                                     jul(7),aug(8),sep(9),oct(10),nov(11),dec(12) */
  OPEN_ST_RECURRING_PARAM_END_WEEK,            /**< end week       : none(0),first(1),second(2),third(3),fourth(4),last(5) */
  OPEN_ST_RECURRING_PARAM_END_HOUR,            /**< end hour       : (0..23) */
  OPEN_ST_RECURRING_PARAM_END_MINUTE,          /**< end minute     : (0..59) */
  OPEN_ST_RECURRING_PARAM_OFFSET_MINUTES,      /**< offset minutes : (1..1440) */
  OPEN_ST_RECURRING_PARAM_ZONE_ACRONYM         /**< zone acronym   : String  */
}OPEN_ST_RECURRING_PARAM_t;

/** OpEN uses this enumeration to define the summer time non recurring parameters */
typedef enum
{
  OPEN_ST_NON_RECURRING_PARAM_START_DAY = 1,       /**< start day      : none(0) | (1..31) */
  OPEN_ST_NON_RECURRING_PARAM_START_MONTH,         /**< start month    : none(0),jan(1),feb(2),mar(3),apr(4),may(5),jun(6),
                                                                         jul(7),aug(8),sep(9),oct(10),nov(11),dec(12) */
  OPEN_ST_NON_RECURRING_PARAM_START_YEAR,          /**< start year     : (0 |2000..2097) */
  OPEN_ST_NON_RECURRING_PARAM_START_HOUR,          /**< start hour     : (0..23) */
  OPEN_ST_NON_RECURRING_PARAM_START_MINUTE,        /**< start minute   : (0..59) */
  OPEN_ST_NON_RECURRING_PARAM_END_DAY,             /**< end day        : none(0) | (1..31) */
  OPEN_ST_NON_RECURRING_PARAM_END_MONTH,           /**< end month      : none(0),jan(1),feb(2),mar(3),apr(4),may(5),jun(6),
                                                                         jul(7),aug(8),sep(9),oct(10),nov(11),dec(12) */
  OPEN_ST_NON_RECURRING_PARAM_END_YEAR,            /**< end year       : (0 |2000..2097) */
  OPEN_ST_NON_RECURRING_PARAM_END_HOUR,            /**< end hour       : (0..23) */
  OPEN_ST_NON_RECURRING_PARAM_END_MINUTE,          /**< end minute     : (0..59) */
  OPEN_ST_NON_RECURRING_PARAM_OFFSET_MINUTES,      /**< offset minutes : (1..1440) */
  OPEN_ST_NON_RECURRING_PARAM_ZONE_ACRONYM         /**< zone acronym   : String  */
}OPEN_ST_NON_RECURRING_PARAM_t;

typedef struct open_clocktime_t
{
  time_t seconds;
  time_t nanoseconds;
} open_clocktime;

/*****************************************************************************
 * @purpose   Convert hh:mm string to int values.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     bufd           @b{(input)}  ptr to buffer discription to hold time string
 * @param     pHour          @b{(output)} pointer to Hour
 * @param     pMin           @b{(output)} pointer to minutes
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     time may be between 00:00 and 23:59
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiTimeConvert(openapiClientHandle_t *client_handle, open_buffdesc *bufd, uint32_t *pHour, uint32_t *pMin);

/*****************************************************************************
 * @purpose   Convert minute offset into hour and minutes
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     minutes        @b{(input)}  Holds minutes
 * @param     pHour          @b{(output)} pointer to Hour
 * @param     pMin           @b{(output)} pointer to minutes
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Converts minute offset into hour and minute offset properly
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMinutesOffsetConvert(openapiClientHandle_t *client_handle, int32_t minutes, uint32_t *pHour, uint32_t *pMin);

/*****************************************************************************
 * @purpose   Convert month name to number value
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     name           @b{(input)}  character buffer with week name string
 * @param     number         @b{(output)} pointer to week sequence number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Convert month name to week number value
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiWeekNumberGet(openapiClientHandle_t *client_handle, open_buffdesc *name, uint32_t *number);

/*****************************************************************************
 * @purpose   Set the time zone acronym parameter
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     acronym        @b{(input)}  minutes in timezone structure.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiTimeZoneAcronymSet(openapiClientHandle_t *client_handle, open_buffdesc *acronym);

/*****************************************************************************
 * @purpose   Get the time zone acronym parameter
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     acronym        @b{(output)} minutes in timezone structure.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiTimeZoneAcronymGet(openapiClientHandle_t *client_handle, open_buffdesc *acronym);

/*****************************************************************************
 * @purpose   Get non-recurring summertime parameters.
 *
 * @param     client_handle  @b{(input)} client handle from registration API
 * @param     parameter      @b{(input)} summer-time parameter
 * @param     bufd           @b{(output)} summer-time parameter value to be returned
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiStNonRecurringGet(openapiClientHandle_t *client_handle, OPEN_ST_NON_RECURRING_PARAM_t parameter, open_buffdesc *bufd);

/*****************************************************************************
 * @purpose   Get recurring summertime parameters.
 *
 * @param     client_handle  @b{(input)} client handle from registration API
 * @param     parameter      @b{(input)} summer-time parameter
 * @param     bufd           @b{(output)} summer-time parameter value to be returned 
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiStRecurringGet(openapiClientHandle_t *client_handle, OPEN_ST_RECURRING_PARAM_t parameter, open_buffdesc *bufd);

/*****************************************************************************
 * @purpose   Set recurring summertime parameters.
 *
 * @param     client_handle  @b{(input)} client handle from registration API
 * @param     parameter      @b{(input)} summer-time parameter
 * @param     bufd           @b{(input)} summer-time parameter value to be set
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiStRecurringSet(openapiClientHandle_t *client_handle, OPEN_ST_RECURRING_PARAM_t parameter, open_buffdesc *bufd);

/*****************************************************************************
 * @purpose   Set non-recurring summertime parameters.
 *
 * @param     client_handle  @b{(input)} client handle from registration API
 * @param     parameter      @b{(input)} summer-time parameter 
 * @param     bufd           @b{(input)} summer-time parameter value to be set
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiStNonRecurringSet(openapiClientHandle_t *client_handle, OPEN_ST_NON_RECURRING_PARAM_t parameter, open_buffdesc *bufd);

/*****************************************************************************
 * @purpose   Get the summer-time mode
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     mode           @b{(output)} Summer-time mode to be returned
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiStModeGet(openapiClientHandle_t *client_handle, OPEN_SUMMER_TIME_MODE_t *mode);

/*****************************************************************************
 * @purpose   Set the summer-time mode
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     mode           @b{(input)}  summer-time mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiStModeSet(openapiClientHandle_t *client_handle, OPEN_SUMMER_TIME_MODE_t mode);

/*****************************************************************************
 * @purpose   Gets the timezone acronym string
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     zoneName       @b{(output)}  timezone name
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSystemTimeZoneAcronymGet(openapiClientHandle_t *client_handle, open_buffdesc *zoneName);

/*****************************************************************************
 * @purpose   Get the current synchronized UTC time since the Unix Epoch.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 *
 * @returns   int32_t            seconds since Unix epoch.
 *
 * @notes     This is a utility routine for SNTP internal timestamping.
 * @notes     Note that NTP epoch is different from Unix epoch. This
 * @notes     routine attempts not to return negative time if the clock
 * @notes     has not been synchronized at least once.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 ****************************************************************************/
int32_t openapiUTCTimeNow(openapiClientHandle_t *client_handle );

/*****************************************************************************
 * @purpose   Get the current UTC time since the Unix Epoch.
 * 
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ct             @b{(output)} UTC time
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     NTP epoch is different from Unix epoch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiUTCTimeGet(openapiClientHandle_t *client_handle, open_clocktime * ct);

/*****************************************************************************
 * @purpose   know whether the summer-time is in effect or not.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API      
 *
 * @returns   OPEN_TRUE            when summer-time is in effect
 * @returns   OPEN_FALSE            when summer-time is not in effect
 *
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
OPEN_BOOL_t openapiSummerTimeIsInEffect(openapiClientHandle_t *client_handle);

/*****************************************************************************
 * @purpose   Adjusts current time for timezone and summer time
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     year           @b{(output)} year
 * @param     month          @b{(output)} month of the year
 * @param     day            @b{(output)} day of the month
 * @param     hour           @b{(output)} hour of the day
 * @param     minutes        @b{(output)} minutes of the hour
 * @param     seconds        @b{(output)} seconds of the hour
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAdjustedTimeGet(openapiClientHandle_t *client_handle, uint32_t *year, uint32_t *month, uint32_t * day, uint32_t * hour, uint32_t * minutes, uint32_t * seconds);

/*****************************************************************************
 * @purpose   Converts the given Raw time to UTC and then adds the timezone and summer time offsets if configured.
 * 
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     rawTime        @b{(input)}  Raw time
 *
 * @returns   uint32_t            Adjusted time
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
uint32_t openapiRawAdjustedTimeGet(openapiClientHandle_t *client_handle, uint32_t rawTime);


#endif /* OPENAPI__H_INCLUDED */

