/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_user_manager.h
*
* @purpose   User Manager Configuration and Status 
*
* @component OpEN
*
* @create    18/01/2013
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_USER_MGR_H_INCLUDED
#define OPENAPI_USER_MGR_H_INCLUDED

#include "openapi_common.h"

/** OpEN uses this enumeration to define Login Access Permissions */
typedef enum  
{
  OPEN_LOGIN_ACCESS_NONE        = 0, /**< No Login Access */
  OPEN_LOGIN_ACCESS_READ_ONLY   = 1, /**< Read-only Login Access  */
  OPEN_LOGIN_ACCESS_READ_WRITE  = 2  /**< Read-write Login Access */
} OPEN_LOGIN_ACCESS_LEVELS_t;

/** OpEN uses this enumeration to define access levels */
typedef enum
{ 
  OPEN_ACCESS_LEVEL_UNKNOWN = 0, /**< Unknown Access Level */
  OPEN_ACCESS_LEVEL_LOGIN   = 1, /**< Login Access Level*/
  OPEN_ACCESS_LEVEL_ENABLE  = 2  /**< Enable Access Level */
} OPEN_ACCESS_LEVEL_t;

/** OpEN uses this enumeration to define Authentication method types */
typedef enum
{
  OPEN_AUTH_METHOD_UNDEFINED = 0, /**< Undefined Authentication Method */
  OPEN_AUTH_METHOD_ENABLE    = 1, /**< Enable Authentication Method */
  OPEN_AUTH_METHOD_LINE      = 2, /**< Line Authentication Method */
  OPEN_AUTH_METHOD_LOCAL     = 3, /**< Local Authentication Method */
  OPEN_AUTH_METHOD_NONE      = 4, /**< No Authentication Method */  
  OPEN_AUTH_METHOD_RADIUS    = 5, /**< RADIUS Authentication Method */                
  OPEN_AUTH_METHOD_TACACS    = 6, /**< TACACS Authentication Method */
  OPEN_AUTH_METHOD_REJECT    = 7, /**< Reject Authentication */
  OPEN_AUTH_METHOD_IAS       = 8  /**< Internal */
} OPEN_USER_MGR_AUTH_METHOD_t;

/** OpEN uses this enumeration to define Authorization types */
typedef enum
{
  OPEN_AUTHOR_TYPE_UNKNOWN  = 0, /**< Unknown Authorization type */
  OPEN_AUTHOR_TYPE_COMMANDS = 1, /**< Command Authorization */
  OPEN_AUTHOR_TYPE_EXEC     = 2, /**< Exec Authorization */
  OPEN_AUTHOR_TYPE_MAX      = 3  /**< Useful to check maximum Authorization types */
} OPEN_USER_MGR_AUTHOR_TYPES_t;

/** OpEN uses this enumeration to define Authorization methods */
typedef enum
{ 
  OPEN_AUTHOR_METHOD_UNDEFINED  = 0, /**< Authorization method undefined */
  OPEN_AUTHOR_METHOD_TACACS     = 1, /**< Authorization method TACACS */            
  OPEN_AUTHOR_METHOD_RADIUS     = 2, /**< Authorization method radius */
  OPEN_AUTHOR_METHOD_LOCAL      = 3, /**< Authorization method local */
  OPEN_AUTHOR_METHOD_NONE       = 4, /**< Authorization method none */
  OPEN_AUTHOR_METHOD_MAX        = 5  /**< Enum to track maximum Authorization  methods */
} OPEN_AUTHOR_METHOD_t;

/** OpEN uses this enumeration to define Accounting Types */
typedef enum
{
  OPEN_ACCT_TYPE_UNDEFINED      = 0, /**<Undefined Accounting Type */
  OPEN_ACCT_TYPE_EXEC_SESSION   = 1, /**< Exec Accounting */
  OPEN_ACCT_TYPE_COMMANDS       = 2, /**< Command Accounting */
  OPEN_ACCT_TYPE_MAX            = 3  /**< Useful to identify maximum Accounting types supported */
} OPEN_USER_MGR_ACCT_TYPES_t;

/** OpEN uses this enumeration to define Accounting Record types */
typedef enum
{
  OPEN_ACCT_RECORD_TYPE_UNDEFINED  = 0, /**< Undefined Record Type */
  OPEN_ACCT_RECORD_TYPE_START_STOP = 1, /**< Start-Stop Record Type */
  OPEN_ACCT_RECORD_TYPE_STOP_ONLY  = 2, /**< Stop-only Record Type */
  OPEN_ACCT_RECORD_TYPE_NONE       = 3, /**< No Record */
  OPEN_ACCT_RECORD_TYPE_MAX        = 4  /**< Useful to identify maximum record types supported */
} OPEN_USER_MGR_RECORD_TYPES_t;

/** OpEN uses this enumeration to define Accounting Methods */
typedef enum
{       
  OPEN_ACCT_METHOD_UNDEFINED    = 0, /**< Accounting Methods Undefined */
  OPEN_ACCT_METHOD_TACACS       = 1, /**< TACACS Accounting Method */
  OPEN_ACCT_METHOD_RADIUS       = 2, /**< RADIUS Accounting Method */
  OPEN_ACCT_METHOD_MAX          = 3  /**< Useful to identify maximum Accounting Methods */
} OPEN_USER_MGR_ACCT_METHOD_t;

/** OpEN Session ID list which use to specify the login type. */
typedef enum
{
  OPEN_LOGIN_TYPE_UNKNWN = 0,
  OPEN_LOGIN_TYPE_SERIAL = 1,
  OPEN_LOGIN_TYPE_TELNET = 2,
  OPEN_LOGIN_TYPE_SSH    = 3,
  OPEN_LOGIN_TYPE_HTTP   = 4,
  OPEN_LOGIN_TYPE_HTTPS  = 5,
  OPEN_LOGIN_TYPE_SNMP   = 6,
  OPEN_LOGIN_TYPE_PING   = 7,
  OPEN_LOGIN_TYPE_UDS    = 8,
  OPEN_LOGIN_TYPE_LAST   = 9
} OPEN_LOGIN_TYPE_t;

/** OpEN uses this enumeration to define password encryption type. */
typedef enum  
{ 
  OPEN_ENCRYPT_TYPE_NONE     = 0,     /**< No Encryption type */  
  OPEN_ENCRYPT_TYPE_AES      = 1,     /**< AES Encryption type */
  OPEN_ENCRYPT_TYPE_MD5_SALT = 2,     /**< MD5 Salted Hash Encryption type */
  OPEN_ENCRYPT_TYPE_SHA256_SALT = 3,  /**< SHA256 Salted Hash Encryption type */
  OPEN_ENCRYPT_TYPE_SHA512_SALT = 4,  /**< SHA512 Salted Hash Encryption type */
  OPEN_ENCRYPT_TYPE_LAST     = 5      /**< Useful to identify maximum Encryption types */
} OPEN_ENCRYPT_TYPE_t;

#define OPEN_LOGIN_SIZE 65   /** Login session username buffer size */
#define OPEN_PASSWORD_SIZE  65 /** Unencrypted password buffer size */

/*********************************************************************
* @purpose  Set the user name and password.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    userName         @b{(input)}  User name buff descriptor
* @param    password         @b{(input)}  Password buff descriptor
*
* @returns  OPEN_E_NONE        if user name and password are set successfully.
* @returns  OPEN_E_FAIL        if failed to set username and password.
* @returns  OPEN_E_FULL        if no more users can be added.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
*
* @notes   This API allows to create a new user with password and also change 
*          the password for an existing user. 
* @notes   Calling this API will change the running configuration of the switch.
* @notes   This API creates the user with default access level (RO).
* @notes   Maximum size for userName is determined by calling
*          openapiUserNameSizeGet().
* @notes   Maximum size for password is determined by calling 
*          openapiUserPasswordSizeGet() with arg isEncrypPswd as OPEN_FALSE.
*
* @notes   Password accepts all the special characters except '"' and '?'.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiUserSet(openapiClientHandle_t *client_handle,
                            open_buffdesc *userName,
                            open_buffdesc *password);

/*********************************************************************
* @purpose  Get the first user.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    username         @b{(output)} First User name.
* 
* @returns  OPEN_E_NONE      if the user is returned successfully.
* @returns  OPEN_E_FAIL      if failed to get first user.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes  Maximum size for userName is determined by calling
*         openapiUserNameSizeGet(). 
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiUserFirstGet(openapiClientHandle_t *client_handle,
                                 open_buffdesc *username);

/*********************************************************************
* @purpose  Get next user name.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    prevUserName     @b{(input)}  Previous user name
* @param    nextUserName     @b{(output)} Next user name
*
* @returns  OPEN_E_NONE      if the next user is returned successfully.
* @returns  OPEN_E_FAIL      if no more users exist.
* @returns  OPEN_E_NOT_FOUND if previous user does not exist.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes  Maximum size for user names is determined by calling
*         openapiUserNameSizeGet(). 
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiUserNextGet(openapiClientHandle_t *client_handle,
                                open_buffdesc *prevUserName,
                                open_buffdesc *nextUserName);

/*********************************************************************
* @purpose  Search the next valid ID of SSH session
*
* @param    client_handle     @b{(input)}         Client handle from 
*                                                 registration API
* @param    sessionType       @b{(input)}         The session type which need to be returned
* @param    prevSessionID     @b{(input/output)}  Previous ID of session
*
* @returns  OPEN_E_NONE       if the next ID of session is returned successfully
* @returns  OPEN_E_NOT_FOUND  if the valid next ID wasn't found
* @returns  OPEN_E_PARAM      if invalid parameter is passed
*
* @notes    The result index will be returned as prevSessionIndex pointer
* 
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiSessionIDNextValidGet(openapiClientHandle_t *client_handle,
                                          OPEN_LOGIN_TYPE_t sessionType,
                                          uint32_t *prevSessionID);

/*********************************************************************
* @purpose  Closes the session by ID
*
* @param    client_handle  @b{(input)}         Client handle from 
*                                              registration API
* @param    sessionID      @b{(input)}         ID of session
*
* @returns  OPEN_E_NONE    if the logout of session was successful
* @returns  OPEN_E_FAIL    if the logout wasn't done
* @returns  OPEN_E_PARAM   if invalid parameter is passed
* 
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiSessionLogout(openapiClientHandle_t *client_handle,
                                  uint32_t sessionID);

/*********************************************************************
* @purpose  Get the maximum number of users supported.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    maxUsers         @b{(output)} Maximum number of users
*
* @returns  OPEN_E_NONE      if maximum number of login users is 
*                            returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiMaxUsersSupportedGet(openapiClientHandle_t *client_handle,
                                         uint32_t *maxUsers);

/*********************************************************************
* @purpose  Set the access level for an existing user.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    userName         @b{(input)}  User name buff descriptor
* @param    level            @b{(input)}  Access level: @ref OPEN_LOGIN_ACCESS_LEVELS_t
* 
* @returns  OPEN_E_NONE      if the level is set successfully for the user.
* @returns  OPEN_E_FAIL      if failed to set user level.
* @returns  OPEN_E_NOT_FOUND if the user does not exist.
* @returns  OPEN_E_INTERNAL  if internal error has occurred.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* @notes   Access level cannot be configured for "admin" user.
* @notes   Maximum size for userName is determined by calling
*          openapiUserNameSizeGet(). 
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiUserLevelSet(openapiClientHandle_t *client_handle,
                                 open_buffdesc *userName,
                                 OPEN_LOGIN_ACCESS_LEVELS_t level);

/*********************************************************************
* @purpose  Get the access level of a user.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    userName         @b{(input)}  User name
* @param    level            @b{(output)} Access level of the user.
*
* @returns  OPEN_E_NONE      if the user level is returned successfully.
* @returns  OPEN_E_FAIL      if failed to get user access level.
* @returns  OPEN_E_ERROR     if user does not exist.
* @returns  OPEN_E_INTERNAL  if internal error has occurred.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Maximum size for userName is determined by calling
*          openapiUserNameSizeGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiUserLevelGet(openapiClientHandle_t *client_handle,
                                 open_buffdesc *userName,
                                 OPEN_LOGIN_ACCESS_LEVELS_t *level);

/*********************************************************************
* @purpose  Get the maximum supported size of username.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    userNameSize     @b{(output)} Maximum size of username.
*
* @returns  OPEN_E_NONE      Maximum size of username returned successfully.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiUserNameSizeGet(openapiClientHandle_t *client_handle,
                                    uint32_t *userNameSize);

/*********************************************************************
* @purpose  Validate the username string.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    userName         @b{(input)}  User name to validate
*
* @returns  OPEN_E_NONE      if username contains valid characters.
* @returns  OPEN_E_FAIL      if username contains invalid characters.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes  Valid characters include alphanumeric, '-', '_' and ' '.
* @notes  Maximum size for userName is determined by calling
*         openapiUserNameSizeGet(). 
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiUserNameValidate(openapiClientHandle_t *client_handle,
                                     open_buffdesc *userName);

/*********************************************************************
* @purpose  Get the minimum and maximum supported sizes of encrypted password.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    passwordMinSize  @b{(output)} Minimum size of password
* @param    passwordMaxSize  @b{(output)} Maximum size of password
*
* @returns  OPEN_E_NONE      Maximum and minimum size of the password 
*                            is returned successfully.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiPasswordSizeGet(openapiClientHandle_t *client_handle,
                                    uint32_t *passwordMinSize,
                                    uint32_t *passwordMaxSize);

/*********************************************************************
* @purpose  Get the minimum and maximum password size based on 
*           encrypted or un-encrypted password.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API.
* @param    isEncrypPswd     @b{(input)}  OPEN_TRUE for encrypted password.
*                                         OPEN_FALSE for unencrypted password.
* @param    pPasswordMinSize @b{(output)} Minimum size of password.
* @param    pPasswordMaxSize @b{(output)} Maximum size of password.
*
* @returns  OPEN_E_NONE      Maximum and minimum size of the password 
*                            is returned successfully.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiUserPasswordSizeGet(openapiClientHandle_t *client_handle,
                                        OPEN_BOOL_t isEncrypPswd,
                                        uint32_t *pPasswordMinSize,
                                        uint32_t *pPasswordMaxSize);

/*********************************************************************
* @purpose  Delete user.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    userName         @b{(input)}  User name buff descriptor
*
* @returns  OPEN_E_NONE      if the user is deleted.
* @returns  OPEN_E_FAIL      if failed to delete user or
*                            if user is "admin".
* @returns  OPEN_E_NOT_FOUND if user does not exist.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    "admin" user cannot be deleted.
* @notes    Maximum size for userName is determined by calling
*           openapiUserNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiUserDelete(openapiClientHandle_t *client_handle,
                               open_buffdesc *userName);

/*********************************************************************
* @purpose  Create an Authentication List.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    accessLevel      @b{(input)}  Access levels: @ref OPEN_ACCESS_LEVEL_t
* @param    authListName     @b{(input)}  Authentication list name buff descriptor
*
* @returns  OPEN_E_NONE      if Authentication List is created.
* @returns  OPEN_E_FAIL      if failed to create Authentication List.
* @returns  OPEN_E_FULL      if maximum number of Authentication Lists have been reached.
* @returns  OPEN_E_EXISTS    if the Authentication List already exists.
* @returns  OPEN_E_INTERNAL  if internal error has occurred.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Creation of Authentication List would be complete 
*           only if methods are added successfully.
* @notes    Maximum size for authListName is determined by calling
*           openapiAuthenticationListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthenticationListCreate(openapiClientHandle_t *client_handle,
                                             OPEN_ACCESS_LEVEL_t accessLevel,
                                             open_buffdesc *authListName);

/*********************************************************************
* @purpose  Add methods to the Authentication list.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    accessLevel      @b{(input)}  Access levels: @ref OPEN_ACCESS_LEVEL_t
* @param    authListName     @b{(input)}  Authentication list name buff descriptor
* @param    authMethod       @b{(input)}  Authentication method list.
*                                         authMethod->pstart points to 
*                                         array of authentication methods.
*                                         Refer @ref OPEN_USER_MGR_AUTH_METHOD_t for
*                                         supported authentication methods.
*                                         authMethod->size holds count of 
*                                         authentication methods pointed by 
*                                         authMethod->pstart.
*
* @returns  OPEN_E_NONE      if methods are added to the Authentication List.
* @returns  OPEN_E_FAIL      if failed to add methods to Authentication List.
* @returns  OPEN_E_NOT_FOUND if Authentication List is not found.
* @returns  OPEN_E_ERROR     if invalid methods are passed or
*                            if duplicate methods are passed or
*                            if the Login and Enable methods are being removed 
*                            during reconfiguration of the methods of an Authentication 
*                            list while it is in use by access line Console.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch. 
* @notes    Maximum size for authListName is determined by calling
*           openapiAuthenticationListNameSizeGet(). 
* @notes    Maximum size for authMethod is determined by calling
*           openapiAuthenticationMethodsMaxGet(). 
* @notes    To apply Authentication List on line Console, enable/login methods are mandatory.
*           While reconfiguring the methods of a list that is applied on line Console, enable/login 
*           cannot be removed. Doing so, openapiAuthenticationMethodsAdd() will return OPEN_E_FAIL.
* @notes    Methods followed by "none" will not be added to the list.
* @notes    Creation of Authentication List would be complete only if methods are added successfully.
* @notes    Methods for Login Authentication: enable, line, local, none, radius and TACACS.
* @notes    Methods for Enable Authentication: deny, enable, line, none, radius and TACACS.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/                          
open_error_t openapiAuthenticationMethodsAdd(openapiClientHandle_t *client_handle,
                                             OPEN_ACCESS_LEVEL_t accessLevel,
                                             open_buffdesc *authListName,
                                             open_buffdesc *authMethod);

/*********************************************************************
* @purpose  Get the method(s) assigned to Authentication List.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    authLevel        @b{(input)}  Access levels: @ref OPEN_ACCESS_LEVEL_t
* @param    authListName     @b{(input)}  Buff descriptor to Accounting 
*                                         List name
* @param    methodList       @b{(output)} Order of the methods assigned to the 
*                                         Accounting List.
*                                         methodList->pstart points to 
*                                         array of authentication methods.
*                                         Refer @ref OPEN_USER_MGR_AUTH_METHOD_t for
*                                         supported authentication methods.
*                                         methodList->size holds count of 
*                                         authentication methods pointed by
*                                         methodList->pstart.
*
* @returns  OPEN_E_NONE      if method list is returned
* @returns  OPEN_E_FAIL      if the Authentication List does not exist.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Maximum size for authListName is determined by calling
*           openapiAuthenticationListNameSizeGet().
* @notes    Maximum size for methodList is determined by calling
*           openapiAuthenticationMethodsMaxGet(). 
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthenticationMethodListGet(openapiClientHandle_t *client_handle,
                                                OPEN_ACCESS_LEVEL_t authLevel,
                                                open_buffdesc *authListName,
                                                open_buffdesc *methodList);

/*********************************************************************
* @purpose  Get the first Authentication List name.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    authLevel        @b{(input)}  Access levels: @ref OPEN_ACCESS_LEVEL_t
* @param    authListName     @b{(output)} First Authentication List name
*
* @returns  OPEN_E_NONE      if the first Authentication List name is returned.
* @returns  OPEN_E_FAIL      if failed to return first Authentication List name.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Maximum size for authListName is determined by calling
*           openapiAuthenticationListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthenticationListFirstGet(openapiClientHandle_t *client_handle,
                                               OPEN_ACCESS_LEVEL_t authLevel,
                                               open_buffdesc *authListName);

/*********************************************************************
* @purpose  Get the next Authentication List name.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    authLevel        @b{(input)}  Access levels: @ref OPEN_ACCESS_LEVEL_t
* @param    prevAuthListName @b{(input)}  Previous Authentication List name
* @param    nextAuthListName @b{(output)} Next Authentication List name
*
* @returns  OPEN_E_NONE      if the next Authentication List name is returned.
* @returns  OPEN_E_FAIL      if the previous Authentication List name does not exist or
*                            if no more Authentication List names exist.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Maximum size for the auth names is determined by calling
*           openapiAuthenticationListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthenticationListNextGet(openapiClientHandle_t *client_handle,
                                              OPEN_ACCESS_LEVEL_t authLevel,
                                              open_buffdesc *prevAuthListName,
                                              open_buffdesc *nextAuthListName);
/*********************************************************************
* @purpose  Delete an Authentication List.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    accessLevel      @b{(input)}  Access levels: @ref OPEN_ACCESS_LEVEL_t
* @param    authListName     @b{(input)}  Authentication list name buff descriptor
*
* @returns  OPEN_E_NONE      if Authentication List is deleted successfully.
* @returns  OPEN_E_NOT_FOUND if Authentication List does not exist.
* @returns  OPEN_E_FAIL      if Authentication List is in use by Console/Telnet/SSH
*                            access lines.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* 
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Maximum size for authListName is determined by calling
*           openapiAuthenticationListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthenticationListDelete(openapiClientHandle_t *client_handle,
                                             OPEN_ACCESS_LEVEL_t accessLevel,
                                             open_buffdesc *authListName);

/*********************************************************************
* @purpose  Set Authentication List to an access line.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    accessLine       @b{(input)}  Access line: @ref OPEN_ACCESS_LINE_t
* @param    accessLevel      @b{(input)}  Access levels: @ref OPEN_ACCESS_LEVEL_t
* @param    authListName     @b{(input)}  Authentication list name buff descriptor
*
* @returns  OPEN_E_NONE      if Authentication List is applied to access line.
* @returns  OPEN_E_NOT_FOUND if Authentication List does not exist.
* @returns  OPEN_E_ERROR     if Authentication List being configured to 
*                            line Console does not have enable/local methods.
* @returns  OPEN_E_FAIL      if Authentication List is not configured.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Maximum size for authListName is determined by calling
*           openapiAuthenticationListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthenticationListLineSet(openapiClientHandle_t *client_handle,
                                              OPEN_ACCESS_LINE_t  accessLine,
                                              OPEN_ACCESS_LEVEL_t accessLevel,
                                              open_buffdesc *authListName);

/*********************************************************************
* @purpose  Get Authentication List assigned to an access line.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    accessLine       @b{(input)}  Access line: @ref OPEN_ACCESS_LINE_t
* @param    accessLevel      @b{(input)}  Access levels: @ref OPEN_ACCESS_LEVEL_t
* @param    authListName     @b{(output)} Authentication list name buff descriptor
*
* @returns  OPEN_E_NONE      if Authentication List assigned to access line is returned.
* @returns  OPEN_E_FAIL      if failed to return Authentication List of an access level.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Maximum size for authListName is determined by calling
*           openapiAuthenticationListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthenticationListLineGet(openapiClientHandle_t *client_handle,
                                              OPEN_ACCESS_LINE_t  accessLine,
                                              OPEN_ACCESS_LEVEL_t accessLevel,
                                              open_buffdesc *authListName);

/*********************************************************************
* @purpose  Delete Authentication List from an access line.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    accessLine       @b{(input)}  Access line: @ref OPEN_ACCESS_LINE_t
* @param    accessLevel      @b{(input)}  Access levels: @ref OPEN_ACCESS_LEVEL_t
*
* @returns  OPEN_E_NONE      if the Authentication List is deleted from access line.
* @returns  OPEN_E_FAIL      if Authentication List is not configured.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    This API sets the respective access lines to default methods.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthenticationListLineDelete(openapiClientHandle_t *client_handle,
                                                 OPEN_ACCESS_LINE_t  accessLine,
                                                 OPEN_ACCESS_LEVEL_t accessLevel);

/*********************************************************************
* @purpose  Get the maximum size of Authentication List name.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    authListNameSize @b{(output)} Authentication list name 
*                                         maximum size
*
* @returns  OPEN_E_NONE      if maximum size of Authentication List name is 
*                            returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthenticationListNameSizeGet(openapiClientHandle_t *client_handle,
                                                  uint32_t *authListNameSize);

/*********************************************************************
* @purpose  Get the maximum Authentication Methods supported.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    maxAuthMethods   @b{(output)} Maximum Authentication 
*                                         methods supported
*
* @returns  OPEN_E_NONE      if maximum number of Authentication methods 
*                            is returned successfully.
* @returns  OPEN_E_FAIL      if failed to return maximum 
*                            Authentication methods.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthenticationMethodsMaxGet(openapiClientHandle_t *client_handle,
                                                uint32_t *maxAuthMethods);


/*********************************************************************
* @purpose  Validate Authentication list name.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    authListName     @b{(input)}  Authentication list name.
*
* @returns  OPEN_E_NONE      if Authentication list name is valid.
* @returns  OPEN_E_FAIL      if the validation fails.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Valid characters include alphanumeric, '-', '_' and ' '.
* @notes    Maximum size for authListName is determined by calling
*           openapiAuthenticationListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthenticationListNameValidate(openapiClientHandle_t *client_handle,
                                                   open_buffdesc *authListName);

/*********************************************************************
* @purpose  Validate Accounting list name.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    acctListName     @b{(input)}  Accounting list name.
*
* @returns  OPEN_E_NONE      if Accounting list name is valid.
* @returns  OPEN_E_FAIL      if the validation fails.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* 
* @notes    Maximum size for acctListName is determined by calling
*           openapiAccountingListNameSizeGet().
* @notes    Valid characters include alphanumeric, '-', '_' and ' ' 
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAccountingListNameValidate(openapiClientHandle_t *client_handle,
                                               open_buffdesc *acctListName);

/*********************************************************************
* @purpose  Validate Authorization list name.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    authorListName   @b{(input)}  Authorization list name.
*
* @returns  OPEN_E_NONE      if Authorization list name is valid.
* @returns  OPEN_E_FAIL      if the validation fails.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* 
* @notes    Maximum size for authListName is determined by calling
*           openapiAuthenticationListNameSizeGet().
* @notes    Valid characters include alphanumeric, '-', '_' and ' ' 
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthorizationListNameValidate(openapiClientHandle_t *client_handle,
                                                  open_buffdesc *authorListName);

/*********************************************************************
* @purpose  Create an Authorization List.
*
* @param    client_handle         @b{(input)}  Client handle from 
*                                              registration API
* @param    type                  @b{(input)}  Authorization type: @ref OPEN_USER_MGR_AUTHOR_TYPES_t
* @param    authorListName        @b{(input)}  Buff descriptor to Authorization 
*                                              list name
*
* @returns  OPEN_E_NONE      if Authorization List name is created successfully.
* @returns  OPEN_E_FAIL      if failed to create Authorization List name.
* @returns  OPEN_E_EXISTS    if Authorization List name already exists.
* @returns  OPEN_E_FULL      if maximum number of Authorization Lists have been reached.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* 
* @notes    Maximum size for authorListName is determined by calling
*           openapiAuthorizationListNameSizeGet().
* @notes    Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthorizationListCreate(openapiClientHandle_t *client_handle,
                                            OPEN_USER_MGR_AUTHOR_TYPES_t type,
                                            open_buffdesc *authorListName);

/*********************************************************************
* @purpose  Add methods to Authorization List.
*
* @param    client_handle         @b{(input)}  Client handle from 
*                                              registration API
* @param    type                  @b{(input)}  Authorization type @ref OPEN_USER_MGR_AUTHOR_TYPES_t
* @param    authorListName        @b{(input)}  Buff descriptor to Authorization 
*                                              list name
* @param    authorMethod          @b{(input)}  Authorization method
*
* @returns  OPEN_E_NONE      if methods are added to the Authorization List.
* @returns  OPEN_E_FAIL      if failed to add methods.
* @returns  OPEN_E_ERROR     if duplicate methods are passed.
* @returns  OPEN_E_NOT_FOUND if Authorization List does not exist.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Maximum size for authorListName is determined by calling
*           openapiAuthorizationListNameSizeGet().
* @notes    Maximum size for authMethod is determined by calling
*           openapiAuthenticationMethodsMaxGet(). 
* @notes    Creation of Authorization List would be complete only if methods are added successfully.
* @notes    Methods followed by "none" will not be added to the list.
* @notes    Valid methods for Command Authorization: radius, TACACS and none.
* @notes    Valid methods for Exec Authorization: local, radius, TACACS and none.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthorizationMethodsAdd(openapiClientHandle_t *client_handle,
                                            OPEN_USER_MGR_AUTHOR_TYPES_t type,
                                            open_buffdesc *authorListName,
                                            open_buffdesc *authorMethod);

/*********************************************************************
* @purpose  Get the maximum and minimum size of Authorization List name.
*
* @param    client_handle       @b{(input)}  Client handle from 
*                                            registration API
* @param    authorListNameSize  @b{(output)} Maximum size of the Authorization 
*                                            List name
*
* @returns  OPEN_E_NONE      if Authorization List name size is 
*                            returned successfully.
* @returns  OPEN_E_FAIL      if failed to return Authorization List name size.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthorizationListNameSizeGet(openapiClientHandle_t *client_handle,
                                                 uint32_t *authorListNameSize);

/*********************************************************************
* @purpose  Get maximum Authorization Methods supported.
*
* @param    client_handle     @b{(input)}  Client handle from 
*                                          registration API
* @param    maxAuthorMethods  @b{(output)} Maximum Authorization methods 
*                                          supported.
*
* @returns  OPEN_E_NONE      if maximum number of Authorization methods 
*                            is returned successfully.
* @returns  OPEN_E_FAIL      if failed to return maximum Authorization methods.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthorizationMethodsMaxGet(openapiClientHandle_t *client_handle,
                                               uint32_t *maxAuthorMethods);

/*********************************************************************
* @purpose  Delete an Authorization List.
*
* @param    client_handle         @b{(input)}  Client handle from 
*                                              registration API
* @param    type                  @b{(input)}  Authorization type: @ref OPEN_USER_MGR_AUTHOR_TYPES_t
* @param    authorListName        @b{(input)}  Buff descriptor to 
*                                              Authorization List name
*
* @returns  OPEN_E_NONE        if Authorization List is deleted successfully.
* @returns  OPEN_E_NOT_FOUND   if Authorization List does not exist or invalid.
* @returns  OPEN_E_FAIL        if Authorization List is in use by Console/Telnet/SSH
*                              access lines.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
* 
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Maximum size for authorListName is determined by calling
*           openapiAuthorizationListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthorizationListDelete(openapiClientHandle_t *client_handle,
                                            OPEN_USER_MGR_AUTHOR_TYPES_t type,
                                            open_buffdesc *authorListName);

/*********************************************************************
* @purpose  Get the first Authorization List name.
*
* @param    client_handle         @b{(input)}  Client handle from 
*                                              registration API
* @param    type                  @b{(input)}  Authorization type: @ref OPEN_USER_MGR_AUTHOR_TYPES_t
* @param    authorListName        @b{(output)} Buff descriptor to 
*                                              Authorization List name
*
* @returns  OPEN_E_NONE        if first Authorization List is returned successfully.
* @returns  OPEN_E_FAIL        if Authorization List name does not exist.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
*
* @notes    Maximum size for authorListName is determined by calling
*           openapiAuthorizationListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthorizationListFirstGet(openapiClientHandle_t *client_handle,
                                              OPEN_USER_MGR_AUTHOR_TYPES_t type,
                                              open_buffdesc *authorListName);

/*********************************************************************
* @purpose  Get the next Authorization List name.
*
* @param    client_handle         @b{(input)}  Client handle from 
*                                              registration API
* @param    type                  @b{(input)}  Authorization type: @ref OPEN_USER_MGR_AUTHOR_TYPES_t
* @param    prevAuthorListName    @b{(input)}  Buff descriptor to previous
*                                              Authorization List name
* @param    nextAuthorListName    @b{(output)} Buff descriptor to next
*                                              Authorization List name
*
* @returns  OPEN_E_NONE        if next Authorization List is returned successfully.
* @returns  OPEN_E_FAIL        if the previous Authorization List name does not exist or 
*                              if no more Authorization List names exist.
* @returns  OPEN_E_UNAVAIL     if the Authorization feature is not supported.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
*
* @notes    Maximum size for the author names is determined by calling
*           openapiAuthorizationListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthorizationListNextGet(openapiClientHandle_t *client_handle,
                                             OPEN_USER_MGR_AUTHOR_TYPES_t type,
                                             open_buffdesc *prevAuthorListName,
                                             open_buffdesc *nextAuthorListName);

/*********************************************************************
* @purpose  Get the method(s) assigned to Authorization List.
*
* @param    client_handle         @b{(input)}  Client handle from 
*                                              registration API
* @param    type                  @b{(input)}  Authorization type @ref OPEN_USER_MGR_AUTHOR_TYPES_t
* @param    authorListName        @b{(input)}  Buff descriptor to 
*                                              Authorization List name
* @param    authorMethodList      @b{(output)} Order of the methods assigned to 
*                                              Authorization List
*
* @returns  OPEN_E_NONE        if method list associated with Authorization List 
*                              is returned.
* @returns  OPEN_E_FAIL        if Authorization List does not exist.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
*
* @notes    Maximum size for authorListName is determined by calling
*           openapiAuthorizationListNameSizeGet().
* @notes    Maximum size for authMethod is determined by calling
*           openapiAuthenticationMethodsMaxGet(). 
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthorizationMethodListGet(openapiClientHandle_t *client_handle,
                                               OPEN_USER_MGR_AUTHOR_TYPES_t type,
                                               open_buffdesc *authorListName,
                                               open_buffdesc *authorMethodList);

/*********************************************************************
* @purpose  Set Authorization List to an access line.
*
* @param    client_handle         @b{(input)}  Client handle from registration API
* @param    accessLine            @b{(input)}  Access line: @ref OPEN_ACCESS_LINE_t
* @param    type                  @b{(input)}  Authorization type: @ref OPEN_USER_MGR_AUTHOR_TYPES_t
* @param    authorListName        @b{(input)}  Buff descriptor to Authorization List name
*
* @returns  OPEN_E_NONE      if Authorization List is applied to access line.
* @returns  OPEN_E_FAIL      if Authorization List is not configured.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Maximum size for authorListName is determined by calling
*           openapiAuthorizationListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthorizationListLineSet(openapiClientHandle_t *client_handle,
                                             OPEN_ACCESS_LINE_t  accessLine,
                                             OPEN_USER_MGR_AUTHOR_TYPES_t type,
                                             open_buffdesc *authorListName);

/*********************************************************************
* @purpose  Get the Authorization List name assigned to an access level.
*
* @param    client_handle         @b{(input)}  Client handle from registration API
* @param    accessLine            @b{(input)}  Access line: @ref OPEN_ACCESS_LINE_t
* @param    type                  @b{(input)}  Authorization type: @ref OPEN_USER_MGR_AUTHOR_TYPES_t
* @param    authorListName        @b{(output)} Buff descriptor to Authorization List name
*
* @returns  OPEN_E_NONE        if the Authorization List assigned to access 
*                              line is returned.
* @returns  OPEN_E_FAIL        if the Authorization List does not exist.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
*
* @notes    Maximum size for authorListName is determined by calling
*           openapiAuthorizationListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthorizationListLineGet(openapiClientHandle_t *client_handle,
                                             OPEN_ACCESS_LINE_t accessLine,
                                             OPEN_USER_MGR_AUTHOR_TYPES_t type,
                                             open_buffdesc *authorListName);

/*********************************************************************
* @purpose  Delete Authorization List from an access line.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    accessLine       @b{(input)}  Access line: @ref OPEN_ACCESS_LINE_t
* @param    type             @b{(input)}  Authorization type: @ref OPEN_USER_MGR_AUTHOR_TYPES_t
*
* @returns  OPEN_E_NONE      if the Authorization is deleted from access line.
* @returns  OPEN_E_FAIL      if Authorization List is not configured.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Maximum size for authorListName is determined by calling
*           openapiAuthorizationListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthorizationListLineDelete(openapiClientHandle_t *client_handle,
                                                OPEN_ACCESS_LINE_t accessLine,
                                                OPEN_USER_MGR_AUTHOR_TYPES_t type);

/*********************************************************************
* @purpose  Create an Accounting List.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    acctType         @b{(input)}  Accounting type: @ref OPEN_USER_MGR_ACCT_TYPES_t
* @param    recordType       @b{(input)}  Accounting record type: @ref OPEN_USER_MGR_RECORD_TYPES_t
* @param    acctListName     @b{(input)}  Accounting list name
*
* @returns  OPEN_E_NONE      if Accounting List is created successfully.
* @returns  OPEN_E_FAIL      if failed to create Accounting List or 
*                            if failed to set record type.
* @returns  OPEN_E_EXISTS    if Accounting List already exists.
* @returns  OPEN_E_FULL      if maximum number of Accounting Lists have been reached.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Maximum size for acctListName is determined by calling
*           openapiAccountingListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAccountingListCreate(openapiClientHandle_t *client_handle,
                                         OPEN_USER_MGR_ACCT_TYPES_t acctType,
                                         OPEN_USER_MGR_RECORD_TYPES_t recordType,
                                         open_buffdesc *acctListName);

/*********************************************************************
* @purpose  Add methods to Accounting List.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    acctType         @b{(input)}  Accounting type: @ref OPEN_USER_MGR_ACCT_TYPES_t
* @param    acctListName     @b{(input)}  Accounting list name
* @param    acctMethod       @b{(input)}  Accounting method
*
* @returns  OPEN_E_NONE      if methods are added to the Accounting List.
* @returns  OPEN_E_FAIL      if failed to add methods or
*                            if invalid methods are passed.
* @returns  OPEN_E_EXISTS    if the methods are already configured or
                             if duplicate methods are configured.
* @returns  OPEN_E_NOT_FOUND if the Accounting List name is not found.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Maximum size for acctListName is determined by calling
*           openapiAccountingListNameSizeGet().
* @notes    Maximum size for acctMethod is determined by calling
*           openapiAuthenticationMethodsMaxGet(). 
* @notes    Creation of Accounting list would be complete only if methods are added successfully.
* @notes    Methods followed by "none" will not be added to the list.
* @notes    Methods for Commands Accounting: TACACS.
* @notes    Methods for Exec Accounting: RADIUS, TACACS.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAccountingMethodsAdd(openapiClientHandle_t *client_handle,
                                         OPEN_USER_MGR_ACCT_TYPES_t acctType,
                                         open_buffdesc *acctListName,
                                         open_buffdesc *acctMethod);

/*********************************************************************
* @purpose  Get the method(s) assigned to the Accounting List.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    acctType         @b{(input)}  Accounting type: @ref OPEN_USER_MGR_ACCT_TYPES_t
* @param    acctListName     @b{(input)}  Buff descriptor to Accounting List name
* @param    methodList       @b{(output)} Order of the methods assigned to the 
*                                         Accounting List.
*
* @returns  OPEN_E_NONE      if method list is returned
* @returns  OPEN_E_FAIL      if the Accounting List does not exist.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Maximum size for acctListName is determined by calling
*           openapiAccountingListNameSizeGet().
* @notes    Maximum size for methodList is determined by calling
*           openapiAuthenticationMethodsMaxGet(). 
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAccountingMethodListGet(openapiClientHandle_t *client_handle,
                                            OPEN_USER_MGR_ACCT_TYPES_t acctType,
                                            open_buffdesc *acctListName,
                                            open_buffdesc *methodList);

/*********************************************************************
* @purpose  Get the record type assigned to the Accounting List.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    acctType         @b{(input)}  Accounting type: @ref OPEN_USER_MGR_ACCT_TYPES_t
* @param    acctListName     @b{(input)}  Buff descriptor to Accounting List name
* @param    recordType       @b{(output)} Record type associated with 
*                                         Accounting list: @ref OPEN_USER_MGR_RECORD_TYPES_t
*
* @returns  OPEN_E_NONE      if Accounting record type is returned.
* @returns  OPEN_E_FAIL      if the Accounting List does not exist.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Maximum size for acctListName is determined by calling
*           openapiAccountingListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAccountingListRecordTypeGet(openapiClientHandle_t *client_handle,
                                                OPEN_USER_MGR_ACCT_TYPES_t acctType,
                                                open_buffdesc *acctListName,
                                                OPEN_USER_MGR_RECORD_TYPES_t *recordType);
                                                 
/*********************************************************************
* @purpose  Get the first Accounting List name.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    acctType         @b{(input)}  Accounting type: @ref OPEN_USER_MGR_ACCT_TYPES_t
* @param    acctListName     @b{(output)} Buff descriptor to Accounting List name
*
* @returns  OPEN_E_NONE      if first Accounting List name is returned.
* @returns  OPEN_E_FAIL      if Accounting List name does not exist.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Maximum size for acctListName is determined by calling
*           openapiAccountingListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAccountingListFirstGet(openapiClientHandle_t *client_handle,
                                           OPEN_USER_MGR_ACCT_TYPES_t acctType,
                                           open_buffdesc *acctListName);

/*********************************************************************
* @purpose  Get the next Accounting List name.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    acctType         @b{(input)}  Accounting type: @ref OPEN_USER_MGR_ACCT_TYPES_t
* @param    prevAcctListName @b{(input)}  Buff descriptor to previous 
*                                         Accounting List name
* @param    nextAcctListName @b{(output)} Buff descriptor to next 
*                                         Accounting List name
*
* @returns  OPEN_E_NONE      if next Accounting List name is returned.
* @returns  OPEN_E_FAIL      if previous Accounting List name does not exist or
*                            if no more Accounting List names exist.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Maximum size for the acct names is determined by calling
*           openapiAccountingListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAccountingListNextGet(openapiClientHandle_t *client_handle,
                                          OPEN_USER_MGR_ACCT_TYPES_t acctType,
                                          open_buffdesc *prevAcctListName,
                                          open_buffdesc *nextAcctListName);

/*********************************************************************
* @purpose  Delete an Accounting List. 
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    acctType         @b{(input)}  Accounting type: @ref OPEN_USER_MGR_ACCT_TYPES_t
* @param    acctListName     @b{(input)}  Buff descriptor to Accounting 
*                                         List name
*
* @returns  OPEN_E_NONE      if Accounting List is deleted.
* @returns  OPEN_E_FAIL      if invalid Accounting List name is passed or
*                            if the Accounting List name is not found or
*                            if Accounting List is in use by Console/Telnet/SSH
*                            access lines.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Maximum size for acctListName is determined by calling
*           openapiAccountingListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAccountingListDelete(openapiClientHandle_t *client_handle,
                                         OPEN_USER_MGR_ACCT_TYPES_t acctType,
                                         open_buffdesc *acctListName);

/*********************************************************************
* @purpose  Set Accounting List to an access line.
*
* @param    client_handle         @b{(input)}  Client handle from registration API
* @param    accessLine            @b{(input)}  Access line: @ref OPEN_ACCESS_LINE_t
* @param    acctType              @b{(input)}  Accounting type: @ref OPEN_USER_MGR_ACCT_TYPES_t
* @param    acctListName          @b{(input)}  Buff descriptor to Accounting List name
*
* @returns  OPEN_E_NONE      if Accounting List is applied to access line.
* @returns  OPEN_E_FAIL      if Accounting List name is not configured.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Maximum size for acctListName is determined by calling
*           openapiAccountingListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAccountingListLineSet(openapiClientHandle_t *client_handle,
                                          OPEN_ACCESS_LINE_t  accessLine,
                                          OPEN_USER_MGR_ACCT_TYPES_t acctType,
                                          open_buffdesc *acctListName);

/*********************************************************************
* @purpose  Get the Accounting list assigned to an access line.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    accessLine       @b{(input)}  Access line: @ref OPEN_ACCESS_LINE_t
* @param    acctType         @b{(input)}  Accounting type: @ref OPEN_USER_MGR_ACCT_TYPES_t
* @param    acctListName     @b{(output)} Buff descriptor to Accounting List name
*
* @returns  OPEN_E_NONE      if Accounting List assigned to access line is returned.
* @returns  OPEN_E_FAIL      if Accounting List is not assigned to the access line.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Maximum size for acctListName is determined by calling
*           openapiAccountingListNameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAccountingListLineGet(openapiClientHandle_t *client_handle,
                                          OPEN_ACCESS_LINE_t accessLine,
                                          OPEN_USER_MGR_ACCT_TYPES_t acctType,
                                          open_buffdesc *acctListName);

/*********************************************************************
* @purpose  Delete Accounting List from an access line.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    accessLine       @b{(input)}  Access line: @ref OPEN_ACCESS_LINE_t
* @param    acctType         @b{(input)}  Accounting type @ref OPEN_USER_MGR_ACCT_TYPES_t
*
* @returns  OPEN_E_NONE      if the Accounting is deleted from access line.
* @returns  OPEN_E_FAIL      if Accounting List is not configured.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAccountingListLineDelete(openapiClientHandle_t *client_handle,
                                             OPEN_ACCESS_LINE_t accessLine,
                                             OPEN_USER_MGR_ACCT_TYPES_t acctType);

/*********************************************************************
* @purpose  Get the maximum Accounting Methods supported.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    maxAcctMethods   @b{(output)} Maximum Accounting methods supported
*
* @returns  OPEN_E_NONE      if maximum number of Accounting methods is returned 
*                            successfully.
* @returns  OPEN_E_FAIL      if failed to return maximum Accounting methods.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAccountingMethodsMaxGet(openapiClientHandle_t *client_handle,
                                            uint32_t *maxAcctMethods);
                                            
/*********************************************************************
* @purpose  Get the maximum size of Accounting List name.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    acctListNameSize @b{(output)} Maximum size of Accounting List name
*
* @returns  OPEN_E_NONE      if maximum size of Accounting list name is returned.
* @returns  OPEN_E_FAIL      if failed to return Accounting List name size.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAccountingListNameSizeGet(openapiClientHandle_t *client_handle,
                                              uint32_t *acctListNameSize);

/*********************************************************************
* @purpose  Set enable password.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    enablePswd       @b{(input)}  Enable password.
*
* @returns  OPEN_E_NONE      if enable password is set successfully.
* @returns  OPEN_E_FAIL      if password is invalid.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Maximum size for password is determined by calling 
*           openapiUserPasswordSizeGet() with arg isEncrypPswd as OPEN_FALSE.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiEnablePasswordSet(openapiClientHandle_t *client_handle,
                                      open_buffdesc *enablePswd);

/*********************************************************************
* @purpose  Delete enable password.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
*
* @returns  OPEN_E_NONE      if enable password is deleted successfully.
* @returns  OPEN_E_FAIL      if failed to delete enable password.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiEnablePasswordRemove(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Authenticate Login User.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    userName         @b{(input)}  User name to authenticate
* @param    password         @b{(input)}  Password for the user
* @param    accessLevel      @b{(output)} Access level of the user: read-write/read-only
*
* @returns  OPEN_E_NONE      if user is authenticated.
* @returns  OPEN_E_FAIL      if user is not authenticated due to invalid username/password or
*                            if the user is configured with blocked access level
* @returns  OPEN_E_ERROR     if all configured Authentication methods failed to provide 
*                            response due to timeout, etc.
* @returns  OPEN_E_INTERNAL  if internal error has occurred.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Maximum size for userName is determined by calling
*          openapiUserNameSizeGet(). 
* @notes   Maximum size for password is determined by calling
*          openapiPasswordSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiAuthenticateUser(openapiClientHandle_t *client_handle,
                                     open_buffdesc *userName,
                                     open_buffdesc *password,
                                     OPEN_LOGIN_ACCESS_LEVELS_t *accessLevel);

/*********************************************************************
* @purpose  Create session for an existing user. 
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    userName         @b{(input)}  User name to authenticate
* @param    password         @b{(input)}  Password for the user
* @param    logintype        @b{(input)}  Login protocol HTTP/HTTPS
* @param    inetaddr         @b{(input)}  Client IP address
* @param    accessLevel      @b{(output)} Access level of the user 
* @param    sessionId        @b{(output)} Session ID of the login.
*
* @returns  OPEN_E_NONE      if user is authenticated.
* @returns  OPEN_E_FAIL      if user is not authenticated due to invalid username/password or
*                            if the user is configured with blocked access level
* @returns  OPEN_E_ERROR     if all configured Authentication methods failed to provide 
*                            response due to timeout, etc.
* @returns  OPEN_E_INTERNAL  if internal error has occurred.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_FULL      if maximum login sessions exist already.
*
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiWebSessionUserLogin(openapiClientHandle_t *client_handle,
                                        open_buffdesc *userName,
                                        open_buffdesc *password,
                                        open_buffdesc *logintype,
                                        open_buffdesc *inetaddr,
                                        uint32_t      *accessLevel,
                                        open_buffdesc *sessionId);


/****************************************************************************
* @purpose  validate session for an existing user. 
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    sessionId        @b{(input)}  Session ID of the login.
* @param    logintype        @b{(input)}  Login protocol HTTP/HTTPS
* @param    inetaddr         @b{(input)}  Client IP address
* @param    accessLevel      @b{(output)} Access Level of the session 
*
* @returns  OPEN_E_NONE      if session is valid.
* @returns  OPEN_E_FAIL      if session is not valid.
*
* @supported-in-version OpEN API Version: 1.6
*
* @end
*******************************************************************************/
open_error_t openapiWebSessionValidate(openapiClientHandle_t *client_handle,
                                       open_buffdesc *sessionId,
                                       open_buffdesc *logintype,
                                       open_buffdesc *inetaddr,
                                       uint32_t      *accessLevel);

/****************************************************************************
* @purpose  End session for an existing user. 
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    sessionId        @b{(input)} Session ID of the login.
*
* @returns  OPEN_E_NONE      if session is valid.
* @returns  OPEN_E_FAIL      if session is not valid.
*
* @supported-in-version OpEN API Version: 1.6
*
* @end
*******************************************************************************/
open_error_t openapiWebSessionLogout(openapiClientHandle_t *client_handle,
                                     open_buffdesc *sessionId);

/*********************************************************************
* @purpose  Get whether Authentication is required.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    accessLine       @b{(input)}  Access line: @ref OPEN_ACCESS_LINE_t
* @param    accessLevel      @b{(input)}  Access levels: @ref OPEN_ACCESS_LEVEL_t
* @param    authStatus       @b{(output)} Authentication Status: @ref OPEN_BOOL_t
*
* @returns  OPEN_E_NONE      if Authentication requirement status is returned.
* @returns  OPEN_E_FAIL      if failed to return Authentication requirement status.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    
*           
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIsAuthenticationRequiredGet(openapiClientHandle_t *client_handle,
                                                OPEN_ACCESS_LINE_t  accessLine,
                                                OPEN_ACCESS_LEVEL_t accessLevel,
                                                OPEN_BOOL_t *authStatus);

/*********************************************************************
* @purpose  Get the password for specified user.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    userName         @b{(input)}  User name
* @param    userPassword     @b{(output)} Password for the user
*
* @returns  OPEN_E_NONE      if user password can be get successfully.
* @returns  OPEN_E_FAIL      if user password get fail.
* @returns  OPEN_E_NOT_FOUND if user is not found.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiUserPasswordGet(openapiClientHandle_t *client_handle,
                                    open_buffdesc *userName,
                                    open_buffdesc *userPassword);

/*********************************************************************
* @purpose  Get the login session username.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    id               @b{(input)}  Session ID
* @param    name             @b{(output)} User name
*
* @returns  OPEN_E_NONE      On success.
* @returns  OPEN_E_FAIL      On failure.
* @returns  OPEN_E_PARAM     Invalid parameter was passed.
*
* @notes Minimum size of name buffer is @ref OPEN_LOGIN_SIZE.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiLoginSessionUserGet(openapiClientHandle_t *client_handle,
                                        uint32_t id,
                                        open_buffdesc *name);

/*********************************************************************
* @purpose  Get the login session remote IP address.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    id               @b{(input)}  Session ID
* @param    addr             @b{(output)} IP address
*
* @returns  OPEN_E_NONE      On success.
* @returns  OPEN_E_FAIL      On failure.
* @returns  OPEN_E_PARAM     Invalid parameter was passed.
*
* @notes 
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiLoginSessionRemoteIpAddrGet(openapiClientHandle_t *client_handle,
                                                uint32_t id,
                                                open_inet_addr_t *addr);

/*********************************************************************
* @purpose  Get the login session idle time.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    id               @b{(input)}  Session ID
* @param    timeVal          @b{(output)} Idle time in seconds.
*
* @returns  OPEN_E_NONE      On success.
* @returns  OPEN_E_FAIL      On failure.
* @returns  OPEN_E_PARAM     Invalid parameter was passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiLoginSessionIdleTimeGet(openapiClientHandle_t *client_handle,
                                            uint32_t id,
                                            uint32_t *timeVal);

/*********************************************************************
* @purpose  Get the login session time.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    id               @b{(input)}  Session ID
* @param    timeVal          @b{(output)} Session time in seconds.
*
* @returns  OPEN_E_NONE      On success.
* @returns  OPEN_E_FAIL      On failure.
* @returns  OPEN_E_PARAM     Invalid parameter was passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiLoginSessionTimeGet(openapiClientHandle_t *client_handle,
                                        uint32_t id,
                                        uint32_t *timeVal);

/*********************************************************************
* @purpose  Get the login session type.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    id               @b{(input)}  Session ID
* @param    type             @b{(output)} Session type
*
* @returns  OPEN_E_NONE      On success.
* @returns  OPEN_E_FAIL      On failure.
* @returns  OPEN_E_PARAM     Invalid parameter was passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiLoginSessionTypeGet(openapiClientHandle_t *client_handle,
                                        uint32_t id,
                                        OPEN_LOGIN_TYPE_t *type);

/*********************************************************************
* @purpose  Set the user name, password and encryption type.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    userName         @b{(input)}  User name buff descriptor
* @param    password         @b{(input)}  Password buff descriptor
* @param    encryptType      @b{(input)}  Encryption type: @ref OPEN_ENCRYPT_TYPE_t
* @param    encrypted        @b{(input)}  flag indicating whether the password param is encrypted
*
* @returns  OPEN_E_NONE        if user name, password and 
*                              encryption type are set successfully.
* @returns  OPEN_E_FAIL        if failed to set username, password and encryption type.
* @returns  OPEN_E_FULL        if no more users can be added.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
* 
* @notes   This API allows to create a new user with password and encryption type
*          and also change the password for an existing user. 
* @notes   Calling this API will change the running configuration of the switch.
* @notes   This API creates the user with default access level (RO).
* @notes   Maximum size for userName is determined by calling
*          openapiUserNameSizeGet().
* @notes   Maximum size for password is determined by calling
*          openapiPasswordSizeGet(). 
* @notes   Password accepts all the special characters except '"' and '?'.
* 
* @supported-in-version OpEN API Version: 1.21
*
* @end
*********************************************************************/
open_error_t openapiUserPasswordEncryptTypeSet(openapiClientHandle_t *client_handle,
                                               open_buffdesc *userName,
                                               open_buffdesc *password,
                                               OPEN_ENCRYPT_TYPE_t encryptType,
                                               OPEN_BOOL_t encrypted);

/*********************************************************************
* @purpose  Get the password encryption type of a user.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    userName         @b{(input)}  User name
* @param    encryptType      @b{(output)} Password encryption type 
*
* @returns  OPEN_E_NONE      if the password encryption type is returned successfully.
* @returns  OPEN_E_FAIL      if failed to get password encryption type.
* @returns  OPEN_E_ERROR     if user does not exist.
* @returns  OPEN_E_INTERNAL  if internal error has occurred.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Maximum size for userName is determined by calling
*          openapiUserNameSizeGet().
*
* @supported-in-version OpEN API Version: 1.21
*
* @end
*********************************************************************/
open_error_t openapiUserPasswordEncryptTypeGet(openapiClientHandle_t *client_handle,
                                               open_buffdesc *userName,
                                               OPEN_ENCRYPT_TYPE_t *encryptType);

/*********************************************************************
* @purpose  Set enable password and encryption type.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    enablePswd       @b{(input)}  Enable password.
* @param    encryptType      @b{(input)}  Encryption type: @ref OPEN_ENCRYPT_TYPE_t
* @param    encrypted        @b{(input)}  flag indicating whether the password param is encrypted
*
* @returns  OPEN_E_NONE      if enable password and encryption type is set successfully.
* @returns  OPEN_E_FAIL      if password is invalid.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Maximum size for password is determined by calling
*           openapiPasswordSizeGet().
* 
* @supported-in-version OpEN API Version: 1.21
*
* @end
*********************************************************************/
open_error_t openapiEnablePasswordEncryptTypeSet(openapiClientHandle_t *client_handle,
                                                 open_buffdesc *enablePswd,
                                                 OPEN_ENCRYPT_TYPE_t encryptType,
                                                 OPEN_BOOL_t encrypted);

/*********************************************************************
* @purpose  Get enable password encryption type.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    encryptType      @b{(output)} Enable password encryption type 
*
* @returns  OPEN_E_NONE      if the password encryption type is returned successfully.
* @returns  OPEN_E_FAIL      if failed to get password encryption type.
* @returns  OPEN_E_INTERNAL  if internal error has occurred.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @supported-in-version OpEN API Version: 1.21
*
* @end
*********************************************************************/
open_error_t openapiEnablePasswordEncryptTypeGet(openapiClientHandle_t *client_handle,
                                                 OPEN_ENCRYPT_TYPE_t *encryptType);

/*********************************************************************
 * @purpose   Get user's multifactor authentication state.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API.
 * @param     pUserName       @b{(input)}  user name.
 * @param     pMfaPwdEnable   @b{(output)} user level multifactor authentication enable state.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches user level multifactor authentication.
 * @notes     Maximum size for pUserName is determined by calling openapiUserNameSizeGet().
 *
 * @supported-in-version OpEN API Version: 1.1
 *
 * @end
 *****************************************************************************/
open_error_t openapiUserMultiFactorAuthStateGet(openapiClientHandle_t *client_handle, open_buffdesc *pUserName, OPEN_BOOL_t *pMfaPwdEnable);

/*****************************************************************************
 * @purpose   Set user's multifactor authentication state
 *
 * @param     client_handle   @b{(input)}  client handle from registration API.
 * @param     pUserName       @b{(input)}  user name.
 * @param     mfaPwdEnable    @b{(input)}  user level multifactor authentication state.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API allows to enable/disable user level multifactor authentication.
 * @notes     Maximum size for pUserName is determined by calling openapiUserNameSizeGet().
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.1
 *
 * @end
 *****************************************************************************/
open_error_t openapiUserMultiFactorAuthStateSet(openapiClientHandle_t *client_handle, open_buffdesc *pUserName, OPEN_BOOL_t mfaPwdEnable);

/*****************************************************************************
 * @purpose   Get user's Configured emailId information.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API.
 * @param     pUserName        @b{(input)}  user name.
 * @param     pEmailInfo       @b{(output)} email Id.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches user's Emaild information.
 * @notes     Maximum size for pUserName is determined by calling openapiUserNameSizeGet().
 *
 * @supported-in-version OpEN API Version: 1.1
 *
 * @end
 *****************************************************************************/
open_error_t openapiUserEmailIdGet(openapiClientHandle_t *client_handle, open_buffdesc *pUserName, open_buffdesc *pEmailInfo);

/*****************************************************************************
 * @purpose   Set user's emailId information.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API.
 * @param     pUserName        @b{(input)}  user name.
 * @param     pEmailInfo       @b{(input)}  email id.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API allows to configure email information for user.
 * @notes     Maximum size for pUserName is determined by calling openapiUserNameSizeGet().
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.1
 *
 * @end
 *****************************************************************************/
open_error_t openapiUserEmailIdSet(openapiClientHandle_t *client_handle, open_buffdesc *pUserName, open_buffdesc *pEmailInfo);

/*****************************************************************************
 * @purpose   Get global multifactor authentication state
 *
 * @param     client_handle   @b{(input)}  client handle from registration API.
 * @param     pMfaCnfgEnable  @b{(output)} Global multifactor authentication enable state.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches global level multifactor authentication.
 * @notes     Maximum size for pUserName is determined by calling openapiUserNameSizeGet().
 *
 * @supported-in-version OpEN API Version: 1.1
 *
 * @end
 *****************************************************************************/
open_error_t openapiGlobalMultiFactorAuthGet(openapiClientHandle_t *client_handle, OPEN_BOOL_t *pMfaCnfgEnable);

/*****************************************************************************
 * @purpose   Set global multifactor authentication state.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API.
 * @param     mfaCnfgEnable   @b{(input)}  Global multifactor authentication enable state.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 *
 * @supported-in-version OpEN API Version: 1.1
 *
 * @end
 *****************************************************************************/
open_error_t openapiGlobalMultiFactorAuthSet(openapiClientHandle_t *client_handle, OPEN_BOOL_t mfaCnfgEnable);
/*****************************************************************************
 * @purpose   Add an authentication method at the specified index in the APL.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     accessLine     @b{(input)}  APL access line
 * @param     accessLevel    @b{(input)}  APL access level
 * @param     apl_name          @b{(input)}  pointer to APL name
 * @param     order          @b{(input)}  order this method should be attempted in the APL
 * @param     auth_method         @b{(input)}  authentication method value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets add an authentication method to an APL.
 * @notes     To remove an auth method, issue a set with L7_AUTH_METHOD_UNDEFINED.
 * @notes     All auth methods will be attempted in the APL until a definitive response is found.
 * @notes     If there is an UNDEFINED entry in the middle of the APL, that entry will be skipped
 * @notes     and the remaining entries will be used.
 * @notes     For HTTP and HTTPS, the allowed methods are local, radius, TACACS and none.
 * @notes     UNDEFINED method is used to remove a valid method.
 * @notes     Additionally for HTTP, HTTPS and DOT1X only the default list is updated, irrespective of input list name.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthenticationMethodSet(openapiClientHandle_t *client_handle, OPEN_ACCESS_LINE_t accessLine, OPEN_ACCESS_LEVEL_t accessLevel, open_buffdesc *apl_name, uint32_t order, OPEN_USER_MGR_AUTH_METHOD_t auth_method);

/*****************************************************************************
 * @purpose   Returns the authentication method at the specified index in the APL.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     accessLine     @b{(input)}  APL access line
 * @param     accessLevel    @b{(input)}  APL access level
 * @param     apl_name       @b{(input)}  pointer to APL name
 * @param     order          @b{(input)}  order this method should be attempted in the APL
 * @param     auth_method    @b{(output)} authentication method value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Order begins with 0, The authentication methods should be attempted in order
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthenticationMethodGet(openapiClientHandle_t *client_handle, OPEN_ACCESS_LINE_t accessLine, OPEN_ACCESS_LEVEL_t accessLevel, open_buffdesc *apl_name, uint32_t order, OPEN_USER_MGR_AUTH_METHOD_t *auth_method);

/*****************************************************************************
 * @purpose   Validates user provided password.
 *
 * @param     client_handle  @b{(input)} client handle from registration API
 * @param     pUser          @b{(input)} user name
 * @param     pEnteredPwd    @b{(input)} user provided password
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Validates user provided password against user configured password.
 *
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiValidateUserPassword(openapiClientHandle_t *client_handle,
                                         open_buffdesc *pUser,
                                         open_buffdesc *pEnteredPwd);

/*********************************************************************
 * @purpose  Set the password lockout count.
 *
 * @param    client_handle   @b{(input)}  Client handle from registration API.
 * @param    lockoutAttempts @b{(input)}  Number of attempts after which the
 *                                        user account is locked.
 *
 * @returns  OPEN_E_NONE     If lockout count value is set successfully.
 * @returns  OPEN_E_FAIL     If lockout count value set is failed.
 * @returns  OPEN_E_PARAM    If invalid parameter is passed.
 *
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *********************************************************************/
open_error_t openapiUserLockoutAttemptsSet(openapiClientHandle_t *client_handle,
                                           uint32_t lockoutAttempts);

/*********************************************************************
 * @purpose  Get the password lockout count.
 *
 * @param    client_handle   @b{(input)}   Client handle from registration API.
 * @param    lockoutAttempts @b{(output)}  Number of attempts after which the
 *                                         user account is locked.
 *
 * @returns  OPEN_E_NONE     If lockout count value is fetched successfully.
 * @returns  OPEN_E_FAIL     If lockout count value get is failed.
 * @returns  OPEN_E_PARAM    If invalid parameter is passed.
 *
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *********************************************************************/
open_error_t openapiUserLockoutAttemptsGet(openapiClientHandle_t *client_handle,
                                           uint32_t *lockoutAttempts);

/*********************************************************************
 * @purpose  Set the time in minutes to unlock the user.
 *
 * @param    client_handle   @b{(input)}  Client handle from registration API.
 * @param    unlockTime      @b{(input)}  Time to unlock the user.
 *
 * @returns  OPEN_E_NONE     If unlock time value is set successfully.
 * @returns  OPEN_E_FAIL     If unlock time value set is failed.
 * @returns  OPEN_E_PARAM    If invalid parameter is passed.
 *
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *********************************************************************/
open_error_t openapiUserUnlockTimeSet(openapiClientHandle_t *client_handle,
                                      uint32_t unlockTime);

/*********************************************************************
 * @purpose  Get the time in minutes to unlock the user.
 *
 * @param    client_handle   @b{(input)}  Client handle from registration API.
 * @param    unlockTime      @b{(output)} Time to unlock the user.
 *
 * @returns  OPEN_E_NONE     If unlock time value is fetched successfully.
 * @returns  OPEN_E_FAIL     If unlock time value get is failed.
 * @returns  OPEN_E_PARAM    If invalid parameter is passed.
 *
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *********************************************************************/
open_error_t openapiUserUnlockTimeGet(openapiClientHandle_t *client_handle,
                                      uint32_t *unlockTime);

/*********************************************************************
 * @purpose  Set the user unlock timer mode.
 *
 * @param    client_handle   @b{(input)}  Client handle from registration API.
 * @param    unlockTimerMode @b{(input)}  Unlock timer mode.
 *
 * @returns  OPEN_E_NONE    If unlock timer mode is set successfully.
 * @returns  OPEN_E_FAIL    If unlock timer mode set is failed.
 * @returns  OPEN_E_PARAM   If invalid parameter is passed. 
 *
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *********************************************************************/
open_error_t openapiUserUnlockTimerModeSet(openapiClientHandle_t *client_handle,
                                           OPEN_CONTROL_t unlockTimerMode);

/*********************************************************************
 * @purpose  Get the user unlock timer mode.
 *
 * @param    client_handle   @b{(input)}   Client handle from registration API.
 * @param    unlockTimerMode @b{(output)}  Unlock timer mode.
 *
 * @returns  OPEN_E_NONE    If unlock timer mode is fetched successfully.
 * @returns  OPEN_E_FAIL    If unlock timer mode get is failed.
 * @returns  OPEN_E_PARAM   If invalid parameter is passed. 
 *
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *********************************************************************/
open_error_t openapiUserUnlockTimerModeGet(openapiClientHandle_t *client_handle,
                                           OPEN_CONTROL_t *unlockTimerMode);

/*********************************************************************
 * @purpose Authenticate user using current authentication method.
 *
 * @param   client_handle    @b{(input)}  Client handle from 
 *                                        registration API.
 * @param   userName         @b{(input)}  User name to authenticate.
 * @param   password         @b{(input)}  Password for the user.
 * @param   accessLine       @b{(input)}  Access line.
 * @param   remoteIpAddr     @b{(input)}  Remote IP address initiating access.
 * @param   accessLevel      @b{(output)} Access level of the user.
 *                                        read-write/read-only.
 * @param   authMethod       @b{(output)} Method using which the user
 *                                        is authenticated.
 * @param   isDefPassword    @b{(output)} Returns @ref OPEN_TRUE if the
 *                                        entered password is default password
 *                                        else @ref OPEN_FALSE for default
 *                                        users on the switch.
 *
 * @returns OPEN_E_NONE      If user is authenticated.
 * @returns OPEN_E_FAIL      If user is not authenticated due to invalid
 *                           username/password or if the user is configured
 *                           with blocked access level.
 * @returns OPEN_E_ERROR     If all configured Authentication methods failed
 *                           to provide response due to timeout, etc.
 * @returns OPEN_E_INTERNAL  If internal error has occurred.
 * @returns OPEN_E_PARAM     If invalid parameter is passed.
 *
 * 
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *********************************************************************/
open_error_t openapiUserSessionAuthentication(openapiClientHandle_t *client_handle,
                                              open_buffdesc *userName,
                                              open_buffdesc *password,
                                              OPEN_ACCESS_LINE_t accessLine,
                                              open_buffdesc *remoteIpAddr,
                                              OPEN_LOGIN_ACCESS_LEVELS_t *accessLevel,
                                              OPEN_USER_MGR_AUTH_METHOD_t *authMethod,
                                              OPEN_BOOL_t *isDefPassword);

/*********************************************************************
 * @purpose  Get the lockout status of a user.
 *
 * @param    client_handle    @b{(input)}  Client handle from 
 *                                         registration API.
 * @param    userName         @b{(input)}  User name.
 * @param    lockoutStatus    @b{(output)} Lockout status of the user.
 *
 * @returns  OPEN_E_NONE      If the user lockout status is returned
 *                            successfully.
 * @returns  OPEN_E_FAIL      If failed to get user lockout status.
 * @returns  OPEN_E_ERROR     If user does not exist.
 * @returns  OPEN_E_INTERNAL  If internal error has occurred.
 * @returns  OPEN_E_PARAM     If invalid parameter is passed.
 *
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *********************************************************************/
open_error_t openapiUserLockoutStatusGet(openapiClientHandle_t *client_handle,
                                         open_buffdesc *userName,
                                         OPEN_BOOL_t *lockoutStatus);
#endif
