/*********************************************************************
*
*  Copyright 2016-2023 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_vlan.h
*
* @purpose   VLAN Configuration and Status
*
* @component OPEN
*
* @create    08/29/2012
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_VLAN_H_INCLUDED
#define OPENAPI_VLAN_H_INCLUDED

#include <stdio.h>
#include <stddef.h>
#include <sys/un.h>

#include "openapi_common.h"

/**< Dot1q port modes. */
typedef enum
{
  OPEN_DOT1Q_SWPORT_MODE_NONE                   = 0,  /**< This is used by applications like gvrp who do not care about the mode */
  OPEN_DOT1Q_SWPORT_MODE_GENERAL                = 1,  /**< Switchport general mode */
  OPEN_DOT1Q_SWPORT_MODE_ACCESS                 = 2,  /**< Switchport access mode */
  OPEN_DOT1Q_SWPORT_MODE_TRUNK                  = 3,  /**< Switchport trunk mode */
  OPEN_DOT1Q_SWPORT_MODE_PRIVATE_HOST           = 4,  /**< Switchport private host mode */
  OPEN_DOT1Q_SWPORT_MODE_PRIVATE_PROMISC        = 5,  /**< Switchport private promisc mode */
  OPEN_DOT1Q_SWPORT_MODE_PRIVATE_PROMISC_TRUNK  = 6,  /**< Switchport private promisc trunk mode */
  OPEN_DOT1Q_SWPORT_MODE_PRIVATE_ISOLATED_TRUNK = 7,  /**< Switchport private isolated trunk mode */
  OPEN_DOT1Q_SWPORT_MODE_AUTO_TRUNK             = 8,  /**< Switchport auto trunk mode */
} OPEN_DOT1Q_SWPORT_MODE_t;

/** VLAN list */
#define OPEN_VLAN_LIST_MAX_ENTRIES 4096

typedef struct
{
  uint32_t  numEntries;                       /**< number of valid list entries */
  uint16_t  ids[OPEN_VLAN_LIST_MAX_ENTRIES];  /**< densely packed array of VLAN ID values making up the list */
} OPEN_VLAN_LIST_t;

/**< Dot1q private vlan types */
typedef enum
{
  OPEN_PRIVATE_VLAN_UNCONFIGURED = 0,    /**< Unconfigured */
  OPEN_PRIVATE_VLAN_PRIMARY,             /**< Primary */
  OPEN_PRIVATE_VLAN_SECONDARY_ISOLATED,  /**< Secondary isolated */             
  OPEN_PRIVATE_VLAN_SECONDARY_COMMUNITY, /**< Secondary community */             
} OPEN_DOT1Q_PVLAN_TYPE_t;


/**< Dot1q private vlan port types */
typedef enum
{
  OPEN_PRIVATE_VLAN_UNCONFIGURED_PORT,  /**< Unconfigured port */               
  OPEN_PRIVATE_VLAN_HOST_PORT,          /**< Host port */              
  OPEN_PRIVATE_VLAN_COMMUNITY_PORT,     /**< Community port */              
  OPEN_PRIVATE_VLAN_PROMISCOUS_PORT    /**< Promiscous port */                 
} OPEN_DOT1Q_PVLAN_PORT_TYPE_t; 

/**< VLAN types */
typedef enum
{
  OPEN_DOT1Q_VLAN_TYPE_NOT_PARTICIPATING,     /**< Not Participating */
  OPEN_DOT1Q_VLAN_TYPE_DEFAULT,               /**< Default VLAN, always valued at 1 */
  OPEN_DOT1Q_VLAN_TYPE_ADMIN,                 /**< Administratively configured, static VLAN */
  OPEN_DOT1Q_VLAN_TYPE_GVRP_REGISTERED,       /**< Dynamically configured GVRP VLAN */  
  OPEN_DOT1Q_VLAN_TYPE_WS_L2_TNNL_REGISTERED, /**< Dynamically configured L2 Tunnel VLAN */
  OPEN_DOT1Q_VLAN_TYPE_IP_INTERNAL_VLAN,      /**< Dynamically configured IP internal VLAN */
  OPEN_DOT1Q_VLAN_TYPE_AUTHMGR_REGISTERED,    /**< Dynamically configured Auth manager/DOT1X VLAN */
  OPEN_DOT1Q_VLAN_TYPE_MVRP_REGISTERED,       /**< Dynamically configured MVRP VLAN */  
  OPEN_DOT1Q_VLAN_TYPE_OPENFLOW_REGISTERED,   /**< Dynamically configured OPENFLOW VLAN */  
  OPEN_DOT1Q_VLAN_TYPE_AUTO_VIDEO_REGISTERED, /**< Auto-Video VLAN */
  OPEN_DOT1Q_VLAN_TYPE_EVC_REGISTERED,        /**< EVC VLAN */
  OPEN_DOT1Q_VLAN_TYPE_PBB_REGISTERED         /**< PBB VLAN */
} OPEN_DOT1Q_VLAN_TYPE_t;

#define OPEN_DOT1Q_UNTAGGED 0
#define OPEN_DOT1Q_TAGGED 1

/*********************************************************************
* @purpose  Gets the next VLAN number that has been configured on the
*           switch.  Call with vlanNum set to 0 to get the first VLAN.
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    vlanNum        @b{(input)}   Current VLAN ID.
* @param    nextVlanNum    @b{(output)}  Next VLAN ID, if one exists.
*
* @returns  OPEN_E_NONE        The next VLAN ID returned.
* @returns  OPEN_E_FAIL        No next VLAN ID was found.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes   Passing a zero vlanNum indicates a request to return the
*          first available VLAN.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiVlanNextGet(openapiClientHandle_t *client_handle,
                                uint32_t vlanNum,
                                uint32_t *nextVlanNum);

/*********************************************************************
* @purpose  Test if the given VLAN has been configured on the switch.
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    vlanNum        @b{(input)}   VLAN ID.
*
* @returns  OPEN_E_NONE        The VLAN is configured.
* @returns  OPEN_E_FAIL        The VLAN is not configured.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiVlanCreatedCheck(openapiClientHandle_t *client_handle,
                                     uint32_t vlanNum);

/*********************************************************************
* @purpose  Returns whether the VLAN is static.
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    vlanNum        @b{(input)}   VLAN ID.
*
* @returns  0        The VLAN is dynamic.
*           1        The VLAN is static.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
int openapiVlanIsStatic(openapiClientHandle_t *client_handle,
                        uint32_t vlanNum);

/*********************************************************************
* @purpose  Creates the VLAN with the ID specified on the
*           switch.
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    vlanNum        @b{(input)}   VLAN ID.
*
* @returns  OPEN_E_NONE        The VLAN was successfully created.
* @returns  OPEN_E_FAIL        The VLAN was not successfully created.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiVlanCreate(openapiClientHandle_t *client_handle,
                               uint32_t vlanNum);

/*********************************************************************
* @purpose  Deletes the VLAN with the ID specified on the
*           switch.
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    vlanNum        @b{(input)}   VLAN ID.
*
* @returns  OPEN_E_NONE        The VLAN was successfully deleted.
* @returns  OPEN_E_FAIL        The VLAN was not successfully deleted.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     The default VLAN deletion is not allowed.
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiVlanDelete(openapiClientHandle_t *client_handle,
                               uint32_t vlanNum);

/*********************************************************************
* @purpose  Get the Acceptable Frame Type configuration for a port
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  physical or logical interface to be configured
* @param    -               Minimum value 1
* @param    -               Maximum value openapiMaxInterfaceCountGet()
* @param    type            @b{(output)} Frame type
*
* @returns  OPEN_E_NONE     Frame type was successfully retrieved.
* @returns  OPEN_E_FAIL     Frame type was not successfully retrieved.
* @returns  OPEN_E_PARAM    Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiVlanIfAcceptFrameTypeGet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum,
                                             OPEN_DOT1Q_ACCEPT_FRAME_TYPE_t *type);

/*********************************************************************
* @purpose  Set the Acceptable Frame Type configuration for a port
*
* @param    client_handle   @b{(input)} client handle from registration API
* @param    ifNum           @b{(input)} physical or logical interface to be configured
* @param    -               Minimum value 1
* @param    -               Maximum value openapiMaxInterfaceCountGet()
* @param    type            @b{(input)} Frame Type
*
* @returns  OPEN_E_NONE     Success
* @returns  OPEN_E_FAIL     Frame type not set
* @returns  OPEN_E_PARAM    Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiVlanIfAcceptFrameTypeSet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum,
                                             OPEN_DOT1Q_ACCEPT_FRAME_TYPE_t type);

/*********************************************************************
* @purpose  Gets an interface's PVID.
*
* @param    client_handle     @b{(input)} client handle from registration API
* @param    ifNum             @b{(input)} Interface ID
* @param    -                 Minimum value 1
* @param    -                 Maximum value openapiMaxInterfaceCountGet()
* @param    vlanNum           @b{(output)} VLAN ID
*
* @returns  OPEN_E_NONE        PVID was successfully retrieved.
* @returns  OPEN_E_FAIL        PVID was not successfully retrieved.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiVlanIfPvidGet(openapiClientHandle_t *client_handle,
                                  uint32_t ifNum,
                                  uint32_t *vlanNum);

/*********************************************************************
* @purpose  Gets an interface's tagging setting for the specified VLAN.
*
* @param    client_handle     @b{(input)} client handle from registration API
* @param    ifNum             @b{(input)} Interface ID
* @param    -                 Minimum value 1
* @param    -                 Maximum value openapiMaxInterfaceCountGet()
* @param    vlanNum           @b{(input)} VLAN ID
* @param    tagging           @b{(output)} OPEN_ENABLE if port is sending tagged
*                                          packets on the VLAN
*
* @returns  OPEN_E_NONE        PVID was successfully retrieved.
* @returns  OPEN_E_FAIL        PVID was not successfully retrieved.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiVlanIfTaggingGet(openapiClientHandle_t *client_handle,
                                     uint32_t ifNum,
                                     uint32_t vlanNum,
                                     OPEN_CONTROL_t *tagging);

/*********************************************************************
* @purpose  Gets an interface's participation status for the specified
*           VLAN.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    vlanNum          @b{(input)}   VLAN ID
* @param    ifNum            @b{(input)}   Interface ID
* @param    -                Minimum value 1
* @param    -                Maximum value openapiMaxInterfaceCountGet()
* @param    participating    @b{(output)}  1 if participating, 0 if not
*
* @returns  OPEN_E_NONE        Participation status was successfully retrieved.
* @returns  OPEN_E_FAIL        Participation status was not successfully retrieved.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiVlanIfParticipationStatusGet(openapiClientHandle_t *client_handle,
                                                 uint32_t vlanNum,
                                                 uint32_t ifNum,
                                                 uint32_t *participating);

/*********************************************************************
* @purpose  Gets the next VLAN which has a pending change or a
*           pending deletion.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    vlanNum          @b{(input)}   ID of the current VLAN ID
* @param    nextVlanNum      @b{(output)}  Storage for the returned VLAN ID
* @param    delete_pending   @b{(output)}  Flag Indicating that the VLAN
*                                          is pending for deletion.
*
* @returns  OPEN_E_NONE        The next VLAN of the specified type returned.
* @returns  OPEN_E_FAIL        No next VLAN of the specified type was found.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiVlanNextChangedGet(openapiClientHandle_t *client_handle,
                                       uint32_t vlanNum,
                                       uint32_t *nextVlanNum,
                                       uint32_t *delete_pending);

/*********************************************************************
* @purpose  Sets an interface's participation mode configuration for the specified
*           VLAN.
*
* @param    client_handle     @b{(input)} client handle from registration API
* @param    vlanNum           @b{(input)}   VLAN ID
* @param    ifNum             @b{(input)}   Interface ID
* @param    -                 Minimum value 1
* @param    -                 Maximum value openapiMaxInterfaceCountGet()
* @param    participationMode @b{(input)}   Mode of participation
*
* @returns  OPEN_E_NONE        Participation mode was successfully set.
* @returns  OPEN_E_FAIL        Participation mode was not successfully set.
* @returns  OPEN_E_UNAVAIL     Setting participation mode not valid for the
*                              specified interface.
* @returns  OPEN_E_NOT_FOUND   The VLAN specified is not created.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiVlanIfParticipationSet(openapiClientHandle_t *client_handle,
                                           uint32_t vlanNum,
                                           uint32_t ifNum,
                                           OPEN_VLAN_PARTICIPATION_MODE_t participationMode);

/*********************************************************************
* @purpose  Gets an interface's configured participation mode for the specified
*           VLAN.
*
* @param    client_handle     @b{(input)} client handle from registration API
* @param    vlanNum           @b{(input)}   VLAN ID
* @param    ifNum             @b{(input)}   Interface ID
* @param    -                 Minimum value 1
* @param    -                 Maximum value openapiMaxInterfaceCountGet()
* @param    participationMode @b{(output)}   Mode of participation
*
* @returns  OPEN_E_NONE        Participation mode was successfully deleted.
* @returns  OPEN_E_FAIL        Participation mode was not successfully deleted.
* @returns  OPEN_E_NOT_FOUND   The VLAN specified is not created.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiVlanIfParticipationGet(openapiClientHandle_t *client_handle,
                                           uint32_t vlanNum,
                                           uint32_t ifNum,
                                           OPEN_VLAN_PARTICIPATION_MODE_t *participationMode);

/*********************************************************************
* @purpose  Sets an interface's VLAN ID (PVID).
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    ifNum            @b{(input)}   Interface ID
* @param    -                Minimum value 1
* @param    -                Maximum value openapiMaxInterfaceCountGet()
* @param    vlanNum          @b{(input)}   VLAN ID
*
* @returns  OPEN_E_NONE        PVID was successfully set.
* @returns  OPEN_E_FAIL        PVID was not successfully set.
* @returns  OPEN_E_UNAVAIL     Setting PVID not valid for the
*                             specified interface.
* @returns  OPEN_E_NOT_FOUND   The VLAN specified is not created.
*
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiVlanIfPvidSet(openapiClientHandle_t *client_handle,
                                  uint32_t ifNum,
                                  uint32_t vlanNum);

/*********************************************************************
* @purpose  Sets an interface's tagging mode for the given VLAN.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    ifNum            @b{(input)}   Interface ID
* @param    -                Minimum value 1
* @param    -                Maximum value openapiMaxInterfaceCountGet()
* @param    vlanNum          @b{(input)}   VLAN ID
* @param    tagging          @b{(input)}   VLAN tagging
*
* @returns  OPEN_E_NONE        Tagging mode was successfully set.
* @returns  OPEN_E_FAIL        Tagging mode was not successfully set.
* @returns  OPEN_E_UNAVAIL     Setting tagging mode not valid for the
*                             specified interface.
* @returns  OPEN_E_NOT_FOUND   The VLAN specified is not created.
*
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiVlanIfTaggingSet(openapiClientHandle_t *client_handle,
                                     uint32_t ifNum,
                                     uint32_t vlanNum,
                                     OPEN_CONTROL_t tagging);

/*********************************************************************
* @purpose  Gets an interface's number of traffic classes supported.
*
* @param    client_handle     @b{(input)} client handle from registration API
* @param    ifNum             @b{(input)} Interface ID
* @param    numTrafficClasses @b{(output)} number of traffic classes
*
* @returns  OPEN_E_NONE      Number of traffic classes was successfully retrieved.
* @returns  OPEN_E_FAIL      Number of traffic classes was not successfully retrieved.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiVlanIfNumTrafficClassesGet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum,
                                               uint32_t *numTrafficClasses);

/*********************************************************************
* @purpose  Sets an interface's default user priority. This is the priority
*           assigned to packets received untagged.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    ifNum            @b{(input)} Interface ID
* @param    priority         @b{(input)} Priority assigned to packets. Acceptable
* @param    -                Minimum value 0
* @param    -                Maximum value @ref openapiVlanIfNumTrafficClassesGet() - 1
*
* @returns  OPEN_E_NONE      Port default priority was successfully set.
* @returns  OPEN_E_FAIL      Port default priority was not successfully set.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiVlanIfDefaultUserPrioritySet(openapiClientHandle_t *client_handle,
                                                 uint32_t ifNum,
                                                 uint32_t priority);

/*********************************************************************
* @purpose  Gets an interface's default user priority. This is the priority
*           assigned to packets received untagged.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    ifNum            @b{(input)} Interface ID
* @param    priority         @b{(output)} Priority assigned to packets.
*
* @returns  OPEN_E_NONE      Port default priority was successfully retrieved.
* @returns  OPEN_E_FAIL      Port default priority was not successfully retrieved.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiVlanIfDefaultUserPriorityGet(openapiClientHandle_t *client_handle,
                                                 uint32_t ifNum,
                                                 uint32_t *priority);

/*********************************************************************
* @purpose  Gets the VLAN Name string length. This length is used to
*           allocate and initialize the open_buffdesc type argument buffers
*           used by various VLAN OpEN APIs.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    strLen          @b{(output)}  string length
*
* @returns  OPEN_E_NONE        String length Get successful.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiVlanNameLenGet(openapiClientHandle_t *client_handle,
                                   uint32_t *strLen);

/*********************************************************************
* @purpose  Configures a VLAN Name
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    vlanNum               @b{(input)}   VLAN ID
* @param    vlanName              @b{(input)}   VLAN name
*                                               Populate vlanName:pstart and size with a char buffer
*                                               containing the VLAN name
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Maximum size for vlanName is determined by calling
*           openapiVlanNameLenGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiVlanNameSet(openapiClientHandle_t *client_handle,
                                uint32_t vlanNum,
                                open_buffdesc *vlanName);

/*********************************************************************
* @purpose  Gets the VLAN Name
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    vlanNum               @b{(input)}   VLAN ID
* @param    vlanName              @b{(output)}  VLAN name
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes    Maximum size for vlanName is determined by calling
*           openapiVlanNameLenGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiVlanNameGet(openapiClientHandle_t *client_handle,
                                uint32_t vlanNum,
                                open_buffdesc *vlanName);

/*********************************************************************
* @purpose  Set the Switchport Configuration Mode for a port
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    intf             @b{(input)}  Interface Number
* @param    -                Minimum value 1
* @param    -                Maximum value openapiMaxInterfaceCountGet()
* @param    mode             @b{(input)}  New Switchport Mode to be set
*
* @returns  OPEN_E_NONE      if Switchport Mode set is successful.
* @returns  OPEN_E_FAIL      if Switchport Mode set failed.
* @returns  OPEN_E_PARAM     invalid parameter(s)
*
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiDot1qSwPortModeSet(openapiClientHandle_t *client_handle,
                                       uint32_t intf, OPEN_DOT1Q_SWPORT_MODE_t mode);

/*********************************************************************
* @purpose  Get the Switchport Configuration Mode for a port
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    intf             @b{(input)}  Interface Number
* @param    -                Minimum value 1
* @param    -                Maximum value openapiMaxInterfaceCountGet()
* @param    mode             @b{(output)} Switchport Mode
*
* @returns  OPEN_E_NONE      if Switchport Mode get is successful.
* @returns  OPEN_E_FAIL      if Switchport Mode get failed.
* @returns  OPEN_E_PARAM     invalid parameter(s)
*
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiDot1qSwPortModeGet(openapiClientHandle_t *client_handle,
                                       uint32_t intf, OPEN_DOT1Q_SWPORT_MODE_t *mode);

/*********************************************************************
* @purpose  Set the Access VLAN ID for a port.
*
* @param    client_handle    @b{(input)}   Client handle from
*                                          registration API
* @param    intf             @b{(input)}   Interface Number
* @param    -                Minimum value 1
* @param    -                Maximum value openapiMaxInterfaceCountGet()
* @param    vlanId           @b{(input)}   VLAN ID to be set
*
* @returns  OPEN_E_NONE      if Access VLAN set is successful.
* @returns  OPEN_E_FAIL      if Access VLAN set failed.
* @returns  OPEN_E_PARAM     invalid parameter(s)
*
* @supported-in-version OpEN API Version:    1.11
*
* @end
*********************************************************************/
open_error_t openapiDot1qSwPortAccessVlanSet(openapiClientHandle_t *client_handle,
                                             uint32_t intf, uint32_t vlanId);

/*********************************************************************
* @purpose  Get the Access VLAN ID for a port.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    intf             @b{(input)}  Interface Number
* @param    -                Minimum value 1
* @param    -                Maximum value openapiMaxInterfaceCountGet()
* @param    vlanId           @b{(output)} Storage for VLAN ID
*
* @returns  OPEN_E_NONE      if Access VLAN get is successful.
* @returns  OPEN_E_FAIL      if Access VLAN get failed.
* @returns  OPEN_E_PARAM     invalid parameter(s)
*
* @supported-in-version OpEN API Version:    1.11
*
* @end
*********************************************************************/
open_error_t openapiDot1qSwPortAccessVlanGet(openapiClientHandle_t *client_handle,
                                             uint32_t intf, uint32_t *vlanId);

/*********************************************************************
* @purpose  Set the native VLAN ID for a port when in trunking mode.
*
* @param    client_handle    @b{(input)} Client handle from
*                                        registration API
* @param    intf             @b{(input)} Interface Number
* @param    -                Minimum value 1
* @param    -                Maximum value openapiMaxInterfaceCountGet()
* @param    vlanId           @b{(input)} VLAN ID to be set
*
* @returns  OPEN_E_NONE      if native VLAN set is successful.
* @returns  OPEN_E_FAIL      if native VLAN set failed.
* @returns  OPEN_E_PARAM     invalid parameter(s)
* @returns  OPEN_E_DISABLED  when specified VLAN is not in allowed VLAN list
*
* @supported-in-version OpEN API Version:    1.11
*
* @end
*********************************************************************/
open_error_t openapiDot1qSwPortTrunkNativeVlanSet(openapiClientHandle_t *client_handle,
                                                  uint32_t intf, uint32_t vlanId);

/*********************************************************************
* @purpose  Get the Native VLAN ID for a port when in trunking mode.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    intf             @b{(input)}  Interface Number
* @param    -                Minimum value 1
* @param    -                Maximum value openapiMaxInterfaceCountGet()
* @param    vlanId           @b{(output)} Storage space for the VLAN ID
*
* @returns  OPEN_E_NONE      if Native VLAN get is successful.
* @returns  OPEN_E_FAIL      if Native VLAN get failed.
* @returns  OPEN_E_PARAM     invalid parameter(s)
*
* @supported-in-version OpEN API Version:    1.11
*
* @end
*********************************************************************/
open_error_t openapiDot1qSwPortTrunkNativeVlanGet(openapiClientHandle_t *client_handle,
                                                  uint32_t intf, uint32_t *vlanId);

/*********************************************************************
* @purpose  Set the list of VLANs allowed on an interface when the
*           interface is in trunking mode.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    intf             @b{(input)}  Interface Number
* @param    vidList          @b{(input)}  VLAN ID List
*
* @returns  OPEN_E_NONE      if Allowed VLAN list set is successful.
* @returns  OPEN_E_FAIL      if Allowed VLAN list set failed.
* @returns  OPEN_E_PARAM     invalid parameter(s)
*
* @supported-in-version OpEN API Version:    1.11
*
* @end
*********************************************************************/
open_error_t openapiDot1qSwPortTrunkAllowedVlanListSet(openapiClientHandle_t *client_handle,
                                                       uint32_t intf,
                                                       OPEN_VLAN_LIST_t *vidList);

/*********************************************************************
* @purpose  Get the list of VLANs allowed on an interface when the
*           interface is in trunking mode.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intf             @b{(input)}  Interface Number
* @param    vidList          @b{(output)} VLAN ID List
*
* @returns  OPEN_E_NONE      if Allowed VLAN list get is successful.
* @returns  OPEN_E_FAIL      if Allowed VLAN list get failed.
* @returns  OPEN_E_PARAM     invalid parameter(s)
*
* @supported-in-version OpEN API Version:    1.11
*
* @end
*********************************************************************/
open_error_t openapiDot1qSwPortTrunkAllowedVlanListGet(openapiClientHandle_t *client_handle,
                                                       uint32_t intf,
                                                       OPEN_VLAN_LIST_t *vidList);

/*********************************************************************
* @purpose  Get the forbidden VLAN list for the interface when the interface
*           is in general mode.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intf             @b{(input)}  Interface Number
* @param    vidList          @b{(output)} VLAN ID List
*
* @returns  OPEN_E_NONE      if Forbidden VLAN list get is successful.
* @returns  OPEN_E_FAIL      if Forbidden VLAN list get failed.
* @returns  OPEN_E_PARAM     invalid parameter(s)
*
* @supported-in-version OpEN API Version:    1.11
*
* @end
*********************************************************************/
open_error_t openapiDot1qSwPortForbiddenVlanListGet(openapiClientHandle_t *client_handle,
                                                    uint32_t intf,
                                                    OPEN_VLAN_LIST_t *vidList);

/*********************************************************************
* @purpose  Get the untagged VLAN list for the interface when the interface
*           is in general mode.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intf             @b{(input)}  Interface Number
* @param    vidList          @b{(output)} VLAN ID List
*
* @returns  OPEN_E_NONE      if Untagged VLAN list get is successful.
* @returns  OPEN_E_FAIL      if Untagged VLAN list get failed.
* @returns  OPEN_E_PARAM     invalid parameter(s)
*
* @supported-in-version OpEN API Version:    1.11
*
* @end
*********************************************************************/
open_error_t openapiDot1qSwPortUntaggedVlanListGet(openapiClientHandle_t *client_handle,
                                                   uint32_t intf,
                                                   OPEN_VLAN_LIST_t *vidList);

/*********************************************************************
* @purpose  Get the tagged VLAN list for the interface when the interface
*           is in general mode.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intf             @b{(input)}  Interface Number
* @param    vidList          @b{(output)} VLAN ID List
*
* @returns  OPEN_E_NONE    if Tagged VLAN list get is successful.
* @returns  OPEN_E_FAIL    if Tagged VLAN list get failed.
* @returns  OPEN_E_PARAM   invalid parameter(s)
*
* @supported-in-version OpEN API Version:    1.11
*
* @end
*********************************************************************/
open_error_t openapiDot1qSwPortTaggedVlanListGet(openapiClientHandle_t *client_handle,
                                                 uint32_t intf,
                                                 OPEN_VLAN_LIST_t *vidList);

/*********************************************************************
* @purpose  Get the dynamically added VLAN list for the interface when the interface
*           is in general mode.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intf             @b{(input)}  Interface Number
* @param    vidList          @b{(output)} VLAN ID List
*
* @returns  OPEN_E_NONE    if Dynamically Added VLAN list get is successful.
* @returns  OPEN_E_FAIL    if Dynamically Added VLAN list get failed.
* @returns  OPEN_E_PARAM   invalid parameter(s)
*
* @supported-in-version OpEN API Version:    1.11
*
* @end
*********************************************************************/
open_error_t openapiDot1qSwPortDynamicallyAddedVlanListGet(openapiClientHandle_t *client_handle,
                                                           uint32_t intf,
                                                           OPEN_VLAN_LIST_t *vidList);

/*********************************************************************
* @purpose  Get the private VLAN mode for an interface.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    intf             @b{(input)}  Interface Number
* @param    -                Minimum value 1
* @param    -                Maximum value openapiMaxInterfaceCountGet()
* @param    mode             @b{(output)} Private VLAN Mode
*
* @returns  OPEN_E_NONE      PVLAN mode get successful.
* @returns  OPEN_E_FAIL      PVLAN mode get failed.
* @returns  OPEN_E_NOT_FOUND Interface not configured as a PVLAN.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.18
* 
* @comments This is similar to openapiDot1qSwPortModeGet(), but is focused
*           exclusively on the private VLAN mode configuration of an interface.
*
* @end
*********************************************************************/
open_error_t openapiPvlanIntfModeGet(openapiClientHandle_t *client_handle,
                                     uint32_t intf,
                                     OPEN_DOT1Q_SWPORT_MODE_t *mode);

/*********************************************************************
* @purpose  Get the private VLAN association for a host mode interface.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intf             @b{(input)}  Interface Number
* @param    -                Minimum value 1
* @param    -                Maximum value openapiMaxInterfaceCountGet()
* @param    primaryVid       @b{(output)} Primary VLAN ID of the Private VLAN
* @param    secondaryVid     @b{(output)} Secondary VLAN ID of the Private VLAN
*
* @returns  OPEN_E_NONE      Host mode PVLAN bindings get successful.
* @returns  OPEN_E_FAIL      Host mode PVLAN bindings get failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @supported-in-version OpEN API Version:    1.18
* 
* @comments An output value of 0 for the primaryVid and/or secondaryVid means
*           there is no host VLAN association configured for this interface.
*
* @end
*********************************************************************/
open_error_t openapiPvlanIntfHostAssocGet(openapiClientHandle_t *client_handle,
                                          uint32_t intf,
                                          uint32_t *primaryVid,
                                          uint32_t *secondaryVid);

/*********************************************************************
* @purpose  Get the private VLAN association for a promiscuous mode interface.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intf             @b{(input)}  Interface Number
* @param    -                Minimum value 1
* @param    -                Maximum value openapiMaxInterfaceCountGet()
* @param    primaryVid       @b{(output)} Primary VLAN ID of the Private VLAN
* @param    secondaryVidList @b{(output)} Secondary VLAN ID List that is mapped to primary VLAN ID
*
* @returns  OPEN_E_NONE      Promiscuous mode PVLAN bindings get successful.
* @returns  OPEN_E_FAIL      Promiscuous mode PVLAN bindings get failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @supported-in-version OpEN API Version:    1.18
*
* @comments An output value of 0 for the primaryVid and/or no entries in the secondaryVidList
*           means there is no promiscuous VLAN association for this interface.
*
* @end
*********************************************************************/
open_error_t openapiPvlanIntfPromiscAssocGet(openapiClientHandle_t *client_handle,
                                             uint32_t intf,
                                             uint32_t *primaryVid,
                                             OPEN_VLAN_LIST_t *secondaryVidList);

/*********************************************************************
* @purpose  Get the list of operational VLAN(s) for a private VLAN interface.
*
* @param    client_handle      @b{(input)}  Client handle from registration API
* @param    intf               @b{(input)}  Interface Number
* @param    operationalVidList @b{(output)} Operational VLAN ID List carried by PVLAN trunk
*
* @returns  OPEN_E_NONE      Operational VLAN list get successful.
* @returns  OPEN_E_FAIL      Operational VLAN list get failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @supported-in-version OpEN API Version:    1.18
* 
* @comments The VLANs that are in operation for an interface depend on its
*           currently configured private VLAN mode and type.
*
* @end
*********************************************************************/
open_error_t openapiPvlanIntfOperAssocGet(openapiClientHandle_t *client_handle,
                                          uint32_t intf,
                                          OPEN_VLAN_LIST_t *operationalVidList);

/*********************************************************************
* @purpose  Get the next primary VLAN and any associated secondary
*           VLAN(s) for a promiscuous trunk interface in a private VLAN.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intf             @b{(input)}  Interface Number
* @param    primaryVid       @b{(input)}  Primary VLAN ID to begin search
* @param    nextPrimaryVid   @b{(output)} Next Primary VLAN ID associated with this trunk
* @param    secondaryVidList @b{(output)} Secondary VLAN ID List that is mapped to next primary VLAN ID
*
* @returns  OPEN_E_NONE      Promiscuous trunk association next get successful.
* @returns  OPEN_E_FAIL      Promiscuous trunk association next get failed.
* @returns  OPEN_E_NOT_FOUND No more instances available.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @supported-in-version OpEN API Version:    1.18
* 
* @comments This API is meaningful only for an interface configured as a promiscuous
*           trunk, as indicated by openapiPvlanIntfModeGet().
*
* @comments Use 0 for the primaryVid to obtain the first associated primary VLAN.
* 
* @end
*********************************************************************/
open_error_t openapiPvlanIntfPromiscTrunkAssocNextGet(openapiClientHandle_t *client_handle,
                                                      uint32_t intf,
                                                      uint32_t primaryVid,
                                                      uint32_t *nextPrimaryVid,
                                                      OPEN_VLAN_LIST_t *secondaryVidList);

/*********************************************************************
* @purpose  Get the next primary VLAN and any associated secondary
*           VLAN for an isolated trunk interface in a private VLAN.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intf             @b{(input)}  Interface Number
* @param    primaryVid       @b{(input)}  Primary VLAN ID to begin search
* @param    nextPrimaryVid   @b{(output)} Next Primary VLAN ID associated with this trunk
* @param    secondaryVid     @b{(output)} Secondary VLAN ID that is mapped to the next primary VLAN ID
*
* @returns  OPEN_E_NONE      Isolated trunk association next get successful.
* @returns  OPEN_E_FAIL      Isolated trunk association next get failed.
* @returns  OPEN_E_NOT_FOUND No more instances available.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @supported-in-version OpEN API Version:    1.18
* 
* @comments This API is meaningful only for an interface configured as an isolated
*           trunk, as indicated by openapiPvlanIntfModeGet().
*
* @comments Use 0 for the primaryVid to obtain the first associated primary VLAN.
* 
* @end
*********************************************************************/
open_error_t openapiPvlanIntfIsolatedTrunkAssocNextGet(openapiClientHandle_t *client_handle,
                                                       uint32_t intf,
                                                       uint32_t primaryVid,
                                                       uint32_t *nextPrimaryVid,
                                                       uint32_t *secondaryVid);

/*********************************************************************
* @purpose  Get the native VLAN for a promiscuous or isolated trunk interface
*           in a private VLAN.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intf             @b{(input)}  Interface Number
* @param    nativeVid        @b{(output)} Native VLAN ID
*
* @returns  OPEN_E_NONE      PVLAN trunk native VLAN get successful.
* @returns  OPEN_E_FAIL      PVLAN trunk native VLAN get failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @supported-in-version OpEN API Version:    1.18
* 
* @comments This API is meaningful only for an interface configured as a promiscuous
*           or isolated trunk, as indicated by openapiPvlanIntfModeGet().
* 
* @comments The trunk native VLAN is the one assigned to untagged packets received
*           on the specified trunk interface in a private VLAN. An output value of 0 means
*           the native VLAN is not currently configured for this trunk interface.
*
* @end
*********************************************************************/
open_error_t openapiPvlanIntfTrunkNativeVlanGet(openapiClientHandle_t *client_handle,
                                                uint32_t intf,
                                                uint32_t *nativeVid);

/*********************************************************************
* @purpose  Get the list of normal VLAN(s) carried by a promiscuous or
*           isolated trunk interface in a private VLAN.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intf             @b{(input)}  Interface Number
* @param    normalVidList    @b{(output)} Normal VLAN ID List carried by PVLAN trunk
*
* @returns  OPEN_E_NONE      PVLAN trunk normal VLAN list get successful.
* @returns  OPEN_E_FAIL      PVLAN trunk normal VLAN list get failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @supported-in-version OpEN API Version:    1.18
* 
* @comments This API is meaningful only for an interface configured as a promiscuous
*           or isolated trunk, as indicated by openapiPvlanIntfModeGet().
*
* @end
*********************************************************************/
open_error_t openapiPvlanIntfTrunkAllowedVlanListGet(openapiClientHandle_t *client_handle,
                                                     uint32_t intf,
                                                     OPEN_VLAN_LIST_t *normalVidList);

/*********************************************************************
* @purpose  Setup a VLAN as private VLAN
*
* @param    client_handle         @b{(input)}  client handle from registration API
* @param    vlanId                @b{(input)}  VLAN ID 
* @param    pvlanType             @b{(input)}  private VLAN type
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiDot1qPrivateVlanEnable(openapiClientHandle_t *client_handle,
                                           uint32_t vlanId,
                                           OPEN_DOT1Q_PVLAN_TYPE_t pvlanType);

/*********************************************************************
* @purpose  Get the configured private VLAN type
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    vlanId             @b{(input)}  VLAN ID 
* @param    pvlanType          @b{(output)} private VLAN type
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiDot1qPrivateVlanGet(openapiClientHandle_t *client_handle,
                                        uint32_t vlanId,
                                        OPEN_DOT1Q_PVLAN_TYPE_t *pvlanType);

/*********************************************************************
* @purpose  Setup a VLAN bindings for private VLAN domain
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    vlanId             @b{(input)}  VLAN ID 
* @param    vidList            @b{(input)}  Secondary VLAN list
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiDot1qApiPrivateVlanAssocAdd(openapiClientHandle_t *client_handle,
                                                uint32_t vlanId,
                                                OPEN_VLAN_LIST_t vidList);

/*********************************************************************
* @purpose  Remove VLAN bindings for private VLAN domain
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    vlanId             @b{(input)}  VLAN ID 
* @param    vidList            @b{(input)}  Secondary VLAN list
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiDot1qApiPrivateVlanAssocRemove(openapiClientHandle_t *client_handle,
                                                   uint32_t vlanId,
                                                   OPEN_VLAN_LIST_t vidList);

/*********************************************************************
* @purpose  Get VLAN bindings for private VLAN domain
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    vlanId             @b{(input)}  VLAN ID 
* @param    vidList            @b{(output)} Secondary VLAN list
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiDot1qPrivateVlanAssociationGet(openapiClientHandle_t *client_handle,
                                                   uint32_t vlanId,
                                                   OPEN_VLAN_LIST_t *vidList);

/*********************************************************************
* @purpose  Set VLAN bindings for private VLAN interface
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    ifNum              @b{(input)}  internal interface number
* @param    primary_vlan       @b{(input)}  primary VLAN ID 
* @param    sec_vlan           @b{(input)}  Secondary VLAN ID
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiDot1qIntfPVlanHostAssocSet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum, 
                                               uint32_t primary_vlan, 
                                               uint32_t sec_vlan);

/*********************************************************************
* @purpose  Clear VLAN bindings for private VLAN interface 
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    ifNum              @b{(input)}  internal interface number
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiDot1qPVlanIntfHostAssocRemove(openapiClientHandle_t *client_handle,
                                                  uint32_t ifNum);

/*********************************************************************
* @purpose  Set VLAN bindings for private VLAN interface 
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    ifNum              @b{(input)}  internal interface number
* @param    primary_vlan       @b{(input)}  primary VLAN ID 
* @param    vidList            @b{(input)}  Secondary VLAN ID List
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiDot1qIntfPVlanPromiscAssocSet(openapiClientHandle_t *client_handle,
                                                  uint32_t ifNum, 
                                                  uint32_t primary_vlan, 
                                                  OPEN_VLAN_LIST_t vidList);

/*********************************************************************
* @purpose  Clear VLAN bindings for private VLAN interface
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    ifNum              @b{(input)}  internal interface number
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiDot1qPVlanIntfPromiscAssocRemove(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum);

 /*********************************************************************
* @purpose  To add a mac to a VLAN.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    mac           @b{(input)}  Mac address
* @param    vlanId        @b{(input)}  Vlan id
*
* @returns  OPEN_E_NONE        Addition of MAC to VLAN is successful.
* @returns  OPEN_E_FAIL        Addition of MAC to VLAN is failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* @returns  OPEN_E_UNAVAIL     Feature not available.
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    MAC address string should be in one of the below formats:
*           "xx:xx:xx:xx:xx:xx", "xx-xx-xx-xx-xx-xx", "xxxx.xxxx.xxxx" 
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
**********************************************************************/
open_error_t openapiVlanMacAdd(openapiClientHandle_t *client_handle,
                               open_buffdesc *mac, uint32_t vlanId);

/*********************************************************************
* @purpose  To delete a mac from a VLAN.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    mac           @b{(input)}  Mac address
* @param    vlanId        @b{(input)}  Vlan id
*
* @returns  OPEN_E_NONE        Deletion of MAC to VLAN is successful.
* @returns  OPEN_E_FAIL        Deletion of MAC to VLAN is failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* @returns  OPEN_E_UNAVAIL     Feature not available.
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    MAC address string should be in one of the below formats:
*           "xx:xx:xx:xx:xx:xx", "xx-xx-xx-xx-xx-xx", "xxxx.xxxx.xxxx" 
* 
* @supported-in-version OpEN API Version: 1.25
*
* @end
**********************************************************************/
open_error_t openapiVlanMacDelete(openapiClientHandle_t *client_handle,
                                  open_buffdesc *mac, uint32_t vlanId);

/*********************************************************************
* @purpose  To get the details of the mac vlan entry.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    mac           @b{(input)}  Mac address
* @param    vlanId        @b{(output)} vlan id
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* @returns  OPEN_E_UNAVAIL     Feature not available.
*
* @notes    MAC address string should be in one of the below formats:
*           "xx:xx:xx:xx:xx:xx", "xx-xx-xx-xx-xx-xx", "xxxx.xxxx.xxxx" 
* 
* @supported-in-version OpEN API Version: 1.25
*
* @end
**********************************************************************/
open_error_t openapiVlanMacGet(openapiClientHandle_t *client_handle,
                               open_buffdesc *mac, uint32_t *vlanId);

/*********************************************************************
* @purpose  Gets the next MAC vlan entry details of a given MAC.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    mac           @b{(input)}  Mac address
* @param    nextmac       @b{(output)} Next created mac
* @param    nextvlanId    @b{(output)} Next vlan id
*
* @returns  OPEN_E_NONE        Get next successful.
* @returns  OPEN_E_FAIL        Get next failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Feature not available.
*
* @notes    MAC address string should be in one of the below formats:
*           "xx:xx:xx:xx:xx:xx", "xx-xx-xx-xx-xx-xx", "xxxx.xxxx.xxxx" 
* @notes    nextmac buffer should have 18 bytes memory.
* 
* @supported-in-version OpEN API Version: 1.25
*
* @end
**********************************************************************/
open_error_t openapiVlanMacGetNext(openapiClientHandle_t *client_handle,
                                   open_buffdesc *mac, open_buffdesc *nextmac,
                                   uint32_t *nextvlanId);

/*********************************************************************
*
* @purpose  Get the Auto-Trunk administrative mode.
*
* @param client_handle  @b{(input)}  client handle from registration API.
* @param adminMode      @b{(output)} auto-trunk administrative mode value.
*
* @returns  OPEN_E_NONE     Auto-Trunk global admin mode get
*                           successfully.
* @returns  OPEN_E_PARAM    Error in parameter passed.
* @returns  OPEN_E_UNAVAIL  Auto-Trunk Feature not present.
* @returns  OPEN_E_INTERNAL Internal application failure.
*
* @supported-in-version OpEN API Version: 1.26
*
* @comments  none
*
* @end
*********************************************************************/
open_error_t openapiVlanGlobalAutoTrunkGet(openapiClientHandle_t *client_handle,
                                           OPEN_CONTROL_t *adminMode);

/*********************************************************************
*
* @purpose  Set the Auto-Trunk administrative mode.
*
* @param client_handle  @b{(input)} client handle from registration API.
* @param adminMode      @b{(input)} auto-trunk administrative mode value.
*
* @returns  OPEN_E_NONE     Auto-Trunk global admin mode set
*                           successfully.
* @returns  OPEN_E_PARAM    Error in parameter passed.
* @returns  OPEN_E_UNAVAIL  Auto-Trunk Feature not present.
* @returns  OPEN_E_INTERNAL Internal application failure.
*
* @supported-in-version OpEN API Version: 1.26
*
* @comments  none
*
* @end
*********************************************************************/
open_error_t openapiVlanGlobalAutoTrunkSet(openapiClientHandle_t *client_handle,
                                           OPEN_CONTROL_t adminMode);

/*********************************************************************
*
* @purpose  Get the operational auto-trunk mode of the interface.
*
* @param client_handle     @b{(input)}  client handle from registration API.
* @param intIfNum          @b{(input)}  internal interface number.
* @param autoTrunkOperMode @b{(output)} operational auto-trunk mode.
*
* @returns  OPEN_E_NONE               Operational auto-trunk get successfully.
* @returns  OPEN_E_FAIL               Failed to get operational auto-trunk mode.
* @returns  OPEN_E_PARAM              Error in parameter passed.
* @returns  OPEN_E_UNAVAIL            Auto-Trunk Feature not present.
* @returns  OPEN_E_INTERNAL           Internal application failure.
*
* @supported-in-version OpEN API Version: 1.26
*
* @comments  none
*
* @end
*********************************************************************/
open_error_t openapiDot1qOperAutoTrunkModeGet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum,
                                              OPEN_BOOL_t *autoTrunkOperMode);

/*****************************************************************************
 * @purpose   Get dot1q Max Supported VLANs.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     val            @b{(output)} max supported VLAN count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1qMaxSupportedVlansGet(openapiClientHandle_t *client_handle, 
                                              uint32_t *val);

/*****************************************************************************
 * @purpose   Get the max VLAN entries allowed.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     val            @b{(output)} number of VLANs in use on device
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiMostVlanEntriesGet(openapiClientHandle_t *client_handle, 
                                       uint32_t *val);

/*****************************************************************************
 * @purpose   Get the number of static VLANs.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     val            @b{(output)} number of static VLANs
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiStaticVlanEntriesGet(openapiClientHandle_t *client_handle, 
                                         uint32_t *val);

/*****************************************************************************
 * @purpose   Get the number of dynamic VLANs.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     val            @b{(output)} number of dynamic VLANs
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDynamicVlanEntriesGet(openapiClientHandle_t *client_handle,
                                          uint32_t *val);

/*****************************************************************************
 * @purpose   Get the number of times a VLAN entry has been deleted from DOT1Q 
 *            table.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     val            @b{(output)} number of times VLANs are deleted
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1qVlanNumDeletesGet(openapiClientHandle_t *client_handle,
                                           uint32_t *val);

/*****************************************************************************
 * @purpose   Get the tagging configuration of a port.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN Id
 * @param     ifNum          @b{(input)}  internal interface number of physical
 *                                        or logical interface
 * @param     val            @b{(output)} tagged status of a port - tagged or 
 *                                        untagged
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVlanTaggedSetGet(openapiClientHandle_t *client_handle, 
                                     uint32_t vlanId, uint32_t ifNum, 
                                     uint32_t *val);

/*****************************************************************************
 * @purpose   Gets the next VLAN ID.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vid            @b{(input)}  current VLAN Id
 * @param     nextVid        @b{(output)} next VLAN id
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiNextVlanGet(openapiClientHandle_t *client_handle, 
                                uint32_t vid, uint32_t *nextVid);

/*****************************************************************************
 * @purpose   Get the Ingress Filtering configuration for a port.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  internal interface number of physical
 *                                        or logical interface
 * @param     val            @b{(output)} enable (1) or disable (0)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiQportsEnableIngressFilteringGet(openapiClientHandle_t *client_handle, 
                                                    uint32_t ifNum, 
                                                    uint32_t *val);

/*****************************************************************************
 * @purpose   Set the Ingress Filtering configuration for a port.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  internal interface number of physical
 *                                        or logical interface or all interfaces 
 * @param     mode           @b{(input)}  enable (1) or disable (0)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiQportsEnableIngressFilteringSet(openapiClientHandle_t *client_handle, 
                                                    uint32_t ifNum, 
                                                    uint32_t mode);

/*****************************************************************************
 * @purpose   Get VLAN ingress filtering status per port.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  internal interface number of physical
 *                                        or logical interface
 * @param     val            @b{(output)} ingress filter mode - enable (1) or disable (0)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1qPortIngressFiltering(openapiClientHandle_t *client_handle, 
                                              uint32_t ifNum, uint32_t *val);

/*****************************************************************************
 * @purpose   Get the Current Ingress Filtering configuration for a port.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  internal interface number of physical
 *                                        or logical interface
 * @param     val            @b{(output)} ingress filter mode - enable (1) 
 *                                        or disable (0)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiQportsCurrentIngressFilteringGet(openapiClientHandle_t *client_handle, 
                                                     uint32_t ifNum, 
                                                     uint32_t *val);

/*****************************************************************************
 * @purpose   Get a list of VLANs an interface is a member of.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     intfNum        @b{(input)}  internal interface number of physical
 *                                        or logical interface
 * @param     data           @b{(output)} pointer to data
 * @param     numOfElements  @b{(output)} length of the VLAN list
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVlanVIDListGet(openapiClientHandle_t *client_handle, 
                                   uint32_t intfNum, OPEN_VLAN_LIST_t *data, 
                                   uint32_t *numOfElements);

/*****************************************************************************
 * @purpose  Get the total number of received packets that are not forwarded 
 *           of the interafce.
 *
 * @param    client_handle           @b{(input)}  Client handle from 
 *                                                registration API.
 * @param    ifNum                   @b{(input)}  Internal interface number
 * @param    totalRxPktsNotForwarded @b{(output)} Total number of received 
 *                                                packets that are not forwarded.
 *
 * @returns  OPEN_E_NONE             On success.
 * @returns  OPEN_E_FAIL             On failure.
 * @returns  OPEN_E_PARAM            Invalid argument.
 * @returns  OPEN_E_NOT_FOUND        If this Counter is not supported.
 *
 * @notes    None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1qRxPacketsNotForwardedCountGet(openapiClientHandle_t *client_handle, 
                                                       uint32_t ifNum,
                                                       uint64_t *totalRxPktsNotForwarded);

/*********************************************************************
 * @purpose  Gets the VLAN type for the specified VLAN.
 *
 * @param    client_handle      @b{(input)}   Client handle from
 *                                            registration API.
 * @param    vlanNum            @b{(input)}   VLAN ID.
 * @param    vlanType           @b{(output)}  VLAN type.
 *
 * @returns  OPEN_E_NONE        On success.
 * @returns  OPEN_E_FAIL        On failure.
 * @returns  OPEN_E_PARAM       Invalid argument.
 * @returns  OPEN_E_NOT_FOUND   The VLAN specified is not created.
 *
 * @notes    None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *********************************************************************/
open_error_t openapiVlanTypeGet(openapiClientHandle_t *client_handle,
                                uint32_t vlanNum,
                                OPEN_DOT1Q_VLAN_TYPE_t *vlanType);
#endif /* OPENAPI_VLAN_H_INCLUDED */
