
/*********************************************************************
*
*  Copyright 2012-2023 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  acl_example.c
*
* @purpose   Security Access Lists Example.
*
* @component OPEN
*
* @comments
*
* @create    09/12/2013
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_acl.h"

#define PRINTSANITYRESULTS(result, test, msg, feat) \
if (result==OPEN_E_UNAVAIL) { printf("Sanity test skipped.\n"); } \
else if ((result==OPEN_E_NONE) && (test)) { printf("Sanity Success - %s - %s.\n", msg, feat); } \
else { printf("Sanity Failure - %s - %s.\n", msg, feat); }

#define PRINTBADRESULT(result, msg) \
if (result==OPEN_E_UNAVAIL) { printf("Feature not supported - %s (err %d).\n", msg, result); } \
else if (result!=OPEN_E_NONE) { printf("Test Failure - %s (err %d).\n", msg, result); }

#define ACL_COUNTERS_ENABLE 1
#define ACL_COUNTERS_DISABLE 0

/*********************************************************************
* @purpose  Create a new Access List
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* @param    aclType             @b{(input)}  ACL type
* @param    aclName             @b{(input)}  ACL name
* @param    aclId               @b{(output)} Newly created ACL identifier
* 
* @returns  none
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testAclCreate(openapiClientHandle_t *clientHandle, 
                           OPEN_ACL_TYPE_t aclType,
                           char *aclName,
                           uint32_t *aclId)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc aclDesc;
  char str[32];
  uint32_t tmp1, tmp2;

  strcpy(str, aclName);
  aclDesc.pstart = str;
  aclDesc.size = strlen(str) + 1;

  /* start fresh */
  openapiAclDeleteByName(clientHandle, aclType, &aclDesc); 
  result = openapiAclCreate(clientHandle, aclType, &aclDesc, &tmp1);
  PRINTBADRESULT(result, "openapiAclCreate");

  if (result == OPEN_E_NONE)
  {
    result = openapiAclGet(clientHandle, aclType, &aclDesc, &tmp2);
    PRINTBADRESULT(result, "openapiAclGet");
  }

  PRINTSANITYRESULTS(result, tmp1==tmp2, __func__, "");

  *aclId = tmp2;

  return (result);
}

/*********************************************************************
* @purpose  Rename and test named Access List
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* @param    aclType             @b{(input)}  ACL type
* @param    aclName             @b{(input)}  ACL name
* @param    aclRename           @b{(input)}  New ACL name
*
* @returns  none
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testAclRename(openapiClientHandle_t *clientHandle, 
                   OPEN_ACL_TYPE_t aclType,
                   char *aclName,
                   char *aclRename)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc aclNameDesc;
  open_buffdesc aclRenameDesc;
  open_buffdesc aclTmpDesc;
  char aclNameStr[32];
  char aclRenameStr[32];
  char aclTmpStr[32];
  uint32_t aclId;

  strcpy(aclNameStr, aclName);
  aclNameDesc.pstart = aclNameStr;
  aclNameDesc.size = strlen(aclNameStr)+1;

  strcpy(aclRenameStr, aclRename);
  aclRenameDesc.pstart = aclRenameStr;
  aclRenameDesc.size = strlen(aclRenameStr)+1;

  aclTmpDesc.pstart = aclTmpStr;
  aclTmpDesc.size = sizeof(aclTmpStr)-1;

  if (result == OPEN_E_NONE)
  {
    result = openapiAclGet(clientHandle, aclType, &aclNameDesc, &aclId);
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiAclNameGet(clientHandle, aclId, &aclTmpDesc);
    PRINTBADRESULT(result, "openapiAclNameGet");
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiAclRename(clientHandle, aclId, &aclRenameDesc);
    PRINTBADRESULT(result, "openapiAclRename");
    PRINTSANITYRESULTS(result, (result == OPEN_E_NONE), "Renamed", (char *)aclRenameDesc.pstart);
  }

}

/*********************************************************************
* @purpose  Perform some generic ACL retrieval and iteration tests.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* 
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void testAclRetrieval(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_ACL_TYPE_t aclType = OPEN_ACL_TYPE_IP;
  uint32_t aclId;
  uint32_t iTmp;
  uint32_t i;

  /* Get first named IP ACL created in the system */
  aclId = 0;
  result = openapiAclGetFirst(clientHandle, aclType, &aclId);
  PRINTBADRESULT(result, "openapiAclGetFirst - IP ACL");
  PRINTSANITYRESULTS(result, (aclId>0),"openapiAclGetFirst", "IP ACL");

  /* Create temporary ACL for iteration purposes */
  result = testAclCreate(clientHandle, aclType, "another_test_ip", &iTmp);
  PRINTBADRESULT(result, "testAclCreate - temporary");
  PRINTSANITYRESULTS(result, (iTmp>0), "testAclCreate", "temporary");

  /* Start from the beginning and iterate through the available named ACL type */
  i = 0;
  aclId = 0;
  while (openapiAclGetNext(clientHandle, aclType, aclId, &aclId) == OPEN_E_NONE) { i++; }
  PRINTSANITYRESULTS(result, (i>1), __func__, "openapiAclGetNext");

  /* Remove temporary ACL */
  if (result == OPEN_E_NONE)
  {
    openapiAclDelete(clientHandle, iTmp);
  }
}

/*********************************************************************
* @purpose  Create some named IP ACL rules for demonstration purposes
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    aclId              @b{(input)}  Newly created ACL identifier
* 
* @returns  none
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testAclAddIpRules(openapiClientHandle_t *clientHandle, 
                       uint32_t aclId)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t aclRule;
  uint32_t iVal, iTmp;
  bool bTmp;
  openIpAclRulesFields_t ipAclRulesFields;

  memset(&ipAclRulesFields, 0x0, sizeof(openIpAclRulesFields_t));

  /* Create a Permit action rule */
  {
    aclRule = 0; /* Let the system determine first available rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_PERMIT);
    PRINTBADRESULT(result, "openapiAclRuleActionAdd - PERMIT");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleActionGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleActionGet - PERMIT");
      PRINTSANITYRESULTS(result, OPEN_ACL_PERMIT==iTmp, __func__, "PERMIT");
    }
  }

  memset(&ipAclRulesFields, 0x0, sizeof(openIpAclRulesFields_t));
  result = openapiAclFieldsConfiguredMaskGet(clientHandle, aclId, aclRule, &ipAclRulesFields);
  if (OPEN_E_NONE == result)
  {
    if (0 != OPEN_ISMASKBITSET(ipAclRulesFields.fieldsConfigured, OPEN_IP_ACL_ACTION))
    {
      printf("\r\nField Action is configured for ACL ID:%d ACL Rule:%d\r\n", aclId, aclRule);
    }
    else
    {
      printf("\r\nField Action is not configured for ACL ID:%d ACL Rule:%d\r\n", aclId, aclRule);
    }
  }
  else
  {
    printf("\r\n result:%d Bad code trying to get fields configured for"
           " ACL ID:%d ACL Rule:%d\r\n", result, aclId, aclRule);
  }

  /* Add match criteria - match-all */
  {
    result = openapiAclRuleMatchEveryAdd(clientHandle, aclId, aclRule, true);
    PRINTBADRESULT(result, "openapiAclRuleMatchEveryAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchEveryGet(clientHandle, aclId, aclRule, &bTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchEveryGet");
      PRINTSANITYRESULTS(result, true==bTmp, __func__, "Every");
    }
  }

  /* Add match criteria - assign max queue id */
  {
    result = openapiAclMaxAssignQueueGet(clientHandle, &iVal);
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchAssignQueueAdd(clientHandle, aclId, aclRule, iVal);
      PRINTBADRESULT(result, "openapiAclRuleMatchAssignQueueAdd");
    }
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchAssignQueueGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchAssignQueueGet");
      PRINTSANITYRESULTS(result, iVal==iTmp, __func__, "Assign Queue");
    }
  }

  /* Add match criteria - logging */
  {
    result = openapiAclRuleMatchLoggingAdd(clientHandle, aclId, aclRule, true);
    PRINTBADRESULT(result, "openapiAclRuleMatchLoggingAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchLoggingGet(clientHandle, aclId, aclRule, &bTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchLoggingGet");
      PRINTSANITYRESULTS(result, true==bTmp, __func__, "Logging");
    }
  }

  /* Add match criteria - sFlow remote agent */
  {
    result = openapiAclRuleMatchSflowSamplingAdd(clientHandle, aclId, aclRule, true);
    PRINTBADRESULT(result, "openapiAclRuleMatchSflowSamplingAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchSflowSamplingGet(clientHandle, aclId, aclRule, &bTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchSflowSamplingGet");
      PRINTSANITYRESULTS(result, true==bTmp, __func__, "sflow-remote-agent");
    }
  }

  /* Add match criteria - Packet Mirroring */
  {
    uint32_t intf;

    result = openapiIfFirstGet(clientHandle, OPEN_INTF_TYPE_PHY, &intf);
    PRINTBADRESULT(result, "openapiIfFirstGet");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchMirrorAdd(clientHandle, aclId, aclRule, intf);
      PRINTBADRESULT(result, "openapiAclRuleMatchMirrorAdd");
    }
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchMirrorGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchMirrorGet");
      PRINTSANITYRESULTS(result, intf==iTmp, __func__, "Packet Mirroring Interface");
    }
  }

  /* Create a Permit action rule */
  {
    aclRule++; /* Increment to next rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_PERMIT);
    PRINTBADRESULT(result, "openapiAclRuleActionAdd - PERMIT");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleActionGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleActionGet - PERMIT");
      PRINTSANITYRESULTS(result, OPEN_ACL_PERMIT==iTmp, __func__, "PERMIT");
    }
  }

  /* Add match criteria - source address and prefix */
  {
    open_inet_addr_t ipAddr;
    open_inet_addr_t maskAddr;
    char *ipStr = "10.10.10.1";
    char *maskStr = "255.255.255.255";
    char ipStrBuff[32];
    char maskStrBuff[32];

    memset(&ipAddr, 0, sizeof(ipAddr));
    inet_pton(AF_INET, ipStr, &(ipAddr.addr.ipv4));
    ipAddr.family = OPEN_AF_INET;

    memset(&maskAddr, 0, sizeof(maskAddr));
    inet_pton(AF_INET, maskStr, &(maskAddr.addr.ipv4));
    maskAddr.family = OPEN_AF_INET;

    /* IPv4 addresses in open_inet_addr_t structures are in host byte order */
    ipAddr.addr.ipv4 = ntohl(ipAddr.addr.ipv4);
    maskAddr.addr.ipv4 = ntohl(maskAddr.addr.ipv4);

    result = openapiAclRuleMatchIpSrcMaskAdd(clientHandle, aclId, aclRule, ipAddr, maskAddr);
    PRINTBADRESULT(result, "openapiAclRuleMatchIpSrcMaskAdd");

    memset(&ipAddr, 0, sizeof(ipAddr));
    memset(&maskAddr, 0, sizeof(maskAddr));
    result = openapiAclRuleMatchIpSrcMaskGet(clientHandle, aclId, aclRule, &ipAddr, &maskAddr);
    PRINTBADRESULT(result, "openapiAclRuleMatchIpSrcMaskGet");

    if (result == OPEN_E_NONE)
    {
      /* IPv4 addresses in open_inet_addr_t structures are in host byte order */
      ipAddr.addr.ipv4 = htonl(ipAddr.addr.ipv4);
      maskAddr.addr.ipv4 = htonl(maskAddr.addr.ipv4);

      inet_ntop(AF_INET, &(ipAddr.addr.ipv4), ipStrBuff, sizeof(ipStrBuff));
      inet_ntop(AF_INET, &(maskAddr.addr.ipv4), maskStrBuff, sizeof(maskStrBuff));
      PRINTSANITYRESULTS(result,
                         ((strcmp(ipStr, ipStrBuff) == 0) && (strcmp(maskStr, maskStrBuff) == 0)),
                         __func__, "Source IPv4 Address/Mask");
    }
  }

  /* Add match criteria - destination address and prefix */
  {
    open_inet_addr_t ipAddr;
    open_inet_addr_t maskAddr;
    char *ipStr = "20.20.20.1";
    char *maskStr = "255.0.0.0";
    char ipStrBuff[32];
    char maskStrBuff[32];

    memset(&ipAddr, 0, sizeof(ipAddr));
    inet_pton(AF_INET, ipStr, &(ipAddr.addr.ipv4));
    ipAddr.family = OPEN_AF_INET;

    memset(&maskAddr, 0, sizeof(maskAddr));
    inet_pton(AF_INET, maskStr, &(maskAddr.addr.ipv4));
    maskAddr.family = OPEN_AF_INET;

    /* IPv4 addresses in open_inet_addr_t structures are in host byte order */
    ipAddr.addr.ipv4 = ntohl(ipAddr.addr.ipv4);
    maskAddr.addr.ipv4 = ntohl(maskAddr.addr.ipv4);

    result = openapiAclRuleMatchIpDstMaskAdd(clientHandle, aclId, aclRule, ipAddr, maskAddr);
    PRINTBADRESULT(result, "openapiAclRuleMatchIpDstMaskAdd");

    memset(&ipAddr, 0, sizeof(ipAddr));
    memset(&maskAddr, 0, sizeof(maskAddr));
    result = openapiAclRuleMatchIpDstMaskGet(clientHandle, aclId, aclRule, &ipAddr, &maskAddr);
    PRINTBADRESULT(result, "openapiAclRuleMatchIpDstMaskGet");
    if (result == OPEN_E_NONE)
    {
      /* IPv4 addresses in open_inet_addr_t structures are in host byte order */
      ipAddr.addr.ipv4 = htonl(ipAddr.addr.ipv4);
      maskAddr.addr.ipv4 = htonl(maskAddr.addr.ipv4);

      inet_ntop(AF_INET, &(ipAddr.addr.ipv4), ipStrBuff, sizeof(ipStrBuff));
      inet_ntop(AF_INET, &(maskAddr.addr.ipv4), maskStrBuff, sizeof(maskStrBuff));
      PRINTSANITYRESULTS(result,
                         ((strcmp(ipStr, ipStrBuff) == 0) && (strcmp(maskStr, maskStrBuff) == 0)),
                         __func__, "Destination IPv4 Address/Mask");
    }
  }
  
  /* Add match criteria - PIM protocol */
  {
    result = openapiAclRuleMatchProtocolAdd(clientHandle, aclId, aclRule, OPENAPI_ACL_PROTOCOL_PIM);
    PRINTBADRESULT(result, "openapiAclRuleMatchProtocolAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchProtocolGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchProtocolGet");
      PRINTSANITYRESULTS(result, OPENAPI_ACL_PROTOCOL_PIM==iTmp, __func__, "PIM Protocol");
    }
  }

  /* Add match criteria - DSCP value */
  {
    result = openapiAclRuleMatchDscpAdd(clientHandle, aclId, aclRule, OPENAPI_ACL_IP_DSCP_AF11);
    PRINTBADRESULT(result, "openapiAclRuleMatchDscpAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchDscpGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchDscpGet");
      PRINTSANITYRESULTS(result, OPENAPI_ACL_IP_DSCP_AF11==iTmp, __func__, "DSCP Value");
    }
  }

  /* Add match criteria - TTL value */
  {
    result = openapiAclRuleMatchTtlAdd(clientHandle, aclId, aclRule, 11);
    PRINTBADRESULT(result, "openapiAclRuleMatchTtlAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchTtlGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchTtlGet");
      PRINTSANITYRESULTS(result, 11==iTmp, __func__, "TTL Value");
    }
  }

  /* Add match criteria - IP fragments */
  {
    result = openapiAclRuleMatchFragmentsAdd(clientHandle, aclId, aclRule, true);
    PRINTBADRESULT(result, "openapiAclRuleMatchFragmentsAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchFragmentsGet(clientHandle, aclId, aclRule, &bTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchFragmentsGet");
      PRINTSANITYRESULTS(result, true==bTmp, __func__, "IP Fragments");
    }
  }

  /* Add match criteria - Precedence value */
  {
    result = openapiAclRuleMatchPrecedenceAdd(clientHandle, aclId, aclRule, 7);
    PRINTBADRESULT(result, "openapiAclRuleMatchPrecedenceAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchPrecedenceGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchPrecedenceGet");
      PRINTSANITYRESULTS(result, 7==iTmp, __func__, "Precedence Value");
    }
  }

  /* Add match criteria - ToS value and mask */
  {
    open_buffdesc tosValDesc;
    open_buffdesc tosMaskDesc;
    char tosValStr[32];
    char tosMaskStr[32];

    open_buffdesc tmpTosValDesc;
    open_buffdesc tmpTosMaskDesc;
    char tmpTosValStr[32];
    char tmpTosMaskStr[32];

    strcpy(tosValStr,"a0");
    tosValDesc.pstart = tosValStr;
    tosValDesc.size = strlen(tosValStr)+1;
    strcpy(tosMaskStr,"a2");
    tosMaskDesc.pstart = tosMaskStr;
    tosMaskDesc.size = strlen(tosMaskStr)+1;

    tmpTosValDesc.pstart = tmpTosValStr;
    tmpTosValDesc.size = sizeof(tmpTosValDesc)-1;
    tmpTosMaskDesc.pstart = tmpTosMaskStr;
    tmpTosMaskDesc.size = sizeof(tmpTosMaskDesc)-1;

    result = openapiAclRuleMatchTosAdd(clientHandle, aclId, aclRule, &tosValDesc, &tosMaskDesc);
    PRINTBADRESULT(result, "openapiAclRuleMatchTosAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchTosGet(clientHandle, aclId, aclRule, &tmpTosValDesc, &tmpTosMaskDesc);
      PRINTBADRESULT(result, "openapiAclRuleMatchTosGet");
      PRINTSANITYRESULTS(result,
                         ((strcmp(tosValDesc.pstart,tmpTosValDesc.pstart)==0)&&
                         (strcmp(tosMaskDesc.pstart,tmpTosMaskDesc.pstart)==0)),
                         __func__, "ToS and Tos Mask Value");
    }
  }

  /* Add match criteria - Packet Redirection */
  {
    uint32_t intf;

    result = openapiIfFirstGet(clientHandle, OPEN_INTF_TYPE_PHY, &intf);
    PRINTBADRESULT(result, "openapiIfFirstGet");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchRedirectAdd(clientHandle, aclId, aclRule, intf);
      PRINTBADRESULT(result, "openapiAclRuleMatchRedirectAdd");
    }
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchRedirectGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchRedirectGet");
      PRINTSANITYRESULTS(result, intf==iTmp, __func__, "Packet Redirection Interface");
    }
  }

  /* Create a Permit action rule */
  {
    aclRule++; /* Increment to next rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_PERMIT);
    PRINTBADRESULT(result, "openapiAclRuleActionAdd - PERMIT");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleActionGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleActionGet - PERMIT");
      PRINTSANITYRESULTS(result, OPEN_ACL_PERMIT==iTmp, __func__, "PERMIT");
    }
  }

  /* Add match criteria - TCP protocol */
  {
    result = openapiAclRuleMatchProtocolAdd(clientHandle, aclId, aclRule, OPENAPI_ACL_PROTOCOL_TCP);
    PRINTBADRESULT(result, "openapiAclRuleMatchProtocolAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchProtocolGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchProtocolGet");
      PRINTSANITYRESULTS(result, OPENAPI_ACL_PROTOCOL_TCP==iTmp, __func__, "TCP Protocol");
    }
  }

  /* Add match criteria - Packet Redirection using an external agent */
  {
    result = openapiAclRuleMatchRedirectAgentAdd(clientHandle, aclId, aclRule, 100);
    PRINTBADRESULT(result, "openapiAclRuleMatchRedirectAgentAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchRedirectAgentGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchRedirectAgentGet");
      PRINTSANITYRESULTS(result, 100==iTmp, __func__, "Packet Redirect External Agent");
    }
  }

  /* Add match criteria - Rate Limit */
  {
    result = openapiAclRuleMatchRateLimitAdd(clientHandle, aclId, aclRule, 4294967295UL, 128);
    PRINTBADRESULT(result, "openapiAclRuleMatchRateLimitAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchRateLimitGet(clientHandle, aclId, aclRule, &iVal, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchRateLimitGet");
      PRINTSANITYRESULTS(result, (4294967295UL==iVal&&128==iTmp), __func__, "Simple traffic rate-limit policing");
    }
  }

  /* Add match criteria - Control when rule is in effect based on time */
  {
    OPEN_ACL_RULE_STATUS_t status;
    open_buffdesc timeDesc;
    open_buffdesc tmpTimeDesc;
    char timeStr[32];
    char tmpStr[32];

    strcpy(timeStr, "ipv4-time-range-name");
    timeDesc.pstart = timeStr;
    timeDesc.size = strlen(timeStr) + 1;

    tmpTimeDesc.pstart = timeStr;
    tmpTimeDesc.size = sizeof(tmpStr)-1;

    result = openapiAclRuleMatchTimeRangeAdd(clientHandle, aclId, aclRule, &timeDesc);
    PRINTBADRESULT(result, "openapiAclRuleMatchTimeRangeAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchTimeRangeGet(clientHandle, aclId, aclRule, &tmpTimeDesc);
      PRINTBADRESULT(result, "openapiAclRuleMatchTimeRangeGet");
      PRINTSANITYRESULTS(result, (strcmp(timeDesc.pstart,tmpTimeDesc.pstart)==0), __func__, "Time based activation");
    }
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchTimeRangeStatusGet(clientHandle, aclId, aclRule, &status);
      PRINTBADRESULT(result, "openapiAclRuleMatchTimeRangeStatusGet");
      PRINTSANITYRESULTS(result, (status==status), __func__, "Time Range status");
    }
  }

  memset(&ipAclRulesFields, 0x0, sizeof(openIpAclRulesFields_t));
  result = openapiAclFieldsConfiguredMaskGet(clientHandle, aclId, aclRule, &ipAclRulesFields);
  if (OPEN_E_NONE == result)
  {
    if (0 != OPEN_ISMASKBITSET(ipAclRulesFields.fieldsConfigured, OPEN_IP_ACL_TIME_RANGE_NAME))
    {
      printf("\r\nField Timerange is configured for ACL ID:%d ACL Rule:%d\r\n", aclId, aclRule);
    }
    else
    {
      printf("\r\nField Timerange is not configured for ACL ID:%d ACL Rule:%d\r\n", aclId, aclRule);
    }
  }
  else
  {
    printf("\r\n result:%d Bad code trying to get fields configured for"
           " ACL ID:%d ACL Rule:%d\r\n", result, aclId, aclRule);
  }

  /* Create a Permit action rule */
  {
    aclRule++; /* Increment to next rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_PERMIT);
    PRINTBADRESULT(result, "openapiAclRuleActionAdd - PERMIT");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleActionGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleActionGet - PERMIT");
      PRINTSANITYRESULTS(result, OPEN_ACL_PERMIT==iTmp, __func__, "PERMIT");
    }
  }

  /* Add match criteria - ICMP protocol */
  {
    result = openapiAclRuleMatchProtocolAdd(clientHandle, aclId, aclRule, OPENAPI_ACL_PROTOCOL_ICMP);
    PRINTBADRESULT(result, "openapiAclRuleMatchProtocolAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchProtocolGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchProtocolGet");
      PRINTSANITYRESULTS(result, OPENAPI_ACL_PROTOCOL_ICMP==iTmp, __func__, "ICMP Protocol");
    }
  }

  /* Add match criteria - ICMP Echo messaging */
  {
    uint32_t ECHO_TYPE = 8;
    uint32_t ECHO_CODE = 0;
    bool addCode = true;
    uint32_t tmpType;
    uint32_t tmpCode;

    result = openapiAclRuleMatchIcmpTypeCodeAdd(clientHandle, aclId, aclRule, addCode, ECHO_TYPE, ECHO_CODE);
    PRINTBADRESULT(result, "openapiAclRuleMatchIcmpTypeCodeAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchIcmpTypeCodeGet(clientHandle, aclId, aclRule, addCode, &tmpType, &tmpCode);
      PRINTBADRESULT(result, "openapiAclRuleMatchIcmpTypeCodeGet");
      PRINTSANITYRESULTS(result, (ECHO_TYPE==tmpType&&ECHO_CODE==tmpCode), __func__, "ICMP messaging for 'Echo'");
    }
  }

  /* Create a Permit action rule */
  {
    aclRule++; /* Increment to next rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_PERMIT);
    PRINTBADRESULT(result, "openapiAclRuleActionAdd - PERMIT");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleActionGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleActionGet - PERMIT");
      PRINTSANITYRESULTS(result, OPEN_ACL_PERMIT==iTmp, __func__, "PERMIT");
    }
  }

  /* Add match criteria - IGMP protocol */
  {
    result = openapiAclRuleMatchProtocolAdd(clientHandle, aclId, aclRule, OPENAPI_ACL_PROTOCOL_IGMP);
    PRINTBADRESULT(result, "openapiAclRuleMatchProtocolAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchProtocolGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchProtocolGet");
      PRINTSANITYRESULTS(result, OPENAPI_ACL_PROTOCOL_IGMP==iTmp, __func__, "IGMP Protocol");
    }
  }

  /* Add match criteria - IGMP Multicast Traceroute messaging */
  {
    uint32_t TRACE_TYPE = 30; /* 0x1e */
    uint32_t tmpType;

    result = openapiAclRuleMatchIgmpTypeAdd(clientHandle, aclId, aclRule, TRACE_TYPE);
    PRINTBADRESULT(result, "openapiAclRuleMatchIgmpTypeAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchIgmpTypeGet(clientHandle, aclId, aclRule, &tmpType);
      PRINTBADRESULT(result, "openapiAclRuleMatchIgmpTypeGet");
      PRINTSANITYRESULTS(result, (TRACE_TYPE==tmpType), __func__, "IGMP messaging for 'Multicast Traceroute'");
    }
  }

  /* Create a Permit action rule */
  {
    aclRule++; /* Increment to next rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_PERMIT);
    PRINTBADRESULT(result, "openapiAclRuleActionAdd - PERMIT");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleActionGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleActionGet - PERMIT");
      PRINTSANITYRESULTS(result, OPEN_ACL_PERMIT==iTmp, __func__, "PERMIT");
    }
  }

  /* Add match criteria - TCP protocol */
  {
    result = openapiAclRuleMatchProtocolAdd(clientHandle, aclId, aclRule, OPENAPI_ACL_PROTOCOL_TCP);
    PRINTBADRESULT(result, "openapiAclRuleMatchProtocolAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchProtocolGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchProtocolGet");
      PRINTSANITYRESULTS(result, OPENAPI_ACL_PROTOCOL_TCP==iTmp, __func__, "TCP Protocol");
    }
  }

  /* Add match criteria - TCP Flags */
  {
    uint32_t flagVal=0, flagMask=0;
    uint32_t tmpVal, tmpMask;

    /* Set Finish flag */
    flagMask |= (true << OPENAPI_ACL_TCP_FLAG_FIN);
    flagVal  |= (true << OPENAPI_ACL_TCP_FLAG_FIN);

    /* Set Synchronize flag */
    flagMask |= (true << OPENAPI_ACL_TCP_FLAG_SYN);
    flagVal  |= (true << OPENAPI_ACL_TCP_FLAG_SYN);

    /* Clear Acknowledge flag */
    flagMask |= (true << OPENAPI_ACL_TCP_FLAG_ACK);
    flagVal  |= (false << OPENAPI_ACL_TCP_FLAG_ACK);

    result = openapiAclRuleMatchTcpFlagsAdd(clientHandle, aclId, aclRule, flagVal, flagMask);
    PRINTBADRESULT(result, "openapiAclRuleMatchTcpFlagsAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchTcpFlagsGet(clientHandle, aclId, aclRule, &tmpVal, &tmpMask);
      PRINTBADRESULT(result, "openapiAclRuleMatchTcpFlagsGet");
      PRINTSANITYRESULTS(result, ((flagVal==tmpVal)&&(flagMask==tmpMask)), __func__, "Miscellaneous TCP Flags test");
    }
  }

  /* Add match criteria - Source Layer 4 port 80 */
  {
    OPEN_ACL_L4_PORT_OPERATOR_t tmpOper;
    uint32_t tmpStartPort;
    uint32_t tmpEndPort;

    result = openapiAclRuleMatchL4SrcPortAdd(clientHandle, aclId, aclRule, OPEN_ACL_L4_PORT_OPERATOR_EQUAL_TO, 80, 0);
    PRINTBADRESULT(result, "openapiAclRuleMatchL4SrcPortAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchL4SrcPortGet(clientHandle, aclId, aclRule, &tmpOper, &tmpStartPort, &tmpEndPort);
      PRINTBADRESULT(result, "openapiAclRuleMatchL4SrcPortGet");
      PRINTSANITYRESULTS(result, ((OPEN_ACL_L4_PORT_OPERATOR_EQUAL_TO==tmpOper)&&(80==tmpStartPort)), __func__, "Source Layer 4 port 80");
    }
  }

  /* Create a Permit action rule */
  {
    aclRule++; /* Increment to next rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_PERMIT);
    PRINTBADRESULT(result, "openapiAclRuleActionAdd - PERMIT");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleActionGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleActionGet - PERMIT");
      PRINTSANITYRESULTS(result, OPEN_ACL_PERMIT==iTmp, __func__, "PERMIT");
    }
  }

  /* Add match criteria - UDP protocol */
  {
    result = openapiAclRuleMatchProtocolAdd(clientHandle, aclId, aclRule, OPENAPI_ACL_PROTOCOL_UDP);
    PRINTBADRESULT(result, "openapiAclRuleMatchProtocolAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchProtocolGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchProtocolGet");
      PRINTSANITYRESULTS(result, OPENAPI_ACL_PROTOCOL_UDP==iTmp, __func__, "UDP Protocol");
    }
  }

  /* Add match criteria - Source Layer 4 port < 10000 */
  {
    OPEN_ACL_L4_PORT_OPERATOR_t tmpOper;
    uint32_t tmpStartPort;
    uint32_t tmpEndPort;

    result = openapiAclRuleMatchL4SrcPortAdd(clientHandle, aclId, aclRule, OPEN_ACL_L4_PORT_OPERATOR_LESS_THAN, 10000, 0);
    PRINTBADRESULT(result, "openapiAclRuleMatchL4SrcPortAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchL4SrcPortGet(clientHandle, aclId, aclRule, &tmpOper, &tmpStartPort, &tmpEndPort);
      PRINTBADRESULT(result, "openapiAclRuleMatchL4SrcPortGet");
      PRINTSANITYRESULTS(result, ((OPEN_ACL_L4_PORT_OPERATOR_LESS_THAN==tmpOper)&&(10000==tmpStartPort)), __func__, "Source Layer 4 port < 10000");
    }
  }

  /* Add match criteria - Destination Layer 4 port > 20000 */
  {
    OPEN_ACL_L4_PORT_OPERATOR_t tmpOper;
    uint32_t tmpStartPort;
    uint32_t tmpEndPort;

    result = openapiAclRuleMatchL4DstPortAdd(clientHandle, aclId, aclRule, OPEN_ACL_L4_PORT_OPERATOR_GREATER_THAN, 20000, 0);
    PRINTBADRESULT(result, "openapiAclRuleMatchL4DstPortAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchL4DstPortGet(clientHandle, aclId, aclRule, &tmpOper, &tmpStartPort, &tmpEndPort);
      PRINTBADRESULT(result, "openapiAclRuleMatchL4DstPortGet");
      PRINTSANITYRESULTS(result, ((OPEN_ACL_L4_PORT_OPERATOR_GREATER_THAN==tmpOper)&&(20000==tmpStartPort)), __func__, "Destination Layer 4 port > 20000");
    }
  }

  /* Create a Deny action rule */
  {
    aclRule = 0; /* Let the system determine first available rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_DENY);
    PRINTBADRESULT(result, "openapiAclRuleActionAdd - DENY");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleActionGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleActionGet - DENY");
      PRINTSANITYRESULTS(result, OPEN_ACL_DENY==iTmp, __func__, "DENY");
    }
  }
}

/*********************************************************************
* @purpose  Create some named IPv6 ACL rules for demonstration purposes
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    aclId              @b{(input)}  Newly created ACL identifier
* 
* @returns  none
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testAclAddIpv6Rules(openapiClientHandle_t *clientHandle, 
                         uint32_t aclId)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t aclRule;
  uint32_t iVal, iTmp;
  bool bTmp;
  openIpAclRulesFields_t ipAclRulesFields;

  memset(&ipAclRulesFields, 0x0, sizeof(openIpAclRulesFields_t));

  /* Create a Permit action rule */
  {
    aclRule = 0; /* Let the system determine first available rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_PERMIT);
    PRINTBADRESULT(result, "openapiAclRuleActionAdd - PERMIT");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleActionGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleActionGet - PERMIT");
      PRINTSANITYRESULTS(result, OPEN_ACL_PERMIT==iTmp, __func__, "PERMIT" );
    }
  }

  /* Add match criteria - match-all */
  {
    result = openapiAclRuleMatchEveryAdd(clientHandle, aclId, aclRule, true);
    PRINTBADRESULT(result, "openapiAclRuleMatchEveryAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchEveryGet(clientHandle, aclId, aclRule, &bTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchEveryGet");
      PRINTSANITYRESULTS(result, true==bTmp, __func__, "Every");
    }
  }

  /* Add match criteria - assign max queue id */
  {
    result = openapiAclMaxAssignQueueGet(clientHandle, &iVal);
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchAssignQueueAdd(clientHandle, aclId, aclRule, iVal);
      PRINTBADRESULT(result, "openapiAclRuleMatchAssignQueueAdd");
    }
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchAssignQueueGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchAssignQueueGet");
      PRINTSANITYRESULTS(result, iVal==iTmp, __func__, "Assign Queue");
    }
  }

  memset(&ipAclRulesFields, 0x0, sizeof(openIpAclRulesFields_t));
  result = openapiAclFieldsConfiguredMaskGet(clientHandle, aclId, aclRule, &ipAclRulesFields);
  if (OPEN_E_NONE == result)
  {
    if (0 != OPEN_ISMASKBITSET(ipAclRulesFields.fieldsConfigured, OPEN_IP_ACL_ASSIGN_QUEUEID))
    {
      printf("\r\nField Assign Queue is configured for ACL ID:%d ACL Rule:%d\r\n", aclId, aclRule);
    }
    else
    {
      printf("\r\nField Assign Queue is not configured for ACL ID:%d ACL Rule:%d\r\n", aclId, aclRule);
    }
  }
  else
  {
    printf("\r\n result:%d Bad code trying to get fields configured for"
           " ACL ID:%d ACL Rule:%d\r\n", result, aclId, aclRule);
  }

  /* Add match criteria - logging */
  {
    result = openapiAclRuleMatchLoggingAdd(clientHandle, aclId, aclRule, true);
    PRINTBADRESULT(result, "openapiAclRuleMatchLoggingAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchLoggingGet(clientHandle, aclId, aclRule, &bTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchLoggingGet");
      PRINTSANITYRESULTS(result, true==bTmp, __func__, "Logging");
    }
  }

  /* Add match criteria - Packet Mirroring */
  {
    uint32_t intf;

    result = openapiIfFirstGet(clientHandle, OPEN_INTF_TYPE_PHY, &intf);
    PRINTBADRESULT(result, "openapiIfFirstGet");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchMirrorAdd(clientHandle, aclId, aclRule, intf);
      PRINTBADRESULT(result, "openapiAclRuleMatchMirrorAdd");
    }
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchMirrorGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchMirrorGet");
      PRINTSANITYRESULTS(result, intf==iTmp, __func__, "Packet Mirroring Interface");
    }
  }

  /* Create a Permit action rule */
  {
    aclRule++; /* Increment to next rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_PERMIT);
    PRINTBADRESULT(result, "openapiAclRuleActionAdd - PERMIT");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleActionGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleActionGet - PERMIT");
      PRINTSANITYRESULTS(result, OPEN_ACL_PERMIT==iTmp, __func__, "PERMIT");
    }
  }

  /* Add match criteria - PIM protocol */
  {
    result = openapiAclRuleMatchProtocolAdd(clientHandle, aclId, aclRule, OPENAPI_ACL_PROTOCOL_PIM);
    PRINTBADRESULT(result, "openapiAclRuleMatchProtocolAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchProtocolGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchProtocolGet");
      PRINTSANITYRESULTS(result, OPENAPI_ACL_PROTOCOL_PIM==iTmp, __func__, "PIM Protocol");
    }
  }

  /* Add match criteria - IPv6 source address and prefix */
  {
    open_inet_addr_t ipAddr;
    char *ipStr = "1001::";
    uint32_t pfxLen = 24;
    open_inet_addr_t tmpIpAddr;
    uint32_t tmpPfxLen;
    char ipStrBuff[64];

    memset(&ipAddr, 0, sizeof(ipAddr));
    inet_pton(AF_INET6, ipStr, &(ipAddr.addr.ipv6));
    ipAddr.family = OPEN_AF_INET6;

    result = openapiAclRuleMatchIpv6SrcPfxAdd(clientHandle, aclId, aclRule, ipAddr, pfxLen);
    PRINTBADRESULT(result, "openapiAclRuleMatchIpv6SrcPfxAdd");
    if (result == OPEN_E_NONE)
    {
      memset(&tmpIpAddr, 0, sizeof(tmpIpAddr));
      tmpPfxLen = 0;
      result = openapiAclRuleMatchIpv6SrcPfxGet(clientHandle, aclId, aclRule, &tmpIpAddr, &tmpPfxLen);
      PRINTBADRESULT(result, "openapiAclRuleMatchIpv6SrcPfxGet");
      if (result == OPEN_E_NONE)
      {
        inet_ntop(AF_INET6, &(tmpIpAddr.addr.ipv6), ipStrBuff, sizeof(ipStrBuff));
        PRINTSANITYRESULTS(result,
                           ((strcmp(ipStr, ipStrBuff) == 0) && (pfxLen == tmpPfxLen)),
                           __func__, "Source IPv6 Address/prefix");
      }
    }
  }

  /* Add match criteria - destination address and prefix */
  {
    open_inet_addr_t ipAddr;
    char *ipStr = "2001::";
    uint32_t pfxLen = 24;
    open_inet_addr_t tmpIpAddr;
    uint32_t tmpPfxLen;
    char ipStrBuff[64];

    memset(&ipAddr, 0, sizeof(ipAddr));
    inet_pton(AF_INET6, ipStr, &(ipAddr.addr.ipv6));
    ipAddr.family = OPEN_AF_INET6;

    result = openapiAclRuleMatchIpv6DstPfxAdd(clientHandle, aclId, aclRule, ipAddr, pfxLen);
    PRINTBADRESULT(result, "openapiAclRuleMatchIpv6DstPfxAdd");
    if (result == OPEN_E_NONE)
    {
      memset(&tmpIpAddr, 0, sizeof(tmpIpAddr));
      tmpPfxLen = 0;
      result = openapiAclRuleMatchIpv6DstPfxGet(clientHandle, aclId, aclRule, &tmpIpAddr, &tmpPfxLen);
      PRINTBADRESULT(result, "openapiAclRuleMatchIpv6DstPfxGet");
      if (result == OPEN_E_NONE)
      {
        inet_ntop(AF_INET6, &(tmpIpAddr.addr.ipv6), ipStrBuff, sizeof(ipStrBuff));
        PRINTSANITYRESULTS(result,
                           ((strcmp(ipStr, ipStrBuff) == 0) && (pfxLen == tmpPfxLen)),
                           __func__, "Destination IPv6 Address/prefix");
      }
    }
  }

  /* Add match criteria - DSCP value */
  {
    result = openapiAclRuleMatchDscpAdd(clientHandle, aclId, aclRule, OPENAPI_ACL_IP_DSCP_AF11);
    PRINTBADRESULT(result, "openapiAclRuleMatchDscpAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchDscpGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchDscpGet");
      PRINTSANITYRESULTS(result, OPENAPI_ACL_IP_DSCP_AF11==iTmp, __func__, "DSCP Value");
    }
  }

  /* Add match criteria - IP fragments */
  {
    result = openapiAclRuleMatchFragmentsAdd(clientHandle, aclId, aclRule, true);
    PRINTBADRESULT(result, "openapiAclRuleMatchFragmentsAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchFragmentsGet(clientHandle, aclId, aclRule, &bTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchFragmentsGet");
      PRINTSANITYRESULTS(result, true==bTmp, __func__, "IP Fragments");
    }
  }

  /* Add match criteria - Packet Redirection */
  {
    uint32_t intf;

    result = openapiIfFirstGet(clientHandle, OPEN_INTF_TYPE_PHY, &intf);
    PRINTBADRESULT(result, "openapiIfFirstGet");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchRedirectAdd(clientHandle, aclId, aclRule, intf);
      PRINTBADRESULT(result, "openapiAclRuleMatchRedirectAdd");
    }
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchRedirectGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchRedirectGet");
      PRINTSANITYRESULTS(result, intf==iTmp, __func__, "Packet Redirection Interface");
    }
  }

  /* Create a Permit action rule */
  {
    aclRule++; /* Increment to next rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_PERMIT);
    PRINTBADRESULT(result, "openapiAclRuleActionAdd - PERMIT");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleActionGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleActionGet - PERMIT");
      PRINTSANITYRESULTS(result, OPEN_ACL_PERMIT==iTmp, __func__, "PERMIT");
    }
  }

  /* Add match criteria - TCP protocol */
  {
    result = openapiAclRuleMatchProtocolAdd(clientHandle, aclId, aclRule, OPENAPI_ACL_PROTOCOL_TCP);
    PRINTBADRESULT(result, "openapiAclRuleMatchProtocolAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchProtocolGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchProtocolGet");
      PRINTSANITYRESULTS(result, OPENAPI_ACL_PROTOCOL_TCP==iTmp, __func__, "TCP Protocol");
    }
  }

  /* Add match criteria - Packet Redirection using an external agent */
  {
    result = openapiAclRuleMatchRedirectAgentAdd(clientHandle, aclId, aclRule, 100);
    PRINTBADRESULT(result, "openapiAclRuleMatchRedirectAgentAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchRedirectAgentGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchRedirectAgentGet");
      PRINTSANITYRESULTS(result, 100==iTmp, __func__, "Packet Redirect External Agent");
    }
  }

  /* Add match criteria - Rate Limit */
  {
    result = openapiAclRuleMatchRateLimitAdd(clientHandle, aclId, aclRule, 4294967295UL, 128);
    PRINTBADRESULT(result, "openapiAclRuleMatchRateLimitAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchRateLimitGet(clientHandle, aclId, aclRule, &iVal, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchRateLimitGet");
      PRINTSANITYRESULTS(result, (4294967295UL==iVal&&128==iTmp), __func__, "Simple traffic rate-limit policing");
    }
  }

  memset(&ipAclRulesFields, 0x0, sizeof(openIpAclRulesFields_t));
  result = openapiAclFieldsConfiguredMaskGet(clientHandle, aclId, aclRule, &ipAclRulesFields);
  if (OPEN_E_NONE == result)
  {
    if (0 != OPEN_ISMASKBITSET(ipAclRulesFields.fieldsConfigured, OPEN_IP_ACL_RATE_LIMIT))
    {
      printf("\r\nField Ratelimit is configured for ACL ID:%d ACL Rule:%d\r\n", aclId, aclRule);
    }
    else
    {
      printf("\r\nField Ratelimit is not configured for ACL ID:%d ACL Rule:%d\r\n", aclId, aclRule);
    }
  }
  else
  {
    printf("\r\n result:%d Bad code trying to get fields configured for"
           " ACL ID:%d ACL Rule:%d\r\n", result, aclId, aclRule);
  }
  /* Add match criteria - Control when rule is in effect based on time */
  {
    OPEN_ACL_RULE_STATUS_t status;
    open_buffdesc timeDesc;
    open_buffdesc tmpTimeDesc;
    char timeStr[32];
    char tmpStr[32];

    strcpy(timeStr, "ipv6-time-range-name");
    timeDesc.pstart = timeStr;
    timeDesc.size = strlen(timeStr) + 1;

    tmpTimeDesc.pstart = tmpStr;
    tmpTimeDesc.size = sizeof(tmpStr);

    result = openapiAclRuleMatchTimeRangeAdd(clientHandle, aclId, aclRule, &timeDesc);
    PRINTBADRESULT(result, "openapiAclRuleMatchTimeRangeAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchTimeRangeGet(clientHandle, aclId, aclRule, &tmpTimeDesc);
      PRINTBADRESULT(result, "openapiAclRuleMatchTimeRangeGet");
      PRINTSANITYRESULTS(result, (strcmp(timeDesc.pstart, tmpTimeDesc.pstart) == 0), __func__, "Time based activation");
    }
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchTimeRangeStatusGet(clientHandle, aclId, aclRule, &status);
      PRINTBADRESULT(result, "openapiAclRuleMatchTimeRangeStatusGet");
      PRINTSANITYRESULTS(result, OPEN_TRUE, __func__, "Time Range status");
    }
  }

  /* Add match criteria - Flow Label */
  {
    result = openapiAclRuleMatchFlowLabelAdd(clientHandle, aclId, aclRule, 1048575);
    PRINTBADRESULT(result, "openapiAclRuleMatchFlowLabelAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchFlowLabelGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchFlowLabelGet");
      PRINTSANITYRESULTS(result, (1048575==iTmp), __func__, "Flow Label");
    }
  }

  /* Add match criteria - Routing Flag */
  {
    result = openapiAclRuleMatchRoutingAdd(clientHandle, aclId, aclRule, true);
    PRINTBADRESULT(result, "openapiAclRuleMatchRoutingAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchRoutingGet(clientHandle, aclId, aclRule, &bTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchRoutingGet");
      PRINTSANITYRESULTS(result, (true==bTmp), __func__, "Routing Flag");
    }
  }

  /* Create a Permit action rule */
  {
    aclRule++; /* Increment to next rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_PERMIT);
    PRINTBADRESULT(result, "openapiAclRuleActionAdd - PERMIT");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleActionGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleActionGet - PERMIT");
      PRINTSANITYRESULTS(result, OPEN_ACL_PERMIT==iTmp, __func__, "PERMIT");
    }
  }

  /* Add match criteria - ICMPv6 protocol */
  {
    result = openapiAclRuleMatchProtocolAdd(clientHandle, aclId, aclRule, OPENAPI_ACL_PROTOCOL_ICMPV6);
    PRINTBADRESULT(result, "openapiAclRuleMatchProtocolAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchProtocolGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchProtocolGet");
      PRINTSANITYRESULTS(result, OPENAPI_ACL_PROTOCOL_ICMPV6==iTmp, __func__, "ICMPv6 Protocol");
    }
  }

  /* Add match criteria - ICMPv6 Port-Unreachable messaging */
  {
    uint32_t UNREACH_TYPE = 1;
    uint32_t UNREACH_CODE = 4;
    bool addCode = true;
    uint32_t tmpType;
    uint32_t tmpCode;

    result = openapiAclRuleMatchIcmpTypeCodeAdd(clientHandle, aclId, aclRule, addCode, UNREACH_TYPE, UNREACH_CODE);
    PRINTBADRESULT(result, "openapiAclRuleMatchIcmpTypeCodeAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchIcmpTypeCodeGet(clientHandle, aclId, aclRule, addCode, &tmpType, &tmpCode);
      PRINTBADRESULT(result, "openapiAclRuleMatchIcmpTypeCodeGet");
      PRINTSANITYRESULTS(result, (UNREACH_TYPE==tmpType&&UNREACH_CODE==tmpCode), __func__, "ICMPv6 messaging for 'Port-Unreachable'");
    }
  }

  /* Create a Permit action rule */
  {
    aclRule++; /* Increment to next rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_PERMIT);
    PRINTBADRESULT(result, "openapiAclRuleActionAdd - PERMIT");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleActionGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleActionGet - PERMIT");
      PRINTSANITYRESULTS(result, OPEN_ACL_PERMIT==iTmp, __func__, "PERMIT");
    }
  }

  /* Add match criteria - TCP protocol */
  {
    result = openapiAclRuleMatchProtocolAdd(clientHandle, aclId, aclRule, OPENAPI_ACL_PROTOCOL_TCP);
    PRINTBADRESULT(result, "openapiAclRuleMatchProtocolAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchProtocolGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchProtocolGet");
      PRINTSANITYRESULTS(result, OPENAPI_ACL_PROTOCOL_TCP==iTmp, __func__, "TCP Protocol");
    }
  }

  /* Add match criteria - TCP Flags */
  {
    uint32_t flagVal=0, flagMask=0;
    uint32_t tmpVal, tmpMask;

    /* Set Finish flag */
    flagMask |= (true << OPENAPI_ACL_TCP_FLAG_RST);
    flagVal  |= (true << OPENAPI_ACL_TCP_FLAG_RST);

    /* Set Synchronize flag */
    flagMask |= (true << OPENAPI_ACL_TCP_FLAG_PSH);
    flagVal  |= (true << OPENAPI_ACL_TCP_FLAG_PSH);

    /* Clear Acknowledge flag */
    flagMask |= (true << OPENAPI_ACL_TCP_FLAG_URG);
    flagVal  |= (false << OPENAPI_ACL_TCP_FLAG_URG);

    result = openapiAclRuleMatchTcpFlagsAdd(clientHandle, aclId, aclRule, flagVal, flagMask);
    PRINTBADRESULT(result, "openapiAclRuleMatchTcpFlagsAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchTcpFlagsGet(clientHandle, aclId, aclRule, &tmpVal, &tmpMask);
      PRINTBADRESULT(result, "openapiAclRuleMatchTcpFlagsGet");
      PRINTSANITYRESULTS(result, ((flagVal==tmpVal)&&(flagMask==tmpMask)), __func__, "Miscellaneous TCP Flags test");
    }
  }

  /* Add match criteria - Source Layer 4 port range from 20000 to 20010 */
  {
    OPEN_ACL_L4_PORT_OPERATOR_t tmpOper;
    uint32_t tmpStartPort;
    uint32_t tmpEndPort;

    result = openapiAclRuleMatchL4SrcPortAdd(clientHandle, aclId, aclRule, OPEN_ACL_L4_PORT_OPERATOR_RANGE, 20000, 20010);
    PRINTBADRESULT(result, "openapiAclRuleMatchL4SrcPortAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchL4SrcPortGet(clientHandle, aclId, aclRule, &tmpOper, &tmpStartPort, &tmpEndPort);
      PRINTBADRESULT(result, "openapiAclRuleMatchL4SrcPortGet");
      PRINTSANITYRESULTS(result,
                         ((OPEN_ACL_L4_PORT_OPERATOR_RANGE==tmpOper)&&(20000==tmpStartPort)&&(20010==tmpEndPort)),
                          __func__,
                         "Source Layer 4 port range 20000 - 20010");
    }
  }

  /* Add match criteria - Destination Layer 4 port > 60000 */
  {
    OPEN_ACL_L4_PORT_OPERATOR_t tmpOper;
    uint32_t tmpStartPort;
    uint32_t tmpEndPort;

    result = openapiAclRuleMatchL4DstPortAdd(clientHandle, aclId, aclRule, OPEN_ACL_L4_PORT_OPERATOR_GREATER_THAN, 60000, 0);
    PRINTBADRESULT(result, "openapiAclRuleMatchL4DstPortAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchL4DstPortGet(clientHandle, aclId, aclRule, &tmpOper, &tmpStartPort, &tmpEndPort);
      PRINTBADRESULT(result, "openapiAclRuleMatchL4DstPortGet");
      PRINTSANITYRESULTS(result, ((OPEN_ACL_L4_PORT_OPERATOR_GREATER_THAN==tmpOper)&&(60000==tmpStartPort)), __func__, "Destination Layer 4 port > 60000");
    }
  }

  /* Create a Deny action rule */
  {
    aclRule = 0; /* Let the system determine first available rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_DENY);
    PRINTBADRESULT(result, "openapiAclRuleActionAdd - DENY");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleActionGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleActionGet - DENY");
      PRINTSANITYRESULTS(result, OPEN_ACL_DENY==iTmp, __func__, "DENY");
    }
  }
}

/*********************************************************************
* @purpose  Create some named MAC ACL rules for demonstration purposes
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    aclId              @b{(input)}  Newly created ACL identifier
* 
* @returns  none
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testAclAddMacRules(openapiClientHandle_t *clientHandle, 
                         uint32_t aclId)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t aclRule;
  uint32_t iVal, iTmp;
  bool bTmp;
  openMacAclRulesFields_t macAclRulesFields;

  memset(&macAclRulesFields, 0x0, sizeof(openMacAclRulesFields_t));

  /* Create a Permit action rule */
  {
    aclRule = 0; /* Let the system determine first available rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_PERMIT);
    PRINTBADRESULT(result, "openapiAclRuleActionAdd - PERMIT");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleActionGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleActionGet - PERMIT");
      PRINTSANITYRESULTS(result, OPEN_ACL_PERMIT==iTmp, __func__, "PERMIT");
    }
  }

  /* Add match criteria - match-all */
  {
    result = openapiAclRuleMatchEveryAdd(clientHandle, aclId, aclRule, true);
    PRINTBADRESULT(result, "openapiAclRuleMatchEveryAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchEveryGet(clientHandle, aclId, aclRule, &bTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchEveryGet");
      PRINTSANITYRESULTS(result, true==bTmp, __func__, "Every");
    }
  }

  /* Add match criteria - assign max queue id */
  {
    result = openapiAclMaxAssignQueueGet(clientHandle, &iVal);
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchAssignQueueAdd(clientHandle, aclId, aclRule, iVal);
      PRINTBADRESULT(result, "openapiAclRuleMatchAssignQueueAdd");
    }
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchAssignQueueGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchAssignQueueGet");
      PRINTSANITYRESULTS(result, iVal==iTmp, __func__, "Assign Queue");
    }
  }

  memset(&macAclRulesFields, 0x0, sizeof(openMacAclRulesFields_t));
  result = openapiAclMacFieldsConfiguredMaskGet(clientHandle, aclId, aclRule, &macAclRulesFields);
  if (OPEN_E_NONE == result)
  {
    if (0 != OPEN_ISMASKBITSET(macAclRulesFields.fieldsConfigured, OPEN_MAC_ACL_ASSIGN_QUEUEID))
    {
      printf("\r\nField Assign Queue is configured for ACL ID:%d ACL Rule:%d\r\n", aclId, aclRule);
    }
    else
    {
      printf("\r\nField Assign Queue is not configured for ACL ID:%d ACL Rule:%d\r\n", aclId, aclRule);
    }
  }
  else
  {
    printf("\r\n result:%d Bad code trying to get fields configured for"
           " ACL ID:%d ACL Rule:%d\r\n", result, aclId, aclRule);
  }

  /* Add match criteria - logging */
  {
    result = openapiAclRuleMatchLoggingAdd(clientHandle, aclId, aclRule, true);
    PRINTBADRESULT(result, "openapiAclRuleMatchLoggingAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchLoggingGet(clientHandle, aclId, aclRule, &bTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchLoggingGet");
      PRINTSANITYRESULTS(result, true==bTmp, __func__, "Logging");
    }
  }

  /* Add match criteria - Packet Mirroring */
  {
    uint32_t intf;

    result = openapiIfFirstGet(clientHandle, OPEN_INTF_TYPE_PHY, &intf);
    PRINTBADRESULT(result, "openapiIfFirstGet");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchMirrorAdd(clientHandle, aclId, aclRule, intf);
      PRINTBADRESULT(result, "openapiAclRuleMatchMirrorAdd");
    }
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchMirrorGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchMirrorGet");
      PRINTSANITYRESULTS(result, intf==iTmp, __func__, "Packet Mirroring Interface");
    }
  }

  /* Create a Permit action rule */
  {
    aclRule++; /* Increment to next rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_PERMIT);
    PRINTBADRESULT(result, "openapiAclRuleActionAdd - PERMIT");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleActionGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleActionGet - PERMIT");
      PRINTSANITYRESULTS(result, OPEN_ACL_PERMIT==iTmp, __func__, "PERMIT");
    }
  }

  /* Add match criteria - source MAC address and mask */
  {
    open_buffdesc macDesc;
    open_buffdesc maskDesc;
    open_buffdesc tmpMacDesc;
    open_buffdesc tmpMaskDesc;
    char macStr[18];
    char maskStr[18];
    char tmpMacStr[18];
    char tmpMaskStr[18];
    strcpy(macStr, "00:11:22:33:44:55");
    strcpy(maskStr, "ff:ff:ff:00:00:00");

    macDesc.pstart = macStr;
    macDesc.size = strlen(macStr)+1;
    maskDesc.pstart = maskStr;
    maskDesc.size = strlen(maskStr)+1;

    tmpMacDesc.pstart = tmpMacStr;
    tmpMacDesc.size = sizeof(tmpMacStr);
    tmpMaskDesc.pstart = tmpMaskStr;
    tmpMaskDesc.size = sizeof(tmpMaskStr);

    result = openapiAclRuleMatchMacSrcMacAdd(clientHandle, aclId, aclRule, &macDesc, &maskDesc);
    PRINTBADRESULT(result, "openapiAclRuleMatchMacSrcMacAdd");

    result = openapiAclRuleMatchMacSrcMacGet(clientHandle, aclId, aclRule, &tmpMacDesc, &tmpMaskDesc);
    PRINTBADRESULT(result, "openapiAclRuleMatchMacSrcMacGet");

    if (result == OPEN_E_NONE)
    {
      PRINTSANITYRESULTS(result,
                         ((strcmp(macDesc.pstart,tmpMacDesc.pstart)==0)&&(strcmp(maskDesc.pstart,tmpMaskDesc.pstart)==0)),
                         __func__, "Source MAC Address/Mask");
    }
  }

  /* Add match criteria - destination MAC address and mask */
  {
    open_buffdesc macDesc;
    open_buffdesc maskDesc;
    open_buffdesc tmpMacDesc;
    open_buffdesc tmpMaskDesc;
    char macStr[18];
    char maskStr[18];
    char tmpMacStr[18];
    char tmpMaskStr[18];

    strcpy(macStr, "55:44:33:22:11:00");
    strcpy(maskStr, "00:00:00:ff:ff:ff");

    macDesc.pstart = macStr;
    macDesc.size = strlen(macStr)+1;
    maskDesc.pstart = maskStr;
    maskDesc.size = strlen(maskStr)+1;

    tmpMacDesc.pstart = tmpMacStr;
    tmpMacDesc.size = sizeof(tmpMacStr);
    tmpMaskDesc.pstart = tmpMaskStr;
    tmpMaskDesc.size = sizeof(tmpMaskStr);

    result = openapiAclRuleMatchMacDstMacAdd(clientHandle, aclId, aclRule, &macDesc, &maskDesc);
    PRINTBADRESULT(result, "openapiAclRuleMatchMacDstMacAdd");

    result = openapiAclRuleMatchMacDstMacGet(clientHandle, aclId, aclRule, &tmpMacDesc, &tmpMaskDesc);
    PRINTBADRESULT(result, "openapiAclRuleMatchMacDstMacGet");

    if (result == OPEN_E_NONE)
    {
      PRINTSANITYRESULTS(result,
                         ((strcmp(macDesc.pstart,tmpMacDesc.pstart)==0)&&(strcmp(maskDesc.pstart,tmpMaskDesc.pstart)==0)),
                         __func__, "Destination MAC Address/Mask");
    }
  }

  /* Add match criteria - Packet Redirection */
  {
    uint32_t intf;

    result = openapiIfFirstGet(clientHandle, OPEN_INTF_TYPE_PHY, &intf);
    PRINTBADRESULT(result, "openapiIfFirstGet");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchRedirectAdd(clientHandle, aclId, aclRule, intf);
      PRINTBADRESULT(result, "openapiAclRuleMatchRedirectAdd");
    }
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchRedirectGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchRedirectGet");
      PRINTSANITYRESULTS(result, intf==iTmp, __func__, "Packet Redirection Interface");
    }
  }

  /* Create a Permit action rule */
  {
    aclRule++; /* Increment to next rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_PERMIT);
    PRINTBADRESULT(result, "openapiAclRuleActionAdd - PERMIT");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleActionGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleActionGet - PERMIT");
      PRINTSANITYRESULTS(result, OPEN_ACL_PERMIT==iTmp, __func__, "PERMIT");
    }
  }

  /* Add match criteria - Packet Redirection using an external agent */
  {
    result = openapiAclRuleMatchRedirectAgentAdd(clientHandle, aclId, aclRule, 100);
    PRINTBADRESULT(result, "openapiAclRuleMatchRedirectAgentAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchRedirectAgentGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchRedirectAgentGet");
      PRINTSANITYRESULTS(result, 100==iTmp, __func__, "Packet Redirect External Agent");
    }
  }

  /* Add match criteria - Rate Limit */
  {
    result = openapiAclRuleMatchRateLimitAdd(clientHandle, aclId, aclRule, 4294967295UL, 128);
    PRINTBADRESULT(result, "openapiAclRuleMatchRateLimitAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchRateLimitGet(clientHandle, aclId, aclRule, &iVal, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchRateLimitGet");
      PRINTSANITYRESULTS(result, (4294967295UL==iVal&&128==iTmp), __func__, "Simple traffic rate-limit policing");
    }
  }

  /* Add match criteria - Control when rule is in effect based on time */
  {
    OPEN_ACL_RULE_STATUS_t status;
    open_buffdesc timeDesc;
    open_buffdesc tmpTimeDesc;
    char timeStr[32];
    char tmpStr[32];

    strcpy(timeStr, "mac-time-range-name");
    timeDesc.pstart = timeStr;
    timeDesc.size = strlen(timeStr) + 1;

    tmpTimeDesc.pstart = timeStr;
    tmpTimeDesc.size = sizeof(tmpStr)-1;

    result = openapiAclRuleMatchTimeRangeAdd(clientHandle, aclId, aclRule, &timeDesc);
    PRINTBADRESULT(result, "openapiAclRuleMatchTimeRangeAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchTimeRangeGet(clientHandle, aclId, aclRule, &tmpTimeDesc);
      PRINTBADRESULT(result, "openapiAclRuleMatchTimeRangeGet");
      PRINTSANITYRESULTS(result, (strcmp(timeDesc.pstart,tmpTimeDesc.pstart)==0), __func__, "Time based activation");
    }
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchTimeRangeStatusGet(clientHandle, aclId, aclRule, &status);
      PRINTBADRESULT(result, "openapiAclRuleMatchTimeRangeStatusGet");
      PRINTSANITYRESULTS(result, (status==status), __func__, "Time Range status");
    }
  }

  memset(&macAclRulesFields, 0x0, sizeof(openMacAclRulesFields_t));
  result = openapiAclMacFieldsConfiguredMaskGet(clientHandle, aclId, aclRule, &macAclRulesFields);
  if (OPEN_E_NONE == result)
  {
    if (0 != OPEN_ISMASKBITSET(macAclRulesFields.fieldsConfigured, OPEN_MAC_ACL_TIME_RANGE_NAME))
    {
      printf("\r\nField Timerange is configured for ACL ID:%d ACL Rule:%d\r\n", aclId, aclRule);
    }
    else
    {
      printf("\r\nField Timerange is not configured for ACL ID:%d ACL Rule:%d\r\n", aclId, aclRule);
    }
  }
  else
  {
    printf("\r\n result:%d Bad code trying to get fields configured for"
           " ACL ID:%d ACL Rule:%d\r\n", result, aclId, aclRule);
  }

  /* Add match criteria - Ethertype 0x809B */
  {
    result = openapiAclRuleMatchMacEtherTypeAdd(clientHandle, aclId, aclRule, 0x809B);
    PRINTBADRESULT(result, "openapiAclRuleMatchMacEtherTypeAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchMacEtherTypeGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchMacEtherTypeGet");
      PRINTSANITYRESULTS(result, (0x809B==iTmp), __func__, "Ethertype 0x809B");
    }
  }

  /* Add match criteria - COS 0 */
  {
    result = openapiAclRuleMatchMacCosAdd(clientHandle, aclId, aclRule, 0);
    PRINTBADRESULT(result, "openapiAclRuleMatchMacCosAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchMacCosGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchMacCosGet");
      PRINTSANITYRESULTS(result, (0==iTmp), __func__, "CoS 0");
    }
  }

  /* Add match criteria - Secondary COS 7 */
  {
    result = openapiAclRuleMatchMacCos2Add(clientHandle, aclId, aclRule, 7);
    PRINTBADRESULT(result, "openapiAclRuleMatchMacCos2Add");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchMacCos2Get(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleMatchMacCos2Get");
      PRINTSANITYRESULTS(result, (7==iTmp), __func__, "Secondary CoS 7");
    }
  }

  memset(&macAclRulesFields, 0x0, sizeof(openMacAclRulesFields_t));
  result = openapiAclMacFieldsConfiguredMaskGet(clientHandle, aclId, aclRule, &macAclRulesFields);
  if (OPEN_E_NONE == result)
  {
    if (0 != OPEN_ISMASKBITSET(macAclRulesFields.fieldsConfigured, OPEN_MAC_ACL_COS2))
    {
      printf("\r\nField CoS2 is configured for ACL ID:%d ACL Rule:%d\r\n", aclId, aclRule);
    }
    else
    {
      printf("\r\nField CoS2 is not configured for ACL ID:%d ACL Rule:%d\r\n", aclId, aclRule);
    }
  }
  else
  {
    printf("\r\n result:%d Bad code trying to get fields configured for"
           " ACL ID:%d ACL Rule:%d\r\n", result, aclId, aclRule);
  }

  /* Add match criteria - VLAN ID Range 2 - 100 */
  {
    uint32_t start = 2;
    uint32_t end = 100;
    OPEN_ACL_VLAN_OPERATOR_t oper = OPEN_ACL_VLAN_OPERATOR_RANGE;
    uint32_t tStart;
    uint32_t tEnd;
    OPEN_ACL_VLAN_OPERATOR_t tOper;

    result = openapiAclRuleMatchMacVlanAdd(clientHandle, aclId, aclRule, oper, start, end);
    PRINTBADRESULT(result, "openapiAclRuleMatchMacVlanAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchMacVlanGet(clientHandle, aclId, aclRule, &tOper, &tStart, &tEnd);
      PRINTBADRESULT(result, "openapiAclRuleMatchMacVlanGet");
      PRINTSANITYRESULTS(result, ((oper==tOper)&&(start==tStart)&&(end==tEnd)), __func__, "VLAN ID Range 2 - 100");
    }
  }

  /* Add match criteria - Secondary VLAN ID */
  {
    uint32_t start = 200;
    uint32_t end = 0;
    OPEN_ACL_VLAN_OPERATOR_t oper = OPEN_ACL_VLAN_OPERATOR_EQUAL_TO;
    uint32_t tStart;
    uint32_t tEnd;
    OPEN_ACL_VLAN_OPERATOR_t tOper;

    result = openapiAclRuleMatchMacSecondaryVlanAdd(clientHandle, aclId, aclRule, oper, start, end);
    PRINTBADRESULT(result, "openapiAclRuleMatchMacSecondaryVlanAdd");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleMatchMacSecondaryVlanGet(clientHandle, aclId, aclRule, &tOper, &tStart, &tEnd);
      PRINTBADRESULT(result, "openapiAclRuleMatchMacSecondaryVlanGet");
      PRINTSANITYRESULTS(result, ((oper==tOper)&&(start==tStart)), __func__, "Secondary VLAN ID 200");
    }
  }

  /* Create a Deny action rule */
  {
    aclRule = 0; /* Let the system determine first available rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_DENY);
    PRINTBADRESULT(result, "openapiAclRuleActionAdd - DENY");
    if (result == OPEN_E_NONE)
    {
      result = openapiAclRuleActionGet(clientHandle, aclId, aclRule, &iTmp);
      PRINTBADRESULT(result, "openapiAclRuleActionGet - DENY");
      PRINTSANITYRESULTS(result, OPEN_ACL_DENY==iTmp, __func__, "DENY");
    }
  }
}

/*********************************************************************
* @purpose  Assign an access group for the given ACL and interface
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    aclType            @b{(input)}  ACL type
* @param    aclName            @b{(input)}  ACL name
* @param    intf               @b{(input)}  internal interface number
* @param    dir                @b{(input)}  inbound or outbound
* @param    seqNum             @b{(input)}  ACL evaluation order sequence number
* 
* @returns  none
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testAclTrafficFilterIntfAdd(openapiClientHandle_t *clientHandle,                         
                                 OPEN_ACL_TYPE_t aclType,
                                 char *aclName,
                                 uint32_t intf,
                                 OPEN_ACL_DIRECTION_t dir,
                                 uint32_t seqNum)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc aclNameDesc;
  char aclNameStr[32];
  char str[256];
  uint32_t aclId;

  strcpy(aclNameStr, aclName);
  aclNameDesc.pstart = aclNameStr;
  aclNameDesc.size = strlen(aclNameStr)+1;

  if (result == OPEN_E_NONE)
  {
    result = openapiAclGet(clientHandle, aclType, &aclNameDesc, &aclId);
    PRINTBADRESULT(result, "openapiAclGet");
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiAclIntfDirAdd(clientHandle, intf, dir, aclId, seqNum);
    PRINTBADRESULT(result, "openapiAclIntfDirAdd");
    
    sprintf(str, "ACL:%s intf:%d dir:%d seq:%d", (char *)aclNameDesc.pstart, intf, dir, seqNum);
    PRINTSANITYRESULTS(result, result==OPEN_E_NONE, __func__, str);
  }
}

/*********************************************************************
* @purpose  Assign an access group for the given ACL and VLAN
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    aclType            @b{(input)}  ACL type
* @param    aclName            @b{(input)}  ACL name
* @param    vlan               @b{(input)}  VLAN id
* @param    dir                @b{(input)}  inbound or outbound
* @param    seqNum             @b{(input)}  ACL evaluation order sequence number
* 
* @returns  none
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testAclTrafficFilterVlanAdd(openapiClientHandle_t *clientHandle,                         
                                 OPEN_ACL_TYPE_t aclType,
                                 char *aclName,
                                 uint32_t vlan,
                                 OPEN_ACL_DIRECTION_t dir,
                                 uint32_t seqNum)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc aclNameDesc;
  char aclNameStr[32];
  char str[256];
  uint32_t aclId;

  strcpy(aclNameStr, aclName);
  aclNameDesc.pstart = aclNameStr;
  aclNameDesc.size = strlen(aclNameStr)+1;

  if (result == OPEN_E_NONE)
  {
    result = openapiAclGet(clientHandle, aclType, &aclNameDesc, &aclId);
    PRINTBADRESULT(result, "openapiAclGet");
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiAclVlanDirAdd(clientHandle, vlan, dir, aclId, seqNum);
    PRINTBADRESULT(result, "openapiAclVlanDirAdd");    

    sprintf(str, "ACL:%s VLAN:%d dir:%d seq:%d", (char *)aclNameDesc.pstart, vlan, dir, seqNum);
    PRINTSANITYRESULTS(result, result==OPEN_E_NONE, __func__, str);
  }
}

/*********************************************************************
* @purpose  Remove an access group for the given ACL and interface/direction
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    aclType            @b{(input)}  ACL type
* @param    aclName            @b{(input)}  ACL name
* @param    intf               @b{(input)}  internal interface number
* @param    dir                @b{(input)}  inbound or outbound
* 
* @returns  none
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testAclTrafficFilterIntfDelete(openapiClientHandle_t *clientHandle,                         
                                    OPEN_ACL_TYPE_t aclType,
                                    char *aclName,
                                    uint32_t intf,
                                    OPEN_ACL_DIRECTION_t dir)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc aclNameDesc;
  char aclNameStr[32];
  uint32_t aclId;

  strcpy(aclNameStr, aclName);
  aclNameDesc.pstart = aclNameStr;
  aclNameDesc.size = strlen(aclNameStr)+1;

  if (result == OPEN_E_NONE)
  {
    result = openapiAclGet(clientHandle, aclType, &aclNameDesc, &aclId);
    PRINTBADRESULT(result, "openapiAclGet");
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiAclIntfDirDelete(clientHandle, intf, dir, aclId);
    PRINTBADRESULT(result, "openapiAclIntfDirDelete");
  }
}

/*********************************************************************
* @purpose  Remove an access group for the given ACL and VLAN/direction
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    aclType            @b{(input)}  ACL type
* @param    aclName            @b{(input)}  ACL name
* @param    vlan               @b{(input)}  VLAN id
* @param    dir                @b{(input)}  inbound or outbound
* 
* @returns  none
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testAclTrafficFilterVlanDelete(openapiClientHandle_t *clientHandle,                         
                                    OPEN_ACL_TYPE_t aclType,
                                    char *aclName,
                                    uint32_t vlan,
                                    OPEN_ACL_DIRECTION_t dir)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc aclNameDesc;
  char aclNameStr[32];
  uint32_t aclId;

  strcpy(aclNameStr, aclName);
  aclNameDesc.pstart = aclNameStr;
  aclNameDesc.size = strlen(aclNameStr)+1;

  if (result == OPEN_E_NONE)
  {
    result = openapiAclGet(clientHandle, aclType, &aclNameDesc, &aclId);
    PRINTBADRESULT(result, "openapiAclGet");
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiAclVlanDirDelete(clientHandle, vlan, dir, aclId);
    PRINTBADRESULT(result, "openapiAclVlanDirDelete");
  }
}

/*********************************************************************
* @purpose  Enable and Disable ACL counter mode
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* 
* @returns  none
* 
* @notes    none
* 
* @end
*********************************************************************/
void testAclCountersAdminModeSet(openapiClientHandle_t *clientHandle) 
{
  open_error_t result = OPEN_E_NONE;

  result = openapiAclCountersAdminModeSet(clientHandle, ACL_COUNTERS_ENABLE);
  if (result != OPEN_E_NONE)
  {
    printf ("Unable to enable ACL counter mode\n");
  }
  else
  {
    printf ("Successfully enabled ACL counter mode\n");
  }

  printf ("Disable ACL counter mode ...\n");
  result = openapiAclCountersAdminModeSet(clientHandle, ACL_COUNTERS_DISABLE);
  if (result != OPEN_E_NONE)
  {
    printf ("Unable to disable ACL counter mode\n");
  }
  else
  {
    printf ("Successfully disabled ACL counter mode\n");
  }
}

/*********************************************************************
* @purpose  ACL add & remove on all interfaces, stats get & clear
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    aclType            @b{(input)}  ACL type
* @param    aclName            @b{(input)}  ACL name
* @param    seqNum             @b{(input)}  ACL evaluation order sequence number
* 
* @returns  none
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testAclStatsAndGlobalAddRemove(openapiClientHandle_t *clientHandle,                         
                                    OPEN_ACL_TYPE_t aclType,
                                    char *aclName,
                                    uint32_t seqNum)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc aclNameDesc;
  char aclNameStr[32];
  char str[256];
  uint32_t aclId;
  uint32_t aclRule;
  int asic = 0;
  uint64_t hitPktCount = 0;
  uint32_t nextRule;

  strcpy(aclNameStr, aclName);
  aclNameDesc.pstart = aclNameStr;
  aclNameDesc.size = strlen(aclNameStr)+1;

  result = openapiAclCountersAdminModeSet(clientHandle, ACL_COUNTERS_ENABLE);
  PRINTBADRESULT(result, "openapiAclCountersAdminModeSet");

  if (result == OPEN_E_NONE)
  {
    result = openapiAclGet(clientHandle, aclType, &aclNameDesc, &aclId);
    PRINTBADRESULT(result, "openapiAclGet");
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiAclRuleGlobalApply(clientHandle, asic, aclId, seqNum);
    PRINTBADRESULT(result, "openapiAclRuleGlobalApply");
    
    sprintf(str, "ACL:%s seq:%d apply on all interfaces ", (char *)aclNameDesc.pstart, seqNum);
    PRINTSANITYRESULTS(result, result==OPEN_E_NONE, __func__, str);
  }

  if (result == OPEN_E_NONE)
  {
    /* Get the last rule number */
    if (openapiAclRuleGetFirst(clientHandle, aclId, &aclRule) == OPEN_E_NONE)
    {
      if (aclRule != 0)
      {
        while (openapiAclRuleGetNext(clientHandle, aclId, aclRule, &nextRule) == OPEN_E_NONE)
        {
          aclRule = nextRule;
        }
      }
    }

    if (0 == aclRule)
    {
      PRINTBADRESULT(OPEN_E_ERROR, "openapiAclRuleGetLast");
    }
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiAclRuleCountGet(clientHandle, asic, aclId, aclRule, OPEN_ACL_PERMIT, (uint64_t *)&hitPktCount);

    sprintf(str, "ACL:%s Get counters for rule %d", (char *)aclNameDesc.pstart, aclRule);
    PRINTSANITYRESULTS(result, result==OPEN_E_NONE, __func__, str);
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiAclRuleCountersClear(clientHandle, asic, aclId, aclRule);

    sprintf(str, "ACL:%s Clear counters for rule %d", (char *)aclNameDesc.pstart, aclRule);
    PRINTSANITYRESULTS(result, result==OPEN_E_NONE, __func__, str);
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiAclRuleDelete(clientHandle, asic, aclId, aclRule);

    sprintf(str, "ACL:%s Delete ACL rule %d", (char *)aclNameDesc.pstart, aclRule);
    PRINTSANITYRESULTS(result, result==OPEN_E_NONE, __func__, str);
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiAclGlobalDelete(clientHandle, asic, aclId);
    PRINTBADRESULT(result, "openapiAclGlobalDelete");
    
    sprintf(str, "ACL:%s delete on all interfaces", (char *)aclNameDesc.pstart);
    PRINTSANITYRESULTS(result, result==OPEN_E_NONE, __func__, str);
  }
}

/*********************************************************************
* @purpose  Iterate and display interfaces that are inuse with matching direction
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* 
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void testAclShowDirectionInterfaces(openapiClientHandle_t *clientHandle)
{
  OPEN_ACL_DIRECTION_t dir = OPEN_ACL_INBOUND_ACL;
  uint32_t intf = 0;
  OPEN_CONTROL_t inUse;

  printf ("\nACL interface direction assignments");
  printf ("\nInterface Direction");
  printf ("\n--------- ---------");

  /* Iterate through all ACL capable interfaces */
  while (openapiAclIntfDirGetNext(clientHandle, intf, dir, &intf, &dir) == OPEN_E_NONE)
  {
    /* Capture all the interfaces with ACL assignments */
    if (openapiAclIsIntfInUse(clientHandle, intf, dir, &inUse) == OPEN_E_NONE)
    {
      if (inUse == OPEN_ENABLE)
      {
        printf("\n%9d %s", intf, (dir==OPEN_ACL_INBOUND_ACL) ? "Inbound" : "Outbound");
      }
    }
  }
}

/*********************************************************************
* @purpose  Retrieve a list of ACLs for the given interface and direction
*           and display its contents.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    intf               @b{(input)}  internal interface number
* @param    dir                @b{(input)}  inbound or outbound
* 
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void testAclShowInterfaceDirectionDetails(openapiClientHandle_t *clientHandle,
                                          uint32_t intf,
                                          OPEN_ACL_DIRECTION_t dir)
{
  OPEN_ACL_INTF_DIR_LIST_t listInfo;
  open_error_t result = OPEN_E_NONE;
  uint32_t idx;

  result = openapiAclIntfDirListGet(clientHandle, intf, dir, &listInfo);

  if (result == OPEN_E_NONE)
  {
    printf ("\n\nACL Interface %d, %s details", intf, (dir==OPEN_ACL_INBOUND_ACL) ? "Inbound" : "Outbound");
    printf ("\nACL id Type Seq Num");
    printf ("\n------ ---- -------");
    for (idx = 0; idx < listInfo.count; idx++)
    {
      printf("\n%6d %4d %7d",
             listInfo.listEntry[idx].aclId,
             listInfo.listEntry[idx].aclType,
             listInfo.listEntry[idx].seqNum);
    }
  }
}

/*********************************************************************
* @purpose  Retrieve a list of ACLs for the given VLAN and direction
*           and display its contents.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    vlan               @b{(input)}  VLAN id
* @param    dir                @b{(input)}  inbound or outbound
* 
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void testAclShowVlanDirectionDetails(openapiClientHandle_t *clientHandle,
                                     uint32_t vlan,
                                     OPEN_ACL_DIRECTION_t dir)
{
  OPEN_ACL_VLAN_DIR_LIST_t listInfo;
  open_error_t result = OPEN_E_NONE;
  uint32_t idx;

  result = openapiAclVlanDirListGet(clientHandle, vlan, dir, &listInfo);
  if (result == OPEN_E_NONE)
  {
    printf ("\n\nACL VLAN %d, %s details", vlan, (dir==OPEN_ACL_INBOUND_ACL) ? "Inbound" : "Outbound");
    printf ("\nACL id Type Seq Num");
    printf ("\n------ ---- -------");
    for (idx = 0; idx < listInfo.count; idx++)
    {
      printf("\n%6d %4d %7d",
             listInfo.listEntry[idx].aclId,
             listInfo.listEntry[idx].aclType,
             listInfo.listEntry[idx].seqNum);
    }
  }
}

/*********************************************************************
* @purpose  Retrieve a list of Interfaces for the given ACL and direction
*           and display its contents.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    aclType            @b{(input)}  ACL type
* @param    aclName            @b{(input)}  ACL name
* @param    dir                @b{(input)}  inbound or outbound
* 
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void testAclShowIntfAclDirectionDetails(openapiClientHandle_t *clientHandle,
                                        OPEN_ACL_TYPE_t aclType,
                                        char *aclName,
                                        OPEN_ACL_DIRECTION_t dir)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_ACL_ASSIGNED_INTF_LIST_t intfList;
  open_buffdesc aclNameDesc;
  char aclNameStr[32];
  uint32_t aclId;
  uint32_t idx;

  strcpy(aclNameStr, aclName);
  aclNameDesc.pstart = aclNameStr;
  aclNameDesc.size = strlen(aclNameStr)+1;

  if (result == OPEN_E_NONE)
  {
    result = openapiAclGet(clientHandle, aclType, &aclNameDesc, &aclId);
    PRINTBADRESULT(result, "openapiAclGet");
  }

  result = openapiAclAssignedIntfDirListGet(clientHandle, aclId, dir, &intfList);

  if (result == OPEN_E_NONE)
  {
    printf ("\n\nACL %s %s Interfaces", aclName, (dir==OPEN_ACL_INBOUND_ACL) ? "Inbound" : "Outbound");
    printf ("\nInterface");
    printf ("\n---------");
    for (idx = 0; idx < intfList.count; idx++)
    {
      printf("\n%d", intfList.intf[idx]);
    }
  }
}

/*********************************************************************
* @purpose  Retrieve a list of VLANs for the given ACL and direction
*           and display its contents.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    aclType            @b{(input)}  ACL type
* @param    aclName            @b{(input)}  ACL name
* @param    dir                @b{(input)}  inbound or outbound
* 
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void testAclShowVlanAclDirectionDetails(openapiClientHandle_t *clientHandle,
                                        OPEN_ACL_TYPE_t aclType,
                                        char *aclName,
                                        OPEN_ACL_DIRECTION_t dir)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_ACL_ASSIGNED_VLAN_LIST_t vlanList;
  open_buffdesc aclNameDesc;
  char aclNameStr[32];
  uint32_t aclId;
  uint32_t idx;

  strcpy(aclNameStr, aclName);
  aclNameDesc.pstart = aclNameStr;
  aclNameDesc.size = strlen(aclNameStr)+1;

  if (result == OPEN_E_NONE)
  {
    result = openapiAclGet(clientHandle, aclType, &aclNameDesc, &aclId);
    PRINTBADRESULT(result, "openapiAclGet");
  }

  result = openapiAclAssignedVlanDirListGet(clientHandle, aclId, dir, &vlanList);

  if (result == OPEN_E_NONE)
  {
    printf ("\n\nACL %s %s VLANs", aclName, (dir==OPEN_ACL_INBOUND_ACL) ? "Inbound" : "Outbound");
    printf ("\nVLAN");
    printf ("\n---");
    for (idx = 0; idx < vlanList.count; idx++)
    {
      printf("\n%d", vlanList.vlan[idx]);
    }
  }
}

/*********************************************************************
* @purpose  Apply an ACL as a route filter.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    aclType            @b{(input)}  ACL type
* @param    aclName            @b{(input)}  ACL name
* 
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void testAclRouteFilterAdd(openapiClientHandle_t *clientHandle,                         
                           OPEN_ACL_TYPE_t aclType,
                           char *aclName)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_ACL_ACTION_t aclAction = OPEN_ACL_DENY;
  open_buffdesc aclNameDesc;
  char aclNameStr[32];
  uint32_t aclId;
  uint32_t routePrefix = 0;
  uint32_t routeMask = 0;
  char *routePrefixStr = "10.10.10.1";
  char *routeMaskStr = "0.0.0.0";

  strcpy(aclNameStr, aclName);
  aclNameDesc.pstart = aclNameStr;
  aclNameDesc.size = strlen(aclNameStr)+1;

  if (result == OPEN_E_NONE)
  {
    result = openapiAclGet(clientHandle, aclType, &aclNameDesc, &aclId);
    PRINTBADRESULT(result, "openapiAclGet");
  }

  inet_pton(AF_INET, routePrefixStr, &routePrefix);
  inet_pton(AF_INET, routeMaskStr, &routeMask);
  /* IPv4 match criteria are in host byte order */
  routePrefix = ntohl(routePrefix);
  routeMask = ntohl(routeMask);

  if (result == OPEN_E_NONE)
  {
    result = openapiAclRouteFilter(clientHandle, aclId, routePrefix, routeMask, &aclAction);

    PRINTBADRESULT(result, "openapiAclRouteFilter - PERMIT");
    PRINTSANITYRESULTS(result, OPEN_ACL_PERMIT==aclAction, __func__, "PERMIT");
  }
}

/*********************************************************************
* @purpose  Retrieve a list of ACLs for the given VLAN and direction
*           and display its contents.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    vlan               @b{(input)}  VLAN id
* @param    dir                @b{(input)}  inbound or outbound
* 
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void testAclShowSummary(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;
  char str[256];
  uint32_t tmp;

  printf("\n\n");

  result = openapiCpuIntfGet(clientHandle, &tmp);
  PRINTBADRESULT(result, "openapiCpuIntfGet");
  sprintf(str, "CPU Control Plane Interface ... %d", tmp);
  PRINTSANITYRESULTS(result, (1==1), "openapiCpuIntfGet", str);

  result = openapiAclMaxAclIntfCountGet(clientHandle, &tmp);
  PRINTBADRESULT(result, "openapiAclMaxAclIntfCountGet");
  sprintf(str, "Number of allowed ACL Interfaces ... %d", tmp);
  PRINTSANITYRESULTS(result, (1==1), "openapiAclMaxAclIntfCountGet", str);

  result = openapiAclMaxAclVlanCountGet(clientHandle, &tmp);
  PRINTBADRESULT(result, "openapiAclMaxAclVlanCountGet");
  sprintf(str, "Number of allowed ACL VLANs ... %d", tmp);
  PRINTSANITYRESULTS(result, (1==1), "openapiAclMaxAclVlanCountGet", str);

  result = openapiAclCountGet(clientHandle, &tmp);
  PRINTBADRESULT(result, "openapiAclCountGet");
  sprintf(str, "Total number of configured ACLs ... %d", tmp);
  PRINTSANITYRESULTS(result, (1==1), "openapiAclCountGet", str);

  result = openapiAclMacCountGet(clientHandle, &tmp);
  PRINTBADRESULT(result, "openapiAclMacCountGet");
  sprintf(str, "Number of configured MAC ACLs ... %d", tmp);
  PRINTSANITYRESULTS(result, (1==1), "openapiAclMacCountGet", str);
}

/*********************************************************************
* @purpose  Display list of ACL IDs present in the system 
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* 
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void testAclDisplayAclIdList(openapiClientHandle_t *clientHandle)
{
  uint32_t aclId;

  printf("\n");

  if (OPEN_E_NONE == openapiAclIdGetFirst(clientHandle, &aclId))
  {
    do 
    {
      printf ("%u ", aclId);  
    } while (OPEN_E_NONE == openapiAclIdGetNext(clientHandle, aclId, &aclId)); 
  }
  else
  {
    printf("ACLs are not configured.");  
  }
  printf ("\n");
}
 
/*******************************************************************
*
* @brief  This is the main function that will demonstrate 
*         ACL OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main (int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result = OPEN_E_NONE;
  uint32_t aclId = 0;

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result =
       openapiClientRegister ("acl_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf ("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit (2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck (&clientHandle) != OPEN_E_NONE)
  {
    sleep (1);
  }

  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Starting ACL API example application");
  printf ("\n");

  /* Execute sanity tests */
  printf ("Begin Sanity tests...\n");

  printf ("Create named IP ACL...\n");
  testAclRename(&clientHandle, OPEN_ACL_TYPE_IP,   "test_ip_renamed",   "test_ip");
  result = testAclCreate(&clientHandle, OPEN_ACL_TYPE_IP, "test_ip", &aclId);
  if (result == OPEN_E_NONE)
  {
    testAclAddIpRules(&clientHandle, aclId);
  }

  printf ("Create named IPv6 ACL...\n");
  testAclRename(&clientHandle, OPEN_ACL_TYPE_IPV6, "test_ipv6_renamed", "test_ipv6");
  result = testAclCreate(&clientHandle, OPEN_ACL_TYPE_IPV6, "test_ipv6", &aclId);
  if (result == OPEN_E_NONE)
  {
    testAclAddIpv6Rules(&clientHandle, aclId);
  }

  printf ("Create named MAC ACL...\n");
  testAclRename(&clientHandle, OPEN_ACL_TYPE_MAC,  "test_mac_renamed",  "test_mac");
  result = testAclCreate(&clientHandle, OPEN_ACL_TYPE_MAC, "test_mac", &aclId);
  if (result == OPEN_E_NONE)
  {
    testAclAddMacRules(&clientHandle, aclId);
  }

  printf ("Perform some generic retrieval tests...\n");
  testAclRetrieval(&clientHandle);

  printf ("Rename then Delete ACLs...\n");
  testAclRename(&clientHandle, OPEN_ACL_TYPE_IP,   "test_ip",   "test_ip_renamed");
  testAclRename(&clientHandle, OPEN_ACL_TYPE_IPV6, "test_ipv6", "test_ipv6_renamed");
  testAclRename(&clientHandle, OPEN_ACL_TYPE_MAC,  "test_mac",  "test_mac_renamed");

  /* Traffic-Filters/Access-Groups */

  printf ("Demonstrate traffic filter assignment for interfaces...\n");
  testAclTrafficFilterIntfAdd(&clientHandle, OPEN_ACL_TYPE_IP,   "test_ip_renamed",   1, OPEN_ACL_INBOUND_ACL,  100);
  testAclTrafficFilterIntfAdd(&clientHandle, OPEN_ACL_TYPE_IP,   "test_ip_renamed",   2, OPEN_ACL_INBOUND_ACL,  200);
  testAclTrafficFilterIntfAdd(&clientHandle, OPEN_ACL_TYPE_IPV6, "test_ipv6_renamed", 3, OPEN_ACL_INBOUND_ACL,  300);
  testAclTrafficFilterIntfAdd(&clientHandle, OPEN_ACL_TYPE_MAC,  "test_mac_renamed",  4, OPEN_ACL_INBOUND_ACL,  400);

  printf ("Demonstrate traffic filter assignment for VLANs...\n");
  testAclTrafficFilterVlanAdd(&clientHandle, OPEN_ACL_TYPE_IP,   "test_ip_renamed",   5, OPEN_ACL_INBOUND_ACL,  500);
  testAclTrafficFilterVlanAdd(&clientHandle, OPEN_ACL_TYPE_IP,   "test_ip_renamed",   6, OPEN_ACL_INBOUND_ACL,  600);
  testAclTrafficFilterVlanAdd(&clientHandle, OPEN_ACL_TYPE_IPV6, "test_ipv6_renamed", 7, OPEN_ACL_INBOUND_ACL,  700);
  testAclTrafficFilterVlanAdd(&clientHandle, OPEN_ACL_TYPE_MAC,  "test_mac_renamed",  8, OPEN_ACL_INBOUND_ACL,  800);

  printf ("Demonstrate traffic filter removal of interface...\n");
  testAclTrafficFilterIntfDelete(&clientHandle, OPEN_ACL_TYPE_IP, "test_ip_renamed",  2, OPEN_ACL_INBOUND_ACL);

  printf ("Demonstrate traffic filter removal of VLAN...\n");
  testAclTrafficFilterVlanDelete(&clientHandle, OPEN_ACL_TYPE_IP, "test_ip_renamed",  5, OPEN_ACL_INBOUND_ACL);

  printf ("Demonstrate route filter...\n");
  testAclRouteFilterAdd(&clientHandle, OPEN_ACL_TYPE_IP, "test_ip_renamed");

  printf ("Enable ACL counter mode ...\n");
  testAclCountersAdminModeSet(&clientHandle);

  printf ("Test ACL add & remove on all interfaces, stats get & clear and deleting of ACL rule...\n");
  testAclStatsAndGlobalAddRemove(&clientHandle, OPEN_ACL_TYPE_IP,   "test_ip_renamed", 900);

  printf ("Display various ACL lists and information...\n");
  testAclShowDirectionInterfaces(&clientHandle);
  testAclShowInterfaceDirectionDetails(&clientHandle, 1, OPEN_ACL_INBOUND_ACL);
  testAclShowVlanDirectionDetails(&clientHandle, 4, OPEN_ACL_INBOUND_ACL);
  testAclShowIntfAclDirectionDetails(&clientHandle, OPEN_ACL_TYPE_IP, "test_ip_renamed", OPEN_ACL_INBOUND_ACL);
  testAclShowVlanAclDirectionDetails(&clientHandle, OPEN_ACL_TYPE_IP, "test_ip_renamed", OPEN_ACL_INBOUND_ACL);

  testAclShowSummary(&clientHandle);

  printf("\n\nDisplay list of ACL IDs present in the system.\n");
  testAclDisplayAclIdList(&clientHandle);

  printf ("\nComplete.\n");

  /* Log goodbye message with OPEN */
  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Stopping ACL API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

