/*********************************************************************
 *
 * Copyright 2016-2023 Broadcom.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 ***********************************************************************
 *
 * @filename  clitxtcfg_example.c
 *
 * @purpose   CLI text based configuration
 *
 * @detail    This is an example application using which you can
 *            exercise CLI Text Based Configuration APIs
 *
 * @component OpEN
 *
 * @note none
 *
 * @create    11/10/2014
 *
 * @end
 *
 **********************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/stat.h>
#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_cli_script.h"

char *get_current_dir_name (void);

#define MAX_ARGUMENTS 2
static char *fileArguments[MAX_ARGUMENTS];
static int errFlag = 0;

/*********************************************************************
*
* @purpose  Example application usage Information 
*
* @param  app  application name (clitxtcfg_example)
*
* @returns  None
*
* @notes
*
* @end
*********************************************************************/
void printUsage (char *app)
{
  printf ("Usage: %s [options]\n", app);
  printf ("  -a, <input_file> <output_file>   apply CLI configuration script\n");
  printf ("  -e, <input_file> <output_file>   execute show CLI commands stored in <input_file> and write output to file\n");
  printf ("  -g, <output_file>                generate running-config and write output to file\n");
  printf ("  -n, <input_file> <output_file>   gracefully apply CLI configuration script\n");
  printf ("  -s,                              save running-config to startup-config\n");
  printf ("  -v, <input_file> <output_file>   validate CLI configuration script\n");
  printf ("  -h, --help                       display this message\n");
  printf ("  -c, <input_file> <output_file>   compress configuration script\n");
  printf ("  -u, <input_file> <output_file>   uncompress configuration script\n");
  exit (EXIT_FAILURE);
}

/*********************************************************************
* @purpose validate file
*
* @param fileName
* @return 1 if the file is regular file
* @return 0 otherwise
*
* @comments
*
* @end
*********************************************************************/
int validateFile (char *fileName)
{
  int ret = 0;
  struct stat status;

  if (stat (fileName, &status) == -1)
  {
    return ret;
  }
  if (status.st_mode & S_IFREG)
  {
    ret = 1;
  }
  return ret;
}

/*********************************************************************
* @purpose capture the arguments and store them
*
* @param index   index of current argument
* @param argc    count of the arguments
* @param argv    an array of pointers to strings
* @param maxArgs maximum number of allowed arguments for an option
*
* @comments
*
* @end
*********************************************************************/
void validateArguments (int index, int argc, char **argv, int maxArgs)
{
  char *next;
  int args = 0;
  char *buf;
  char *pPosition;
  char *tmpStr;

  buf = (char *) get_current_dir_name ();
  if (maxArgs == 0)
  {
    if ((argc - index - 1) != maxArgs)
    {
      errFlag = 1;
      return;
    }
  }
  while (index < argc)
  {
    next = strdup (argv[index]);
    index++;
    if ((next[0] != '-') || (args < maxArgs))
    {
      pPosition = strchr (next, '/');
      if (pPosition != NULL)
      {
        fileArguments[args] = next;
      }
      else
      {
        tmpStr = (char *) malloc (2 + strlen (buf) + strlen (next));
        strcpy (tmpStr, buf);
        strcat (tmpStr, "/");
        strcat (tmpStr, next);
        fileArguments[args] = tmpStr;
      }
    }
    else
    {
      break;
    }
    args++;
  }

  if (args != maxArgs)
  {
    errFlag = 1;
  }
}

/*********************************************************************
* @purpose perform CLI Text Based configuration action
*
* @param opt action to perform on a given option
*
* @comments
*
* @end
*********************************************************************/
void performAction (int opt)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  open_buffdesc inputFileName;
  open_buffdesc outputFileName;
  open_buffdesc showCliCommand;
  char showCliCommandStr[100] = {0};
  FILE *fp = NULL;  

  l7proc_crashlog_register ();

  /* Register with OpEN */
  if ((result = openapiClientRegister ("cli text config",
                                       &client_handle)) != OPEN_E_NONE)
  {
    printf ("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit (2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck (&client_handle) != OPEN_E_NONE)
  {
    sleep (1);
  }

  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0,
               "Starting CLI Text Configuration API example application");

  printf ("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof (switch_os_revision_string);
  if (openapiNetworkOSVersionGet (&client_handle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf ("Network OS version = %s\n", switch_os_revision_string);
  }
  else
  {
    printf ("Network OS version retrieve error\n");
  }

  switch (opt)
  {
  case 'a':
    inputFileName.pstart = fileArguments[0];
    inputFileName.size = strlen (inputFileName.pstart) + 1;
    outputFileName.pstart = strdup (fileArguments[1]);
    if (outputFileName.pstart == 0)
    {
      printf ("Unable to allocate memory\n");
      exit (1);
    }
    outputFileName.size = strlen (outputFileName.pstart) + 1;

    if ((result = openapiTxtCfgApply (&client_handle, &inputFileName,
                                      &outputFileName)) != OPEN_E_NONE)
    {
      printf ("Bad return code trying to apply the configuration. (result = %d)\n", result);
    }
    else
    {
      printf ("Successfully applied the configuration. (result = %d)\n", result);
    }
    free(outputFileName.pstart);
    break;
  case 'e':
    outputFileName.pstart = strdup (fileArguments[1]);
    if (outputFileName.pstart == 0)
    {
      printf("Unable to allocate memory\n");
      exit(1);
    }
    outputFileName.size = strlen(outputFileName.pstart) + 1;

    fp = fopen(fileArguments[0], "r");
    if (fp == NULL)
    {
      printf("Could not open file for reading...\n");
      free(outputFileName.pstart);
      break;
    }

    while (fgets(showCliCommandStr, sizeof(showCliCommandStr), fp) != NULL)
    {
      /* strip new line */
      if (showCliCommandStr[strlen(showCliCommandStr) - 1] == '\n')
      {
        showCliCommandStr[strlen(showCliCommandStr) - 1] = '\0';
      }
      showCliCommand.pstart = showCliCommandStr;
      showCliCommand.size = strlen (outputFileName.pstart) + 1;
      if ((result = openapiCliShowExecuteToFile(&client_handle, &showCliCommand,
                                                &outputFileName)) != OPEN_E_NONE)
      {
        printf ("Bad return code trying to execute CLI show command. (result = %d) (command = %s)\n", result, showCliCommandStr);
      }
      else
      {
        printf ("Successfully executed CLI show command. (result = %d) (command = %s)\n", result, showCliCommandStr);
      }
    }
    fclose(fp);
    free(outputFileName.pstart);
    break;
  case 'n':
    inputFileName.pstart = fileArguments[0];
    inputFileName.size = strlen (inputFileName.pstart) + 1;
    outputFileName.pstart = strdup (fileArguments[1]);
    if (outputFileName.pstart == 0)
    {
      printf ("Unable to allocate memory\n");
      exit (1);
    }
    outputFileName.size = strlen (outputFileName.pstart) + 1;
    
    if ((result = openapiTxtCfgNDCApply (&client_handle, &inputFileName,
                                         &outputFileName)) != OPEN_E_NONE)
    {
      printf ("Bad return code trying to apply the configuration. (result = %d)\n", result);
    }
    else
    {
      printf ("Gracefully applied the configuration. (result = %d)\n", result);
    }
    free(outputFileName.pstart);
    break;
  case 'g':
    outputFileName.pstart = fileArguments[0];
    outputFileName.size = strlen (outputFileName.pstart) + 1;
    if ((result = openapiRunningCfgGenerate (&client_handle,
                                             &outputFileName)) != OPEN_E_NONE)
    {
      printf ("Bad return code trying to generate the running configuration. (result = %d)\n", result);
    }
    else
    {
      printf ("Successfully generated the running configuration. (result = %d)\n", result);
    }
    break;
  case 's':
    if ((result = openapiSaveConfig (&client_handle)) != OPEN_E_NONE)
    {
      printf ("Bad return code trying to save the configuration. (result = %d)\n", result);
    }
    else
    {
      printf ("Successfully saved the configuration. (result = %d)\n", result);
    }
    break;
  case 'v':
    inputFileName.pstart = fileArguments[0];
    inputFileName.size = strlen (inputFileName.pstart) + 1;
    outputFileName.pstart = fileArguments[1];
    outputFileName.size = strlen (outputFileName.pstart) + 1;
    if ((result = openapiTxtCfgValidate (&client_handle, &inputFileName,
                                         &outputFileName)) != OPEN_E_NONE)
    {
      printf ("Bad return code trying to validate the configuration. (result = %d)\n", result);
    }
    else
    {
      printf ("Successfully validated the configuration. (result = %d)\n", result);
    }
    break;
  case 'c':
    inputFileName.pstart = fileArguments[0];
    inputFileName.size = strlen(inputFileName.pstart);
    outputFileName.pstart = fileArguments[1];
    outputFileName.size = strlen (outputFileName.pstart);
    if ((result = openapiTxtCfgCompress(&client_handle, &inputFileName,
                                        &outputFileName)) != OPEN_E_NONE)
    {
      printf ("Bad return code trying to compress the configuration file. (result = %d)\n", result);
    }
    else
    {
      printf ("Successfully compressed the configuration file. (result = %d)\n", result);
    }
    break;

  case 'u':
    inputFileName.pstart = fileArguments[0];
    inputFileName.size = strlen(inputFileName.pstart);
    outputFileName.pstart = fileArguments[1];
    outputFileName.size = strlen (outputFileName.pstart);
    if ((result = openapiTxtCfgUnCompress(&client_handle, &inputFileName,
                                          &outputFileName)) != OPEN_E_NONE)
    {
      printf ("Bad return code trying to uncompress the configuration file. (result = %d)\n", result);
    }
    else
    {
      printf ("Successfully uncompressed the configuration file. (result = %d)\n", result);
    }
    break;
  }

  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0,
               "Stopping CLI Text Configuration API example application");
}

/*********************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates CLI Text Based configuration OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure
*
*********************************************************************/
int main (int argc, char **argv)
{
  int c;
  int applyFlag = 0;
  int saveFlag = 0;
  int generateFlag = 0;
  int validateFlag = 0;
  int allOpts = 0;
  int ndcFlag = 0;
  int execFlag = 0;
  int cmpFlag = 0;
  int unCmpFlag = 0;

  opterr = 0;
  while ((c = getopt (argc, argv, "hn:a:e:g:sv:c:u:")) != -1)
  {
    switch (c)
    {
    case 'a':
      applyFlag = 1;
      validateArguments (optind - 1, argc, argv, MAX_ARGUMENTS);
      break;
    case 'e':
      execFlag = 1;
      validateArguments (optind - 1, argc, argv, MAX_ARGUMENTS);
      break;
    case 'n':
      ndcFlag = 1;
      validateArguments (optind - 1, argc, argv, MAX_ARGUMENTS);
      break;
    case 'g':
      generateFlag = 1;
      validateArguments (optind - 1, argc, argv, 1);
      break;
    case 's':
      saveFlag = 1;
      validateArguments (optind - 1, argc, argv, 0);
      break;
    case 'v':
      validateFlag = 1;
      validateArguments (optind - 1, argc, argv, MAX_ARGUMENTS);
      break;
    case 'h':
      printUsage (argv[0]);
      break;
    case '?':
      printUsage (argv[0]);
      break;
    case 'c':
      cmpFlag = 1;
      validateArguments (optind - 1, argc, argv, MAX_ARGUMENTS);
      break;
    case 'u':
      unCmpFlag = 1;
      validateArguments (optind - 1, argc, argv, MAX_ARGUMENTS);
      break;
    default:
      printUsage (argv[0]);
    }
  }

  /* make sure only one option at a time and without errors in options */
  allOpts = applyFlag + generateFlag + saveFlag + validateFlag + ndcFlag + execFlag + cmpFlag + unCmpFlag;
  if ((allOpts > 1) || (allOpts == 0) || (argc == 1) || (errFlag == 1))
  {
    printUsage (argv[0]);
  }

  if (applyFlag)
  {
    if (validateFile (fileArguments[0]))
    {
      /* perform script apply action */
      performAction ('a');
    }
    else
    {
      printf ("Error: Not a valid file or file does not exist.\n");
    }
  }
  else if (execFlag)
  {
    if (validateFile (fileArguments[0]))
    {
      /* perform CLI commands execution to the file */
      performAction ('e');
    }
    else
    {
      printf ("Error: Not a valid file or file does not exist.\n");
    }
  }
  else if (ndcFlag)
  {
    if (validateFile (fileArguments[0]))
    {
      /* perform gracefull script apply action */
      performAction ('n');
    }
    else
    {
      printf ("Error: Not a valid file or file does not exist.\n");
    }
  }
  else if (generateFlag)
  {
    /* perform running configuration action */
    performAction ('g');
  }
  else if (saveFlag)
  {
    /* perform save running configuration to startup configuration action */
    performAction ('s');
  }
  else if (validateFlag)
  {
    if (validateFile (fileArguments[0]))
    {
      /* perform script validation action */
      performAction ('v');
    }
    else
    {
      printf ("Error: Not a valid file or file does not exist.\n");
    }
  }
  else if (cmpFlag)
  {
    if (validateFile (fileArguments[0]))
    {
      /* perform script compression action */
      performAction ('c');
    }
    else
    {
      printf ("Error: Not a valid file or file does not exist.\n");
    }
  }
  else if (unCmpFlag)
  {
    if (validateFile (fileArguments[0]))
    {
      /* perform script compression action */
      performAction ('u');
    }
    else
    {
      printf ("Error: Not a valid file or file does not exist.\n");
    }
  }
  else
  {
    printf ("unknown option\n");
  }

  return 0;
}

