
/*********************************************************************
*
*  Copyright 2012-2023 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  cos_example.c
*
* @purpose   class of service Example.
*
* @component OPEN
*
* @comments
*
* @create    09/05/2017
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_cos.h"

#define PRINTBADRESULT(result, msg) \
if (result==OPEN_E_UNAVAIL) { printf("Feature not supported - %s (err %d).\n", msg, result); } \
else if (result!=OPEN_E_NONE) { printf("Test Failure - %s (err %d).\n", msg, result); }

#define COS_WRED_DROP_PREC        0     /* Drop precedence 0 */
#define COS_WRED_MIN_THRESHOLD    22    /* WRED minimum threshold */
#define COS_WRED_MAX_THRESHOLD    55    /* WRED maximum threshold */
#define COS_WRED_ECN_ENABLE       1     /* Enable ECN */
#define COS_WRED_ENABLE           1     /* Enable WRED */
#define COS_WRED_DISABLE          0     /* Disable WRED */

static void displayCoqQueueInfo(char *pCosQueueInfo)
{
  char *pQueueInfo = pCosQueueInfo;
  char *tknComma;
  char qInfo[32] = {0};
  char *qInfoStr;
  char *tknColn;
  while ((tknComma = strtok_r(pQueueInfo, ",", &pQueueInfo)))
  {
    snprintf(qInfo, sizeof(qInfo),"%s", tknComma);
    qInfoStr = qInfo;
    while ((tknColn = strtok_r(qInfoStr, ":", &qInfoStr)))
    {
      printf("| %20s   |", tknColn);
    }
    printf("\n");
  }

}
/*********************************************************************
* @purpose  Set WRED drop parameters.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* 
* @returns  OPEN_E_NONE         If successful (If lag string is found)
*           OPEN_E_ERROR        On error.
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testWredDropParamsSet(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;
  uint8_t      wredMinThreshold    = COS_WRED_MIN_THRESHOLD;
  uint8_t      wredMaxThreshold    = COS_WRED_MAX_THRESHOLD;
  uint8_t      wredDropProbability = 100;
  uint8_t      pri                 = 1;

  result = openapiCosQueueWredDropParamsSet(clientHandle, OPEN_COS_ALL_INTERFACES, pri, COS_WRED_DROP_PREC,
                                            COS_WRED_ECN_ENABLE, wredMinThreshold, wredMaxThreshold, wredDropProbability);

  if (OPEN_E_NONE != result)
  {
    PRINTBADRESULT(result, "openapiCosQueueWredDropParamsSet");
  }

  return (result);
}

/*********************************************************************
* @purpose  Reset WRED drop parameters.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* 
* @returns  OPEN_E_NONE         If successful (If lag string is found)
*           OPEN_E_ERROR        On error.
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testWredDropParamsReset(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;
  uint8_t      pri                 = 1;

  result = openapiCosQueueWredDropParamsReset(clientHandle, OPEN_COS_ALL_INTERFACES, pri);

  if (OPEN_E_NONE != result)
  {
    PRINTBADRESULT(result, "openapiCosQueueWredDropParamsReset");
  }

  return (result);
}

/*********************************************************************
* @purpose  Enable and Disable Cos Queue WRED configuration.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* 
* @returns  OPEN_E_NONE         If successful (If lag string is found)
*           OPEN_E_ERROR        On error.
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testCosQueueWredSet(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;
  uint8_t      wredMinThreshold    = COS_WRED_MIN_THRESHOLD;
  uint8_t      wredMaxThreshold    = COS_WRED_MAX_THRESHOLD;
  uint8_t      wredDropProbability = 100;
  uint8_t      pri                 = 1;

  result = openapiCosQueueWredDropParamsSet(clientHandle, OPEN_COS_ALL_INTERFACES, pri, COS_WRED_DROP_PREC,
                                            COS_WRED_ECN_ENABLE, wredMinThreshold, wredMaxThreshold, wredDropProbability);

  if (OPEN_E_NONE == result)
  {
    printf("Enable WRED on all interfaces \n");
    result = openapiCosQueueWredSet(clientHandle, OPEN_COS_ALL_INTERFACES, pri, COS_WRED_ENABLE);
  
    if (OPEN_E_NONE != result)
    {
      PRINTBADRESULT(result, "testCosQueueWredSet");
      return (result);
    }
  }

  if (OPEN_E_NONE == result)
  {
    printf("Disable WRED on all interfaces \n");
    result = openapiCosQueueWredSet(clientHandle, OPEN_COS_ALL_INTERFACES, pri, COS_WRED_DISABLE);
  
    if (OPEN_E_NONE != result)
    {
      PRINTBADRESULT(result, "testCosQueueWredSet");
      return (result);
    }
  }
  return (result);
}

/*********************************************************************
* @purpose  Test COS queue WRED stats get.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* 
* @returns  OPEN_E_NONE         If successful (If lag string is found)
*           OPEN_E_ERROR        On error.
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testCosQueueWredStatsGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result              = OPEN_E_NONE;
  uint8_t      wredMinThreshold    = COS_WRED_MIN_THRESHOLD;
  uint8_t      wredMaxThreshold    = COS_WRED_MAX_THRESHOLD;
  uint8_t      wredDropProbability = 100;
  uint8_t      pri                 = 1;
  uint64_t     wredDropCount       = 0;
  uint64_t     ecnMarked           = 0;

  result = openapiCosQueueWredDropParamsSet(clientHandle, OPEN_COS_ALL_INTERFACES, pri, COS_WRED_DROP_PREC,
                                            COS_WRED_ECN_ENABLE, wredMinThreshold, wredMaxThreshold, wredDropProbability);

  if (OPEN_E_NONE == result)
  {
    printf("Enable WRED on all interfaces \n");
    result = openapiCosQueueWredSet(clientHandle, OPEN_COS_ALL_INTERFACES, pri, COS_WRED_ENABLE);
  
    if (OPEN_E_NONE == result)
    {
      printf("Get WRED stats from all interfaces \n");
      result = openapiCosQueueWredStatsGet(clientHandle, OPEN_COS_ALL_INTERFACES, &wredDropCount, &ecnMarked);
    }
  }

  if (OPEN_E_NONE != result)
  {
    PRINTBADRESULT(result, "testCosQueueWredStatsGet");
  }

  return (result);
}

/*********************************************************************
* @purpose  Test COS queue WRED stats clear.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* 
* @returns  OPEN_E_NONE         If successful (If lag string is found)
*           OPEN_E_ERROR        On error.
* 
* @notes    none
* 
* @end
*********************************************************************/
open_error_t testCosQueueWredStatsClear(openapiClientHandle_t *clientHandle)
{
  open_error_t result              = OPEN_E_NONE;
  uint8_t      wredMinThreshold    = COS_WRED_MIN_THRESHOLD;
  uint8_t      wredMaxThreshold    = COS_WRED_MAX_THRESHOLD;
  uint8_t      wredDropProbability = 100;
  uint8_t      pri                 = 1;

  result = openapiCosQueueWredDropParamsSet(clientHandle, OPEN_COS_ALL_INTERFACES, pri, COS_WRED_DROP_PREC,
                                            COS_WRED_ECN_ENABLE, wredMinThreshold, wredMaxThreshold, wredDropProbability);

  if (OPEN_E_NONE == result)
  {
    printf("Enable WRED on all interfaces \n");
    result = openapiCosQueueWredSet(clientHandle, OPEN_COS_ALL_INTERFACES, pri, COS_WRED_ENABLE);
  
    if (OPEN_E_NONE == result)
    {
      printf("Clear WRED stats from all interfaces \n");
      result = openapiCosQueueWredStatsClear(clientHandle, OPEN_COS_ALL_INTERFACES);
    }
  }

  if (OPEN_E_NONE != result)
  {
    PRINTBADRESULT(result, "testCosQueueWredStatsClear");
  }

  return (result);
}

/*******************************************************************
*
* @brief  This function prints the qoscos Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Sets the CoS global trust mode: %s 1 <trustMode>\n", name);
  printf("Test 2: Gets CoS global trust mode: %s 2 \n", name);
  printf("Test 3: Sets the CoS interface trust mode: %s 3 <intIfNum> <trustMode>\n", name);
  printf("Test 4: Gets CoS internal trust mode: %s 4 <intIfNum> \n", name);
  printf("Test 5: Gets specified trust mode interface index status: %s 5 <intIfNum>\n", name);
  printf("Test 6: Gets next sequential specified trust mode interface: %s 6 <intIfNum> \n", name);
  printf("Test 7: Gets CoS untrusted port default traffic class for given internal interface: %s 7 <intIfNum>\n", name);
  printf("Test 8: Gets specified queue config interface index status: %s 8 <intIfNum>\n", name);
  printf("Test 9: Gets next sequential specified queue config interface index: %s 9 <intIfNum> \n", name);
  printf("Test 10: Gets if the specified interface is valid for COS queue config: %s 10 <intIfNum> \n", name);
  printf("Test 11: Gets specified queueId  index status: %s 11 <queueId>\n", name);
  printf("Test 12: Gets next sequential specified queueId index: %s 12 <queueId>\n", name);
  printf("Test 13: Sets minimum bandwidth for a specific queue on a given interface: %s 13 <intIfNum> <queueId> <minBwVal>\n", name);
  printf("Test 14: Gets minimum bandwidth list for all queues on a given interface: %s 14 <intIfNum>\n", name);
  printf("Test 15: Sets scheduler type for a specific queue on a given interface: %s 15 <intIfNum> <queueId> <schTypeVal>\n", name);
  printf("Test 16: Gets scheduler type list for all queues on a given interface: %s 16 <intIfNum>\n", name);
  printf("Test 17: Sets queue management type for a specific queue on a given interface: %s 17 <intIfNum> <queueId> <qMgmtType>\n", name);
  printf("Test 18: Gets queue management type list for all queues on a given interface: %s 18 <intIfNum>\n", name);
  printf("Test 19: Sets dot1d traffic class: %s 19 <intIfNum> <priority> <tc>\n", name);
  printf("Test 20: Gets dot1d traffic class information: %s 20 <intIfNum> <priority>\n", name);
  printf("Test 21: Gets status of user priority to traffic class mapping for a given interface: %s 21 <intIfNum> <priority>\n", name);
  printf("Test 22: Gets next sequential user priority to traffic class mapping: %s 22 <intIfNum> <priority>\n", name);
  printf("Test 23: Gets status of specified IP DSCP mapping table index: %s 23 <dscp>\n", name);
  printf("Test 24: Gets next sequential ip DSCP mapping table index: %s 24 <dscp>\n", name);
  printf("Test 25: Sets the assigned traffic class (queue) for given Ip DSCP: %s 25 <intIfNum> <dscp> <tc>\n", name);
  printf("Test 26: Gets the assigned traffic class (queue) for given ip DSCP: %s 26 <intIfNum> <dscp>\n", name);
  printf("Test 27: Gets the default traffic class mapping for specified ip DSCP value: %s 27 <intIfNum> <dscp>\n", name);
  printf("Test 28: Restore default ip DSCP mappings for given interface: %s 28 <intIfNum>\n", name);
  printf("Test 29: Gets status of specified ip DSCP mapping table interface index: %s 29 <intIfNum>\n", name);
  printf("Test 30: Gets next sequential ip DSCP mapping table interface index: %s 30 <intIfNum>\n", name);
  printf("Test 31: Run API sanity checks: %s 31 \n", name);
  printf("Test 32: Gets interface information to denote global config operation: %s 32 \n", name);
  printf("Test 33: Gets factory default dot1d traffic class information: %s 33 <intIfNum> <priority>\n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    client_handle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *pClientHandle)
{
  open_error_t result;
  uint32_t nxtIntIfNum;
  uint32_t queueId = 1;
  OPEN_QOS_COS_MAP_TRUST_MODE_t trustMode;
  uint32_t priority;
  uint32_t tc;
  uint32_t dscp = 5;
  uint32_t intIfNum = 10;
  OPEN_BOOL_t isQueueIntfValid;
  uint32_t nxtqueueId;
  uint32_t nxtDscp;
  uint32_t maxCosQueueId;
  char buffer[255];
  open_buffdesc buf = {.pstart = buffer, .size = sizeof(buffer)};
  open_buffdesc zeroLenBuf = {.pstart = buffer, .size = 0};
  open_buffdesc badBufdescPointer = {.pstart = NULL, .size = sizeof(buffer)};

  printf("Testing qoscos OpEN APIs sanity:\n");

  printf("Testing openapiQosCosMapGlobalTrustModeGet():\n");
  result = openapiQosCosMapGlobalTrustModeGet(NULL, &trustMode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosCosMapGlobalTrustModeGet(pClientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiQosCosMapInterfaceTrustModeGet():\n");
  result = openapiQosCosMapInterfaceTrustModeGet(NULL, intIfNum, &trustMode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosCosMapInterfaceTrustModeGet(pClientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiQosCosMapTrustModeIntfIndexGetNext():\n");
  result = openapiQosCosMapTrustModeIntfIndexGetNext(NULL, intIfNum, &nxtIntIfNum);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosCosMapTrustModeIntfIndexGetNext(pClientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiQosCosMapUntrustedPortDefaultTrafficClassGet():\n");
  result = openapiQosCosMapUntrustedPortDefaultTrafficClassGet(NULL, intIfNum, &tc);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosCosMapUntrustedPortDefaultTrafficClassGet(pClientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiQosCosQueueIntfIndexGetNext():\n");
  result = openapiQosCosQueueIntfIndexGetNext(NULL, intIfNum, &nxtIntIfNum);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosCosQueueIntfIndexGetNext(pClientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiQosCosQueueIntfIsValid():\n");
  result = openapiQosCosQueueIntfIsValid(NULL, intIfNum, &isQueueIntfValid);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosCosQueueIntfIsValid(pClientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiQosCosQueueIdIndexGetNext():\n");
  result = openapiQosCosQueueIdIndexGetNext(NULL, queueId, &nxtqueueId);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosCosQueueIdIndexGetNext(pClientHandle, queueId, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiQosCosQueueMinBandwidthListGet():\n");
  result = openapiQosCosQueueMinBandwidthListGet(NULL, intIfNum, &buf);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosCosQueueMinBandwidthListGet(pClientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiQosCosQueueMinBandwidthListGet(pClientHandle, intIfNum, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiQosCosQueueMinBandwidthListGet(pClientHandle, intIfNum, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);

  printf("Testing openapiQosCosQueueSchedulerTypeListGet():\n");
  result = openapiQosCosQueueSchedulerTypeListGet(NULL, intIfNum, &buf);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosCosQueueSchedulerTypeListGet(pClientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiQosCosQueueSchedulerTypeListGet(pClientHandle, intIfNum, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiQosCosQueueSchedulerTypeListGet(pClientHandle, intIfNum, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);

  printf("Testing openapiQosCosQueueMgmtTypeListGet():\n");
  result = openapiQosCosQueueMgmtTypeListGet(NULL, intIfNum, &buf);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosCosQueueMgmtTypeListGet(pClientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiQosCosQueueMgmtTypeListGet(pClientHandle, intIfNum, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiQosCosQueueMgmtTypeListGet(pClientHandle, intIfNum, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);

  printf("Testing openapiQosCosDot1dTrafficClassGet():\n");
  result = openapiQosCosDot1dTrafficClassGet(NULL, intIfNum, priority, &tc);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosCosDot1dTrafficClassGet(pClientHandle, intIfNum, priority, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiQosCosDot1dDefaultTrafficClassGet():\n");
  result = openapiQosCosDot1dDefaultTrafficClassGet(NULL, intIfNum, priority, &tc);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosCosDot1dDefaultTrafficClassGet(pClientHandle, intIfNum, priority, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiQosCosDot1dTrafficClassEntryNextGet():\n");
  result = openapiQosCosDot1dTrafficClassEntryNextGet(NULL, &intIfNum, &priority);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosCosDot1dTrafficClassEntryNextGet(pClientHandle, NULL, &priority);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiQosCosDot1dTrafficClassEntryNextGet(pClientHandle, &intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiQosCosMapIpDscpIndexGetNext():\n");
  result = openapiQosCosMapIpDscpIndexGetNext(NULL, dscp, &nxtDscp);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosCosMapIpDscpIndexGetNext(pClientHandle, dscp, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiQosCosMapIpDscpTrafficClassGet():\n");
  result = openapiQosCosMapIpDscpTrafficClassGet(NULL, intIfNum, dscp, &tc);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosCosMapIpDscpTrafficClassGet(pClientHandle, intIfNum, dscp, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiQosCosMapIpDscpDefaultTrafficClassGet():\n");
  result = openapiQosCosMapIpDscpDefaultTrafficClassGet(NULL, intIfNum, dscp, &tc);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosCosMapIpDscpDefaultTrafficClassGet(pClientHandle, intIfNum, dscp, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiQosCosMapIpDscpIntfIndexGetNext():\n");
  result = openapiQosCosMapIpDscpIntfIndexGetNext(NULL, intIfNum, &nxtIntIfNum);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosCosMapIpDscpIntfIndexGetNext(pClientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf ("Verify WRED drop params set ...\n");
  result = testWredDropParamsSet(pClientHandle);
  printf ("\nComplete.\n");

  printf ("Verify WRED drop params reset ...\n");
  result = testWredDropParamsReset(pClientHandle);
  printf ("\nComplete.\n");

  printf ("Verify COS queue WRED set ...\n");
  result = testCosQueueWredSet(pClientHandle);
  printf ("\nComplete.\n");

  printf ("Verify COS queue WRED stats get ...\n");
  result = testCosQueueWredStatsGet(pClientHandle);
  printf ("\nComplete.\n");

  printf ("Verify COS queue WRED stats clear ...\n");
  result = testCosQueueWredStatsClear(pClientHandle);
  printf ("\nComplete.\n");

  printf ("Get the max COS queue ID value ...\n");
  if (OPEN_E_NONE == openapiCosQueueMaxIdGet(pClientHandle, &maxCosQueueId))
  {
    printf ("Max COS queue ID value is %u ...\n", maxCosQueueId);
  }
  else
  {
    printf ("Unable to get the max cos queue ID value ...\n");
  }

  return;
}


/*********************************************************************
* @purpose  Sets the CoS global trust mode.
*
* @param    pClientHandle   @b{(input)}  client handle from registration API
* @param    trustMode       @b{(input)}  Trust mode

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosMapGlobalTrustModeSet(openapiClientHandle_t *pClientHandle, OPEN_QOS_COS_MAP_TRUST_MODE_t trustMode)
{
  open_error_t result;

  if ((result = openapiQosCosMapGlobalTrustModeSet(pClientHandle, trustMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the CoS global trust mode. (result = %d)\n", result);
  }
  else
  {
    printf("Global trust mode is successfully set to %d.\n", trustMode);
  }
  return;
}


/*********************************************************************
* @purpose  Gets CoS global trust mode.
*
* @param    pClientHandle    @b{(input)}  client handle from registration API
* @param    pTrustMode       @b{(output)}  Trust mode

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosMapGlobalTrustModeGet(openapiClientHandle_t *pClientHandle, OPEN_QOS_COS_MAP_TRUST_MODE_t *pTrustMode)
{
  open_error_t result;

  if ((result = openapiQosCosMapGlobalTrustModeGet(pClientHandle, pTrustMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets CoS global trust mode. (result = %d)\n", result);
  }
  else
  {
    printf("Global Trust mode fetched information is %d. \n", *pTrustMode);
  }
  return;
}


/*********************************************************************
* @purpose  Sets the CoS interface trust mode.
*
* @param    pClientHandle   @b{(input)}  client handle from registration API
* @param    intIfNum        @b{(input)}  Internal interface number
* @param    trustMode       @b{(input)}  Trust mode

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosMapInterfaceTrustModeSet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, OPEN_QOS_COS_MAP_TRUST_MODE_t trustMode)
{
  open_error_t result;

  if ((result = openapiQosCosMapInterfaceTrustModeSet(pClientHandle, intIfNum, trustMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the CoS interface trust mode. (result = %d)\n", result);
  }
  else
  {
    printf("Trust mode of interface:%d is successfully set to %d.\n", intIfNum, trustMode);
  }
  return;
}


/*********************************************************************
* @purpose  Gets CoS internal trust mode.
*
* @param    pClientHandle     @b{(input)}  client handle from registration API
* @param    intIfNum          @b{(input)}  Internal interface number
* @param    pTrustMode        @b{(output)}  Trust mode

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosMapInterfaceTrustModeGet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, OPEN_QOS_COS_MAP_TRUST_MODE_t *pTrustMode)
{
  open_error_t result;

  if ((result = openapiQosCosMapInterfaceTrustModeGet(pClientHandle, intIfNum, pTrustMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets CoS internal trust mode. (result = %d)\n", result);
  }
  else
  {
    printf("Trust mode fetched information for interface %d is %d. \n", intIfNum, *pTrustMode);
  }
  return;
}


/*********************************************************************
* @purpose  Gets specified trust mode interface index status.
*
* @param    pClientHandle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosMapTrustModeIntfIndexGet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum)
{
  open_error_t result;

  if ((result = openapiQosCosMapTrustModeIntfIndexGet(pClientHandle, intIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets specified trust mode interface index status. (result = %d)\n", result);
  }
  else
  {
    printf("Trust mode exists for interface %d. \n",intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets next sequential specified trust mode interface.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum            @b{(input)}  Internal interface number
* @param    pNxtIntIfNum        @b{(output)}  Next internal interface number

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosMapTrustModeIntfIndexGetNext(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, uint32_t *pNxtIntIfNum)
{
  open_error_t result;

  if ((result = openapiQosCosMapTrustModeIntfIndexGetNext(pClientHandle, intIfNum, pNxtIntIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets next sequential specified trust mode interface. (result = %d)\n", result);
  }
  else
  {
    printf("Next trust mode interface is %d for the given interface %d.\n", *pNxtIntIfNum, intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets CoS untrusted port default traffic class for given internal interface.
*
* @param    pClientHandle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    pTcVal         @b{(output)}  untrusted traffic class value

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosMapUntrustedPortDefaultTrafficClassGet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, uint32_t *pTcVal)
{
  open_error_t result;

  if ((result = openapiQosCosMapUntrustedPortDefaultTrafficClassGet(pClientHandle, intIfNum, pTcVal)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets CoS untrusted port default traffic class for given internal interface. (result = %d)\n", result);
  }
  else
  {
    printf("Untrusted port default traffic class for interface %d is %d.\n", intIfNum, *pTcVal);
  }
  return;
}


/*********************************************************************
* @purpose  Gets specified queue config interface index status.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosQueueIntfIndexGet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum)
{
  open_error_t result;

  if ((result = openapiQosCosQueueIntfIndexGet(pClientHandle, intIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets specified queue config interface index status. (result = %d)\n", result);
  }
  else
  {
    printf("Queue config interface index exists for interface:%d.\n", intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets next sequential specified queue config interface index.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    pNxtIntIfNum       @b{(output)}  Next internal interface number

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosQueueIntfIndexGetNext(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, uint32_t *pNxtIntIfNum)
{
  open_error_t result;

  if ((result = openapiQosCosQueueIntfIndexGetNext(pClientHandle, intIfNum, pNxtIntIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets next sequential specified queue config interface index. (result = %d)\n", result);
  }
  else
  {
    printf("Next sequential interface in queue config after the given interface %d is %d\n", intIfNum, *pNxtIntIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets if the specified interface is valid for COS queue config.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    pIsQueueIntfValid       @b{(output)}  If interface is valid for queue config then OPEN_TRUE, else OPEN_FALSE

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosQueueIntfIsValid(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, OPEN_BOOL_t *pIsQueueIntfValid)
{
  open_error_t result;

  if ((result = openapiQosCosQueueIntfIsValid(pClientHandle, intIfNum, pIsQueueIntfValid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets if the specified interface is valid for COS queue config. (result = %d)\n", result);
  }
  else
  {
    printf("Interface %d has %s COS queue config.\n", intIfNum,
           (OPEN_TRUE == *pIsQueueIntfValid) ? "valid" : "invalid" );
  }
  return;
}


/*********************************************************************
* @purpose  Gets specified queueId  index status.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    queueId       @b{(input)}  queueId

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosQueueIdIndexGet(openapiClientHandle_t *pClientHandle, uint32_t queueId)
{
  open_error_t result;

  if ((result = openapiQosCosQueueIdIndexGet(pClientHandle, queueId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets specified queueId  index status. (result = %d)\n", result);
  }
  else
  {
    printf("Specified queueId index %d exists.\n", queueId);
  }
  return;
}


/*********************************************************************
* @purpose  Gets next sequential specified queueId index.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    queueId       @b{(input)}  Queue id
* @param    pNxtqueueId       @b{(output)}  Next Queue id

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosQueueIdIndexGetNext(openapiClientHandle_t *pClientHandle, uint32_t queueId, uint32_t *pNxtqueueId)
{
  open_error_t result;

  if ((result = openapiQosCosQueueIdIndexGetNext(pClientHandle, queueId, pNxtqueueId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets next sequential specified queueId index. (result = %d)\n", result);
  }
  else
  {
    printf("Next sequential queueId in queue config after the given queueId %d is %d\n", queueId, *pNxtqueueId);
  }
  return;
}


/*********************************************************************
* @purpose  Sets minimum bandwidth for a specific queue on a given interface.
*
* @param    pClientHandle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    queueId        @b{(input)}  Queue id
* @param    minBwVal       @b{(input)}  Min bandwidth value

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosQueueMinBandwidthSet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, uint32_t queueId, uint32_t minBwVal)
{
  open_error_t result;

  if ((result = openapiQosCosQueueMinBandwidthSet(pClientHandle, intIfNum, queueId, minBwVal)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets minimum bandwidth for a specific queue on a given interface. (result = %d)\n", result);
  }
  else
  {
    printf("Successfully set minimum bandwidth %d for queue %d on interface %d.\n", minBwVal, queueId, intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets minimum bandwidth list for all queues on a given interface.
*
* @param    pClientHandle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    pMinBwList     @b{(input)}  Minimum bandwidth list

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosQueueMinBandwidthListGet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, open_buffdesc *pMinBwList)
{
  open_error_t result;

  if ((result = openapiQosCosQueueMinBandwidthListGet(pClientHandle, intIfNum, pMinBwList)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets minimum bandwidth list for all queues on a given interface. (result = %d)\n", result);
  }
  else
  {
    printf("| %20s   || %20s   |\n","Queue ID", "Min Bandwidth" );
    displayCoqQueueInfo(pMinBwList->pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Sets scheduler type for a specific queue on a given interface.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    queueId       @b{(input)}  Queue id
* @param    schTypeVal       @b{(input)}  Scheduler type value

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosQueueSchedulerTypeSet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, uint32_t queueId, uint32_t schTypeVal)
{
  open_error_t result;

  if ((result = openapiQosCosQueueSchedulerTypeSet(pClientHandle, intIfNum, queueId, schTypeVal)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets scheduler type for a specific queue on a given interface. (result = %d)\n", result);
  }
  else
  {
    printf("Sets scheduler type %d for a queue %d on interface %d.\n", schTypeVal, queueId, intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets scheduler type list for all queues on a given interface.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    pSchTypeList       @b{(input)}  Scheduler type list

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosQueueSchedulerTypeListGet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, open_buffdesc *pSchTypeList)
{
  open_error_t result;

  if ((result = openapiQosCosQueueSchedulerTypeListGet(pClientHandle, intIfNum, pSchTypeList)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets scheduler type list for all queues on a given interface. (result = %d)\n", result);
  }
  else
  {
    printf("| %20s   || %20s   |\n","Queue ID", "Scheduler Type" );
    displayCoqQueueInfo(pSchTypeList->pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Sets queue management type for a specific queue on a given interface.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    queueId       @b{(input)}  Queue id
* @param    qMgmtType       @b{(input)}  Queue mgmt type value

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosQueueMgmtTypeSet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, uint32_t queueId, uint32_t qMgmtType)
{
  open_error_t result;

  if ((result = openapiQosCosQueueMgmtTypeSet(pClientHandle, intIfNum, queueId, qMgmtType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets queue management type for a specific queue on a given interface. (result = %d)\n", result);
  }
  else
  {
    printf("Sets queue management type %d for queue %d on a interface %d\n", intIfNum, queueId, qMgmtType);
  }
  return;
}


/*********************************************************************
* @purpose  Gets queue management type list for all queues on a given interface.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    pMinBwList       @b{(input)}  queue mgmt type list

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosQueueMgmtTypeListGet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, open_buffdesc *pQmgmtList)
{
  open_error_t result;

  if ((result = openapiQosCosQueueMgmtTypeListGet(pClientHandle, intIfNum, pQmgmtList)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets queue management type list for all queues on a given interface. (result = %d)\n", result);
  }
  else
  {
    printf("| %20s   || %20s   |\n","Queue ID", "Queue Management Type" );
    displayCoqQueueInfo(pQmgmtList->pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Sets dot1d traffic class.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    priority       @b{(input)}  priority
* @param    tc       @b{(input)}  traffic class

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosDot1dTrafficClassSet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, uint32_t priority, uint32_t tc)
{
  open_error_t result;

  if ((result = openapiQosCosDot1dTrafficClassSet(pClientHandle, intIfNum, priority, tc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets dot1d traffic class. (result = %d)\n", result);
  }
  else
  {
    printf("Sets dot1d traffic class %d for priority %d on interface %d \n", tc, priority, intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets  factory default dot1d traffic class information.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum            @b{(input)}  Internal interface number
* @param    priority            @b{(input)}  priority
* @param    pTc                 @b{(output)}  traffic class

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosDot1dDefaultTrafficClassGet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, uint32_t priority, uint32_t *pTc)
{
  open_error_t result;

  if ((result = openapiQosCosDot1dDefaultTrafficClassGet(pClientHandle, intIfNum, priority, pTc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets dot1d traffic class information. (result = %d)\n", result);
  }
  else
  {
    printf("Factory Default tc information fetched for priority :%d on interface %d is %d.\n", priority, intIfNum, *pTc);
  }
  return;
}

/*********************************************************************
* @purpose  Gets dot1d traffic class information.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    priority       @b{(input)}  priority
* @param    pTc       @b{(output)}  traffic class

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosDot1dTrafficClassGet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, uint32_t priority, uint32_t *pTc)
{
  open_error_t result;

  if ((result = openapiQosCosDot1dTrafficClassGet(pClientHandle, intIfNum, priority, pTc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets dot1d traffic class information. (result = %d)\n", result);
  }
  else
  {
    printf("Tc information fetched for priority :%d on interface %d is %d.\n", priority, intIfNum, *pTc);
  }
  return;
}

/*********************************************************************
* @purpose  Gets status of user priority to traffic class mapping for a given interface.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    priority       @b{(input)}  priority

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosDot1dTrafficClassEntryGet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, uint32_t priority)
{
  open_error_t result;

  if ((result = openapiQosCosDot1dTrafficClassEntryGet(pClientHandle, intIfNum, priority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets status of user priority to traffic class mapping for a given interface. (result = %d)\n", result);
  }
  else
  {
    printf("user priority to traffic class mapping is successfully fetched for priority:%d on interface:%d.\n", priority, intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets next sequential user priority to traffic class mapping.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    pIntIfNum       @b{(inout)}  Internal interface number
* @param    pPriority       @b{(inout)}  priority

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosDot1dTrafficClassEntryNextGet(openapiClientHandle_t *pClientHandle, uint32_t *pIntIfNum, uint32_t *pPriority)
{
  open_error_t result;
  uint32_t intIfNum = *pIntIfNum;
  uint32_t priority = *pPriority;

  if ((result = openapiQosCosDot1dTrafficClassEntryNextGet(pClientHandle, pIntIfNum, pPriority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets next sequential user priority to traffic class mapping. (result = %d)\n", result);
  }
  else
  {
    printf("Next sequential user priority to traffic class mapping fetched for priority:%d on interface:%d is priority%d and interface:%d .\n", 
        priority, intIfNum, *pPriority, *pIntIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets status of specified IP DSCP mapping table index.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    dscp       @b{(input)}  Ip DSCP

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosMapIpDscpIndexGet(openapiClientHandle_t *pClientHandle, uint32_t dscp)
{
  open_error_t result;

  if ((result = openapiQosCosMapIpDscpIndexGet(pClientHandle, dscp)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets status of specified IP DSCP mapping table index. (result = %d)\n", result);
  }
  else
  {
    printf("Successfully fetched IP DSCP mapping table index for dscp:%d.\n", dscp);
  }
  return;
}


/*********************************************************************
* @purpose  Gets next sequential ip DSCP mapping table index.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    dscp       @b{(input)}  Ip DSCP
* @param    pNxtDscp       @b{(output)}  Next Ip DSCP

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosMapIpDscpIndexGetNext(openapiClientHandle_t *pClientHandle, uint32_t dscp, uint32_t *pNxtDscp)
{
  open_error_t result;

  if ((result = openapiQosCosMapIpDscpIndexGetNext(pClientHandle, dscp, pNxtDscp)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets next sequential ip DSCP mapping table index. (result = %d)\n", result);
  }
  else
  {
    printf("Next sequential ip DSCP mapping table index for given dscp:%d is %d.\n", dscp, *pNxtDscp);
  }
  return;
}


/*********************************************************************
* @purpose  Sets the assigned traffic class (queue) for given Ip DSCP.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    dscp       @b{(input)}  Ip DSCP
* @param    tc       @b{(input)}  traffic class

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosMapIpDscpTrafficClassSet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, uint32_t dscp, uint32_t tc)
{
  open_error_t result;

  if ((result = openapiQosCosMapIpDscpTrafficClassSet(pClientHandle, intIfNum, dscp, tc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the assigned traffic class (queue) for given Ip DSCP. (result = %d)\n", result);
  }
  else
  {
    printf("Succesully set the assigned traffic class:%d  (queue) for Ip DSCP:%d on interface:%d. \n", tc, dscp, intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets the assigned traffic class (queue) for given ip DSCP.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    dscp       @b{(input)}  dscp
* @param    pTc       @b{(output)}  traffic class

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosMapIpDscpTrafficClassGet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, uint32_t dscp, uint32_t *pTc)
{
  open_error_t result;

  if ((result = openapiQosCosMapIpDscpTrafficClassGet(pClientHandle, intIfNum, dscp, pTc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the assigned traffic class (queue) for given ip DSCP. (result = %d)\n", result);
  }
  else
  {
    printf("Assigned traffic class (queue) for given ip DSCP:%d on interface:%d is %d.\n", dscp, intIfNum, *pTc);
  }
  return;
}


/*********************************************************************
* @purpose  Gets the default traffic class mapping for specified ip DSCP value.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    dscp       @b{(input)}  dscp
* @param    pTc       @b{(output)}  traffic class

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosMapIpDscpDefaultTrafficClassGet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, uint32_t dscp, uint32_t *pTc)
{
  open_error_t result;

  if ((result = openapiQosCosMapIpDscpDefaultTrafficClassGet(pClientHandle, intIfNum, dscp, pTc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the default traffic class mapping for specified ip DSCP value. (result = %d)\n", result);
  }
  else
  {
    printf("Default traffic class mapping for ip DSCP %d on interface %d is %d.\n", dscp, intIfNum, *pTc);
  }
  return;
}


/*********************************************************************
* @purpose  Restore default ip DSCP mappings for given interface.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosMapIpDscpDefaultsRestore(openapiClientHandle_t *pClientHandle, uint32_t intIfNum)
{
  open_error_t result;

  if ((result = openapiQosCosMapIpDscpDefaultsRestore(pClientHandle, intIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to restore default ip DSCP mappings for given interface. (result = %d)\n", result);
  }
  else
  {
    printf("Restore default ip DSCP mappings for interface %d.\n", intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets status of specified ip DSCP mapping table interface index.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosMapIpDscpIntfIndexGet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum)
{
  open_error_t result;

  if ((result = openapiQosCosMapIpDscpIntfIndexGet(pClientHandle, intIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets status of specified ip DSCP mapping table interface index. (result = %d)\n", result);
  }
  else
  {
    printf("Successfully fetched information of ip DSCP mapping table for interface :%d.\n", intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets next sequential ip DSCP mapping table interface index.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    pNxtIntIfNum       @b{(output)}  Next Internal interface number

*
* @returns  none
*
* @end
*********************************************************************/
void qosCosMapIpDscpIntfIndexGetNext(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, uint32_t *pNxtIntIfNum)
{
  open_error_t result;

  if ((result = openapiQosCosMapIpDscpIntfIndexGetNext(pClientHandle, intIfNum, pNxtIntIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets next sequential ip DSCP mapping table interface index. (result = %d)\n", result);
  }
  else
  {
    printf("Next sequential index for ip DSCP mapping table for interface %d is %d.\n", intIfNum, *pNxtIntIfNum);
  }
  return;
}


/*******************************************************************
*
* @brief  This is the main function that will demonstrate 
*         COS OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main (int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result = OPEN_E_NONE;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  OPEN_QOS_COS_MAP_TRUST_MODE_t trustMode;
  uint32_t intIfNum, nxtIntIfNum, tcVal, queueId, nxtQueueId;
  uint32_t minBwVal, schTypeVal, qMgmtType, priority, dscp, nxtDscp;
  OPEN_BOOL_t isQueueIntfValid;
  char qInfo[255] = {0};
  open_buffdesc qInfoBufd = {.pstart = qInfo, .size = sizeof(qInfo)};
  

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result =
       openapiClientRegister ("cos_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf ("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit (2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck (&clientHandle) != OPEN_E_NONE)
  {
    sleep (1);
  }

  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Starting COS API example application");

  printf ("\n");

 printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        trustMode = atoi(argv[2]);
        qosCosMapGlobalTrustModeSet(&clientHandle, trustMode);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 2)
      {
        qosCosMapGlobalTrustModeGet(&clientHandle, &trustMode);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 4)
      {
        intIfNum = atoi(argv[2]);
        trustMode = atoi(argv[3]);
        qosCosMapInterfaceTrustModeSet(&clientHandle, intIfNum, trustMode);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 3)
      {
        intIfNum = atoi(argv[2]);
        qosCosMapInterfaceTrustModeGet(&clientHandle, intIfNum, &trustMode);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 3)
      {
        intIfNum = atoi(argv[2]);
        qosCosMapTrustModeIntfIndexGet(&clientHandle, intIfNum);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 3)
      {
        intIfNum = atoi(argv[2]);
        qosCosMapTrustModeIntfIndexGetNext(&clientHandle, intIfNum, &nxtIntIfNum);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 3)
      {
        intIfNum = atoi(argv[2]);
        qosCosMapUntrustedPortDefaultTrafficClassGet(&clientHandle, intIfNum, &tcVal);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 3)
      {
        intIfNum = atoi(argv[2]);
        qosCosQueueIntfIndexGet(&clientHandle, intIfNum);
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 3)
      {
        intIfNum = atoi(argv[2]);
        qosCosQueueIntfIndexGetNext(&clientHandle, intIfNum, &nxtIntIfNum);
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 3)
      {
        intIfNum = atoi(argv[2]);
        qosCosQueueIntfIsValid(&clientHandle, intIfNum, &isQueueIntfValid);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 3)
      {
        queueId = atoi(argv[2]);
        qosCosQueueIdIndexGet(&clientHandle, queueId);
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 3)
      {
        queueId = atoi(argv[2]);
        qosCosQueueIdIndexGetNext(&clientHandle, queueId, &nxtQueueId);
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 5)
      {
        intIfNum = atoi(argv[2]);
        queueId = atoi(argv[3]);
        minBwVal = atoi(argv[4]);
        qosCosQueueMinBandwidthSet(&clientHandle, intIfNum, queueId, minBwVal);
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 3)
      {
        intIfNum = atoi(argv[2]);
        memset(qInfo, 0, sizeof(qInfo));
        qInfoBufd.size = sizeof(qInfo);
        qosCosQueueMinBandwidthListGet(&clientHandle, intIfNum, &qInfoBufd);
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 5)
      {
        intIfNum = atoi(argv[2]);
        queueId = atoi(argv[3]);
        schTypeVal = atoi(argv[4]);
        qosCosQueueSchedulerTypeSet(&clientHandle, intIfNum, queueId, schTypeVal);
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 3)
      {
        intIfNum = atoi(argv[2]);
        memset(qInfo, 0, sizeof(qInfo));
        qInfoBufd.size = sizeof(qInfo);
        qosCosQueueSchedulerTypeListGet(&clientHandle, intIfNum, &qInfoBufd);
        show_help = 0;
      }
      break;
    case 17:
      if (argc == 5)
      {
        intIfNum = atoi(argv[2]);
        queueId = atoi(argv[3]);
        qMgmtType = atoi(argv[4]);
        qosCosQueueMgmtTypeSet(&clientHandle, intIfNum, queueId, qMgmtType);
        show_help = 0;
      }
      break;
    case 18:
      if (argc == 3)
      {
        intIfNum = atoi(argv[2]);
        memset(qInfo, 0, sizeof(qInfo));
        qInfoBufd.size = sizeof(qInfo);
        qosCosQueueMgmtTypeListGet(&clientHandle, intIfNum, &qInfoBufd);
        show_help = 0;
      }
      break;
    case 19:
      if (argc == 5)
      {
        intIfNum = atoi(argv[2]);
        priority = atoi(argv[3]);
        tcVal = atoi(argv[4]);
        qosCosDot1dTrafficClassSet(&clientHandle, intIfNum, priority, tcVal);
        show_help = 0;
      }
      break;
    case 20:
      if (argc == 4)
      {
        intIfNum = atoi(argv[2]);
        priority = atoi(argv[3]);
        qosCosDot1dTrafficClassGet(&clientHandle, intIfNum, priority, &tcVal);
        show_help = 0;
      }
      break;
    case 21:
      if (argc == 4)
      {
        intIfNum = atoi(argv[2]);
        priority = atoi(argv[3]);
        qosCosDot1dTrafficClassEntryGet(&clientHandle, intIfNum, priority);
        show_help = 0;
      }
      break;
    case 22:
      if (argc == 4)
      {
        intIfNum = atoi(argv[2]);
        priority = atoi(argv[3]);
        qosCosDot1dTrafficClassEntryNextGet(&clientHandle, &intIfNum, &priority);
        show_help = 0;
      }
      break;
    case 23:
      if (argc == 3)
      {
        dscp = atoi(argv[2]);
        qosCosMapIpDscpIndexGet(&clientHandle, dscp);
        show_help = 0;
      }
      break;
    case 24:
      if (argc == 3)
      {
        dscp = atoi(argv[2]);
        qosCosMapIpDscpIndexGetNext(&clientHandle, dscp, &nxtDscp);
        show_help = 0;
      }
      break;
    case 25:
      if (argc == 5)
      {
        intIfNum = atoi(argv[2]);
        dscp = atoi(argv[3]);
        tcVal = atoi(argv[4]);
        qosCosMapIpDscpTrafficClassSet(&clientHandle, intIfNum, dscp, tcVal);
        show_help = 0;
      }
      break;
    case 26:
      if (argc == 4)
      {
        intIfNum = atoi(argv[2]);
        dscp = atoi(argv[3]);
        qosCosMapIpDscpTrafficClassGet(&clientHandle, intIfNum, dscp, &tcVal);
        show_help = 0;
      }
      break;
    case 27:
      if (argc == 4)
      {
        intIfNum = atoi(argv[2]);
        dscp = atoi(argv[3]);
        qosCosMapIpDscpDefaultTrafficClassGet(&clientHandle, intIfNum, dscp, &tcVal);
        show_help = 0;
      }
      break;
    case 28:
      if (argc == 3)
      {
        intIfNum = atoi(argv[2]);
        qosCosMapIpDscpDefaultsRestore(&clientHandle, intIfNum);
        show_help = 0;
      }
      break;
    case 29:
      if (argc == 3)
      {
        intIfNum = atoi(argv[2]);
        qosCosMapIpDscpIntfIndexGet(&clientHandle, intIfNum);
        show_help = 0;
      }
      break;
    case 30:
      if (argc == 3)
      {
        intIfNum = atoi(argv[2]);
        qosCosMapIpDscpIntfIndexGetNext(&clientHandle, intIfNum, &nxtIntIfNum);
        show_help = 0;
      }
      break;
    case 31:
      if (argc == 2)
      {
        runSanity(&clientHandle);
        show_help = 0;
      }
      break;
    case 32:
      if (argc == 2)
      {
        openapiQosCosMapGetIntfIndexForGlobalConfig(&clientHandle, &intIfNum);
        printf("Interface number for global config is %d.\n", intIfNum);
        show_help = 0;
      }
      break;
    case 33:
      if (argc == 4)
      {
        intIfNum = atoi(argv[2]);
        priority = atoi(argv[3]);
        qosCosDot1dDefaultTrafficClassGet(&clientHandle, intIfNum, priority, &tcVal);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OPEN */
  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Stopping COS API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
