/*********************************************************************
*
* Copyright 2017-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  dcbx_example.c
*
* @purpose   Data center bridging - DCBX API Example.
*
* @component OPEN
*
* @comments
*
* @create    1/31/2017
*
* @end
*
**********************************************************************/

#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_dcbx.h"

static char *dcbxVersionToStr(OPEN_DCBX_VERSION_t version)
{
  char *ret = "Unknown";

  if (version == OPEN_DCBX_AUTO)
  {
    ret = "OPEN_DCBX_AUTO";
  }   
  else if (version == OPEN_DCBX_IEEE)
  {
    ret = "OPEN_DCBX_IEEE";
  }
  else if (version == OPEN_DCBX_CIN)
  {
    ret = "OPEN_DCBX_CIN";
  }
  else if (version == OPEN_DCBX_CEE)
  {
    ret = "OPEN_DCBX_CEE";
  }
  return ret;
}

static char *dcbxOpStatusToStr(OPEN_DCBX_OP_STATUS status)
{
  char *ret = "Unknown";

  if (status == OPEN_DCBX_OP_STATUS_ENABLE)
  {
    ret = "OPEN_DCBX_OP_STATUS_ENABLE";
  }   
  else if (status == OPEN_DCBX_OP_STATUS_DISABLE)
  {
    ret = "OPEN_DCBX_OP_STATUS_DISABLE";
  }
  else if (status == OPEN_DCBX_OP_STATUS_MULTI_PEER)
  {
    ret = "OPEN_DCBX_OP_STATUS_MULTI_PEER";
  }
  return ret;
}

static char *dcbxPortRoleToStr(OPEN_DCBX_PORT_ROLE_t role)
{
  char *ret = "Unknown";

  if (role == OPEN_DCBX_PORT_ROLE_MANUAL)
  {
    ret = "OPEN_DCBX_PORT_ROLE_MANUAL";
  }   
  else if (role == OPEN_DCBX_PORT_ROLE_AUTO_UPSTREAM)
  {
    ret = "OPEN_DCBX_PORT_ROLE_AUTO_UPSTREAM";
  }
  else if (role == OPEN_DCBX_PORT_ROLE_AUTO_DOWNSTREAM)
  {
    ret = "OPEN_DCBX_PORT_ROLE_AUTO_DOWNSTREAM";
  }
  else if (role == OPEN_DCBX_PORT_ROLE_AUTO_CFG_SRC)
  {
    ret = "OPEN_DCBX_PORT_ROLE_AUTO_CFG_SRC";
  }
  return ret;
}

void dumpDcbxCaps(OPEN_DCBX_CAPABILITIES_t mask)
{
  if (mask & OPEN_DCBX_CAP_PFC_BITMASK)
  {
    printf("  OPEN_DCBX_CAP_PFC_BITMASK\n");
  }
  if (mask & OPEN_DCBX_CAP_ETS_CFG_BITMASK)
  {
    printf("  OPEN_DCBX_CAP_ETS_CFG_BITMASK\n");
  }
  if (mask & OPEN_DCBX_CAP_ETS_REC_BITMASK)
  {
    printf("  OPEN_DCBX_CAP_ETS_REC_BITMASK\n");
  }
  if (mask & OPEN_DCBX_CAP_APP_PRI_BITMASK)
  {
    printf("  OPEN_DCBX_CAP_APP_PRI_BITMASK\n");
  }
  if (mask & OPEN_DCBX_CAP_CONTROL_BITMASK)
  {
    printf("  OPEN_DCBX_CAP_CONTROL_BITMASK\n");
  }
}

/*********************************************************************
* @purpose  Set DCBX version for a given interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   interface number
* @param    version          @b{(input)}   desired version
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/

void setDcbxVersion(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_DCBX_VERSION_t version)
{
  open_error_t ret;

  ret = openapiDcbxVersionSet(client_handle, intIfNum, version);
  if (ret != OPEN_E_NONE)
  {
    printf("unable to set version %s for interface %d: error %d\n", dcbxVersionToStr(version), intIfNum, ret);
  }
}

/*********************************************************************
* @purpose  Set bitmask of enabled TLVs
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   interface number
* @param    dcbxCap          @b{(input)}   mask of enabled DCBX TLVs.
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/

void setDcbxTLVs(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_DCBX_CAPABILITIES_t dcbxCaps)
{
  open_error_t ret;

  ret = openapiDcbxPortConfigTLVsEnabledSet(client_handle, intIfNum, dcbxCaps);
  if (ret != OPEN_E_NONE)
  {
    printf("unable to set caps %x for interface %d: error %d\n", dcbxCaps, intIfNum, ret);
  }
}

/*********************************************************************
* @purpose  Set auto config port role.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   interface number
* @param    role             @b{(input)}   desired role
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/

void setPortRole(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_DCBX_PORT_ROLE_t role)
{
  open_error_t ret;

  ret = openapiDcbxAutoCfgPortRoleSet(client_handle, intIfNum, role);
  if (ret == OPEN_E_UNAVAIL)
  {
    printf("unable to set port role %s for interface %d: interface not supported\n", dcbxPortRoleToStr(role), intIfNum);
  }
  else if (ret != OPEN_E_NONE)
  {
    printf("unable to set port role %s for interface %d: error %d\n", dcbxPortRoleToStr(role), intIfNum, ret);
  }
}

/*********************************************************************
* @purpose  Show the configuration port number.
*
* @param    client_handle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/

void showConfigurationPortNumber(openapiClientHandle_t *client_handle)
{
  uint32_t ifNum;
  OPEN_BOOL_t selected;
  open_error_t ret;

  ret = openapiDcbxCfgSrcGet(client_handle, &ifNum, &selected);
  if (ret == OPEN_E_UNAVAIL)
  {
    printf("unable to get configuration src: interface not supported\n");
  }
  else if (ret != OPEN_E_NONE)
  {
    printf("unable to get configuration src: error %d\n", ret);
  }
  else
  {
    printf("configuration source interface: %d selected: %s\n", ifNum, selected == OPEN_TRUE?"true":"false");
  }
}

/*********************************************************************
* @purpose  Display DCBX status for a given interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/

void showDcbx(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t appIndex)
{
  open_error_t ret;
  OPEN_DCBX_OP_STATUS status;
  OPEN_DCBX_PORT_ROLE_t role;
  OPEN_DCBX_CAPABILITIES_t caps;
  OPEN_DCBX_VERSION_t version;
  uint32_t txCount;
  uint32_t rxCount;
  uint32_t errorFrames;
  uint32_t unknownTlv;
  int i;
  OPEN_BOOL_t isCfgSrc;
  uint32_t timedOut;
  uint32_t multiPeer;
  uint32_t operVer;
  uint32_t seqNum;
  uint32_t rcvdAckNumber;
  OPEN_BOOL_t willing;
  uint16_t protocolId;
  uint32_t protoSel;
  uint8_t priority;
  OPEN_BOOL_t valid;
  OPEN_BOOL_t appStatus;
  open_buffdesc macAddr;
  open_buffdesc desc;
  open_buffdesc schedList;
  open_buffdesc bwList;
  unsigned char macAddrBuf[OPEN_MAC_ADDR_LEN];
  unsigned char descBuf[OPEN_LLDP_MGMT_STRING_SIZE_MAX];
  unsigned char schedListBuf[OPEN_ETS_TCG_NUM_MAX+1];
  unsigned char bwListBuf[OPEN_ETS_TCG_NUM_MAX+1];

  memset(macAddrBuf, 0, sizeof(macAddrBuf));
  macAddr.pstart = macAddrBuf;
  macAddr.size = sizeof(macAddrBuf);

  memset(descBuf, 0, sizeof(descBuf));
  desc.pstart = descBuf;
  desc.size = sizeof(descBuf);

  memset(schedListBuf, 0, sizeof(schedListBuf));
  schedList.pstart = schedListBuf;
  schedList.size = sizeof(schedListBuf);

  memset(bwListBuf, 0, sizeof(bwListBuf));
  bwList.pstart = bwListBuf;
  bwList.size = sizeof(bwListBuf);

  printf("Interface %d\n", intIfNum);
  ret = openapiDcbxIsValidIntf(client_handle, intIfNum, &valid);
  if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiDcbxIsValidIntf failed (%d)\n", __FUNCTION__, ret);
  }
  else
  {
    printf("interface is DCBX valid: %s\n", valid == OPEN_TRUE? "true":"false");
  }

  ret = openapiDcbxOperStatusGet(client_handle, intIfNum, &status);
  if (ret != OPEN_E_NONE) 
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to get operational status: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxOperStatusGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("operational status: %s\n", dcbxOpStatusToStr(status));
  }
  ret = openapiDcbxAutoCfgPortRoleGet(client_handle, intIfNum, &role);
  if (ret != OPEN_E_NONE) 
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to get auto config port role: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxAutoCfgPortRoleGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("auto config port role: %s\n", dcbxPortRoleToStr(role));
  }
  ret = openapiDcbxVersionGet(client_handle, intIfNum, &version);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to get version: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxVersionGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("version: %s\n", dcbxVersionToStr(role));
  }
  ret = openapiDcbxPeerVersionGet(client_handle, intIfNum, &version);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to get peer version: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxPeerVersionGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("peer version: %s\n", dcbxVersionToStr(version));
  }
  ret = openapiDcbxCounterGet(client_handle, intIfNum, &txCount, &rxCount, 
                              &errorFrames, &unknownTlv);

  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to get dcbx counters: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxCounterGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("counters: tx: %d rx: %d error frames: %d unknown TLVs: %d\n",
           txCount, rxCount, errorFrames, unknownTlv); 
  }
  ret = openapiDcbxPeerMacGet(client_handle, intIfNum, &macAddr);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to get peer MAC address: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxPeerMacGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("peer mac addr %02X:%02X:%02X:%02X:%02X:%02X\n", 
            macAddrBuf[0], macAddrBuf[1], macAddrBuf[2], macAddrBuf[3], 
            macAddrBuf[4], macAddrBuf[5]);
  }
  ret = openapiDcbxPeerDescGet(client_handle, intIfNum, &desc);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to get system description of peer device: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxPeerDescGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("peer system description: '%s'\n", descBuf);
  }
  ret = openapiDcbxIntfIsCfgSrcGet(client_handle, intIfNum, &isCfgSrc);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to determine if interface if cfg source: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxIntfIsCfgSrcGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("interface is cfg source: %s\n", isCfgSrc == OPEN_TRUE?"true":"false");
  }
  ret = openapiDcbxTimedOutAndMultiPeerCountGet(client_handle, intIfNum, &timedOut, &multiPeer);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to timed-out and multi-peer counts: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxTimedOutAndMultiPeerCountGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("timed-out count %d multi-peer count %d\n", timedOut, multiPeer);
  }
  ret = openapiDcbxPortOperationalModeGet(client_handle, intIfNum, &version);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to read port operational mode: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxPortOperationalModeGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("port operational mode %s\n",  dcbxVersionToStr(version));
  }
  ret = openapiDcbxPeerTlvMaskGet(client_handle, intIfNum, &caps);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to read peer TLV mask: interface not supported.\n");
    }
    else if (ret == OPEN_E_NOT_FOUND)
    {
      printf("Unable to read peer TLV mask: peer not found.\n");
    }
    else
    {
      printf("%s: openapiDcbxPeerTlvMaskGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("peer TLV mask:\n");
    dumpDcbxCaps(caps);
  }
  ret = openapiDcbxPortLegacyRemOperVerGet(client_handle, intIfNum, &operVer);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to read port legacy remote oper version: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxPortLegacyRemOperVerGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("port legacy remote oper version: %d\n", operVer);
  }
  ret = openapiDcbxPortLegacyRemMaxVerGet(client_handle, intIfNum, &operVer);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to read max port legacy remote oper version: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxPortLegacyRemMaxVerGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("max port legacy remote oper version: %d\n", operVer);
  }
  ret = openapiDcbxPortLegacyRemSeqNumGet(client_handle, intIfNum, &seqNum);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to read port legacy remote sequence number: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxPortLegacyRemSeqNumGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("port legacy remote sequence number: %d\n", seqNum);
  }
  ret = openapiDcbxPortLegacyRemRcvdAckNumGet(client_handle, intIfNum, &rcvdAckNumber);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to read port legacy remote received ack number: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxPortLegacyRemRcvdAckNumGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("port legacy remote received ack number: %d\n", rcvdAckNumber);
  }
  ret = openapiDcbxPortWillingGet(client_handle, intIfNum, &willing);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to read port willing: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxPortWillingGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("port willing: %s\n", willing == OPEN_TRUE ? "true":"false");
  }
  ret = openapiDcbxLocalEtsSchedGet(client_handle, intIfNum, &schedList);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to read local ETS scheduler type: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxLocalEtsSchedGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("local ETS scheduler type:\n");
    for (i = OPEN_ETS_TCG_NUM_MIN; i <= OPEN_ETS_TCG_NUM_MAX; i++)
    {
      printf("%d: %d\n", i, schedListBuf[i]);
    }
  }
  ret = openapiDcbxLocalEtsBwGet(client_handle, intIfNum, &bwList);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to read local ETS bandwidth: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxLocalEtsBwGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    for (i = OPEN_ETS_TCG_NUM_MIN; i <= OPEN_ETS_TCG_NUM_MAX; i++)
    {
      printf("%d: %d\n", i, bwListBuf[i]);
    }
  }
  ret = openapiDcbxAppPriTableGet(client_handle, intIfNum, appIndex,
                                  &protocolId, &protoSel, &priority, &appStatus);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to read application priority table: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxAppPriTableGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("application priority table: index: %d protocol ID: %d protocol sel: %d priority: %d app status: %s\n",
           appIndex, protocolId, protoSel, priority, appStatus == OPEN_TRUE?"true":"false");
  }
  ret = openapiDcbxPeerAppPriTableGet(client_handle, intIfNum, appIndex,
                                  &protocolId, &protoSel, &priority, &appStatus);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to read peer application priority table: interface not supported.\n");
    }
    else if (ret == OPEN_E_NOT_FOUND)
    {
      printf("Unable to read peer application priority table: peer not found.\n");
    }
    else
    {
      printf("%s: openapiDcbxPeerAppPriTableGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("peer application priority table: index: %d protocol ID: %d protocol sel: %d priority: %d app status: %s\n",
           appIndex, protocolId, protoSel, priority, 
           appStatus == OPEN_TRUE?"true":"false");
  }
  ret = openapiDcbxPortConfigTLVsEnabledGet(client_handle, intIfNum, &caps);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("Unable to read mask of enabled DCBX TLVs: interface not supported.\n");
    }
    else
    {
      printf("%s: openapiDcbxPortConfigTLVsEnabledGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("enabled port config TLVs:\n");
    dumpDcbxCaps(caps);
  }
}

void print_usage(const char *name)
{
  printf("%s [-a appIndex] [-r role] [-v version] [-c caps]  -i interface\n", name);
  printf("-a appIndex -- application index (openapiDcbxAppPriTableGet, default 0)\n");
  printf("-r role -- set port role. Use one of the following to specify the role:\n");
  printf(" 1: OPEN_DCBX_PORT_ROLE_MANUAL \n");
  printf(" 2: OPEN_DCBX_PORT_ROLE_AUTO_UPSTREAM \n");
  printf(" 3: OPEN_DCBX_PORT_ROLE_AUTO_DOWNSTREAM \n");
  printf(" 4: OPEN_DCBX_PORT_ROLE_AUTO_CFG_SRC \n");
  printf("-v version -- set version. Use one of the following to specify the version:\n");
  printf(" 1: OPEN_DCBX_AUTO \n");
  printf(" 2: OPEN_DCBX_IEEE \n");
  printf(" 3: OPEN_DCBX_CIN \n");
  printf(" 4: OPEN_DCBX_CEE \n");
  printf("-c caps -- set capabilities. Argument must be a base 10 integer that contains a bitmask of the following values:\n");
  printf(" 1: OPEN_DCBX_CAP_PFC_BITMASK\n");
  printf(" 2: OPEN_DCBX_CAP_ETS_CFG_BITMASK\n");
  printf(" 4: OPEN_DCBX_CAP_ETS_REC_BITMASK\n");
  printf(" 8: OPEN_DCBX_CAP_APP_PRI_BITMASK\n");
  printf(" 16: OPEN_DCBX_CAP_CONTROL_BITMASK\n");
  printf("-i interface -- interface id (required argument)\n");
  exit(0);
}

int main(int argc, char *argv[])
{
  openapiClientHandle_t clientHandle;
  int option = 0;
  uint32_t intIfNum;
  uint32_t appIndex = 0;
  bool sawRole = false;
  bool sawVersion = false;
  bool sawCaps = false;
  bool sawif = false;
  bool sawShowCfg = false;
  OPEN_DCBX_VERSION_t version;
  OPEN_DCBX_CAPABILITIES_t caps;
  OPEN_DCBX_PORT_ROLE_t role;

  int rc;

  while ((option = getopt(argc, argv,"a:r:v:i:c:s")) != -1) {
    switch (option) {
      case 'a':
        appIndex = atoi(optarg);
        break;
      case 'r':
        role = atoi(optarg);
        sawRole = true;
        break;
      case 'v':
        version = atoi(optarg);
        sawVersion = true;
        break;
      case 'c':
        caps = atoi(optarg);
        sawCaps = true;
        break;
      case 's':
        sawShowCfg = true;
        break;
      case 'i' : 
        intIfNum = atoi(optarg);
        sawif = true;
        break;
      default: 
        print_usage(argv[0]); 
        break;
    }
  }

  if (sawif == false)
  {
    print_usage(argv[0]); 
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((rc = openapiClientRegister("routing_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", rc);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  if (sawShowCfg)
  {
    showConfigurationPortNumber(&clientHandle);
  }

  if (sawRole)
  {
    setPortRole(&clientHandle, intIfNum, role);
  }

  if (sawVersion)
  {
    setDcbxVersion(&clientHandle, intIfNum, version);
  }

  if (sawCaps)
  {
    setDcbxTLVs(&clientHandle, intIfNum, caps);
  }

  showDcbx(&clientHandle, intIfNum, appIndex);

  return 0;
}


