#!/bin/python
# Copyright 2016 Broadcom.
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#
# Python 2.6.6
#*****************************************************************************
# @filename tunnel_example.py
#
# @purpose  Test tunnel code
# @detail  
# @component 
# @comments none
# @end
#*****************************************************************************

from struct import *
import OpEN_py as OpEN
from OpENUtil import *
from time import time
from select import select
import fcntl
import os
import errno
import socket
import struct

import time
import sys
import getopt
import ctypes
from ctypes import *
import array

class Olay:
  def __init__(self, client):
    self.clientHandle = client
    self.display_count = 1
    self.macAddr_len = 6

#****************************************************************************
# @purpose  Returns the byte equivalent of the mac address
#
# @params   pszMACAddress - mac address
# @params   pbyAddress - byte values of mac address 
#
# @returns  pbyAddress
#
#****************************************************************************
  def convertMacAddressStringIntoByte(self, pszMACAddress, pbyAddress):

    iNumber = ''
    pbyAddress = []
    pointer = pszMACAddress.split('-')
    
    for i in range(self.macAddr_len):
      iNumber = iNumber + str(pointer[i])

    pbyAddress = ''.join(chr(int(iNumber[i:i+2], 16)) for i in range(0, len(iNumber), 2))
      
    return pbyAddress

#****************************************************************************
# @purpose  Given an OpEN error code, return it as a string 
#
# @params   error - OpEN error
#
# @returns  error as string
#
#****************************************************************************
  def openapiErrorStrGet( self, error ):
     name = ""
     if error == OpEN.OPEN_E_NONE:
       name = "OPEN_E_NONE"
     elif error == OpEN.OPEN_E_RPC:
       name = "OPEN_E_RPC"
     elif error == OpEN.OPEN_E_INTERNAL: 
       name = "OPEN_E_INTERNAL"
     elif error == OpEN.OPEN_E_PARAM:    
       name = "OPEN_E_PARAM"
     elif error == OpEN.OPEN_E_FULL:    
       name = "OPEN_E_FULL"
     elif error == OpEN.OPEN_E_EXISTS:   
       name = "OPEN_E_EXISTS"
     elif error == OpEN.OPEN_E_TIMEOUT:  
       name = "OPEN_E_TIMEOUT"
     elif error == OpEN.OPEN_E_FAIL:     
       name = "OPEN_E_FAIL" 
     elif error == OpEN.OPEN_E_DISABLED: 
       name = "OPEN_E_DISABLED"
     elif error == OpEN.OPEN_E_UNAVAIL:  
       name = "OPEN_E_UNAVAIL"
     elif error == OpEN.OPEN_E_NOT_FOUND: 
       name = "OPEN_E_NOT_FOUND" 
     elif error == OpEN.OPEN_E_EMPTY:     
       name = "OPEN_E_EMPTY"
     elif error == OpEN.OPEN_E_ERROR:     
       name = "OPEN_E_ERROR"
     else:
       name = "Unknown or Undefined Error"
     return name;

#*************************************************************************
# create tenant
#
# @param[in] tenant tenant ID
# @param[in] tunnelType type of tunnel, e.g., L7_L2OL3_TUNNEL_TYPE_GRE
# @param[in] vlanId VLAN on switch associated with the tenant
# @param[in] flags unknown, not used by L7_L2OL3_TUNNEL_TYPE_GRE
#*************************************************************************
  def create_tenant(self, tenant, tunnelType, vlanId, srcIP, flags):

    configType = OpEN.OPEN_L2OL3_CONFIG_CREATE

    config = OpEN.open_l2ol3TenantConfig_t()
    config.tunnelType = tunnelType;
    config.vlanId = vlanId;
    config.localTepIpAddr = srcIP;

    retVal = OpEN.openapiL2oL3TenantCreate(self.clientHandle, configType, tenant, config)
    if retVal == OpEN.OPEN_E_NONE:
      print("\nTenant created\n")
    else:
      value = self.openapiErrorStrGet(retVal)
      print("\nTenant creation failed, rc = %s\n" % value)
  
    return

#*************************************************************************
# delete the specified tenant.
#
# @param[in] tenant ID of tenant to delete
#*************************************************************************
  def delete_tenant(self, tenant):
    retVal = OpEN.openapiL2oL3TenantDelete(self.clientHandle, tenant)

    if retVal == OpEN.OPEN_E_NONE:
      print("\nTenant deleted\n")
    else:
      value = self.openapiErrorStrGet(retVal)
      print("\nTenant deletion failed, rc = %s\n" %value)

    return

#**************************************************************************
# Display tenant information
#
# @param[in] tenant     tenant ID
#
#**************************************************************************
  def display_tenants(self, tenant):
    nextTenant_p = OpEN.new_uint32_tp()
    nextTenantConfig = OpEN.open_l2ol3TenantConfig_t()
    display_count = 0

    while True:
      retVal = OpEN.openapiL2oL3TenantNextGet(self.clientHandle, tenant,
                                              nextTenant_p, nextTenantConfig)
      if retVal == OpEN.OPEN_E_NONE:
        nextTenant = OpEN.uint32_tp_value(nextTenant_p)
        tenant = nextTenant
        display_count = display_count + 1
        if nextTenantConfig.tunnelType == OpEN.OPEN_L2OL3_TUNNEL_TYPE_VXLAN:
          str =  "VXLAN" 
        else:
          str = "NVGRE"
        print ("  %-8u    %-8u    %s         %-5u     %-15s  \n") % (display_count,
              nextTenant,
              str,
              nextTenantConfig.vlanId,
              socket.inet_ntop(socket.AF_INET, struct.pack("!I", nextTenantConfig.localTepIpAddr)))
      else:
        break

    OpEN.delete_uint32_tp(nextTenant_p)

    return
      
#***************************************************************************
# get tenant information
#
# @param[in] tenant     tenant ID
#
# @notes     if tenant = 0, gets all tenants info
# @notes     if tenant != 0, gets tenant tunnel type and vlan
#
#***************************************************************************
  def get_tenant(self, tenant):
    config = OpEN.open_l2ol3TenantConfig_t()
    count = 1    

    if tenant:
      retVal = OpEN.openapiL2oL3TenantGet(self.clientHandle, tenant, config)
      if retVal == OpEN.OPEN_E_NONE:
        print ("+----------------------------------------------------------------+\n");
        print ("  Sl          Tenant     Tunnel-Type    VLAN ID   Local TEP  \n");
        print ("+----------------------------------------------------------------+\n");

        if config.tunnelType == OpEN.OPEN_L2OL3_TUNNEL_TYPE_VXLAN:
          str = "VXLAN"
        else:
          str = "NVGRE" 
        print ("  %-8u    %-8u    %s         %-5u     %-15s  \n" % (count,
               tenant,
               str,
               config.vlanId, socket.inet_ntop(socket.AF_INET, struct.pack("!I", config.localTepIpAddr))))
        count = count + 1
        print ("+----------------------------------------------------------------+\n")
      else:
        print ("Tenant %d info get failed. rc=%s\n") % (tenant, 
              self.openapiErrorStrGet (retVal))
    else:
      print ("+----------------------------------------------------------------+\n");
      print ("  Sl          Tenant     Tunnel-Type    VLAN ID   Local TEP  \n");
      print ("+----------------------------------------------------------------+\n");

      self.display_tenants(0) 

      print ("+----------------------------------------------------------------+\n");

    return

#***************************************************************************
# Create tunnel
#
# @param[in] tenant tenant ID
# @param[in] remoteIP IPV4 remote IP address
# @param[in] flags configuration flags
# @param[in] tunnelHandle tunnel handle
#***************************************************************************
  def create_tunnel(self, tenant, remoteIP, flags, tunnelHandle):
    handle_p = OpEN.new_uint32_tp()

    configType = OpEN.OPEN_L2OL3_CONFIG_CREATE
  
    retVal = OpEN.openapiL2oL3TunnelCreate(self.clientHandle, 
                                           configType,
                                           tenant, 
                                           remoteIP, handle_p)
    
    handle = OpEN.uint32_tp_value(handle_p)
    if retVal == OpEN.OPEN_E_NONE:
      print("\nTunnel created, handle %d\n" %handle)
    else:
      value = self.openapiErrorStrGet(retVal)
      print("\nTunnel deletion failed, rc = %s\n" %value)
    
    OpEN.delete_uint32_tp(handle_p)
    return

#*************************************************************************
# Get status of specified tunnel.
#
# @param[in] tunnelHandle tunnel handle.
#*************************************************************************
  def get_tunnel_status(self, tunnelHandle):
    state_p = OpEN.new_uint32_tp()
    reason_p = OpEN.new_uint32_tp()
 
    retVal = OpEN.openapiL2oL3TunnelStatusGet(self.clientHandle, tunnelHandle, state_p, reason_p)
    if retVal == OpEN.OPEN_E_NONE:
      state = OpEN.uint32_tp_value(state_p)
      if state:
        stat = "Up"
      else: 
        stat = "Down"
      print("\nTunnel Status:")
      print("\nState  - %s\n" %(state))
      
      reason = OpEN.uint32_tp_value(reason_p)
      if reason == OpEN.OPEN_L2OL3_TUNNEL_REASON_INITIATOR_NOT_RESOLVED:
        print("  Reason - INITIATOR_NOT_RESOLVED\n")
      elif reason ==  OpEN.OPEN_L2OL3_TUNNEL_REASON_INITIATOR_HW_INSTALL_FAILED:
        print("  Reason - INITIATOR_HW_INSTALL_FAILED\n")
      elif reason == OpEN.OPEN_L2OL3_TUNNEL_REASON_TERMINATOR_HW_INSTALL_FAILED:
        print("  Reason - TERMINATOR_HW_INSTALL_FAILED\n")
      else:
        print("  Reason - None\n")

      value = self.openapiErrorStrGet(retVal)
      print("\nTunnel status get success, rc = %s\n" %value)
    else:
      value = self.openapiErrorStrGet(retVal)
      print("\nTunnel status get failed, rc = %s\n" %value)

    OpEN.delete_uint32_tp(state_p)
    OpEN.delete_uint32_tp(reason_p)
    return

#*************************************************************************
# Get counters for the specified tunnel
#
# @param[in] tunnelHandle tunnel
#
#*************************************************************************
  def tunnel_stats_get(self, tunnelHandle):
    tunnelStats_p = OpEN.new_open_l2ol3TunnelStats_tp()

    retVal = OpEN.openapiL2oL3TunnelCounterGet(self.clientHandle, tunnelHandle, tunnelStats_p)
    if retVal == OpEN.OPEN_E_NONE:
      tunnelStats = OpEN.l2ol3TunnelStats_tp_value(tunnelStats_p)
      print("Tunnel counters for handle = %d:" %tunnelHandle)
      print("rxPkts  = %u" %tunnelStats.rxPkts)
      print("rxBytes = %u" %tunnelStats.rxBytes)
      print("txPkts  = %u" %tunnelStats.txPkts)
      print("txBytes = %u" %tunnelStats.txBytes)

      value = self.openapiErrorStrGet(retVal)
      print("\nTunnel counters get success, rc = %s\n" %value)
  
    else:
      value = self.openapiErrorStrGet(retVal)
      print("\nTunnel counters get failed, rc = %s\n" %value)

    OpEN.delete_open_l2ol3TunnelStats_tp(tunnelStats_p)
    return

#*************************************************************************
# Clear the tunnel counters.
#
# @param[in] tunnelHandle tunnel
#
#*************************************************************************
  def tunnel_stats_clear(self, tunnelHandle):
    retVal = OpEN.openapiL2oL3TunnelCounterClear(self.clientHandle, tunnelHandle)

    if retVal == OpEN.OPEN_E_NONE:
      value = self.openapiErrorStrGet(retVal)
      print("\nTunnel counters clear success. rc = %s\n" %value)
    else:
      value = self.openapiErrorStrGet(retVal)
      print("\nTunnel counters clear failed, rc = %s\n" %value)

    return

#*************************************************************************
# delete the specified tunnel.
#
# @param[in] tunnelHandle tunnel to delete.
#*************************************************************************
  def delete_tunnel(self, tunnelHandle):
    retVal = OpEN.openapiL2oL3TunnelDelete(self.clientHandle, tunnelHandle)
    if retVal == OpEN.OPEN_E_NONE:
      print("\nTunnel deleted\n")
    else:
      value = self.openapiErrorStrGet(retVal)
      print("\nTunnel deletion failed, rc %s" %value)

    return

#*************************************************************************
# purpose  add_l2Entry adding mac address
#
# @param[in] tenant tenant ID
# @param[in] macaddr is the byte representative of the mac address
# @param[in] ifNum is the tunnel handle
#
# @returns  none
#
# @notes
#
# @end
#*************************************************************************
  def add_l2Entry(self, tenant, macAddr, ifNum):
     macAddrBuf, macAddrValue = self.macAddrToOpenBuf(macAddr)  

     retVal = OpEN.openapiL2oL3TenantL2FwdEntryAdd(self.clientHandle, tenant, macAddrBuf, ifNum)
     
     if retVal != OpEN.OPEN_E_NONE:
       value = self.openapiErrorStrGet(retVal)
       print("\nL2 entry add failed, rc = %s\n" %value)
     else:
       value = self.openapiErrorStrGet(retVal)
       print("\nL2 entry add success, rc = %s\n" %value)

     return

#*********************************************************************
#
# @purpose  del L2 forwarding entry of a tenant
#
# @params   tenant   Tenant Identifier
# @param[in] macaddr is the byte representation of the mac address
#
# @returns  none
#
# @notes
#
# @end
#*********************************************************************
  def del_l2Entry(self, tenant, macAddr):
    macAddrBuf, macAddrValue = self.macAddrToOpenBuf(macAddr)  

    retVal = OpEN.openapiL2oL3TenantL2FwdEntryDelete(self.clientHandle, tenant, macAddrBuf)

    if retVal != OpEN.OPEN_E_NONE:
      value = self.openapiErrorStrGet(retVal)
      print("\nL2 entry delete failed, rc = %s\n" %value )
    else:
      value = self.openapiErrorStrGet(retVal)
      print("\nL2 entry delete success, rc = %s\n" % value)

    return

#*********************************************************************
#
# @purpose  Get L2 forwarding entry of tenant
#
# @params   tenant   Tenant Identifier
# @params   macAddr  Mac address to lookup
#
# @returns  none
#
# @notes
#
# @end
#*********************************************************************/
  def get_l2Entry(self, tenant, macAddr):
    print("get_l2Entry not implemented")
    return

    open = OpENUtil()
    
    macAddrBuf, macAddrValue = self.macAddrToOpenBuf(macAddr)  
    macEntry = OpEN.open_l2ol3L2FwdEntry_t()
 
    count = 0

    retVal = OpEN.openapiL2oL3TenantL2FwdEntryGet(self.clientHandle, tenant,
                                                  macAddrBuf,
                                                  macEntry)
    if retVal == OpEN.OPEN_E_NONE:
      print("Found MAC entry!\n\n")

      print("+--------------------------------------------------------+\n")
      print("| Sl    Tenant        MAC Address    destHandle   Static |\n")
      print("+--------------------------------------------------------+\n")

      a = ord(mac[0])
      b = ord(mac[1]) 
      c = ord(mac[2]) 
      d = ord(mac[3]) 
      e = ord(mac[4]) 
      f = ord(mac[5]) 
      print("| %-5u %-10d %02x:%02x:%02x:%02x:%02x:%02x    %-5d        %d   |\n" % (count, macEntry.tenantId, a, b, c, d, e, f, macEntry.destHandle, macEntry.isStatic))
      count = count + 1
      print("+--------------------------------------------------------+\n")
    else:
      value = self.openapiErrorStrGet(retVal)
      print("Failed to find MAC entry. RC=%s\n" % (value))

    return

#*********************************************************************
#
# @purpose  Get L2 forwarding entries of tenant
#
# @params   tenant   Tenant Identifier
#
# @returns  none
#
# @notes
#
# @end
#*********************************************************************/
  def get_l2Entries(self, tenant):
    print("get_l2Entries not implemented")
    return

    open = OpENUtil()
    
    macAddrBuf, macAddrValue = self.macAddrToOpenBuf("")  
    nextMacEntry = OpEN.open_l2ol3L2FwdEntry_t()

    count = 0 
    seed = tenant

    if tenant:
      print ("Forwarding entries for tenant=%d:\n\n") % (tenant)
    else:
      print ("Forwarding entries in database:\n\n")

    print ("+--------------------------------------------------------+\n");
    print ("| Sl    Tenant        MAC Address    destHandle   Static |\n");
    print ("+--------------------------------------------------------+\n");

    while True:
      if count >= 50:
        break
      retVal = OpEN.openapiL2oL3TenantL2FwdEntryNextGet(self.clientHandle, tenant,
                                                        macAddrBuf, nextMacEntry )
      if retVal == OpEN.OPEN_E_NONE:
        if (seed == 0) or (seed and (seed == nextMacEntry.tenantId)):
          macAddress = nextMacEntry.macAddress
          a = ord(macAddress[0])
          b = ord(macAddress[1]) 
          c = ord(macAddress[2]) 
          d = ord(macAddress[3]) 
          e = ord(macAddress[4]) 
          f = ord(macAddress[5]) 
          count = count + 1
          print ("| %-5u %-10d %02x:%02x:%02x:%02x:%02x:%02x    %-5d        %d   |\n") % (count, nextMacEntry.tenantId,
          a, b, c, d, e, f,
          nextMacEntry.destHandle, nextMacEntry.isStatic)
        tenant = nextMacEntry.tenantId
        macAddrBuf, macAddrValue = self.macAddrToOpenBuf(nextMacEntry.macAddress)
      else:
        break

    print ("+--------------------------------------------------------+\n")

    return

  '''
  /**
  * To get the switch overlay tunnel configuration mode
  * of a specified tunnel type.
  *
  * @param[in[ clientHandle - Client handle
  * @param[in] tunnelType - type of tunnel.
  *                         e.g., OPEN_L2OL3_TUNNEL_TYPE_NVGRE
  *
  */
  '''
  def tunnel_services_config_get(self, tunnelType):
    mode_p = OpEN.new_OPEN_CONTROL_tp()

    retVal = OpEN.openapiL2oL3SwitchTunnelModeGet(self.clientHandle, tunnelType, mode_p)
    if retVal != OpEN.OPEN_E_NONE:
      print ("Overlay Configuration Mode Get Failed for Tunnel Type=%d rc=%s\n" % (tunnelType, self.openapiErrorStrGet(retVal)))
    else:
      mode = OpEN.OPEN_CONTROL_tp_value(mode_p)
      if mode == OpEN.OPEN_ENABLE:
        str = "Enabled"
      else:
        str = "Disabled"
      OpEN.delete_OPEN_CONTROL_tp(mode_p)
      print ("Overlay Configuration Mode is %s for Tunnel Type=%d. rc=%s\n" % (str, tunnelType, self.openapiErrorStrGet(retVal)))

    if tunnelType != OpEN.OPEN_L2OL3_TUNNEL_TYPE_VXLAN:
      return

    udpPort_p = OpEN.new_uint32_tp()
    retVal = OpEN.openapiL2oL3SwitchVxlanUdpDestPortGet(self.clientHandle, udpPort_p)
    if retVal == OpEN.OPEN_E_NONE:
      udpPort = OpEN.uint32_tp_value(udpPort_p)
      print ("VXLAN UDP port is configured to %d.\n" % (udpPort))
    else:
      print ("VXLAN UDP port configured get has failed. rc=%s.\n" % (self.openapiErrorStrGet(retVal)))
    OpEN.delete_uint32_tp(udpPort_p)

    return

  '''
  /**
  * To set the switch overlay tunnel configuration mode
  * of a specified tunnel type.
  *
  * @param[in[ clientHandle - Client handle
  * @param[in] tunnelType - type of tunnel.
  *                         e.g., OPEN_L2OL3_TUNNEL_TYPE_NVGRE
  * @param[in[ mode - Enable or Disable
  *
  */
  '''
  def tunnel_services_config_set(self, tunnelType, mode):
    retVal = OpEN.openapiL2oL3SwitchTunnelModeSet(self.clientHandle, tunnelType, mode)
    if retVal != OpEN.OPEN_E_NONE:
      print ("Overlay Configuration Mode Set Failed for Tunnel Type=%d rc=%s\n" % (tunnelType, self.openapiErrorStrGet(retVal)))
    else:
      if mode == OpEN.OPEN_ENABLE:
        str = "Enable"
      else:
        str = "Disable" 
      print ("Overlay Configuration Mode is set to %s for Tunnel Type=%d. rc=%s\n" % (str, tunnelType, self.openapiErrorStrGet(retVal)))
    return

  def tunnel_services_config_vxlan_udp_port(self, udpPort):
    retVal = OpEN.openapiL2oL3SwitchVxlanUdpDestPortSet(self.clientHandle, udpPort)
    if retVal != OpEN.OPEN_E_NONE:
      print ("VXLAN UDP destination port=%d configuration on Swtich failed. rc=%s\n" % (udpPort, self.openapiErrorStrGet(retVal)))
    else:
      print ("VXLAN UDP destination port=%d configuration on Switch is successful. rc=%s\n" % (udpPort, self.openapiErrorStrGet(retVal)))

    return

  def get_access_port(self, accessId):
    accessInfo = OpEN.open_l2ol3AccessPortInfo_t()

    if accessId:
      rc = OpEN.openapiL2oL3AccessPortGet(self.clientHandle, accessId, accessInfo)
      if rc == OpEN.OPEN_E_NONE:
        print ("------------------------------------------------------------------------\n")
        print (" Tenant     Tunnel-Type   Access-Handle  VLAN  intIfNum  Match-Type  \n")
        print ("------------------------------------------------------------------------\n")
        if accessInfo.tunnelType == OpEN.OPEN_L2OL3_TUNNEL_TYPE_VXLAN:
          str = "VXLAN"
        else:
          str = "NVGRE"
        print (" %-10d    %s          %-4d       %-4d    %d         %d\n" % (accessInfo.tenantId, str, accessInfo.accessHandle, accessInfo.vlanId, accessInfo.intIfNum, accessInfo.matchType))
        print ("------------------------------------------------------------------------\n")
      else:
        print ("Access port identifier = %d information not found!\n" % (accessId))
    else:
      print ("Invalid access port identifier = %d\n" % (accessId))


  def get_access_ports(self, tenant):
    count = 0;
    accessHandle = 0;
    accessInfoNext = OpEN.open_l2ol3AccessPortInfo_t()

    print ("------------------------------------------------------------------------\n")
    print (" Tenant     Tunnel-Type   Access-Handle  VLAN  intIfNum  Match-Type  \n")
    print ("------------------------------------------------------------------------\n")

    if tenant:
      while True:
        rc = OpEN.openapiL2oL3AccessPortNextGet(self.clientHandle, accessHandle, accessInfoNext)
        if rc == OpEN.OPEN_E_NONE:
          if tenant == accessInfoNext.tenantId:
            if accessInfoNext.tunnelType == OpEN.OPEN_L2OL3_TUNNEL_TYPE_VXLAN:
              str = "VXLAN"
            else:
              str = "NVGRE"
            print (" %-10d    %s          %-4d       %-4d    %d         %d\n" % (accessInfoNext.tenantId, str, accessInfoNext.accessHandle, accessInfoNext.vlanId, accessInfoNext.intIfNum, accessInfoNext.matchType))
            count = count + 1
          accessHandle = accessInfoNext.accessHandle
        else:
          break

      print ("------------------------------------------------------------------------\n")
      print ("\n\nTenant %d has %d Access Ports\n" % (tenant, count))
    else:
      while True:
        rc = OpEN.openapiL2oL3AccessPortNextGet(self.clientHandle, accessHandle, accessInfoNext)
        if OpEN.OPEN_E_NONE == rc:
          if accessInfoNext.tunnelType == OpEN.OPEN_L2OL3_TUNNEL_TYPE_VXLAN:
            str = "VXLAN"
          else:
            str = "NVGRE"
          print (" %-10d    %s          %-4d       %-4d    %d         %d\n" % (accessInfoNext.tenantId, str, accessInfoNext.accessHandle, accessInfoNext.vlanId, accessInfoNext.intIfNum, accessInfoNext.matchType))

          accessHandle = accessInfoNext.accessHandle
          count = count + 1
        else:
          break
      print ("------------------------------------------------------------------------\n")
      print ("\n\nTotal Access Ports of All Tenants = %d\n" % (count))
    return

  '''
  /**
  * Get all tenant tunnels.
  *
  */
  '''
  def get_all_tunnels(self):
    tunnelHandle = 0
    count = 0;
    tunnelInfoNext = OpEN.open_l2ol3TunnelInfo_t()

    print ("------------------------------------------------------------------------\n")
    print (" Tenant     Tunnel-Type Tunnel-Handle    Remote-TEP      Status Reason  \n")
    print ("------------------------------------------------------------------------\n")

    while True:
      rc = OpEN.openapiL2oL3TunnelNextGet(self.clientHandle, tunnelHandle, tunnelInfoNext)
      if OpEN.OPEN_E_NONE == rc:
        if tunnelInfoNext.tunnelType == OpEN.OPEN_L2OL3_TUNNEL_TYPE_VXLAN:
          str = "VXLAN"
        else:
          str = "NVGRE"
        print (" %-9d    %s           %-4d       %-15s     %d    %d\n" % (tunnelInfoNext.tenantId, str, tunnelInfoNext.tunnelHandle, socket.inet_ntop(socket.AF_INET, struct.pack("!I", tunnelInfoNext.config.remoteIPAddr)), tunnelInfoNext.status.status, tunnelInfoNext.status.reasonFlags))

        tunnelHandle = tunnelInfoNext.tunnelHandle
        count = count + 1
      else:
        break

    print ("------------------------------------------------------------------------\n")
    print ("\n\nTotal Tenant Tunnels = %d\n" % (count))


  def resource_stats_get(self):
    resourceLimits = OpEN.open_l2ol3ResourceLimits_t()
    resourceUsage = OpEN.open_l2ol3ResourceStats_t()

    retVal = OpEN.openapiL2oL3ResourceLimitsGet(self.clientHandle, resourceLimits)
    if retVal != OpEN.OPEN_E_NONE:
      print ("\n\nERROR: Failed to get resource limits. RC=%s\n\n" % (self.openapiErrorStrGet(retVal)))
      return

    print ("\n\n<<<<< RESOURCE ALLOWED LIMITS >>>>>\n\n")
    print ("Maximum Number of Tenants Allowed          : %d\n" % (resourceLimits.maxNumTenants))
    print ("Maximum Number of Access Ports Allowed     : %d\n" % (resourceLimits.maxNumAccessPorts))
    print ("Maximum Number of Tenant Tunnels Allowed   : %d\n" % (resourceLimits.maxNumTunnelPorts))
    print ("Maximum Number of Shared Tunnels Allowed   : %d\n" % (resourceLimits.maxNumTunnelsInHw))
    print ("Maximum Number of User configured host         \n"
          "entries allowed per Physical/Lag Interface : %d\n" % (resourceLimits.maxNumLocalStaticFwdEntriesPerInterface))
    print ("Maximum Number of User configured host         \n"
          "entries allowed per Tenant                 : %d\n" % (resourceLimits.maxNumRemoteStaticFwdEntriesPerTenant))
    print ("Maximum Number of User configured host         \n"
          "entries allowed on Switch                  : %d\n" % (resourceLimits.maxNumRemoteStaticFwdEntriesPerSwitch))
    print ("L2 forwarding database size                : %d\n" % (resourceLimits.maxNumFwdEntries))


    retVal = OpEN.openapiL2oL3ResourceStatsGet(self.clientHandle, resourceUsage)
    if retVal != OpEN.OPEN_E_NONE:
      print ("\n\nERROR: Failed to get resource limits. RC=%s\n\n" % (openapiErrorStrGet(retVal)))
      return

    print ("\n\n<<<<< RESOURCE USAGE STATISTICS >>>>>\n\n")
    print ("Number of Tenants Configured         : %d\n" % (resourceUsage.numTenants))
    print ("Number of Tenant VLANs Configured    : %d\n" % (resourceUsage.numTenantVlans))
    print ("Number of Source TEPs Configured     : %d\n" % (resourceUsage.numTenantLocalTeps))
    print ("Number of Access Ports Configured    : %d\n" % (resourceUsage.numAccessPorts))
    print ("Number of Tenant Tunnels Configured  : %d\n" % (resourceUsage.numTenantTunnels))
    print ("Number of Forwarding Entries Learnt  : %d\n" % (resourceUsage.numLearnedFwdEntries))
    print ("Number of Entries Configured by User : %d\n" % (resourceUsage.numStaticFwdEntries))
    print ("Total Forwarding Entries in Database : %d\n" % (resourceUsage.numTotalFwdEntries))


    print ("\n\n<<<<< SOFTWARE ERROR STATISTICS >>>>>\n\n")
    print ("globalTunnelModeFailures       : %d\n" % (resourceUsage.errStats.globalTunnelModeFailures))
    print ("numTenantCreateFailures        : %d\n" % (resourceUsage.errStats.numTenantCreateFailures))
    print ("numTenantDeleteFailures        : %d\n" % (resourceUsage.errStats.numTenantDeleteFailures))
    print ("numTenantVlanAddFailures       : %d\n" % (resourceUsage.errStats.numTenantVlanAddFailures))
    print ("numTenantVlanDeleteFailures    : %d\n" % (resourceUsage.errStats.numTenantVlanDeleteFailures))
    print ("numTenantSrcTepAddFailures     : %d\n" % (resourceUsage.errStats.numTenantSrcTepAddFailures))
    print ("numTenantSrcTepDeleteFailures  : %d\n" % (resourceUsage.errStats.numTenantSrcTepDeleteFailures))
    print ("numAccessCreateFailures        : %d\n" % (resourceUsage.errStats.numAccessCreateFailures))
    print ("numAccessDeleteFailures        : %d\n" % (resourceUsage.errStats.numAccessDeleteFailures))
    print ("numTenantTunnelCreateFailures  : %d\n" % (resourceUsage.errStats.numTenantTunnelCreateFailures))
    print ("numTenantTunnelModifyFailures  : %d\n" % (resourceUsage.errStats.numTenantTunnelModifyFailures))
    print ("numTenantTunnelDeleteFailures  : %d\n" % (resourceUsage.errStats.numTenantTunnelDeleteFailures))
    print ("numVxlanUdpDstCfgFailures      : %d\n" % (resourceUsage.errStats.numVxlanUdpDstCfgFailures))
    print ("numStaticFwdAddFailures        : %d\n" % (resourceUsage.errStats.numStaticFwdAddFailures))
    print ("numStaticFwdDeleteFailures     : %d\n" % (resourceUsage.errStats.numStaticFwdDeleteFailures))
    print ("numLearnedFwdAddFailures       : %d\n" % (resourceUsage.errStats.numLearnedFwdAddFailures))
    print ("numLearnedFwdDeleteFailures    : %d\n" % (resourceUsage.errStats.numLearnedFwdDeleteFailures))

    return

  '''
  /**
  * Get status of specified tunnel.
  *
  * @param[in] tunnelHandle tunnel handle.
  */
  '''

  def get_tunnel_status(self, tunnelHandle):
    tunnelInfo = OpEN.open_l2ol3TunnelInfo_t()
    retVal = OpEN.openapiL2oL3TunnelGet(self.clientHandle, tunnelHandle, tunnelInfo)
    if retVal == OpEN.OPEN_E_NONE:
      print("Tunnel Status:\n")
      if tunnelInfo.status.status:
        str = "UP"
      else:
        str = "DOWN"
      print("  State  - %s\n" % (str))

      if tunnelInfo.status.reasonFlags == OpEN.OPEN_L2OL3_TUNNEL_REASON_INITIATOR_NOT_RESOLVED:
        print("  Reason - INITIATOR_NOT_RESOLVED\n")
      elif tunnelInfo.status.reasonFlags == OpEN.OPEN_L2OL3_TUNNEL_REASON_INITIATOR_HW_INSTALL_FAILED:
        print("  Reason - INITIATOR_HW_INSTALL_FAILED\n")
      elif tunnelInfo.status.reasonFlags == OpEN.OPEN_L2OL3_TUNNEL_REASON_TERMINATOR_HW_INSTALL_FAILED:
        print("  Reason - TERMINATOR_HW_INSTALL_FAILED\n")
      else:
        print("  Reason - None\n")

      print("\nTunnel status get success, rc=%s\n" % (self.openapiErrorStrGet(retVal)))
    else:
      print("Tunnel status get failed, rc=%s\n" % (self.openapiErrorStrGet(retVal)))
    return

  '''
  /**
  * Get counters for the specified tunnel
  *
  * @param[in] tunnelHandle tunnel
  */
  '''

  def tunnel_stats_get(self, tunnelHandle):
    tunnelStats = OpEN.open_l2ol3TunnelStats_t()

    retVal = OpEN.openapiL2oL3TunnelCounterGet(self.clientHandle, tunnelHandle, tunnelStats)

    if retVal == OpEN.OPEN_E_NONE:
      print("Tunnel counters for handle=%d:\n" % (tunnelHandle))
      print("rxPackts = %u\n" % (tunnelStats.rxPkts))
      print("rxBytes  = %u\n" % (tunnelStats.rxBytes))
      print("txPackts = %u\n" % (tunnelStats.txPkts))
      print("txBytes  = %u\n" % (tunnelStats.txBytes))

      print("\nTunnel counters get success, rc=%s\n" % (self.openapiErrorStrGet(retVal)))
    else:
      print("Tunnel counters get failed, rc=%s\n" % (self.openapiErrorStrGet(retVal)))

    return

  '''
  /**
  * clear the tunnel counters.
  *
  * @param[in] tunnelHandle tunnel
  */
  '''

  def tunnel_stats_clear(self, tunnelHandle):
    retVal = OpEN.openapiL2oL3TunnelCounterClear(self.clientHandle, tunnelHandle)

    if retVal == OpEN.OPEN_E_NONE:
      print ("Tunnel counters clear success. rc=%s\n" % (self.openapiErrorStrGet(retVal)))
    else:
      print ("Tunnel counters clear failed, rc=%s\n" % (self.openapiErrorStrGet(retVal)))

    return

#*********************************************************************
# @purpose  Convert a mac address into an openbuff_desc
#
# @returns  pointer to buffer
#
# @end
#*********************************************************************
  def macAddrToOpenBuf(self,mac):
    open = OpENUtil()
    try:
      macAddr = open.getByteBuffer(self.macAddr_len, mac)
    except OpENBufferSizeError:
      print("macAddrToOpenBuf: getByteBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("macAddrToOpenBuf: getByteBuffer raised TypeError")
      return
    macAddrBuf = OpEN.open_buffdesc()
    macAddrBuf.pstart = macAddr
    macAddrBuf.size = self.macAddr_len
    return (macAddrBuf, macAddr)

#*********************************************************************
# @purpose  
#
# @returns  0: Success
# @returns  1: Failure
#
# @notes
#
# @end
#*********************************************************************
  def printUsage(self, app):
    '''
    print("Usage: %s -o <option> -i <intf> -t <tenant> -r <remoteIP>"
         "-a -d -g -v <vlanId> -y <tunnel Type> -s <srcIP>  -u <tunnel>  -c <config flags>  -h ", app)
    print("")
    print("\t o \t Options should be one of the following based on action desired, ")
    print("\t\t init   - to initialize overlay API service.\n ")
    print("\t\t finish - to reset overlay API service.\n ")
    print("\t\t tenant - to create/delete/get a tenant.\n"
          "\t\t          options needed to create: -t -y [-v] [-c]\n"
          "\t\t          options needed to delete: -t -d\n"
          "\t\t          options needed to get: [-t] [-y] -g")
    print("\t\t tunnel - to create/delete/get a tunnel.\n"
          "\t\t          options needed to create: -t -y -r -s [-c]\n"
          "\t\t          options needed to delete: -u -d\n"
          "\t\t          options needed to get: -u -g\n")
    print("\t\t l2entry - to create/delete/get bridging entries.")
    print("\t\t           options needed to create: -t -m <[-i] | [-u]>")
    print("\t\t           options needed to delete: -t -m -d ")
    print("\t\t           options needed to get: -t -g \n")
    print("\t\t stats   - to get or clear tunnel counters.\n"
          "\t\t           options needed to get: -u -g\n"
          "\t\t           options needed to clear: -u -d\n")
    print("\t\t flags -a to add/create and -d to delete. Default action is -a")
    print("\t y \t Tunnel type of the tunnel ")
    print("\t v \t Vlan Id ")
    print("\t r \t Remote IP for the tunnel (a.b.c.d) ")
    print("\t s \t SRC IP for the tunnel (a.b.c.d)")
    print("\t u \t Tunnel handle ")
    print("\t m \t Mac Address (xx-xx-xx-xx-xx-xx)")
    print("\t i \t Interface number ")
    print("\t c \t Configuration flags ")
    print("\t h \t Usage help")
    '''
    print("Usage: %s -o <option> -t <tenant> -y <tunnel Type> -v <vlanId>"
          " -s <srcIP> -r <remoteIP> -u <tunnel Id> -i <access Id>"
          " -c <config flags> -a -d -g -h \n") % (app)
    print("\n");
    print("\t o \t Options should be one of the following based on action desired, \n");
    print("\t\t config - to initialize overlay API service. \n"
          "\t\t          options needed to enable:   -y [-p] [-a] \n"
          "\t\t          options needed to disable:  -y -d \n"
          "\t\t          options needed to get:      -y -g \n");
    print("\t\t tenant - to create/modify/delete/get a tenant.\n"
          "\t\t          options needed to create/modify:   -t -y <[-v] | [-s]> -c\n"
          "\t\t          options needed to delete:   -t -d\n"
          "\t\t          options needed to get:     [-t ] -g\n");
    print("\t\t access - to get list of configured access ports.\n"
          "\t\t          options needed to get:      -g [-t | -i]\n");
    print("\t\t tunnel - to create/delete/get a tunnel.\n"
          "\t\t          options needed to create:   -t -r -c \n"
          "\t\t          options needed to delete:   -d -u\n"
          "\t\t          options needed to get:      -g [-u]\n");
    print("\t\t l2entry - to create/delete/get bridging entries.\n");
    print("\t\t           options needed to create:  -t -m <[-i] | [-u]>\n");
    print("\t\t           options needed to delete:  -t -m -d \n");
    print("\t\t           options needed to get:     [-t] [-m] -g\n");
    print("\t\t stats   - to get stats or clear tunnel counters.\n"
          "\t\t           to get tunnel stats :      -u -g\n"
          "\t\t           to clear tunnel stats :    -u -d\n"
          "\t\t     Note: Resource & error stats:    -g \n");
    print("\t\t flags -a to add/create and -d to delete. Default action is -a\n");
    print("\t t \t Tenant Identifier (VNID for VXLAN / VSID for NVGRE)\n");
    print("\t y \t Tenant or Tunnel type (VXLAN - 2, NVGRE - 4)\n");
    print("\t v \t VLAN Identifier or Tenant VLAN\n");
    print("\t r \t IP address of Remote TEP (Tunnel end point) in a.b.c.d format \n");
    print("\t s \t IP address of Local/Source TEP or Local Gateway in a.b.c.d format\n");
    print("\t p \t VXLAN destination UDP port\n");
    print("\t u \t Tunnel handle \n");
    print("\t m \t MAC Address (xx:xx:xx:xx:xx:xx)\n");
    print("\t i \t Access Interface Handle.\n");
    print("\t c \t Configuration flags. 1-Create, 2-Modify \n");
    print("\t h \t Usage help\n");

'''
/*********************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates Overlay OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure
*
*********************************************************************/
'''
def main(argc, argv):
  ifNum = 0 
  srcIP = 0 
  opterr = 0
  vlanId = 0 
  addFlag = 1
  getFlag = 0
  remoteIP = 0 
  macFlag = 1
  dstUdp = 0 
  udpFlag = 0
  configFlags = 0 
  tunnelHandle = 0
  macAddr = range(6)
  nullMac = range(6)
  option_name = []
  open = OpENUtil()
  tunnel_type = 0 
  tenant = 0  
  ret = open.connect("tunnel_app")
  l2DestIntf = 0
  
  clientHandle = open.get_client()
  if ret == OpEN.OPEN_E_NONE:
    if argc < 4:
       option_name.append('There are no options specified')

    opts, args = getopt.getopt(argv[1:], "hadgy:c:v:t:m:o:i:r:s:u:p") 
    for opt, optarg in opts:
      if opt == '-a':
        addFlag = 1
      elif opt == '-g':
        getFlag = 1
      elif opt == '-t':
        tenant = int(optarg)
      elif opt == '-u':
        tunnelHandle = int(optarg)
      elif opt == '-i':
        ifNum = int(optarg)
      elif opt == '-v':
        vlanId = int(optarg)
      elif opt == '-c':
        configFlags = int(optarg)
      elif opt == '-y':
        tunnel_type = int(optarg)
      elif opt == '-r':
        remoteIP = unpack('I', socket.inet_pton(socket.AF_INET, optarg) )[0]
        remoteIP = socket.ntohl(remoteIP)
      elif opt == '-s':
        srcIP = unpack('I', socket.inet_pton(socket.AF_INET, optarg))[0]
        srcIP = socket.ntohl(srcIP)
      elif opt == '-m':
         macAddr = Olay(clientHandle).convertMacAddressStringIntoByte(optarg, macAddr)
         macFlag = 1
      elif opt == '-d':
        addFlag = 0
      elif opt == '-p':
        dstUdp = int(optarg)
        udpFlag = 1
      elif opt == '-o':
        option_name = optarg
      elif opt == '-h':
        Olay(clientHandle).printUsage(argv[0])
        return 0
      elif opt == '?':
        if isprint(opt):
          print(stderr, "Unknown option `-%c'.\n" %opt)
        else:
          print(stderr, "Unknown option character `\\x%x'.\n" %optopt)
        return 1
      else:
        sys.exit()

    nullMacStr = "00-00-00-00-00-00"
    nullMac = Olay(clientHandle).convertMacAddressStringIntoByte(nullMacStr, nullMac)
    if option_name == "config":
      if getFlag and tunnel_type:
        Olay(clientHandle).tunnel_services_config_get(tunnel_type)
      elif addFlag and tunnel_type:
        Olay(clientHandle).tunnel_services_config_set(tunnel_type, OpEN.OPEN_ENABLE)

        if udpFlag and tunnel_type == 2:
          Olay(clientHandle).tunnel_services_config_vxlan_udp_port(dstUdp)
      elif tunnel_type:
        Olay(clientHandle).tunnel_services_config_set(tunnel_type, OpEN.OPEN_DISABLE)
      else:
        Olay(clientHandle).printUsage(argv[0])
        return 1
    elif option_name == "l2entry":
      if getFlag and macFlag: 
        Olay(clientHandle).get_l2Entry(tenant, macAddr)
      elif getFlag:
        Olay(clientHandle).get_l2Entries(tenant)
      elif addFlag and macFlag and (tunnelHandle or ifNum):
        if tenant == 0 or (macAddr == nullMac): 
          Olay(clientHandle).printUsage(argv[0])
          return 1
        if tunnelHandle:
          l2DestIntf = tunnelHandle
        else:
          l2DestIntf = ifNum
        Olay(clientHandle).add_l2Entry(tenant, macAddr, l2DestIntf)

      elif addFlag == 0 and macFlag and tenant: 
        if macAddr == nullMac:
         Olay(clientHandle).printUsage(argv[0])
         return 1
        Olay(clientHandle).del_l2Entry(tenant, macAddr)
      else:
         Olay(clientHandle).printUsage(argv[0])
         return 1
    elif option_name == "tenant":
      if getFlag:
        Olay(clientHandle).get_tenant(tenant)
      elif addFlag:
        Olay(clientHandle).create_tenant(tenant, tunnel_type, vlanId, srcIP, configFlags)
      else:
        Olay(clientHandle).delete_tenant(tenant)

    elif option_name == "access":
      if getFlag and ifNum:
        Olay(clientHandle).get_access_port(ifNum)
      elif getFlag:
        Olay(clientHandle).get_access_ports(tenant)
      else:
        print("Incorrect option specified %s \n" % option_name)
        Olay(clientHandle).printUsage(argv[0]);
        return 1
    elif option_name == "tunnel":
      if getFlag:
        if tunnelHandle:
          Olay(clientHandle).get_tunnel_status(tunnelHandle)
        else:
          Olay(clientHandle).get_all_tunnels()
      elif addFlag:
        Olay(clientHandle).create_tunnel(tenant, remoteIP, configFlags, tunnelHandle)
      else:
        Olay(clientHandle).delete_tunnel(tunnelHandle)

    elif option_name == "stats":
      if getFlag and tunnelHandle:
        Olay(clientHandle).tunnel_stats_get(tunnelHandle)
      elif not addFlag and tunnelHandle:
        Olay(clientHandle).tunnel_stats_clear(tunnelHandle)
      elif getFlag:
        Olay(clientHandle).resource_stats_get()
      else:
        print("Incorrect option specified %s \n" % option_name)
        Olay(clientHandle).printUsage(argv[0])
    else:
      print("Incorrect option specified %s \n" % option_name)
      Olay(clientHandle).printUsage(argv[0])
    return 0
 
  else :
    print("Failure: Unable to connect")

if __name__ == "__main__":
    main(len(sys.argv), sys.argv)

