/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  dns_example.c
*
* @purpose   DNS Configuration OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    04/01/2013
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dns.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the DNS Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printDNSAppMenu()
{
  printf("Usage:  dns_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Set DNS default domain name: dns_example 1 <domain-name> \n");
  printf("Test 2: Get DNS default domain name: dns_example 2 \n");
  printf("Test 3: Clear DNS default domain name: dns_example 3 \n");
  printf("Test 4: Add DNS IP name server: dns_example 4 <ip-address> \n");
  printf("Test 5: Delete DNS IP name server: dns_example 5 <ip-address> \n");
  printf("Test 6: Get the DNS IP name servers: dns_example 6 \n");
  printf("Test 7: Set the host name to IP address mapping:  dns_example 7 <hostname> <ip-address> \n");
  printf("Test 8: Delete the host name to IP address mapping:  dns_example 8 <hostname> \n");
  printf("Test 9: Show static DNS host name to IP address mappings: dns_example 9 \n");
  printf("Test 10: Get the IP address for the given DNS client host name: dns_example 10 <hostname> \n");
  printf("Test 11: Test DNS OpEN API Sanity: dns_example 11 \n");
  printf("Test 12: Set DNS Admin Mode: dns_example 12 <mode> \n");
  printf("Test 13: Get DNS Admin Mode: dns_example 13 \n");
  printf("Test 14: Get the DNS IP name servers with preference: dns_example 14 \n");
  printf("Test 15: Get the DNS domain hostnames: dns_example 15 \n");
  printf("Test 16: Delete the host name to IPv4/Ipv6 address mapping for a given VRF name:  dns_example 16 <vrfname> <hostname> <family>\n");
  printf("Test 17: Show static DNS host name to address mappings  for a given VRF name: dns_example 17 <vrfname>\n");

  return;
}

/*********************************************************************
* @purpose  Set the DNS default domain name 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    domainName       @b{(input)}   DNS default domain name
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dnsDefaultDomainNameSet(openapiClientHandle_t *clientHandle, char *domainName)
{
  open_error_t result;
  open_buffdesc dnsDefDomainName;
  char str[256];

  memset(str, 0, sizeof(str));
  strncpy(str, domainName, (sizeof(str) - 1));
  dnsDefDomainName.pstart = str;
  dnsDefDomainName.size = strlen(str)+1;

  if ((result = openapiDNSDefaultDomainNameSet(clientHandle, &dnsDefDomainName)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set DNS default domain name. (result = %d)\n", result);
  }
  else
  {
    printf("DNS default domain name is set successfully. (result = %d)\n", result);
  }
  
  return;
}

/*********************************************************************
* @purpose  Get the DNS default domain name 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes 
* 
* @end
*********************************************************************/
void dnsDefaultDomainNameGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t domainNameMaxLen = 0;
  open_buffdesc dnsDefDomainName;
  char *str;

  if ((result = openapiDNSDomainNameMaxSizeGet(clientHandle, &domainNameMaxLen)) != OPEN_E_NONE)
  {
    printf(" Bad return code trying to get DNS domain name maximum length. (result = %d)\n", result);
    return;
  }

  if ((str = (char *)malloc(domainNameMaxLen)) == NULL)
  {
    printf("Could not allocate memory. \n");
    return;
  }

  memset(str, 0, domainNameMaxLen);
  dnsDefDomainName.pstart = str;
  dnsDefDomainName.size = domainNameMaxLen;

  if ((result = openapiDNSDefaultDomainNameGet(clientHandle, &dnsDefDomainName)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get DNS default domain name. (result = %d)\n", result);
  }
  else
  {
    printf("DNS default domain name: %s \n", (char *)dnsDefDomainName.pstart);
  }

  free(str);
  return; 
}

/*********************************************************************
* @purpose  Delete the DNS default domain name 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dnsDefaultDomainNameDelete(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  if ((result = openapiDNSDefaultDomainNameDelete(clientHandle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete DNS default domain name. (result = %d)\n", result);
  }
  else
  {
    printf("DNS default domain name deleted successfully. (result = %d)\n", result);
  }
  
  return;   
}

/*********************************************************************
* @purpose  Add an IP name server
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    ipStr            @b{(input)}   IP address
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dnsIpNameServerAdd(openapiClientHandle_t *clientHandle, char *ipStr)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  open_buffdesc ipAddrStr;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, ipStr, (sizeof(str) - 1));
  ipAddrStr.pstart = str;
  ipAddrStr.size = sizeof(str);


  if ((result = openapiInetAddrGet(clientHandle, &ipAddrStr, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }


  if ((result = openapiDNSIpNameServerAdd(clientHandle, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add DNS IP name server. (result = %d)\n", result);
  }
  else
  {
    printf("DNS IP name server added successfully. (result = %d)\n", result);
  }
  
  return; 
}

/*********************************************************************
* @purpose  Delete an IP name server
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    ipStr            @b{(input)}   IP address
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* @notes  To delete all the name servers pass ipStr as "all"
* 
* @end
*********************************************************************/
void dnsIpNameServerDelete(openapiClientHandle_t *clientHandle,
                           char *ipStr)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  open_buffdesc ipAddrStr;
  char str[40];

  if (strcmp(ipStr, "all") == 0)
  {
    memset(&ipAddr, 0, sizeof(open_inet_addr_t));
  }
  else
  {
    memset(str, 0, sizeof(str));
    strncpy(str, ipStr, (sizeof(str) - 1));
    ipAddrStr.pstart = str;
    ipAddrStr.size = sizeof(str);
    if ((result = openapiInetAddrGet(clientHandle, &ipAddrStr, &ipAddr)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to convert IP address. (result = %d)\n", result);
      return;
    }  
  }

  if ((result = openapiDNSIpNameServerDelete(clientHandle, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete DNS IP name server. (result = %d)\n", result);
  }
  else
  {
    printf("DNS IP name server deleted successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get IP name servers.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void dnsIpNameServersGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  char ip6AddrStr[40];
  uint32_t maxIpNameServers = 0;
  struct in_addr ipAddrStr;

  if ((result = openapiDNSIpNameServersMaxSupportedGet(clientHandle, &maxIpNameServers)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get maximum IP name servers supported. (result = %d) \n", result);
    return;
  }

  memset(&ipAddr, 0, sizeof(ipAddr));
  if ((result = openapiDNSIpNameServersNextGet(clientHandle, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get first IP name server. (result = %d) \n", result); 
    return;
  }

  printf("Maximum IP name serves supported: %d\n", maxIpNameServers);

  printf("DNS IP name servers:\n");
  printf("--------------------\n");
  
  do
  {
    if (ipAddr.family == OPEN_AF_INET)
    {
      ipAddrStr.s_addr = htonl(ipAddr.addr.ipv4);
      printf("%s\n", inet_ntoa(ipAddrStr));
    }
    else
    {
      memset(ip6AddrStr, 0, sizeof(ip6AddrStr));
      if (inet_ntop(AF_INET6, (void*)&(ipAddr.addr.ipv6), ip6AddrStr, sizeof(ip6AddrStr)) == NULL)
      {
        printf("Bad return code trying to convert IPv6 address.\n");
        return;
      }
      printf("%s\n", ip6AddrStr);
    }

  }while(openapiDNSIpNameServersNextGet(clientHandle, &ipAddr) == OPEN_E_NONE);

  return;
}

/*********************************************************************
* @purpose  Set the hostname to IP address mapping
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    hostname         @b{(input)}   Host name
* @param    ipStr            @b{(input)}   IP address
* 
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch   
* 
* @end
*********************************************************************/
void dnsStaticHostAddressMappingSet(openapiClientHandle_t *clientHandle, char *hostname, char *ipStr)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  open_buffdesc dnsHostName;
  char str[256];
  open_buffdesc ipAddrStr;
  char str1[40];


  memset(str, 0, sizeof(str));
  strncpy(str, hostname, (sizeof(str) - 1));
  dnsHostName.pstart = str;
  dnsHostName.size = strlen(str)+1;
  
  memset(&ipAddr, 0, sizeof(ipAddr));

  memset(str1, 0, sizeof(str1));
  strncpy(str1, ipStr, (sizeof(str1) - 1));
  ipAddrStr.pstart = str1;
  ipAddrStr.size = sizeof(str1);


  if ((result = openapiInetAddrGet(clientHandle, &ipAddrStr, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }

  if ((result = openapiDNSStaticHostAddressMappingSet(clientHandle, &dnsHostName, &ipAddr)) != OPEN_E_NONE) 
  {
    printf("Bad return code trying to set IP host address mapping. (result = %d)\n", result);
  }
  else
  {
    printf("Host name to IP address mapping successful. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Delete static hostname.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    hostname         @b{(input)}   Host name
* 
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch   
*   
* @end
*********************************************************************/
void dnsStaticHostAddressMappingRemove(openapiClientHandle_t *clientHandle, char *hostname)
{
  open_error_t result;
  open_buffdesc dnsHostName;
  char str[256];

  memset(str, 0, sizeof(str));
  strncpy(str, hostname, (sizeof(str) - 1));
  dnsHostName.pstart = str;
  dnsHostName.size = strlen(str)+1;

  if ((result = openapiDNSStaticHostAddressMappingRemove(clientHandle, &dnsHostName)) != OPEN_E_NONE) 
  {
    printf("Bad return code trying to remove IP host address mapping. (result = %d)\n", result);
  }
  else
  {
    printf("Host name to IP address mapping removed successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Show static hostnames mapped to IP addresses.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*
* @notes
*   
* @end
*********************************************************************/
void dnsStaticHostToIpMappingsShow(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t dnsHostNameMaxSize = 0;
  open_buffdesc hostName;
  char *hostNameStr;
  open_inet_addr_t ipAddr;
  char ipAddrStr[OPEN_IP_ADDR_DISP_LENGTH];
  open_buffdesc ipAddrBufd;

  memset(ipAddrStr, 0x0, sizeof(ipAddrStr));
  ipAddrBufd.pstart = ipAddrStr;
  ipAddrBufd.size = sizeof(ipAddrStr);

  if ((result = openapiDNSDomainNameMaxSizeGet(clientHandle, &dnsHostNameMaxSize)) != OPEN_E_NONE)
  {
    printf(" Bad return code trying to get displayable DNS domain name maximum length. (result = %d)\n", result);
    return;
  }

  if ((hostNameStr = (char *)malloc(dnsHostNameMaxSize)) == NULL)
  {
    printf("Could not allocate memory. \n");
    return;
  }

  memset(hostNameStr, 0, dnsHostNameMaxSize);
  hostName.pstart = hostNameStr;
  hostName.size = dnsHostNameMaxSize;

  if ((result = openapiDNSStaticHostNextGet(clientHandle, &hostName)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get first static host name. (result = %d)\n", result);
    free(hostNameStr);
    return;
  }

  do 
  {
    printf("Hostname: %s\n", (char *) hostName.pstart);
    printf("IP Address: ");

    if ((result = openapiDNSStaticHostIpAddrGet(clientHandle, &hostName, &ipAddr)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get ip address mapped to the host name. (result = %d)\n", result);
    }
    else
    {
      (void) openapiOpenIPtoStringGet(clientHandle, ipAddr, &ipAddrBufd);
      printf("%s\n", ipAddrStr);
      printf("\n");
    }

   printf("\n");
   hostName.size = dnsHostNameMaxSize;
   memset(&ipAddr, 0, sizeof(ipAddr));
   memset(ipAddrStr, 0x0, sizeof(ipAddrStr));
   ipAddrBufd.size = sizeof(ipAddrStr);
  }while(openapiDNSStaticHostNextGet(clientHandle, &hostName) == OPEN_E_NONE);

  free(hostNameStr);
  return;
}

/*********************************************************************
* @purpose  Get the IP address for the given DNS client host name
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    hostname         @b{(input)}   Host name
* 
* @returns  none
*
* @notes
*   
* @end
*********************************************************************/
void dnsClientInetNameLookup(openapiClientHandle_t *clientHandle, char *hostname)
{
  open_error_t result;
  open_buffdesc dnsHostName;
  open_buffdesc dnsAddress;
  uint32_t iPMaxSize = 0;
  char str[256];
  char *ipStr;

  memset(str, 0, sizeof(str));
  strncpy(str, hostname, (sizeof(str) - 1));
  dnsHostName.pstart = str;
  dnsHostName.size = strlen(str)+1;

  if ((result = openapiIpAddressMaxStringLength(clientHandle, &iPMaxSize)) != OPEN_E_NONE)
  {
    printf(" Bad return code trying to get IP maximum length. (result = %d)\n", result);
    return;
  }

  if ((ipStr = (char *)malloc(iPMaxSize)) == NULL)
  {
    printf("Could not allocate memory for IP address. \n");
    return;
  }

  memset(ipStr, 0, iPMaxSize);
  dnsAddress.pstart = ipStr;
  dnsAddress.size = iPMaxSize;

  if ((result = openapiDNSClientInetNameLookup(clientHandle, &dnsHostName, &dnsAddress)) != OPEN_E_NONE) 
  {
    printf("Bad return code trying to get IP address. (result = %d)\n", result);
  }
  else
  {
    printf("Host name %s IP address %s\n", (char *)dnsHostName.pstart, (char *)dnsAddress.pstart);
  }

  free(ipStr);
  return;
}

/*********************************************************************
* @purpose  Set DNS Client administrative mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    mode             @b{(input)}   DNS Client admin mode
*
* @returns  none
*
* @notes   Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void dnsAdminModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDNSAdminModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set DNS Client Admin Mode. (result = %d)\n", result);
  }
  else
  {
    printf("DNS Client Admin Mode is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Print the DNS Client administrative mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void dnsAdminModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  if ((result = openapiDNSAdminModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get DNS Client Admin Mode. (result = %d)\n", result);
  }
  else
  {
    printf("DNS Client Admin Mode: %d. (result = %d)\n", mode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Perform Sanity of DNS OpEN APIs.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*
* @notes
*   
* @end
*********************************************************************/
void dnsOpENAPIsSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t dnsDomainNameMaxSize = 0;
  uint32_t maxIpNameServers = 0;
  char *str;
  open_buffdesc buffdesc;
  open_inet_addr_t ipAddr;
  OPEN_CONTROL_t adminMode = OPEN_DISABLE;

  if ((result = openapiDNSDomainNameMaxSizeGet(clientHandle, &dnsDomainNameMaxSize)) != OPEN_E_NONE)
  {
    printf(" Bad return code trying to get DNS domain name maximum length. (result = %d)\n", result);
    return;
  }

  if ((str = (char *)malloc(dnsDomainNameMaxSize)) == NULL)
  {
    printf("Could not allocate memory for DNS domain name. \n");
    return;
  }

  memset(str, 0, dnsDomainNameMaxSize);
  buffdesc.pstart = str;
  buffdesc.size = dnsDomainNameMaxSize;

  memset(&ipAddr, 0, sizeof(ipAddr));

  /* openapiDNSDefaultDomainNameSet() */
  printf("\nTesting openapiDNSDefaultDomainNameSet(): \n");

  result = openapiDNSDefaultDomainNameSet(NULL, &buffdesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiDNSDefaultDomainNameSet(clientHandle, NULL);
  printf("NULL buff descriptor to DNS default domain name. (result = %d)\n", result);

  printf("openapiDNSDefaultDomainNameSet() sanity successful. \n");

  /* openapiDNSDefaultDomainNameGet() */
  printf("\nTesting openapiDNSDefaultDomainNameGet(): \n");

  result = openapiDNSDefaultDomainNameGet(NULL, &buffdesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiDNSDefaultDomainNameGet(clientHandle, NULL);
  printf("NULL buff descriptor to DNS default domain name. (result = %d)\n", result);

  printf("openapiDNSDefaultDomainNameGet() sanity successful. \n");

  /* openapiDNSDomainNameMaxSizeGet() */
  printf("\nTesting openapiDNSDomainNameMaxSizeGet(): \n");

  result = openapiDNSDomainNameMaxSizeGet(NULL, &dnsDomainNameMaxSize);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiDNSDomainNameMaxSizeGet(clientHandle, NULL);
  printf("NULL default domain name maximum length. (result = %d)\n", result);

  printf("openapiDNSDomainNameMaxSizeGet() sanity successful. \n");

  /* openapiDNSDefaultDomainNameDelete() */
  printf("\nTesting openapiDNSDefaultDomainNameDelete(): \n");

  result = openapiDNSDefaultDomainNameDelete(NULL);
  printf("NULL Client Handle. (result = %d)\n", result);

  printf("openapiDNSDefaultDomainNameDelete() sanity successful. \n");

  /* openapiDNSIpNameServerAdd() */ 
  printf("\nTesting openapiDNSIpNameServerAdd(): \n");

  result = openapiDNSIpNameServerAdd(NULL, &ipAddr);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiDNSIpNameServerAdd(clientHandle, NULL);
  printf("NULL ipAddr. (result = %d)\n", result);

  printf("openapiDNSIpNameServerAdd() sanity successful. \n");

  /* openapiDNSIpNameServerDelete() */ 
  printf("\nTesting openapiDNSIpNameServerDelete(): \n");

  result = openapiDNSIpNameServerDelete(NULL, &ipAddr);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiDNSIpNameServerDelete(clientHandle, NULL);
  printf("NULL ipAddr. (result = %d)\n", result);

  printf("openapiDNSIpNameServerDelete() sanity successful. \n");

  /* openapiDNSIpNameServersNextGet() */ 
  printf("\nTesting openapiDNSIpNameServersNextGet(): \n");

  result = openapiDNSIpNameServersNextGet(NULL, &ipAddr);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiDNSIpNameServersNextGet(clientHandle, NULL);
  printf("NULL ipAddr. (result = %d)\n", result);

  printf("openapiDNSIpNameServersNextGet() sanity successful. \n");

  /* openapiDNSIpNameServersMaxSupportedGet() */
  printf("\nTesting openapiDNSIpNameServersMaxSupportedGet(): \n");

  result = openapiDNSIpNameServersMaxSupportedGet(NULL, &maxIpNameServers);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiDNSIpNameServersMaxSupportedGet(clientHandle, NULL);
  printf("NULL maximum IP name servers. (result = %d)\n", result);

  printf("openapiDNSIpNameServersMaxSupportedGet() sanity successful. \n");

  /* openapiDNSStaticHostAddressMappingSet() */
  printf("\nTesting openapiDNSStaticHostAddressMappingSet(): \n");

  result = openapiDNSStaticHostAddressMappingSet(NULL, &buffdesc, &ipAddr);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiDNSStaticHostAddressMappingSet(clientHandle, NULL, &ipAddr);
  printf("NULL buff descriptor to DNS hostname. (result = %d)\n", result);

  result = openapiDNSStaticHostAddressMappingSet(clientHandle, &buffdesc, NULL);
  printf("NULL ipAddr. (result = %d)\n", result);

  printf("openapiDNSStaticHostAddressMappingSet() sanity successful. \n");

  /* openapiDNSStaticHostAddressMappingRemove() */
  printf("\nTesting openapiDNSStaticHostAddressMappingRemove(): \n");

  result = openapiDNSStaticHostAddressMappingRemove(NULL, &buffdesc); 
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiDNSStaticHostAddressMappingRemove(clientHandle, NULL); 
  printf("NULL buff descriptor to DNS hostname. (result = %d)\n", result);

  printf("openapiDNSStaticHostAddressMappingRemove() sanity successful. \n");

  /* openapiDNSStaticHostNextGet() */
  printf("\nTesting openapiDNSStaticHostNextGet(): \n");

  result = openapiDNSStaticHostNextGet(NULL, &buffdesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiDNSStaticHostNextGet(clientHandle, NULL);
  printf("NULL buff descriptor to DNS hostname. (result = %d)\n", result);

  printf("openapiDNSStaticHostNextGet() sanity successful. \n");

  /* openapiDNSStaticHostIpAddrGet() */
  printf("\nTesting openapiDNSStaticHostIpAddrGet(): \n");

  result = openapiDNSStaticHostIpAddrGet(NULL, &buffdesc, &ipAddr);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiDNSStaticHostIpAddrGet(clientHandle, NULL, &ipAddr);
  printf("NULL buff descriptor to DNS hostname. (result = %d)\n", result);

  result = openapiDNSStaticHostIpAddrGet(clientHandle, &buffdesc, NULL);
  printf("NULL ipAddr. (result = %d)\n", result);  

  printf("openapiDNSStaticHostIpAddrGet() sanity successful. \n");

/* openapiDNSAdminModeSet() */
  printf("\nTesting openapiDNSAdminModeSet(): \n");

  result = openapiDNSAdminModeSet(NULL, adminMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiDNSAdminModeSet(clientHandle, 4);
  printf("Invalid DNS Client admin mode. (result = %d)\n", result);

  printf("openapiDNSAdminModeSet() sanity successful.\n");

  /* openapiDNSAdminModeGet() */
  printf("\nTesting openapiDNSAdminModeGet(): \n");

  result = openapiDNSAdminModeGet(NULL, &adminMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiDNSAdminModeGet(clientHandle, NULL);
  printf("NULL DNS Client admin mode. (result = %d)\n", result);

  printf("openapiDNSAdminModeGet sanity successful.\n");
  
  free(str);
  return;
}

/*********************************************************************
* @purpose  Get IP name servers with preference
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void dnsIpNameServersWithPrefGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  char ip6AddrStr[40];
  struct in_addr ipAddrStr;
  uint32_t preference = 0;

  memset(&ipAddr, 0, sizeof(ipAddr));
  if ((result = openapiDNSIpNameServersWithPrefNextGet (clientHandle, &ipAddr, &preference))
                                                     != OPEN_E_NONE)
  {
    printf ("Bad return code trying to get first IP name server. (result = %d) \n",
            result);
    return;
  }

  printf("\nDNS IP name servers:\n");
  printf("--------------------\n");

  do
  {
    if (ipAddr.family == OPEN_AF_INET)
    {
      ipAddrStr.s_addr = htonl(ipAddr.addr.ipv4);
      printf("\n%s\n", inet_ntoa(ipAddrStr));
    }
    else if (ipAddr.family == OPEN_AF_INET6)
    {
      memset(ip6AddrStr, 0, sizeof(ip6AddrStr));
      if (inet_ntop(AF_INET6, (void*)&(ipAddr.addr.ipv6), ip6AddrStr, sizeof(ip6AddrStr)) == NULL)
      {
        printf("Bad return code trying to convert IPv6 address.\n");
        return;
      }
      printf("\n%s\n", ip6AddrStr);
    }
    else
    {
      printf ("Bad address family: %d \n", ipAddr.family);
      return;
    }

    printf ("  Preference: %u\n", preference);

  } while (openapiDNSIpNameServersWithPrefNextGet (clientHandle, &ipAddr, &preference) == OPEN_E_NONE);

  return;
}

/*********************************************************************
* @purpose  Get DNS domain hostnames
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void
dnsIpDomainHostNameServersGet (openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  char ip6AddrStr[40];
  struct in_addr ipAddrStr;
  openDnsDynamicHostEntry_t domainNameEntry;

  memset(&domainNameEntry, 0, sizeof(domainNameEntry));

  if ((result = openapiDnsDomainNameEntryNextGet (clientHandle, &domainNameEntry))
                                               != OPEN_E_NONE)
  {
    printf ("Bad return code trying to get first Domain hostname server. (result = %d) \n",
            result);
    return;
  }

  printf("\nDNS Domain Hostnames:\n");
  printf("---------------------\n");

  do
  {
    printf ("\n  Hostname: %s\n", domainNameEntry.hostName);
    printf ("    TTL: %u\n", domainNameEntry.ttl);
    printf ("    TTL elapsed: %u\n", domainNameEntry.ttlElapsed);
    printf ("    Entry type: %u\n", domainNameEntry.entryType);

    if (domainNameEntry.hostAddr.family == OPEN_AF_INET)
    {
      ipAddrStr.s_addr = htonl(domainNameEntry.hostAddr.addr.ipv4);
      printf("    IPv4 address: %s\n", inet_ntoa(ipAddrStr));
    }
    else if (domainNameEntry.hostAddr.family == OPEN_AF_INET6)
    {
      memset(ip6AddrStr, 0, sizeof(ip6AddrStr));
      if (inet_ntop(AF_INET6, (void*)&(domainNameEntry.hostAddr.addr.ipv6), ip6AddrStr, sizeof(ip6AddrStr)) == NULL)
      {
        printf("Bad return code trying to convert IPv6 address.\n");
        return;
      }
      printf("    IPv6 address: %s\n", ip6AddrStr);
    }
    else
    {
      printf ("Bad address family: %d \n", ipAddr.family);
    }

  } while (openapiDnsDomainNameEntryNextGet (clientHandle, &domainNameEntry) == OPEN_E_NONE);

  return;
}

/*********************************************************************
 * @purpose  Delete static hostname.
 * 
 * @param    clientHandle  @b{(input)}   Client handle from registration API.
 * @param    vrf Name      @b{(input)}   VRF name.
 * @param    hostname      @b{(input)}   Host name.
 * @param    family        @b{(input)}   IPv4/Ipv6 Address.
 * 
 * @returns  none
 *
 * @notes  Calling this API will change the running configuration of the switch   
 *   
 * @end
 *********************************************************************/
void dnsStaticHostAddressMappingDelete(openapiClientHandle_t *clientHandle,
                                       char *vrfName,
                                       char *hostname,
                                       OPEN_AF_t family)
{
  open_error_t result;
  open_buffdesc dnsHostName;
  char str[256];
  open_buffdesc vrfNameBufd;
  char str1[256];

  memset(str, 0, sizeof(str));
  strncpy(str, hostname, sizeof(str));
  dnsHostName.pstart = str;
  dnsHostName.size = strlen(str) + 1;

  memset(str1, 0, sizeof(str1));
  strncpy(str1, vrfName, sizeof(str1));
  vrfNameBufd.pstart = str1;
  vrfNameBufd.size = strlen(str1) + 1;

  if ((result = openapiDNSStaticHostAddressMappingDelete(clientHandle, &vrfNameBufd,
                                                         &dnsHostName, family)) != OPEN_E_NONE) 
  {
    printf("Bad return code trying to remove IP host address mapping. (result = %d)\n", result);
  }
  else
  {
    printf("Host name to IP address mapping removed successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
 * @purpose  Show static hostnames mapped to IP addresses.
 * 
 * @param    clientHandle  @b{(input)}   Client handle from registration API.
 * @param    vrf Name      @b{(input)}   VRF name.
 * 
 * @returns  none
 *
 * @notes
 *   
 * @end
 *********************************************************************/
void dnsStaticHostToIpAddressMappingsShow(openapiClientHandle_t *clientHandle,
                                          char *vrfName)
{
  open_error_t result;
  uint32_t dnsHostNameMaxSize = 0;
  open_buffdesc hostName;
  char *hostNameStr;
  char ipAddrStr[OPEN_IP_ADDR_DISP_LENGTH];
  open_inet_addr_t ipAddr;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  open_buffdesc ipAddrBufd;

  memset(&ipAddr, 0, sizeof(ipAddr));
  memset(ipAddrStr, 0x0, sizeof(ipAddrStr));
  ipAddrBufd.pstart = ipAddrStr;
  ipAddrBufd.size = sizeof(ipAddrStr);

  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr));
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  if ((result = openapiDNSDomainNameMaxSizeGet(clientHandle, &dnsHostNameMaxSize)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get displayable DNS domain name maximum "
           "length. (result = %d)\n", result);
    return;
  }

  if ((hostNameStr = (char *)malloc(dnsHostNameMaxSize)) == NULL)
  {
    printf("Could not allocate memory. \n");
    return;
  }

  memset(hostNameStr, 0, dnsHostNameMaxSize);
  hostName.pstart = hostNameStr;
  hostName.size = dnsHostNameMaxSize;

  if ((result = openapiDNSStaticHostAddressMappingNextGet(clientHandle, &vrfNameBufd,
                                                          &hostName, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get first static host name. (result = %d)\n", result);
    free(hostNameStr);
    return;
  }

  do 
  {
    printf("Hostname: %s\n", (char *) hostName.pstart);
    printf("IP Address: ");

    (void) openapiOpenIPtoStringGet(clientHandle, ipAddr, &ipAddrBufd);
    printf("%s\n", ipAddrStr);
    printf("\n");

    hostName.size = dnsHostNameMaxSize;
    memset(ipAddrStr, 0x0, sizeof(ipAddrStr));
    ipAddrBufd.size = sizeof(ipAddrStr);
  }while (openapiDNSStaticHostAddressMappingNextGet(clientHandle, &vrfNameBufd, &hostName, &ipAddr) == OPEN_E_NONE);

  free(hostNameStr);
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  uint32_t arg1;
  
  if (argc < 2)
  {
    printDNSAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("dns_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting DNS API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3) 
      {
        printDNSAppMenu();
        exit(1);
      }
      dnsDefaultDomainNameSet(&clientHandle, argv[2]);
      break;
    case 2:
      if (argc != 2)
      {
        printDNSAppMenu();
        exit(1);
      }
      dnsDefaultDomainNameGet(&clientHandle);
      break;
    case 3:
      if (argc != 2)
      {
        printDNSAppMenu();
        exit(1);
      }
      dnsDefaultDomainNameDelete(&clientHandle);
      break;
    case 4:
      if (argc != 3)
      {
        printDNSAppMenu();
        exit(1);
      }
      dnsIpNameServerAdd(&clientHandle, argv[2]);
      break;       
    case 5:
      if (argc != 3)
      {
        printDNSAppMenu();
        exit(1);
      }
      dnsIpNameServerDelete(&clientHandle, argv[2]);
      break;
    case 6:
      if (argc != 2)
      {
        printDNSAppMenu();
        exit(1);
      }
      dnsIpNameServersGet(&clientHandle);
      break;
    case 7:
      if (argc != 4)
      {
        printDNSAppMenu();
        exit(1);
      }
      dnsStaticHostAddressMappingSet(&clientHandle, argv[2], argv[3]);
      break;
    case 8:
      if (argc != 3)
      {
        printDNSAppMenu();
        exit(1);
      }
      dnsStaticHostAddressMappingRemove(&clientHandle, argv[2]);      
      break;
    case 9:
      if (argc != 2)
      {
        printDNSAppMenu();
        exit(1);
      }
      dnsStaticHostToIpMappingsShow(&clientHandle); 
      break;
    case 10:
     if (argc != 3)
      {
        printDNSAppMenu();
        exit(1);
      }
     dnsClientInetNameLookup(&clientHandle, argv[2]);
     break;
    case 11:
     if (argc != 2)
      {
        printDNSAppMenu();
        exit(1);
      }
     dnsOpENAPIsSanity(&clientHandle);
     break;

    case 12:
      if (argc != 3)
      {
        printDNSAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dnsAdminModeSet(&clientHandle, arg1);
      break;

    case 13:
      if (argc != 2)
      {
        printDNSAppMenu();
        exit(1);
      }
      dnsAdminModeGet(&clientHandle);
      break;

    case 14:
      if (argc != 2)
      {
        printDNSAppMenu();
        exit(1);
      }
      dnsIpNameServersWithPrefGet(&clientHandle);
      break;

    case 15:
      if (argc != 2)
      {
        printDNSAppMenu();
        exit(1);
      }
      dnsIpDomainHostNameServersGet(&clientHandle);
      break;

    case 16:
      if (argc != 5)
      {
        printDNSAppMenu();
        exit(1);
      }
      dnsStaticHostAddressMappingDelete(&clientHandle, argv[2], argv[3], atoi(argv[4]));      
      break;

    case 17:
      if (argc != 3)
      {
        printDNSAppMenu();
        exit(1);
      }
      dnsStaticHostToIpAddressMappingsShow(&clientHandle, argv[2]);
      break;

    default:
      printDNSAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping DNS API example application");
        
  (void) openapiClientTearDown(&clientHandle);        
  return 0;
}
