/*********************************************************************
*
*  Copyright 2022-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  dos_example.c
*
* @purpose   OpEN DoS example.
*
* @component OpEN
*
* @create    11/21/2022
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dos.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the DoS Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Sets source IP address=destination IP address (SIP=DIP) denial of service protection: %s 1 <mode>\n", name);
  printf("Test 2: Sets source MAC address=destination MAC address (SMAC=DMAC) denial of service protection: %s 2 <mode>\n", name);
  printf("Test 3: Sets first fragment denial of service protection: %s 3 <mode>\n", name);
  printf("Test 4: Sets minimum TCP header size denial of service protection: %s 4 <minTCPHdrSize>\n", name);
  printf("Test 5: Sets TCP fragment denial of service protection: %s 5 <mode>\n", name);
  printf("Test 6: Sets TCP offset denial of service protection: %s 6 <mode>\n", name);
  printf("Test 7: Sets TCP flag denial of service protection: %s 7 <mode>\n", name);
  printf("Test 8: Sets TCP flag and sequence denial of service protection: %s 8 <mode>\n", name);
  printf("Test 9: Sets TCP port denial of service protection: %s 9 <mode>\n", name);
  printf("Test 10: Sets UDP port denial of service protection: %s 10 <mode>\n", name);
  printf("Test 11: Sets TCP SYN denial of service protection: %s 11 <mode>\n", name);
  printf("Test 12: Sets TCP SYN and FIN denial of service protection: %s 12 <mode>\n", name);
  printf("Test 13: Sets TCP FIN,URG and PSH denial of service protection: %s 13 <mode>\n", name);
  printf("Test 14: Sets L4 Port denial of service protection: %s 14 <mode>\n", name);
  printf("Test 15: Sets ICMP denial of service protection: %s 15 <mode>\n", name);
  printf("Test 16: Sets maximum ICMP size denial of service protection: %s 16 <maxICMPSize>\n", name);
  printf("Test 17: Sets maximum ICMPv6 size denial of service protection: %s 17 <maxICMPv6Size>\n", name);
  printf("Test 18: Sets ICMP Fragment denial of service protection: %s 18 <mode>\n", name);
  printf("Test 19: Gets source IP address=destination IP address (SIP=DIP) denial of service protection mode: %s 19\n", name);
  printf("Test 20: Gets source MAC address=destination MAC address (SMAC=DMAC) denial of service protection mode: %s 20\n", name);
  printf("Test 21: Gets first fragment denial of service protection mode: %s 21\n", name);
  printf("Test 22: Gets TCP Partial denial of service protection mode: %s 22\n", name);
  printf("Test 23: Gets denial of service protection Minimum TCP Header Size: %s 23\n", name);
  printf("Test 24: Gets TCP Offset denial of service protection mode: %s 24\n", name);
  printf("Test 25: Gets TCP port denial of service protection mode: %s 25\n", name);
  printf("Test 26: Gets UDP port denial of service protection mode: %s 26\n", name);
  printf("Test 27: Gets TCP Fragment denial of service protection mode: %s 27\n", name);
  printf("Test 28: Gets TCP flag denial of service protection mode: %s 28\n", name);
  printf("Test 29: Gets TCP flag and sequence denial of service protection mode: %s 29\n", name);
  printf("Test 30: Gets TCP SYN denial of service protection mode: %s 30\n", name);
  printf("Test 31: Gets TCP SYN and FIN denial of service protection mode: %s 31\n", name);
  printf("Test 32: Gets TCP FIN,URG and PSH denial of service protection mode: %s 32\n", name);
  printf("Test 33: Gets L4 Port denial of service protection mode: %s 33\n", name);
  printf("Test 34: Gets ICMP denial of service protection mode: %s 34\n", name);
  printf("Test 35: Gets denial of service protection maximum ICMP size: %s 35\n", name);
  printf("Test 36: Sets ping flooding denial of service protection on an interface: %s 36 <interface><mode><param>\n", name);
  printf("Test 37: Gets ping flooding denial of service protection mode on an interface: %s 37 <interface>\n", name);
  printf("Test 38: Sets  preventing Smurf denial of service protection on an interface: %s 38 <interface><mode>\n", name);
  printf("Test 39: Gets Smurf denial of service protection mode on an interface: %s 39 <interface>\n", name);
  printf("Test 40: Gets SynAck Flooding denial of service protection mode on an interface: %s 40 <interface>\n", name);
  printf("Test 41: Sets ping flooding denial of service protection: %s 41 <mode><param>\n", name);
  printf("Test 42: Gets ping flooding denial of service protection mode: %s 42\n", name);
  printf("Test 43: Sets preventing Smurf denial of service protection: %s 43 <mode>\n", name);
  printf("Test 44: Gets Smurf denial of service protection mode: %s 44\n", name);
  printf("Test 45: Sets preventing SynAck Flooding denial of service protection: %s 45 <mode><param>\n", name);
  printf("Test 46: Gets preventing SynAck Flooding denial of service protection mode: %s46\n", name);
  printf("Test 47: Gets denial of service protection Maximum ICMPv4 payload size: %s 47\n", name);
  printf("Test 48: Gets denial of service protection Maximum ICMPv6 payload size: %s 48\n", name);
  printf("Test 49: Gets ICMP Fragment denial of service protection mode: %s 49\n", name);
  printf("Test 50: Gets ICMPv6 denial of service protection mode: %s 50\n", name);
  printf("Test 51: Sets ICMPv6 denial of service protection: %s 51 <mode>\n", name);
  printf("Test 52: Gets port D-disable state for denial of service protection: %s 52\n", name);
  printf("Test 53: Sets port D-disable for denial of service protection: %s 53 <mode>\n", name);
  printf("Test 54: Gets Auto denial of service protection mode: %s 54\n", name);
  printf("Test 55: Sets Auto denial of service protection: %s 55 <mode>\n", name);
  printf("Test 56: Gets Vlan0 Tagged and SNAP Drop denial of service protection mode: %s 56\n", name);
  printf("Test 57: Sets Vlan0 Tagged and SNAP Drop denial of service protection: %s 57 <mode>\n", name);
  printf("Test 58: Run API sanity checks: %s 58 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t maxICMPSize;
  OPEN_CONTROL_t mode;
  uint32_t minTCPHdrSize;
  uint32_t param;
  uint32_t interface=10;

  printf("Testing DoS OpEN APIs sanity:\n");

  printf("Testing openapiDoSSIPDIPModeGet():\n");
  result = openapiDoSSIPDIPModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSSIPDIPModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSSMACDMACModeGet():\n");
  result = openapiDoSSMACDMACModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSSMACDMACModeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDoSFirstFragModeGet():\n");
  result = openapiDoSFirstFragModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSFirstFragModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSTCPPartialModeGet():\n");
  result = openapiDoSTCPPartialModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSTCPPartialModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSFirstFragSizeGet():\n");
  result = openapiDoSFirstFragSizeGet(NULL, &minTCPHdrSize);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSFirstFragSizeGet(clientHandle, NULL);
  printf("NULL first Frag Size argument:(result = %d)\n", result);

  printf("Testing openapiDoSTCPOffsetModeGet():\n");
  result = openapiDoSTCPOffsetModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSTCPOffsetModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSTCPPortModeGet():\n");
  result = openapiDoSTCPPortModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSTCPPortModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSUDPPortModeGet():\n");
  result = openapiDoSUDPPortModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSUDPPortModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSTCPFragModeGet():\n");
  result = openapiDoSTCPFragModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSTCPFragModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSTCPFlagModeGet():\n");
  result = openapiDoSTCPFlagModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSTCPFlagModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSTCPFlagSeqModeGet():\n");
  result = openapiDoSTCPFlagSeqModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSTCPFlagSeqModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSTCPSynModeGet():\n");
  result = openapiDoSTCPSynModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSTCPSynModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSTCPSynFinModeGet():\n");
  result = openapiDoSTCPSynFinModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSTCPSynFinModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSTCPFinUrgPshModeGet():\n");
  result = openapiDoSTCPFinUrgPshModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSTCPFinUrgPshModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSL4PortModeGet():\n");
  result = openapiDoSL4PortModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSL4PortModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSICMPModeGet():\n");
  result = openapiDoSICMPModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSICMPModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSICMPSizeGet():\n");
  result = openapiDoSICMPSizeGet(NULL, &maxICMPSize);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSICMPSizeGet(clientHandle, NULL);
  printf("NULL min ICMP size argument 2:(result = %d)\n", result);

  printf("Testing openapiDoSIntfPingFloodingModeGet():\n");
  result = openapiDoSIntfPingFloodingModeGet(NULL, interface, &mode, &param);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSIntfPingFloodingModeGet(clientHandle, interface, NULL, &param);
  printf("NULL mode argument:(result = %d)\n", result);
  result = openapiDoSIntfPingFloodingModeGet(clientHandle, interface, &mode, NULL);
  printf("NULL param argument:(result = %d)\n", result);

  printf("Testing openapiDoSIntfSmurfAttackModeGet():\n");
  result = openapiDoSIntfSmurfAttackModeGet(NULL, interface, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSIntfSmurfAttackModeGet(clientHandle, interface, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSIntfSynAckFloodingModeGet():\n");
  result = openapiDoSIntfSynAckFloodingModeGet(NULL, interface, &mode, &param);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSIntfPingFloodingModeGet(clientHandle, interface, NULL, &param);
  printf("NULL mode argument:(result = %d)\n", result);
  result = openapiDoSIntfPingFloodingModeGet(clientHandle, interface, &mode, NULL);
  printf("NULL param argument:(result = %d)\n", result);

  printf("Testing openapiDoSGlobalPingFloodingModeGet():\n");
  result = openapiDoSGlobalPingFloodingModeGet(NULL, &mode, &param);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSGlobalPingFloodingModeGet(clientHandle, NULL, &param);
  printf("NULL mode argument:(result = %d)\n", result);
  result = openapiDoSGlobalPingFloodingModeGet(clientHandle, &mode, NULL);
  printf("NULL param argument:(result = %d)\n", result);

  printf("Testing openapiDoSGlobalSmurfAttackModeGet():\n");
  result = openapiDoSGlobalSmurfAttackModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSGlobalSmurfAttackModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSGlobalSynAckFloodingModeGet():\n");
  result = openapiDoSGlobalSynAckFloodingModeGet(NULL, &mode, &param);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSGlobalSynAckFloodingModeGet(clientHandle, NULL, &param);
  printf("NULL mode argument:(result = %d)\n", result);
  result = openapiDoSGlobalSynAckFloodingModeGet(clientHandle, &mode, NULL);
  printf("NULL param argument:(result = %d)\n", result);

  printf("Testing openapiDoSICMPv4SizeGet():\n");
  result = openapiDoSICMPv4SizeGet(NULL, &maxICMPSize);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSICMPv4SizeGet(clientHandle, NULL);
  printf("NULL max ICMPv4 size argument :(result = %d)\n", result);

  printf("Testing openapiDoSICMPv6SizeGet():\n");
  result = openapiDoSICMPv6SizeGet(NULL, &maxICMPSize);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSICMPv6SizeGet(clientHandle, NULL);
  printf("NULL max ICMPv6 size argument:(result = %d)\n", result);

  printf("Testing openapiDoSICMPFragModeGet():\n");
  result = openapiDoSICMPFragModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSICMPFragModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSICMPv6ModeGet():\n");
  result = openapiDoSICMPv6ModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSICMPv6ModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSPortDdisableModeGet():\n");
  result = openapiDoSPortDdisableModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSPortDdisableModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSAutoModeGet():\n");
  result = openapiDoSAutoModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSAutoModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  printf("Testing openapiDoSVlan0TaggedSnapDropModeGet():\n");
  result = openapiDoSVlan0TaggedSnapDropModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDoSVlan0TaggedSnapDropModeGet(clientHandle, NULL);
  printf("NULL mode argument:(result = %d)\n", result);

  return;
}


/*********************************************************************
* @purpose  Sets source IP address=destination IP address (SIP=DIP) denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSSIPDIPModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSSIPDIPModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets source IP address=destination IP address (SIP=DIP) denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("Source IP address=destination IP address (SIP=DIP) denial of service protection is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets source MAC address=destination MAC address (SMAC=DMAC) denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSSMACDMACModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSSMACDMACModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets source MAC address=destination MAC address (SMAC=DMAC) denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("Source MAC address=destination MAC address (SMAC=DMAC) denial of service protection is successfully %s\n", 
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets first fragment denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSFirstFragModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSFirstFragModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets first fragment denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("First fragment denial of service protection is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets minimum TCP header size denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    minTCPHdrSize       @b{(input)}  minimum TCP header size

*
* @returns  none
*
* @end
*********************************************************************/
void doSFirstFragSizeSet(openapiClientHandle_t *client_handle, uint32_t minTCPHdrSize)
{
  open_error_t result;

  if ((result = openapiDoSFirstFragSizeSet(client_handle, minTCPHdrSize)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets minimum TCP header size denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("Minimum TCP header size for denial of service protection is set to %d\n", minTCPHdrSize);
  }
  return;
}


/*********************************************************************
* @purpose  Sets TCP fragment denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSTCPFragModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSTCPFragModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets TCP fragment denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("TCP fragment denial of service protection is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets TCP offset denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSTCPOffsetModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSTCPOffsetModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets TCP offset denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("TCP offset denial of service protection is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets TCP flag denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSTCPFlagModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSTCPFlagModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets TCP flag denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("TCP flag denial of service protection is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets TCP flag and sequence denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSTCPFlagSeqModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSTCPFlagSeqModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets TCP flag and sequence denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("TCP flag and sequence denial of service protection is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets TCP port denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSTCPPortModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSTCPPortModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets TCP port denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("TCP port denial of service protection is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets UDP port denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSUDPPortModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSUDPPortModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets UDP port denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("UDP port denial of service protection is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets TCP SYN denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSTCPSynModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSTCPSynModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets TCP SYN denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("TCP SYN denial of service protection is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets TCP SYN and FIN denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSTCPSynFinModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSTCPSynFinModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets TCP SYN and FIN denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("TCP SYN and FIN denial of service protection is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets TCP FIN,URG and PSH denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSTCPFinUrgPshModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSTCPFinUrgPshModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets TCP FIN,URG and PSH denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("TCP FIN,URG and PSH denial of service protection is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets L4 Port denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSL4PortModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSL4PortModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets L4 Port denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("L4 Port denial of service protection is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets ICMP denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSICMPModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSICMPModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets ICMP denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("ICMP denial of service protection is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets maximum ICMP size denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    maxICMPSize       @b{(input)}  minimum ICMP size

*
* @returns  none
*
* @end
*********************************************************************/
void doSICMPSizeSet(openapiClientHandle_t *client_handle, uint32_t maxICMPSize)
{
  open_error_t result;

  if ((result = openapiDoSICMPSizeSet(client_handle, maxICMPSize)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets maximum ICMP size denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("Maximum ICMPv4 size denial of service protection is set to %d\n", maxICMPSize);
  }
  return;
}


/*********************************************************************
* @purpose  Sets maximum ICMPv6 size denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    maxICMPv6Size       @b{(input)}  minimum ICMPv6 size

*
* @returns  none
*
* @end
*********************************************************************/
void doSICMPv6SizeSet(openapiClientHandle_t *client_handle, uint32_t maxICMPv6Size)
{
  open_error_t result;

  if ((result = openapiDoSICMPv6SizeSet(client_handle, maxICMPv6Size)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets maximum ICMPv6 size denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("Maximum ICMPv6 size denial of service protection is set to %d\n", maxICMPv6Size);
  }
  return;
}


/*********************************************************************
* @purpose  Sets ICMP Fragment denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSICMPFragModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSICMPFragModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets ICMP Fragment denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("ICMP Fragment denial of service protection is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets source IP address=destination IP address (SIP=DIP) denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSSIPDIPModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSSIPDIPModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets source IP address=destination IP address (SIP=DIP) denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("Source IP address=destination IP address (SIP=DIP) denial of service protection is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets source MAC address=destination MAC address (SMAC=DMAC) denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSSMACDMACModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSSMACDMACModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets source MAC address=destination MAC address (SMAC=DMAC) denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("Source MAC address=destination MAC address (SMAC=DMAC) denial of service protection mode is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets first fragment denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSFirstFragModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSFirstFragModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets first fragment denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("First fragment denial of service protection is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets TCP Partial denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSTCPPartialModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSTCPPartialModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets TCP Partial denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("TCP Partial denial of service protection mode is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets denial of service protection Minimum TCP Header Size.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMinTCPHdrSize       @b{(output)}  minimum TCP header size

*
* @returns  none
*
* @end
*********************************************************************/
void doSFirstFragSizeGet(openapiClientHandle_t *client_handle, uint32_t *pMinTCPHdrSize)
{
  open_error_t result;

  if ((result = openapiDoSFirstFragSizeGet(client_handle, pMinTCPHdrSize)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets denial of service protection Minimum TCP Header Size. (result = %d)\n", result);
  }
  else
  {
    printf("Denial of service protection Minimum TCP Header Size is %d\n", *pMinTCPHdrSize);
  }
  return;
}


/*********************************************************************
* @purpose  Gets TCP Offset denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSTCPOffsetModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSTCPOffsetModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets TCP Offset denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("TCP Offset denial of service protection mode is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets TCP port denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSTCPPortModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSTCPPortModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets TCP port denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("TCP port denial of service protection mode is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets UDP port denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSUDPPortModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSUDPPortModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets UDP port denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("UDP port denial of service protection mode is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets TCP Fragment denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSTCPFragModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSTCPFragModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets TCP Fragment denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("TCP Fragment denial of service protection mode is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets TCP flag denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSTCPFlagModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSTCPFlagModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets TCP flag denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("TCP flag denial of service protection mode is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets TCP flag and sequence denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSTCPFlagSeqModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSTCPFlagSeqModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets TCP flag and sequence denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("TCP flag and sequence denial of service protection mode is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets TCP SYN denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSTCPSynModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSTCPSynModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets TCP SYN denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("TCP SYN denial of service protection mode is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets TCP SYN and FIN denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSTCPSynFinModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSTCPSynFinModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets TCP SYN and FIN denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("TCP SYN and FIN denial of service protection mode is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets TCP FIN,URG and PSH denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSTCPFinUrgPshModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSTCPFinUrgPshModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets TCP FIN,URG and PSH denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("TCP FIN,URG and PSH denial of service protection mode is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets L4 Port denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSL4PortModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSL4PortModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets L4 Port denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("L4 Port denial of service protection mode is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets ICMP denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSICMPModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSICMPModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets ICMP denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("ICMP denial of service protection mode is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets denial of service protection maximum ICMP size.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMaxICMPSize        @b{(output)} maximum ICMP size

*
* @returns  none
*
* @end
*********************************************************************/
void doSICMPSizeGet(openapiClientHandle_t *client_handle, uint32_t *pMaxICMPSize)
{
  open_error_t result;

  if ((result = openapiDoSICMPSizeGet(client_handle, pMaxICMPSize)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets denial of service protection maximum ICMP size. (result = %d)\n", result);
  }
  else
  {
    printf("Denial of service protection maximum ICMP size is %d\n", *pMaxICMPSize);
  }
  return;
}


/*********************************************************************
* @purpose  Sets ping flooding denial of service protection on an interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    interface       @b{(input)}  interface number
* @param    mode       @b{(input)}  Mode [Enable/Disable]
* @param    param       @b{(input)}  Param value

*
* @returns  none
*
* @end
*********************************************************************/
void doSIntfPingFloodingModeSet(openapiClientHandle_t *client_handle, uint32_t interface, OPEN_CONTROL_t mode, uint32_t param)
{
  open_error_t result;

  if ((result = openapiDoSIntfPingFloodingModeSet(client_handle, interface, mode, param)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets ping flooding denial of service protection on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Ping flooding denial of service protection on an interface is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets ping flooding denial of service protection mode on an interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    interface       @b{(input)}  interface number
* @param    pMode       @b{(output)}  Mode [Enable/Disable]
* @param    pParam       @b{(output)}  Param value

*
* @returns  none
*
* @end
*********************************************************************/
void doSIntfPingFloodingModeGet(openapiClientHandle_t *client_handle, uint32_t interface, OPEN_CONTROL_t *pMode, uint32_t *pParam)
{
  open_error_t result;

  if ((result = openapiDoSIntfPingFloodingModeGet(client_handle, interface, pMode, pParam)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets ping flooding denial of service protection mode on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Ping flooding denial of service protection on an interface is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets  preventing Smurf denial of service protection on an interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    interface       @b{(input)}  interface number
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSIntfSmurfAttackModeSet(openapiClientHandle_t *client_handle, uint32_t interface, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSIntfSmurfAttackModeSet(client_handle, interface, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets  preventing Smurf denial of service protection on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Preventing Smurf denial of service protection on an interface is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets Smurf denial of service protection mode on an interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    interface       @b{(input)}  interface number
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSIntfSmurfAttackModeGet(openapiClientHandle_t *client_handle, uint32_t interface, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSIntfSmurfAttackModeGet(client_handle, interface, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets Smurf denial of service protection mode on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Preventing Smurf denial of service protection on an interface is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets SynAck Flooding denial of service protection mode on an interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    interface       @b{(input)}  interface number
* @param    pMode       @b{(output)}  Mode [Enable/Disable]
* @param    pParam       @b{(output)}  Param value

*
* @returns  none
*
* @end
*********************************************************************/
void doSIntfSynAckFloodingModeGet(openapiClientHandle_t *client_handle, uint32_t interface, OPEN_CONTROL_t *pMode, uint32_t *pParam)
{
  open_error_t result;

  if ((result = openapiDoSIntfSynAckFloodingModeGet(client_handle, interface, pMode, pParam)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets SynAck Flooding denial of service protection mode on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("SynAck Flooding denial of service protection mode on an interface is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets ping flooding denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]
* @param    param       @b{(input)}  Param value

*
* @returns  none
*
* @end
*********************************************************************/
void doSGlobalPingFloodingModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode, uint32_t param)
{
  open_error_t result;

  if ((result = openapiDoSGlobalPingFloodingModeSet(client_handle, mode, param)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets ping flooding denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("Ping flooding denial of service protection is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets ping flooding denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]
* @param    pParam       @b{(output)}  Param value

*
* @returns  none
*
* @end
*********************************************************************/
void doSGlobalPingFloodingModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode, uint32_t *pParam)
{
  open_error_t result;

  if ((result = openapiDoSGlobalPingFloodingModeGet(client_handle, pMode, pParam)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets ping flooding denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("Ping flooding denial of service protection is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets preventing Smurf denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSGlobalSmurfAttackModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSGlobalSmurfAttackModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets preventing Smurf denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("Preventing Smurf denial of service protection is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets Smurf denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSGlobalSmurfAttackModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSGlobalSmurfAttackModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets Smurf denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("Preventing Smurf denial of service protection is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets preventing SynAck Flooding denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]
* @param    param       @b{(input)}  Param value

*
* @returns  none
*
* @end
*********************************************************************/
void doSGlobalSynAckFloodingModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode, uint32_t param)
{
  open_error_t result;

  if ((result = openapiDoSGlobalSynAckFloodingModeSet(client_handle, mode, param)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets preventing SynAck Flooding denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("Preventing SynAck Flooding denial of service protection is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets preventing SynAck Flooding denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]
* @param    pParam       @b{(output)}  Param value

*
* @returns  none
*
* @end
*********************************************************************/
void doSGlobalSynAckFloodingModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode, uint32_t *pParam)
{
  open_error_t result;

  if ((result = openapiDoSGlobalSynAckFloodingModeGet(client_handle, pMode, pParam)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets preventing SynAck Flooding denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("Preventing SynAck Flooding denial of service protection is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets denial of service protection Maximum ICMPv4 payload size.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMaxICMPSize       @b{(output)}   Maximum ICMPv4 payload size

*
* @returns  none
*
* @end
*********************************************************************/
void doSICMPv4SizeGet(openapiClientHandle_t *client_handle, uint32_t *pMaxICMPSize)
{
  open_error_t result;

  if ((result = openapiDoSICMPv4SizeGet(client_handle, pMaxICMPSize)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets denial of service protection Maximum ICMPv4 payload size. (result = %d)\n", result);
  }
  else
  {
    printf("Denial of service protection Maximum ICMPv4 payload size is %d\n", *pMaxICMPSize);
  }
  return;
}


/*********************************************************************
* @purpose  Gets denial of service protection Maximum ICMPv6 payload size.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMaxICMPSize       @b{(output)}   Maximum ICMPv6 payload size

*
* @returns  none
*
* @end
*********************************************************************/
void doSICMPv6SizeGet(openapiClientHandle_t *client_handle, uint32_t *pMaxICMPSize)
{
  open_error_t result;

  if ((result = openapiDoSICMPv6SizeGet(client_handle, pMaxICMPSize)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets denial of service protection Maximum ICMPv6 payload size. (result = %d)\n", result);
  }
  else
  {
    printf("Denial of service protection Maximum ICMPv6 payload size is %d\n", *pMaxICMPSize);
  }
  return;
}


/*********************************************************************
* @purpose  Gets ICMP Fragment denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSICMPFragModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSICMPFragModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets ICMP Fragment denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("ICMP Fragment denial of service protection mode is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets ICMPv6 denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSICMPv6ModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSICMPv6ModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets ICMPv6 denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("ICMPv6 denial of service protection mode is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets ICMPv6 denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSICMPv6ModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSICMPv6ModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets ICMPv6 denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("ICMPv6 denial of service protection mode is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets port D-disable state for denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSPortDdisableModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSPortDdisableModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets port D-disable state for denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("Port D-disable state for denial of service protection is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets port D-disable for denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSPortDdisableModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSPortDdisableModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets port D-disable for denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("Port D-disable state for denial of service protection is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets Auto denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSAutoModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSAutoModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets Auto denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("Auto denial of service protection mode is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets Auto denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSAutoModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSAutoModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets Auto denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("Auto denial of service protection mode is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets Vlan0 Tagged and SNAP Drop denial of service protection mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSVlan0TaggedSnapDropModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDoSVlan0TaggedSnapDropModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets Vlan0 Tagged and SNAP Drop denial of service protection mode. (result = %d)\n", result);
  }
  else
  {
    printf("Vlan0 Tagged and SNAP Drop denial of service protection mode is in %s state\n",
            (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets Vlan0 Tagged and SNAP Drop denial of service protection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void doSVlan0TaggedSnapDropModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDoSVlan0TaggedSnapDropModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets Vlan0 Tagged and SNAP Drop denial of service protection. (result = %d)\n", result);
  }
  else
  {
    printf("Vlan0 Tagged and SNAP Drop denial of service protection mode is successfully %s\n",
            (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for DoS
*
* @returns   0: Success
* @returns  -1: Failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  OPEN_CONTROL_t mode;
  uint32_t recvMode;
  uint32_t minTCPHdrSize;
  uint32_t maxICMPSize;
  uint32_t param;


  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("DoS example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting DoS API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSSIPDIPModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSSMACDMACModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSFirstFragModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 3)
      {
        doSFirstFragSizeSet(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSTCPFragModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSTCPOffsetModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSTCPFlagModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSTCPFlagSeqModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSTCPPortModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSUDPPortModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSTCPSynModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSTCPSynFinModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSTCPFinUrgPshModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSL4PortModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSICMPModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 3)
      {
        doSICMPSizeSet(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 17:
      if (argc == 3)
      {
        doSICMPv6SizeSet(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 18:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSICMPFragModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 19:
      if (argc == 2)
      {
        doSSIPDIPModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 20:
      if (argc == 2)
      {
        doSSMACDMACModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 21:
      if (argc == 2)
      {
        doSFirstFragModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 22:
      if (argc == 2)
      {
        doSTCPPartialModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 23:
      if (argc == 2)
      {
        doSFirstFragSizeGet(&clientHandle, &minTCPHdrSize);
        show_help = 0;
      }
      break;
    case 24:
      if (argc == 2)
      {
        doSTCPOffsetModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 25:
      if (argc == 2)
      {
        doSTCPPortModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 26:
      if (argc == 2)
      {
        doSUDPPortModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 27:
      if (argc == 2)
      {
        doSTCPFragModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 28:
      if (argc == 2)
      {
        doSTCPFlagModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 29:
      if (argc == 2)
      {
        doSTCPFlagSeqModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 30:
      if (argc == 2)
      {
        doSTCPSynModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 31:
      if (argc == 2)
      {
        doSTCPSynFinModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 32:
      if (argc == 2)
      {
        doSTCPFinUrgPshModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 33:
      if (argc == 2)
      {
        doSL4PortModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 34:
      if (argc == 2)
      {
        doSICMPModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 35:
      if (argc == 2)
      {
        doSICMPSizeGet(&clientHandle, &maxICMPSize);
        show_help = 0;
      }
      break;
    case 36:
      if (argc == 5)
      {
        recvMode = atoi(argv[3]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSIntfPingFloodingModeSet(&clientHandle, atoi(argv[2]), mode, atoi(argv[4]));
        }
        show_help = 0;
      }
      break;
    case 37:
      if (argc == 3)
      {
        doSIntfPingFloodingModeGet(&clientHandle, atoi(argv[2]), &mode, &param);
        show_help = 0;
      }
      break;
    case 38:
      if (argc == 4)
      {
        recvMode = atoi(argv[3]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSIntfSmurfAttackModeSet(&clientHandle, atoi(argv[2]), mode);
        }
        show_help = 0;
      }
      break;
    case 39:
      if (argc == 3)
      {
        doSIntfSmurfAttackModeGet(&clientHandle, atoi(argv[2]), &mode);
        show_help = 0;
      }
      break;
    case 40:
      if (argc == 3)
      {
        doSIntfSynAckFloodingModeGet(&clientHandle, atoi(argv[2]), &mode, &param);
        show_help = 0;
      }
      break;
    case 41:
      if (argc == 4)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSGlobalPingFloodingModeSet(&clientHandle, mode, atoi(argv[3]));
        }
        show_help = 0;
      }
      break;
    case 42:
      if (argc == 2)
      {
        doSGlobalPingFloodingModeGet(&clientHandle, &mode, &param);
        show_help = 0;
      }
      break;
    case 43:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSGlobalSmurfAttackModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 44:
      if (argc == 2)
      {
        doSGlobalSmurfAttackModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 45:
      if (argc == 4)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSGlobalSynAckFloodingModeSet(&clientHandle, mode, atoi(argv[3]));
        }
        show_help = 0;
      }
      break;
    case 46:
      if (argc == 2)
      {
        doSGlobalSynAckFloodingModeGet(&clientHandle, &mode, &param);
        show_help = 0;
      }
      break;
    case 47:
      if (argc == 2)
      {
        doSICMPv4SizeGet(&clientHandle, &maxICMPSize);
        show_help = 0;
      }
      break;
    case 48:
      if (argc == 2)
      {
        doSICMPv6SizeGet(&clientHandle, &maxICMPSize);
        show_help = 0;
      }
      break;
    case 49:
      if (argc == 2)
      {
        doSICMPFragModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 50:
      if (argc == 2)
      {
        doSICMPv6ModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 51:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSICMPv6ModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 52:
      if (argc == 2)
      {
        doSPortDdisableModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 53:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSPortDdisableModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 54:
      if (argc == 2)
      {
        doSAutoModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 55:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSAutoModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 56:
      if (argc == 2)
      {
        doSVlan0TaggedSnapDropModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 57:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          doSVlan0TaggedSnapDropModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 58:
      if (argc == 2)
      {
        runSanity(&clientHandle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping DoS API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

