/*********************************************************************
 * *
 * *  Copyright 2023 Broadcom.
 * *
 * *  Licensed under the Apache License, Version 2.0 (the "License");
 * *  you may not use this file except in compliance with the License.
 * *  You may obtain a copy of the License at
 * *
 * *      http://www.apache.org/licenses/LICENSE-2.0
 * *
 * *  Unless required by applicable law or agreed to in writing, software
 * *  distributed under the License is distributed on an "AS IS" BASIS,
 * *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * *  See the License for the specific language governing permissions and
 * *  limitations under the License.
 * *
 * **********************************************************************
 * *
 * * @filename  dot1x_eap_stats_example.c
 * *
 * * @purpose   OpEN dot1x_eap_stats example.
 * *
 * * @component OpEN
 * *
 * * @create    02/07/2023
 * *
 * * @end
 * *
 * **********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dot1x.h"

/*
 *    OpEN API set functions are processed asynchronously. There may be some
 *    delay between when the set function call returns and when the system
 *    state is updated to reflect the change. These parameters control how
 *    long the test code retries the get functions to retrieve a change.
 *
*/

/*******************************************************************
 * *
 * * @brief  This function prints the dot1x_eap_stats Example Application Menu.
 * *
 * * @param  name   @b{(input)} program name
 * *
 * * @returns  none
 * *
 * * @end
 * *********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Get the dot1x port Eapol Frames Received Stats: %s 1 <intIfNum>\n", name);
  printf("Test 2: Get the dot1x port Eapol Frames Transmitted Stats: %s 2 <intIfNum>\n", name);
  printf("Test 3: Get the dot1x port Eapol Start Frames Received Stats: %s 3 <intIfNum>\n", name);
  printf("Test 4: Get the dot1x port Eapol Logoff Frames Received Stat: %s 4 <intIfNum>\n", name);
  printf("Test 5: Get number of EAP Response/Identity frames received: %s 5 <intIfNum>\n", name);
  printf("Test 6: Get the dot1x port Eap Response Frames Received Stat: %s 6 <intIfNum>\n", name);
  printf("Test 7: Get the dot1x port Eap Request Id Frames Transmitted Stat: %s 7 <intIfNum>\n", name);
  printf("Test 8: Get the dot1x port Eap Request Frames Transmitted Stat: %s 8 <intIfNum>\n", name);
  printf("Test 9: Get the dot1x port Invalid Eapol Frames Received Stat: %s 9 <intIfNum>\n", name);
  printf("Test 10: Get the dot1x port Eap Length error Frames Received Stat: %s 10 <intIfNum>\n", name);
  printf("Test 11: Get the dot1x port Last Eapol Frame Version Stat: %s 11 <intIfNum>\n", name);
  printf("Test 12: Get the dot1x port Last Eapol Frame Source Stat: %s 12 <intIfNum>\n", name);
  printf("Test 13: Clear the dot1x stats on the specified port: %s 13 <intIfNum>\n", name);
  printf("Test 14: Run API sanity checks: %s 14 \n", name);

  return;
}

/***************************************************************************
 * * @purpose  Display results when incorrect inputs are passed to API.
 * *
 * * @param    clientHandle    @b{(input)}   client handle from registration API
 * *
 * * @returns  none
 * *
 * * @end
 * ****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  uint32_t eapolLogoffFramesRx;
  uint32_t eapolFramesRx;
  uint32_t eapolFramesTx;
  uint32_t lastEapolFrameVersion;
  uint32_t eapReqFramesTx;
  uint32_t eapRespIdFramesRx;
  uint32_t eapolStartFramesRx;
  uint32_t eapResponseFramesRx;
  uint32_t invalidEapolFramesRx;
  uint32_t eapLengthErrorFramesRx;
  uint32_t intIfNum = 1;
  uint32_t eapReqIdFramesTx;
  open_buffdesc lastEapolFrameSource;
  char buffer[128];
  open_buffdesc zeroLenBuf;
  open_buffdesc badBufdescPointer;

  badBufdescPointer.pstart = (void *) NULL;
  badBufdescPointer.size = sizeof(buffer);
  badBufdescPointer.size = 0;
  zeroLenBuf.pstart = buffer;
  zeroLenBuf.size = 0;

  printf("Testing dot1x_eap_stats OpEN APIs sanity:\n");

  printf("Testing openapiDot1xPortEapolFramesRxGet():\n");
  result = openapiDot1xPortEapolFramesRxGet(NULL, intIfNum, &eapolFramesRx);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortEapolFramesRxGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1xPortEapolFramesTxGet():\n");
  result = openapiDot1xPortEapolFramesTxGet(NULL, intIfNum, &eapolFramesTx);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortEapolFramesTxGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1xPortEapolStartFramesRxGet():\n");
  result = openapiDot1xPortEapolStartFramesRxGet(NULL, intIfNum, &eapolStartFramesRx);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortEapolStartFramesRxGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1xPortEapolLogoffFramesRxGet():\n");
  result = openapiDot1xPortEapolLogoffFramesRxGet(NULL, intIfNum, &eapolLogoffFramesRx);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortEapolLogoffFramesRxGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1xPortEapRespIdFramesRxGet():\n");
  result = openapiDot1xPortEapRespIdFramesRxGet(NULL, intIfNum, &eapRespIdFramesRx);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortEapRespIdFramesRxGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1xPortEapResponseFramesRxGet():\n");
  result = openapiDot1xPortEapResponseFramesRxGet(NULL, intIfNum, &eapResponseFramesRx);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortEapResponseFramesRxGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1xPortEapReqIdFramesTxGet():\n");
  result = openapiDot1xPortEapReqIdFramesTxGet(NULL, intIfNum, &eapReqIdFramesTx);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortEapReqIdFramesTxGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1xPortEapReqFramesTxGet():\n");
  result = openapiDot1xPortEapReqFramesTxGet(NULL, intIfNum, &eapReqFramesTx);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortEapReqFramesTxGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1xPortInvalidEapolFramesRxGet():\n");
  result = openapiDot1xPortInvalidEapolFramesRxGet(NULL, intIfNum, &invalidEapolFramesRx);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortInvalidEapolFramesRxGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1xPortEapLengthErrorFramesRxGet():\n");
  result = openapiDot1xPortEapLengthErrorFramesRxGet(NULL, intIfNum, &eapLengthErrorFramesRx);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortEapLengthErrorFramesRxGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1xPortLastEapolFrameVersionGet():\n");
  result = openapiDot1xPortLastEapolFrameVersionGet(NULL, intIfNum, &lastEapolFrameVersion);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortLastEapolFrameVersionGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1xPortLastEapolFrameSourceGet():\n");
  result = openapiDot1xPortLastEapolFrameSourceGet(NULL, intIfNum, &lastEapolFrameSource);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortLastEapolFrameSourceGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiDot1xPortLastEapolFrameSourceGet(clientHandle, intIfNum, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiDot1xPortLastEapolFrameSourceGet(clientHandle, intIfNum, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);

  return;
}


/*********************************************************************
 * * @purpose  Get the dot1x port Eapol Frames Received Stats
 * *
 * * @param    clientHandle       @b{(input)}  client handle from registration API
 * * @param    intIfNum       @b{(input)}  internal interface number
 * * @param    eapolFramesRx       @b{(output)}  EAPOL start frames received
 *
 * *
 * * @returns  none
 * *
 * * @end
 * *********************************************************************/
void dot1xPortEapolFramesRxGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t *eapolFramesRx)
{
  open_error_t result;

  if ((result = openapiDot1xPortEapolFramesRxGet(clientHandle, intIfNum, eapolFramesRx)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the dot1x port Eapol Frames Received Stats (result = %d)\n", result);
  }
  else
  {
    printf("EAPOL frames received :- %u\n", *eapolFramesRx);
  }
  return;
}


/*********************************************************************
 * * @purpose  Get the dot1x port Eapol Frames Transmitted Stats
 * *
 * * @param    clientHandle       @b{(input)}  client handle from registration API
 * * @param    intIfNum       @b{(input)}  internal interface number
 * * @param    eapolFramesTx       @b{(output)}  EAPOL frames transmitted
 *
 * *
 * * @returns  none
 * *
 * * @end
 * *********************************************************************/
void dot1xPortEapolFramesTxGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t *eapolFramesTx)
{
  open_error_t result;

  if ((result = openapiDot1xPortEapolFramesTxGet(clientHandle, intIfNum, eapolFramesTx)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the dot1x port Eapol Frames Transmitted Stats (result = %d)\n", result);
  }
  else
  {
    printf("EAPOL frames transmitted :- %u\n", *eapolFramesTx);
  }
  return;
}


/*********************************************************************
 * * @purpose  Get the dot1x port Eapol Start Frames Received Stats
 * *
 * * @param    clientHandle       @b{(input)}  client handle from registration API
 * * @param    intIfNum       @b{(input)}  internal interface number
 * * @param    eapolStartFramesRx       @b{(output)}  EAPOL start frames received
 *
 * *
 * * @returns  none
 * *
 * * @end
 * *********************************************************************/
void dot1xPortEapolStartFramesRxGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t *eapolStartFramesRx)
{
  open_error_t result;

  if ((result = openapiDot1xPortEapolStartFramesRxGet(clientHandle, intIfNum, eapolStartFramesRx)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the dot1x port Eapol Start Frames Received Stats (result = %d)\n", result);
  }
  else
  {
    printf("EAPOL start frames received :- %u\n", *eapolStartFramesRx);
  }
  return;
}


/*********************************************************************
 * * @purpose  Get the dot1x port Eapol Logoff Frames Received Stat
 * *
 * * @param    clientHandle       @b{(input)}  client handle from registration API
 * * @param    intIfNum       @b{(input)}  internal interface number
 * * @param    eapolLogoffFramesRx       @b{(output)}  EAPOL logoff frames received
 *
 * *
 * * @returns  none
 * *
 * * @end
 * *********************************************************************/
void dot1xPortEapolLogoffFramesRxGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t *eapolLogoffFramesRx)
{
  open_error_t result;

  if ((result = openapiDot1xPortEapolLogoffFramesRxGet(clientHandle, intIfNum, eapolLogoffFramesRx)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the dot1x port Eapol Logoff Frames Received Stat (result = %d)\n", result);
  }
  else
  {
    printf("EAPOL logoff frames received :- %u\n", *eapolLogoffFramesRx);
  }
  return;
}


/*********************************************************************
 * * @purpose  Get number of EAP Response/Identity frames received
 * *
 * * @param    clientHandle       @b{(input)}  client handle from registration API
 * * @param    intIfNum       @b{(input)}  internal interface number
 * * @param    eapRespIdFramesRx       @b{(output)}  EAP Resp/Id frames received
 *
 * *
 * * @returns  none
 * *
 * * @end
 * *********************************************************************/
void dot1xPortEapRespIdFramesRxGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t *eapRespIdFramesRx)
{
  open_error_t result;

  if ((result = openapiDot1xPortEapRespIdFramesRxGet(clientHandle, intIfNum, eapRespIdFramesRx)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get number of EAP Response/Identity frames received (result = %d)\n", result);
  }
  else
  {
    printf("EAP Resp/Id frames received :- %u\n", *eapRespIdFramesRx);
  }
  return;
}


/*********************************************************************
 * * @purpose  Get the dot1x port Eap Response Frames Received Stat
 * *
 * * @param    clientHandle       @b{(input)}  client handle from registration API
 * * @param    intIfNum       @b{(input)}  internal interface number
 * * @param    eapResponseFramesRx       @b{(output)}  EAP Response frames received
 *
 * *
 * * @returns  none
 * *
 * * @end
 * *********************************************************************/
void dot1xPortEapResponseFramesRxGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t *eapResponseFramesRx)
{
  open_error_t result;

  if ((result = openapiDot1xPortEapResponseFramesRxGet(clientHandle, intIfNum, eapResponseFramesRx)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the dot1x port Eap Response Frames Received Stat (result = %d)\n", result);
  }
  else
  {
    printf("EAP Response frames received :- %u\n", *eapResponseFramesRx);
  }
  return;
}


/*********************************************************************
 * * @purpose  Get the dot1x port Eap Request Id Frames Transmitted Stat
 * *
 * * @param    clientHandle       @b{(input)}  client handle from registration API
 * * @param    intIfNum       @b{(input)}  internal interface number
 * * @param    eapReqIdFramesTx       @b{(output)}  EAP Req/Id frames transmitted
 *
 * *
 * * @returns  none
 * *
 * * @end
 * *********************************************************************/
void dot1xPortEapReqIdFramesTxGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t *eapReqIdFramesTx)
{
  open_error_t result;

  if ((result = openapiDot1xPortEapReqIdFramesTxGet(clientHandle, intIfNum, eapReqIdFramesTx)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the dot1x port Eap Request Id Frames Transmitted Stat (result = %d)\n", result);
  }
  else
  {
    printf("EAP Req/Id frames transmitted :- %u\n", *eapReqIdFramesTx);
  }
  return;
}


/*********************************************************************
 * * @purpose  Get the dot1x port Eap Request Frames Transmitted Stat
 * *
 * * @param    clientHandle       @b{(input)}  client handle from registration API
 * * @param    intIfNum       @b{(input)}  internal interface number
 * * @param    eapReqFramesTx       @b{(output)}  EAP Request frames transmitted
 *
 * *
 * * @returns  none
 * *
 * * @end
 * *********************************************************************/
void dot1xPortEapReqFramesTxGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t *eapReqFramesTx)
{
  open_error_t result;

  if ((result = openapiDot1xPortEapReqFramesTxGet(clientHandle, intIfNum, eapReqFramesTx)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the dot1x port Eap Request Frames Transmitted Stat (result = %d)\n", result);
  }
  else
  {
    printf("EAP Request frames transmitted :- %u\n", *eapReqFramesTx);
  }
  return;
}


/*********************************************************************
 * * @purpose  Get the dot1x port Invalid Eapol Frames Received Stat
 * *
 * * @param    clientHandle       @b{(input)}  client handle from registration API
 * * @param    intIfNum       @b{(input)}  internal interface number
 * * @param    invalidEapolFramesRx       @b{(output)}  Invalid EAPOL frames received
 *
 * *
 * * @returns  none
 * *
 * * @end
 * *********************************************************************/
void dot1xPortInvalidEapolFramesRxGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t *invalidEapolFramesRx)
{
  open_error_t result;

  if ((result = openapiDot1xPortInvalidEapolFramesRxGet(clientHandle, intIfNum, invalidEapolFramesRx)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the dot1x port Invalid Eapol Frames Received Stat (result = %d)\n", result);
  }
  else
  {
    printf("Invalid EAPOL frames received :- %u\n", *invalidEapolFramesRx);
  }
  return;
}


/*********************************************************************
 * * @purpose  Get the dot1x port Eap Length error Frames Received Stat
 * *
 * * @param    clientHandle       @b{(input)}  client handle from registration API
 * * @param    intIfNum       @b{(input)}  internal interface number
 * * @param    eapLengthErrorFramesRx       @b{(output)}  EAP length error frames
 *
 * *
 * * @returns  none
 * *
 * * @end
 * *********************************************************************/
void dot1xPortEapLengthErrorFramesRxGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t *eapLengthErrorFramesRx)
{
  open_error_t result;

  if ((result = openapiDot1xPortEapLengthErrorFramesRxGet(clientHandle, intIfNum, eapLengthErrorFramesRx)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the dot1x port Eap Length error Frames Received Stat (result = %d)\n", result);
  }
  else
  {
    printf("EAP length error frames :- %u\n", *eapLengthErrorFramesRx);
  }
  return;
}


/*********************************************************************
 * * @purpose  Get the dot1x port Last Eapol Frame Version Stat
 * *
 * * @param    clientHandle       @b{(input)}  client handle from registration API
 * * @param    intIfNum       @b{(input)}  internal interface number
 * * @param    lastEapolFrameVersion       @b{(output)}  last EAPOL version received
 *
 * *
 * * @returns  none
 * *
 * * @end
 * *********************************************************************/
void dot1xPortLastEapolFrameVersionGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t *lastEapolFrameVersion)
{
  open_error_t result;

  if ((result = openapiDot1xPortLastEapolFrameVersionGet(clientHandle, intIfNum, lastEapolFrameVersion)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the dot1x port Last Eapol Frame Version Stat (result = %d)\n", result);
  }
  else
  {
    printf("last EAPOL version received :- %u\n", *lastEapolFrameVersion);
  }
  return;
}


/*********************************************************************
 * * @purpose  Get the dot1x port Last Eapol Frame Source Stat
 * *
 * * @param    clientHandle       @b{(input)}  client handle from registration API
 * * @param    intIfNum       @b{(input)}  internal interface number
 * * @param    lastEapolFrameSource       @b{(output)}  last EAPOL version received
 *
 * *
 * * @returns  none
 * *
 * * @end
 * *********************************************************************/
void dot1xPortLastEapolFrameSourceGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  open_buffdesc lastEapolFrameSource;
  unsigned char macAddrBuf[OPEN_MAC_ADDR_LEN];
  memset(macAddrBuf, 0, sizeof(macAddrBuf));
  lastEapolFrameSource.pstart = macAddrBuf;
  lastEapolFrameSource.size = sizeof(macAddrBuf);

  if ((result = openapiDot1xPortLastEapolFrameSourceGet(clientHandle, intIfNum, &lastEapolFrameSource)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the dot1x port Last Eapol Frame Source Stat (result = %d)\n", result);
  }
  else
  {
    printf("last EAPOL version received :- %02X:%02X:%02X:%02X:%02X:%02X\n",
            macAddrBuf[0], macAddrBuf[1], macAddrBuf[2], macAddrBuf[3],
            macAddrBuf[4], macAddrBuf[5]);
  }
  return;
}


/*********************************************************************
 * * @purpose  Clear the dot1x stats on the specified port
 * *
 * * @param    clientHandle       @b{(input)}  client handle from registration API
 * * @param    intIfNum       @b{(input)}  internal interface number
 *
 * *
 * * @returns  none
 * *
 * * @end
 * *********************************************************************/
void dot1xPortStatsClear(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;

  if ((result = openapiDot1xPortStatsClear(clientHandle, intIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear the dot1x stats on the specified port (result = %d)\n", result);
  }
  else
  {
    printf("Dot1x stats cleared for interface %u\n", intIfNum);
  }
  return;
}



/*******************************************************************
 * *
 * * @brief  This is the main() function of the example application that
 * *         demonstrates OpEN APIs for dot1x_eap_stats
 * *
 * * @returns   0: Success
 * * @returns  -1: Failure 
 * *
 * *********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  uint32_t eapolLogoffFramesRx;
  uint32_t eapolFramesRx;
  uint32_t eapolFramesTx;
  uint32_t lastEapolFrameVersion;
  uint32_t eapReqFramesTx;
  uint32_t eapRespIdFramesRx;
  uint32_t eapolStartFramesRx;
  uint32_t eapResponseFramesRx;
  uint32_t invalidEapolFramesRx;
  uint32_t eapLengthErrorFramesRx;
  uint32_t eapReqIdFramesTx;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("dot1x_eap_stats example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting dot1x_eap_stats API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        dot1xPortEapolFramesRxGet(&clientHandle, atoi(argv[2]), &eapolFramesRx);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 3)
      {
        dot1xPortEapolFramesTxGet(&clientHandle, atoi(argv[2]), &eapolFramesTx);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 3)
      {
        dot1xPortEapolStartFramesRxGet(&clientHandle, atoi(argv[2]), &eapolStartFramesRx);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 3)
      {
        dot1xPortEapolLogoffFramesRxGet(&clientHandle, atoi(argv[2]), &eapolLogoffFramesRx);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 3)
      {
        dot1xPortEapRespIdFramesRxGet(&clientHandle, atoi(argv[2]), &eapRespIdFramesRx);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 3)
      {
        dot1xPortEapResponseFramesRxGet(&clientHandle, atoi(argv[2]), &eapResponseFramesRx);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 3)
      {
        dot1xPortEapReqIdFramesTxGet(&clientHandle, atoi(argv[2]), &eapReqIdFramesTx);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 3)
      {
        dot1xPortEapReqFramesTxGet(&clientHandle, atoi(argv[2]), &eapReqFramesTx);
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 3)
      {
        dot1xPortInvalidEapolFramesRxGet(&clientHandle, atoi(argv[2]), &invalidEapolFramesRx);
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 3)
      {
        dot1xPortEapLengthErrorFramesRxGet(&clientHandle, atoi(argv[2]), &eapLengthErrorFramesRx);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 3)
      {
        dot1xPortLastEapolFrameVersionGet(&clientHandle, atoi(argv[2]), &lastEapolFrameVersion);
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 3)
      {
        dot1xPortLastEapolFrameSourceGet(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 3)
      {
        dot1xPortStatsClear(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 2)
      {
        runSanity(&clientHandle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping dot1x_eap_stats API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

