/*********************************************************************
*
* Copyright 2018-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  dai_example.c
*
* @purpose   Dynamic ARP Inspection (DAI) OpEN APIs Example 
*
* @component OpEN
*
* @comments
*
* @create    07/09/2018
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dynamic_arp_inspection.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the DAI 
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/
/*******************************************************************
*
* @brief  This function prints the DAI Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printDaiAppMenu()
{
  printf("Usage:  dai_example <test#> <arg1> ... \n");
  printf("Test 1: Set global source MAC address validation mode: dai_example 1 <0|1> \n");
  printf("Test 2: Get global source MAC address validation mode: dai_example 2 \n");
  printf("Test 3: Set global destination MAC address validation mode: dai_example 3 <0|1> \n");
  printf("Test 4: Get global destination MAC address validation mode: dai_example 4 \n");
  printf("Test 5: Set global IP address validation mode: dai_example 5 <0|1> \n");
  printf("Test 6: Get global IP address validation mode: dai_example 6 \n");
  printf("Test 7: Get API for whether DAI considers a port trusted: dai_example 7 <ifNum>\n");
  printf("Test 8: Set API for whether DAI considers a port trusted: dai_example 8 <ifNum> <trustMode>\n");
  printf("Test 9: Get rate limit value of incoming ARP packets on untrusted interfaces: dai_example 9 <ifNum>\n");
  printf("Test 10: Set rate limit value of incoming ARP packets on untrusted interfaces: dai_example 10 <ifNum> <rate>\n");
  printf("Test 11: Get burst interval value of incoming ARP packets on untrusted interfaces: dai_example 11 <ifNum> <burstInterval>\n");
  printf("Test 12: Set burst interval value of incoming ARP packets on untrusted interfaces: dai_example 12 <ifNum> <burstInterval>\n");
  printf("Test 13: Get DAI statistics on a VLAN: dai_example 13 <vlanId>\n");
  printf("Test 14: Get the next vlan id eligible for DAI configuration, given the current vlan id: dai_example 14 <vlanId>\n");
  printf("Test 15: Get API for whether Dynamic ARP Inspection is enabled or disabled on a VLAN: dai_example 15 <vlanId>\n");
  printf("Test 16: Set API to enable or disable Dynamic ARP Inspection on a VLAN: dai_example 16 <vlanId><val>\n");
  printf("Test 17: Get API for whether DAI logging is enabled or disabled on a VLAN: dai_example 17 <vlanId>\n");
  printf("Test 18: Set API to enable or disable DAI logging for a VLAN: dai_example 18 <vlanId><val>\n");
  printf("Test 19: Get the ARP ACL static flag for a vlan: dai_example 19 <vlanId>\n");
  printf("Test 20: Set the ARP ACL static flag for a vlan: dai_example 20 <vlanId><staticFlag>\n");
  printf("Test 21: Clear DAI statistics on all vlans: dai_example 21 \n");
  printf("Test 22: API to add rule to ARP ACL: dai_example 22 <aclName><seqNo><ipAddr><macAddr><action>\n");
  printf("Test 23: API to remove rule from ARP ACL: dai_example 23 <aclName><seqNo>\n");
  printf("Test 24: Get the ARP ACL Name for a vlan (Extended Version): dai_example 24 <vlanId>\n");
  printf("Test 25: Apply the ARP ACL to a range of VLANs: dai_example 25 <aclName><vlanStart><vlanEnd><val><staticFlag>\n");
  printf("Test 26: Set the ARP ACL Name for a vlan (Extended Version): dai_example 26 <vlanId><aclName>\n");
  printf("Test 27: API to create ARP ACL (extensible version): dai_example 27 <aclName>\n");
  printf("Test 28: API to delete ARP ACL (extensible version): dai_example 28 <aclName>\n");

  printf("Test 29: DAI OpEN APIs Sanity: dai_example 29 \n");
  return;
}

/*********************************************************************
* @purpose  Set the global DAI source MAC validate mode.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API.
* @param    mode             @b{(input)}   Source MAC validate mode.
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch.
* 
* @end
*********************************************************************/
void daiSourceMacValidateModeSet(openapiClientHandle_t *clientHandle,
                                 OPEN_CONTROL_t        mode)
{
  open_error_t result;

  if ((result = openapiDaiSourceMacValidateModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set global DAI source MAC validate mode. (result = %d)\n", result);
  }
  else
  {
    printf("DAI source MAC validate mode is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the global DAI source MAC validate mode.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API.
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void daiSourceMacValidateModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t   result;
  OPEN_CONTROL_t mode;

  if ((result = openapiDaiSourceMacValidateModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get global source MAC validate mode. (result = %d)\n", result);
  }
  else
  {
    printf("DAI global source MAC validate mode: %d (result = %d)\n", mode, result);
  }

  return; 
}

/*********************************************************************
* @purpose  Set the global DAI destination MAC validate mode.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API.
* @param    mode             @b{(input)}   Destination MAC validate mode.
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch.
* 
* @end
*********************************************************************/
void daiDestMacValidateModeSet(openapiClientHandle_t *clientHandle,
                               OPEN_CONTROL_t        mode)
{
  open_error_t result;

  if ((result = openapiDaiDestMacValidateModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set global DAI destination MAC validate mode. (result = %d)\n", result);
  }
  else
  {
    printf("DAI destination MAC validate mode is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the global DAI destination MAC validate mode.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API.
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void daiDestMacValidateModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t   result;
  OPEN_CONTROL_t mode;

  if ((result = openapiDaiDestMacValidateModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get global destination MAC validate mode. (result = %d)\n", result);
  }
  else
  {
    printf("DAI global destination MAC validate mode: %d (result = %d)\n", mode, result);
  }

  return; 
}

/*********************************************************************
* @purpose  Set the global DAI IP address validate mode.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API.
* @param    mode             @b{(input)}   IP address validate mode.
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch.
* 
* @end
*********************************************************************/
void daiIpAddrValidateModeSet(openapiClientHandle_t *clientHandle,
                              OPEN_CONTROL_t        mode)
{
  open_error_t result;

  if ((result = openapiDaiIpAddrValidateModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set global DAI IP address validate mode. (result = %d)\n", result);
  }
  else
  {
    printf("DAI IP address validate mode is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the global DAI IP address validate mode.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API.
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void daiIpAddrValidateModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t   result;
  OPEN_CONTROL_t mode;

  if ((result = openapiDaiIpAddrValidateModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get global IP address validate mode. (result = %d)\n", result);
  }
  else
  {
    printf("DAI global IP address validate mode: %d (result = %d)\n", mode, result);
  }

  return; 
}

/*********************************************************************
* @purpose  Get API for whether DAI considers a port trusted.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface
* @param    trustMode       @b{(output)} Port trust mode
*
* @returns  none
*
* @end
*********************************************************************/
void daiIntfTrustGet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                     OPEN_BOOL_t *trustMode)
{
  open_error_t result;

  if ((result = openapiDaiIntfTrustGet(client_handle, ifNum, trustMode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get API for whether DAI considers a port"
           " trusted. (result = %d)\n", result);
  }
  else
  {
    printf("Trust mode status is %u.\n", *trustMode);
  }
  return;
}

/*********************************************************************
* @purpose  Set API for whether DAI considers a port trusted.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum               @b{(input)}  Interface
* @param    trustMode           @b{(input)}  Port trust mode
*
* @returns  none
*
* @end
*********************************************************************/
void daiIntfTrustSet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                     OPEN_BOOL_t trustMode)
{
  open_error_t result;

  if ((result = openapiDaiIntfTrustSet(client_handle, ifNum, trustMode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set API for whether DAI considers a "
           "port trusted. (result = %d)\n", result);
  }
  else
  {
    printf("Port trust mode status set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get rate limit value of incoming ARP packets on untrusted interfaces.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum               @b{(input)}  Interface
* @param    rate                @b{(output)} Upper limit for number of incoming
*                                            packets processed per second
*
* @returns  none
*
* @end
*********************************************************************/
void daiIntfRateLimitGet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                         int32_t *rate)
{
  open_error_t result;

  if ((result = openapiDaiIntfRateLimitGet(client_handle, ifNum, rate)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get rate limit value of incoming ARP "
           "packets on untrusted interfaces. (result = %d)\n", result);
  }
  else
  {
    printf("Rate limit value of incoming ARP packets is %u.\n", *rate);
  }
  return;
}

/*********************************************************************
* @purpose  Set rate limit value of incoming ARP packets on untrusted interfaces.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum               @b{(input)}  Interface
* @param    rate                @b{(input)}  Upper limit for number of incoming 
*                                            packets processed per second
*
* @returns  none
*
* @end
*********************************************************************/
void daiIntfRateLimitSet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                         int32_t rate)
{
  open_error_t result;

  if ((result = openapiDaiIntfRateLimitSet(client_handle, ifNum, rate)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set rate limit value of incoming ARP packets"
           " on untrusted interfaces. (result = %d)\n", result);
  }
  else
  {
    printf("Rate limit value of incoming ARP packets set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get burst interval value of incoming ARP packets on untrusted interfaces.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum               @b{(input)}  Interface
* @param    burstInterval       @b{(output)} Consecutive interval in seconds, 
*                                            over which the interface is monitored
*                                            for a high rate of ARP packets
*
* @returns  none
*
* @end
*********************************************************************/
void daiIntfBurstIntervalGet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                             int32_t *burstInterval)
{
  open_error_t result;

  if ((result = openapiDaiIntfBurstIntervalGet(client_handle, ifNum, burstInterval))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get burst interval value of incoming ARP "
           "packets on untrusted interfaces. (result = %d)\n", result);
  }
  else
  {
    printf("Burst interval value of incoming ARP packets is %u.\n", *burstInterval);
  }
  return;
}

/*********************************************************************
* @purpose  Set burst interval value of incoming ARP packets on untrusted interfaces.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum               @b{(input)}  Interface
* @param    burstInterval       @b{(input)}  Consecutive interval in seconds, 
*                                            over which the interface is monitored
*                                            for a high rate of ARP packets
*
* @returns  none
*
* @end
*********************************************************************/
void daiIntfBurstIntervalSet(openapiClientHandle_t *client_handle, uint32_t ifNum,
                             int32_t burstInterval)
{
  open_error_t result;

  if ((result = openapiDaiIntfBurstIntervalSet(client_handle, ifNum, 
                                               burstInterval)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set burst interval value of incoming ARP"
           " packets on untrusted interfaces. (result = %d)\n", result);
  }
  else
  {
    printf("Burst interval value of incoming ARP packets set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get DAI statistics on a VLAN.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    vlanId          @b{(input)}  VLAN Id
* @param    vlanStats       @b{(output)} Statistics for a given VLAN
*
* @returns  none
*
* @end
*********************************************************************/
void daiVlanStatsGet(openapiClientHandle_t *client_handle, uint32_t vlanId,
                     OPEN_DAI_VLAN_STATS_t *vlanStats)
{
  open_error_t result;

  if ((result = openapiDaiVlanStatsGet(client_handle, vlanId, vlanStats)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get DAI statistics on a VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("VLAN ID : %u\n", vlanId);  
    printf("Forwarded : %u\n", vlanStats->forwarded);
    printf("Dropped : %u\n", vlanStats->dropped);
    printf("DHCP Drops : %u\n", vlanStats->dhcpDrops);
    printf("DHCP Permits : %u\n", vlanStats->dhcpPermits);
    printf("ACL Drops : %u\n", vlanStats->aclDrops);
    printf("ACL permits : %u\n", vlanStats->aclPermits);
    printf("ACL Denials : %u\n", vlanStats->aclDenials);
    printf("Source MAC Failures : %u\n", vlanStats->sMacFailures);
    printf("Destination MAC Failures : %u\n", vlanStats->dMacFailures);
    printf("IP Validation Failures : %u\n", vlanStats->ipValidFailures);
  }
  return;
}

/*********************************************************************
* @purpose  Get the next vlan id eligible for DAI configuration, given 
*           the current vlan id.
*
* @param    client_handle  @b{(input)}  client handle from regn API
* @param    vlanId         @b{(input)}  VLAN Id
* @param    nextVlan       @b{(output)} next VLAN Id
*
* @returns  none
*
* @end
*********************************************************************/
void daiVlanNextGet(openapiClientHandle_t *client_handle, uint32_t vlanId, 
                    uint32_t *nextVlan)
{
  open_error_t result;

  if ((result = openapiDaiVlanNextGet(client_handle, vlanId, nextVlan)) != 
      OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next vlan id eligible for DAI " 
           "configuration, given the current vlan id. (result = %d)\n", result);
  }
  else
  {
    printf("Next VLAN eligible for DAI configuration given current vlan "
           "%d is %d\n", vlanId, *nextVlan);
  }
  return;
}

/*********************************************************************
* @purpose  Get API for whether Dynamic ARP Inspection is enabled or 
*           disabled on a VLAN.
*
* @param    client_handle  @b{(input)}  client handle from regn API
* @param    vlanId         @b{(input)}  VLAN Id
* @param    val            @b{(output)} L7_ENABLE if DAI is enabled 
*                                       on the VLAN
*
*
* @returns  none
*
* @end
*********************************************************************/
void daiVlanEnableGet(openapiClientHandle_t *client_handle, uint32_t vlanId, 
                      uint32_t *val)
{
  open_error_t result;

  if ((result = openapiDaiVlanEnableGet(client_handle, vlanId, val)) != 
      OPEN_E_NONE)
  {
    printf("Bad return code trying to get API for whether Dynamic ARP "
           "Inspection is enabled or disabled on a VLAN. (result = %d)\n", 
           result);
  }
  else
  {
    printf("DAI is %s on VLAN %d\n", (*val == OPEN_ENABLE)?"enabled":"disabled", 
           vlanId);
  }
  return;
}

/*********************************************************************
* @purpose  Set API to enable or disable Dynamic ARP Inspection on a 
*           VLAN.
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    vlanId        @b{(input)}  VLAN Id
* @param    val           @b{(input)}  L7_ENABLE or L7_DISABLE
*
* @returns  none
*
* @end
*********************************************************************/
void daiVlanEnableSet(openapiClientHandle_t *client_handle, uint32_t vlanId, 
                      uint32_t val)
{
  open_error_t result;

  if ((result = openapiDaiVlanEnableSet(client_handle, vlanId, val)) != 
      OPEN_E_NONE)
  {
    printf("Bad return code trying to set API to enable or disable Dynamic "
           "ARP Inspection on a VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("DAI is %s on VLAN %d successfully\n", (val == OPEN_ENABLE)?
           "enabled":"disabled", vlanId);
  }
  return;
}

/*********************************************************************
* @purpose  Get API for whether DAI logging is enabled or disabled on 
*           a VLAN.
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    vlanId        @b{(input)}  VLAN Id
* @param    val           @b{(output)} L7_ENABLE if DAI logging is 
*                                      enabled on the VLAN
*
* @returns  none
*
* @end
*********************************************************************/
void daiVlanLoggingEnableGet(openapiClientHandle_t *client_handle, 
                             uint32_t vlanId, uint32_t *val)
{
  open_error_t result;

  if ((result = openapiDaiVlanLoggingEnableGet(client_handle, vlanId, val)) != 
      OPEN_E_NONE)
  {
    printf("Bad return code trying to get API for whether DAI logging is enabled "
           "or disabled on a VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("DAI logging for VLAN %d is %s\n", vlanId, (*val == OPEN_ENABLE)?
           "enabled":"disabled");
  }
  return;
}

/*********************************************************************
* @purpose  Set API to enable or disable DAI logging for a VLAN.
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    vlanId        @b{(input)}  VLAN Id
* @param    val           @b{(input)}  L7_ENABLE or L7_DISABLE
*
*
* @returns  none
*
* @end
*********************************************************************/
void daiVlanLoggingEnableSet(openapiClientHandle_t *client_handle, 
                             uint32_t vlanId, uint32_t val)
{
  open_error_t result;

  if ((result = openapiDaiVlanLoggingEnableSet(client_handle, vlanId, val)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set API to enable or disable DAI "
           "logging for a VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("DAI logging for VLAN %d is %s successfully\n", vlanId, 
           (val == OPEN_ENABLE)? "enabled":"disabled");
  }
  return;
}

/*********************************************************************
* @purpose  Get the ARP ACL static flag for a vlan.
*
* @param    client_handle  @b{(input)}  client handle from regn API
* @param    vlanId         @b{(input)}  VLAN Id
* @param    staticFlag     @b{(output)} static flag for the VLAN

*
* @returns  none
*
* @end
*********************************************************************/
void daiVlanArpAclStaticFlagGet(openapiClientHandle_t *client_handle, 
                                uint32_t vlanId, uint32_t *staticFlag)
{
  open_error_t result;

  if ((result = openapiDaiVlanArpAclStaticFlagGet(client_handle, vlanId, 
      staticFlag)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the ARP ACL static flag for a vlan."
           " (result = %d)\n", result);
  }
  else
  {
    printf("DAI VLAN ARP ACL static flag for vlan %d is %s\n", vlanId, 
           (*staticFlag == OPEN_ENABLE)? "enabled":"disabled");
  }
  return;
}

/*********************************************************************
* @purpose  Set the ARP ACL static flag for a vlan.
*
* @param    client_handle  @b{(input)}  client handle from regn API
* @param    vlanId         @b{(input)}  VLAN Id
* @param    staticFlag     @b{(input)}  static flag for the VLAN

*
* @returns  none
*
* @end
*********************************************************************/
void daiVlanArpAclStaticFlagSet(openapiClientHandle_t *client_handle, 
                                uint32_t vlanId, uint32_t staticFlag)
{
  open_error_t result;

  if ((result = openapiDaiVlanArpAclStaticFlagSet(client_handle, vlanId, 
      staticFlag)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the ARP ACL static flag for a vlan. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("DAI VLAN ARP ACL static flag for VLAN %d is %s successfully\n", 
           vlanId, (staticFlag == OPEN_ENABLE)? "enabled":"disabled");
  }
  return;
}

/*********************************************************************
* @purpose  Clear DAI statistics on all vlans.
*
* @param    client_handle  @b{(input)}  client handle from regn API
*
* @returns  none
*
* @end
*********************************************************************/
void daiVlanStatsClear(openapiClientHandle_t *client_handle)
{
  open_error_t result;

  if ((result = openapiDaiVlanStatsClear(client_handle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear DAI statistics on all vlans. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("DAI VLAN statistics are cleared successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  API to add rule to ARP ACL.
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    bufd          @b{(input)}  buffer descriptor containing 
*                                      ACL Name
* @param    seqNo         @b{(input)}  Rule Sequence Number
* @param    ipAddr        @b{(input)}  Sender's IP address
* @param    macAddr       @b{(input)}  Sender's MAC address
* @param    action        @b{(input)}  PERMIT or DENY action to be 
*                                      performed
*
* @returns  none
*
* @end
*********************************************************************/
void arpAclRuleAdd(openapiClientHandle_t *client_handle, open_buffdesc *bufd, 
                   uint32_t seqNo, uint32_t ipAddr, open_buffdesc *macAddr, 
                   OPEN_ARP_ACL_RULE_ACTION_t action)
{
  open_error_t result;

  if ((result = openapiArpAclRuleAdd(client_handle, bufd, seqNo, ipAddr, 
      macAddr, action)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add rule to ARP ACL. (result = %d)\n", 
           result);
  }
  else
  {
    printf("ARP ACL rule addition is successful\n");
  }
  return;
}

/*********************************************************************
* @purpose  API to remove rule from ARP ACL.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    bufd          @b{(input)}  buffer descriptor containing ACL Name
* @param    seqNo         @b{(input)}  Rule Sequence Number
*
* @returns  none
*
* @end
*********************************************************************/
void arpAclRuleDelete(openapiClientHandle_t *client_handle, open_buffdesc *bufd, 
                      uint32_t seqNo)
{
  open_error_t result;

  if ((result = openapiArpAclRuleDelete(client_handle, bufd, seqNo)) != 
      OPEN_E_NONE)
  {
    printf("Bad return code trying to aPI to remove rule from ARP ACL. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("ARP ACL rule deletion is successful\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the ARP ACL Name for a vlan (Extended Version).
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    vlanId        @b{(input)}  VLAN Id
* @param    aclName       @b{(output)} ARP ACL name for this VLAN

*
* @returns  none
*
* @end
*********************************************************************/
void daiVlanArpAclGet(openapiClientHandle_t *client_handle, uint32_t vlanId, 
                      open_buffdesc *aclName)
{
  open_error_t result;

  if ((result = openapiDaiVlanArpAclGet(client_handle, vlanId, aclName)) != 
      OPEN_E_NONE)
  {
    printf("Bad return code trying to get the ARP ACL Name for a vlan "
           "(Extended Version). (result = %d)\n", result);
  }
  else
  {
    if (aclName && aclName->pstart)
    {
      printf("DAI ARP ACL name for VLAN %d is %s\n", vlanId, (char *)aclName->pstart);
    }
  }
  return;
}

/*********************************************************************
* @purpose  Apply the ARP ACL to a range of VLANs.
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    aclName       @b{(input)}  ACL name
* @param    vlanStart     @b{(input)}  Low end of VLAN range
* @param    vlanEnd       @b{(input)}  High end of VLAN range
* @param    val           @b{(input)}  L7_ENABLE if ARP ACL is enabled 
*                                      on all VLANs in the range
* @param    staticFlag    @b{(input)}  If flag is enabled, packets that 
*                                      do not match any permit statements 
*                                      of ARP ACL are dropped without 
*                                      consulting the DHCP snooping bindings.
*
* @returns  none
*
* @end
*********************************************************************/
void daiVlanARPAclRangeSet(openapiClientHandle_t *client_handle, 
                           open_buffdesc *aclName, uint32_t vlanStart, 
                           uint32_t vlanEnd, uint32_t val, 
                           uint32_t staticFlag)
{
  open_error_t result;

  if ((result = openapiDaiVlanARPAclRangeSet(client_handle, aclName, vlanStart,
      vlanEnd, val, staticFlag)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to apply the ARP ACL to a range of VLANs. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("ARP ACL for range of VLANs is set successful\n");
  }
  return;
}

/*********************************************************************
* @purpose  Set the ARP ACL Name for a vlan (Extended Version).
*
* @param    client_handle  @b{(input)}  client handle from regn API
* @param    vlanId         @b{(input)}  VLAN Id
* @param    aclName        @b{(input)}  ARP ACL name for this VLAN

*
* @returns  none
*
* @end
*********************************************************************/
void daiVlanArpAclSet(openapiClientHandle_t *client_handle, uint32_t vlanId,
                      open_buffdesc *aclName)
{
  open_error_t result;

  if ((result = openapiDaiVlanArpAclSet(client_handle, vlanId, aclName)) != 
      OPEN_E_NONE)
  {
    printf("Bad return code trying to set the ARP ACL Name for a vlan "
           "(Extended Version). (result = %d)\n", result);
  }
  else
  {
    printf("ARP ACL for a VLAN %d is set successfully\n", vlanId);
  }
  return;
}

/*********************************************************************
* @purpose  API to create ARP ACL (extensible version).
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    bufd          @b{(input)}  buffer descriptor containing 
*                                      ACL Name
*
* @returns  none
*
* @end
*********************************************************************/
void arpAclCreate(openapiClientHandle_t *client_handle, open_buffdesc *bufd)
{
  open_error_t result;

  if ((result = openapiArpAclCreate(client_handle, bufd)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to aPI to create ARP ACL (extensible "
           "version). (result = %d)\n", result);
  }
  else
  {
    printf("ARP ACL creation is successful\n");
  }
  return;
}

/*********************************************************************
* @purpose  API to delete ARP ACL (extensible version).
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    bufd          @b{(input)}  buffer descriptor containing 
*                                      ACL Name
*
* @returns  none
*
* @end
*********************************************************************/
void arpAclDelete(openapiClientHandle_t *client_handle, open_buffdesc *bufd)
{
  open_error_t result;

  if ((result = openapiArpAclDelete(client_handle, bufd)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to aPI to delete ARP ACL (extensible "
           "version). (result = %d)\n", result);
  }
  else
  {
    printf("ARP ACL deletion is successful\n");
  }
  return;
}

/*********************************************************************
* @purpose  DAI OpEN APIs Sanity.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes 
* 
* @end
*********************************************************************/
void daiOpENAPIsSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t              result;
  OPEN_CONTROL_t            mode = OPEN_DISABLE;
  OPEN_BOOL_t trustMode;
  OPEN_DAI_VLAN_STATS_t vlanStats;
  int32_t burstInterval = 1;
  int32_t rate = 0;
  uint32_t vlanId = 1, nextVlan = 0;
  uint32_t ifNum = 1, val = 0, staticFlag = 0;
  open_buffdesc aclName, badBufdescPointer;
  open_buffdesc bufd, zeroLenBuf;
  char acl[OPEN_ARP_ACL_NAME_LEN_MAX+1] = {0};

  aclName.pstart = acl;
  aclName.size = OPEN_ARP_ACL_NAME_LEN_MAX+1;

  badBufdescPointer.pstart = NULL;
  zeroLenBuf.size = 0;

  /* openapiDaiSourceMacValidateModeSet() */ 
  printf("\nTesting openapiDaiSourceMacValidateModeSet(): \n");
  result = openapiDaiSourceMacValidateModeSet(NULL, mode);
  printf("NULL client handle. (result = %d)\n", result);
  result = openapiDaiSourceMacValidateModeSet(clientHandle, 9);
  printf("Invalid source MAC validate mode. (result = %d)\n", result);
  printf("openapiDaiSourceMacValidateModeSet() sanity successful.\n");

  /* openapiDaiSourceMacValidateModeGet() */
  printf("\nTesting openapiDaiSourceMacValidateModeGet(): \n");
  result = openapiDaiSourceMacValidateModeGet(NULL, &mode);
  printf("NULL client handle. (result = %d)\n", result);
  result = openapiDaiSourceMacValidateModeGet(clientHandle, NULL);
  printf("NULL DAI source MAC validate mode parameter. (result = %d)\n", result);
  printf("openapiDaiSourceMacValidateGet() sanity successful.\n");

  /* openapiDaiDestMacValidateModeSet() */ 
  printf("\nTesting openapiDaiDestMacValidateModeSet(): \n");
  result = openapiDaiDestMacValidateModeSet(NULL, mode);
  printf("NULL client handle. (result = %d)\n", result);
  result = openapiDaiDestMacValidateModeSet(clientHandle, 9);
  printf("Invalid destination MAC validate mode. (result = %d)\n", result);
  printf("openapiDaiDestMacValidateModeSet() sanity successful.\n");

  /* openapiDaiDestMacValidateModeGet() */
  printf("\nTesting openapiDaiDestMacValidateModeGet(): \n");
  result = openapiDaiDestMacValidateModeGet(NULL, &mode);
  printf("NULL client handle. (result = %d)\n", result);
  result = openapiDaiDestMacValidateModeGet(clientHandle, NULL);
  printf("NULL DAI destination MAC validate mode parameter. (result = %d)\n", result);
  printf("openapiDaiDestMacValidateGet() sanity successful.\n");

  /* openapiDaiIpAddrValidateModeSet() */ 
  printf("\nTesting openapiDaiIpAddrValidateModeSet(): \n");
  result = openapiDaiIpAddrValidateModeSet(NULL, mode);
  printf("NULL client handle. (result = %d)\n", result);
  result = openapiDaiIpAddrValidateModeSet(clientHandle, 9);
  printf("Invalid IP address validate mode. (result = %d)\n", result);
  printf("openapiDaiIpAddrValidateModeSet() sanity successful.\n");

  /* openapiDaiIpAddrValidateModeGet() */
  printf("\nTesting openapiDaiIpAddrValidateModeGet(): \n");
  result = openapiDaiIpAddrValidateModeGet(NULL, &mode);
  printf("NULL client handle. (result = %d)\n", result);
  result = openapiDaiIpAddrValidateModeGet(clientHandle, NULL);
  printf("NULL DAI IP address validate mode parameter. (result = %d)\n", result);
  printf("openapiDaiIpAddrValidateGet() sanity successful.\n");

  printf("Testing openapiDaiIntfTrustGet():\n");
  result = openapiDaiIntfTrustGet(NULL, ifNum, &trustMode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDaiIntfTrustGet(clientHandle, ifNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDaiIntfRateLimitGet():\n");
  result = openapiDaiIntfRateLimitGet(NULL, ifNum, &rate);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDaiIntfRateLimitGet(clientHandle, ifNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDaiIntfRateLimitSet():\n");
  result = openapiDaiIntfRateLimitSet(NULL, ifNum, rate);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDaiIntfRateLimitSet(clientHandle, ifNum, OPEN_DAI_RATE_LIMIT_MIN - 1);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiDaiIntfRateLimitSet(clientHandle, ifNum, OPEN_DAI_RATE_LIMIT_MAX + 1);
  printf("greater than maximum value argument 3:(result = %d)\n", result);

  printf("Testing openapiDaiIntfBurstIntervalGet():\n");
  result = openapiDaiIntfBurstIntervalGet(NULL, ifNum, &burstInterval);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDaiIntfBurstIntervalGet(clientHandle, ifNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDaiIntfBurstIntervalSet():\n");
  result = openapiDaiIntfBurstIntervalSet(NULL, ifNum, burstInterval);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDaiIntfBurstIntervalSet(clientHandle, ifNum, OPEN_DAI_BURST_INTERVAL_MIN - 1);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiDaiIntfBurstIntervalSet(clientHandle, ifNum, OPEN_DAI_BURST_INTERVAL_MAX + 1);
  printf("greater than maximum value argument 3:(result = %d)\n", result);

  printf("Testing openapiDaiVlanStatsGet():\n");
  result = openapiDaiVlanStatsGet(NULL, vlanId, &vlanStats);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDaiVlanStatsGet(clientHandle, vlanId, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDaiVlanNextGet():\n");
  result = openapiDaiVlanNextGet(NULL, vlanId, &nextVlan);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDaiVlanNextGet(clientHandle, vlanId, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDaiVlanEnableGet():\n");
  result = openapiDaiVlanEnableGet(NULL, vlanId, &val);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDaiVlanEnableGet(clientHandle, vlanId, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDaiVlanLoggingEnableGet():\n");
  result = openapiDaiVlanLoggingEnableGet(NULL, vlanId, &val);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDaiVlanLoggingEnableGet(clientHandle, vlanId, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDaiVlanArpAclStaticFlagGet():\n");
  result = openapiDaiVlanArpAclStaticFlagGet(NULL, vlanId, &staticFlag);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDaiVlanArpAclStaticFlagGet(clientHandle, vlanId, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDaiVlanStatsClear():\n");
  result = openapiDaiVlanStatsClear(NULL);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDaiVlanArpAclGet():\n");
  result = openapiDaiVlanArpAclGet(NULL, vlanId, &aclName);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDaiVlanArpAclGet(clientHandle, vlanId, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiDaiVlanArpAclGet(clientHandle, vlanId, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiDaiVlanArpAclGet(clientHandle, vlanId, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);

  printf("Testing openapiDaiVlanArpAclSet():\n");
  result = openapiDaiVlanArpAclSet(NULL, vlanId, &aclName);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDaiVlanArpAclSet(clientHandle, vlanId, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiDaiVlanArpAclSet(clientHandle, vlanId, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiDaiVlanArpAclSet(clientHandle, vlanId, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);

  printf("Testing openapiArpAclCreate():\n");
  result = openapiArpAclCreate(NULL, &bufd);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiArpAclCreate(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);
  result = openapiArpAclCreate(clientHandle, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 2:(result = %d)\n", result);
  result = openapiArpAclCreate(clientHandle, &zeroLenBuf);
  printf("buffdesc 0 length size argument 2:(result = %d)\n", result);

  printf("Testing openapiArpAclDelete():\n");
  result = openapiArpAclDelete(NULL, &bufd);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiArpAclDelete(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);
  result = openapiArpAclDelete(clientHandle, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 2:(result = %d)\n", result);
  result = openapiArpAclDelete(clientHandle, &zeroLenBuf);
  printf("buffdesc 0 length size argument 2:(result = %d)\n", result);


  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum, val1;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  uint32_t arg1, ipAddr;
  OPEN_BOOL_t state;
  int32_t  val;
  OPEN_DAI_VLAN_STATS_t vlanStats;
  open_buffdesc bufd, macAddr, aclName, ipBuffdesc;
  char str[OPEN_ARP_ACL_NAME_LEN_MAX], macStr[OPEN_MFDB_MAC_LENGTH];
  char ipAddrStr[40];

  if (argc < 2)
  {
    printDaiAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("dai_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Dynamic ARP Inspection (DAI) API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printDaiAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      daiSourceMacValidateModeSet(&clientHandle, arg1);
      break;
    case 2:
      if (argc != 2)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiSourceMacValidateModeGet(&clientHandle);
      break;
    case 3:
      if (argc != 3)
      {
        printDaiAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      daiDestMacValidateModeSet(&clientHandle, arg1);
      break;
    case 4:
      if (argc != 2)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiDestMacValidateModeGet(&clientHandle);
      break;
    case 5:
      if (argc != 3)
      {
        printDaiAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      daiIpAddrValidateModeSet(&clientHandle, arg1);
      break;
    case 6:
      if (argc != 2)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiIpAddrValidateModeGet(&clientHandle);
      break;
    case 7:
      if (argc != 3)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiIntfTrustGet(&clientHandle, atoi(argv[2]), &state);
      break;
    case 8:
      if (argc != 4)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiIntfTrustSet(&clientHandle, atoi(argv[2]), atoi(argv[3]));
      break;
    case 9:
      if (argc != 3)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiIntfRateLimitGet(&clientHandle, atoi(argv[2]), &val);
      break;
    case 10:
      if (argc != 4)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiIntfRateLimitSet(&clientHandle, atoi(argv[2]), atoi(argv[3]));
      break;
    case 11:
      if (argc != 3)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiIntfBurstIntervalGet(&clientHandle, atoi(argv[2]), &val);
      break;
    case 12:
      if (argc != 4)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiIntfBurstIntervalSet(&clientHandle, atoi(argv[2]), atoi(argv[3]));
      break;
    case 13:
      if (argc != 3)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiVlanStatsGet(&clientHandle, atoi(argv[2]), &vlanStats);
      break;
    case 14:
      if (argc != 3)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiVlanNextGet(&clientHandle, atoi(argv[2]), &val1);
      break;
    case 15:
      if (argc != 3)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiVlanEnableGet(&clientHandle, atoi(argv[2]), &val1);
      break;
    case 16:
      if (argc != 4)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiVlanEnableSet(&clientHandle, atoi(argv[2]), atoi(argv[3]));
      break;
    case 17:
      if (argc != 3)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiVlanLoggingEnableGet(&clientHandle, atoi(argv[2]), &val1);
      break;
    case 18:
      if (argc != 4)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiVlanLoggingEnableSet(&clientHandle, atoi(argv[2]), atoi(argv[3]));
      break;
    case 19:
      if (argc != 3)
      {
        printDaiAppMenu();
        exit(1);
      }

      daiVlanArpAclStaticFlagGet(&clientHandle, atoi(argv[2]), &val1);
      break;
    case 20:
      if (argc != 4)
      {
        printDaiAppMenu();
        exit(1);
      }

      daiVlanArpAclStaticFlagSet(&clientHandle, atoi(argv[2]), atoi(argv[3]));
      break;
    case 21:
      if (argc != 2)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiVlanStatsClear(&clientHandle);
      break;
    case 22:
      if (argc != 7)
      {
        printDaiAppMenu();
        exit(1);
      }

      memset(str, 0, sizeof(str));
      strncpy(str, argv[2], sizeof(str));
      bufd.pstart = str;
      bufd.size = strlen(str) + 1;

      memset(ipAddrStr, 0, sizeof(ipAddrStr));
      strncpy(ipAddrStr, argv[4], strlen(argv[4]));
      ipBuffdesc.pstart = ipAddrStr;
      ipBuffdesc.size = strlen(ipAddrStr) + 1;
      if ((result = openapiInetAton(&clientHandle, &ipBuffdesc, &ipAddr)) !=
           OPEN_E_NONE)
      {
        printf("ERROR: Bad return code trying to convert ip address string "
               "to a 32 bit integer. result = %d.\n", result);
        exit(1);
      }

      memset(macStr, 0, sizeof(macStr));
      strncpy(macStr, argv[5], (sizeof(macStr) - 1));
      macAddr.pstart = macStr;
      macAddr.size = strlen(macStr) + 1;

      arpAclRuleAdd(&clientHandle, &bufd, atoi(argv[3]), ipAddr, &macAddr, atoi(argv[6]));
      break;
    case 23:
      if (argc != 4)
      {
        printDaiAppMenu();
        exit(1);
      }

      memset(str, 0, sizeof(str));
      strncpy(str, argv[2], sizeof(str));
      bufd.pstart = str;
      bufd.size = strlen(str) + 1;

      arpAclRuleDelete(&clientHandle, &bufd, atoi(argv[3]));
      break;
    case 24:
      if (argc != 3)
      {
        printDaiAppMenu();
        exit(1);
      }

      memset(str, 0, sizeof(str));
      aclName.pstart = str;
      aclName.size = OPEN_ARP_ACL_NAME_LEN_MAX;

      daiVlanArpAclGet(&clientHandle, atoi(argv[2]), &aclName);
      break;
    case 25:
      if (argc != 7)
      {
        printDaiAppMenu();
        exit(1);
      }

      memset(str, 0, sizeof(str));
      strncpy(str, argv[2], sizeof(str));
      aclName.pstart = str;
      aclName.size = strlen(str) + 1;

      daiVlanARPAclRangeSet(&clientHandle, &aclName, atoi(argv[3]), 
                            atoi(argv[4]), atoi(argv[5]), atoi(argv[6]));
      break;
    case 26:
      if (argc != 4)
      {
        printDaiAppMenu();
        exit(1);
      }

      memset(str, 0, sizeof(str));
      strncpy(str, argv[3], sizeof(str));
      aclName.pstart = str;
      aclName.size = strlen(str) + 1;

      daiVlanArpAclSet(&clientHandle, atoi(argv[2]), &aclName);
      break;
    case 27:
      if (argc != 3)
      {
        printDaiAppMenu();
        exit(1);
      }

      memset(str, 0, sizeof(str));
      strncpy(str, argv[2], sizeof(str));
      bufd.pstart = str;
      bufd.size = strlen(str) + 1;

      arpAclCreate(&clientHandle, &bufd);
      break;
    case 28:
      if (argc != 3)
      {
        printDaiAppMenu();
        exit(1);
      }

      memset(str, 0, sizeof(str));
      strncpy(str, argv[2], sizeof(str));
      bufd.pstart = str;
      bufd.size = strlen(str) + 1;

      arpAclDelete(&clientHandle, &bufd);
      break;
    case 29:
      if (argc != 2)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiOpENAPIsSanity(&clientHandle);
      break;
    default:
      printDaiAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Dynamic ARP Inspection (DAI) API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
