
/*********************************************************************
*
*  Copyright 2012-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  fdb_example.c
*
* @purpose   FDB APIs Example.
*
* @component OpEN
*
* @comments
*
* @create    05/31/2015
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_fdb.h"

/*******************************************************************
*
* @brief  This is the main function that will demonstrate
*         FDB OpEN APIs.
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result = OPEN_E_NONE;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];

  open_buffdesc mac;
  char buf[100];

  uint32_t vlanId;
  uint32_t fdbIfNum;
  OPEN_FDB_ADDR_FLAG_t fdbEntryType;

  uint32_t fdbId;
  uint32_t value;
  OPEN_VL_TYPE_t vlType;

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("fdb_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (Result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fails until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting FDB API example application");

  printf ("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet (&clientHandle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf ("Network OS version = %s\n", switch_os_revision_string);
  }
  else
  {
    printf ("Network OS version retrieve error.\n");
  }

  /* exercise various OPEN API FDB functions */
  do
  {
    /* Test retrieving the first entry from the forwarding database */
    memset(buf, 0, sizeof(buf));
    mac.pstart = buf;
    mac.size = sizeof(buf);
    vlanId = 0;
    fdbIfNum = 0;
    fdbEntryType = 0;

    result = openapiFdbEntryNextGet(&clientHandle, &mac, &vlanId, &fdbIfNum, &fdbEntryType);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to get the first entry from the forwarding database (Result = %d)\n", result);
      break;
    }
    else
    {
      printf("Success: Get the first entry from the forwarding database. Found entry with VLAN-MAC: %d-%s\n", vlanId, (char *)mac.pstart);

      /* Test retrieving the next entry from the forwarding database */
      result = openapiFdbEntryNextGet(&clientHandle, &mac, &vlanId, &fdbIfNum, &fdbEntryType);

      if (OPEN_E_NONE != result)
      {
        if (OPEN_E_NOT_FOUND != result)
        {
          printf("Failed to get the next entry from the forwarding database (Result = %d)\n", result);
          break;
        }
        printf("Success: Cannot get the next entry from the forwarding database - entry not found (Result = %d)\n", result);
      }
      else
      {
        printf("Success: Get the next entry from the forwarding database. Found entry with VLAN-MAC: %d-%s\n", vlanId, (char *)mac.pstart);
      }
    }

    /* Test verification is a FDB with the specified ID exists */
    fdbId = 0;
    result = openapiFdbIdGet(&clientHandle, fdbId);

    if (OPEN_E_NONE == result)
    {
      printf("Success: FDB with ID %d exists.\n", fdbId);
    }
    else if (OPEN_E_NOT_FOUND == result)
    {
      printf("Success: FDB with ID %d does not exist.\n", fdbId);
    }
    else
    {
      printf("Failed to verify FDB with ID %d exists. (Result = %d)\n", fdbId, result);
      break;
    }

    /* Testing Fdb Address Aging Timeout - only if FDB exists */
    if (OPEN_E_NONE == result)
    {
      result = openapiFdbAddrAgingTimeoutGet(&clientHandle, fdbId, &value);

      if (OPEN_E_NONE == result)
      {
        printf("Success: Fdb Address Aging Timeout is %d.\n", value);
      }
      else
      {
        printf("Failed to get Fdb Address Aging Timeout for FDB with ID %d. (Result = %d)\n", fdbId, result);
      }

      /* Test Set Fdb Address Aging Timeout */
      value += 1;
      result = openapiFdbAddrAgingTimeoutSet(&clientHandle, fdbId, value);

      if (OPEN_E_NONE == result)
      {
        printf("Success: Set Fdb Address Aging Timeout value: %d.\n", value);
      }
      else
      {
        printf("Failed to set Fdb Address Aging Timeout value: %d. (Result = %d)\n", value, result);
      }
    }

    /* Testing maximum number of entries that the FDB table can hold */
    result = openapiFdbMaxTableEntriesGet(&clientHandle, &value);

    if (OPEN_E_NONE == result)
    {
      printf("Success: The maximum number of FDB entries supported on the platform: %d.\n", value);
    }
    else
    {
      printf("Failed to get maximum number of FDB entries supported on the platform. (Result = %d)\n", result);
    }

    /* Test current number of entries in the FDB table */
    result = openapiActiveAddrEntriesGet(&clientHandle, &value);

    if (OPEN_E_NONE == result)
    {
      printf("Success: The current number of FDB entries: %d.\n", value);
    }
    else
    {
      printf("Failed to get the current number of FDB entries. (Result = %d)\n", result);
    }

    /* Test greatest number of entries in the FDB table */
    result = openapiFdbMaxCountGet(&clientHandle, &value);

    if (OPEN_E_NONE == result)
    {
      printf("Success: The greatest number of FDB entries: %d.\n", value);
    }
    else
    {
      printf("Failed to get the greatest number of FDB entries. (Result = %d)\n", result);
    }

    /* Test resetting the greatest number of entries in the FDB table */
    result = openapiMaxCountReset(&clientHandle);

    if (OPEN_E_NONE == result)
    {
      printf("Success: The count greatest number of FDB entries is reset.\n");
    }
    else
    {
      printf("Failed to reset the greatest number of FDB entries. (Result = %d)\n", result);
    }

    /* Test getting the count of dynamic entries in the L2FDB */
    result = openapiFdbDynamicEntriesGet(&clientHandle, &value);

    if (OPEN_E_NONE == result)
    {
      printf("Success: The count of dynamic entries in the L2FDB: %d.\n", value);
    }
    else
    {
      printf("Failed to get the count of dynamic entries in the L2FDB. (Result = %d)\n", result);
    }

    /* Test getting the count of internally added entries in the L2FDB */
    result = openapiFdbInternalEntriesGet(&clientHandle, &value);

    if (OPEN_E_NONE == result)
    {
      printf("Success: The count of internally added entries in the L2FDB: %d.\n", value);
    }
    else
    {
      printf("Failed to get the count of internally added entries in the L2FDB. (Result = %d)\n", result);
    }

    /* Test if specified MAC is restricted */
    result = openapiIsRestrFilterMacGet(&clientHandle, &mac);

    if (OPEN_E_NONE == result)
    {
      printf("Success: The specified MAC is not restricted.\n");
    }
    else if (OPEN_E_DISABLED == result)
    {
      printf("Success: The specified MAC is restricted.\n");
    }
    else
    {
      printf("Failed to verify if specified MAC is restricted. (Result = %d)\n", result);
    }

    /* Test retrieving of the supported VL type */
    result = openapiFdbTypeOfVLGet(&clientHandle, &vlType);

    if (OPEN_E_NONE == result)
    {
      printf("Success: Get the supported VLAN learning type: %s (%d).\n", 
             (vlType == OPEN_IVL_TYPE) ? "IVL":"SVL", vlType);
    }
    else
    {
      printf("Failed to get the supported VLAN learning type. (Result = %d)\n", result);
    }

    /* Test retrieving the next fdbID starting from a certain index */
    result = openapiFdbIdNextGet(&clientHandle, fdbId, &value);

    if (OPEN_E_NONE == result)
    {
      printf("Success: Get the next fdbID starting from a '%d' index.\n", fdbId);
    }
    else
    {
      printf("Cannot get the next fdbID starting from a '%d' index. (Result = %d)\n", fdbId, result);
    }

    /* Test retrieving the number of failed FDB insert attempts due to failure to allocate space */
    result = openapiFdbBadAddsGet(&clientHandle, &value);

    if (OPEN_E_NONE == result)
    {
      printf("Success: Get the number of failed FDB insert: %d.\n", value);
    }
    else
    {
      printf("Failed to get the number of failed FDB insert. (Result = %d)\n", result);
    }

    /* Test retrieving the max limit of MAC entries set in the hardware */
    result = openapiFdbMacMaxHwLimitGet(&clientHandle, &value);

    if (OPEN_E_NONE == result)
    {
      printf("Success: Get the max limit of MAC entries supported by hardware: %d.\n", value);
    }
    else
    {
      printf("Failed to get the max limit of MAC entries supported by hardware. (Result = %d)\n", result);
    }

    /* Test retrieving the value of dynamic entries in the L2FDB for a specific interface */
    result = openapiFdbIntfDynamicEntriesGet(&clientHandle, fdbIfNum, &value);

    if (OPEN_E_NONE == result)
    {
      printf("Success: Get the value of dynamic entries in the L2FDB for a %d interface: %d.\n", fdbIfNum, value);
    }
    else
    {
      printf("Failed to get the value of dynamic entries in the L2FDB for a %d interface. (Result = %d)\n", fdbIfNum, result);
    }

    /* Test retrieving the value of dynamic entries in the L2FDB for a specific VLAN */
    result = openapiFdbVlanDynamicEntriesGet(&clientHandle, vlanId, &value);

    if (OPEN_E_NONE == result)
    {
      printf("Success: Get the value of dynamic entries in the L2FDB for a VLAN %d: %d.\n", vlanId, value);
    }
    else
    {
      printf("Failed to get the value of dynamic entries in the L2FDB for a VLAN %d. (Result = %d)\n", vlanId, result);
    }

    /* Test retrieving the value of internal entries in the L2FDB for a specific interface */
    result = openapiFdbIntfInternalEntriesGet(&clientHandle, fdbIfNum, &value);

    if (OPEN_E_NONE == result)
    {
      printf("Success: Get the value of internal entries in the L2FDB for a %d interface: %d.\n", fdbIfNum, value);
    }
    else
    {
      printf("Failed to get the value of internal entries in the L2FDB for a %d interface. (Result = %d)\n", fdbIfNum, result);
    }

    /* Test retrieving the value of internal entries in the L2FDB for a specific VLAN */
    result = openapiFdbVlanInternalEntriesGet(&clientHandle, vlanId, &value);

    if (OPEN_E_NONE == result)
    {
      printf("Success: Get the value of internal entries in the L2FDB for a VLAN %d: %d.\n", vlanId, value);
    }
    else
    {
      printf("Failed to get the value of internal entries in the L2FDB for a VLAN %d. (Result = %d)\n", vlanId, result);
    }

    /* Test deleting the specified FDB entry. Get the first FDB entry to delete it */
    memset(buf, 0x00, sizeof(buf));
    mac.size = sizeof(buf);
    vlanId = 0;
    fdbIfNum = 0;
    fdbEntryType = 0;

    if (OPEN_E_NONE == openapiFdbEntryNextGet(&clientHandle, &mac, &vlanId, &fdbIfNum, &fdbEntryType))
    {
      result = openapiFdbEntryDelete(&clientHandle, &mac, vlanId, fdbIfNum);

      if (OPEN_E_NONE == result)
      {
        printf("Success: The specified FDB entry (VLAN = %d, MAC = %s, ifNum = %d) is deleted.\n",
               vlanId, (char *)mac.pstart, fdbIfNum);
      }
      else
      {
        printf("Failed to delete the specified FDB entry. (Result = %d)\n", result);
      }
    }

    /* Test flushing all entries in fdb learned on the specified interface */
    result = openapiFdbFlushByPort(&clientHandle, fdbIfNum);

    if (OPEN_E_NONE == result)
    {
      printf("Success: Flushed all entries in fdb learned on interface %d.\n", fdbIfNum);
    }
    else
    {
      printf("Failed to flush all entries in fdb learned on interface %d. (Result = %d)\n", fdbIfNum, result);
    }

    /* Test flushing all entries in fdb learned on the specified VLAN */
    result = openapiFdbFlushByVlan(&clientHandle, vlanId);

    if (OPEN_E_NONE == result)
    {
      printf("Success: Flushed all entries in fdb learned on VLAN %d.\n", vlanId);
    }
    else
    {
      printf("Failed to flush all entries in fdb learned on VLAN %d. (Result = %d)\n", vlanId, result);
    }

    /* Test flushing all entries in fdb learned on the specified MAC */
    result = openapiFdbFlushByMac(&clientHandle, &mac);

    if (OPEN_E_NONE == result)
    {
      printf("Success: Flushed all entries in fdb learned for MAC = %s.\n", (char *)mac.pstart);
    }
    else
    {
      printf("Failed to flush all entries in fdb learned for MAC = %s. (Result = %d)\n", (char *)mac.pstart, result);
    }

    /* Test flushing all learned entries from the L2FDB */
    result = openapiFdbFlush(&clientHandle);

    if (OPEN_E_NONE == result)
    {
      printf("Success: Flushed all learned entries from the L2FDB.\n");
    }
    else
    {
      printf("Failed to flush all learned entries from the L2FDB. (Result = %d)\n", result);
    }
  }
  while (0);

  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping FDB API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
