/*********************************************************************
*
*  Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  eee_example.c
*
* @purpose   Green ethernet configuration OpEN APIs example 
*
* @component OpEN
*
* @comments
*
* @create    05/19/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_system.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the eee Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printEeeAppMenu()
{
  printf("Usage:  eee_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: testGreenEthrEeeIsValidIntf:eee_example 1 <interface>\n");
  printf("Test 2: testGreenEthrEeeValidIntfFirstGet:eee_example 2\n");
  printf("Test 3: testGreenEthrEeeValidIntfNextGet:eee_example 3 <interface>\n");
  printf("Test 4: testGreenEthrEeeAdminModeGet:eee_example 4 <interface>\n");
  printf("Test 5: testGreenEthrEeeTxIdleTimeGet:eee_example 5 <interface>\n");
  printf("Test 6: testGreenEthrEeeTxWakeTimeGet:eee_example 6 <interface>\n");
  printf("Test 7: testGreenEthrEeeAdminModeSet:eee_example 7 <interface> <mode>\n");
  printf("Test 8: testGreenEthrEeeTxIdleTimeSet:eee_example 8 <interface> <time>\n");
  printf("Test 9: testGreenEthrEeeTxWakeTimeSet:eee_example 9 <interface> <time>\n");
  printf("Test 10: testGreenEthrEeeStatisticsClear:eee_example 10 <interface>\n");
  printf("Test 11: testGreenEthrEeeTxEventsGet:eee_example 11 <interface>\n");
  printf("Test 12: testGreenEthrEeeTxDurationGet:eee_example 12 <interface>\n");
  printf("Test 13: testGreenEthrEeeRxEventsGet:eee_example 13 <interface>\n");
  printf("Test 14: testGreenEthrEeeRxDurationGet:eee_example 14 <interface>\n");
  printf("Test 15: testGreenEthrEeeTxTwsysGet:eee_example 15 <interface>\n");
  printf("Test 16: testGreenEthrEeeRxTwsysGet:eee_example 16 <interface>\n");
  printf("Test 17: testGreenEthrEeeFbTwsysGet:eee_example 17 <interface>\n");
  printf("Test 18: testGreenEthrEeeEchoTxTwsysGet:eee_example 18 <interface>\n");
  printf("Test 19: testGreenEthrEeeEchoRxTwsysGet:eee_example 19 <interface>\n");
  printf("Test 20: testGreenEthrEeeRemTxTwsysGet:eee_example 20 <interface>\n");
  printf("Test 21: testGreenEthrEeeRemTxTwsysEchoGet:eee_example 21 <interface>\n");
  printf("Test 22: testGreenEthrEeeRemRxTwsysGet:eee_example 22 <interface>\n");
  printf("Test 23: testGreenEthrEeeRemRxTwsysEchoGet:eee_example 23 <interface>\n");
  printf("Test 24: testGreenEthrEeeRemFbTwsysGet:eee_example 24 <interface>\n");
  printf("Test 25: testGreenEthrEeeTxDllEnableGet:eee_example 25 <interface>\n");
  printf("Test 26: testGreenEthrEeeTxDllReadyGet:eee_example 26 <interface>\n");
  printf("Test 27: testGreenEthrEeeRxDllEnableGet:eee_example 27 <interface>\n");
  printf("Test 28: testGreenEthrEeeRxDllReadyGet:eee_example 28 <interface>\n");
  printf("Test 29: testGreenEthrEeeTotalLpiTimeGet:eee_example 29\n");
  printf("Test 30: testGreenEthrEeeLocalSummaryGet:eee_example 30 <interface>\n");
  printf("Test 31: testGreenEthrEeeRemoteSummaryGet:eee_example 31 <interface>\n");
  printf("Test 32: testGreenEthrGlobalEeeAdminModeGet:eee_example 32\n");
  printf("Test 33: testGreenEthrGlobalEeeAdminModeSet:eee_example 33 <mode>\n");
  printf("Test 34: testGreenEthrEeeTimeSinceLastStatsResetGet:eee_example 34 <interface>\n");
  printf("Test 35: testxGreenEthrEeeTimeSinceLastStatsResetGet:eee_example 35 <interface>\n");

  return;
}

/*********************************************************************
* @purpose  Determine if the interface is valid 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void testGreenEthrEeeIsValidIntf(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  OPEN_BOOL_t valid;

  if ((result = openapiGreenEthrEeeIsValidIntf(clientHandle, intIfNum, &valid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get valid interface. (result = %d)\n", result);
  }
  else
  {
    printf("Interface valid: %d. (result = %d)\n", valid, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the first interface that is valid for green Ethernet EEE 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void testGreenEthrEeeValidIntfFirstGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t intIfNum;

  if ((result = openapiGreenEthrEeeValidIntfFirstGet(clientHandle, &intIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get first interface. (result = %d)\n", result);
  }
  else
  {
    printf("Valid interface: %d. (result = %d)\n", intIfNum, result);
  }

  return;
}


/*********************************************************************
* @purpose  Get the next interface that is valid for green Ethernet EEE 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    preIntIf         @b{(input)} previous internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeValidIntfNextGet(openapiClientHandle_t *clientHandle, uint32_t preIntIf)
{
  open_error_t result;
  uint32_t intIfNum;

  if ((result = openapiGreenEthrEeeValidIntfNextGet(clientHandle, preIntIf, &intIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get next interface. (result = %d)\n", result);
  }
  else
  {
    printf("Next interface: %d. (result = %d)\n", intIfNum, result);
  }

  return;
}


/*********************************************************************
* @purpose  Get EEE admin mode 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeAdminModeGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  OPEN_CONTROL_t adminMode;

  if ((result = openapiGreenEthrEeeAdminModeGet(clientHandle, intIfNum, &adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get EEE admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("EEE admin mode: %d. (result = %d)\n", adminMode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get EEE Transmit Idle Time 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeTxIdleTimeGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t time;

  if ((result = openapiGreenEthrEeeTxIdleTimeGet(clientHandle, intIfNum, &time)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get EEE Transmit Idle Time. (result = %d)\n", result);
  }
  else
  {
    printf("EEE Transmit Idle Time: %d. (result = %d)\n", time, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get EEE Transmit wake Time 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeTxWakeTimeGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t time;

  if ((result = openapiGreenEthrEeeTxWakeTimeGet(clientHandle, intIfNum, &time)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get EEE Transmit wake Time. (result = %d)\n", result);
  }
  else
  {
    printf("EEE Transmit wake Time: %d. (result = %d)\n", time, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set EEE admin mode 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeAdminModeSet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t mode)
{
  open_error_t result;

  if ((result = openapiGreenEthrEeeAdminModeSet(clientHandle, intIfNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set EEE admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("EEE admin mode: %d. (result = %d)\n", mode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set EEE Transmit Idle Time 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* @param    time             @b{(input)} time
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeTxIdleTimeSet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t time)
{
  open_error_t result;

  if ((result = openapiGreenEthrEeeTxIdleTimeSet(clientHandle, intIfNum, time)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set idle time. (result = %d)\n", result);
  }
  else
  {
    printf("Set idle time: %d. (result = %d)\n", time, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set EEE transmit wake time 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* @param    time             @b{(input)} time
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeTxWakeTimeSet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t time)
{
  open_error_t result;

  if ((result = openapiGreenEthrEeeTxWakeTimeSet(clientHandle, intIfNum, time)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set wake time. (result = %d)\n", result);
  }
  else
  {
    printf("Set wake time: %d. (result = %d)\n", time, result);
  }

  return;
}

/*********************************************************************
* @purpose  Clear EEE statistics counters 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeStatisticsClear(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;

  if ((result = openapiGreenEthrEeeStatisticsClear(clientHandle, intIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to Clear EEE statistics counters. (result = %d)\n", result);
  }
  else
  {
    printf("Clear EEE statistics counters. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get EEE Transmit LPI event count 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeTxEventsGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiGreenEthrEeeTxEventsGet(clientHandle, intIfNum, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to EEE Transmit LPI event count. (result = %d)\n", result);
  }
  else
  {
    printf("EEE Transmit LPI event count: %d. (result = %d)\n", count, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get EEE Transmit LPI duration 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeTxDurationGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t high;
  uint32_t low;

  if ((result = openapiGreenEthrEeeTxDurationGet(clientHandle, intIfNum, &high, &low)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to EEE Transmit duration. (result = %d)\n", result);
  }
  else
  {
    printf("EEE Transmit LPI duration: high %d  low %d. (result = %d)\n", high, low, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get EEE receive LPI event count 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeRxEventsGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiGreenEthrEeeRxEventsGet(clientHandle, intIfNum, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to EEE receive LPI event count. (result = %d)\n", result);
  }
  else
  {
    printf("EEE receive LPI event count: %d. (result = %d)\n", count, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get EEE receive LPI duration 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeRxDurationGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t high;
  uint32_t low;

  if ((result = openapiGreenEthrEeeRxDurationGet(clientHandle, intIfNum, &high, &low)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to EEE receive duration. (result = %d)\n", result);
  }
  else
  {
    printf("EEE receive LPI duration: high %d  low %d. (result = %d)\n", high, low, result);
  }

  return;
}
/*********************************************************************
* @purpose  Get EEE transmit Tw_sys 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeTxTwsysGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiGreenEthrEeeTxTwsysGet(clientHandle, intIfNum, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to EEE transmit Tw_sys. (result = %d)\n", result);
  }
  else
  {
    printf("EEE transmit Tw_sys: %d. (result = %d)\n", count, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get EEE receive Tw_sys
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeRxTwsysGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiGreenEthrEeeRxTwsysGet(clientHandle, intIfNum, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to EEE receive Tw_sys. (result = %d)\n", result);
  }
  else
  {
    printf("EEE receive Tw_sys: %d. (result = %d)\n", count, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get EEE fallback Tw_sys
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeFbTwsysGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiGreenEthrEeeFbTwsysGet(clientHandle, intIfNum, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to EEE fallback Tw_sys. (result = %d)\n", result);
  }
  else
  {
    printf("EEE fallback Tw_sys: %d. (result = %d)\n", count, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get EEE echo transmit Tw_sys 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeEchoTxTwsysGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiGreenEthrEeeEchoTxTwsysGet(clientHandle, intIfNum, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to EEE echo transmit Tw_sys. (result = %d)\n", result);
  }
  else
  {
    printf("EEE echo transmit Tw_sys: %d. (result = %d)\n", count, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get EEE echo receive Tw_sys
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeEchoRxTwsysGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiGreenEthrEeeEchoRxTwsysGet(clientHandle, intIfNum, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to EEE echo receive Tw_sys. (result = %d)\n", result);
  }
  else
  {
    printf("EEE echo receive Tw_sys: %d. (result = %d)\n", count, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get remote EEE transmit Tw_sys 
* 
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    intIfNum       @b{(input)} internal interface number
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this entry
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeRemTxTwsysGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum,
                                   uint32_t remIndex,uint32_t timestamp)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiGreenEthrEeeRemTxTwsysGet(clientHandle, intIfNum, remIndex, timestamp, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to remote EEE transmit Tw_sys. (result = %d)\n", result);
  }
  else
  {
    printf("Remote EEE transmit Tw_sys: %d. (result = %d)\n", count, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get remote EEE transmit Tw_sys echo
* 
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    intIfNum       @b{(input)} internal interface number
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this entry
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeRemTxTwsysEchoGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum,
                                   uint32_t remIndex,uint32_t timestamp)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiGreenEthrEeeRemTxTwsysEchoGet(clientHandle, intIfNum, remIndex, timestamp, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to remote EEE transmit Tw_sys echo. (result = %d)\n", result);
  }
  else
  {
    printf("Remote EEE transmit Tw_sys echo: %d. (result = %d)\n", count, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get remote EEE receive Tw_sys 
* 
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    intIfNum       @b{(input)} internal interface number
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this entry
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeRemRxTwsysGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum,
                                   uint32_t remIndex,uint32_t timestamp)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiGreenEthrEeeRemRxTwsysGet(clientHandle, intIfNum, remIndex, timestamp, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to remote EEE receive Tw_sys. (result = %d)\n", result);
  }
  else
  {
    printf("Remote EEE receive Tw_sys: %d. (result = %d)\n", count, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get remote EEE receive Tw_sys echo
* 
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    intIfNum       @b{(input)} internal interface number
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this entry
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeRemRxTwsysEchoGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum,
                                   uint32_t remIndex,uint32_t timestamp)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiGreenEthrEeeRemRxTwsysEchoGet(clientHandle, intIfNum, remIndex, timestamp, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to remote EEE receive Tw_sys echo. (result = %d)\n", result);
  }
  else
  {
    printf("Remote EEE receive Tw_sys echo: %d. (result = %d)\n", count, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get remote EEE fallback Tw_sys 
* 
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    intIfNum       @b{(input)} internal interface number
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this entry
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeRemFbTwsysGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum,
                                   uint32_t remIndex,uint32_t timestamp)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiGreenEthrEeeRemFbTwsysGet(clientHandle, intIfNum, remIndex, timestamp, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to remote EEE fallback Tw_sys. (result = %d)\n", result);
  }
  else
  {
    printf("Remote EEE fallback Tw_sys: %d. (result = %d)\n", count, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get Transmit DLL Enable Status
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeTxDllEnableGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  OPEN_BOOL_t enable;

  if ((result = openapiGreenEthrEeeTxDllEnableGet(clientHandle, intIfNum, &enable)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Transmit DLL Enable Status. (result = %d)\n", result);
  }
  else
  {
    printf("Transmit DLL Enable Status: %d. (result = %d)\n", enable, result);
  }

  return;
}
/*********************************************************************
* @purpose  Get Transmit DLL ready status
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeTxDllReadyGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  OPEN_BOOL_t enable;

  if ((result = openapiGreenEthrEeeTxDllReadyGet(clientHandle, intIfNum, &enable)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Transmit DLL ready Status. (result = %d)\n", result);
  }
  else
  {
    printf("Transmit DLL ready Status: %d. (result = %d)\n", enable, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get receive DLL enable status
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeRxDllEnableGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  OPEN_BOOL_t enable;

  if ((result = openapiGreenEthrEeeRxDllEnableGet(clientHandle, intIfNum, &enable)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get receive DLL Enable Status. (result = %d)\n", result);
  }
  else
  {
    printf("Receive DLL Enable Status: %d. (result = %d)\n", enable, result);
  }

  return;
}
/*********************************************************************
* @purpose  Get receive DLL ready status
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeRxDllReadyGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  OPEN_BOOL_t enable;

  if ((result = openapiGreenEthrEeeRxDllReadyGet(clientHandle, intIfNum, &enable)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get receive DLL ready Status. (result = %d)\n", result);
  }
  else
  {
    printf("Receive DLL ready Status: %d. (result = %d)\n", enable, result);
  }

  return;
}
/*********************************************************************
* @purpose  Determine time of last reset
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeTimeSinceLastStatsResetGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  uint64_t val;

  if ((result = openapiGreenEthrEeeTimeSinceLastStatsResetGet(clientHandle, intIfNum, &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get time of last reset. (result = %d)\n", result);
  }
  else
  {
    printf("Time of last reset: %ld. (result = %d)\n", val, result);
  }

  return;
}
/*********************************************************************
* @purpose  Determine time of last reset
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testxGreenEthrEeeTimeSinceLastStatsResetGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t d, h, m, s;

  if ((result = openapixGreenEthrEeeTimeSinceLastStatsResetGet(clientHandle, intIfNum, &d, &h, &m, &s)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get time of last reset. (result = %d)\n", result);
  }
  else
  {
    printf("Time of last reset: D %d H %d M %d S %d (result = %d)\n", d, h, m, s, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get percentage of Total LPI time spent for all interfaces
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeTotalLpiTimeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiGreenEthrEeeTotalLpiTimeGet(clientHandle, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to percentage of Total LPI time. (result = %d)\n", result);
  }
  else
  {
    printf("Percentage of Total LPI time: %d. (result = %d)\n", count, result);
  }

  return;
}
/*********************************************************************
* @purpose  green ethernet eee counters
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeLocalSummaryGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t t, te, r, re, tf, tde, tdr, rde, rdr; 

  if ((result = openapiGreenEthrEeeLocalSummaryGet(clientHandle, intIfNum,
       &t, &te, &r, &re, &tf, &tde, &tdr, &rde, &rdr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get eee counters. (result = %d)\n", result);
  }
  else
  {
    printf("eee counters (result = %d)\n", result);
    printf("twSysTx  %d\n", t);
    printf("twSysTxEcho  %d\n", te);
    printf("twSysRx  %d\n", r);
    printf("twSysRxEcho  %d\n", re);
    printf("twSysTxFb  %d\n", tf);
    printf("txDllEnabled  %d\n", tde);
    printf("txDllReady  %d\n", tdr);
    printf("rxDllEnabled  %d\n", rde);
    printf("rxDllReady  %d\n", rdr);
  }

  return;
}

/*********************************************************************
* @purpose  Remote green ethernet eee counters
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeRemoteSummaryGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t t, te, r, re, tf; 

  if ((result = openapiGreenEthrEeeRemoteSummaryGet(clientHandle, intIfNum,
                                      &t, &te, &r, &re, &tf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get remote eee counters. (result = %d)\n", result);
  }
  else
  {
    printf("Remote eee counters (result = %d)\n", result);
    printf("twSysTx  %d\n", t);
    printf("twSysTxEcho  %d\n", te);
    printf("twSysRx  %d\n", r);
    printf("twSysRxEcho  %d\n", re);
    printf("twSysTxFb  %d\n", tf);
  }

  return;
}

/*********************************************************************
* @purpose  Get the Global EEE Administrative mode 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrGlobalEeeAdminModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t adminMode;

  if ((result = openapiGreenEthrGlobalEeeAdminModeGet(clientHandle, &adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get global EEE administrative mode. (result = %d)\n", result);
  }
  else
  {
    printf("Global EEE administrative mode: %d. (result = %d)\n", adminMode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set the Global EEE Administrative mode 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrGlobalEeeAdminModeSet(openapiClientHandle_t *clientHandle, uint32_t mode)
{
  open_error_t result;
  OPEN_CONTROL_t adminMode;

  if (1 == mode)
  {
    adminMode = OPEN_ENABLE;
  }
  else if (0 == mode)
  {
    adminMode = OPEN_DISABLE;
  }
  else
  {
    printf("Invalid input params\n");
    return;
  }

  if ((result = openapiGreenEthrGlobalEeeAdminModeSet(clientHandle, adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set global EEE administrative mode. (result = %d)\n", result);
  }
  else
  {
    printf("Global EEE administrative mode. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Sanity of eee OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note
* 
* @end
*********************************************************************/
void eeeOpENAPISanity(openapiClientHandle_t *clientHandle)
{
  printf("Sanity successful.\n");

  return;
}


/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  open_buffdesc switch_os_revision;
  open_revision_data_t openApiVersion;
  char switch_os_revision_string[OPENAPI_OS_REVISION_STR_SIZE];
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t arg1, arg2, arg3;
  uint32_t testNum;

  if (argc < 2)
  {
    printEeeAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("eee_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting eee API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  if (openapiApiVersionGet(&clientHandle, &openApiVersion) == OPEN_E_NONE)
    printf("Open API Version = %d.%d.%d.%d\n", 
           openApiVersion.release,
           openApiVersion.version,
           openApiVersion.maint_level,
           openApiVersion.build_num);
  else
    printf("Open API Version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeIsValidIntf(&clientHandle,arg1);
      break;
    case 2:
      if (argc != 2)
      {
        printEeeAppMenu();
        exit(1);
      }
      testGreenEthrEeeValidIntfFirstGet(&clientHandle);
      break;
    case 3:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeValidIntfNextGet(&clientHandle, arg1);
      break;
    case 4:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeAdminModeGet(&clientHandle, arg1);
      break;
    case 5:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeTxIdleTimeGet(&clientHandle, arg1);
      break;
    case 6:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeTxWakeTimeGet(&clientHandle, arg1);
      break;
    case 7:
      if (argc != 4)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      testGreenEthrEeeAdminModeSet(&clientHandle, arg1, arg2);
      break;
    case 8:
      if (argc != 4)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      testGreenEthrEeeTxIdleTimeSet(&clientHandle, arg1, arg2);
      break;
    case 9:
      if (argc != 4)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      testGreenEthrEeeTxWakeTimeSet(&clientHandle, arg1, arg2);
      break;
    case 10:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeStatisticsClear(&clientHandle, arg1);
      break;
    case 11:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeTxEventsGet(&clientHandle, arg1);
      break;
    case 12:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeTxDurationGet(&clientHandle, arg1);
      break;
    case 13:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeRxEventsGet(&clientHandle, arg1);
      break;
    case 14:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeRxDurationGet(&clientHandle, arg1);
      break;
    case 15:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeTxTwsysGet(&clientHandle, arg1);
      break;
    case 16:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeRxTwsysGet(&clientHandle, arg1);
      break;
    case 17:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeFbTwsysGet(&clientHandle, arg1);
      break;
    case 18:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeEchoTxTwsysGet(&clientHandle, arg1);
      break;
    case 19:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeEchoRxTwsysGet(&clientHandle, arg1);
      break;
    case 20:
      if (argc != 5)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      testGreenEthrEeeRemTxTwsysGet(&clientHandle, arg1, arg2, arg3);
      break;
    case 21:
      if (argc != 5)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      testGreenEthrEeeRemTxTwsysEchoGet(&clientHandle, arg1, arg2, arg3);
      break;
    case 22:
      if (argc != 5)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      testGreenEthrEeeRemRxTwsysGet(&clientHandle, arg1, arg2, arg3);
      break;
    case 23:
      if (argc != 5)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      testGreenEthrEeeRemRxTwsysEchoGet(&clientHandle, arg1, arg2, arg3);
      break;
    case 24:
      if (argc != 5)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      testGreenEthrEeeRemFbTwsysGet(&clientHandle, arg1, arg2, arg3);
      break;
    case 25:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeTxDllEnableGet(&clientHandle, arg1);
      break;
    case 26:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeTxDllReadyGet(&clientHandle, arg1);
      break;
    case 27:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeRxDllEnableGet(&clientHandle, arg1);
      break;
    case 28:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeRxDllReadyGet(&clientHandle, arg1);
      break;
    case 29:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeTotalLpiTimeGet(&clientHandle);
      break;
    case 30:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeLocalSummaryGet(&clientHandle, arg1);
      break;
    case 31:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeRemoteSummaryGet(&clientHandle, arg1);
      break;
    case 32:
      if (argc != 2)
      {
        printEeeAppMenu();
        exit(1);
      }
      testGreenEthrGlobalEeeAdminModeGet(&clientHandle);
      break;
    case 33:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrGlobalEeeAdminModeSet(&clientHandle, arg1);
      break;
    case 34:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeTimeSinceLastStatsResetGet(&clientHandle, arg1);
      break;
    case 35:
      if (argc != 3)
      {
        printEeeAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testxGreenEthrEeeTimeSinceLastStatsResetGet(&clientHandle, arg1);
      break;

    default:
      printEeeAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping green ethernet API example application");
        
  (void) openapiClientTearDown(&clientHandle);          
  return 0;
}
