/*********************************************************************
*
*  Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  green_pwr_estimates_example.c
*
* @purpose   Green ethernet configuration OpEN APIs example 
*
* @component OpEN
*
* @comments
*
* @create    05/19/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_system.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*********************************************************************
* @purpose Get green mode current power consumption on given interface
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenPwrEstIntfCurPwrGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiGreenPwrEstIntfCurPwrGet(clientHandle, intIfNum, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to green mode current power consumption. (result = %d)\n", result);
  }
  else
  {
    printf("Green mode current power consumption: %d. (result = %d)\n", count, result);
  }

  return;
}

/*********************************************************************
* @purpose Get green mode current power consumption
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenPwrEstGlobalCurPwrGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiGreenPwrEstGlobalCurPwrGet(clientHandle, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to green mode current power consumption. (result = %d)\n", result);
  }
  else
  {
    printf("Green mode current power consumption: %d. (result = %d)\n", count, result);
  }

  return;
}

/*********************************************************************
* @purpose Get green mode current power saving on given interface
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenPwrEstIntfCumSavingGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiGreenPwrEstIntfCumSavingGet(clientHandle, intIfNum, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to green mode current power saving. (result = %d)\n", result);
  }
  else
  {
    printf("Green mode current power saving: %d. (result = %d)\n", count, result);
  }

  return;
}

/*********************************************************************
* @purpose Get green mode current power saving
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenPwrEstGlobalCumSavingGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiGreenPwrEstGlobalCumSavingGet(clientHandle, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to green mode current power saving. (result = %d)\n", result);
  }
  else
  {
    printf("Green mode current power saving: %d. (result = %d)\n", count, result);
  }

  return;
}

/*********************************************************************
* @purpose Get green mode power saving
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrPwrEstGlobalPwrSavingGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiGreenEthrPwrEstGlobalPwrSavingGet(clientHandle, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to green mode power saving. (result = %d)\n", result);
  }
  else
  {
    printf("Green mode power saving: %d. (result = %d)\n", count, result);
  }

  return;
}
/*********************************************************************
* @purpose Clear green mode statistics for the given interface
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrIntfStatisticsClear(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;

  if ((result = openapiGreenEthrIntfStatisticsClear(clientHandle, intIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear green mode statistics. (result = %d)\n", result);
  }
  else
  {
    printf("Clear green mode statistics (result = %d)\n", result);
  }

  return;
}
/*******************************************************************
*
* @brief  This function prints the green pwr estimates Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printGreenPwrEstimatesAppMenu()
{
  printf("Usage:  green_ethernet_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: testGreenPwrEstIntfCurPwrGet: green_pwr_estimates_example 1 <interface>\n");
  printf("Test 2: testGreenPwrEstGlobalCurPwrGet: green_pwr_estimates_example 2\n");
  printf("Test 3: testGreenPwrEstIntfCumSavingGet: green_pwr_estimates_example 3 <interface>\n");
  printf("Test 4: testGreenPwrEstGlobalCumSavingGet: green_pwr_estimates_example 4\n");
  printf("Test 5: testGreenEthrPwrEstGlobalPwrSavingGet: green_pwr_estimates_example 5\n");
  printf("Test 6: testGreenEthrIntfStatisticsClear: green_pwr_estimates_example 6 <interface>\n");

  return;
}

/*********************************************************************
* @purpose  Sanity of green pwr estimates OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note
* 
* @end
*********************************************************************/
void pwrEstimatesOpENAPISanity(openapiClientHandle_t *clientHandle)
{

  return;
}


/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  open_revision_data_t openApiVersion;
  char switch_os_revision_string[OPENAPI_OS_REVISION_STR_SIZE];
  uint32_t arg1;

  if (argc < 2)
  {
    printGreenPwrEstimatesAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("green_ethernet_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting green pwr estimates API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  if (openapiApiVersionGet(&clientHandle, &openApiVersion) == OPEN_E_NONE)
    printf("Open API Version = %d.%d.%d.%d\n", 
           openApiVersion.release,
           openApiVersion.version,
           openApiVersion.maint_level,
           openApiVersion.build_num);
  else
    printf("Open API Version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printGreenPwrEstimatesAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenPwrEstIntfCurPwrGet(&clientHandle, arg1);
      break;
    case 2:
      if (argc != 2)
      {
        printGreenPwrEstimatesAppMenu();
        exit(1);
      }
      testGreenPwrEstGlobalCurPwrGet(&clientHandle);
      break;
    case 3:
      if (argc != 3)
      {
        printGreenPwrEstimatesAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenPwrEstIntfCumSavingGet(&clientHandle, arg1);
      break;
    case 4:
      if (argc != 2)
      {
        printGreenPwrEstimatesAppMenu();
        exit(1);
      }
      testGreenPwrEstGlobalCumSavingGet(&clientHandle);
      break;
    case 5:
      if (argc != 2)
      {
        printGreenPwrEstimatesAppMenu();
        exit(1);
      }
      testGreenEthrPwrEstGlobalPwrSavingGet(&clientHandle);
      break;
    case 6:
      if (argc != 3)
      {
        printGreenPwrEstimatesAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrIntfStatisticsClear(&clientHandle, arg1);
      break;


    default:
      printGreenPwrEstimatesAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping green ethernet API example application");
        
  (void) openapiClientTearDown(&clientHandle);          
  return 0;
}
