
/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  interface_example.c
*
* @purpose   Switch Interface APIs Example.
*
* @component OPEN
*
* @comments
*
* @create    10/15/2012
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_if.h"
#include "openapi_boxs.h"


void getPhysicalModes(uint64_t portCapability)
{
  printf("------------------------------------------------\n");
  if (((portCapability) & (OPEN_PHY_CAP_PORTSPEED_AUTO_NEG)) == (OPEN_PHY_CAP_PORTSPEED_AUTO_NEG))
  {
    printf("Auto Negotiate \n");
  }

  if (( portCapability) & (OPEN_PHY_CAP_PORTSPEED_HALF_10))
  {
    printf("10 Mbps Half Duplex \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_10))
  {
    printf("10 Mbps Full Duplex \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_HALF_100))
  {
    printf("100 Mbps Half Duplex \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_100))
  {
    printf("100 Mbps Full Duplex \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_HALF_1000))
  {
    printf("1000 Mbps Half Duplex \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_1000))
  {
    printf("1000 Mbps Full Duplex \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_2P5G))
  {
    printf("2.5 Gbps Full \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_5G))
  {
    printf("5 Gbps Full \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_10G))
  {
    printf("10 Gbps Full \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_20G))
  {
    printf("20 Gbps Full \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_25G))
  {
    printf("25 Gbps Full \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_40G))
  {
    printf("40 Gbps Full \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_50G))
  {
    printf("50 Gbps Full \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_100G))
  {
    printf("100 Gbps Full \n");
  }
}

static char *reasonToStr(OPEN_DISABLE_REASON_t reason)
{
  char *ret = "Unknown reason";

  switch(reason)
  {
    case OPEN_DISABLE_REASON_NONE:
      ret = "OPEN_DISABLE_REASON_NONE";
      break;
    case OPEN_DISABLE_REASON_UDLD:
      ret = "OPEN_DISABLE_REASON_UDLD";
      break;
    case OPEN_DISABLE_REASON_STP:
      ret = "OPEN_DISABLE_REASON_STP";
      break;
    case OPEN_DISABLE_REASON_BPDUGUARD:
      ret = "OPEN_DISABLE_REASON_BPDUGUARD";
      break;
    case OPEN_DISABLE_REASON_BPDUSTORM:
      ret = "OPEN_DISABLE_REASON_BPDUSTORM";
      break;
    case OPEN_DISABLE_REASON_DHCPSNOOPING:
      ret = "OPEN_DISABLE_REASON_DHCPSNOOPING";
      break;
    case OPEN_DISABLE_REASON_ARPINSPECTION:
      ret = "OPEN_DISABLE_REASON_ARPINSPECTION";
      break;
    case OPEN_DISABLE_REASON_STORMCONTROL_BCAST:
      ret = "OPEN_DISABLE_REASON_STORMCONTROL_BCAST";
      break;
    case OPEN_DISABLE_REASON_STORMCONTROL_MCAST:
      ret = "OPEN_DISABLE_REASON_STORMCONTROL_MCAST";
      break;
    case OPEN_DISABLE_REASON_STORMCONTROL_UCAST:
      ret = "OPEN_DISABLE_REASON_STORMCONTROL_UCAST";
      break;
    case OPEN_DISABLE_REASON_XCEIVER:
      ret = "OPEN_DISABLE_REASON_XCEIVER";
      break;
    case OPEN_DISABLE_REASON_XCEIVERPLUS:
      ret = "OPEN_DISABLE_REASON_XCEIVERPLUS";
      break;
    case OPEN_DISABLE_REASON_LOOPPROTECT:
      ret = "OPEN_DISABLE_REASON_LOOPPROTECT";
      break;
    case OPEN_DISABLE_REASON_PML:
      ret = "OPEN_DISABLE_REASON_PML";
      break;
    case OPEN_DISABLE_REASON_DOS:
      ret = "OPEN_DISABLE_REASON_DOS";
      break;
    case OPEN_DISABLE_REASON_LINK_FLAP:
      ret = "OPEN_DISABLE_REASON_LINK_FLAP";
      break;
    case OPEN_DISABLE_REASON_AUTHMGR_AUTH_VIOLATION:
      ret = "OPEN_DISABLE_REASON_AUTHMGR_AUTH_VIOLATION";
      break;
    default:
      break;
  }
  return ret;
}

static char *specialPortTypeToStr(OPEN_SPECIAL_PORT_TYPE_t specialPortType)
{
  char *ret = "Unknown";

  switch(specialPortType)
  {
    case OPEN_SPECIAL_PORT_TYPE_NORMAL:
      ret = "Normal port";
      break;
    case OPEN_SPECIAL_PORT_TYPE_TRUNK_MEMBER:
      ret = "Trunk member";
      break;
    case OPEN_SPECIAL_PORT_TYPE_MIRRORED_PORT:
      ret = "Mirrored port";
      break;
    case OPEN_SPECIAL_PORT_TYPE_PROBE_PORT:
      ret = "Probe port";
      break;
    case OPEN_SPECIAL_PORT_TYPE_L2TUNNEL_PORT:
      ret = "L2 tunnel port";
      break;
    case OPEN_SPECIAL_PORT_TYPE_SERVICE_PORT:
      ret = "Service port";
      break;
    default:
      break;
  }
  return ret;
}

static char *mediaTypeToStr(OPEN_PORT_CABLE_MEDIA_TYPE_t mediaType, OPEN_SFP_MODULE_TYPE_t moduleType)
{
  char *ret = "Unknown";

  if (mediaType == OPEN_PORT_CABLE_MEDIA_TYPE_COPPER)
  {
    ret = "Copper";
  }
  else if (mediaType == OPEN_PORT_CABLE_MEDIA_TYPE_FIBER)
  {
    switch(moduleType)
    {
      case OPEN_MODULE_NONE:
        ret = "None";
        break;
      case OPEN_MODULE_SFP:
        ret = "SFP";
        break;
      case OPEN_MODULE_SFPPLUS:
        ret = "SFP+";
        break;
      case OPEN_MODULE_QSFP:
        ret = "QSFP";
        break;
      case OPEN_MODULE_DAC:
        ret = "Direct attach cable";
        break;
      case OPEN_MODULE_AX741:
        ret = "XFP 10G plugin module (AX741)";
        break;
      case OPEN_MODULE_AX742:
        ret = "Stacking module (AX742)";
        break;
      case OPEN_MODULE_AX743:
        ret = "SFP+ plugin module (AX743)";
        break;
      case OPEN_MODULE_AX744:
        ret = "CX4 plugin module (AX744)";
        break;
      case OPEN_MODULE_AX745:
        ret = "Copper 10G plugin module (AX745)";
        break;
      default:
        break;
    }
  }
  return ret;
}

/*********************************************************************
* @purpose  Get the Configuration Mask Offset.
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    interface     @b{(input)}  internal interface number
* @param    maskOffset    @b{(output)} config mask offset for a given 
*                                      interface
*
* @returns  none
*
* @end
*********************************************************************/
void configIdMaskOffsetGet(openapiClientHandle_t *client_handle, 
                           uint32_t interface, uint32_t *maskOffset)
{
  open_error_t result;

  if ((result = openapiConfigIdMaskOffsetGet(client_handle, interface, 
      maskOffset)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Configuration Mask Offset. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Configuration mask offset for interface %d is %d\n", interface,
           *maskOffset);
  }

  return;
}

/*********************************************************************
* @purpose  Returns the interface index associated with the internal 
*           interface number.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    interface       @b{(input)}  internal interface number
* @param    ifIndex         @b{(output)} interface index

*
* @returns  none
*
* @end
*********************************************************************/
void ifIndexGet(openapiClientHandle_t *client_handle, uint32_t interface, 
                uint32_t *ifIndex)
{
  open_error_t result;

  if ((result = openapiusmDbIfIndexGet(client_handle, interface, ifIndex)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to returns the interface index associated "
           "with the internal interface number. (result = %d)\n", result);
  }
  else
  {
    printf("Fetched interface index for interface %d is %d\n", interface, 
           *ifIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Gets the active state of the specified interface 
*
* @param    client_handle   @b{(input)}  client handle from regn API
* @param    interface       @b{(input)}  internal interface number
* @param    intfActiveState @b{(output)} active - if link is up and is in 
*                                        STP forwarding state
*
* @returns  none
*
* @end
*********************************************************************/
void ifActiveStateGet(openapiClientHandle_t *client_handle, uint32_t interface, 
                      uint32_t *intfActiveState)
{
  open_error_t result;

  if ((result = openapiIfActiveStateGet(client_handle, interface, intfActiveState)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the Broadcast Storm Control Action "
           "mode of an interface (result = %d)\n", result);
  }
  else
  {
    printf("Interface %d active state is %d\n", interface, *intfActiveState);
  }
  return;
}

static char *expPortModeStr(OPEN_EXPANDABLE_PORTMODE_t mode)
{
  char *ret = "Invalid";

  switch (mode)
  {
    case OPEN_EXPANDABLE_PORTMODE_NONE:
      ret = "None";
      break;
    case OPEN_EXPANDABLE_PORTMODE_SINGLE_10G:
      ret = "Single lane ports 10G";
      break;
    case OPEN_EXPANDABLE_PORTMODE_SINGLE_25G:
      ret = "Single lane ports 25G";
      break;
    case OPEN_EXPANDABLE_PORTMODE_DUAL_50G:
      ret = "Dual lane ports 50G";
      break;
    case OPEN_EXPANDABLE_PORTMODE_QUAD_40G:
      ret = "Quad lane 40g ports";
      break;
    case OPEN_EXPANDABLE_PORTMODE_QUAD_100G:
      ret = "Quad lane 100g ports";
      break;
    case OPEN_EXPANDABLE_PORTMODE_ROOT_DUAL_50G:
      ret = "Single 50GB port with dual lane";
      break;
    case OPEN_EXPANDABLE_PORTMODE_DUAL_25G:
      ret = "Dual 25GB ports";
      break;
    case OPEN_EXPANDABLE_PORTMODE_8x10G:
      ret = "Single lane 10G ports";
      break;
    case OPEN_EXPANDABLE_PORTMODE_8x25G:
      ret = "Single lane 25G ports";
      break;
    case OPEN_EXPANDABLE_PORTMODE_8x50G:
      ret = "Single lane 50G ports";
      break;
    case OPEN_EXPANDABLE_PORTMODE_4x100G:
      ret = "Quad lane 100g ports";
      break;
    case OPEN_EXPANDABLE_PORTMODE_2x100G:
      ret = "Dual lane 100g ports";
      break;
    case OPEN_EXPANDABLE_PORTMODE_2x200G:
      ret = "Quad lane 200g ports"; 
      break;
    case OPEN_EXPANDABLE_PORTMODE_1x400G:
      ret = "Octalane 400g ports";
      break;
    case OPEN_EXPANDABLE_PORTMODE_LAST:
    case OPEN_EXPANDABLE_PORTMODE_INVALID:
    default:
      break;
  }

  return ret;
}

/*********************************************************************
* @purpose  Get the expandable port configuration.
*
* @param    client_handle   @b{(input)}  client handle from regn API
*
* @returns  none
*
* @end
*********************************************************************/
void displayExpandablePortInfo(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t intfNum = 0;
  open_buffdesc intfName;
  char intfNameStr[OPENAPI_INTERFACE_LIST_MAX_LENGTH];
  OPEN_EXPANDABLE_PORTMODE_t runnPortMode = OPEN_EXPANDABLE_PORTMODE_NONE;
  OPEN_EXPANDABLE_PORTMODE_t confPortMode = OPEN_EXPANDABLE_PORTMODE_NONE;
  OPEN_EXPANDABLE_PORTMODE_t portMode = OPEN_EXPANDABLE_PORTMODE_NONE;
  char peerIntfsStr[OPENAPI_INTERFACE_LIST_MAX_LENGTH];
  open_buffdesc peerIntfs;
  uint32_t peerCount = 0;

  memset(intfNameStr, 0, sizeof(intfNameStr));
  intfName.pstart = intfNameStr;
  intfName.size = sizeof(intfNameStr);

  memset(peerIntfsStr, 0, sizeof(peerIntfsStr));
  peerIntfs.pstart = peerIntfsStr;
  peerIntfs.size = sizeof(peerIntfsStr);

  if ((result = openapiIntfExpandablePortNextGet(client_handle, intfNum, &intfNum)) != OPEN_E_NONE)
  {
    printf("\r\nExpandable ports are not supported.\r\n");
    return;
  }

  printf("\r\nExpandable Port Configuration:\r\n");

  do
  {
    memset(intfNameStr, 0, sizeof(intfNameStr));
    intfName.size = sizeof(intfNameStr);
    runnPortMode = OPEN_EXPANDABLE_PORTMODE_NONE;
    confPortMode = OPEN_EXPANDABLE_PORTMODE_NONE;
    memset(peerIntfsStr, 0, sizeof(peerIntfsStr));
    peerIntfs.size = sizeof(peerIntfsStr);
    peerCount = 0;

    /* Interface */
    if ((result = openapiIfNameGet(client_handle, intfNum, &intfName)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get interface name for interface %d. (result = %d).\n",
             intfNum, result);
    }
    else
    {
      printf("  Expandable Interface:%s(%d)\r\n", intfNameStr, intfNum);
    }

    /* Configured Mode */
    if ((result = openapiIntfExpandablePortConfigModeGet(client_handle, intfNum, &confPortMode)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get configured port mode for interface "
             "%s (%d). (result = %d).\n",
             intfNameStr, intfNum, result);
    }
    else
    {
      printf("    Configured Mode:%s(%d)\r\n", expPortModeStr(confPortMode), confPortMode);
    }

    /* Operating Mode */
    if ((result = openapiIntfExpandablePortOperModeGet(client_handle, intfNum, &runnPortMode)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get operating port mode for interface "
             "%s (%d). (result = %d).\n",
             intfNameStr, intfNum, result);
    }
    else
    {
      printf("    Operating Mode:%s(%d)\r\n", expPortModeStr(runnPortMode), runnPortMode);
    }

    /* Peer interfaces and Peer Count */
    portMode = OPEN_EXPANDABLE_PORTMODE_NONE;
    while (OPEN_E_NONE == openapiIntfExpandableModePeersGetByModeGet(client_handle, intfNum, &portMode,
                                                      &peerIntfs, &peerCount))
    {
      printf("    Expanded option(s):%s (%d)\r\n", expPortModeStr(portMode), portMode);
      printf("    Expanded interfaces:%s\r\n", peerIntfsStr);
      printf("    Expanded interfaces count:%d\r\n", peerCount);

      memset(peerIntfsStr, 0, sizeof(peerIntfsStr));
      peerIntfs.size = sizeof(peerIntfsStr);
      peerCount = 0;
    }

    printf("\r\n");
  }while (OPEN_E_NONE == openapiIntfExpandablePortNextGet(client_handle, intfNum, &intfNum));

  return;
}

/*********************************************************************
* @purpose  Set and Get the expandable port configuration.
*
* @param    client_handle   @b{(input)}  client handle from regn API
*
* @returns  none
*
* @end
*********************************************************************/
void setGetExpandablePortMode(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t intfNum = 0;
  open_buffdesc intfName;
  char intfNameStr[OPENAPI_INTERFACE_LIST_MAX_LENGTH];
  OPEN_EXPANDABLE_PORTMODE_t confPortMode = OPEN_EXPANDABLE_PORTMODE_NONE;
  OPEN_EXPANDABLE_PORTMODE_t newPortMode = OPEN_EXPANDABLE_PORTMODE_SINGLE_25G;
  OPEN_BOOL_t isSupported = OPEN_FALSE; 

  memset(intfNameStr, 0, sizeof(intfNameStr));
  intfName.pstart = intfNameStr;
  intfName.size = sizeof(intfNameStr);


  if ((result = openapiIntfExpandablePortNextGet(client_handle, intfNum, &intfNum)) != OPEN_E_NONE)
  {
    printf("\r\nExpandable ports are not supported.\r\n");
    return;
  }

  printf("\r\nSet port mode for expandable interface:%s(%d)\r\n", intfNameStr, intfNum);

  /* Interface */
  if ((result = openapiIfNameGet(client_handle, intfNum, &intfName)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get interface name for interface %d. (result = %d).\n",
           intfNum, result);
  }
  else
  {
    printf("  Expandable Interface:%s(%d)\r\n", intfNameStr, intfNum);
  }

  /* Configured Mode */
  if ((result = openapiIntfExpandablePortConfigModeGet(client_handle, intfNum, &confPortMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get configured port mode for interface "
           "%s (%d). (result = %d).\n",
           intfNameStr, intfNum, result);
  }
  else
  {
    printf("  Current Configured Mode:%s(%d)\r\n", expPortModeStr(confPortMode), confPortMode);
  }

  if ((result = openapiIntfExpandablePortModeSupportedGet(client_handle, newPortMode,
                                                          &isSupported)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to check if the port mode %s(%d) is supported "
           "(result = %d).\n", expPortModeStr(newPortMode), newPortMode, result);
    return;
  }
  else
  {
    if (OPEN_FALSE == isSupported)
    {
      printf("Expandable port mode %s(%d) is not supported.\r\n ", expPortModeStr(newPortMode), newPortMode);
      return;
    } 
  }

  /* Set Configured Mode */
  if ((result = openapiIntfExpandablePortModeSet(client_handle, intfNum,
                                                 newPortMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set port mode %s(%d) for interface "
           "%s (%d). (result = %d).\n", expPortModeStr(newPortMode), newPortMode,
           intfNameStr, intfNum, result);
  }
  else
  {
    printf("Successfully configured the port Mode:%s(%d) on Interface %s (%d).\r\n",
           expPortModeStr(newPortMode), newPortMode, intfNameStr, intfNum);

    confPortMode = OPEN_EXPANDABLE_PORTMODE_NONE;
    if ((result = openapiIntfExpandablePortConfigModeGet(client_handle, intfNum, &confPortMode)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get new configured port mode for interface "
           "%s (%d). (result = %d).\n",
           intfNameStr, intfNum, result);
    }
    else
    {
      printf("  New Configured Mode:%s(%d)\r\n", expPortModeStr(confPortMode), confPortMode);
    }
  }

  return;
}

static char * getFECModeStr(OPEN_PORT_FEC_MODE_t mode)
{
  char *fecMode = "Invalid";

  switch (mode)
  {
    case OPEN_PORT_FEC_ENABLE:
      fecMode = "Enable";
      break;

    case OPEN_PORT_FEC_AUTO:
      fecMode = "Auto";
      break;

    case OPEN_PORT_FEC_BASE_R:
      fecMode = "CL74";
      break;

    case OPEN_PORT_FEC_RS_FEC:
      fecMode = "CL91";
      break;

    case OPEN_PORT_FEC_RS_544:
      fecMode = "RS544";
      break;

    case OPEN_PORT_FEC_RS_272:
      fecMode = "RS272";
      break;

    case OPEN_PORT_FEC_RS_206:
      fecMode = "RS206";
      break;

    case OPEN_PORT_FEC_RS_108:
      fecMode = "RS108";
      break;

    case OPEN_PORT_FEC_RS_545:
      fecMode = "RS545";
      break;

    case OPEN_PORT_FEC_RS_304:
      fecMode = "RS304";
      break;

    case OPEN_PORT_FEC_RS_544_2xN:
      fecMode = "RS544_2xN";
      break;

    case OPEN_PORT_FEC_RS_272_2xN:
      fecMode = "RS272_2xN";
      break;

    case OPEN_PORT_FEC_NOFEC_50G:
      fecMode = "NOFEC_50G";
      break;

    case OPEN_PORT_FEC_DISABLE:
      fecMode = "Disable";
      break;

    case OPEN_PORT_FEC_NOT_AVAILABLE:
      fecMode = "Not Available";
      break;

    default:
      break;

  }

  return fecMode;
}

/*********************************************************************
* @purpose  Get the FEC port configuration.
*
* @param    client_handle   @b{(input)}  client handle from regn API
*
* @returns  none
*
* @end
*********************************************************************/
void displayFECPortInfo(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t intfNum = 0;
  open_buffdesc intfName;
  char intfNameStr[OPENAPI_INTERFACE_LIST_MAX_LENGTH];
  OPEN_PORT_FEC_MODE_t fecConfigMode = OPEN_PORT_FEC_DISABLE;
  OPEN_PORT_FEC_MODE_t fecOperMode = OPEN_PORT_FEC_DISABLE;
  uint64_t portCapability;

  memset(intfNameStr, 0, sizeof(intfNameStr));
  intfName.pstart = intfNameStr;
  intfName.size = sizeof(intfNameStr);

  while (OPEN_E_NONE == openapiIfNextGet(client_handle, OPEN_INTF_TYPE_PHY, intfNum, &intfNum))
  {
    memset(intfNameStr, 0, sizeof(intfNameStr));
    intfName.size = sizeof(intfNameStr);
    fecConfigMode = OPEN_PORT_FEC_DISABLE;
    fecOperMode = OPEN_PORT_FEC_DISABLE;
    portCapability = 0;

    if ((OPEN_E_NONE == openapiIfCapabilitiesGet(client_handle, intfNum, &portCapability)) &&
        ((portCapability) & (OPEN_PHY_CAP_FEC_SUPPORT)))
    {
      printf("\r\n");

      /* Interface */
      if ((result = openapiIfNameGet(client_handle, intfNum, &intfName)) != OPEN_E_NONE)
      {
        printf("Bad return code trying to get interface name for interface %d. (result = %d).\n",
               intfNum, result);
      }
      else
      {
        printf("Interface:%s(%d)\r\n", intfNameStr, intfNum);
      }

      /* Configured and Operational Mode */
      if ((result = openapiIntfFECModeGet(client_handle, intfNum, &fecConfigMode, &fecOperMode)) != OPEN_E_NONE)
      {
        printf("Bad return code trying to get FEC mode for interface "
               "%s (%d). (result = %d).\n",
               intfNameStr, intfNum, result);
      }
      else
      {
        printf("  Configured Mode:%s(%d)\r\n", getFECModeStr(fecConfigMode), fecConfigMode);
        printf("  Operational Mode:%s(%d)\r\n", getFECModeStr(fecOperMode), fecOperMode);
      }
     
      printf("\r\n");
    }
  }

  return;
}

/*********************************************************************
* @purpose  Set the FEC port configuration.
*
* @param    client_handle   @b{(input)}  client handle from regn API
*
* @returns  none
*
* @end
*********************************************************************/
void setGetFECPortInfo(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t intfNum = 0;
  open_buffdesc intfName;
  char intfNameStr[OPENAPI_INTERFACE_LIST_MAX_LENGTH];
  OPEN_PORT_FEC_MODE_t fecConfigMode = OPEN_PORT_FEC_DISABLE;
  OPEN_PORT_FEC_MODE_t fecOperMode = OPEN_PORT_FEC_DISABLE;
  uint64_t portCapability;

  memset(intfNameStr, 0, sizeof(intfNameStr));
  intfName.pstart = intfNameStr;
  intfName.size = sizeof(intfNameStr);

  while (OPEN_E_NONE == openapiIfNextGet(client_handle, OPEN_INTF_TYPE_PHY, intfNum, &intfNum))
  {
    memset(intfNameStr, 0, sizeof(intfNameStr));
    intfName.size = sizeof(intfNameStr);
    fecConfigMode = OPEN_PORT_FEC_DISABLE;
    fecOperMode = OPEN_PORT_FEC_DISABLE;
    portCapability = 0;

    if ((OPEN_E_NONE == openapiIfCapabilitiesGet(client_handle, intfNum, &portCapability)) &&
        ((portCapability) & (OPEN_PHY_CAP_FEC_SUPPORT)))
    {
      printf("\r\n");

      /* Interface */
      if ((result = openapiIfNameGet(client_handle, intfNum, &intfName)) != OPEN_E_NONE)
      {
        printf("Bad return code trying to get interface name for interface %d. (result = %d).\n",
               intfNum, result);
      }
      else
      {
        printf("Interface:%s(%d)\r\n", intfNameStr, intfNum);
      }

      if ((result = openapiIntfFECModeSet(client_handle, intfNum, OPEN_PORT_FEC_DISABLE)) != OPEN_E_NONE)
      {
        printf("Bad return code trying to set FEC mode for interface "
               "%s (%d). (result = %d).\n",
               intfNameStr, intfNum, result);
      }
      else
      {
        printf("FEC mode is set successfully for interface "
               "%s (%d).\n", intfNameStr, intfNum);

        /* Configured and Operational Mode */
        if ((result = openapiIntfFECModeGet(client_handle, intfNum, &fecConfigMode, &fecOperMode)) != OPEN_E_NONE)
        {
          printf("Bad return code trying to get FEC mode for interface "
                 "%s (%d). (result = %d).\n",
                 intfNameStr, intfNum, result);
        }
        else
        {
          printf("  Configured Mode:%s(%d)\r\n", getFECModeStr(fecConfigMode), fecConfigMode);
          printf("  Operational Mode:%s(%d)\r\n", getFECModeStr(fecOperMode), fecOperMode);
        }
      }
      printf("\r\n");
      break;
    }
  }

  return;
}
/*******************************************************************
*
* @brief  This is the main function that will demonstrate 
*         Interface OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure
*
*********************************************************************/

int
main (int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  uint32_t IfNum, mtu, mtu2, efs;
  open_error_t result;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  open_buffdesc intfName;
  char intfNameStr[256];
  char buf[128];
  open_buffdesc mac;
  OPEN_PORT_SPEEDS_t speed;
  uint32_t duplex;
  uint32_t autoneg;
  uint32_t value = 0;
  OPEN_CONTROL_t admin_state;
  uint64_t portCapability;
  OPEN_DISABLE_REASON_t reason;
  OPEN_SPECIAL_PORT_TYPE_t specialPortType;
  OPEN_PORT_CABLE_MEDIA_TYPE_t mediaType = OPEN_PORT_CABLE_MEDIA_TYPE_NONE;
  OPEN_PORT_STATE_t val = OPEN_PORT_DISABLE;
  OPEN_SFP_MODULE_TYPE_t moduleType = OPEN_MODULE_NONE;
  OPEN_BOOL_t rxStatus, txStatus;
  uint32_t allIntfCount = 0, rxTimeout;
  OPEN_CONTROL_t linkTrapMode = OPEN_DISABLE;
  openapiClientRpcStats_t stats;

  l7proc_crashlog_register ();

  /* Register with OpEN */
  if ((result =
       openapiClientRegister ("interface_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf ("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit (2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck (&clientHandle) != OPEN_E_NONE)
  {
    sleep (1);
  }

  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Starting Interface API example application");

  printf ("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof (switch_os_revision_string);
  if (openapiNetworkOSVersionGet (&clientHandle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf ("Network OS version = %s\n", switch_os_revision_string);
  }
  else
  {
    printf ("Network OS version retrieve error\n");
  }

  if ((result = openapiAllIntfNumGet(&clientHandle, &allIntfCount)) != OPEN_E_NONE)
  {
    printf("\nBad return code trying to get the count of all interfaces (result = %d)\n",
           result);
  }
  else
  {
    printf ("Number of all interfaces: %u\n", allIntfCount);
  }

  printf ("\n");
  if ((result = openapiClientReceiveTimeoutGet(&clientHandle, 
                                               &rxTimeout)) != OPEN_E_NONE)
  {
    printf("\nBad return code trying to get the socket receive timeout."
           " (result = %d)\n", result);
  }
  else
  {
    printf ("Current socket receive timeout is : %u\n", rxTimeout);
  }

  /* Setting socket receive time out */
  if ((result = openapiClientReceiveTimeoutSet(&clientHandle, 
                                               10)) != OPEN_E_NONE)
  {
    printf("\nBad return code trying to set the socket receive timeout."
           " (result = %d)\n", result);
  }
  else
  {
    printf ("Socket receive timeout set successfully.\n");
  }

  if ((result = openapiClientReceiveTimeoutGet(&clientHandle, 
                                               &rxTimeout)) != OPEN_E_NONE)
  {
    printf("\nBad return code trying to get the socket receive timeout."
           " (result = %d)\n", result);
  }
  else
  {
    printf ("Socket receive timeout is : %u\n", rxTimeout);
  }

  /* Reset to default */
  if ((result = openapiClientReceiveTimeoutSet(&clientHandle, 
                                               0)) != OPEN_E_NONE)
  {
    printf("\nBad return code trying to set the socket receive timeout."
           " (result = %d)\n", result);
  }
  else
  {
    printf ("Socket receive timeout set successfully.\n");
  }

  /* Get openapi client RPC statistics */
  if ((result = openapiClientRpcStatsGet(&stats)) != OPEN_E_NONE)
  {
    printf("\nBad return code trying to get the client RPC statistics."
           " (result = %d)\n", result);
  }
  else
  {
    printf ("Sent msg success count %u\n", stats.msendok);  
    printf ("Received msg success count %u\n", stats.mrecvok);  
    printf ("Nowait option usage count %u\n", stats.mnowait);  
    printf ("Failed msg send attempts %u\n", stats.msendfail);  
    printf ("Failed msg recv attempts %u\n", stats.mrecvfail);  
    printf ("Msg recv sequence number mismatch %u\n", stats.mrecvseqerr);  
    printf ("Msg recv timed out %u\n", stats.mrecvtimeout);  
    printf ("Truncated recv msgs %u\n", stats.mrecvtrunc);  
    printf ("Failed connection attempts %u\n", stats.mconnfail);  
  }

  /* Print RPC client statistics on console/stdout */
  (void) openapiClientStatsPrint();

  printf ("\n");

  /* exercise various OPEN API interface functions */
  do
  { 
    if ((result = openapiIfFirstGet (&clientHandle, OPEN_INTF_TYPE_PHY,
                                     &IfNum)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get first physical interface. (result = %d)\n",result);
      break;
    }

    intfName.pstart = intfNameStr;
    intfName.size = sizeof(intfNameStr);
    if ((result = openapiIfNameGet(&clientHandle, IfNum, &intfName)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get first physical interface name. (result = %d)\n",result);
      break;
    }

    printf("Using physical interface %d, name %s\n", IfNum, (char *)intfName.pstart);

    if ((result = openapiIfAdminStateGet(&clientHandle, IfNum, &admin_state)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get admin state physical interface. (result = %d)\n",result);
    }
    else
    {
      memset (buf, 0, sizeof(buf));
      if (admin_state == OPEN_ENABLE) sprintf(buf,"ENABLE");
      else sprintf(buf,"DISABLE");
      printf("physical interface %d get admin-state : %s\n", IfNum, buf);
    }

    admin_state = (admin_state == OPEN_ENABLE) ? OPEN_DISABLE : OPEN_ENABLE;

    if ((result = openapiIfAdminStateSet(&clientHandle, IfNum, admin_state)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to set admin state physical interface. (result = %d)\n",result);
    }

    if ((result = openapiIfAdminStateGet(&clientHandle, IfNum, &admin_state)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get admin state physical interface. (result = %d)\n",result);
    }
    else
    {
      memset (buf, 0, sizeof(buf));
      if (admin_state == OPEN_ENABLE) sprintf(buf,"ENABLE");
      else sprintf(buf,"DISABLE");
      printf("physical interface %d set admin-state : %s\n", IfNum, buf);
    }

    if ((result = openapiIfSpeedGet(&clientHandle, IfNum, &speed)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get speed of  physical interface. (result = %d)\n",result);
    }
    else
    {
      memset (buf, 0, sizeof(buf));
      switch (speed)
      {
        case OPEN_PORTCTRL_PORTSPEED_AUTO_NEG:
          sprintf(buf,"AUTO");
          break;
        case OPEN_PORTCTRL_PORTSPEED_HALF_100TX:
          sprintf(buf,"100 half");
          break;
        case OPEN_PORTCTRL_PORTSPEED_FULL_100TX:
          sprintf(buf,"100 full");
          break;
        case OPEN_PORTCTRL_PORTSPEED_HALF_10T:
          sprintf(buf,"10 half");
          break;
        case OPEN_PORTCTRL_PORTSPEED_FULL_10T:
          sprintf(buf,"10 full");
          break;
        case OPEN_PORTCTRL_PORTSPEED_FULL_100FX:
          sprintf(buf,"100 full");
          break;
        case OPEN_PORTCTRL_PORTSPEED_FULL_1000SX:
          sprintf(buf,"1000 full");
          break;
        case OPEN_PORTCTRL_PORTSPEED_FULL_10GSX:
          sprintf(buf,"10G full");
          break;
        case OPEN_PORTCTRL_PORTSPEED_FULL_40GSX:
          sprintf(buf,"40G full");
          break;
        default:
          sprintf(buf,"UNKNOWN");
          break;
      }
      printf("physical interface %d speed %s\n",IfNum,buf);
    }

    if ((result = openapiIfDuplexModeGet(&clientHandle, IfNum, &duplex)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get duplex mode of  physical interface. (result = %d)\n",result);
    }
    else
    {
      memset (buf, 0, sizeof(buf));
      switch (duplex)
      {
        case OPEN_PORT_DUPLEX_HALF:
          sprintf(buf,"half-duplex");
          break;
        case OPEN_PORT_DUPLEX_FULL:
          sprintf(buf,"full-duplex");
          break;
        case OPEN_PORT_DUPLEX_UNKNOWN:
          sprintf(buf,"UNKNOWN");
          break;
        default:
          sprintf(buf,"UNKNOWN");
          break;
      }
      printf("physical interface %d duplex mode %s\n",IfNum,buf);
    }

    if ((result = openapiIfAutoNegGet(&clientHandle, IfNum, &autoneg)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get auto-negotiation mode of  physical interface. (result = %d)\n",result);
    }
    else
    {
      memset (buf, 0, sizeof(buf));
      if (autoneg == OPEN_PORT_AUTO_NEG_ENABLE)
        sprintf(buf,"ENABLE");
      else if(autoneg == OPEN_PORT_AUTO_NEG_DISABLE)
        sprintf(buf,"DISABLE");
      else
        sprintf(buf,"UNKNOWN");

      printf("physical interface %d auto-negotiation mode : %s\n",IfNum,buf);
    }

    printf("setting auto-negotiation to DISABLE on interface %d \n",IfNum);
    autoneg = OPEN_PORT_AUTO_NEG_DISABLE;
    if ((result = openapiIfAutoNegSet(&clientHandle, IfNum, autoneg)) != OPEN_E_NONE)
    {
      if (result == OPEN_E_UNAVAIL)
      {
        printf(" Cannot set auto-negotiation on an unsupported interface. (result = %d)\n", result);   
      }
      else
      {
        printf("  Bad return code trying to set auto-negotiation mode of  physical interface. (result = %d)\n",result);
      }
    }
    else
    {
      sleep(5);
      if ((result = openapiIfAutoNegGet(&clientHandle, IfNum, &autoneg)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get auto-negotiation mode of  physical interface. (result = %d)\n",result);
      }
      else
      {
        memset (buf, 0, sizeof(buf));
        if (autoneg == OPEN_PORT_AUTO_NEG_ENABLE)
          sprintf(buf,"ENABLE");
        else if(autoneg == OPEN_PORT_AUTO_NEG_DISABLE)
          sprintf(buf,"DISABLE");
        else
          sprintf(buf,"UNKNOWN");

        printf("physical interface %d auto-negotiation mode : %s\n",IfNum,buf);
      }

      printf("setting the speed to 100M full-duplex on interface %d\n",IfNum);
      speed = OPEN_PORTCTRL_PORTSPEED_FULL_100TX;
      if ((result = openapiIfSpeedSet(&clientHandle, IfNum, speed)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to set speed of  physical interface. (result = %d)\n",result);
      }
      else
      {
        sleep(5);
        if ((result = openapiIfSpeedGet(&clientHandle, IfNum, &speed)) != OPEN_E_NONE)
        {
          printf("  Bad return code trying to get speed of  physical interface. (result = %d)\n",result);
        }
        else
        {
          memset (buf, 0, sizeof(buf));
          switch (speed)
          {
            case OPEN_PORTCTRL_PORTSPEED_AUTO_NEG:
              sprintf(buf,"AUTO");
              break;
            case OPEN_PORTCTRL_PORTSPEED_HALF_100TX:
              sprintf(buf,"100 half");
              break;
            case OPEN_PORTCTRL_PORTSPEED_FULL_100TX:
              sprintf(buf,"100 full");
              break;
            case OPEN_PORTCTRL_PORTSPEED_HALF_10T:
              sprintf(buf,"10 half");
              break;
            case OPEN_PORTCTRL_PORTSPEED_FULL_10T:
              sprintf(buf,"10 full");
              break;
            case OPEN_PORTCTRL_PORTSPEED_FULL_100FX:
              sprintf(buf,"100 full");
              break;
            case OPEN_PORTCTRL_PORTSPEED_FULL_1000SX:
              sprintf(buf,"1000 full");
              break;
            case OPEN_PORTCTRL_PORTSPEED_FULL_10GSX:
              sprintf(buf,"10G full");
              break;
            case OPEN_PORTCTRL_PORTSPEED_FULL_40GSX:
              sprintf(buf,"40G full");
              break;
            default:
              sprintf(buf,"unknown");
              break;
          }
          printf("physical interface %d speed %s\n",IfNum,buf);
        }
      }
    }

    if ((result = openapiIfEthernetHeaderSizeGet(&clientHandle, IfNum, &efs)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get ethernet frame size of physical interface. (result = %d)\n",result);
    }
    else
    {
      printf("physical interface %d ethernet header size : %d\n", IfNum, efs);
    }

    if ((result = openapiIfMtuSizeGet(&clientHandle, IfNum, &mtu)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get mtu size of physical interface. (result = %d)\n",result);
    }
    else
    {
      printf("physical interface %d MTU size : %d\n", IfNum, mtu);
    }

    mtu = 9216;
    printf("setting the MTU to %d\n", mtu);
    if ((result = openapiIfMtuSizeSetV2(&clientHandle, IfNum, mtu)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to set mtu size of  physical interface. (result = %d)\n",result);
    }
    else
    {
      sleep(5);
      if ((result = openapiIfMtuSizeGet(&clientHandle, IfNum, &mtu2)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get mtu size of  physical interface. (result = %d)\n",result);
      }
      else
      {
        printf("physical interface %d MTU size : %d\n",IfNum, mtu2);
        printf("MTU is%sequal to the set MTU\n", (mtu == mtu2 ? " " : " not "));
      }
    }
    
    mtu = 8148;
    printf("setting the max frame size to %d\n", mtu);
    if ((result = openapiIfMaxFrameSizeSet(&clientHandle, IfNum, mtu)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to set max frame size of physical interface. (result = %d)\n",result);
    }
    else
    {
      sleep(5);
      if ((result = openapiIfMtuSizeGet(&clientHandle, IfNum, &mtu2)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get mtu size of physical interface. (result = %d)\n",result);
      }
      else
      {
        printf("physical interface %d MTU size : %d\n",IfNum,mtu2);
      }
      if ((result = openapiIfMaxFrameSizeGet(&clientHandle, IfNum, &mtu2)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get max frame size of physical interface. (result = %d)\n",result);
      }
      else
      {
        printf("physical interface %d max frame size : %d\n", IfNum, mtu2);
        printf("Max frame size is%sequal to the set max frame size\n", (mtu == mtu2 ? " " : " not "));
      }
    }

    memset (buf, 0, sizeof(buf));
    mac.pstart = buf;
    mac.size = sizeof(buf);
    if ((result = openapiIfMacAddrGet(&clientHandle, IfNum, &mac)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get mac-address of  physical interface. (result = %d)\n",result);
    }
    else
    {
      printf("physical interface %d mac-address : %s\n",IfNum,buf);
    }

    memset (buf, 0, sizeof(buf));
    mac.pstart = buf;
    mac.size = sizeof(buf);
    if ((result = openapiIfDescrGet(&clientHandle, IfNum, &mac)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get description of  physical interface. (result = %d)\n",result);
    }
    else
    {
      printf("physical interface %d description : %s\n",IfNum,buf);
    }

    memset (buf, 0, sizeof(buf));
    sprintf(buf,"test_port_%d_description",IfNum);
    mac.pstart = buf;
    mac.size = strlen(buf) + 1;
    printf("setting the interface %d description to %s\n",IfNum,buf);
    if ((result = openapiIfDescrSet(&clientHandle, IfNum, &mac)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to set description of  physical interface. (result = %d)\n",result);
    }
    else
    {
      memset (buf, 0, sizeof(buf));
      mac.pstart = buf;
      mac.size = sizeof(buf);
      if ((result = openapiIfDescrGet(&clientHandle, IfNum, &mac)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get description of  physical interface. (result = %d)\n",result);
      }
      else
      {
        printf("physical interface %d description : %s\n",IfNum,buf);
      }
    }

    if ((result = openapiIfDisableReasonGet(&clientHandle, IfNum, &reason)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get disable reason. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d disable reason: %d (%s)\n", IfNum, reason, reasonToStr(reason));
    }

    if ((result = openapiIfSpecialPortTypeGet(&clientHandle, IfNum, &specialPortType)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get special port type. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d special port type designation: %d (%s)\n", IfNum, specialPortType, specialPortTypeToStr(specialPortType));
    }

    if ((result = openapiIfActiveMediaTypeGet(&clientHandle, IfNum, &mediaType)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get active media type. result = %d\n", result);
    }
    else if ((mediaType == OPEN_PORT_CABLE_MEDIA_TYPE_FIBER) &&
             ((result = openapiBoxsSfpTypeGet(&clientHandle, IfNum, &moduleType)) != OPEN_E_NONE))
    {
      printf("Bad return code trying to get SFP type. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d active media type: %d (%s)\n", IfNum, mediaType, mediaTypeToStr(mediaType, moduleType));
    }

    if ((result = openapiIfPauseStatusGet(&clientHandle, IfNum, &rxStatus, &txStatus)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get pause status. result = %d\n", result);
    }
    else
    {
      memset (buf, 0, sizeof(buf));
      if ((rxStatus == OPEN_TRUE) || (txStatus == OPEN_TRUE))
      {
        sprintf(buf,"ACTIVE");
      }
      else
      {
        sprintf(buf,"INACTIVE");
      }
      printf("Physical interface %d flow control status: %s\n", IfNum, buf);
    }

    printf("--- interface statistic---\n");
    sleep(2);

    if ((result = openapiIfLastChangeGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get time of last link state change. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Time of last link state change: %d\n", IfNum, value);
    }
    value = 0;
    if ((result = openapiIfInOctetsGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get total number of octets. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Total number of octets: %d\n", IfNum, value);
    }
    value = 0;
    if ((result = openapiIfInUcastPktsGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get number of packets, delivered by this sub-layer to a higher sub-layer. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Number of packets, delivered by this sub-layer to a higher sub-layer: %d\n", IfNum, value);
    }

    value = 0;
    if ((result = openapiIfInDiscardsGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get number of inbound packets which were chosen to be discarded. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Number of inbound packets which were chosen to be discarded: %d\n", IfNum, value);
    }

    value = 0;
    if ((result = openapiIfInErrorsGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get In errors counter. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d In errors counter: %d\n", IfNum, value);
    }

    value = 0;
    if ((result = openapiIfInUnknownProtosGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get number of packets received via the interface \nwhich were discarded because of an unknown or unsupported protocol. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Number of packets received via the interface \nwhich were discarded because of an unknown or unsupported protocol: %d\n", IfNum, value);
    }
   
    value = 0;
    if ((result = openapiIfOutOctetsGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get total number of octets transmitted out of the interface, including framing characters. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Total number of octets transmitted out of the interface, including framing characters: %d\n", IfNum, value);
    }

    value = 0;
    if ((result = openapiIfOutUcastPktsGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get total number of packets that higher-level protocols requested be transmitted. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Total number of packets that higher-level protocols requested be transmitted: %d\n", IfNum, value);
    }

    value = 0;
    if ((result = openapiIfOutDiscardsGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get number of outbound packets which were chosen to be discarded. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Number of outbound packets which were chosen to be discarded: %d\n", IfNum, value);
    }

    value = 0;
    if ((result = openapiIfOutDiscardsGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get number of outbound packets which were chosen to be discarded. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Number of outbound packets which were chosen to be discarded: %d\n", IfNum, value);
    }

    value = 0;
    if ((result = openapiIfOutErrorsGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get out errors counter. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Out errors counter: %d\n", IfNum, value);
    }

    if ((result = openapiIfCapabilitiesGet(&clientHandle, IfNum, &portCapability)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get port capabilities. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d is capable of following speeds: \n", IfNum);
      getPhysicalModes(portCapability);
    }

    if ((result = openapiIfOprStateGet(&clientHandle, IfNum, &val)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get operational port state. result = %d\n", result);
    }  
    else
    {
      memset (buf, 0, sizeof(buf));
      if (val == OPEN_PORT_ENABLE)
      {
        snprintf(buf, sizeof(buf), "ENABLE");
      }  
      else if (val == OPEN_PORT_DISABLE)
      {
        snprintf(buf, sizeof(buf), "DISABLE");
      } 
      else
      {
        snprintf(buf, sizeof(buf), "DIAG_DISABLE");
      } 

      printf("Physical interface %d operational state %s\n", IfNum, buf);
    }

    configIdMaskOffsetGet(&clientHandle, IfNum, &value);
    ifIndexGet(&clientHandle, IfNum, &value);
    ifActiveStateGet(&clientHandle, IfNum, &value);

    if ((result = openapiIfLinkTrapGet(&clientHandle, IfNum, &linkTrapMode)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get the link trap mode on interface %s(%d). "
             "result = %d\n", intfNameStr, IfNum, result);
    }
    else
    {
      printf("Successfully fetched the link trap mode on interface %s(%d) Mode:%s.\n",
             intfNameStr, IfNum, (linkTrapMode == OPEN_ENABLE) ? "Enable" :"Disable" );
    }

    linkTrapMode = (linkTrapMode == OPEN_ENABLE) ? OPEN_DISABLE : OPEN_ENABLE;
    if ((result = openapiIfLinkTrapSet(&clientHandle, IfNum, linkTrapMode)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to set the link trap mode on interface %s(%d). "
             "result = %d\n", intfNameStr, IfNum, result);
    }
    else
    {
      printf("Successfully configured the link trap mode on interface %s(%d) Mode:%s.\n",
             intfNameStr, IfNum, (linkTrapMode == OPEN_ENABLE) ? "Enable" :"Disable" );
    }
  
    if ((result = openapiIfLinkTrapGet(&clientHandle, IfNum, &linkTrapMode)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get the link trap mode on interface %s(%d). "
             "result = %d\n", intfNameStr, IfNum, result);
    }
    else
    {
      printf("Successfully fetched the link trap mode on interface %s(%d) Mode:%s.\n",
             intfNameStr, IfNum, (linkTrapMode == OPEN_ENABLE) ? "Enable" :"Disable" );
    }

    break;
  }
  while (0);


  /* Display the expandable port details if supported. */
  displayExpandablePortInfo(&clientHandle);

  /* Configure the expandable port mode */
  setGetExpandablePortMode(&clientHandle);

  /* Display the FEC port details if supported. */
  displayFECPortInfo(&clientHandle);

  /* Display the FEC port details if supported. */
  setGetFECPortInfo(&clientHandle);

  /* Log goodbye message with OPEN */
  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0,
               "Stopping Interface API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
