/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  ip6map_example.c
*
* @purpose   OpEN IP6Map example.
*
* @component OpEN
*
* @create    01/10/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>
#include <arpa/inet.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_ip6map.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the IP6Map Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Get the administrative mode of the router for given VRF: %s 1 <vrfName>\n", name);
  printf("Test 2: Set the administrative mode of the router for given VRF: %s 2 <vrfName> <mode>\n", name);
  printf("Test 3: Get the dynamic renewal mode of IPv6 neighbors in a given VRF: %s 3 <vrfName>\n", name);
  printf("Test 4: Set the dynamic renewal mode of IPv6 neighbors in a given VRF: %s 4 <vrfName> <mode>\n", name);
  printf("Test 5: Get the NUD backoff-multiple value set for calculation of timeouts for NS transmissions during NUD for a given VRF: %s 5 <vrfName>\n", name);
  printf("Test 6: Set the NUD backoff-multiple value for calculation of timeouts for NS transmissions during NUD for a given VRF: %s 6 <vrfName> <val>\n", name);
  printf("Test 7: Get the maximum unicast neighbor solicitations sent during NUD for a given VRF: %s 7 <vrfName>\n", name);
  printf("Test 8: Set the maximum unicast neighbor solicitations sent during NUD for a given VRF: %s 8 <vrfName> <val>\n", name);
  printf("Test 9: Get the maximum multicast neighbor solicitations sent during NUD for a given VRF: %s 9 <vrfName>\n", name);
  printf("Test 10: Set the maximum multicast neighbor solicitations sent during NUD for a given VRF: %s 10 <vrfName> <val>\n", name);
  printf("Test 11: Get the rate limit value set for the unresolved IPv6 data packets coming to CPU: %s 11\n", name);
  printf("Test 12: Set the rate limit value for the unresolved IPv6 data packets coming to CPU: %s 12 <val>\n", name);
  printf("Test 13: Get the default IPv6 router TTL value for a given VRF: %s 13 <vrfName>\n", name);
  printf("Test 14: Set the default IPv6 router TTL value for a given VRF: %s 14 <vrfName> <val>\n", name);
  printf("Test 15: Get the administrative state of a given interface: %s 15 <ifNum>\n", name);
  printf("Test 16: Get the operational state of a given interface: %s 16 <ifNum>\n", name);
  printf("Test 17: Set the administrative state of a given interface: %s 17 <ifNum> <mode>\n", name);
  printf("Test 18: Get the IPv6 AutoState mode of a given interface: %s 18 <ifNum>\n", name);
  printf("Test 19: Set the IPv6 AutoState mode of a given interface: %s 19 <ifNum> <mode>\n", name);
  printf("Test 20: Determine if IPv6 interface exists: %s 20 <ifNum>\n", name);
  printf("Test 21: Determine if the interface is valid for participation in the IPv6 component: %s 21 <ifNum>\n", name);
  printf("Test 22: Determine if the interface is valid to be configured with an IPv6 neighbor: %s 22 <ifNum>\n", name);
  printf("Test 23: Get the IPv6 MTU of a given interface: %s 23 <ifNum>\n", name);
  printf("Test 24: Get the IPv6 effective MTU of a given interface: %s 24 <ifNum>\n", name);
  printf("Test 25: Set the IPv6 MTU of a given interface: %s 25 <ifNum> <val>\n", name);
  printf("Test 26: Get the interface bandwidth of a given interface: %s 26 <ifNum>\n", name);
  printf("Test 27: Get the first valid interface for participation in IPv6 component: %s 27\n", name);
  printf("Test 28: Get the next valid interface for participation in IPv6 component: %s 28 <prevIfNum>\n", name);
  printf("Test 29: Get the maximum IP MTU that may be set on an interface: %s 29 <ifNum>\n", name);
  printf("Test 30: Get IPv6 statistics for the given interface: %s 30 <intIfNum>\n", name);
  printf("Test 31: Get IPv6 ICMP statistics for the given interface: %s 31 <intIfNum>\n", name);
  printf("Test 32: Clear IPv6 statistics for the given interface: %s 32 <intIfNum>\n", name);
  printf("Test 33: Get total of each IPv6 statistic on all interfaces: %s 33 \n", name);
  printf("Test 34: Clear IPv6 statistics on all interfaces: %s 34 \n", name);
  printf("Test 35: Get the IPv6 Route table: %s 35 <vrfName> <bestRouteOnly (0-Flase, 1-True)>\n", name);
  printf("Test 36: Get the number of routes in the routing table: %s 36 <vrfName> <bestRouteOnly (0-Flase, 1-True)>\n", name);
  printf("Test 37: Get the configured IPv6 addresses on the interface: %s 37 <intIfNum> <numAddr>\n", name);
  printf("Test 38: Get the number of neighbor solicits configured to be set for duplicate address detection (DAD) on an interface: %s 38 <intIfNum>\n", name);
  printf("Test 39: Set the number of neighbor solicits to be sent for duplicate address detection (DAD) on an interface: %s 39 <intIfNum> <val>\n", name);
  printf("Test 40: Get the interval between IPv6 neighbor solicitation retransmissions on an interface: %s 40 <intIfNum>\n", name);
  printf("Test 41: Set the interval between IPv6 neighbor solicitation retransmissions on an interface: %s 41 <intIfNum> <val>\n", name);
  printf("Test 42: Get the amount of time that a remote IPv6 node is reachable: %s 42 <intIfNum>\n", name);
  printf("Test 43: Set the amount of time that a remote IPv6 node is reachable: %s 43 <intIfNum> <val>\n", name);
  printf("Test 44: Get the configured mode for ICMPv6 unreachable messages: %s 44 <intIfNum>\n", name);
  printf("Test 45: Set the configured mode for ICMPv6 unreachable messages: %s 45 <intIfNum> <mode (0-Disable, 1-Enable)>\n", name);
  printf("Test 46: Get the IPv6 address details: %s 46 <intIfNum> <ipv6Address>\n", name);
  printf("Test 47: Get the IPv6 address details of all the entries: %s 47\n", name);
  printf("Test 48: Get the IPv6 address prefix details: %s 48 <intIfNum> <ipv6Address> <pfxLen>\n", name);
  printf("Test 49: Get the IPv6 addresses of the interface: %s 49 <intIfNum> <numAddr>\n", name);
  printf("Test 50: Get the IPv6 address of the interface: %s 50 <intIfNum>\n", name);
  printf("Test 51: Get the IPv6 neighbor entries: %s 51 <vrfName> <intIfNum>\n", name);
  printf("Test 52: Flush the IPv6 neighbor entries: %s 52 <vrfName> <intIfNum>\n", name);
  printf("Test 53: Get the static IPv6 routes from the routing table: %s 53\n", name);

  return;
}

/*********************************************************************
* @purpose  Get the IPv6 administrative mode of the router for given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    vrfName        @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrRtrAdminModeGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t adminMode = OPEN_DISABLE; /* OPEN_DISABLE/OPEN_ENABLE */
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrRtrAdminModeGet(client_handle, &vrfNameBufd, &adminMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 administrative mode of the router for given VRF is %u (1-Enabled, 0-Disabled).\n", adminMode);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the IPv6 administrative mode of the router for given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the IPv6 administrative mode of the router for given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    vrfName        @b{(input)}  VRF name
* @param    mode           @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrRtrAdminModeSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t mode)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t openMode = OPEN_DISABLE;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  if (1 < mode)
  {
    printf("\nERROR: Invalid mode value. Expected 0(for Disable) or 1(for Enable).\n");
    return;
  }

  if (1 == mode)
  {
    openMode = OPEN_ENABLE;
  }

  result = openapiIp6VrRtrAdminModeSet(client_handle, &vrfNameBufd, openMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the IPv6 administrative mode of the router for given VRF.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the IPv6 administrative mode of the router for given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the dynamic renewal mode of IPv6 neighbors in a given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    vrfName        @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrNbrsDynRenewGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t adminMode = OPEN_DISABLE; /* OPEN_DISABLE/OPEN_ENABLE */
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrNbrsDynRenewGet(client_handle, &vrfNameBufd, &adminMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe dynamic renewal mode of IPv6 neighbors for default VRF is %u (1-Enabled, 0-Disabled).\n", adminMode);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the dynamic renewal mode of IPv6 neighbors in a given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the dynamic renewal mode of IPv6 neighbors in a given VRF.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vrfName       @b{(input)}  VRF name
* @param    mode          @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrNbrsDynRenewSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t mode)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t openMode = OPEN_DISABLE;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  if (1 < mode)
  {
    printf("\nERROR: Invalid mode value. Expected 0(for Disable) or 1(for Enable).\n");
    return;
  }

  if (1 == mode)
  {
    openMode = OPEN_ENABLE;
  }

  result = openapiIp6VrNbrsDynRenewSet(client_handle, &vrfNameBufd, openMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the dynamic renewal mode of IPv6 neighbors in the given VRF.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the dynamic renewal mode of IPv6 neighbors in a given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the NUD backoff-multiple value set for calculation of
*           timeouts for NS transmissions during NUD for a given VRF.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrNudBackoffMultipleGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t val = 0; /* The backoff multiple value */

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrNudBackoffMultipleGet(client_handle, &vrfNameBufd, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe NUD backoff-multiple value set for calculation of timeouts for NS transmissions during NUD for given VRF is %u.\n", val);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get NUD backoff-multiple value set for calculation of timeouts for NS transmissions during NUD for a given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the NUD backoff-multiple value for calculation of timeouts
*           for NS transmissions during NUD for a given VRF.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vrfName       @b{(input)}  VRF name
* @param    val           @b{(input)}  The backoff multiple value
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrNudBackoffMultipleSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t val)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrNudBackoffMultipleSet(client_handle, &vrfNameBufd, val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the NUD backoff-multiple value for calculation of timeouts for NS transmissions during NUD for the given VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the NUD backoff-multiple value for calculation of timeouts for NS transmissions during NUD for given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the maximum unicast neighbor solicitations sent during NUD for a given VRF.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrNudMaxUnicastSolicitsGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t val = 0; /* The maximum unicast solicits value */

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrNudMaxUnicastSolicitsGet(client_handle, &vrfNameBufd, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe maximum unicast neighbor solicitations sent during NUD for the given VRF is %u.\n", val);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the maximum unicast neighbor solicitations sent during NUD for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the maximum unicast neighbor solicitations sent during NUD for a given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    vrfName        @b{(input)}  VRF name
* @param    val            @b{(input)}  The maximum unicast solicits value
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrNudMaxUnicastSolicitsSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t val)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrNudMaxUnicastSolicitsSet(client_handle, &vrfNameBufd, val);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the maximum unicast neighbor solicitations sent during NUD for the given VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the maximum unicast neighbor solicitations sent during NUD for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the maximum multicast neighbor solicitations sent during NUD for a given VRF.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vrfName       @b{(input)}  VRF name
* @param    val           @b{(output)} The maximum multicast solicits value
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrNudMaxMcastSolicitsGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t val = 0; /* The maximum multicast solicits value */

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrNudMaxMcastSolicitsGet(client_handle, &vrfNameBufd, &val);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe maximum multicast neighbor solicitations sent during NUD for the given VRF is %u.\n", val);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the maximum multicast neighbor solicitations sent during NUD for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the maximum multicast neighbor solicitations sent during NUD for a given VRF.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vrfName       @b{(input)}  VRF name
* @param    val           @b{(input)}  The maximum multicast solicits value
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrNudMaxMcastSolicitsSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t val)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrNudMaxMcastSolicitsSet(client_handle, &vrfNameBufd, val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the maximum multicast neighbor solicitations sent during NUD for the given VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the maximum multicast neighbor solicitations sent during NUD for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the rate limit value set for the unresolved IPv6 data packets coming to CPU.
*
* @param    client_handle @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void ip6UnresolvedRateLimitGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t val = 0; /* The bandwidth value in bytes */

  result = openapiIp6UnresolvedRateLimitGet(client_handle, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe rate limit value set for the unresolved IPv6 data packets coming to CPU is %u.\n", val);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the rate limit value set for the unresolved IPv6 data packets coming to CPU. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the rate limit value for the unresolved IPv6 data packets coming to CPU.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    val           @b{(input)}  The rate-limt value in pps(packet-per-second)
*
* @returns  none
*
* @end
*********************************************************************/
void ip6UnresolvedRateLimitSet(openapiClientHandle_t *client_handle, uint32_t val)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIp6UnresolvedRateLimitSet(client_handle, val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the rate limit value for the unresolved IPv6 data packets coming to CPU.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the rate limit value for the unresolved IPv6 data packets coming to CPU. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the default IPv6 router TTL value for a given VRF.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrRtrDefaultTTLGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t val = 0; /* The TTL value */

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrRtrDefaultTTLGet(client_handle, &vrfNameBufd, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe default IPv6 router TTL value for the given VRF is %u.\n", val);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the default IPv6 router TTL value for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the default IPv6 router TTL value for a given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    vrfName        @b{(input)}  VRF name
* @param    val            @b{(input)}  The TTL value
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrRtrDefaultTTLSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t val)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrRtrDefaultTTLSet(client_handle, &vrfNameBufd, val);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the default IPv6 router TTL value for the given VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the default IPv6 router TTL value for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the IPv6 administrative state of a given interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t adminMode = OPEN_DISABLE; /* OPEN_DISABLE/OPEN_ENABLE */

  result = openapiIp6RtrIntfModeGet(client_handle, ifNum, &adminMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 administrative state of a given interface is %u (1-Enabled, 0-Disabled).\n", adminMode);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the IPv6 administrative state of a given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the operational state of a given interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfOperModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t adminMode = OPEN_DISABLE; /* OPEN_DISABLE/OPEN_ENABLE */

  result = openapiIp6RtrIntfOperModeGet(client_handle, ifNum, &adminMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe operational state of a given interface is %u (1-Enabled, 0-Disabled).\n", adminMode);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the operational state of a given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the IPv6 administrative state of a given interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal Interface Number
* @param    mode           @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t mode)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t openMode = OPEN_DISABLE;

  if (1 < mode)
  {
    printf("\n Invalid mode value. Expected 0(for Disable) or 1(for Enable).\n");
    return;
  }

  if (1 == mode)
  {
    openMode = OPEN_ENABLE;
  }

  result = openapiIp6RtrIntfModeSet(client_handle, ifNum, openMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the IPv6 administrative state of the given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the IPv6 administrative state of the given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the IPv6 AutoState mode of a given interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfAutoStateModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t adminMode = OPEN_DISABLE; /* OPEN_DISABLE/OPEN_ENABLE */

  result = openapiIp6RtrIntfAutoStateModeGet(client_handle, ifNum, &adminMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 AutoState mode of a given interface is %u (1-Enabled, 0-Disabled).\n", adminMode);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the IPv6 AutoState mode of a given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the IPv6 AutoState mode of a given interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal Interface Number
* @param    mode           @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfAutoStateModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t autoStateMode)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t openMode = OPEN_DISABLE;

  if (1 < autoStateMode)
  {
    printf("\nERROR: Invalid mode value. Expected 0(for Disable) or 1(for Enable).\n");
    return;
  }

  if (1 == autoStateMode)
  {
    openMode = OPEN_ENABLE;
  }

  result = openapiIp6RtrIntfAutoStateModeSet(client_handle, ifNum, openMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the IPv6 AutoState mode of the given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the IPv6 AutoState mode of a given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Determine if IPv6 interface exists.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6IntfExists(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  OPEN_BOOL_t result = OPEN_FALSE;

  result = openapiIp6IntfExists(client_handle, ifNum);
  switch(result)
  {
    case OPEN_TRUE:
         printf("\nTRUE.\n");
         break;

    case OPEN_FALSE:
         printf("\nFALSE.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to determine if IPv6 interface exists. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Determine if the interface is valid for participation in the IPv6 component.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6MapIntfIsValid(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  OPEN_BOOL_t result = OPEN_FALSE;

  result = openapiIp6MapIntfIsValid(client_handle, ifNum);
  switch(result)
  {
    case OPEN_TRUE:
         printf("\nTRUE.\n");
         break;

    case OPEN_FALSE:
         printf("\nFALSE.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to determine if the interface is valid for participation in the IPv6 component. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Determine if the interface is valid to be configured with an IPv6 neighbor.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6MapNbrIntfIsValid(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  OPEN_BOOL_t result = OPEN_FALSE;

  result = openapiIp6MapNbrIntfIsValid(client_handle, ifNum);
  switch(result)
  {
    case OPEN_TRUE:
         printf("\nTRUE.\n");
         break;

    case OPEN_FALSE:
         printf("\nFALSE.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to determine if the interface is valid to be configured with an IPv6 neighbor. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the IPv6 MTU of a given interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfIpv6MtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t val = 0; /* The IPv6 MTU value in bytes */

  result = openapiIp6RtrIntfIpv6MtuGet(client_handle, ifNum, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 MTU of the given interface is %u.\n", val);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the IPv6 MTU of a given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the IPv6 effective MTU of a given interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfIpv6EffectiveMtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t val = 0; /* The IPv6 effective MTU value in bytes */

  result = openapiIp6RtrIntfIpv6EffectiveMtuGet(client_handle, ifNum, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 effective MTU of the given interface is %u.\n", val);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the IPv6 effective MTU of a given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the maximum IP MTU that may be set on an interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfMaxIpv6MtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t val = 0; /* The IPv6 maximum MTU value in bytes */

  result = openapiIp6RtrIntfMaxIpv6MtuGet(client_handle, ifNum, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe maximum IPv6 MTU that may be set on the given interface is %u.\n", val);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the maximum IPv6 MTU that may be set on the given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the IPv6 MTU of a given interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal Interface Number
* @param    val            @b{(input)}  The IPv6 MTU value in bytes
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfIpv6MtuSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t val)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIp6RtrIntfIpv6MtuSet(client_handle, ifNum, val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the IPv6 MTU of the given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the IPv6 MTU of a given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the interface bandwidth of a given interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfBandwidthGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t val = 0; /* The bandwidth value in bytes */

  result = openapiIp6RtrIntfBandwidthGet(client_handle, ifNum, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe interface bandwidth of a given interface is %u.\n", val);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the interface bandwidth of a given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the first valid interface for participation in IPv6 component.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void ip6MapIntfValidFirstGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t ifNum = 0; /* First Internal Interface Number */

  result = openapiIp6MapIntfValidFirstGet(client_handle, &ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe first valid interface for participation in IPv6 component is %u.\n", ifNum);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the first valid interface for participation in IPv6 component. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the next valid interface for participation in IPv6 component.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    prevIfNum      @b{(input)}  Previous Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6MapIntfValidNextGet(openapiClientHandle_t *client_handle, uint32_t prevIfNum)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t nextIfNum = 0; /* Next Internal Interface Number */

  result = openapiIp6MapIntfValidNextGet(client_handle, prevIfNum, &nextIfNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe next valid interface for participation in IPv6 component is %u.\n", nextIfNum);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the next valid interface for participation in IPv6 component. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get IPv6 statistics for the given interface
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal Interface number

*
* @returns  none
*
* @end
*********************************************************************/
void ipv6IfStatsGet(openapiClientHandle_t *client_handle, uint32_t intIfNum)
{
  open_error_t result;
  openapiIpv6IfStats_t ipv6IfStats;

  result = openapiIpv6IfStatsGet (client_handle, intIfNum, &ipv6IfStats);
  switch(result)
  {
    case OPEN_E_NONE:
         printf ("\nIPv6 STATISTIC\n");
         printf ("Total Datagrams Received................................. %lu\n", ipv6IfStats.ipv6IfStatsInReceives);
         printf ("Received Datagrams Locally Delivered..................... %lu\n", ipv6IfStats.ipv6IfStatsInDelivers);
         printf ("Received Datagrams Discarded Due To Header Errors........ %lu\n", ipv6IfStats.ipv6IfStatsInHdrErrors);
         printf ("Received Datagrams Discarded Due To MTU.................. %lu\n", ipv6IfStats.ipv6IfStatsInTooBigErrors);
         printf ("Received Datagrams Discarded Due To No Route............. %lu\n", ipv6IfStats.ipv6IfStatsInNoRoutes);
         printf ("Received Datagrams With Unknown Protocol................. %lu\n", ipv6IfStats.ipv6IfStatsInUnknownProtos);
         printf ("Received Datagrams Discarded Due To Invalid Address...... %lu\n", ipv6IfStats.ipv6IfStatsInAddrErrors);
         printf ("Received Datagrams Discarded Due To Truncated Data....... %lu\n", ipv6IfStats.ipv6IfStatsInTruncatedPkts);
         printf ("Received Datagrams Discarded Other....................... %lu\n", ipv6IfStats.ipv6IfStatsInDiscards);
         printf ("Received Datagrams Reassembly Required................... %lu\n", ipv6IfStats.ipv6IfStatsReasmReqds);
         printf ("Datagrams Successfully Reassembled....................... %lu\n", ipv6IfStats.ipv6IfStatsReasmOKs);
         printf ("Datagrams Failed To Reassemble........................... %lu\n", ipv6IfStats.ipv6IfStatsReasmFails);
         printf ("Datagrams Forwarded...................................... %lu\n", ipv6IfStats.ipv6IfStatsOutForwDatagrams);
         printf ("Datagrams Locally Transmitted............................ %lu\n", ipv6IfStats.ipv6IfStatsOutRequests);
         printf ("Datagrams Transmit Failed................................ %lu\n", ipv6IfStats.ipv6IfStatsOutDiscards);
         printf ("Datagrams Successfully Fragmented........................ %lu\n", ipv6IfStats.ipv6IfStatsOutFragOKs);
         printf ("Datagrams Failed To Fragment............................. %lu\n", ipv6IfStats.ipv6IfStatsOutFragFails);
         printf ("Fragments Created........................................ %lu\n", ipv6IfStats.ipv6IfStatsOutFragCreates);
         printf ("Multicast Datagrams Received............................. %lu\n", ipv6IfStats.ipv6IfStatsInMcastPkts);
         printf ("Multicast Datagrams Transmitted.......................... %lu\n", ipv6IfStats.ipv6IfStatsOutMcastPkts);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the stats on interface. (result = %d)\n", result);
         break;
  }
  return;
}


/*********************************************************************
* @purpose  Get IPv6 ICMP statistics for the given interface
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal Interface number

*
* @returns  none
*
* @end
*********************************************************************/
void ipv6IfIcmpStatsGet(openapiClientHandle_t *client_handle, uint32_t intIfNum)
{
  open_error_t result;
  openapiIpstkIpv6IfIcmpStats_t ipv6IfIcmpStats;

  result = openapiIpv6IfIcmpStatsGet(client_handle, intIfNum, &ipv6IfIcmpStats);
  switch(result)
  {
    case OPEN_E_NONE:
         printf ("\nICMPv6 STATISTICS\n");
         printf ("Total ICMPv6 Messages Received........................... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInMsgs);
         printf ("ICMPv6 Messages With Errors Received..................... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInErrors);
         printf ("ICMPv6 Destination Unreachable Messages Received......... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInDestUnreachs);
         printf ("ICMPv6 Messages Prohibited Administratively Received..... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInAdminProhibs);
         printf ("ICMPv6 Time Exceeded Messages Received................... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInTimeExcds);
         printf ("ICMPv6 Parameter Problem Messages Received............... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInParmProblems);
         printf ("ICMPv6 Packet Too Big Messages Received.................. %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInPktTooBigs);
         printf ("ICMPv6 Echo Request Messages Received.................... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInEchos);
         printf ("ICMPv6 Echo Reply Messages Received...................... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInEchoReplies);
         printf ("ICMPv6 Router Solicit Messages Received.................. %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInRouterSolicits);
         printf ("ICMPv6 Router Advertisement Messages Received............ %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInRouterAdvertisements);
         printf ("ICMPv6 Neighbor Solicit Messages Received................ %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInNeighborSolicits);
         printf ("ICMPv6 Neighbor Advertisement Messages Received.......... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInNeighborAdvertisements);
         printf ("ICMPv6 Redirect Messages Received........................ %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInRedirects);
         printf ("ICMPv6 Group Membership Query Messages Received.......... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInGroupMembQueries);
         printf ("ICMPv6 Group Membership Response Messages Received....... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInGroupMembResponses);
         printf ("ICMPv6 Group Membership Reduction Messages Received...... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInGroupMembReductions);
         printf ("Total ICMPv6 Messages Transmitted........................ %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutMsgs);
         printf ("ICMPv6 Messages Not Transmitted Due To Error............. %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutErrors);
         printf ("ICMPv6 Destination Unreachable Messages Transmitted...... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutDestUnreachs);
         printf ("ICMPv6 Messages Prohibited Administratively Transmitted.. %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutAdminProhibs);
         printf ("ICMPv6 Time Exceeded Messages Transmitted................ %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutTimeExcds);
         printf ("ICMPv6 Parameter Problem Messages Transmitted............ %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutParmProblems);
         printf ("ICMPv6 Packet Too Big Messages Transmitted............... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutPktTooBigs);
         printf ("ICMPv6 Echo Request Messages Transmitted................. %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutEchos);
         printf ("ICMPv6 Echo Reply Messages Transmitted................... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutEchoReplies);
         printf ("ICMPv6 Router Solicit Messages Transmitted............... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutRouterSolicits);
         printf ("ICMPv6 Router Advertisement Messages Transmitted......... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutRouterAdvertisements);
         printf ("ICMPv6 Neighbor Solicit Messages Transmitted............. %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutNeighborSolicits);
         printf ("ICMPv6 Neighbor Advertisement Messages Transmitted....... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutNeighborAdvertisements);
         printf ("ICMPv6 Redirect Messages Transmitted..................... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutRedirects);
         printf ("ICMPv6 Group Membership Query Messages Transmitted....... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutGroupMembQueries);
         printf ("ICMPv6 Group Membership Response Messages Transmitted.... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutGroupMembResponses);
         printf ("ICMPv6 Group Membership Reduction Messages Transmitted... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutGroupMembReductions);
         printf ("ICMPv6 Duplicate Address Detects......................... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpDupAddrDetects);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the stats on interface. (result = %d)\n", result);
         break;
  }
  return;
}


/*********************************************************************
* @purpose  Clear IPv6 statistics for the given interface
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal Interface number

*
* @returns  none
*
* @end
*********************************************************************/
void ipv6IfStatsClear(openapiClientHandle_t *client_handle, uint32_t intIfNum)
{
  open_error_t result;

  result = openapiIpv6IfStatsClear (client_handle, intIfNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nIPv6 Statistics are cleared on interface - %u.\n", intIfNum);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the stats on interface. (result = %d)\n", result);
         break;
  }
  return;
}


/*********************************************************************
* @purpose  Get total of each IPv6 statistic on all interfaces
*
* @param    client_handle       @b{(input)}  client handle from registration API

*
* @returns  none
*
* @end
*********************************************************************/
void ipv6TotalStatsGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  openapiIpv6IfStats_t ipv6IfStats;
  openapiIpstkIpv6IfIcmpStats_t ipv6IfIcmpStats;

  result = openapiIpv6TotalStatsGet (client_handle, &ipv6IfStats, &ipv6IfIcmpStats);
  switch(result)
  {
    case OPEN_E_NONE:
         printf ("\nIPv6 STATISTIC\n");
         printf ("Total Datagrams Received................................. %lu\n", ipv6IfStats.ipv6IfStatsInReceives);
         printf ("Received Datagrams Locally Delivered..................... %lu\n", ipv6IfStats.ipv6IfStatsInDelivers);
         printf ("Received Datagrams Discarded Due To Header Errors........ %lu\n", ipv6IfStats.ipv6IfStatsInHdrErrors);
         printf ("Received Datagrams Discarded Due To MTU.................. %lu\n", ipv6IfStats.ipv6IfStatsInTooBigErrors);
         printf ("Received Datagrams Discarded Due To No Route............. %lu\n", ipv6IfStats.ipv6IfStatsInNoRoutes);
         printf ("Received Datagrams With Unknown Protocol................. %lu\n", ipv6IfStats.ipv6IfStatsInUnknownProtos);
         printf ("Received Datagrams Discarded Due To Invalid Address...... %lu\n", ipv6IfStats.ipv6IfStatsInAddrErrors);
         printf ("Received Datagrams Discarded Due To Truncated Data....... %lu\n", ipv6IfStats.ipv6IfStatsInTruncatedPkts);
         printf ("Received Datagrams Discarded Other....................... %lu\n", ipv6IfStats.ipv6IfStatsInDiscards);
         printf ("Received Datagrams Reassembly Required................... %lu\n", ipv6IfStats.ipv6IfStatsReasmReqds);
         printf ("Datagrams Successfully Reassembled....................... %lu\n", ipv6IfStats.ipv6IfStatsReasmOKs);
         printf ("Datagrams Failed To Reassemble........................... %lu\n", ipv6IfStats.ipv6IfStatsReasmFails);
         printf ("Datagrams Forwarded...................................... %lu\n", ipv6IfStats.ipv6IfStatsOutForwDatagrams);
         printf ("Datagrams Locally Transmitted............................ %lu\n", ipv6IfStats.ipv6IfStatsOutRequests);
         printf ("Datagrams Transmit Failed................................ %lu\n", ipv6IfStats.ipv6IfStatsOutDiscards);
         printf ("Datagrams Successfully Fragmented........................ %lu\n", ipv6IfStats.ipv6IfStatsOutFragOKs);
         printf ("Datagrams Failed To Fragment............................. %lu\n", ipv6IfStats.ipv6IfStatsOutFragFails);
         printf ("Fragments Created........................................ %lu\n", ipv6IfStats.ipv6IfStatsOutFragCreates);
         printf ("Multicast Datagrams Received............................. %lu\n", ipv6IfStats.ipv6IfStatsInMcastPkts);
         printf ("Multicast Datagrams Transmitted.......................... %lu\n", ipv6IfStats.ipv6IfStatsOutMcastPkts);

         printf ("\nICMPv6 STATISTICS\n");
         printf ("Total ICMPv6 Messages Received........................... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInMsgs);
         printf ("ICMPv6 Messages With Errors Received..................... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInErrors);
         printf ("ICMPv6 Destination Unreachable Messages Received......... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInDestUnreachs);
         printf ("ICMPv6 Messages Prohibited Administratively Received..... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInAdminProhibs);
         printf ("ICMPv6 Time Exceeded Messages Received................... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInTimeExcds);
         printf ("ICMPv6 Parameter Problem Messages Received............... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInParmProblems);
         printf ("ICMPv6 Packet Too Big Messages Received.................. %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInPktTooBigs);
         printf ("ICMPv6 Echo Request Messages Received.................... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInEchos);
         printf ("ICMPv6 Echo Reply Messages Received...................... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInEchoReplies);
         printf ("ICMPv6 Router Solicit Messages Received.................. %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInRouterSolicits);
         printf ("ICMPv6 Router Advertisement Messages Received............ %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInRouterAdvertisements);
         printf ("ICMPv6 Neighbor Solicit Messages Received................ %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInNeighborSolicits);
         printf ("ICMPv6 Neighbor Advertisement Messages Received.......... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInNeighborAdvertisements);
         printf ("ICMPv6 Redirect Messages Received........................ %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInRedirects);
         printf ("ICMPv6 Group Membership Query Messages Received.......... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInGroupMembQueries);
         printf ("ICMPv6 Group Membership Response Messages Received....... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInGroupMembResponses);
         printf ("ICMPv6 Group Membership Reduction Messages Received...... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpInGroupMembReductions);
         printf ("Total ICMPv6 Messages Transmitted........................ %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutMsgs);
         printf ("ICMPv6 Messages Not Transmitted Due To Error............. %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutErrors);
         printf ("ICMPv6 Destination Unreachable Messages Transmitted...... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutDestUnreachs);
         printf ("ICMPv6 Messages Prohibited Administratively Transmitted.. %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutAdminProhibs);
         printf ("ICMPv6 Time Exceeded Messages Transmitted................ %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutTimeExcds);
         printf ("ICMPv6 Parameter Problem Messages Transmitted............ %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutParmProblems);
         printf ("ICMPv6 Packet Too Big Messages Transmitted............... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutPktTooBigs);
         printf ("ICMPv6 Echo Request Messages Transmitted................. %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutEchos);
         printf ("ICMPv6 Echo Reply Messages Transmitted................... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutEchoReplies);
         printf ("ICMPv6 Router Solicit Messages Transmitted............... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutRouterSolicits);
         printf ("ICMPv6 Router Advertisement Messages Transmitted......... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutRouterAdvertisements);
         printf ("ICMPv6 Neighbor Solicit Messages Transmitted............. %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutNeighborSolicits);
         printf ("ICMPv6 Neighbor Advertisement Messages Transmitted....... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutNeighborAdvertisements);
         printf ("ICMPv6 Redirect Messages Transmitted..................... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutRedirects);
         printf ("ICMPv6 Group Membership Query Messages Transmitted....... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutGroupMembQueries);
         printf ("ICMPv6 Group Membership Response Messages Transmitted.... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutGroupMembResponses);
         printf ("ICMPv6 Group Membership Reduction Messages Transmitted... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpOutGroupMembReductions);
         printf ("ICMPv6 Duplicate Address Detects......................... %lu\n", ipv6IfIcmpStats.ipv6IfIcmpDupAddrDetects);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the stats on interface. (result = %d)\n", result);
         break;
  }
  return;
}


/*********************************************************************
* @purpose  Clear IPv6 statistics on all interfaces
*
* @param    client_handle       @b{(input)}  client handle from registration API

*
* @returns  none
*
* @end
*********************************************************************/
void ipv6TotalStatsClear(openapiClientHandle_t *client_handle)
{
  open_error_t result;

  result = openapiIpv6TotalStatsClear(client_handle);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nIPv6 Statistics are cleared on all the interfaces.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the stats on interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the IPv6 Route table
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    *vrfName       @b{(input)}  VRF Name
* @param    bestRouteOnly  @b{(input)}  Get best routes only.
*                                       0 for False and 1 for True
*
* @returns  none
*
* @end
*********************************************************************/
void ipv6RouteTableGet (openapiClientHandle_t *client_handle,
                        char *vrfName,
                        uint32_t bestRouteOnly)
{
  open_error_t         result = OPEN_E_FAIL;
  openapiRoute6Entry_t ipv6Route;
  open_buffdesc        vrfNameBufd;
  char                 vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t             c = 1, i = 0;
  char                 ipStr[80];

  memset (vrfNameStr, 0, sizeof(vrfNameStr));
  memset (&ipv6Route, 0, sizeof (openapiRoute6Entry_t));

  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }

  strncpy (vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;
  
  /* Get first entry */
  result = openapiIpv6RouteEntryGetNext (client_handle, &vrfNameBufd,
                                         &ipv6Route, (OPEN_BOOL_t) bestRouteOnly);
  switch (result)
  {
    case OPEN_E_NONE:
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the route table for a given VRF. (result = %d)\n", result);
         break;
  }

  do
  {
    memset (ipStr, 0 , sizeof (ipStr));
    (void) inet_ntop (AF_INET6, &(ipv6Route.ip6Addr.u.addr8), ipStr, sizeof (ipStr));
    printf ("\nRoute %-5u: %s/%d", c, ipStr, ipv6Route.ip6PrefixLen);
    printf ("\n    nexthops:");
    for (i = 0; i < ipv6Route.ecmpRoutes.numOfRoutes; i++)
    {
      memset (ipStr, 0 , sizeof (ipStr));
      (void) inet_ntop (AF_INET6, ipv6Route.ecmpRoutes.equalCostPath[i].ip6Addr.u.addr8,
                        ipStr, sizeof (ipStr));
      if (0 != ipv6Route.ecmpRoutes.equalCostPath[i].intIfNum)
      {
        printf ("\n      %s via %u", ipStr, ipv6Route.ecmpRoutes.equalCostPath[i].intIfNum);
      }
      else
      {
        printf ("\n      %s", ipStr);
      }
    }
    printf ("\n    preference: %u", ipv6Route.pref);
    printf ("\n    protocol: %u\n", ipv6Route.protocol);
    c++;

  } while (OPEN_E_NONE == openapiIpv6RouteEntryGetNext (client_handle, &vrfNameBufd,
                                                        &ipv6Route, bestRouteOnly));
  return;
}

/*********************************************************************
* @purpose  Get the number of routes in the routing table
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    *vrfName         @b{(input)}  VRF Name
* @param    bestRouteOnly    @b{(input)}  Get best routes only.
*                                         0 for False and 1 for True

*
* @returns  none
*
* @end
*********************************************************************/
void ipv6RouteCountGet (openapiClientHandle_t *client_handle,
                        char *vrfName,
                        uint32_t bestRouteOnly)
{
  open_error_t    result;
  open_buffdesc   vrfNameBufd;
  char            vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t        count = 0;

  memset (vrfNameStr, 0, sizeof(vrfNameStr));

  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }

  strncpy (vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;
  
  /* Get first entry */
  result = openapiIpv6RouteCountGet (client_handle, &vrfNameBufd,
                                     &count, (OPEN_BOOL_t) bestRouteOnly);
  switch (result)
  {
    case OPEN_E_NONE:
         printf ("\nTotal number of IPv6 routes for VRF \"%s\" : %u\n", vrfName, count);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the route table for a given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the configured IPv6 addresses and modes on the interface
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum            @b{(input)}  Internal interface number
* @param    numAddr             @b{(inout)}  Number IPv6 addressed to be fetched
*                                             or configured on interface
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrCfgIntfAddressesGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t numAddr)
{
  open_error_t                result;
  uint32_t                    i = 0;
  openapi_ip6_cfg_addr_list_t ipv6AddrList;
  OPEN_CONTROL_t              ipv6Mode = OPEN_DISABLE;
  OPEN_CONTROL_t              autoConfigMode = OPEN_DISABLE;
  OPEN_CONTROL_t              dhcpMode = OPEN_DISABLE;
  char                        ipStr[80];
  uint32_t                    count = numAddr;

  memset (&ipv6AddrList, 0, sizeof (openapi_ip6_cfg_addr_list_t));

  if ((result = openapiIp6RtrCfgIntfAddressesGet (client_handle, intIfNum,
                                                  &count, &ipv6AddrList, &ipv6Mode,
                                                  &autoConfigMode, &dhcpMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the configured IPv6 addresses on the interface (result = %d)\n", result);
  }
  else
  {
    printf ("\n\nIPv6 Mode on intf-%u: %s", intIfNum,
            (ipv6Mode == OPEN_ENABLE) ? "ENABLED" : "DISABLED");
    printf ("\nConfig modes on intf-%u: %s  %s\n", intIfNum,
           (dhcpMode == OPEN_ENABLE) ? "DHCP" : "",
           (autoConfigMode == OPEN_ENABLE) ? "AUTO CONF" : "");
    printf ("\nNumber of address available on interface : %u", count);
    printf ("\nInterface Addresses:");
    for (i = 0; i < count; i++)
    {
      memset (ipStr, 0 , sizeof (ipStr));
      (void) inet_ntop (AF_INET6, &(ipv6AddrList.ipAddrList[i].u.addr8), ipStr, sizeof (ipStr));
      printf ("\n     %s/%u", ipStr, ipv6AddrList.pLen[i]);
    }
    printf ("\n\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the number of neighbor solicits configured to be set for duplicate address detection (DAD) on an interface
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number

*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrDadTransmitsGet(openapiClientHandle_t *client_handle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t     val = 0;

  if ((result = openapiIp6RtrDadTransmitsGet (client_handle, intIfNum, &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the number of neighbor solicits configured to be set for duplicate address detection (DAD) on an interface (result = %d)\n", result);
  }
  else
  {
    printf ("\nNumber of solicits to be sent for DAD on intf-%u is %u\n\n", intIfNum, val);
  }
  return;
}


/*********************************************************************
* @purpose  Set the number of neighbor solicits to be sent for duplicate address detection (DAD) on an interface
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    val       @b{(input)}  Number of neighbor solicits to be sent for DAD

*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrDadTransmitsSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t val)
{
  open_error_t result;

  if ((result = openapiIp6RtrDadTransmitsSet(client_handle, intIfNum, val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the number of neighbor solicits to be sent for duplicate address detection (DAD) on an interface (result = %d)\n", result);
  }
  else
  {
    printf ("\n%u Solicits configured to be sent for DAD on intf-%u\n\n", val, intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Get the interval between IPv6 neighbor solicitation retransmissions on an interface
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number

*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrNdNsIntervalGet(openapiClientHandle_t *client_handle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t     val = 0;

  if ((result = openapiIp6RtrNdNsIntervalGet(client_handle, intIfNum, &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the interval between IPv6 neighbor solicitation retransmissions on an interface (result = %d)\n", result);
  }
  else
  {
    printf ("\nInterval between solicits for intf-%u is %u\n\n", intIfNum, val);
  }
  return;
}


/*********************************************************************
* @purpose  Set the interval between IPv6 neighbor solicitation retransmissions on an interface
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    val       @b{(input)}  Interval between IPv6 neighbor solicitation retransmissions in milliseconds

*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrNdNsIntervalSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t val)
{
  open_error_t result;

  if ((result = openapiIp6RtrNdNsIntervalSet(client_handle, intIfNum, val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the interval between IPv6 neighbor solicitation retransmissions on an interface (result = %d)\n", result);
  }
  else
  {
    printf ("\nInterval between solicits for intf-%u configured to %u\n\n", intIfNum, val);
  }
  return;
}


/*********************************************************************
* @purpose  Get the amount of time that a remote IPv6 node is reachable
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number

*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrNdReachableTimeGet(openapiClientHandle_t *client_handle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t     val;

  if ((result = openapiIp6RtrNdReachableTimeGet(client_handle, intIfNum, &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the amount of time that a remote IPv6 node is reachable (result = %d)\n", result);
  }
  else
  {
    printf ("\nTime for which remote node is reachable on intf-%u is %u\n\n", intIfNum, val);
  }
  return;
}


/*********************************************************************
* @purpose  Set the amount of time that a remote IPv6 node is reachable
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    val       @b{(input)}  Reachable time in milliseconds

*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrNdReachableTimeSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t val)
{
  open_error_t result;

  if ((result = openapiIp6RtrNdReachableTimeSet(client_handle, intIfNum, val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the amount of time that a remote IPv6 node is reachable (result = %d)\n", result);
  }
  else
  {
    printf ("\nTime for which remote node is reachable on intf-%u is configured to %u\n\n", intIfNum, val);
  }
  return;
}


/*********************************************************************
* @purpose  Get the configured mode for ICMPv6 unreachable messages
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number

*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfICMPUnreachablesModeGet(openapiClientHandle_t *client_handle, uint32_t intIfNum)
{
  open_error_t   result;
  OPEN_CONTROL_t mode = OPEN_DISABLE;

  if ((result = openapiIp6RtrIntfICMPUnreachablesModeGet(client_handle, intIfNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the configured mode for ICMPv6 unreachable messages (result = %d)\n", result);
  }
  else
  {
    printf("\n ICMP Unreachable messages on intf-%u are %s\n\n",
           intIfNum, (mode == OPEN_ENABLE) ? "ENABLED" : "DISABLED");
  }
  return;
}


/*********************************************************************
* @purpose  Set the configured mode for ICMPv6 unreachable messages
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    mode       @b{(input)}  Enable or Disable ICMPv6 unreachable messages

*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfICMPUnreachablesModeSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiIp6RtrIntfICMPUnreachablesModeSet(client_handle, intIfNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the configured mode for ICMPv6 unreachable messages (result = %d)\n", result);
  }
  else
  {
    printf("\n ICMP Unreachable messages on intf-%u are configured to be %s\n\n",
           intIfNum, (mode == OPEN_ENABLE) ? "ENABLED" : "DISABLED");
  }
  return;
}

/*********************************************************************
* @purpose  Get the IPv6 address details.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    intIfNum            @b{(inout)}  Internal Interface Number.
* @param    ipv6AddrString      @b{(inout)}  IPv6 address.
*
* @returns  none
*
* @end
*********************************************************************/
void ipv6AddrEntryGet (openapiClientHandle_t *client_handle, uint32_t intIfNum, char* ipv6AddrString)
{
  open_error_t result;
  openapiIpv6AddrEntry_t ipv6AddrEntry;
  uint32_t i = 0;
  char ipStr[80];

  memset (&ipv6AddrEntry, 0, sizeof (openapiIpv6AddrEntry_t));

  ipv6AddrEntry.ipv6IfIndex = intIfNum;
  i = inet_pton (AF_INET6, ipv6AddrString, (void*)&(ipv6AddrEntry.ipv6Address));

  if ((0 == i) ||
      (0 > i))
  {
    printf("Bad return code trying to convert ip address - %s.\n", ipv6AddrString);
  }

  if ((result = openapiIpv6AddrEntryGet (client_handle, 1, &ipv6AddrEntry)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the IPv6 address details. (result = %d)\n", result);
  }
  else
  {
    memset (ipStr, 0 , sizeof (ipStr));
    (void) inet_ntop (AF_INET6, &(ipv6AddrEntry.ipv6Address.u.addr8), ipStr, sizeof (ipStr));
    printf ("\n\nAddress Configured on interface %u is %s/%u\n", ipv6AddrEntry.ipv6IfIndex,
            ipStr, ipv6AddrEntry.ipv6AddrPfxLength);
    printf ("\n addr Type is %5u", ipv6AddrEntry.ipv6AddrType);
    printf ("\n anycast flag is %5u", ipv6AddrEntry.ipv6AddrAnycastFlag);
    printf ("\n EUI 4 flag is %5u", ipv6AddrEntry.ipv6AddrEui64Flag);
    printf ("\n address status is %5u", ipv6AddrEntry.ipv6AddrStatus);
    printf ("\n\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the IPv6 address details of the next entry.
*
* @param    client_handle   @b{(input)}  client handle from registration API.
*
* @returns  none
*
* @end
*********************************************************************/
void ipv6AddrEntryGetNext(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  openapiIpv6AddrEntry_t ipv6AddrEntry;
  char ipStr[80];

  memset (&ipv6AddrEntry, 0, sizeof (openapiIpv6AddrEntry_t));

  if ((result = openapiIpv6AddrEntryGetNext(client_handle, 2, &ipv6AddrEntry)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the IPv6 address details of the next entry. (result = %d)\n", result);
  }
  else
  {
    do {
      printf ("\n\nAddress Configured on interface %u :", ipv6AddrEntry.ipv6IfIndex);
      memset (ipStr, 0 , sizeof (ipStr));
      (void) inet_ntop (AF_INET6, &(ipv6AddrEntry.ipv6Address.u.addr8), ipStr, sizeof (ipStr));
      printf ("\n\n   %s/%u\n", ipStr, ipv6AddrEntry.ipv6AddrPfxLength);
      printf ("\n      addr Type is %5u", ipv6AddrEntry.ipv6AddrType);
      printf ("\n      anycast flag is %5u", ipv6AddrEntry.ipv6AddrAnycastFlag);
      printf ("\n      EUI 4 flag is %5u", ipv6AddrEntry.ipv6AddrEui64Flag);
      printf ("\n      address status is %5u", ipv6AddrEntry.ipv6AddrStatus);
    } while ((openapiIpv6AddrEntryGetNext (client_handle, 2, &ipv6AddrEntry)) == OPEN_E_NONE);
    printf ("\n\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the IPv6 address prefix details.
*
* @param    client_handle    @b{(input)}  client handle from registration API.
* @param    intIfNum         @b{(inout)}  Internal Interface Number.
* @param    ipv6AddrString   @b{(inout)}  IPv6 address.
* @param    pfxLen           @b{(inout)}  Prefix length.
*
* @returns  none
*
* @end
*********************************************************************/
void ipv6AddrPrefixEntryGet (openapiClientHandle_t *client_handle, uint32_t intIfNum, char* ipv6AddrString, uint32_t pfxLen)
{
  open_error_t result;
  openapiIpv6AddrPrefixEntry_t ipv6AddrPrefixEntry;
  char ipStr[80];
  uint32_t i = 0;

  memset (&ipv6AddrPrefixEntry, 0, sizeof (openapiIpv6AddrPrefixEntry_t));

  ipv6AddrPrefixEntry.ipv6IfIndex = intIfNum;
  i = inet_pton (AF_INET6, ipv6AddrString, (void*)&(ipv6AddrPrefixEntry.ipv6AddrPrefix));
  ipv6AddrPrefixEntry.ipv6AddrPrefixLength = pfxLen;

  if ((0 == i) ||
      (0 > i))
  {
    printf("Bad return code trying to convert ip address - %s.\n", ipv6AddrString);
  }

  if ((result = openapiIpv6AddrPrefixEntryGet (client_handle, 1, &ipv6AddrPrefixEntry)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the IPv6 address prefix details. (result = %d)\n", result);
  }
  else
  {
    memset (ipStr, 0 , sizeof (ipStr));
    (void) inet_ntop (AF_INET6,
                      &(ipv6AddrPrefixEntry.ipv6AddrPrefix.u.addr8),
                      ipStr, sizeof (ipStr));
    printf ("\n\n Address configured on interface %u is %s/%u\n",
            ipv6AddrPrefixEntry.ipv6IfIndex, ipStr,
            ipv6AddrPrefixEntry.ipv6AddrPrefixLength);
    printf ("\n On link flag is %5u", ipv6AddrPrefixEntry.ipv6AddrPrefixOnLinkFlag);
    printf ("\n Autonomous flag is %5u", ipv6AddrPrefixEntry.ipv6AddrPrefixAutonomousFlag);
    printf ("\n preferred lifetime is %5u", ipv6AddrPrefixEntry.ipv6AddrPrefixAdvPreferredLifetime);
    printf ("\n valid lifetime is %5u", ipv6AddrPrefixEntry.ipv6AddrPrefixAdvValidLifetime);
    printf ("\n\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the IPv6 addresses of the interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API.
* @param    intIfNum       @b{(input)}  Internal interface number.
* @param    numAddr        @b{(inout)}  Number of addresses to be fetched
*                                        or actual available on interface.
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfAddressesGet (openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t numAddr)
{
  open_error_t result;
  openIpv6IntfAddrList_t addrList;
  uint32_t i = 0, count = numAddr;
  char ipStr[80];

  memset (&addrList, 0, sizeof (openIpv6IntfAddrList_t));

  if ((result = openapiIp6RtrIntfAddressesGet(client_handle, intIfNum, &count, &addrList)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the IPv6 addresses of the interface. (result = %d)\n", result);
  }
  else
  {
    printf ("\n\nNumber of addresses configured on interface %u : %u", intIfNum, count);
    for (i = 0; i < count; i++)
    {
      memset (ipStr, 0 , sizeof (ipStr));
      (void) inet_ntop (AF_INET6,
                        &(addrList.ipv6AddrList[i].ipv6Addr.u.addr8),
                        ipStr, sizeof (ipStr));
      printf ("\n %u:    %s/%u", i, ipStr, addrList.ipv6AddrList[i].ipv6PrefixLen);
      printf ("\n    state : %u", addrList.ipv6AddrList[i].ipv6AddrState);
    }
    printf ("\n\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the IPv6 address of the interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API.
* @param    intIfNum       @b{(input)}  Internal interface number.
* @param    inetAddr       @b{(output)}  Address on the interface.
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfAddressGet(openapiClientHandle_t *client_handle, uint32_t intIfNum)
{
  open_error_t result;
  open_inet_addr_t inetAddr;
  char ipStr[80];

  memset (&inetAddr, 0, sizeof (open_inet_addr_t));

  if ((result = openapiIp6RtrIntfAddressGet (client_handle, intIfNum, &inetAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the IPv6 address of the interface. (result = %d)\n", result);
  }
  else
  {
    memset (ipStr, 0 , sizeof (ipStr));
    (void) inet_ntop (AF_INET6, &(inetAddr.addr.ipv6.u.addr8), ipStr, sizeof (ipStr));
    printf ("\n\nPrimary address on the interface %u is %s\n\n", intIfNum, ipStr);
  }
  return;
}


/*********************************************************************
* @purpose  Get the IPv6 neighbor entries.
*
* @param    client_handle   @b{(input)}  client handle from registration API.
* @param    vrfName         @b{(input)}  VRF name.
* @param    intIfNum        @b{(inout)}  Internal interface number.
*
* @returns  none
*
* @end
*********************************************************************/
void ipv6NbrTblEntryGet (openapiClientHandle_t *client_handle, char *vrfName, uint32_t intIfNum)
{
  open_error_t result;
  open_buffdesc vrfNameBufd;
  open_inet_addr_t ipv6Addr;
  open_buffdesc macAddr;
  OPEN_IP6_NBR_TYPE_t ipv6Type = 0;
  OPEN_IP6_NBR_STATE_t nbrState = 0;
  uint32_t lastUpdate = 0;
  OPEN_BOOL_t isRouter = OPEN_FALSE;
  uint32_t ipv6ScopeId = 0;
  char ipStr[80];
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  char mac[18];
  uint32_t ifIndex = intIfNum;
  char nbrStateStr[32];
  char ipv6TypeStr[32];

  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  memset (vrfNameStr, 0, sizeof (vrfNameStr));
  memset (nbrStateStr, 0, sizeof (nbrStateStr));
  memset (ipv6TypeStr, 0, sizeof (ipv6TypeStr));

  strncpy (vrfNameStr, vrfName, strlen(vrfName));
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1; 
  memset (&ipv6Addr, 0, sizeof (open_inet_addr_t));
  memset (mac, 0, 18);
  macAddr.pstart = mac;
  macAddr.size = 18;
  
  if ((result = openapiIpv6NbrTblEntryGet (client_handle, &vrfNameBufd,
                                           &ipv6Addr, &macAddr,
                                           &ifIndex, &ipv6Type,
                                           &nbrState, &lastUpdate,
                                           &isRouter, &ipv6ScopeId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the IPv6 neighbor entries. (result = %d)\n", result);
  }
  else
  {
    do {
      if (0 == intIfNum)
      {
        printf ("\n\n IPv6 Neighbors :");
        ifIndex = 0;
      }
      else
        printf ("\n\n IPv6 Neighbors on interface %u : ", intIfNum);

      memset (ipStr, 0 , sizeof (ipStr));
      (void) inet_ntop (AF_INET6, &(ipv6Addr.addr.ipv6.u.addr8), ipStr, sizeof (ipStr));

      memset (nbrStateStr, 0, sizeof (nbrStateStr));
      switch (nbrState)
      {
        case OPEN_IP6_NBR_STATE_REACHABLE:
          snprintf(nbrStateStr, sizeof(nbrStateStr), "%s", "Reachable");
          break;

        case OPEN_IP6_NBR_STATE_STALE:
          snprintf(nbrStateStr, sizeof(nbrStateStr), "%s", "Stale");
          break;

        case OPEN_IP6_NBR_STATE_DELAY:
          snprintf(nbrStateStr, sizeof(nbrStateStr), "%s", "Delay");
          break;

        case OPEN_IP6_NBR_STATE_PROBE:
          snprintf(nbrStateStr, sizeof(nbrStateStr), "%s", "Probe");
          break;

        case OPEN_IP6_NBR_STATE_INCOMPLETE:
          snprintf(nbrStateStr, sizeof(nbrStateStr), "%s", "Incomplete");
          break;

        case OPEN_IP6_NBR_STATE_UNREACHABLE:
          snprintf(nbrStateStr, sizeof(nbrStateStr), "%s", "Unreachable");
          break;

        case OPEN_IP6_NBR_STATE_PERMANENT:
          snprintf(nbrStateStr, sizeof(nbrStateStr), "%s", "Permanent");
          break;

        case OPEN_IP6_NBR_STATE_UNKNOWN:
          snprintf(nbrStateStr, sizeof(nbrStateStr), "%s", "Unknown");
        default:
          break;
      }

      memset (ipv6TypeStr, 0, sizeof (ipv6TypeStr));
      switch (ipv6Type)
      {
        case OPEN_IP6_NBR_TYPE_DYNAMIC:
          snprintf(ipv6TypeStr, sizeof(ipv6TypeStr), "%s", "Dynamic");
          break;

        case OPEN_IP6_NBR_TYPE_STATIC:
          snprintf(ipv6TypeStr, sizeof(ipv6TypeStr), "%s", "Static");
          break;

        case OPEN_IP6_NBR_TYPE_LOCAL:
          snprintf(ipv6TypeStr, sizeof(ipv6TypeStr), "%s", "Local");
          break;

        case OPEN_IP6_NBR_TYPE_OTHER:
        default:
          snprintf(ipv6TypeStr, sizeof(ipv6TypeStr), "%s", "Other");
          break;
      }

      printf ("\n    Neighbor address is %s", ipStr);
      printf ("\n     Mac address is %s", mac);
      printf ("\n     interface is %u", ifIndex);
      printf ("\n     scope id is %u", ipv6ScopeId);
      printf ("\n     type is %s(%u)", ipv6TypeStr, ipv6Type);
      printf ("\n     state is %s(%u)", nbrStateStr, nbrState);
      printf ("\n     isRouter - %s", ((isRouter == OPEN_TRUE)? "TRUE" : "FALSE"));
      printf ("\n     time since last update %u", lastUpdate);

    } while (OPEN_E_NONE == openapiIpv6NbrTblEntryGet (client_handle, &vrfNameBufd,
                                           &ipv6Addr, &macAddr,
                                           &ifIndex, &ipv6Type,
                                           &nbrState, &lastUpdate,
                                           &isRouter, &ipv6ScopeId));
    printf ("\n\n");
  }
  return;
}


/*********************************************************************
* @purpose  Flush the IPv6 neighbor entries.
*
* @param    client_handle  @b{(input)}  client handle from registration API.
* @param    vrfName        @b{(input)}  VRF name.
* @param    intIfNum       @b{(input)}  Internal interface number.
*
* @returns  none
*
* @end
*********************************************************************/
void ipv6NdpFlush(openapiClientHandle_t *client_handle, char *vrfName, uint32_t intIfNum)
{
  open_error_t result;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  memset (vrfNameStr, 0, sizeof (vrfNameStr));
  strncpy (vrfNameStr, vrfName, strlen(vrfName));
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  if ((result = openapiIpv6NdpFlush(client_handle, &vrfNameBufd, intIfNum, NULL)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the IPv6 address of the interface. (result = %d)\n", result);
  }
  else
  {
    printf("\n\nNeighbor are cleared.\n\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the next IPv6 route from the routing table.
*
* @param    client_handle    @b{(input)}  client handle from registration API.
*
* @returns  none
*
* @end
*********************************************************************/
void snmpIpv6StaticRouteGetNext (openapiClientHandle_t *client_handle)
{
  open_error_t result;
  openIpv6RouteEntry_t routeEntry;
  char ipStr[80];

  memset (&routeEntry, 0, sizeof (openIpv6RouteEntry_t));

  if ((result = openapiSnmpIpv6StaticRouteGetNext (client_handle, &routeEntry)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next IPv6 route from the routing table. (result = %d)\n", result);
  }
  else
  {
    printf ("\n\n Static Routes configured :");
    do {
      memset (ipStr, 0 , sizeof (ipStr));
      (void) inet_ntop (AF_INET6,
                        &(routeEntry.ipv6RouteDest.u.addr8),
                        ipStr, sizeof (ipStr));
      printf ("\n    %s/%u", ipStr, routeEntry.ipv6RoutePfxLength);
      printf ("\n     interface %u", routeEntry.ipv6RouteIfIndex);
      memset (ipStr, 0 , sizeof (ipStr));
      (void) inet_ntop (AF_INET6,
                        &(routeEntry.ipv6RouteNextHop.u.addr8),
                        ipStr, sizeof (ipStr));
      printf ("\n     next hop is %s", ipStr);
      printf ("\n     protocol is %u", routeEntry.ipv6RouteProtocol);
      printf ("\n     metric is %u", routeEntry.ipv6Metric);
      printf ("\n     weight is %u", routeEntry.ipv6RouteWeight);

    } while (OPEN_E_NONE == openapiSnmpIpv6StaticRouteGetNext (client_handle, &routeEntry));
    printf ("\n\n");
  }
  return;
}


/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for IP6Map
*
* @returns   0: Success
* @returns  -1: Failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  uint32_t value = 0;
  uint32_t ifNum = 0;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("IP6Map example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting IP6Map API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1: /* Test 1: Get the administrative mode of the router for given VRF: ip6map_example 1 <vrfName> */
      if (argc == 3)
      {
        ip6VrRtrAdminModeGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 2: /* Test 2: Set the administrative mode of the router for given VRF: ip6map_example 2 <vrfName> <mode> */
      if (argc == 4)
      {
        value = atoi(argv[3]);
        ip6VrRtrAdminModeSet(&client_handle, argv[2], value);
        show_help = 0;
      }
      break;
    case 3: /* Test 3: Get the dynamic renewal mode of IPv6 neighbors in a given VRF: ip6map_example 3 <vrfName> */
      if (argc == 3)
      {
        ip6VrNbrsDynRenewGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 4: /* Test 4: Set the dynamic renewal mode of IPv6 neighbors in a given VRF: ip6map_example 4 <vrfName> <mode> */
      if (argc == 4)
      {
        value = atoi(argv[3]);
        ip6VrNbrsDynRenewSet(&client_handle, argv[2], value);
        show_help = 0;
      }
      break;
    case 5: /* Test 5: Get the NUD backoff-multiple value set for calculation of timeouts for NS transmissions during NUD for a given VRF: ip6map_example 5 <vrfName> */
      if (argc == 3)
      {
        ip6VrNudBackoffMultipleGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 6: /* Test 6: Set the NUD backoff-multiple value for calculation of timeouts for NS transmissions during NUD for a given VRF: ip6map_example 6 <vrfName> <val> */
      if (argc == 4)
      {
        value = atoi(argv[3]);
        ip6VrNudBackoffMultipleSet(&client_handle, argv[2], value);
        show_help = 0;
      }
      break;
    case 7: /* Test 7: Get the maximum unicast neighbor solicitations sent during NUD for a given VRF: ip6map_example 7 <vrfName> */
      if (argc == 3)
      {
        ip6VrNudMaxUnicastSolicitsGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 8: /* Test 8: Set the maximum unicast neighbor solicitations sent during NUD for a given VRF: ip6map_example 8 <vrfName> <val> */
      if (argc == 4)
      {
        value = atoi(argv[3]);
        ip6VrNudMaxUnicastSolicitsSet(&client_handle, argv[2], value);
        show_help = 0;
      }
      break;
    case 9: /* Test 9: Get the maximum multicast neighbor solicitations sent during NUD for a given VRF: %s 9 <vrfName> */
      if (argc == 3)
      {
        ip6VrNudMaxMcastSolicitsGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 10: /* Test 10: Set the maximum multicast neighbor solicitations sent during NUD for a given VRF: ip6map_example 10 <vrfName> <val> */
      if (argc == 4)
      {
        value = atoi(argv[3]);
        ip6VrNudMaxMcastSolicitsSet(&client_handle, argv[2], value);
        show_help = 0;
      }
      break;
    case 11: /* Test 11: Get the rate limit value set for the unresolved IPv6 data packets coming to CPU: ip6map_example 11 */
      if (argc == 2)
      {
        ip6UnresolvedRateLimitGet(&client_handle);
        show_help = 0;
      }
      break;
    case 12: /* Test 12: Set the rate limit value for the unresolved IPv6 data packets coming to CPU: ip6map_example 12 <val> */
      if (argc == 3)
      {
        value = atoi(argv[2]);
        ip6UnresolvedRateLimitSet(&client_handle, value);
        show_help = 0;
      }
      break;
    case 13: /* Test 13: Get the default IPv6 router TTL value for a given VRF: ip6map_example 13 <vrfName> */
      if (argc == 3)
      {
        ip6VrRtrDefaultTTLGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 14: /* Test 14: Set the default IPv6 router TTL value for a given VRF: ip6map_example 14 <vrfName> <val> */
      if (argc == 4)
      {
        value = atoi(argv[3]);
        ip6VrRtrDefaultTTLSet(&client_handle, argv[2], value);
        show_help = 0;
      }
      break;
    case 15: /* Test 15: Get the administrative state of a given interface: ip6map_example 15 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrIntfModeGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 16: /* Test 16: Get the operational state of a given interface: ip6map_example 16 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrIntfOperModeGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 17: /* Test 17: Set the administrative state of a given interface: ip6map_example 17 <ifNum> <mode> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        value = atoi(argv[3]);
        ip6RtrIntfModeSet(&client_handle, ifNum, value);
        show_help = 0;
      }
      break;
    case 18: /* Test 18: Get the IPv6 AutoState mode of a given interface: ip6map_example 18 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrIntfAutoStateModeGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 19: /* Test 19: Set the IPv6 AutoState mode of a given interface: ip6map_example 19 <ifNum> <mode> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        value = atoi(argv[3]);
        ip6RtrIntfAutoStateModeSet(&client_handle, ifNum, value);
        show_help = 0;
      }
      break;
    case 20: /* Test 20: Determine if IPv6 interface exists: ip6map_example 20 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6IntfExists(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 21: /* Test 21: Determine if the interface is valid for participation in the IPv6 component: ip6map_example 21 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6MapIntfIsValid(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 22: /* Test 22: Determine if the interface is valid to be configured with an IPv6 neighbor: ip6map_example 22 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6MapNbrIntfIsValid(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 23: /* Test 23: Get the IPv6 MTU of a given interface: ip6map_example 23 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrIntfIpv6MtuGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 24: /* Test 24: Get the IPv6 effective MTU of a given interface: ip6map_example 24 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrIntfIpv6EffectiveMtuGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 25: /* Test 25: Set the IPv6 MTU of a given interface: ip6map_example 25 <ifNum> <val> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        value = atoi(argv[3]);
        ip6RtrIntfIpv6MtuSet(&client_handle, ifNum, value);
        show_help = 0;
      }
      break;
    case 26: /* Test 26: Get the interface bandwidth of a given interface: ip6map_example 26 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrIntfBandwidthGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 27: /* Test 27: Get the first valid interface for participation in IPv6 component: ip6map_example 27 */
      if (argc == 2)
      {
        ip6MapIntfValidFirstGet(&client_handle);
        show_help = 0;
      }
      break;
    case 28: /* Test 28: Get the next valid interface for participation in IPv6 component: ip6map_example 28 <prevIfNum> */
      if (argc == 3)
      {
        value = atoi(argv[2]);
        ip6MapIntfValidNextGet(&client_handle, value);
        show_help = 0;
      }
      break;
    case 29: /* Test 29: Get the maximum IP MTU that may be set on an interface: ip6map_example 29 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrIntfMaxIpv6MtuGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 30: /* Test 30: Get IPv6 statistics for the given interface: ip6map_example 30 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ipv6IfStatsGet (&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 31: /* Test 31: Get IPv6 ICMP statistics for the given interface: ip6map_example 31 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ipv6IfIcmpStatsGet (&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 32: /* Test 32: Clear IPv6 statistics for the given interface: ip6map_example 32 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ipv6IfStatsClear (&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 33: /* Test 33: Get total of each IPv6 statistic on all interfaces: ip6map_example 33 */
      if (argc == 2)
      {
        ipv6TotalStatsGet(&client_handle);
        show_help = 0;
      }
      break;
    case 34: /* Test 34: Clear IPv6 statistics on all interfaces: ip6map_example 34 */
      if (argc == 2)
      {
        ipv6TotalStatsClear(&client_handle);
        show_help = 0;
      }
      break;
    case 35: /* Test 35: Get the IPv6 Route table:
              * ip6map_example 35 <vrfname> <bestRouteOnly (0-Flase, 1-True)>
              */
      if (argc == 4)
      {
        value  = atoi(argv[3]);
        ipv6RouteTableGet (&client_handle, argv[2], value);
        show_help = 0;
      }
      break;
    case 36: /* Test 36: Clear IPv6 statistics on all interfaces:
              * ip6map_example 36 <vrfName> <bestRouteOnly (0-Flase, 1-True)>
              */
      if (argc == 4)
      {
        value  = atoi(argv[3]);
        ipv6RouteCountGet (&client_handle, argv[2], value);
        show_help = 0;
      }
      break;
    case 37: /* Test 37: Get the configured IPv6 addresses on the interface:
              * ipv6intfconfig_example 37 <intIfNum> <numAddr>
              */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        value = atoi(argv[3]);
        ip6RtrCfgIntfAddressesGet (&client_handle, ifNum, value);
        show_help = 0;
      }
      break;
    case 38: /* Test 38: Get the number of neighbor solicits configured to be
              * set for duplicate address detection (DAD) on an interface:
              * ipv6intfconfig_example 38 <intIfNum>
              */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrDadTransmitsGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 39: /* Test 39: Set the number of neighbor solicits to be sent
              * for duplicate address detection (DAD) on an interface:
              * ipv6intfconfig_example 39 <intIfNum> <val>
              */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        value = atoi(argv[3]);
        ip6RtrDadTransmitsSet(&client_handle, ifNum, value);
        show_help = 0;
      }
      break;
    case 40: /* Test 40: Get the interval between IPv6 neighbor
              * solicitation retransmissions on an interface:
              * ipv6intfconfig_example 40 <intIfNum>
              */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrNdNsIntervalGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 41: /* Test 41: Set the interval between IPv6 neighbor
              * solicitation retransmissions on an interface:
              * ipv6intfconfig_example 41 <intIfNum> <val>
              */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        value = atoi(argv[3]);
        ip6RtrNdNsIntervalSet(&client_handle, ifNum, value);
        show_help = 0;
      }
      break;
    case 42: /* Test 42: Get the amount of time that a remote IPv6 node is reachable: ipv6intfconfig_example 42 <intIfNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrNdReachableTimeGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 43: /* Test 43: Set the amount of time that a remote IPv6 node is reachable: ipv6intfconfig_example 43 <intIfNum> <val> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        value = atoi(argv[3]);
        ip6RtrNdReachableTimeSet(&client_handle, ifNum, value);
        show_help = 0;
      }
      break;
    case 44: /* Test 44: Get the configured mode for ICMPv6 unreachable messages: ipv6intfconfig_example 44 <intIfNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrIntfICMPUnreachablesModeGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 45: /* Test 45: Set the configured mode for ICMPv6 unreachable messages:
              * ipv6intfconfig_example 45 <intIfNum> <mode (0-Disable, 1-Enable)>
              */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        value = atoi(argv[3]);
        ip6RtrIntfICMPUnreachablesModeSet(&client_handle, ifNum, value);
        show_help = 0;
      }
      break;
    case 46:
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        ipv6AddrEntryGet (&client_handle, ifNum, argv[3]);
        show_help = 0;
      }
      break;
    case 47:
      if (argc == 2)
      {
        ipv6AddrEntryGetNext (&client_handle);
        show_help = 0;
      }
      break;
    case 48:
      if (argc == 5)
      {
        ifNum = atoi(argv[2]);
        value = atoi(argv[4]);
        ipv6AddrPrefixEntryGet (&client_handle, ifNum, argv[3], value);
        show_help = 0;
      }
      break;
    case 49:
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        value = atoi(argv[3]);
        ip6RtrIntfAddressesGet (&client_handle, ifNum, value);
        show_help = 0;
      }
      break;
    case 50:
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrIntfAddressGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 51:
      if (argc == 4)
      {
        ifNum = atoi(argv[3]);
        ipv6NbrTblEntryGet (&client_handle, argv[2], ifNum);
        show_help = 0;
      }
      break;
    case 52:
      if (argc == 4)
      {
        ifNum = atoi(argv[3]);
        ipv6NdpFlush (&client_handle, argv[2], ifNum);
        show_help = 0;
      }
      break;
    case 53:
      if (argc == 2)
      {
        snmpIpv6StaticRouteGetNext (&client_handle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping IP6Map API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}

