/*********************************************************************
*
*  Copyright 2022-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  ipsla_example.c
*
* @purpose   OpEN IPSLA example.
*
* @component OpEN
*
* @create    12/27/2022
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_ipsla.h"

#define OPEN_DISP_ADDR_LEN   46
/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the IPSLA Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test  1: Create an IP SLA operation for the given IP SLA number: %s 1 <ipSlaId>\n", name);
  printf("Test  2: Delete an IP SLA operation for the given IP SLA number: %s 2 <ipSlaId>\n", name);
  printf("Test  3: Schedule an IP SLA operation to start or stop probing: %s 3 <ipSlaId> <toSchedule>\n", name);
  printf("Test  4: Get the config information of the given IP SLA number: %s 4 <ipSlaId>\n", name);
  printf("Test  5: Get the operational info of the given IP SLA number: %s 5 <ipSlaId>\n", name);
  printf("Test  6: To get the IPSLA ID of the first IPSLA probe operational. Entry from the IPSLA Table: %s 6\n", name);
  printf("Test  7: To get the ipSla id of the next IPSLA probe operational entry from the IPSLA Table: %s 7 <ipSlaId>\n", name);
  printf("Test  8: Get the operational status of the given IP SLA number: %s 8 <ipSlaId>\n", name);
  printf("Test  9: Get the tracking status of the given Track number: %s 9 <trackId>\n", name);
  printf("Test 10: Get the config information of the given track number: %s 10 <trackId>\n", name);
  printf("Test 11: Get the operational info of the given track number: %s 11 <trackId>\n", name);
  printf("Test 12: To get the track Id of the first track object's operational entry from the Track Table: %s 12\n", name);
  printf("Test 13: To get the track Id of the next track object's operational entry from the TrackTable: %s 13 <trackId>\n", name);
  printf("Test 14: Create a Track object for the given track number: %s 14 <trackId> <ipSlaId> <trackingType>\n", name);
  printf("Test 15: Delete the Track object for the given track number: %s 15 <trackId>\n", name);
  printf("Test 16: Set the given Track object's Up delay time value: %s 16 <trackId> <upDelayTime>\n", name);
  printf("Test 17: Set the given Track object's Down delay time value: %s 17 <trackId> <downDelayTime>\n", name);
  printf("Test 18: Set the given IP SLA Operation's ICMP-ECHO Type params: %s 18 <ipSlaId> <destIpAddress> <srcIntIfNum>\n", name);
  printf("Test 19: Set the given IP SLA Operation's ICMP-ECHO Frequency value: %s 19 <ipSlaId> <frequency>\n", name);
  printf("Test 20: Set the given IP SLA Operation's ICMP-ECHO Timeout value: %s 20 <ipSlaId> <timeout>\n", name);
  printf("Test 21: Set the given IP SLA Operation's ICMP-ECHO Threshold value: %s 21 <ipSlaId> <threshold>\n", name);
  printf("Test 22: Set the given IP SLA Operation's ICMP-ECHO VRF Name value: %s 22 <ipSlaId> <vrfName>\n", name);
  printf("Test 23: Clear IP SLA statistical information for given IP SLA operation or all IP SLAs: %s 23 <ipSlaId>\n", name);

  return;
}

/**************************************************************************
* @purpose  map L7_AF to AF
*   
* @param    l7af         L7_AF type
* 
* @returns  stack AF
* 
* @comments static use only. here since osapi callers may not have access
* to AF_ values of stack. Values are NOT the same.
* 
* @end 
* 
*************************************************************************/
static uint32_t utilFromL7AF(uint32_t l7af)
{ 
  switch(l7af)
  {
  case 1: /*L7_AF_INET*/
      return AF_INET;
  case 2: /*L7_AF_INET6*/
      return AF_INET6;
  default:
      break;
  }
  return 0; 
} 

static char *utilInetNtop(uint32_t family, char *addr, char *str, uint32_t len)
{ 
   const char *rp;
  
   family = utilFromL7AF(family);
   rp =  inet_ntop(family,addr,str,len);
   return (char *)rp;
}

/**************************************************************************
* @purpose     Convert an IP address from a hex value to an ASCII string 
* 
* @param       ipAddr @b{(input)}  IP address to be converted (host byte order)
* @param       buf    @b{(output)} location to store IP address string
* 
* @returns     void
*
* @comments    Caller must provide an output buffer of at least 16 bytes.
* @comments    'ipAddr' is supplied in HOST BYTE ORDER
* 
* @end
*************************************************************************/
static void utilInetNtoa (uint32_t ipAddr, char *buf)
{
  sprintf(buf, "%d.%d.%d.%d",
           (ipAddr & 0xff000000) >> 24,
           (ipAddr & 0x00ff0000) >> 16,
           (ipAddr & 0x0000ff00) >> 8,
           (ipAddr & 0x000000ff));
}

/*********************************************************************
* @purpose  Convert an OPEN address to a string
*
* @param    openAddr  @b{(input)}  IP address in OPEN structure format
* @param    buf       @b{(output)} Pointer to char buffer
*
* @returns  NULL if conversion failed
*           0 on success
*           2 on error
*
* @notes
*
* @end
*********************************************************************/
/* convert an OPEN address to a string */
static char *utilOpenAddrPrint(open_inet_addr_t *openAddr, char *buf)
{
  if ((!openAddr) || (!buf))
  {
    return NULL;
  }

  memset(buf, 0, OPEN_DISP_ADDR_LEN);

  if (openAddr->family == OPEN_AF_INET)
  {
    utilInetNtoa(openAddr->addr.ipv4, buf);
  }
  else if (openAddr->family == OPEN_AF_INET6)
  {
    utilInetNtop(OPEN_AF_INET6, (char*) &openAddr->addr.ipv6, buf, OPEN_DISP_ADDR_LEN);
  }
  else
  {
    sprintf(buf, OPEN_DISP_ADDR_LEN, "(invalid family)");
  }
  return buf;
}

/*********************************************************************
* @purpose  Create an IP SLA operation for the given IP SLA number.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ipSlaId        @b{(input)}  IPSLA operation number
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaIpSlaIdCreate(openapiClientHandle_t *client_handle, uint32_t ipSlaId)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIpSlaIpSlaIdCreate(client_handle, ipSlaId);
  switch(result)
  {    
    case OPEN_E_NONE:
         printf("\nSuccessfully created an IPSLA operation for the given IP SLA number.\n");
         break;
       
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");     
         break;
  
    default:
         printf("\nERROR: Bad return code trying to create an IP SLA operation for the given IP SLA number. (result = %d)\n", result);
         break;
  }    
       
  return;
}

/*********************************************************************
* @purpose  Delete an IP SLA operation for the given IP SLA number.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ipSlaId       @b{(input)}  IPSLA operation number
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaIpSlaIdDelete(openapiClientHandle_t *client_handle, uint32_t ipSlaId)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIpSlaIpSlaIdDelete(client_handle, ipSlaId);
  switch(result)
  {    
    case OPEN_E_NONE:
         printf("\nSuccessfully deleted an IPSLA operation for the given IP SLA number.\n");
         break;
       
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");     
         break;
  
    default:
         printf("\nERROR: Bad return code trying to delete an IP SLA operation for the given IP SLA number. (result = %d)\n", result);
         break;
  }    
       
  return;
}

/*********************************************************************
* @purpose  Schedule an IP SLA operation to start or stop probing.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ipSlaId       @b{(input)}  IPSLA operation number
* @param    toSchedule    @b{(input)}  Schedule mode (1 for TRUE or 0 for FALSE)
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaIpSlaSchedule(openapiClientHandle_t *client_handle, uint32_t ipSlaId, uint32_t toSchedule)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t openFlag = OPEN_FALSE;

  if (toSchedule > 1)
  {
    printf("\nERROR: Invalid toSchedule value. Expected 0 (for FALSE) or 1 (for TRUE).\n");
    return;
  }

  if (1 == toSchedule)
  {
    openFlag = OPEN_TRUE;
  }

  result = openapiIpSlaIpSlaSchedule(client_handle, ipSlaId, openFlag);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the given schedule mode of IPSLA operation for the given IP SLA number.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
  
    default:
         printf("\nERROR: Bad return code trying to schedule an IP SLA operation to start or stop probing. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Get the config information of the given IP SLA number.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ipSlaId       @b{(input)}  IPSLA operation number
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaIpSlaIdNodeCfgInfoGet(openapiClientHandle_t *client_handle, uint32_t ipSlaId)
{
  open_error_t result = OPEN_E_FAIL;
  openIpSlaCfg_t ipSlaCfgInfo; /* IPSLA's config info */
  char addrStr[46];

  memset(addrStr, 0x0, sizeof(addrStr));
  memset(&ipSlaCfgInfo, 0x0, sizeof(ipSlaCfgInfo));
  result = openapiIpSlaIpSlaIdNodeCfgInfoGet(client_handle, ipSlaId, &ipSlaCfgInfo);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nFollowing is the configuration info of the given IPSLA number: ");
         printf("\n inUse: %u",ipSlaCfgInfo.inUse);
         printf("\n IPSLA ID: %u",ipSlaCfgInfo.ipSlaId);
         printf("\n IPSLA Type: %u",ipSlaCfgInfo.ipSlaType);
         printf("\n isScheduled: %u",ipSlaCfgInfo.isScheduled);
         printf("\n Destination IP address: %s", utilOpenAddrPrint(&ipSlaCfgInfo.u.ipSlaEchoCfgParams.destIpAddr, addrStr));
         printf("\n Source interface: %u",ipSlaCfgInfo.u.ipSlaEchoCfgParams.srcIntIfNum);
         printf("\n Frequency: %u",ipSlaCfgInfo.u.ipSlaEchoCfgParams.frequency);
         printf("\n Threshold in millisec: %u",ipSlaCfgInfo.u.ipSlaEchoCfgParams.threshold);
         printf("\n Timeout in millisec: %u",ipSlaCfgInfo.u.ipSlaEchoCfgParams.timeout);
         printf("\n VRF Name: [%s]\n",ipSlaCfgInfo.u.ipSlaEchoCfgParams.vrfName);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the config information of the given IPSLA number. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the operational info of the given IP SLA number.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    ipSlaId         @b{(input)}  IPSLA operation number
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaIpSlaIdNodeOperInfoGet(openapiClientHandle_t *client_handle, uint32_t ipSlaId)
{
  open_error_t result = OPEN_E_FAIL;
  openIpSlaOperData_t ipSlaOperInfo; /* IPSLA's oper info */
  char addrStr[46];

  memset(addrStr, 0x0, sizeof(addrStr));
  memset(&ipSlaOperInfo, 0x0, sizeof(ipSlaOperInfo));
  result = openapiIpSlaIpSlaIdNodeOperInfoGet(client_handle, ipSlaId, &ipSlaOperInfo);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nFollowing is the operational info of the given IPSLA number: ");
         printf("\n IPSLA ID: %u",ipSlaOperInfo.ipSlaId);
         printf("\n Family:   %u",ipSlaOperInfo.family);
         printf("\n VRF ID:   %u",ipSlaOperInfo.ipSlaVrfId);
         printf("\n isIpSlaEntryActive: %u",ipSlaOperInfo.isIpSlaEntryActive);
         printf("\n If IPv4 routing is enabled on the interface: %u",ipSlaOperInfo.isSrcIntfV4RoutingEnabled);
         printf("\n If IPv6 routing is enabled on the interface: %u",ipSlaOperInfo.isSrcIntfV6RoutingEnabled);
         printf("\n SrcIP to be used for probes for this IPSLA operation:  %s",utilOpenAddrPrint(&ipSlaOperInfo.srcIpAddr,addrStr));
         printf("\n lastProbeRTTvalue: %u",ipSlaOperInfo.lastProbeRTTvalue);
         printf("\n Count of Successful Probes (Return Code is 'OK'): %u",ipSlaOperInfo.numSuccesses);
         printf("\n Count of Failed Probes (Return Code is not 'OK'): %u",ipSlaOperInfo.numFailures);
         printf("\n overThresholdsOccurred:  %u",ipSlaOperInfo.overThresholdsOccurred);
         printf("\n Up time since lastest probe start time: %u milliseconds",ipSlaOperInfo.latestOperationStartUpTime);
         switch (ipSlaOperInfo.probeReturnCode)
         {
           case OPEN_IPSLA_TRACK_RC_OK:
                printf("\n ReturnCode of the last probe: OK\n");
                break;
           case OPEN_IPSLA_TRACK_RC_TIMEOUT:
                printf("\n ReturnCode of the last probe: Timeout\n");
                break;
           case OPEN_IPSLA_TRACK_RC_OVERTHRESHOLD:
                printf("\n ReturnCode of the last probe: OverThreshold\n");
                break;
           default:
                /* OPEN_IPSLA_TRACK_RC_UNKNOWN */
                printf("\n ReturnCode of the last probe: Unknown\n");
                break;
         }
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the operational info of the given IPSLA number. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  To get the IPSLA ID of the first IPSLA probe operational entry from the IPSLA Table.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    isShowRunningCfg @b{(input)}  Is for show running-config output (OPEN_TRUE/OPEN_FALSE)
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaIpSlaIdCfgNodeFirstGet(openapiClientHandle_t *client_handle, OPEN_BOOL_t isShowRunningCfg)
{
  uint32_t ipSlaId = 0; /*IPSLA operation number */
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t openFlag = OPEN_FALSE;

  if (isShowRunningCfg > 1)
  {
    printf("\nERROR: Invalid 'isShowRunningCfg' value. Expected 0 (for FALSE) or 1 (for TRUE).");
    return;
  }

  if (1 == isShowRunningCfg)
  {
    openFlag = OPEN_TRUE;
  }

  result = openapiIpSlaIpSlaIdCfgNodeFirstGet(client_handle, openFlag, &ipSlaId);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPSLA ID of the first IPSLA probe operational entry from the IPSLA Table is %u.\n", ipSlaId);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the IPSLA ID of the first IPSLA probe operational entry from the IPSLA Table. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the IPSLA ID of the next IPSLA probe operational entry from the IPSLA Table.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    ipSlaId           @b{(input)}  Current IPSLA operation number
* @param    isShowRunningCfg  @b{(input)}  Is for show running-config output (OPEN_TRUE/OPEN_FALSE)
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaIpSlaIdCfgNodeNextGet(openapiClientHandle_t *client_handle, uint32_t ipSlaId, OPEN_BOOL_t isShowRunningCfg)
{
  uint32_t nextIpSlaId = 0; /* Next IPSLA operation number */
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t openFlag = OPEN_FALSE;

  if (isShowRunningCfg > 1)
  {
    printf("\nERROR: Invalid 'isShowRunningCfg' value. Expected 0 (for FALSE) or 1 (for TRUE).\n");
    return;
  }

  if (1 == isShowRunningCfg)
  {
    openFlag = OPEN_TRUE;
  }

  result = openapiIpSlaIpSlaIdCfgNodeNextGet(client_handle, ipSlaId, openFlag, &nextIpSlaId);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPSLA ID of the next IPSLA probe operational entry for given current IPSLA ID from the IPSLA Table is %u.\n", nextIpSlaId);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the IPSLA ID of the next IPSLA probe operational entry for given current IPSLA ID from the IPSLA Table. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the operational status of the given IP SLA number.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    ipSlaId             @b{(input)}  Current IPSLA operation number
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaIpSlaIdOperStatusGet(openapiClientHandle_t *client_handle, uint32_t ipSlaId)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t isIpSlaEntryActive; /* IPSLA's oper status(OPEN_TRUE/OPEN_FALSE) */

  result = openapiIpSlaIpSlaIdOperStatusGet(client_handle, ipSlaId, &isIpSlaEntryActive);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe operational status of the given IP SLA number is %u.\n", isIpSlaEntryActive);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");     
         break;
    default:
         printf("\nERROR: Bad return code trying to get the operational status of the given IP SLA number. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the tracking status of the given Track number.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    trackId           @b{(input)}  Track object number
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaTrackIdTrackingStateGet(openapiClientHandle_t *client_handle, uint32_t trackId)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t isTrackStatusUp; /* Track object' tracking status(OPEN_TRUE/OPEN_FALSE) */

  result = openapiIpSlaTrackIdTrackingStateGet(client_handle, trackId, &isTrackStatusUp);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe tracking status of the given Track number is %u.\n", isTrackStatusUp);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");     
         break;
    default:
         printf("\nERROR: Bad return code trying to get the tracking status of the given Track number. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the config information of the given track number.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    trackId       @b{(input)}  Track object number
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaTrackIdNodeCfgInfoGet(openapiClientHandle_t *client_handle, uint32_t trackId)
{
  open_error_t result = OPEN_E_FAIL;
  openIpSlaTrackCfg_t trackCfgInfo; /* track's config info */

  result = openapiIpSlaTrackIdNodeCfgInfoGet(client_handle, trackId, &trackCfgInfo);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nFollowing is the configurational info of the given track number: ");
         printf("\n Track ID: %u",trackCfgInfo.trackId);
         printf("\n If this entry contains a Track object config info: %u",trackCfgInfo.inUse);
         printf("\n Tracked IP SLA operation number: %u",trackCfgInfo.trackedIpSlaId);
         printf("\n Tracking type [(0)Reachability/(1)State]: %u",trackCfgInfo.trackingType);
         printf("\n Time to delay notification of Up event: %u seconds",trackCfgInfo.delayUp);
         printf("\n Time to delay notification of Down event: %u seconds\n",trackCfgInfo.delayDown);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the config information of the given track number. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the operational info of the given track number.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    trackId         @b{(input)}  Track object number
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaTrackIdNodeOperInfoGet(openapiClientHandle_t *client_handle, uint32_t trackId)
{
  open_error_t result = OPEN_E_FAIL;
  openIpSlaTrackOperData_t trackOperInfo; /* track's oper info */

  result = openapiIpSlaTrackIdNodeOperInfoGet(client_handle, trackId, &trackOperInfo);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nFollowing is the operational info of the given track number: ");
         printf("\n Track ID: %u",trackOperInfo.trackId);
         printf("\n Family:   %u",trackOperInfo.family);
         printf("\n isTrackEntryActive: %u",trackOperInfo.isTrackEntryActive);
         printf("\n Tracking/Reachability Status: %u",trackOperInfo.trackingStatus);
         printf("\n Previous Tracking Status: %u",trackOperInfo.prevTrackingStatus);
         printf("\n IgnoreDelayConfig:  %u",trackOperInfo.ignoreDelayConfig);
         printf("\n Time since the last reachability status change update arrived with different value: %u seconds",trackOperInfo.delayStartTime);
         printf("\n Time since the last reachability status change: %u seconds",trackOperInfo.lastStatusChangeUpdateTime);
         printf("\n Count of Status change updates received since TrackEntryActive: %u",trackOperInfo.numStatusChangeUpdates);
         switch (trackOperInfo.latestOperationReturnCode)
         {
           case OPEN_IPSLA_TRACK_RC_OK:
                printf("\n ReturnCode when the last TrackingStatus was changed: OK\n");
                break;
           case OPEN_IPSLA_TRACK_RC_TIMEOUT:
                printf("\n ReturnCode when the last TrackingStatus was changed: Timeout\n");
                break;
           case OPEN_IPSLA_TRACK_RC_OVERTHRESHOLD:
                printf("\n ReturnCode when the last TrackingStatus was changed: OverThreshold\n");
                break;
           default:
                /* OPEN_IPSLA_TRACK_RC_UNKNOWN */
                printf("\n ReturnCode when the last TrackingStatus was changed: Unknown\n");
                break;
         }
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the operational info of the given track number. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  To get the track Id of the first track object's operational entry from the Track Table.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaTrackIdCfgNodeFirstGet(openapiClientHandle_t *client_handle)
{
  uint32_t trackId = 0; /* Track object number */
  open_error_t result = OPEN_E_FAIL;

  result = openapiIpSlaTrackIdCfgNodeFirstGet(client_handle, &trackId);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe track ID of the first track object's operational entry from the Track Table is %u.\n", trackId);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the track ID of the first track object's operational entry from the Track Table. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  To get the track Id of the next track object's operational entry from the TrackTable.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    trackId       @b{(input)}  Current track object number
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaTrackIdCfgNodeNextGet(openapiClientHandle_t *client_handle, uint32_t trackId)
{
  uint32_t nextTrackId = 0; /* Next track object number */
  open_error_t result = OPEN_E_FAIL;

  result = openapiIpSlaTrackIdCfgNodeNextGet(client_handle, trackId, &nextTrackId);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe track ID of the next track object's operational entry from the Track Table is %u.\n", nextTrackId);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the track ID of the next track object's operational entry from the Track Table. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Create a Track object for the given track number.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    trackId       @b{(input)}  Track object number
* @param    ipSlaId       @b{(input)}  IPSLA operation number
* @param    type          @b{(input)}  Tracking-type value (OPEN_IPSLA_TRACKING_TYPE_REACHABILITY=0,`
*                                                           OPEN_IPSLA_TRACKING_TYPE_STATE=1)
*
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaTrackIdCreate(openapiClientHandle_t *client_handle, uint32_t trackId, uint32_t ipSlaId, uint32_t type)
{
  open_error_t result = OPEN_E_FAIL;
  openIpSlaTrackTypes_t trackingType = OPEN_IPSLA_TRACKING_TYPE_REACHABILITY;

  if (type > 1)
  {
    printf("\nERROR: Invalid type value. Expected 0(for REACHABILITY) or 1(for STATE).");
    return;
  }

  if (1 == type)
  {
    trackingType = OPEN_IPSLA_TRACKING_TYPE_STATE;
  }

  result = openapiIpSlaTrackIdCreate(client_handle, trackId, ipSlaId, trackingType);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully created a Track object for the given track number.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");     
         break;
    default:
         printf("\nERROR: Bad return code trying to create a Track object for the given track number. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Delete the Track object for the given track number.
*
* @param    client_handle @b{(input)} Client handle from registration API
* @param    trackId       @b{(input)} Track object number
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaTrackIdDelete(openapiClientHandle_t *client_handle, uint32_t trackId)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIpSlaTrackIdDelete(client_handle, trackId);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully deleted the Track object for the given track number.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to delete the Track object for the given track number. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the given Track object's Up delay time value.
*
* @param    client_handle @b{(input)} Client handle from registration API
* @param    trackId       @b{(input)} Track object number
* @param    upDelayTime   @b{(input)} Up delay time in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaSetTrackOjectsDelayUpTime(openapiClientHandle_t *client_handle, uint32_t trackId, uint32_t upDelayTime)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIpSlaSetTrackOjectsDelayUpTime(client_handle, trackId, upDelayTime);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set given Track object's Up delay time value.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the given Track object's Up delay time value. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the given Track object's Down delay time value.
*
* @param    client_handle @b{(input)} Client handle from registration API
* @param    trackId       @b{(input)} Track object number
* @param    downDelayTime @b{(input)} Down delay time in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaSetTrackOjectsDelayDownTime(openapiClientHandle_t *client_handle, uint32_t trackId, uint32_t downDelayTime)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIpSlaSetTrackOjectsDelayDownTime(client_handle, trackId, downDelayTime);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set given Track object's Down delay time value.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the given Track object's Down delay time value. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the given IP SLA Operation's ICMP-ECHO Type params.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ipSlaId       @b{(input)}  IPSLA operation number
* @param    ipAddrStr     @b{(input)}  Destination Host's IP address in string format
* @param    srcIntIfNum   @b{(input)}  Source interface's internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaIpSlaProbesIcmpEchoTypeSet(openapiClientHandle_t *client_handle, uint32_t ipSlaId, char *ipAddrStr, uint32_t srcIntIfNum)
{
  open_error_t result = OPEN_E_FAIL;
  open_inet_addr_t destIpAddress;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, ipAddrStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  if (OPEN_E_NONE != openapiInetAddrGet(client_handle, &ipBuffdesc, &destIpAddress))
  {
    printf("\nERROR: Invalid IP-address input.\n");
    return;
  }
  
  result = openapiIpSlaIpSlaProbesIcmpEchoTypeSet(client_handle, ipSlaId, destIpAddress, srcIntIfNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the given IP SLA Operation's ICMP-ECHO Type params.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the given IP SLA Operation's ICMP-ECHO Type params. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the given IP SLA Operation's ICMP-ECHO Frequency value.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ipSlaId       @b{(input)}  IPSLA operation number
* @param    frequency     @b{(input)}  Frequency value
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaSetIpSlaProbesFrequency(openapiClientHandle_t *client_handle, uint32_t ipSlaId, uint32_t frequency)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIpSlaSetIpSlaProbesFrequency(client_handle, ipSlaId, frequency);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the given IP SLA Operation's ICMP-ECHO Frequency value.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the given IP SLA Operation's ICMP-ECHO Frequency value. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the given IP SLA Operation's ICMP-ECHO Timeout value.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ipSlaId       @b{(input)}  IPSLA operation number
* @param    timeout       @b{(input)}  Timeout value
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaSetIpSlaProbesTimeout(openapiClientHandle_t *client_handle, uint32_t ipSlaId, uint32_t timeout)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIpSlaSetIpSlaProbesTimeout(client_handle, ipSlaId, timeout);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the given IP SLA Operation's ICMP-ECHO Timeout value.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the given IP SLA Operation's ICMP-ECHO Timeout value. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the given IP SLA Operation's ICMP-ECHO Threshold value.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ipSlaId       @b{(input)}  IPSLA operation number
* @param    threshold     @b{(input)}  Threshold value
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaSetIpSlaProbesThreshold(openapiClientHandle_t *client_handle, uint32_t ipSlaId, uint32_t threshold)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIpSlaSetIpSlaProbesThreshold(client_handle, ipSlaId, threshold);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the given IP SLA Operation's ICMP-ECHO Threshold value.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the given IP SLA Operation's ICMP-ECHO Threshold value. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the given IP SLA Operation's ICMP-ECHO VRF Name value.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ipSlaId       @b{(input)}  IPSLA operation number
* @param    *vrfName      @b{(input)}  VRF name of VRF in which the IP SLA probe is to run
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaSetIpSlaProbesVrfName(openapiClientHandle_t *client_handle, uint32_t ipSlaId, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIpSlaSetIpSlaProbesVrfName(client_handle, ipSlaId, &vrfNameBufd);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the given IP SLA Operation's ICMP-ECHO VRF-Name value.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    case OPEN_E_UNAVAIL:
         printf("\nERROR: VRF package is not supported.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the given IP SLA Operation's ICMP-ECHO VRF-Name value. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Clear IP SLA statistical information for given IP SLA operation or all IP SLAs.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ipSlaId       @b{(input)}  IPSLA operation number incase a specific probe's stats need to be cleared or else zero incase for all IPSLA's.
*
* @returns  none
*
* @end
*********************************************************************/
void ipSlaIpSlaStatisticsClear(openapiClientHandle_t *client_handle, uint32_t ipSlaId)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIpSlaIpSlaStatisticsClear(client_handle, ipSlaId);
  switch(result)
  {
    case OPEN_E_NONE:
         if (ipSlaId)
         {
           printf("\nSuccessfully cleared the IP SLA statistical information for the given IP SLA operation.\n");
         }
         else
         {
           printf("\nSuccessfully cleared the IP SLA statistical information for all IPSLA's.\n");
         }
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to clear IP SLA statistical information for given IP SLA operation or all IP SLAs. (result = %d)\n", result);
         break;
  }
  return;

}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for IPSLA
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum;
  uint32_t ipSlaId = 0;
  uint32_t trackId = 0;
  uint32_t value = 0;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("IPSLA example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting IPSLA API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1: /* Test 1: Create an IP SLA operation for the given IP SLA number: ipsla_example 1 <ipSlaId> */
      if (argc == 3)
      {
        ipSlaId = atoi(argv[2]);
        ipSlaIpSlaIdCreate(&client_handle, ipSlaId);
        show_help = 0;
      }
      break;
    case 2: /* Test 2: Delete an IP SLA operation for the given IP SLA number: ipsla_example 2 <ipSlaId> */
      if (argc == 3)
      {
        ipSlaId = atoi(argv[2]);
        ipSlaIpSlaIdDelete(&client_handle, ipSlaId);
        show_help = 0;
      }
      break;
    case 3: /* Test 3: Schedule an IP SLA operation to start or stop probing: ipsla_example 3 <ipSlaId> <toSchedule> */
      if (argc == 4)
      {
        ipSlaId = atoi(argv[2]);
        value   = atoi(argv[3]);
        ipSlaIpSlaSchedule(&client_handle, ipSlaId, value);
        show_help = 0;
      }
      break;
    case 4: /* Test 4: Get the config information of the given IP SLA number: ipsla_example 4 <ipSlaId> */
      if (argc == 3)
      {
        ipSlaId = atoi(argv[2]);
        ipSlaIpSlaIdNodeCfgInfoGet(&client_handle, ipSlaId);
        show_help = 0;
      }
      break;
    case 5: /* Test 5: Get the operational info of the given IP SLA number: ipsla_example 5 <ipSlaId> */
      if (argc == 3)
      {
        ipSlaId = atoi(argv[2]);
        ipSlaIpSlaIdNodeOperInfoGet(&client_handle, ipSlaId);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 2)
      {
        value = 0;
        ipSlaIpSlaIdCfgNodeFirstGet(&client_handle, value);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 3)
      {
        ipSlaId = atoi(argv[2]);
        value = 0;
        ipSlaIpSlaIdCfgNodeNextGet(&client_handle, ipSlaId, value);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 3)
      {
        ipSlaId = atoi(argv[2]);
        ipSlaIpSlaIdOperStatusGet(&client_handle, ipSlaId);
        show_help = 0;
      }
      break;
    case 9: /* Test  9: Get the tracking status of the given Track number: ipsla_example 9 <trackId> */
      if (argc == 3)
      {
        trackId = atoi(argv[2]);
        ipSlaTrackIdTrackingStateGet(&client_handle, trackId);
        show_help = 0;
      }
      break;
    case 10: /* Test 10: Get the config information of the given track number: ipsla_example 10 <trackId> */
      if (argc == 3)
      {
        trackId = atoi(argv[2]);
        ipSlaTrackIdNodeCfgInfoGet(&client_handle, trackId);
        show_help = 0;
      }
      break;
    case 11: /* Test 11: Get the operational info of the given track number: ipsla_example 11 <trackId> */
      if (argc == 3)
      {
        trackId = atoi(argv[2]);
        ipSlaTrackIdNodeOperInfoGet(&client_handle, trackId);
        show_help = 0;
      }
      break;
    case 12: /* Test 12: To get the track Id of the first track object's operational entry from the Track Table: ipsla_example 12 */
      if (argc == 2)
      {
        ipSlaTrackIdCfgNodeFirstGet(&client_handle);
        show_help = 0;
      }
      break;
    case 13: /* Test 13: To get the track Id of the next track object's operational entry from the TrackTable: ipsla_example 13 <trackId> */
      if (argc == 3)
      {
        trackId = atoi(argv[2]);
        ipSlaTrackIdCfgNodeNextGet(&client_handle, trackId);
        show_help = 0;
      }
      break;
    case 14: /* Test 14: Create a Track object for the given track number: ipsla_example 14 <trackId> <ipSlaId> <trackingType> */
      if (argc == 5)
      {
        trackId = atoi(argv[2]);
        ipSlaId = atoi(argv[3]);
        value   = atoi(argv[4]);
        ipSlaTrackIdCreate(&client_handle, trackId, ipSlaId, value);
        show_help = 0;
      }
      break;
    case 15: /* Test 15: Delete the Track object for the given track number: ipsla_example 15 <trackId> */
      if (argc == 3)
      {
        trackId = atoi(argv[2]);
        ipSlaTrackIdDelete(&client_handle, trackId);
        show_help = 0;
      }
      break;
    case 16: /* Test 16: Set the given Track object's Up delay time value: ipsla_example 16 <trackId> <upDelayTime> */
      if (argc == 4)
      {
        trackId = atoi(argv[2]);
        value   = atoi(argv[3]);
        ipSlaSetTrackOjectsDelayUpTime(&client_handle, trackId, value);
        show_help = 0;
      }
      break;
    case 17: /* Test 17: Set the given Track object's Down delay time value: ipsla_example 17 <trackId> <downDelayTime> */
      if (argc == 4)
      {
        trackId = atoi(argv[2]);
        value   = atoi(argv[3]);
        ipSlaSetTrackOjectsDelayDownTime(&client_handle, trackId, value);
        show_help = 0;
      }
      break;
    case 18: /* Test 18: Set the given IP SLA Operation's ICMP-ECHO Type params: ipsla_example 18 <ipSlaId> <destIpAddress> <srcIntIfNum> */
      if (argc == 5)
      {
        ipSlaId = atoi(argv[2]);
        value   = atoi(argv[4]);
        ipSlaIpSlaProbesIcmpEchoTypeSet(&client_handle, ipSlaId, argv[3], value);
        show_help = 0;
      }
      break;
    case 19: /* Test 19: Set the given IP SLA Operation's ICMP-ECHO Frequency value: ipsla_example 19 <ipSlaId> <frequency> */
      if (argc == 4)
      {
        ipSlaId = atoi(argv[2]);
        value   = atoi(argv[3]);
        ipSlaSetIpSlaProbesFrequency(&client_handle, ipSlaId, value);
        show_help = 0;
      }
      break;
    case 20: /* Test 20: Set the given IP SLA Operation's ICMP-ECHO Timeout value: ipsla_example 20 <ipSlaId> <timeout> */
      if (argc == 4)
      {
        ipSlaId = atoi(argv[2]);
        value   = atoi(argv[3]);
        ipSlaSetIpSlaProbesTimeout(&client_handle, ipSlaId, value);
        show_help = 0;
      }
      break;
    case 21: /* Test 21: Set the given IP SLA Operation's ICMP-ECHO Threshold value: ipsla_example 21 <ipSlaId> <threshold> */
      if (argc == 4)
      {
        ipSlaId = atoi(argv[2]);
        value   = atoi(argv[3]);
        ipSlaSetIpSlaProbesThreshold(&client_handle, ipSlaId, value);
        show_help = 0;
      }
      break;
    case 22: /* Test 22: Set the given IP SLA Operation's ICMP-ECHO VRF Name value: ipsla_example 22 <ipSlaId> <vrfName> */
      if (argc == 4)
      {
        ipSlaId = atoi(argv[2]);
        ipSlaSetIpSlaProbesVrfName(&client_handle, ipSlaId, argv[3]);
        show_help = 0;
      }
      break;
    case 23: /* Test 23: Clear IP SLA statistical information for given IP SLA operation or all IP SLAs: ipsla_example 23 <ipSlaId> */
      if (argc == 3)
      {
        ipSlaId = atoi(argv[2]);
        ipSlaIpSlaStatisticsClear(&client_handle, ipSlaId);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping IPSLA API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}

