/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  mirror_example.c
*
* @purpose   Mirroring APIs Example.
*
* @component OpEN
*
* @comments
*
* @create    06/10/2015
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_mirror.h"


static void printAppMenu()
{
  printf("Usage: mirror_example API <test number> [<arg1>][<arg2>] ... \n");
  printf("Probe type values should be taken from OPENAPI_MIRROR_DIRECTION_t \n\n\n");
  printf("Test 1:Print example program menu : mirror_example 1 \n");
  printf("Test 2:Get port monitoring mode of session: mirror_example 2 <session_id> \n");
  printf("Test 3:Setting port monitoring mode of session: mirror_example 3 <session_id> <mode-0/1> (0-disable, 1- enable)\n");\
  printf("Test 4:Retrieving the list of source interfaces for session : mirror_example 4 <session_id>\n");  
  printf("Test 5:Get the direction to be mirrored for a source port: mirror_example 5 <session_id> <intf_num> \n");
  printf("Test 6:Adding the interface to the source mirrored ports of session: mirror_example 6 <session_id> <intf_num> <probType> \n");
  printf("Test 7:Removing the interface to the source mirrored ports of session: mirror_example 7 <session_id> <intf_num> \n");
  printf("Test 8:Setting the interface to the destination ports of session: mirror_example 8 <session_id> <intf_num>\n");
  printf("Test 9:Get the destination port of session: mirror_example 9 <session_id> \n");\
  printf("Test 10:Remove the destination port of session: mirror_example 10 <session_id> \n");\

  printf("Test 11:Get session count of a port monitoring mode : mirror_example  11 \n");\
  printf("Test 12:Get the status of interface which is configured as destination port : mirror_example  12 <intf_num> \n");
  printf("Test 13:Get the status of interface which is configured as source port : mirror_example  13 <intf_num> \n");
  printf("Test 14:Get the session id in which a particular interface is source port : mirror_example  14 <intf_num> \n");
  printf("Test 15:Check if a port is valid source port: mirror_example  15 <intf_num> \n");

  printf("Test 16:Check if a port is valid as destination port: mirror_example  16 <intf_num> \n");
  printf("Test 17:Get the first valid interface which is configured as source port : mirror_example  17 \n");
  printf("Test 18:Get the next valid interface which is configured as source port : mirror_example  18 <intf_num> \n");
  printf("Test 19:Get the list of all valid interfaces which can be configured as source port : mirror_example  19 \n");
  printf("Test 20:Get the first valid interface which can be configured as destination port : mirror_example  20 \n");
  printf("Test 21:Get the next valid interface which can be configured as destination port : mirror_example  21 <intf_num> \n");
  printf("Test 22:Get the list of all valid interfaces which can be configured as destination port : mirror_example  22 \n");
  printf("Test 23:Attatch a list of VLANS to a session : mirror_example  23 <1,3,5,7> <0/1> (0-delete, 1- add) \n");
  printf("Test 24:Get the list of VLANs Attatched to a session : mirror_example  24 <session_id>\n");
  printf("Test 25:Get the source mirror type (Port based or VLAN based) of a session : mirror_example  25 <session_id>\n");
  printf("Test 26:Get the destination mirror type (Port based or VLAN based) of a session : mirror_example  26 <session_id>\n");
  printf("Test 27:Get the remote source VLAN of a session : mirror_example  27 <session_id>\n");
  printf("Test 28:Set the remote source VLAN of a session : mirror_example  28 <session_id> <vlan_id> <0/1> (0-delete, 1- add)\n");
  printf("Test 29:Get the remote destination VLAN of a session : mirror_example  29 <session_id>\n");
  printf("Test 30:Set the remote destination VLAN of a session : mirror_example  30 <session_id> <vlan_id> <0/1> (0-delete, 1- add)\n");
  printf("Test 31:Get the monitoring reflector port for session : mirror_example  31 <session_id> \n");
  printf("Test 32:Set the monitoring reflector port for session : mirror_example  32 <session_id> <intf_number> <0/1> (0-delete, 1- add) \n");
  printf("Test 33:Get the IP ACL attached a mirroring session : mirror_example  33 <session_id> \n");
  printf("Test 34:Set the IP ACL attached to a mirroring session : mirror_example  34 <session_id> <ip_acl_num> <0/1> (0-delete, 1- add) \n");
  printf("Test 35:Get the MAC ACL attached a mirroring session : mirror_example  35 <session_id> \n");
  printf("Test 36:Set the MAC ACL attached to a mirroring session : mirror_example  36 <session_id> <ip_acl_num> <0/1> (0-delete, 1- add) \n");
  printf("Test 37:Get the RSPAN VLAN id: mirror_example  37  \n");
  printf("Test 38:Get the next RSPAN VLAN id: mirror_example  38 <vlan_id> \n");
  printf("Test 39:Get the RSPAN VLAN list: mirror_example  39  \n");
  printf("Test 40:Set the RSPAN VLAN : mirror_example  40 <vlan_id> <0/1> (0-delete, 1- add)\n");
  printf("Test 41:Check if VLAN is valid RSPAN VLAN : mirror_example  41 <vlan_id> \n");
  printf("Test 42:Check if VLAN is a source of any mirror session : mirror_example  42 <vlan_id> \n");
  printf("Test 43:Get port monitoring session reserved state : mirror_example  43 <session_id> \n");
  printf("Test 44:Set port monitoring session reserved state : mirror_example  44 <session_id> <state>\n");
  printf("Test 45:Removing port monitoring configuration for one sessions: mirror_example 45 <session_id> \n");
  printf("Test 46:Removing port monitoring configuration for all sessions: mirror_example 46 \n");
}

/*******************************************************************
*
* @brief  This is the main function that will demonstrates
*         Mirroring OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result = OPEN_E_NONE;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100] = {0};

  open_buffdesc intfList;
  char buf_intfList[5000] = {0};
  open_buffdesc srcList;
  char buf_srcList[5000] = {0};
  open_buffdesc dstList;
  char buf_dstList[5000] = {0};
  open_buffdesc vlanList;
  char buf_vlanList[5000] = {0};
  open_buffdesc vlanIdListIn;
  char buf_vlanIdListIn[5000] = {0};
  open_buffdesc vlanIdListOut;
  char buf_vlanIdListOut[5000] = {0};
  open_buffdesc macAccessListName;
  char buf_macAccessListName[5000] = {0};
  uint32_t sessionCount = 0;
  uint32_t sessionNum = 1;
  uint32_t intIfNum = 1;
  uint32_t nextIntIfNum = 0;
  uint16_t vlanId = 0;
  uint32_t mirrorType = 0;
  uint32_t remoteVlan = 0;
  uint32_t nextRemoteVlan = 0;
  uint32_t ipAccessListNum = 0;
  uint32_t test_num = 0;
  OPEN_CONTROL_t mode;
  OPEN_CONTROL_t updateAction = OPEN_ENABLE;
  OPEN_CONTROL_t state = OPEN_DISABLE;
  OPENAPI_MIRROR_DIRECTION_t probType = OPENAPI_MIRROR_UNCONFIGURED;
  OPEN_BOOL_t status = OPEN_FALSE;
  OPEN_BOOL_t valid = OPEN_FALSE;

  intfList.pstart = buf_intfList;
  intfList.size = sizeof(buf_intfList);
  srcList.pstart = buf_srcList;
  srcList.size = sizeof(buf_srcList);
  dstList.pstart = buf_dstList;
  dstList.size = sizeof(buf_dstList);
  vlanList.pstart = buf_vlanList;
  vlanList.size = sizeof(buf_vlanList);
  vlanIdListIn.pstart = buf_vlanIdListIn;
  vlanIdListIn.size = sizeof(buf_vlanIdListIn);
  vlanIdListOut.pstart = buf_vlanIdListOut;
  vlanIdListOut.size = sizeof(buf_vlanIdListOut);
  macAccessListName.pstart = buf_macAccessListName;
  macAccessListName.size = sizeof(buf_macAccessListName);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("mirror_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (Result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fails until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Mirroring API example application");

  printf ("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet (&clientHandle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf ("Network OS version = %s\n", switch_os_revision_string);
  }
  else
  {
    printf ("Network OS version retrieve error\n");
  }
  printf ("\n");

  if (argc < 2)
  {
    printAppMenu();
    return 0;
  }
    
  test_num = atoi(argv[1]);
  switch(test_num)
  {
    case 1:
          printAppMenu();
          return 0;
    case 2:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        result = openapiSwPortMonitorModeGet(&clientHandle, sessionNum, &mode);
        if (OPEN_E_NONE == result)
        {
          printf("The port monitoring mode for session %d is %d\n", sessionNum, mode);
        }
        else
        {
          printf("Failed to get the port monitoring mode for session %d. (Result = %d)\n", sessionNum, result);
        }
        break;
    case 3:
        if (argc < 4)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        mode       = atoi(argv[3]);
        result = openapiSwPortMonitorModeSet(&clientHandle, sessionNum, mode);
        if (OPEN_E_NONE == result)
        {
          printf("The port monitoring mode for session %d set with value %d\n", sessionNum, mode);
        }
        else
        {
          printf("Failed to set the port monitoring mode for session %d. (Result = %d)\n", sessionNum, result);
        }
        break;
    case 4:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        result = openapiSwPortMonitorSourcePortsListGet(&clientHandle, sessionNum, &intfList);
        if (OPEN_E_NONE == result)
        {
          printf("The list of source interfaces for session %d is: %s\n", sessionNum, (char *)intfList.pstart);
        }
        else
        {
          printf("Failed to retrive the list of source interfaces for session %d. (Result = %d)\n", 
                 sessionNum, result);
        }
        break;
    case 5:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        result = openapiMirrorSourcePortDirectionGet(&clientHandle, sessionNum, intIfNum, &probType);
        if (OPEN_E_NONE == result)
        {
          printf("The direction to be mirrored for a source port %d is: %d \n", intIfNum, probType);
          printf("\n The value belongs to OPENAPI_MIRROR_DIRECTION_t enum \n");
        }
        else
        {
          printf("Failed to retrive the direction to be mirrored for a source port %d. (Result = %d)\n", 
                 intIfNum, result);
        }
        break;
    case 6:
        if (argc < 5)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        intIfNum = atoi(argv[3]);
        probType = atoi(argv[4]);
        result = openapiSwPortMonitorSourcePortAdd(&clientHandle, sessionNum, intIfNum, probType);
        if (OPEN_E_NONE == result)
        {
          printf("The interface %d added to the mirrored port of session %d with direction %d\n", 
                  intIfNum, sessionNum, probType);
        }
        else
        {
          printf("Failed to add the interface %d to the mirrored ports of session %d. (Result = %d)\n", 
                 intIfNum, sessionNum, result);
        }
        break;
    case 7:
        if (argc < 4)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        intIfNum = atoi(argv[3]);
        result = openapiSwPortMonitorSourcePortRemove(&clientHandle, sessionNum, intIfNum);
        if (OPEN_E_NONE == result)
        {
          printf("The interface %d removed from the mirrored port of session %d\n", 
                 intIfNum, sessionNum);
        }
        else
        {
          printf("Failed to remove the interface %d from the mirrored ports of session %d. (Result = %d)\n", 
                 intIfNum, sessionNum, result);
        }
        break;
    case 8:
        if (argc < 4)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        intIfNum = atoi(argv[3]);
        /* Testing of the setting the destination port for session */
        result = openapiSwPortMonitorDestPortSet(&clientHandle, sessionNum, intIfNum);
        if (OPEN_E_NONE == result)
        {
          printf("The destination port for session %d set with value %d\n", sessionNum, intIfNum);
        }
        else
        {
          printf("Failed to set the destination port for session %d. (Result = %d)\n", 
                 sessionNum, result);
        }
        break;
    case 9:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        /* Testing of the retrieving the destination port for session */
        result = openapiSwPortMonitorDestPortGet(&clientHandle, sessionNum, &intIfNum);
        if (OPEN_E_NONE == result)
        {
          printf("The destination port for session %d is %d\n", sessionNum, intIfNum);
        }
        else
        {
          printf("Failed to retrieve the destination port for session %d. (Result = %d)\n", 
                 sessionNum, result);
        }
        break;
    case 10:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        /* Testing of the removing the destination port of session */
        result = openapiSwPortMonitorDestPortRemove(&clientHandle, sessionNum);
        if (OPEN_E_NONE == result)
        {
          printf("The interface removed from the mirrored port of session %d\n", 
                 sessionNum);
        }
        else
        {
          printf("Failed to remove the interface from the mirrored ports of session %d. (Result = %d)\n", 
                 sessionNum, result);
        }
        break;
    case 11:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        /* Testing of the retrieving count of the sessions of port monitoring mode */
        result = openapiSwPortMonitorSessionCount(&clientHandle, &sessionCount);
        if (OPEN_E_NONE == result)
        {
          printf("The count of the sessions of port monitoring mode %d\n", sessionCount);
        }
        else
        {
          printf("Failed to retrieve the count of the sessions of port monitoring mode. (Result = %d)\n", 
                 result);
        }
        break;
    case 12:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        intIfNum = atoi(argv[2]);
        /* Testing of the determining status of intIfNum configured as destination port */
        result = openapiSwPortMonitorIsDestConfigured(&clientHandle, intIfNum, &sessionNum, &status);
        if (OPEN_E_NONE == result)
        {
          printf("The status of internal interface number %d for session %d as a destination port is %d\n", 
                 intIfNum, sessionNum, status);
        }
        else
        {
          printf("Failed to determine status of internal interface number %d "
                 "configured as destination port. (Result = %d)\n", intIfNum, result);
        }
        break;
    case 13:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        intIfNum = atoi(argv[2]);
        /* Testing of the determining status of intIfNum configured as source port */
        result = openapiSwPortMonitorIsSrcConfigured(&clientHandle, intIfNum, &status);
        if (OPEN_E_NONE == result)
        {
          printf("The status of internal interface number %d as a source port is %d\n", 
                 intIfNum, status);
        }
        else
        {
          printf("Failed to determine status of internal interface number %d "
                 "configured as source port. (Result = %d)\n", intIfNum, result);
        }
        break;
    case 14:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        intIfNum = atoi(argv[2]);
        /* Testing of the retrieving session in which inIfNum is configured as source port */
        result = openapiSwPortMonitorSrcConfiguredSessionGet(&clientHandle, intIfNum, &sessionNum);
        if (OPEN_E_NONE == result)
        {
          printf("The session in which internal interface number %d" 
                 "is configured as source port is %d\n", intIfNum, sessionNum);
        }
        else
        {
          printf("Failed to retrieve the session in which iinternal interface number %d"
                 "is configured as source port. (Result = %d)\n", intIfNum, result);
        }
        break;
    case 15:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        intIfNum = atoi(argv[2]);
        /* Testing of the retrieving status of the validation of interface for a source port */
        result = openapiSwPortMonitorIsValidSrcPort(&clientHandle, intIfNum, &valid);
        if (OPEN_E_NONE == result)
        {
          printf("The status of the validation of interface %d for a source port is %d\n", 
                 intIfNum, valid);
        }
        else
        {
          printf("Failed to retrieve the status of the validation of interface %d"
                 "for a source port. (Result = %d)\n", intIfNum, result);
        }
        break;
    case 16:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        intIfNum = atoi(argv[2]);
        /* Testing of the retrieving status of the validation of interface for a destination port */
        result = openapiSwPortMonitorIsValidDestPort(&clientHandle, intIfNum, &valid);
        if (OPEN_E_NONE == result)
        {
          printf("The status of the validation of interface %d"
                 "for a destination port is %d\n", intIfNum, valid);
        }
        else
        {
          printf("Failed to retrieve the status of the validation of interface %d "
                 "for a destination port. (Result = %d)\n", intIfNum, result);
        }
        break;
    case 17:
        if (argc < 2)
        {
          printAppMenu();
          return 0;
        }
        /* Testing of the retrieving first valid interface that can be configured as source port */
        result = openapiSwPortMonitorValidSrcIntfFirstGet(&clientHandle, &intIfNum);
        if (OPEN_E_NONE == result)
        {
          printf("The first valid interface that can be configured as source port is %d\n", 
                 intIfNum);
        }
        else
        {
          printf("Failed to retrieve the first valid interface that can be configured as source port." 
                 "(Result = %d)\n", result);
        }
        break;
    case 18:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        intIfNum = atoi(argv[2]);
        /* Testing of the retrieving next valid interface that can be configured as source port */
        result = openapiSwPortMonitorValidSrcIntfNextGet(&clientHandle, intIfNum, &nextIntIfNum);
        if (OPEN_E_NONE == result)
        {
          printf("The next valid interface that can be configured as source port is %d\n", nextIntIfNum);
        }
        else
        {
          printf("Failed to retrieve the next valid interface that can be "
                 "configured as source port. (Result = %d)\n", result);
        }
        break;
    case 19:
        if (argc < 2)
        {
          printAppMenu();
          return 0;
        }
        /* Testing of the retrieving the list of valid interfaces that can be configured as source port */
        result = openapiSwPortMonitorValidSrcIntfListGet(&clientHandle, &srcList);
        if (OPEN_E_NONE == result)
        {
          printf("The list of valid interfaces that can be configured as source port is: %s\n", 
                 (char *)srcList.pstart);
        }
        else
        {
          printf("Failed to retrive the list of valid interfaces "
                 "that can be configured as source port. (Result = %d)\n", result);
        }
        break;
    case 20:
        if (argc < 2)
        {
          printAppMenu();
          return 0;
        }
        /* Testing of the retrieving first valid interface that can be configured as destination port */
        result = openapiSwPortMonitorValidDestIntfFirstGet(&clientHandle, &intIfNum);
        if (OPEN_E_NONE == result)
        {
          printf("The first valid interface that can be configured as destination port is %d\n", intIfNum);
        }
        else
        {
          printf("Failed to retrieve the first valid interface that "
                 "can be configured as destination port. (Result = %d)\n", result);
        }
        break;
    case 21:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        intIfNum = atoi(argv[2]);
        /* Testing of the retrieving next valid interface that can be configured as destination port */
        result = openapiSwPortMonitorValidDestIntfNextGet(&clientHandle, intIfNum, &nextIntIfNum);
        if (OPEN_E_NONE == result)
        {
          printf("The next valid interface that can be configured as destination port is %d\n", 
                 nextIntIfNum);
        }
        else
        {
           printf("Failed to retrieve the next valid interface that can be"
                  "configured as destination port. (Result = %d)\n", result);
        }
        break;
    case 22:
        if (argc < 2)
        {
          printAppMenu();
          return 0;
        }
        /* Testing of the retrieving the list of valid interfaces that can be configured as destination port */
        result = openapiSwPortMonitorValidDestIntfListGet(&clientHandle, &dstList);
        if (OPEN_E_NONE == result)
        {
          printf("The list of valid interfaces that can be configured as destination port is: %s\n", 
                 (char *)dstList.pstart);
        }
        else
        {
          printf("Failed to retrive the list of valid interfaces"
                 " that can be configured as destination port. (Result = %d)\n", result);
        }
        break;
    case 23:
        if (argc < 5)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        strcpy((char *)vlanIdListIn.pstart, argv[3]);
        updateAction = atoi(argv[4]);
        vlanIdListIn.size = strlen((char *)vlanIdListIn.pstart) + 1;
        result = openapiSwMonitorVlanListSet(&clientHandle, sessionNum, &vlanIdListIn, updateAction);
        if (OPEN_E_NONE == result)
        {
          printf("The VLAN id and number of VLANs for session %d set with values %s\n", 
                  sessionNum, (char *)vlanIdListIn.pstart);
        }
        else
        {
          printf("Failed to set the VLAN id and number of VLANs for session %d. (Result = %d)\n", 
                 sessionNum, result);
        }
        break;
    case 24:
        if (argc < 4)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        /* Testing of the retrieving VLAN id and number of VLANs attached to the session */
        result = openapiSwMonitorVlanListGet(&clientHandle, sessionNum, &vlanIdListOut);
        if (OPEN_E_NONE == result)
        {
          printf("The VLAN IDS attached to the session %d is %s\n", 
                 sessionNum, (char *)vlanIdListOut.pstart);
        }
        else
        {
          printf("Failed to retrieve the VLAN IDs attached to the session %d. (Result = %d)\n", 
                 sessionNum, result);
        }
        break;
    case 25:
        if (argc < 4)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        /* Testing of the retrieving source mirror type(Port based or VLAN based) for session */
        result = openapiSwMonitorSrcTypeGet(&clientHandle, sessionNum, &mirrorType);
        if (OPEN_E_NONE == result)
        {
          printf("The source mirror type for session %d is %d\n", 
                 sessionNum, mirrorType);
        }
        else
        {
          printf("Failed to retrieve the source mirror type for session %d. (Result = %d)\n", 
                 sessionNum, result);
        }
        break;
    case 26:
        if (argc < 4)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        /* Testing of the retrieving destination mirror type(Port based or VLAN based) for session */
        result = openapiSwMonitorDstTypeGet(&clientHandle, sessionNum, &mirrorType);
        if (OPEN_E_NONE == result)
        {
          printf("The destination mirror type for session %d is %d\n", 
                 sessionNum, mirrorType);
        }
        else
        {
          printf("Failed to retrieve the destination mirror type for session %d. (Result = %d)\n", 
                 sessionNum, result);
        }
        break;
    case 27:
        if (argc < 4)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        /* Testing of the retrieving remote source VLAN for session */
        result = openapiSwMonitorRemoteSrcVlanGet(&clientHandle, sessionNum, &remoteVlan);
        if (OPEN_E_NONE == result)
        {
          printf("The remote source VLAN for session %d is %d\n", 
                 sessionNum, remoteVlan);
        }
        else
        {
          printf("Failed to retrieve the remote source VLAN for session %d. (Result = %d)\n", 
                 sessionNum, result);
        }
        break;
    case 28:
        if (argc < 5)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        vlanId = atoi(argv[3]);
        updateAction = atoi(argv[4]);
        result = openapiSwRemoteMonitorSrcVlanSet(&clientHandle, sessionNum, vlanId, updateAction);
        if (OPEN_E_NONE == result)
        {
          printf("The remote source VLAN for session %d set with value %d\n", 
                 sessionNum, vlanId);
        }
        else
        {
          printf("Failed to set the remote source VLAN for session %d. (Result = %d)\n", 
                 sessionNum, result);
        }
        break;
    case 29:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        /* Testing of the retrieving remote destination VLAN for session */
        result = openapiSwMonitorRemoteDstVlanGet(&clientHandle, sessionNum, &remoteVlan);
        if (OPEN_E_NONE == result)
        {
          printf("The remote destination VLAN for session %d is %d\n", 
                 sessionNum, remoteVlan);
        }
        else
        {
          printf("Failed to retrieve the remote destination VLAN for session %d. (Result = %d)\n", 
                 sessionNum, result);
        }
        break;
    case 30:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        vlanId = atoi(argv[3]);
        updateAction = atoi(argv[4]);
        /* Testing of the setting remote destination VLAN for session */
        result = openapiSwRemoteMonitorDstVlanSet(&clientHandle, sessionNum, vlanId, updateAction);
        if (OPEN_E_NONE == result)
        {
          printf("The remote destination VLAN for session %d set with value %d\n", 
                 sessionNum, vlanId);
        }
        else
        {
          printf("Failed to set the remote destination VLAN for session %d. (Result = %d)\n", 
                 sessionNum, result);
        }
        break;
    case 31:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        /* Testing of the retrieving monitoring reflector port for session */
        result = openapiSwMonitorReflectorPortGet(&clientHandle, sessionNum, &intIfNum);
        if (OPEN_E_NONE == result)
        {
          printf("The monitoring reflector port for session %d is %d\n", 
                 sessionNum, intIfNum);
        }
        else
        {
          printf("Failed to retrieve the monitoring reflector port for session %d. (Result = %d)\n", 
                 sessionNum, result);
        }
        break;
    case 32:
        if (argc < 5)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        vlanId = atoi(argv[3]);
        updateAction = atoi(argv[4]);
        /* Testing of the setting monitoring reflector port for session */
        result = openapiSwMonitorReflectorPortSet(&clientHandle, sessionNum, intIfNum, updateAction);
        if (OPEN_E_NONE == result)
        {
          printf("The monitoring reflector port for session %d set with value %d\n", 
                 sessionNum, intIfNum);
        }
        else
        {
          printf("Failed to set the monitoring reflector port for session %d. (Result = %d)\n", 
                  sessionNum, result);
        }
        break;
    case 33:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        /* Testing of the retrieving IP ACL attached to the mirroring session */
        result = openapiSwMonitorIpAclGet(&clientHandle, sessionNum, &ipAccessListNum);
        if (OPEN_E_NONE == result)
        {
          printf("The IP ACL attached to the mirroring session %d is %d\n", sessionNum, ipAccessListNum);
        }
        else
        {
          printf("Failed to retrieve the IP ACL attached to the mirroring session %d. (Result = %d)\n", 
                 sessionNum, result);
        }
        break;
    case 34:
        if (argc < 5)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        ipAccessListNum = atoi(argv[3]);
        updateAction = atoi(argv[4]);
        /* Testing of the attaching IP ACL to the mirroring session */
        result = openapiSwMonitorIpAclSet(&clientHandle, sessionNum, ipAccessListNum, updateAction);
        if (OPEN_E_NONE == result)
        {
          printf("The IP ACL attached to the mirroring session %d with value %d\n", 
                 sessionNum, ipAccessListNum);
        }
        else
        {
          printf("Failed to attach the IP ACL to the mirroring session %d. (Result = %d)\n", 
                  sessionNum, result);
        }
        break;
    case 35:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        /* Testing of the retrieving MAC ACL attached to the mirroring session */
        result = openapiSwMonitorMacAclGet(&clientHandle, sessionNum, &macAccessListName);
        if (OPEN_E_NONE == result)
        {
          printf("The MAC ACL attached to the mirroring session %d is %s\n", 
                 sessionNum, (char *)macAccessListName.pstart);
        }
        else
        {
          printf("Failed to retrieve the MAC ACL attached to the mirroring session %d. (Result = %d)\n", 
                 sessionNum, result);
        }
        break;
    case 36:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        strcpy((char *)macAccessListName.pstart, argv[3]);
        updateAction = atoi(argv[4]);
        /* Testing of the attaching MAC ACL to the mirroring session */
        result = openapiSwMonitorMacAclSet(&clientHandle, sessionNum, &macAccessListName, updateAction);
        if (OPEN_E_NONE == result)
        {
          printf("The MAC ACL attached to the mirroring session %d with value %s\n", 
                 sessionNum, (char *)macAccessListName.pstart);
        }
        else
        {
          printf("Failed to attach the MAC ACL to the mirroring session %d. (Result = %d)\n", 
                 sessionNum, result);
        }
        break;
    case 37:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        /* Testing of the RSPAN VLAN */
        result = openapiRemoteSpanVlanGet(&clientHandle, &remoteVlan);
        if (OPEN_E_NONE == result)
        {
          printf("The RSPAN VLAN is %d\n", remoteVlan);
        }
        else
        {
          printf("Failed to retrieve the RSPAN VLAN. (Result = %d)\n", result);
        }
        break;
    case 38:
        if (argc < 4)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        remoteVlan = atoi(argv[3]);

        /* Testing of the retrieving next configured RSPAN VLAN */
        result = openapiRemoteSpanVlanNextGet(&clientHandle, remoteVlan, &nextRemoteVlan);
        if (OPEN_E_NONE == result)
        {
          printf("The next configured RSPAN VLAN is %d\n", nextRemoteVlan);
        }
        else
        {
          printf("Failed to retrieve the next configured RSPAN VLAN. (Result = %d)\n", result);
        }
        break;
    case 39:
        if (argc < 2)
        {
          printAppMenu();
          return 0;
        }
        /* Testing of the retrieving the list of RSPAN VLAN */
        result = openapiRemoteSpanVlanListGet(&clientHandle, &vlanList);
        if (OPEN_E_NONE == result)
        {
          printf("The list of RSPAN VLAN is: %s\n", (char *)vlanList.pstart);
        }
        else
        {
          printf("Failed to retrive the list of RSPAN VLAN. (Result = %d)\n", result);
        }
        break;
    case 40:
        if (argc < 2)
        {
          printAppMenu();
          return 0;
        }
        /* Testing of the setting VLAN as RSPAN VLAN */
        remoteVlan = atoi(argv[2]);
        updateAction = atoi(argv[3]);
        result = openapiRemoteSpanVlanSet(&clientHandle, remoteVlan, updateAction);
        if (OPEN_E_NONE == result)
        {
          printf("The VLAN %d set as RSPAN VLAN\n", remoteVlan);
        }
        else
        {
          printf("Failed to set the VLAN %d set as RSPAN VLAN. (Result = %d)\n", remoteVlan, result);
        }
        break;
    case 41:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        remoteVlan = atoi(argv[2]);
        /* Testing of the checking VLAN as RSPAN VLAN */
        result = openapiMirrorIsRspanVlan(&clientHandle, remoteVlan, &status);
        if (OPEN_E_NONE == result)
        {
          printf("The status of VLAN %d as a RSPAN VLAN is %d\n", remoteVlan, status);
        }
        else
        {
         printf("Failed to check status of VLAN %d. (Result = %d)\n", remoteVlan, result);
        }
        break;
    case 42:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        remoteVlan = atoi(argv[2]);
        /* Testing of the checking if the VLAN is source to any of the mirroring sessions */
        result = openapiMirrorIsSrcVlan(&clientHandle, remoteVlan, &status);
        if (OPEN_E_NONE == result)
        {
          printf("The status of VLAN %d as a source to any of the mirroring sessions is %d\n", 
                 remoteVlan, status);
        }
        else
        {
          printf("Failed to check status of VLAN %d. (Result = %d)\n", 
                  remoteVlan, result);
        }
        break;
    case 43:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        /* Testing of the retrieving port monitoring session reserved state */
        result = openapiMirrorSessionReserveStateGet(&clientHandle, sessionNum, &state);
        if (OPEN_E_NONE == result)
        {
          printf("The port monitoring session %d reserved state %d\n", 
                 sessionNum, state);
        }
        else
        {
          printf("Failed to retrieve the reserved state for port monitoring"
                 " session %d. (Result = %d)\n", sessionNum, result);
        }
        break;
    case 44:
        if (argc < 4)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        state = atoi(argv[3]);
        /* Testing of the setting port monitoring session reserved state */
        result = openapiMirrorSessionReserveStateSet(&clientHandle, sessionNum, state);
        if (OPEN_E_NONE == result)
        {
          printf("The port monitoring session %d reserved state set with value %d\n", 
                  sessionNum, state);
        }
        else
        {
          printf("Failed to set the reserved state for port monitoring session %d. (Result = %d)\n", 
                 sessionNum, result);
        }
        break;
    case 45:
        if (argc < 3)
        {
          printAppMenu();
          return 0;
        }
        sessionNum = atoi(argv[2]);
        /* Testing of the removing port monitoring configuration for session */
        result = openapiSwPortMonitorSessionRemove(&clientHandle, sessionNum);
        if (OPEN_E_NONE == result)
        {
          printf("The port monitoring configuration removed for session %d\n", sessionNum);
        }
        else
        {
          printf("Failed to remove the port monitoring configuration removed "
                 "for session %d. (Result = %d)\n", sessionNum, result);
        }
        break;
    case 46:
        if (argc < 2)
        {
          printAppMenu();
          return 0;
        }
        /* Testing of the removing port monitoring configuration for all sessions */
        result = openapiSwPortMonitorConfigRemove(&clientHandle);
        if (OPEN_E_NONE == result)
        {
          printf("The port monitoring configuration removed for all sessions\n");
        }
        else
        {
          printf("Failed to remove the port monitoring configuration removed "
                 "for all sessions. (Result = %d)\n", result);
        }
        break;
    default:
          printAppMenu();
          return 0;
  }

  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Mirroring API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
