#!/mnt/fastpath/usr/bin/python

"""rx_pkt_example.py: OpEN API Packet Receive example"""

import OpEN_py as OpEN
from OpENUtil import *
import socket
import struct
import sys
import os
import ctypes
from sys import version_info

open_ = OpENUtil()

#
# Copyright 2016 Broadcom.
#
#  Licensed under the Apache License, Version 2.0 (the "License")
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#

#
# Python 2.6.6
#
       
DESTINATION_MAC       = 'Destination MAC'
SOURCE_MAC            = 'Source MAC'
VLAN_ID               = 'VLAN ID'
VLAN_PRIORITY         = 'VLAN Priority'
ETHER_TYPE            = 'Ether Type'
DESTINATION_IP        = 'Destination IPv4'
DESTINATION_IP_MASK   = 'Destination Mask'
SOURCE_IP             = 'Source IPv4'
SOURCE_IP_MASK        = 'Source IPv4 Mask'
DESTINATION_IPV6      = 'Destination IPV6'
DESTINATION_IPV6_MASK = 'Destination IPV6 Mask'
SOURCE_IPV6           = 'Source IPV6'
SOURCE_IPV6_MASK      = 'Source IPV6 Mask'
DONE                  = 'Done'

def ip_to_int(addr) :
  """Convert ipv4 string to integer"""

  return struct.unpack("!I", socket.inet_aton(addr))[0]

def ipv6_to_int(addr) :
  """Convert ipv6 string to integer"""

  str_ = socket.inet_pton(socket.AF_INET6, addr)
  a, b = struct.unpack('!2Q', str_)
  return (a << 64) | b

def print_sanity_results(result, test, msg, feat) :
  """Print overall comparison results"""

  if result == OpEN.OPEN_E_UNAVAIL:
    print("Sanity test skipped.")
  elif result == OpEN.OPEN_E_EXISTS:
    print("Sanity Skipped (already exists) - %s - %s." % (msg, feat))
  elif result == OpEN.OPEN_E_NONE and test == True:
    print("Sanity Success - %s - %s." % (msg, feat))
  else:
    print("Sanity Failure - %s - %s." % (msg, feat))

def print_bad_result(result, msg) :
  """Print some general error messages if the result is bad"""
   
  if result == OpEN.OPEN_E_UNAVAIL:
    print("Feature not supported - %s (err %d)." % (msg, result))
  elif result == OpEN.OPEN_E_NOT_FOUND:
      print("Test Skipped (not found) - ", msg)
  elif result != OpEN.OPEN_E_NONE:
    print("Test Failure - %s (err %d)." % (msg, result))

def get_mac_address(prompt) :
  """Get a MAC address from the user and return it as an unsigned char"""

  while True:
    try:
      mac = raw_input(prompt)
      macstr = mac.replace(':', '').decode('hex')
      if len(macstr) == 6:
        return open_.getUCharBuffer(len(macstr),macstr)
      else:
        print("Invalid MAC address.")
    except Exception as e:
      print("%s" % str(e))

def get_ipv4_address(prompt) :
  """ get an IP address from user and return it as int"""

  while True:
    try:
      address = raw_input(prompt)
      return ip_to_int(address)
    except Exception as e:
      print("%s" % str(e))
    except:
      print("Invalid IPv4 address.")

def get_ipv6_address(prompt) :
  """ get an IP address from user and return it as open_inet_addr_t"""

  addr = OpEN.open_inet_addr_t()

  if not socket.has_ipv6:
    print_bad_result(OpEN.OPEN_E_UNAVAIL, prompt)
    return addr

  while True:
    try:
      address = raw_input(prompt)
      addr.addr.ipv6 = ipv6_to_int(address)
      addr.family = socket.AF_INET6
      return addr
    except Exception as e:
      print("%s" % str(e))
    except:
      print("Invalid IPv6 address.")

def done() :
  """ """

def receive_packets(self, agent_num, pkt_cnt, log_file) :
  """Create a socket end-point to receive frames
  Display packets received, else write to log_file if defined. """

  if log_file == None: log = False
  else: log = True

  # Make sure agent isn't already registered
  OpEN.openapiExtAgentPktUnregister(agent_num)

  # Register new agent
  agent_name = "Test Agent"
  try:
    agent_string      = open_.getStringBuffer(len(agent_name) + 1, agent_name)
  except OpENBufferSizeError:
    print("receive_packets: getStringBuffer raised OpENBufferSizeError")
    return
  except TypeError:
    print("receive_packets: getStringBuffer raised TypeError")
    return
  agent_buff        = OpEN.open_buffdesc()
  agent_buff.pstart = agent_string
  agent_buff.size   = len(agent_name) + 1
  result = OpEN.openapiExtAgentPktRegister(agent_num, agent_buff, OpEN.OPEN_TCAM_EXT_AGENT_PKT_RECEIVE)
  print_sanity_results(result, True, 'openapiExtAgentPktRegister', "Register External Agent: (%s)" % agent_name) 

  if result != OpEN.OPEN_E_NONE: return

  # Socket file name is based on agent id
  fn = '/tmp/rx.%05d' % agent_num

  # Remove previous socket file (must not exist)
  try:
    os.unlink(fn)
  except:
    pass

  s = socket.socket(socket.AF_UNIX, socket.SOCK_DGRAM)
  s.bind(fn)

  print("Receiving Packets...")
  if log: f = open(log_file, 'w')

  pkt_p = OpEN.new_open_sysExtAgentPktRxDesc_tp()
  idx = 0

  while (idx < pkt_cnt):    
    data, addr = s.recvfrom(OpEN.OPEN_TXRX_MAX_FRAME_SIZE + 64)
    OpEN.buf_to_open_sysExtAgentPktRxDesc_t(data, pkt_p)
    pkt = OpEN.open_sysExtAgentPktRxDesc_tp_value(pkt_p)

    if log:
      f.write("\nPacket received from ifNum %d, pktLen %d" % (pkt.rxIfNum, pkt.pktLength))
      f.write("\nPacket content:")
    else:
      print("\nPacket #%d Version:%d Received from ifNum %d, pktLen %d" % (idx, pkt.version, pkt.rxIfNum, pkt.pktLength))
      print("Packet content:")

    if version_info >= (3,0,0):
      for i in range(0, pkt.pktLength-1):
        octet = data[pkt.descLen+i:pkt.descLen+i+1].encode('hex')
        if log: f.write('%s ' % octet)
        else: print(octet, end=" ")
    else:
      for i in xrange(0, pkt.pktLength-1):
        octet = data[pkt.descLen+i:pkt.descLen+i+1].encode('hex')
        if log: f.write('%s ' % octet)
        else: print(octet, end=" ")

    if log: f.write('\n')
    else: print()

    idx += 1

  OpEN.openapiExtAgentPktUnregister(agent_num)
  OpEN.delete_open_sysExtAgentPktRxDesc_tp(pkt_p)
  s.close()
  if log: f.close()

class Policy :
  """TCAM API policy"""

  def __init__(self, client) :
    self.m_client = client
    self.m_open_policy = OpEN.open_tcamApiPolicy_t()
    self.m_initialized = False
    self.m_handle = None

  def __del__(self):
    if self.m_handle:
      OpEN.delete_OPEN_TCAM_API_POLICYID_tp(self.m_handle)

  def initialize(self) :
    """Initialize API and get/store the API versioning in the header"""

    result = OpEN.openapiTcamApiInit(self.m_client)
    if result == OpEN.OPEN_E_NONE:
      version = OpEN.open_tcam_version_t()
      result = OpEN.openapiTcamApiVersionGet(self.m_client, version)
      if result == OpEN.OPEN_E_NONE:
        self.m_open_policy.policyHeader.versionInfo.versionMajor = version.versionMajor
        self.m_open_policy.policyHeader.versionInfo.versionMinor = version.versionMinor
        self.m_open_policy.policyHeader.headerLen = sys.getsizeof(self.m_open_policy.policyHeader)
        self.m_initialized = True

  def match_dst_mac(self) :
    """Prompt user for destination mac address and save it"""
    self.m_open_policy.matchDstMac = get_mac_address(DESTINATION_MAC + ' (xx:xx:xx:xx:xx:xx) : ')

  def match_src_mac(self) :
    """Prompt the user for source mac address and save it"""
    self.m_open_policy.matchSrcMac = get_mac_address(SOURCE_MAC + ' (xx:xx:xx:xx:xx:xx) : ')

  def match_vlan(self) :
    """Prompt user for vlan and save it"""

    while True:
      try:
        self.m_open_policy.matchVlanVal = int(raw_input(VLAN_ID + ' : '))
        break
      except Exception as e:
        print("%s" % str(e))
      except:
        print("Invalid VLAN.")

  def match_vlan_priority(self) :
    """Prompt user for vlan priority and save it"""

    while True:
      try:
        self.m_open_policy.matchVlanPrio = int(raw_input(VLAN_PRIORITY + ' : '))
        break
      except Exception as e:
        print("%s" % str(e))
      except:
        print("Invalid VLAN priority.")

  def match_ether_type(self) :
    """Prompt user for ether type and save it"""

    while True:
      try:
        hexstr = raw_input(ETHER_TYPE + ' (xx) : ')
        self.m_open_policy.matchEthType = int(hexstr[0:2], 16)
        break
      except Exception as e:
        print("%s" % str(e))
      except:
        print("Invalid Ether type.")

  def match_dst_ipv4(self) :
    """Prompt user for IP and save it"""

    self.m_open_policy.matchDstIp = get_ipv4_address(DESTINATION_IP + ' : ')
    self.m_open_policy.matchDstIpMask = get_ipv4_address(DESTINATION_IP_MASK + ' : ')

  def match_src_ipv4(self) :
    """Prompt user for IP and save it"""
    self.m_open_policy.matchSrcIp = get_ipv4_address(SOURCE_IP + ' : ')
    self.m_open_policy.matchSrcIpMask = get_ipv4_address(SOURCE_IP_MASK + ' : ')
        
  def match_dst_ipv6(self) :
    """Prompt user for IP and save it"""

    self.m_open_policy.matchDstIpv6Addr = get_ipv6_address(DESTINATION_IPV6 + ' : ')
    self.m_open_policy.matchDstIpv6Mask = get_ipv6_address(DESTINATION_IPV6_MASK + ' : ')

  def match_src_ipv6(self) :
    """Prompt user for IP and save it"""

    self.m_open_policy.matchSrcIpv6Mask = get_ipv6_address(SOURCE_IPV6 + ' : ')
    self.m_open_policy.matchSrcIpv6Mask = get_ipv6_address(SOURCE_IPV6_MASK + ' : ')

  def get_priority_and_type(self) :
    """Prompt user for policy priority and type and save it"""

    while True:
      try:
        policy_priority = int(raw_input("Enter policy priority : "))
        self.m_open_policy.policyPrio = policy_priority
        break
      except:
        print("Invalid priority, priority must be numeric.")
      
    while True:
      try:
        policy_type = int(raw_input("Enter policy type (1:OpenFlow 2:Gen IPv6 3:Egress) : "))
        if policy_type == 1:
          self.m_open_policy.policyType = OpEN.OPEN_TCAM_POLICY_TYPE_OPENFLOW
          break
        elif policy_type == 2:
          self.m_open_policy.policyType = OpEN.OPEN_TCAM_POLICY_TYPE_GEN
          break
        elif policy_type == 3:
          self.m_open_policy.policyType = OpEN.OPEN_TCAM_POLICY_TYPE_EGRESS
          break
        print("Invalid type.")
      except:
        print("Invalid type, type must be numeric.")

  def get_classifiers(self) :
    """Prompt the user for policy classifiers and save them.
    The input loops until the user is done"""

    type_menu = {
      1: self.match_dst_mac,
      2: self.match_src_mac,
      3: self.match_vlan,
      4: self.match_vlan_priority,
      5: self.match_ether_type,
      6: self.match_dst_ipv4,
      7: self.match_src_ipv4,
      8: self.match_dst_ipv6,
      9: self.match_src_ipv6,
      10: done
    }

    while True:
      try:
        print("Enter classification types")
        print("--------------------------")
        print("1: Match", DESTINATION_MAC)
        print("2: Match", SOURCE_MAC)
        print("3: Match", VLAN_ID)
        print("4: Match", VLAN_PRIORITY)
        print("5: Match", ETHER_TYPE)
        print("6: Match", DESTINATION_IP)
        print("7: Match", SOURCE_IP)
        print("8: Match", DESTINATION_IPV6)
        print("9: Match", SOURCE_IPV6)
        print("10:", DONE)

        type = input("Enter selection : ")
        if type == 10:
          break
        if type in type_menu:
          type_menu[type]()

      except:
        pass

  def get_actions(self, agent_num) :
    """The policy action is hard-coded in this example"""

    self.m_open_policy.actionType |= OpEN.OPEN_TCAM_ACTION_REDIRECT_CPU
    self.m_open_policy.ruleNum = agent_num

  def create(self) :
    """Create the actual policy on the device """

    handle_p = OpEN.new_OPEN_TCAM_API_POLICYID_tp()

    result = OpEN.openapiTcamPolicyCreate(self.m_client, self.m_open_policy, handle_p)
    if result == OpEN.OPEN_E_NONE:
      print()
      print("Policy successfully added.")
      self.m_handle = handle_p
    else:
      print()
      print("Error: Policy could not be created")

  def get_interface(self) :
    """ Prompt for interface and add it to the policy"""

    intf_p = OpEN.new_uint32_tp()

    while True:
      try:
        interface = raw_input("Enter interface name (slot/port) : ")
        try:
          name_str = open_.getStringBuffer(len(interface) + 1, interface)
        except OpENBufferSizeError:
          print("get_interface: getStringBuffer raised OpENBufferSizeError")
          return
        except TypeError:
          print("get_interface: getStringBuffer raised TypeError")
          return
        name_buff = OpEN.open_buffdesc()
        name_buff.pstart = name_str
        name_buff.size = len(interface)+1

        result = OpEN.openapiIfNumGet(self.m_client, name_buff, intf_p) 
        print_bad_result(result, "get interface")
        if result == OpEN.OPEN_E_NONE:
          intf = OpEN.uint32_tp_value(intf_p)
          result = OpEN.openapiTcamPolicyIntfAdd(self.m_client, self.m_handle, intf)
          print_bad_result(result, "add interface")
          break
      except:
        print("Invalid Interface.")

    OpEN.delete_uint32_tp(intf_p)

class RxPktExample :
  """Packet transmission """

  def __init__(self, client) :
    self.m_client = client

  def test_rx_pkt(self, agent_num, pkt_cnt, log_file) :
    """Instantiate and create a policy and assign it to an interface.
    Establish a server socket to accept packets from the agent."""

    policy = Policy(self.m_client)
    policy.initialize()
    policy.get_priority_and_type()
    policy.get_classifiers()
    policy.get_actions(agent_num)
    policy.create()
    policy.get_interface()
    receive_packets(self, agent_num, pkt_cnt, log_file)

def main(argv) :
  """Demonstrate OpEN usage for Packet API"""

  # logfile is optional
  if len(argv) < 2 or len(argv) > 3:
    print("rx_pkt_example.py <agentnum> <packetcount> [logfile]")
    sys.exit()

  try:
    agent_num = int(argv[0])
  except:
    print("Invalid agent number, <agentnum> must be numeric.")
    sys.exit()

  try:
    packet_count = int(argv[1])
  except:
    print("Invalid packet count, <packetcount> must be numeric.")
    sys.exit()

  try:
    log_file = argv[2]
    fp = open (log_file, 'w')
    fp.close()
  except:
    log_file = None

  ret = open_.connect("tx_pkt_example")
  if ret == OpEN.OPEN_E_NONE:
    open_.getNetworkOSVersion()
    client = open_.get_client()
    example = RxPktExample(client)
    example.test_rx_pkt(agent_num, packet_count, log_file)
    open_.terminate()
  else:
    print("Unable to connect")
  
if __name__ == '__main__':
  main(sys.argv[1:])


