/*********************************************************************
*
* Copyright 2017-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  pfc_example.c
*
* @purpose   Data center bridging - Priority-based Flow Control API Example.
*
* @component OPEN
*
* @comments
*
* @create    1/31/2017
*
* @end
*
**********************************************************************/

#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_pfc.h"
#include "openapi_ets.h"

static char *pfcModeToStr(OPEN_PFC_MODE_t mode)
{
  char *ret = "Unknown pfc mode";

  if (mode == OPEN_PFC_MODE_DISABLE)
  {
    ret = "disable";
  }   
  else if (mode == OPEN_PFC_MODE_ENABLE)
  {
    ret = "enable";
  }
  return ret;
}

static char *pfcDropModeToStr(OPEN_PFC_PRI_DROP_MODE_t mode)
{
  char *ret = "Unknown pfc drop mode";

  if (mode == OPEN_PFC_PRI_DROP_MODE)
  {
    ret = "drop";
  }   
  else if (mode == OPEN_PFC_PRI_NODROP_MODE)
  {
    ret = "enable";
  }
  return ret;
}

static char *pfcDDRModeToStr(OPEN_PFC_DDR_MODE_t mode)
{
  char *ret = "Unknown pfc deadlock mode";

  if (mode == OPEN_PFC_DDR_MODE_DISABLE)
  {
    ret = "disable";
  }
  else if (mode == OPEN_PFC_DDR_MODE_ENABLE)
  {
    ret = "enable";
  }
  return ret;
}

static char *pfcDDIntToStr(OPEN_PFC_DDR_DETECT_TIME_t val)
{
  char *ret = "Unknown PFC Deadlock Detect Interval";

  if (OPEN_PFC_DDR_DETECT_TIME_10MS == val)
  {
    ret = "10ms";
  }
  else if (OPEN_PFC_DDR_DETECT_TIME_100MS == val)
  {
    ret = "100ms";
  }
  return ret;
}

static char *pfcDDRActionToStr(OPEN_PFC_DDR_ACTION_t val)
{
  char *ret = "Unknown PFC Deadlock Recoveyr Action";

  if (OPEN_PFC_DDR_ACTION_TRANSMIT == val)
  {
    ret = "transmit";
  }
  else if (OPEN_PFC_DDR_ACTION_DROP == val)
  {
    ret = "drop";
  }
  return ret;
}

/*********************************************************************
* @purpose  Set PFC mode for a given interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   interface number
* @param    mode             @b{(input)}   desired mode
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/

void setPfcMode(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_PFC_MODE_t mode)
{
  open_error_t ret;

  ret = openapiIfPfcModeSet(client_handle, intIfNum, mode);
  if (ret == OPEN_E_UNAVAIL)
  {
    printf("unable to set mode %s for interface %d: feature not supported\n", pfcModeToStr(mode), intIfNum);
  }
  else if (ret != OPEN_E_NONE)
  {
    printf("unable to set mode %s for interface %d: error %d\n", pfcModeToStr(mode), intIfNum, ret);
  }
}

/*********************************************************************
* @purpose  Set PFC drop mode for a given interface and priority.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   interface number
* @param    priority         @b{(input)}   priority
* @param    mode             @b{(input)}   desired mode
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/

void setDropMode(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t priority, OPEN_PFC_PRI_DROP_MODE_t mode)
{
  open_error_t ret;

  ret = openapiIfPfcPriorityModeSet(client_handle, intIfNum, priority, mode);
  if (ret == OPEN_E_UNAVAIL)
  {
    printf("unable to set drop mode %s for interface %d and priority %d: feature not supported\n", pfcDropModeToStr(mode), intIfNum, priority);
  }
  else if (ret == OPEN_E_NOT_FOUND)
  {
    printf("unable to set drop mode %s for interface %d and priority %d: priority out of range\n", pfcDropModeToStr(mode), intIfNum, priority);
  }
  else if (ret != OPEN_E_NONE)
  {
    printf("unable to set mode %s for interface %d and priority %d: error %d\n", pfcDropModeToStr(mode), intIfNum, ret, priority);
  }
}

/*********************************************************************
* @purpose  Display PFC status for a given interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/

void showPfc(openapiClientHandle_t *client_handle, uint32_t intIfNum)
{
  open_error_t ret;
  uint32_t incompatibleCfgCount;
  uint32_t numPfcCap;
  uint32_t prioMin;
  uint32_t prioMax;
  uint32_t allowance;
  uint32_t count;
  OPEN_BOOL_t willing;
  OPEN_BOOL_t mbcStatus;
  OPEN_BOOL_t compat;
  OPEN_PFC_PRI_DROP_MODE_t mode;
  OPEN_PFC_STATUS_t pfcStatus;
  OPEN_PFC_MODE_t pfcMode;
  int i;

  printf("Interface %d\n", intIfNum);

  ret = openapiPfcPeerIncompatibleCfgCountGet(client_handle, intIfNum, &incompatibleCfgCount);
  if (ret != OPEN_E_NONE) 
  {
    printf("%s: openapiPfcPeerIncompatibleCfgCountGet failed (%d)\n", 
           __FUNCTION__, ret);
  }
  else
  {
    printf("Incompatible configurations received from peer: %u\n", incompatibleCfgCount);
  }
  for (i = OPEN_DOT1P_MIN_PRIORITY; i <= OPEN_DOT1P_MAX_PRIORITY; i++)
  {
    printf("\nPriority %d\n", i);
    printf("***********\n");
    ret = openapiPfcAdvPriorityModeGet(client_handle, intIfNum, i, &mode);
    if (ret != OPEN_E_NONE) 
    {
      if (ret == OPEN_E_NOT_FOUND)
      {
        printf("openapiPfcAdvPriorityModeGet: Interface not supported\n");
      }
      else
      {
        printf("%s: openapiPfcAdvPriorityModeGet (%d)\n", __FUNCTION__, ret);
      }
    }
    else
    {
      printf("Advertised priority mode:  %u\n", mode);
    }
    ret = openapiPfcPeerPriorityModeGet(client_handle, intIfNum, i, &mode);
    if (ret != OPEN_E_NONE) 
    {
      if (ret == OPEN_E_NOT_FOUND)
      {
        printf("openapiPfcPeerPriorityModeGet: No peer detected or priority out of range\n");
      }
      else
      {
        printf("%s: openapiPfcPeerPriorityModeGet (%d)\n", __FUNCTION__, ret);
      }
    }
    else
    {
      printf("Peer priority mode:  %u\n", mode);
    }
    ret = openapiIfPfcPriorityModeGet(client_handle, intIfNum, i, &mode);
    if (ret != OPEN_E_NONE) 
    {
      printf("%s: openapiIfPfcPriorityModeGet (%d)\n", __FUNCTION__, ret);
    }
    else
    {
      printf("IF PFC priority mode:  %u\n", mode);
    }
    ret = openapiPfcOprPriorityModeGet(client_handle, intIfNum, i, &mode);
    if (ret != OPEN_E_NONE)
    {
      printf("%s: openapiPfcOprPriorityModeGet (%d)\n", __FUNCTION__, ret);
    }
    else
    {
      printf("Operational PFC Priority participation mode:  %u\n", mode);
    }
    ret = openapiIfPfcRxPriorityStatGet(client_handle, intIfNum, i, &count);
    if (ret != OPEN_E_NONE)
    {
      if (ret == OPEN_E_UNAVAIL)
      {
        printf("%s: openapiIfPfcRxPriorityStatGet feature not supported (%d)\n",
               __FUNCTION__, ret);
      }
      else
      {
        printf("%s: openapiIfPfcRxPriorityStatGet failed (%d)\n",
               __FUNCTION__, ret);
      }
    }
    else
    {
      printf("pfc rx priority stat: %u\n", count);
    }
    ret = openapiPfcTxPriorityStatGet(client_handle, intIfNum, i, &count);
    if (ret != OPEN_E_NONE)
    {
      if (ret == OPEN_E_UNAVAIL)
      {
        printf("%s: openapiPfcTxPriorityStatGet feature not supported (%d)\n",
               __FUNCTION__, ret);
      }
      else
      {
        printf("%s: openapiPfcTxPriorityStatGet failed (%d)\n",
               __FUNCTION__, ret);
      }
    }
    else
    {
      printf("pfc tx priority stat: %u\n", count);
    }
  }
  printf("\n");
  ret = openapiPfcPeerWillingGet(client_handle, intIfNum, &willing);
  if (ret != OPEN_E_NONE) 
  {
    if (ret == OPEN_E_NOT_FOUND)
    {
      printf("Peer not detected, unable to report willingness.\n");
    }
    else
    {
      printf("%s: openapiPfcPeerWillingGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("Willingness of peer: %s\n", willing == OPEN_TRUE ? "true":"false");
  }
  ret = openapiPfcPeerMbcStatusGet(client_handle, intIfNum, &mbcStatus);
  if (ret == OPEN_E_NOT_FOUND)
  {
    printf("openapiPfcPeerMbcStatusGet: No peer detected\n");
  }
  else if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPfcPeerMbcStatusGet (%d)\n", __FUNCTION__, ret);
  }
  else
  {
    printf("MACSEC bypass capability of peer: %s\n", mbcStatus == OPEN_TRUE ? "true":"false");
  }
  ret = openapiPfcPeerCapabilityGet(client_handle, intIfNum, &numPfcCap);
  if (ret != OPEN_E_NONE) 
  {
    printf("%s: openapiPfcPeerCapabilityGet failed (%d)\n", 
           __FUNCTION__, ret);
  }
  else
  {
    printf("Capability of peer: %d\n", numPfcCap);
  }
  ret = openapiPfcMinNoDropPriorityGet(client_handle, &prioMin);
  if (ret != OPEN_E_NONE) 
  {
    printf("%s: openapiPfcMinNoDropPriorityGet failed (%d)\n", 
           __FUNCTION__, ret);
  }
  else
  {
    printf("Minimum no drop priority: %u\n", prioMin);
  }
  ret = openapiPfcMaxNoDropPriorityGet(client_handle, &prioMax);
  if (ret != OPEN_E_NONE) 
  {
    printf("%s: openapiPfcMaxNoDropPriorityGet failed (%d)\n", 
           __FUNCTION__, ret);
  }
  else
  {
    printf("Maximum no drop priority: %u\n", prioMax);
  }
  ret = openapiIfPfcStatusGet(client_handle, intIfNum, &pfcStatus);
  if (ret != OPEN_E_NONE) 
  {
    printf("%s: openapiIfPfcStatusGet failed (%d)\n", 
           __FUNCTION__, ret);
  }
  else
  {
    printf("PFC status: %s\n", pfcStatus == OPEN_PFC_STATUS_ACTIVE ? "active":"inactive");
  }
  ret = openapiIfPfcModeGet(client_handle, intIfNum, &pfcMode);
  if (ret != OPEN_E_NONE) 
  {
    printf("%s: openapiIfPfcModeGet failed (%d)\n", 
           __FUNCTION__, ret);
  }
  else
  {
    printf("PFC mode: %s\n", pfcMode == OPEN_PFC_MODE_DISABLE ? "disable":"enable");
  }
  ret = openapiPfcOperLinkDelayAllowanceGet(client_handle, intIfNum, &allowance);
  if (ret != OPEN_E_NONE) 
  {
    printf("%s: openapiPfcOperLinkDelayAllowanceGet failed (%d)\n", 
           __FUNCTION__, ret);
  }
  else
  {
    printf("Link allowance in bits: %u\n", allowance);
  }
  ret = openapiPfcPeerCfgCompatibleGet(client_handle, intIfNum, &compat);
  if (ret != OPEN_E_NONE) 
  {
    if (ret == OPEN_E_NOT_FOUND)
    {
      printf("Peer not detected, unable to determine peer compatibilty.\n");
    }
    else
    {
      printf("%s: openapiPfcPeerCfgCompatibleGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("pfc peer cfg compatible: %s\n", compat == OPEN_TRUE ? "true":"false");
  }
  ret = openapiPfcPeerCompatibleCfgCountGet(client_handle, intIfNum, &count);
  if (ret != OPEN_E_NONE) 
  {
    printf("%s: openapiPfcPeerCompatibleCfgCountGet failed (%d)\n", 
           __FUNCTION__, ret);
  }
  else
  {
    printf("pfc peer cfg compatible count: %u\n", count);
  }
  ret = openapiIfPfcRxStatGet(client_handle, intIfNum, &count);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("%s: openapiIfPfcRxStatGet feature not supported (%d)\n",
           __FUNCTION__, ret);
    }
    else
    {
      printf("%s: openapiIfPfcRxStatGet failed (%d)\n",
           __FUNCTION__, ret);
    }
  }
  else
  {
    printf("pfc rx stat: %u\n", count);
  }
  ret = openapiIfPfcTxStatGet(client_handle, intIfNum, &count);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("%s: openapiIfPfcTxStatGet feature not supported (%d)\n",
           __FUNCTION__, ret);
    }
    else
    {
      printf("%s: openapiIfPfcTxStatGet failed (%d)\n",
           __FUNCTION__, ret);
    }
  }
  else
  {
    printf("pfc tx stat: %u\n", count);
  }
}

/*********************************************************************
* @purpose  Clears the  PFC stats for the given interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void clearIfPfcStats(openapiClientHandle_t *client_handle, uint32_t intIfNum)
{
  open_error_t ret;
  ret = openapiIfPfcStatsClear(client_handle, intIfNum);
  if (OPEN_E_NONE == ret)
  {
    printf("PFC statistics cleared successfully on the interface:%d\n",intIfNum);
  }
  else
  {
    printf("Unable to clear PFC statistics for the interface:%d, error:%d\n", intIfNum, ret);
  }
  return;
}

/*********************************************************************
* @purpose  Set PFC Deadlock detect mode for the priority on an interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   interface number
* @param    priority         @b{(input)}   priority
* @param    mode             @b{(input)}   desired mode
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void setPfcDDMode(openapiClientHandle_t *client_handle, uint32_t intIfNum,
                  uint32_t priority, OPEN_PFC_DDR_MODE_t mode)
{
  open_error_t ret;

  ret = openapiIfPfcDeadlockConfigSet(client_handle, intIfNum, priority, mode);
  if (ret == OPEN_E_UNAVAIL)
  {
    printf("Feature not supported, error:%d\n", ret);
  }
  else if (ret != OPEN_E_NONE)
  {
    printf("Failed to set admin mode for interface %d: error %d\n", intIfNum, ret);
  }
}

/*********************************************************************
* @purpose  Get PFC Deadlock detect mode for all the priorities on the
*           given  interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getPfcDDMode(openapiClientHandle_t *client_handle, uint32_t intIfNum)
{
  open_error_t ret;
  uint32_t pri;
  OPEN_PFC_DDR_MODE_t mode;

  for (pri = OPEN_QOS_PFC_PRIORITY_MIN; pri <= OPEN_QOS_PFC_PRIORITY_MAX; pri++)
  {
    ret = openapiIfPfcDeadlockConfigGet(client_handle, intIfNum, pri, &mode);
    if (OPEN_E_NONE == ret)
    {
      printf("PFC Deadlock Detect mode for Interface:%d, Priority:%d is %s\n", intIfNum, pri, pfcDDRModeToStr(mode));
    }
    else
    {
      printf("Failed to get PFC Deadlock detect mode for Interface:%d, Priority:%d\n", intIfNum, pri);
    }
  }
  return;
}

/*********************************************************************
* @purpose  Set PFC Deadlock detect interval.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    val              @b{(input)}   PFC Deadlock detect interval
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void setPfcDDInterval(openapiClientHandle_t *client_handle,
                      OPEN_PFC_DDR_DETECT_TIME_t val)
{
  open_error_t ret;

  ret = openapiPfcDeadlockDetectIntervalSet(client_handle, val);
  if (ret == OPEN_E_UNAVAIL)
  {
    printf("Feature not supported, error:%d\n", ret);
  }
  else if (ret != OPEN_E_NONE)
  {
    printf("Failed to set PFC Deadlock detection Interval, error: %d\n", ret);
  }
}

/*********************************************************************
* @purpose  Get PFC Deadlock detect interval
*
* @param    client_handle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getPfcDDInterval(openapiClientHandle_t *client_handle)
{
  open_error_t ret;
  OPEN_PFC_DDR_DETECT_TIME_t val;

  ret = openapiPfcDeadlockDetectIntervalGet(client_handle, &val);
  if (OPEN_E_NONE == ret)
  {
    printf("PFC Deadlock Detection Interval :%s\n", pfcDDIntToStr(val));
  }
  else
  {
    printf("Failed to get PFC DD Interval\n");
  }
}

/*********************************************************************
* @purpose  Set PFC Deadlock recovery action.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    val              @b{(input)}   PFC Deadlock recovery action
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void setPfcDDRecoveryAction(openapiClientHandle_t *client_handle, uint32_t val)
{
  open_error_t ret;

  ret = openapiPfcDeadlockRecoveryActionSet(client_handle, val);
  if (ret == OPEN_E_UNAVAIL)
  {
    printf("Feature not supported, error:%d\n", ret);
  }
  else if (ret != OPEN_E_NONE)
  {
    printf("Failed to set PFC Deadlock Recovery Action, error: %d\n", ret);
  }

}

/*********************************************************************
* @purpose  Get PFC Deadlock recovery action
*
* @param    client_handle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getPfcDDRecoveryAction(openapiClientHandle_t *client_handle)
{
  open_error_t ret;
  OPEN_PFC_DDR_ACTION_t val;

  ret = openapiPfcDeadlockRecoveryActionGet(client_handle, &val);
  if (OPEN_E_NONE == ret)
  {
    printf("PFC Deadlock Recovery Action:%s\n", pfcDDRActionToStr(val));
  }
  else
  {
    printf("Failed to get PFC DD Recovery action\n");
  }

}

/*********************************************************************
* @purpose  Set PFC Deadlock detect time for the given priority.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    priority         @b{(input)}   priority
* @param    val              @b{(input)}   PFC Deadlock detect time
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void setPfcDDTime(openapiClientHandle_t *client_handle, uint32_t priority, uint32_t val)
{
  open_error_t ret;

  ret = openapiPfcDeadlockDetectTimeSet(client_handle, priority, val);
  if (ret == OPEN_E_UNAVAIL)
  {
    printf("Feature not supported, error:%d\n", ret);
  }
  else if (ret != OPEN_E_NONE)
  {
    printf("Failed to set PFC Deadlock Detect Time, error: %d\n", ret);
  }

}

/*********************************************************************
* @purpose  Set PFC Deadlock recovery time for the given priority.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    priority         @b{(input)}   priority
* @param    val              @b{(input)}   PFC Deadlock recovery  time
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void setPfcDRTime(openapiClientHandle_t *client_handle, uint32_t priority, uint32_t val)
{
  open_error_t ret;
  ret = openapiPfcDeadlockRecoveryTimeSet(client_handle, priority, val);
  if (ret == OPEN_E_UNAVAIL)
  {
    printf("Feature not supported, error:%d\n", ret);
  }
  else if (ret != OPEN_E_NONE)
  {
    printf("Failed to set PFC Deadlock Recovery Time, error: %d\n", ret);
  }

}

/*********************************************************************
* @purpose  Get PFC Deadlock detect time and recovery time for the given priority.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    priority         @b{(input)}   priority
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getPfcDDRTime(openapiClientHandle_t *client_handle, uint32_t priority)
{
  open_error_t ret;
  uint32_t det_time, rec_time;

  ret = openapiPfcDeadlockDetectTimeGet(client_handle, priority, &det_time, &rec_time);
  if (OPEN_E_NONE == ret)
  {
    printf("For priority %d PFC Deadlock Detect time:%d, Recovery time:%d\n", priority, det_time, rec_time);
  }
  else
  {
    printf("Failed to get the PFC Deadlock detect time and recovery time");
  }

}

void print_usage(const char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Enable(1)/Disable(0) pfc on an interface: %s 1 <intf> <mode>\n", name);
  printf("Test 2: Set PFC drop mode(drop - 0, noDrop - 1) for a given interface and priority: %s 2 <intf> <priority> <dropMode>\n", name);
  printf("Test 3: clear the PFC stats on an interface: %s 3 <intf>\n", name);
  printf("Test 4: Display PFC status for a given interface : %s 4 <intf>\n", name);

  //PFC DD
  printf("Test 5: Enable(1)/Disable(0) deadlock detect for the priority on an interface: %s 5 <intf> <priority> <mode>\n", name);
  printf("Test 6: Gets deadlock detect mode for an interface: %s 6 <intf>\n", name);
  printf("Test 7: Sets the deadlock detection interval[10ms - 0, 100ms - 1]: %s 7 <detect_interval>\n", name);
  printf("Test 8: Gets the deadlock detection interval: %s 8 \n", name);
  printf("Test 9: Sets the deadlock recovery action[Transmit - 0, Drop - 1]: %s 9 <action>\n", name);
  printf("Test 10: Gets the deadlock recovery action: %s 10\n", name);
  printf("Test 11: Sets the deadlock detect time[0 - 1500]: %s 11 <priority> <detect_time>\n", name);
  printf("Test 12: Sets the deadlock recovery time[0 - 1500]: %s 12 <priority> <recovery_time>\n", name);
  printf("Test 13: Gets the deadlock detect time and recovery time for given priority: %s 13 <priority>\n", name);
  exit(0);
}

int main(int argc, char *argv[])
{
  openapiClientHandle_t clientHandle;
  int option = 0;
  int rc;

  if (argc < 2)
  {
    print_usage(argv[0]);
    exit(1);
  }

  option = atoi(argv[1]);
  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((rc = openapiClientRegister("pfc_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", rc);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  printf("\n");

  switch(option)
  {
    case 1:
      if (argc != 4)
      {
        print_usage(argv[0]);
        exit(1);
      }
      setPfcMode(&clientHandle, atoi(argv[2]), atoi(argv[3]));
      break;

    case 2:
      if (argc != 5)
      {
        print_usage(argv[0]);
        exit(1);
      }
      setDropMode(&clientHandle, atoi(argv[2]), atoi(argv[3]), atoi(argv[4]));
      break;
 
    case 3:
      if (argc != 3)
      {
        print_usage(argv[0]);
        exit(1);
      }
      clearIfPfcStats(&clientHandle, atoi(argv[2]));
      break;

    case 4:
      if (argc != 3)
      {
        print_usage(argv[0]);
        exit(1);
      }
      showPfc(&clientHandle, atoi(argv[2]));
      break;

    case 5:
      if (argc != 5)
      {
        print_usage(argv[0]);
        exit(1);
      }
      setPfcDDMode(&clientHandle, atoi(argv[2]), atoi(argv[3]), atoi(argv[4]));
      break;

    case 6:
      if (argc != 3)
      {
        print_usage(argv[0]);
        exit(1);
      }
      getPfcDDMode(&clientHandle, atoi(argv[2]));
      break;

    case 7:
      if (argc != 3)
      {
        print_usage(argv[0]);
        exit(1);
      }
      setPfcDDInterval(&clientHandle, atoi(argv[2]));
      break;

    case 8:
      if (argc != 2)
      {
        print_usage(argv[0]);
        exit(1);
      }
      getPfcDDInterval(&clientHandle);
      break;

    case 9:
      if (argc != 3)
      {
        print_usage(argv[0]);
        exit(1);
      }
      setPfcDDRecoveryAction(&clientHandle, atoi(argv[2]));
      break;

    case 10:
      if (argc != 2)
      {
        print_usage(argv[0]);
        exit(1);
      }
      getPfcDDRecoveryAction(&clientHandle);
      break;

    case 11:
      if (argc != 4)
      {
        print_usage(argv[0]);
        exit(1);
      }
      setPfcDDTime(&clientHandle, atoi(argv[2]), atoi(argv[3]));
      break;

    case 12:
      if (argc != 4)
      {
        print_usage(argv[0]);
        exit(1);
      }
      setPfcDRTime(&clientHandle, atoi(argv[2]), atoi(argv[3]));
      break;

    case 13:
      if (argc != 3)
      {
        print_usage(argv[0]);
        exit(1);
      }
      getPfcDDRTime(&clientHandle, atoi(argv[2]));
      break;

    default:
        print_usage(argv[0]);
        break;
  }
  return 0;
}
