/*********************************************************************
*
*  Copyright 2018-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  pml_example.c
*
* @purpose   Port MAC locking API example.
*
* @component OPEN
*
* @comments
*
* @create    9/10/2018
*
* @end
*
**********************************************************************/

#include <unistd.h>
#include <stdlib.h>
#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_pml.h"

#define MAC_STRING_SIZE 18
/*********************************************************************
* @purpose  Get the global admin mode.
*
* @param    clientHandle    @b{(input)}   Client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getGlobalAdminMode(openapiClientHandle_t *clientHandle)
{
  open_error_t ret;
  OPEN_CONTROL_t mode;

  ret = openapiPmlGlobalAdminModeGet(clientHandle, &mode);
  if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPmlGlobalAdminModeGet failed (%d)\n",
           __FUNCTION__, ret);
  }
  else
  {
    printf("global admin mode %s\n", (mode == OPEN_ENABLE ? "enable" : "disable"));
  }
}

/*********************************************************************
* @purpose  Get whether an interface is a valid PML interface or not.
*
* @param    clientHandle    @b{(input)}   Client handle from registration API
* @param    ifNum           @b{(input)}   Interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getPmlInterfaceIsValid(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t ret;
  OPEN_BOOL_t isValid;

  ret = openapiPmlIntfIsValid(clientHandle, ifNum, &isValid);
  if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPmlIntfIsValid failed (%d)\n",
           __FUNCTION__, ret);
  }
  else
  {
    printf("interface %d is %svalid\n", ifNum, (isValid == OPEN_TRUE ? "" : "not "));
  }
}

/*********************************************************************
* @purpose  Get mode for a valid PML interface.
*
* @param    clientHandle    @b{(input)}   Client handle from registration API
* @param    ifNum           @b{(input)}   Interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getMode(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t ret;
  OPEN_CONTROL_t mode;

  ret = openapiPmlIntfModeGet(clientHandle, ifNum, &mode);
  if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPmlIntfModeGet failed (%d)\n",
           __FUNCTION__, ret);
  }
  else
  {
    printf("interface %d mode %s\n", ifNum, (mode == OPEN_ENABLE ? "enable" : "disable"));
  }
}

/*********************************************************************
* @purpose  Get dynamic limit for a valid PML interface.
*
* @param    clientHandle    @b{(input)}   Client handle from registration API
* @param    ifNum           @b{(input)}   Interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getDynamicLimit(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t ret;
  uint32_t limit;

  ret = openapiPmlIntfDynamicLimitGet(clientHandle, ifNum, &limit);
  if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPmlIntfDynamicLimitGet failed (%d)\n",
           __FUNCTION__, ret);
  }
  else
  {
    printf("interface %d dynamic limit %u\n", ifNum, limit);
  }
}

/*********************************************************************
* @purpose  Get static limit for a valid PML interface.
*
* @param    clientHandle    @b{(input)}   Client handle from registration API
* @param    ifNum           @b{(input)}   Interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getStaticLimit(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t ret;
  uint32_t limit;

  ret = openapiPmlIntfStaticLimitGet(clientHandle, ifNum, &limit);
  if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPmlIntfStaticLimitGet failed (%d)\n",
           __FUNCTION__, ret);
  }
  else
  {
    printf("interface %d static limit %u\n", ifNum, limit);
  }
}

/*********************************************************************
* @purpose  Get violation trap mode for a valid PML interface.
*
* @param    clientHandle    @b{(input)}   Client handle from registration API
* @param    ifNum           @b{(input)}   Interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getViolationTrapMode(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t ret;
  OPEN_CONTROL_t mode;

  ret = openapiPmlIntfViolationTrapModeGet(clientHandle, ifNum, &mode);
  if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPmlIntfViolationTrapModeGet failed (%d)\n",
           __FUNCTION__, ret);
  }
  else
  {
    printf("interface %d violation trap mode %s\n", ifNum, (mode == OPEN_ENABLE ? "enable" : "disable"));
  }
}

/*********************************************************************
* @purpose  Get violation D-Disable mode for a valid PML interface.
*
* @param    clientHandle    @b{(input)}   Client handle from registration API
* @param    ifNum           @b{(input)}   Interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getViolationDDisableMode(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t ret;
  OPEN_CONTROL_t mode;

  ret = openapiPmlIntfViolationDDisableModeGet(clientHandle, ifNum, &mode);
  if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPmlIntfViolationDDisableModeGet failed (%d)\n",
           __FUNCTION__, ret);
  }
  else
  {
    printf("interface %d violation D-Disable mode %s\n", ifNum, (mode == OPEN_ENABLE ? "enable" : "disable"));
  }
}

/*********************************************************************
* @purpose  Get sticky mode for a valid PML interface.
*
* @param    clientHandle    @b{(input)}   Client handle from registration API
* @param    ifNum           @b{(input)}   Interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getStickyMode(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t ret;
  OPEN_CONTROL_t mode;

  ret = openapiPmlIntfStickyModeGet(clientHandle, ifNum, &mode);
  if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPmlIntfStickyModeGet failed (%d)\n",
           __FUNCTION__, ret);
  }
  else
  {
    printf("interface %d sticky mode %s\n", ifNum, (mode == OPEN_ENABLE ? "enable" : "disable"));
  }
}

/*********************************************************************
* @purpose  Display the statically locked addresses on an interface.
*
* @param    clientHandle    @b{(input)}   Client handle from registration API
* @param    ifNum           @b{(input)}   Interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void displayStaticEntries (openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  
  char mac[MAC_STRING_SIZE] = "";
  open_buffdesc macAddr;
  uint32_t      vlanId;
  OPEN_BOOL_t   isSticky;

  macAddr.pstart = mac;
  macAddr.size   = MAC_STRING_SIZE;

  if (OPEN_E_NONE == openapiPmlIntfStaticEntryGetFirst (clientHandle,
                                                        ifNum, &macAddr,
                                                        &vlanId, &isSticky))
  {
    printf ("Statically configured MAC Address VLAN ID Sticky\n");
    printf ("--------------------------------- ------- ------\n");

    do
    {
      printf ("%-33s %7u %6s\n", (char *) macAddr.pstart, vlanId,
              (isSticky == OPEN_TRUE) ? "Yes": "No");  
    } while (OPEN_E_NONE == openapiPmlIntfStaticEntryGetNext(clientHandle,
                                                             ifNum, &macAddr, 
                                                             vlanId, &macAddr,
                                                             &vlanId, &isSticky));
  }
  else
  {
    printf ("No statically locked MAC entries.\n");  
  }
}

/*********************************************************************
* @purpose  Display the dynamically learned addresses on an interface.
*
* @param    clientHandle    @b{(input)}   Client handle from registration API
* @param    ifNum           @b{(input)}   Interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void displayDynamicEntries (openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  
  char mac[MAC_STRING_SIZE] = "";
  open_buffdesc macAddr;
  uint32_t      vlanId;

  macAddr.pstart = mac;
  macAddr.size   = MAC_STRING_SIZE;

  if (OPEN_E_NONE == openapiPmlIntfDynamicEntryGetFirst (clientHandle,
                                                         ifNum, &macAddr,
                                                         &vlanId))
  {
    printf ("Dynamically configured MAC Address VLAN ID\n");
    printf ("---------------------------------- -------\n");

    do
    {
      printf ("%-34s %7u\n", (char *) macAddr.pstart, vlanId);
    } while (OPEN_E_NONE == openapiPmlIntfDynamicEntryGetNext(clientHandle,
                                                              ifNum, &macAddr, 
                                                              vlanId, &macAddr, 
                                                              &vlanId)); 
  }
  else
  {
    printf ("No dynamically learned MAC entries.\n");  
  }
}

/*********************************************************************
* @purpose  Sets the PML global admin mode.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    adminMode       @b{(input)}  PML global admin mode
*
* @returns  none
*
* @end
*********************************************************************/
void pmlGlobalAdminModeSet(openapiClientHandle_t *client_handle, 
                           OPEN_CONTROL_t adminMode)
{
  open_error_t result;

  if ((result = openapiPmlGlobalAdminModeSet(client_handle, adminMode))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the PML global admin mode."
           " (result = %d)\n", result);
  }
  else
  {
    printf("PML global admin mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Set the Port MAC Locking mode for the specified interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    intfMode       @b{(input)}  PML interface admin mode
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                    OPEN_CONTROL_t intfMode)
{
  open_error_t result;

  if ((result = openapiPmlIntfModeSet(client_handle, ifNum, intfMode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the Port MAC Locking mode for "
           "the specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("PML interface mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the Port MAC Locking maximum dynamic limit for the 
*           specified interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    limit          @b{(output)} Maximum dynamic limit
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfDynamicMaxLimitGet(openapiClientHandle_t *client_handle, 
                               uint32_t ifNum, uint32_t *limit)
{
  open_error_t result;

  if ((result = openapiPmlIntfDynamicMaxLimitGet(client_handle, ifNum, 
                                                 limit)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Port MAC Locking maximum"
           " dynamic limit for the specified interface. (result = %d)\n", 
           result);
  }
  else
  {
    printf ("PML interface maximum dynamic limit is %u.\n", *limit);  
  }
  return;
}

/*********************************************************************
* @purpose  Sets the Port MAC Locking dynamic limit for the specified interface.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    ifNum           @b{(input)}  Interface number
* @param    limit           @b{(input)}  Dynamic limit
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfDynamicLimitSet(openapiClientHandle_t *client_handle, 
                            uint32_t ifNum, uint32_t limit)
{
  open_error_t result;

  if ((result = openapiPmlIntfDynamicLimitSet(client_handle, ifNum, limit))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the Port MAC Locking dynamic "
           "limit for the specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("PML interface dynamic limit set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the Port MAC Locking maximum static limit for the 
*           specified interface.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    ifNum           @b{(input)}  Interface number
* @param    limit           @b{(output)} Maximum static limit
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfStaticMaxLimitGet(openapiClientHandle_t *client_handle, 
                              uint32_t ifNum, uint32_t *limit)
{
  open_error_t result;

  if ((result = openapiPmlIntfStaticMaxLimitGet(client_handle, ifNum, limit)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Port MAC Locking maximum "
           "static limit for the specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("PML interface max static limit is %u.\n", *limit);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the Port MAC Locking static count for the specified interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    limit          @b{(output)} Static count
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfStaticCountGet(openapiClientHandle_t *client_handle, 
                           uint32_t ifNum, uint32_t *limit)
{
  open_error_t result;

  if ((result = openapiPmlIntfStaticCountGet(client_handle, ifNum, limit)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Port MAC Locking static "
           "count for the specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("PML interface static count is %u.\n", *limit);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the Port MAC Locking static limit for the specified interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    limit          @b{(input)}  Static limit
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfStaticLimitSet(openapiClientHandle_t *client_handle, 
                           uint32_t ifNum, uint32_t limit)
{
  open_error_t result;

  if ((result = openapiPmlIntfStaticLimitSet(client_handle, ifNum, limit)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the Port MAC Locking static "
           "limit for the specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("PML interface static limit set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the frequency interval between successive SNMP traps.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    interval      @b{(output)} Time interval in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfTrapFrequencyGet(openapiClientHandle_t *client_handle, 
                             uint32_t ifNum, uint32_t *interval)
{
  open_error_t result;

  if ((result = openapiPmlIntfTrapFrequencyGet(client_handle, ifNum, interval)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the frequency interval between "
           "successive SNMP traps. (result = %d)\n", result);
  }
  else
  {
    printf("PML interface trap frequency is %u.\n", *interval);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the frequency interval between successive SNMP traps.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    interval       @b{(input)}  Time interval in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfTrapFrequencySet(openapiClientHandle_t *client_handle, 
                             uint32_t ifNum, uint32_t interval)
{
  open_error_t result;

  if ((result = openapiPmlIntfTrapFrequencySet(client_handle, ifNum, interval)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the frequency interval between "
           "successive SNMP traps. (result = %d)\n", result);
  }
  else
  {
    printf("PML interface trap frequency set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Adds a statically locked entry to an interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    macAddr       @b{(input)}  MAC address
* @param    vlanId        @b{(input)}  VLAN ID
* @param    sticky        @b{(input)}  Sticky
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfStaticEntryAdd(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                           open_buffdesc *macAddr, uint32_t vlanId, 
                           OPEN_BOOL_t sticky)
{
  open_error_t result;

  if ((result = openapiPmlIntfStaticEntryAdd(client_handle, ifNum, macAddr, 
                                             vlanId, sticky)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to adds a statically locked entry to "
           "an interface. (result = %d)\n", result);
  }
  else
  {
    printf("PML static entry added successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Deletes a statically locked entry from an interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    macAddr       @b{(input)}  MAC address
* @param    vlanId        @b{(input)}  VLAN ID
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfStaticEntryDelete(openapiClientHandle_t *client_handle, 
                              uint32_t ifNum, open_buffdesc *macAddr, 
                              uint32_t vlanId)
{
  open_error_t result;

  if ((result = openapiPmlIntfStaticEntryDelete(client_handle, ifNum, 
                                                macAddr, vlanId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to deletes a statically locked entry "
           "from an interface. (result = %d)\n", result);
  }
  else
  {
    printf("PML static entry deleted successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Converts dynamically locked entries to statically locked 
*           entries on this interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfDynamicToStaticMove(openapiClientHandle_t *client_handle, 
                                uint32_t ifNum)
{
  open_error_t result;

  if ((result = openapiPmlIntfDynamicToStaticMove(client_handle, ifNum)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to converts dynamically locked entries "
           "to statically locked entries on this interface. (result = %d)\n", 
           result);
  }
  else
  {
    printf("Dynamic entries are moved to static.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Configures PML sticky mode on this interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    sticky         @b{(input)}  Sticky mode
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfStickyModeSet(openapiClientHandle_t *client_handle, 
                          uint32_t ifNum, OPEN_CONTROL_t sticky)
{
  open_error_t result;

  if ((result = openapiPmlIntfStickyModeSet(client_handle, ifNum, sticky)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to configures PML sticky mode on "
           "this interface. (result = %d)\n", result);
  }
  else
  {
    printf("PML sticky mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the last violation entry for an interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    macAddr       @b{(output)} MAC address
* @param    vlanId        @b{(output)} VLAN ID
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfLastViolationAddrGet(openapiClientHandle_t *client_handle, 
                                 uint32_t ifNum, open_buffdesc *macAddr, 
                                 uint32_t *vlanId)
{
  open_error_t result;

  if ((result = openapiPmlIntfLastViolationAddrGet(client_handle, ifNum, 
                                                   macAddr, vlanId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the last violation entry "
           "for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Last violation entry details are MAC address : %s, VLAN id %u.\n",
           (char *) macAddr->pstart, *vlanId);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the count of dynamcially learned mac Addresses.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    count          @b{(output)} Learned dynamic address count
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfDynamicLearnedAddrCountGet(openapiClientHandle_t *client_handle, 
                                       uint32_t ifNum, uint32_t *count)
{
  open_error_t result;

  if ((result = openapiPmlIntfDynamicLearnedAddrCountGet(client_handle, ifNum,
                                                         count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the count of dynamcially "
           "learned mac Addresses. (result = %d)\n", result);
  }
  else
  {
    printf("PML dynamic learned address count is %u.\n", *count);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the count of statically learned mac Addresses.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    count         @b{(output)} Static address count
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfStaticLearnedAddrCountGet(openapiClientHandle_t *client_handle, 
                                      uint32_t ifNum, uint32_t *count)
{
  open_error_t result;

  if ((result = openapiPmlIntfStaticLearnedAddrCountGet(client_handle, ifNum, 
                                                        count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the count of statically "
           "learned mac Addresses. (result = %d)\n", result);
  }
  else
  {
    printf("PML static address count is %u.\n", *count);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the count of statically learned mac Addresses.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vlanId        @b{(input)}  VLAN ID
* @param    count         @b{(output)} Static address count
*
* @returns  none
*
* @end
*********************************************************************/
void pmlStaticAddrCountByVlanGet(openapiClientHandle_t *client_handle, 
                                 uint32_t vlanId, uint32_t *count)
{
  open_error_t result;

  if ((result = openapiPmlStaticAddrCountByVlanGet(client_handle, vlanId, count)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the count of statically learned "
           "mac Addresses. (result = %d)\n", result);
  }
  else
  {
    printf("PML static address count for a vlan is %u.\n", *count);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the global count of statically learned mac Addresses.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    count          @b{(output)} Learned static address count
*
* @returns  none
*
* @end
*********************************************************************/
void pmlStaticAddrGlobalCountGet(openapiClientHandle_t *client_handle, 
                                 uint32_t *count)
{
  open_error_t result;

  if ((result = openapiPmlStaticAddrGlobalCountGet(client_handle, count)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the global count of statically"
           " learned mac Addresses. (result = %d)\n", result);
  }
  else
  {
    printf("PML global static address count is %u.\n", *count);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the port MAC locking violation D-Disable mode for the 
*           specified interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    mode          @b{(input)}  Violation D-Disable Mode
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfViolationDDisableModeSet(openapiClientHandle_t *client_handle, 
                                     uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiPmlIntfViolationDDisableModeSet(client_handle, ifNum, 
                                                       mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the port MAC locking violation"
           " D-Disable mode for the specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("PML interface violation D-Disable mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the PML vlan admin mode.
*
* @param    client_handle  @b{(input)}   Client handle from registration API
* @param    adminMode      @b{(output)}  PML vlan admin mode
*
* @returns  none
*
* @end
*********************************************************************/
void pmlVlanAdminModeGet(openapiClientHandle_t *client_handle,
                         OPEN_CONTROL_t *adminMode)
{
  open_error_t result;

  if ((result = openapiPmlVlanAdminModeGet(client_handle, adminMode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the PML vlan admin mode. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("PML vlan admin mode is %u.\n", *adminMode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the PML vlan admin mode.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    adminMode      @b{(input)}  PML vlan admin mode
*
* @returns  none
*
* @end
*********************************************************************/
void pmlVlanAdminModeSet(openapiClientHandle_t *client_handle, 
                         OPEN_CONTROL_t adminMode)
{
  open_error_t result;

  if ((result = openapiPmlVlanAdminModeSet(client_handle, adminMode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the PML vlan admin mode."
           " (result = %d)\n", result);
  }
  else
  {
    printf("PML VLAN admin mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the MAC limit per Vlan.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    vlanId         @b{(input)}  VLAN ID
* @param    limit          @b{(output)} MAC lock limit per vlan
*
* @returns  none
*
* @end
*********************************************************************/
void pmlVlanDynamicLimitGet(openapiClientHandle_t *client_handle, 
                            uint32_t vlanId, uint32_t *limit)
{
  open_error_t result;

  if ((result = openapiPmlVlanDynamicLimitGet(client_handle, vlanId, limit)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the MAC limit per Vlan. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("PML vlan dynamic limit is %u.\n", *limit);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the vlan MAC Locking limit.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vlanId        @b{(input)}  VLAN ID
* @param    limit         @b{(input)}  MAC lock limit per vlan
*
* @returns  none
*
* @end
*********************************************************************/
void pmlVlanDynamicLimitSet(openapiClientHandle_t *client_handle, 
                            uint32_t vlanId, uint32_t limit)
{
  open_error_t result;

  if ((result = openapiPmlVlanDynamicLimitSet(client_handle, vlanId, limit)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the vlan MAC Locking limit."
           " (result = %d)\n", result);
  }
  else
  {
    printf("PML vlan dynamic limit set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets operational vlan MAC locking limit.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    vlanId         @b{(input)}  VLAN ID
* @param    limit          @b{(output)} MAC lock operational limit per vlan
*
* @returns  none
*
* @end
*********************************************************************/
void pmlVlanOperationalLimitGet(openapiClientHandle_t *client_handle, 
                                uint32_t vlanId, uint32_t *limit)
{
  open_error_t result;

  if ((result = openapiPmlVlanOperationalLimitGet(client_handle, vlanId, limit)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get operational vlan MAC locking limit."
           " (result = %d)\n", result);
  }
  else
  {
    printf("PML vlan operational count is %u.\n", *limit);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the MAC Locking violation D-Disable mode for the specified vlan.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vlanId        @b{(input)}  VLAN ID
* @param    mode          @b{(output)} Violation D-Disable Mode
*
* @returns  none
*
* @end
*********************************************************************/
void pmlVlanViolationDDisableModeGet(openapiClientHandle_t *client_handle, 
                                     uint32_t vlanId, OPEN_BOOL_t *mode)
{
  open_error_t result;

  if ((result = openapiPmlVlanViolationDDisableModeGet(client_handle, vlanId, mode))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the MAC Locking violation "
           "D-Disable mode for the specified vlan. (result = %d)\n", result);
  }
  else
  {
    printf("PML vlan d-disable mode is %u.\n", *mode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the MAC Locking violation D-Disable mode for the specified vlan.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vlanId        @b{(input)}  VLAN ID
* @param    mode          @b{(input)}  Violation D-Disable Mode
*
* @returns  none
*
* @end
*********************************************************************/
void pmlVlanViolationDDisableModeSet(openapiClientHandle_t *client_handle, 
                                     uint32_t vlanId, OPEN_BOOL_t mode)
{
  open_error_t result;

  if ((result = openapiPmlVlanViolationDDisableModeSet(client_handle, vlanId, mode))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the MAC Locking violation "
           "D-Disable mode for the specified vlan. (result = %d)\n", result);
  }
  else
  {
    printf("PML vlan d-disable mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the MAC Locking violation trap mode for the specified vlan.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vlanId        @b{(input)}  VLAN ID
* @param    mode          @b{(output)} Violation trap Mode
*
* @returns  none
*
* @end
*********************************************************************/
void pmlVlanViolationTrapModeGet(openapiClientHandle_t *client_handle, 
                                 uint32_t vlanId, OPEN_BOOL_t *mode)
{
  open_error_t result;

  if ((result = openapiPmlVlanViolationTrapModeGet(client_handle, vlanId, mode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the MAC Locking violation trap"
           " mode for the specified vlan. (result = %d)\n", result);
  }
  else
  {
    printf("PML vlan violation trap mode is %u.\n", *mode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the MAC Locking violation trap mode for the specified vlan.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vlanId        @b{(input)}  VLAN ID
* @param    mode          @b{(input)}  Violation trap Mode
*
* @returns  none
*
* @end
*********************************************************************/
void pmlVlanViolationTrapModeSet(openapiClientHandle_t *client_handle, 
                                 uint32_t vlanId, OPEN_BOOL_t mode)
{
  open_error_t result;

  if ((result = openapiPmlVlanViolationTrapModeSet(client_handle, vlanId, mode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the MAC Locking violation "
           "trap mode for the specified vlan. (result = %d)\n", result);
  }
  else
  {
    printf("PML vlan violation trap mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the MAC Locking mode for the specified vlan.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vlanId        @b{(input)}  VLAN ID
* @param    mode          @b{(output)} VLAN MAC Locking mode
*
* @returns  none
*
* @end
*********************************************************************/
void pmlVlanModeGet(openapiClientHandle_t *client_handle, uint32_t vlanId, 
                    uint32_t *mode)
{
  open_error_t result;

  if ((result = openapiPmlVlanModeGet(client_handle, vlanId, mode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the MAC Locking mode "
           "for the specified vlan. (result = %d)\n", result);
  }
  else
  {
    printf("PML vlan mode is %u.\n", *mode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the MAC Locking mode for the specified vlan.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vlanId        @b{(input)}  VLAN ID
* @param    mode          @b{(input)}  VLAN MAC Locking mode
*
* @returns  none
*
* @end
*********************************************************************/
void pmlVlanModeSet(openapiClientHandle_t *client_handle, uint32_t vlanId, 
                    uint32_t mode)
{
  open_error_t result;

  if ((result = openapiPmlVlanModeSet(client_handle, vlanId, mode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the MAC Locking mode for the "
           "specified vlan. (result = %d)\n", result);
  }
  else
  {
    printf("PML vlan mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the Port MAC Locking vlan dynamic limit for the specified 
*           interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    vlanId        @b{(input)}  VLAN ID
* @param    limit         @b{(output)} MAC lock limit per vlan per interface
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfVlanDynamicLimitGet(openapiClientHandle_t *client_handle, 
                                uint32_t ifNum, uint32_t vlanId, 
                                uint32_t *limit)
{
  open_error_t result;

  if ((result = openapiPmlIntfVlanDynamicLimitGet(client_handle, ifNum, vlanId,
                                                  limit)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Port MAC Locking vlan dynamic"
            " limit for the specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("PML interface vlan dynamic limit is %u.\n", *limit);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the Port MAC Locking vlan dynamic limit for the specified interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    vlanId        @b{(input)}  VLAN ID
* @param    limit         @b{(input)}  MAC lock limit per vlan per interface
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfVlanDynamicLimitSet(openapiClientHandle_t *client_handle, 
                                uint32_t ifNum, uint32_t vlanId, uint32_t limit)
{
  open_error_t result;

  if ((result = openapiPmlIntfVlanDynamicLimitSet(client_handle, ifNum, vlanId,
                                                  limit)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the Port MAC Locking vlan dynamic"
           " limit for the specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("PML interface vlan dynamic limit set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the Port MAC Locking vlan dynamic learned address count 
*           for the specified interface and vlan.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    vlanId        @b{(input)}  VLAN ID
* @param    count         @b{(output)} MAC lock learned address count per 
*                                      vlan per interface
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfVlanDynamicLearnedAddrCountGet(openapiClientHandle_t *client_handle, 
                                           uint32_t ifNum, uint32_t vlanId, 
                                           uint32_t *count)
{
  open_error_t result;

  if ((result = openapiPmlIntfVlanDynamicLearnedAddrCountGet(client_handle, ifNum,
                                                             vlanId, count)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Port MAC Locking vlan dynamic "
           "learned address count for the specified interface and vlan. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("PML vlan dynamic learned address count is %u.\n", *count);
  }
  return;
}

/*********************************************************************
* @purpose  Gets secure MAC address aging time for the specified interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    ageTime       @b{(output)} Age time in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfAgeTimeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                       uint32_t *ageTime)
{
  open_error_t result;

  if ((result = openapiPmlIntfAgeTimeGet(client_handle, ifNum, ageTime)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get secure MAC address aging time "
           "for the specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("PML interface age time is %u.\n", *ageTime);
  }
  return;
}

/*********************************************************************
* @purpose  Sets secure MAC address aging time for the specified interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    ageTime       @b{(input)}  Age time in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void pmlIntfAgeTimeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                       uint32_t ageTime)
{
  open_error_t result;

  if ((result = openapiPmlIntfAgeTimeSet(client_handle, ifNum, ageTime)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set secure MAC address aging time "
           "for the specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("PML interface age time set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the next valid PML interface.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    ifNum           @b{(input)}  Interface number
* @param    nextIfNum       @b{(output)} Next valid PML Interface number
*
* @returns  none
*
* @end
*********************************************************************/
void pmlNextValidIntfGet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                         uint32_t *nextIfNum)
{
  open_error_t result;

  if ((result = openapiPmlNextValidIntfGet(client_handle, ifNum, nextIfNum)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next valid PML interface."
           " (result = %d)\n", result);
  }
  else
  {
    printf("PML next valid interface is %u.\n", *nextIfNum);
  }
  return;
}

/*********************************************************************
* @purpose  Display the dynamically learned addresses on an interface per vlan.
*
* @param    clientHandle    @b{(input)}   Client handle from registration API
* @param    ifNum           @b{(input)}   Interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void displayVlanDynamicEntries (openapiClientHandle_t *clientHandle, 
                                uint32_t ifNum)
{
  
  char mac[MAC_STRING_SIZE] = "";
  open_buffdesc macAddr;
  uint32_t      vlanId;

  macAddr.pstart = mac;
  macAddr.size   = MAC_STRING_SIZE;

  if (OPEN_E_NONE == openapiPmlVlanDynamicEntryGetFirst (clientHandle,
                                                         ifNum, &macAddr,
                                                         &vlanId))
  {
    printf ("Dynamically configured MAC Address VLAN ID\n");
    printf ("---------------------------------- -------\n");

    do
    {
      printf ("%-34s %7u\n", (char *) macAddr.pstart, vlanId);
    } while (OPEN_E_NONE == openapiPmlVlanDynamicEntryGetNext (clientHandle,
                                                               ifNum, &macAddr, 
                                                               &macAddr, &vlanId)); 
  }
  else
  {
    printf ("No dynamically learned MAC entries.\n");  
  }
}

/*********************************************************************
* @purpose  Associate the 802.1Qci stream with static PML entry.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    macAddr       @b{(input)}  MAC address
* @param    vlanId        @b{(input)}  VLAN ID
* @param    streamID      @b{(input)}  802.1QCI StreamID
*
* @returns  none
*
* @end
*********************************************************************/
void pmlDot1qciStreamSet(openapiClientHandle_t *client_handle, uint32_t ifNum,
                         open_buffdesc *macAddr, uint32_t vlanId, 
                         uint32_t streamID)
{
  open_error_t result;

  if ((result = openapiPmlDot1qciStreamSet(client_handle, ifNum, macAddr, 
                                           vlanId, streamID)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to associate the 802.1Qci stream with static "
           "PML entry. (result = %d)\n", result);
  }
  else
  {
    printf("Association of 802.1Qci stream with static PML entry is successful.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Deassociate the 802.1Qci stream with static PML entry.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    macAddr       @b{(input)}  MAC address
* @param    vlanId        @b{(input)}  VLAN ID
* @param    streamID      @b{(input)}  802.1QCI StreamID
*
* @returns  none
*
* @end
*********************************************************************/
void pmlDot1qciStreamDelete(openapiClientHandle_t *client_handle, 
                            uint32_t ifNum, open_buffdesc *macAddr, 
                            uint32_t vlanId, uint32_t streamID)
{
  open_error_t result;

  if ((result = openapiPmlDot1qciStreamDelete(client_handle, ifNum, macAddr,
                                              vlanId, streamID)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to deassociate the 802.1Qci stream with "
           "static PML entry. (result = %d)\n", result);
  }
  else
  {
    printf("Deassociation of 802.1Qci stream with static PML entry is "
           "successful.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the 802.1Qci stream ID associated with static PML entry.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    macAddr       @b{(input)}  MAC address
* @param    vlanId        @b{(input)}  VLAN ID
* @param    streamID      @b{(output)} 802.1QCI StreamID
*
* @returns  none
*
* @end
*********************************************************************/
void pmlDot1qciStreamGet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                         open_buffdesc *macAddr, uint32_t vlanId, 
                         uint32_t *streamID)
{
  open_error_t result;

  if ((result = openapiPmlDot1qciStreamGet(client_handle, ifNum, macAddr, 
                                           vlanId, streamID)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1Qci stream ID associated "
           "with static PML entry. (result = %d)\n", result);
  }
  else
  {
    printf("802.1Qci stream ID associated with static PML entry is %u.\n",
           *streamID);
  }
  return;
}

/*********************************************************************
* @purpose  Set violation trap mode for a PML interface.
*
* @param    clientHandle    @b{(input)}   Client handle from registration API
* @param    ifNum           @b{(input)}   Interface number
* @param    mode            @b{(input)}   Violation trap Mode
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void setViolationTrapMode(openapiClientHandle_t *clientHandle, uint32_t ifNum, 
                          OPEN_CONTROL_t trapMode)
{
  open_error_t ret;

  ret = openapiPmlIntfViolationTrapModeSet(clientHandle, ifNum, trapMode);
  if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPmlIntfViolationTrapModeSet failed (%d)\n",
           __FUNCTION__, ret);
  }
  else
  {
    printf("Violation trap mode set successfully.\n");
  }
}

/*******************************************************************
*
* @purpose  This function prints usage for the example program.
*
* @param    name    @b{(input)}   application name
*
* @returns  none
*
* @end
*********************************************************************/
void printAppMenu(char *name)
{
  printf("\nUsage: %s <test#> <arg1> <arg2> ...\n\n", name);
  printf("Test  1: get the global admin mode: %s 1\n", name);
  printf("Test  2: get whether interface is valid PML interface: %s 2 <intf>\n", name);
  printf("Test  3: get the mode for a valid PML interface: %s 3 <intf>\n", name);
  printf("Test  4: get the dynamic limit for a valid PML interface: %s 4 <intf>\n", name);
  printf("Test  5: get the static limit for a valid PML interface: %s 5 <intf>\n", name);
  printf("Test  6: get the violation trap mode for a valid PML interface: %s 6 <intf>\n", name);
  printf("Test  7: get the violation D-Disable mode for a valid PML interface: %s 7 <intf>\n", name);
  printf("Test  8: get the sticky mode for a valid PML interface: %s 8 <intf>\n", name);
  printf("Test  9: Display statically locked MAC addresses on an interface.: %s 9 <intf>\n", name);
  printf("Test 10: Display dynamically learned MAC addresses on an interface.: %s 10 <intf>\n", name);
  printf("Test 11: Sets the PML global admin mode: %s 11 <adminMode>\n", name);
  printf("Test 12: Set the Port MAC Locking mode for the specified interface: %s 12 <ifNum> <intfMode>\n", name);
  printf("Test 13: Gets the Port MAC Locking maximum dynamic limit for the specified interface: %s 13 <ifNum>\n", name);
  printf("Test 14: Sets the Port MAC Locking dynamic limit for the specified interface: %s 14 <ifNum> <limit>\n", name);
  printf("Test 15: Gets the Port MAC Locking maximum static limit for the specified interface: %s 15 <ifNum>\n", name);
  printf("Test 16: Gets the Port MAC Locking static count for the specified interface: %s 16 <ifNum>\n", name);
  printf("Test 17: Sets the Port MAC Locking static limit for the specified interface: %s 17 <ifNum> <limit>\n", name);
  printf("Test 18: Gets the frequency interval between successive SNMP traps: %s 18 <ifNum>\n", name);
  printf("Test 19: Sets the frequency interval between successive SNMP traps: %s 19 <ifNum> <interval>\n", name);
  printf("Test 20: Adds a statically locked entry to an interface: %s 20 <ifNum> <macAddr> <vlanId> <sticky>\n", name);
  printf("Test 21: Deletes a statically locked entry from an interface: %s 21 <ifNum> <macAddr> <vlanId>\n", name);
  printf("Test 22: Converts dynamically locked entries to statically locked entries on this interface: %s 22 <ifNum>\n", name);
  printf("Test 23: Configures PML sticky mode on this interface: %s 23 <ifNum> <sticky>\n", name);
  printf("Test 24: Gets the last violation entry for an interface: %s 24 <ifNum>\n", name);
  printf("Test 25: Gets the count of dynamcially learned mac Addresses: %s 25 <ifNum>\n", name);
  printf("Test 26: Gets the count of statically learned mac Addresses: %s 26 <ifNum>\n", name);
  printf("Test 27: Gets the count of statically learned mac Addresses: %s 27 <vlanId>\n", name);
  printf("Test 28: Gets the global count of statically learned mac Addresses: %s 28\n", name);
  printf("Test 29: Sets the port MAC locking violation D-Disable mode for the specified interface: %s 29 <ifNum> <mode>\n", name);
  printf("Test 30: Gets the PML vlan admin mode: %s 30\n", name);
  printf("Test 31: Sets the PML vlan admin mode: %s 31 <adminMode>\n", name);
  printf("Test 32: Gets the MAC limit per Vlan: %s 32 <vlanId>\n", name);
  printf("Test 33: Sets the vlan MAC Locking limit: %s 33 <vlanId> <limit>\n", name);
  printf("Test 34: Gets operational vlan MAC locking limit: %s 34 <vlanId>\n", name);
  printf("Test 35: Gets the MAC Locking violation D-Disable mode for the specified vlan: %s 35 <vlanId>\n", name);
  printf("Test 36: Sets the MAC Locking violation D-Disable mode for the specified vlan: %s 36 <vlanId> <mode>\n", name);
  printf("Test 37: Gets the MAC Locking violation trap mode for the specified vlan: %s 37 <vlanId>\n", name);
  printf("Test 38: Sets the MAC Locking violation trap mode for the specified vlan: %s 38 <vlanId> <mode>\n", name);
  printf("Test 39: Gets the MAC Locking mode for the specified vlan: %s 39 <vlanId>\n", name);
  printf("Test 40: Sets the MAC Locking mode for the specified vlan: %s 40 <vlanId> <mode>\n", name);
  printf("Test 41: Gets the Port MAC Locking vlan dynamic limit for the specified interface: %s 41 <ifNum> <vlanId>\n", name);
  printf("Test 42: Sets the Port MAC Locking vlan dynamic limit for the specified interface: %s 42 <ifNum> <vlanId> <limit>\n", name);
  printf("Test 43: Gets the Port MAC Locking vlan dynamic learned address count for the specified interface and vlan: %s 43 <ifNum> <vlanId>\n", name);
  printf("Test 44: Gets secure MAC address aging time for the specified interface: %s 44 <ifNum>\n", name);
  printf("Test 45: Sets secure MAC address aging time for the specified interface: %s 45 <ifNum> <ageTime>\n", name);
  printf("Test 46: Gets the next valid PML interface: %s 46 <ifNum>\n", name);
  printf("Test 47: Display dynamically learned MAC addresses on an interface.: %s 47 <intf>\n", name);
  printf("Test 48: Associate the 802.1Qci stream with static PML entry: %s 48 <ifNum> <macAddr> <vlanId> <streamID>\n", name);
  printf("Test 49: Deassociate the 802.1Qci stream with static PML entry: %s 49 <ifNum> <macAddr> <vlanId> <streamID>\n", name);
  printf("Test 50: Get the 802.1Qci stream ID associated with static PML entry: %s 50 <ifNum> <macAddr> <vlanId> <streamID>\n", name);
  printf("Test 51: Set the violation trap mode for a PML interface: %s 51 <intf> <mode>\n", name);
  printf("Test 52: run negative tests: %s 52 <intf>\n", name);
  return;
}

/*******************************************************************
*
* @purpose  Run API negative tests.
*
* @param    clientHandle    @b{(input)}   Client handle from registration API
* @param    intf            @b{(input)}   interface
*
* @returns  none
*
* @end
*********************************************************************/
void runNegativeTests(openapiClientHandle_t *clientHandle, uint32_t intf)
{
  OPEN_CONTROL_t mode;
  OPEN_BOOL_t valid;
  uint32_t invalidIntf = 0xffffffff;
  open_error_t result;
  uint32_t value;

  /* openapiPmlGlobalAdminModeGet() */
  printf("\nTesting openapiPmlGlobalAdminModeGet(): \n");
  result = openapiPmlGlobalAdminModeGet(NULL, &mode);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPmlGlobalAdminModeGet(clientHandle, NULL);
  printf("NULL mode arg. (result = %d)\n", result);

  /* openapiPmlIntfIsValid() */
  printf("\nTesting openapiPmlIntfIsValid(): \n");
  result = openapiPmlIntfIsValid(NULL, intf, &valid);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPmlIntfIsValid(clientHandle, intf, NULL);
  printf("NULL valid arg. (result = %d)\n", result);
  result = openapiPmlIntfIsValid(clientHandle, invalidIntf, &valid);
  printf("Likely invalid intf arg. (result = %d)\n", result);

  /* openapiPmlIntfModeGet() */
  printf("\nTesting openapiPmlIntfModeGet(): \n");
  result = openapiPmlIntfModeGet(NULL, intf, &mode);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPmlIntfModeGet(clientHandle, intf, NULL);
  printf("NULL mode arg. (result = %d)\n", result);
  result = openapiPmlIntfModeGet(clientHandle, invalidIntf, &mode);
  printf("Likely invalid intf arg. (result = %d)\n", result);

  /* openapiPmlIntfDynamicLimitGet() */
  printf("\nTesting openapiPmlIntfDynamicLimitGet(): \n");
  result = openapiPmlIntfDynamicLimitGet(NULL, intf, &value);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPmlIntfDynamicLimitGet(clientHandle, intf, NULL);
  printf("NULL limit argument. (result = %d)\n", result);
  result = openapiPmlIntfDynamicLimitGet(clientHandle, invalidIntf, &value);
  printf("Likely invalid intf arg. (result = %d)\n", result);

  /* openapiPmlIntfStaticLimitGet() */
  printf("\nTesting openapiPmlIntfStaticLimitGet(): \n");
  result = openapiPmlIntfStaticLimitGet(NULL, intf, &value);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPmlIntfStaticLimitGet(clientHandle, intf, NULL);
  printf("NULL limit argument. (result = %d)\n", result);
  result = openapiPmlIntfStaticLimitGet(clientHandle, invalidIntf, &value);
  printf("Likely invalid intf arg. (result = %d)\n", result);

  /* openapiPmlIntfViolationTrapModeGet() */
  printf("\nTesting openapiPmlIntfViolationTrapModeGet(): \n");
  result = openapiPmlIntfViolationTrapModeGet(NULL, intf, &mode);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPmlIntfViolationTrapModeGet(clientHandle, intf, NULL);
  printf("NULL mode arg. (result = %d)\n", result);
  result = openapiPmlIntfViolationTrapModeGet(clientHandle, invalidIntf, &mode);
  printf("Likely invalid intf arg. (result = %d)\n", result);

  /* openapiPmlIntfViolationDDisableModeGet() */
  printf("\nTesting openapiPmlIntfViolationDDisableModeGet(): \n");
  result = openapiPmlIntfViolationDDisableModeGet(NULL, intf, &mode);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPmlIntfViolationDDisableModeGet(clientHandle, intf, NULL);
  printf("NULL mode arg. (result = %d)\n", result);
  result = openapiPmlIntfViolationDDisableModeGet(clientHandle, invalidIntf, &mode);
  printf("Likely invalid intf arg. (result = %d)\n", result);

  /* openapiPmlIntfStickyModeGet() */
  printf("\nTesting openapiPmlIntfStickyModeGet(): \n");
  result = openapiPmlIntfStickyModeGet(NULL, intf, &mode);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPmlIntfStickyModeGet(clientHandle, intf, NULL);
  printf("NULL mode arg. (result = %d)\n", result);
  result = openapiPmlIntfStickyModeGet(clientHandle, invalidIntf, &mode);
  printf("Likely invalid intf arg. (result = %d)\n", result);

  return;
}

int main(int argc, char *argv[])
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  uint32_t intf;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  uint32_t  show_help = 1, val = 0, vlanId = 0;
  open_buffdesc macAddr;
  char mac[MAC_STRING_SIZE] = {0};

  macAddr.pstart = mac;
  macAddr.size   = MAC_STRING_SIZE;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);
  if (argc >= 3)
  {
     intf = atoi(argv[2]);
     vlanId = atoi(argv[2]);
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister(argv[0], &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);

  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 2)
      {
        getGlobalAdminMode(&clientHandle);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 3)
      {
        getPmlInterfaceIsValid(&clientHandle, intf);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 3)
      {
        getMode(&clientHandle, intf);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 3)
      {
        getDynamicLimit(&clientHandle, intf);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 3)
      {
        getStaticLimit(&clientHandle, intf);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 3)
      {
        getViolationTrapMode(&clientHandle, intf);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 3)
      {
        getViolationDDisableMode(&clientHandle, intf);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 3)
      {
        getStickyMode(&clientHandle, intf);
        show_help = 0;
      }
      break;

    case 9:
      if (argc == 3)
      {
        displayStaticEntries(&clientHandle, intf);
        show_help = 0;
      }
      break;

    case 10:
      if (argc == 3)
      {
        displayDynamicEntries(&clientHandle, intf);
        show_help = 0;
      }
      break;

    case 11:
      if (argc == 3)
      {
        pmlGlobalAdminModeSet(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    
    case 12:
      if (argc == 4)
      {
        pmlIntfModeSet(&clientHandle, intf, atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 3)
      {
        pmlIntfDynamicMaxLimitGet(&clientHandle, intf, &val);
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 4)
      {
        pmlIntfDynamicLimitSet(&clientHandle, intf, atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 3)
      {
        pmlIntfStaticMaxLimitGet(&clientHandle, intf, &val);
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 3)
      {
        pmlIntfStaticCountGet(&clientHandle, intf, &val);
        show_help = 0;
      }
      break;
    case 17:
      if (argc == 4)
      {
        pmlIntfStaticLimitSet(&clientHandle, intf, atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 18:
      if (argc == 3)
      {
        pmlIntfTrapFrequencyGet(&clientHandle, intf, &val);
        show_help = 0;
      }
      break;
    case 19:
      if (argc == 4)
      {
        pmlIntfTrapFrequencySet(&clientHandle, intf, atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 20:
      if (argc == 6)
      {
        strncpy (mac, argv[3], MAC_STRING_SIZE-1);  
        pmlIntfStaticEntryAdd(&clientHandle, intf, &macAddr, atoi(argv[4]),
                              atoi(argv[5]));
        show_help = 0;
      }
      break;
    case 21:
      if (argc == 5)
      {
        strncpy (mac, argv[3], MAC_STRING_SIZE-1);  
        pmlIntfStaticEntryDelete(&clientHandle, intf, &macAddr, atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 22:
      if (argc == 3)
      {
        pmlIntfDynamicToStaticMove(&clientHandle, intf);
        show_help = 0;
      }
      break;
    case 23:
      if (argc == 4)
      {
        pmlIntfStickyModeSet(&clientHandle, intf, atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 24:
      if (argc == 3)
      {
        pmlIntfLastViolationAddrGet(&clientHandle, intf, &macAddr, &val);
        show_help = 0;
      }
      break;
    case 25:
      if (argc == 3)
      {
        pmlIntfDynamicLearnedAddrCountGet(&clientHandle, intf, &val);
        show_help = 0;
      }
      break;
    case 26:
      if (argc == 3)
      {
        pmlIntfStaticLearnedAddrCountGet(&clientHandle, intf, &val);
        show_help = 0;
      }
      break;
    case 27:
      if (argc == 3)
      {
        pmlStaticAddrCountByVlanGet(&clientHandle, vlanId, &val);
        show_help = 0;
      }
      break;
    case 28:
      if (argc == 2)
      {
        pmlStaticAddrGlobalCountGet(&clientHandle, &val);
        show_help = 0;
      }
      break;
    case 29:
      if (argc == 4)
      {
        pmlIntfViolationDDisableModeSet(&clientHandle, intf, atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 30:
      if (argc == 2)
      {
        pmlVlanAdminModeGet(&clientHandle, &val);
        show_help = 0;
      }
      break;
    case 31:
      if (argc == 3)
      {
        pmlVlanAdminModeSet(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 32:
      if (argc == 3)
      {
        pmlVlanDynamicLimitGet(&clientHandle, vlanId, &val);
        show_help = 0;
      }
      break;
    case 33:
      if (argc == 4)
      {
        pmlVlanDynamicLimitSet(&clientHandle, vlanId, atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 34:
      if (argc == 3)
      {
        pmlVlanOperationalLimitGet(&clientHandle, vlanId, &val);
        show_help = 0;
      }
      break;
    case 35:
      if (argc == 3)
      {
        pmlVlanViolationDDisableModeGet(&clientHandle, vlanId, &val);
        show_help = 0;
      }
      break;
    case 36:
      if (argc == 4)
      {
        pmlVlanViolationDDisableModeSet(&clientHandle, vlanId, atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 37:
      if (argc == 3)
      {
        pmlVlanViolationTrapModeGet(&clientHandle, vlanId, &val);
        show_help = 0;
      }
      break;
    case 38:
      if (argc == 4)
      {
        pmlVlanViolationTrapModeSet(&clientHandle, vlanId, atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 39:
      if (argc == 3)
      {
        pmlVlanModeGet(&clientHandle, vlanId, &val);
        show_help = 0;
      }
      break;
    case 40:
      if (argc == 4)
      {
        pmlVlanModeSet(&clientHandle, vlanId, atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 41:
      if (argc == 4)
      {
        pmlIntfVlanDynamicLimitGet(&clientHandle, intf, atoi(argv[3]), &val);
        show_help = 0;
      }
      break;
    case 42:
      if (argc == 5)
      {
        pmlIntfVlanDynamicLimitSet(&clientHandle, intf, atoi(argv[3]), 
                                   atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 43:
      if (argc == 4)
      {
        pmlIntfVlanDynamicLearnedAddrCountGet(&clientHandle, intf, 
                                              atoi(argv[3]), &val);
        show_help = 0;
      }
      break;
    case 44:
      if (argc == 3)
      {
        pmlIntfAgeTimeGet(&clientHandle, intf, &val);
        show_help = 0;
      }
      break;
    case 45:
      if (argc == 4)
      {
        pmlIntfAgeTimeSet(&clientHandle, intf, atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 46:
      if (argc == 3)
      {
        pmlNextValidIntfGet(&clientHandle, intf, &val);
        show_help = 0;
      }
      break;
    case 47:
      if (argc == 3)
      {
        displayVlanDynamicEntries(&clientHandle, intf);
        show_help = 0;
      }
      break;
    case 48:
      if (argc == 6)
      {
        strncpy (mac, argv[3], MAC_STRING_SIZE-1);  
        pmlDot1qciStreamSet(&clientHandle, intf, &macAddr, 
                            atoi(argv[4]), atoi(argv[5]));
        show_help = 0;
      }
      break;
    case 49:
      if (argc == 6)
      {
        strncpy (mac, argv[3], MAC_STRING_SIZE-1);  
        pmlDot1qciStreamDelete(&clientHandle, intf, &macAddr,
                               atoi(argv[4]), atoi(argv[5]));
        show_help = 0;
      }
      break;
    case 50:
      if (argc == 5)
      {
        strncpy (mac, argv[3], MAC_STRING_SIZE-1);  
        pmlDot1qciStreamGet(&clientHandle, intf, &macAddr, atoi(argv[4]), &val);
        show_help = 0;
      }
      break;
    case 51:
      if (argc == 4)
      {
        setViolationTrapMode(&clientHandle, intf, atoi(argv[3]));
        show_help = 0;
      }
      break;
 
    case 52:
      if (argc == 3)
      {
        runNegativeTests(&clientHandle, intf);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping PML API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
