/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  protectedport_example.c
*
* @purpose   OpEN ProtectedPort example.
*
* @component OpEN
*
* @create    05/26/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_protectedport.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the ProtectedPort Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Get the name of the group for a given group ID: %s 1 <groupId>\n", name);
  printf("Test 2: Set the name of the group for a given group ID: %s 2 <groupId> <groupName>\n", name);
  printf("Test 3: Get all the protected ports which are members of a given group ID: %s 3 <groupId> \n", name);
  printf("Test 4: Get the group ID for a given interface: %s 4 <ifNum>\n", name);
  printf("Test 5: Get the next group and interface for a given group and interface: %s 5 <ifNum> <groupId>\n", name);
  printf("Test 6: Add an interface to a protected port group: %s 6 <ifNum> <groupId>\n", name);
  printf("Test 7: Delete an interface from a protected port group: %s 7 <ifNum> <groupId>\n", name);
  printf("Test 8: Get the maximum number of protected port groups: %s 8\n", name);
  printf("Test 9: Get the number of protected ports which are members of a given group ID: %s 9 <groupId>\n", name);

  return;
}

/*********************************************************************
* @purpose  Get the name of the group for a given group ID.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    groupId       @b{(input)}  Group identifier
* @param    groupName     @b{(output)} Name of the group
*
* @returns  none
*
* @end
*********************************************************************/
void protectedPortGroupNameGet(openapiClientHandle_t *client_handle, 
                               uint32_t groupId, open_buffdesc *groupName)
{
  open_error_t result;

  if ((result = openapiProtectedPortGroupNameGet(client_handle, groupId, groupName))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the name of the group for a given "
           "group ID. (result = %d)\n", result);
  }
  else
  {
    printf("Group name is %s.\n", (char *) groupName->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Set the name of the group for a given group ID.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    groupId       @b{(input)}  Group identifier
* @param    groupName     @b{(input)}  Name of the group
*
* @returns  none
*
* @end
*********************************************************************/
void protectedPortGroupNameSet(openapiClientHandle_t *client_handle, 
                               uint32_t groupId, open_buffdesc *groupName)
{
  open_error_t result;

  if ((result = openapiProtectedPortGroupNameSet(client_handle, groupId,
                                                 groupName)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the name of the group for a given"
           " group ID. (result = %d)\n", result);
  }
  else
  {
    printf("Group name set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get all the protected ports which are members of a given group ID.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    groupId       @b{(input)}  Group identifier
* @param    intfList      @b{(output)} List of interfaces with comma separation
*
* @returns  none
*
* @end
*********************************************************************/
void protectedPortGroupIntfListGet(openapiClientHandle_t *client_handle, 
                                   uint32_t groupId, open_buffdesc *intfList)
{
  open_error_t result;

  if ((result = openapiProtectedPortGroupIntfListGet(client_handle, groupId,
                                                     intfList)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get all the protected ports which are"
           " members of a given group ID. (result = %d)\n", result);
  }
  else
  {
    printf("List of interfaces in the given group are %s.\n", (char *) intfList->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Get the group ID for a given interface.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    groupId       @b{(output)} Group identifier
*
* @returns  none
*
* @end
*********************************************************************/
void protectedPortGroupGet(openapiClientHandle_t *client_handle, uint32_t ifNum,
                           uint32_t *groupId)
{
  open_error_t result;

  if ((result = openapiProtectedPortGroupGet(client_handle, ifNum, groupId))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the group ID for a given interface."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Group ID is %u.\n", *groupId);
  }
  return;
}

/*********************************************************************
* @purpose  Get the next group and interface for a given group and interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    groupId        @b{(input)}  Group identifier
* @param    NextIfNum      @b{(output)} Next interface
* @param    NextGroupId    @b{(output)} Next group identifier
*
* @returns  none
*
* @end
*********************************************************************/
void protectedPortNextGroupGet(openapiClientHandle_t *client_handle, 
                               uint32_t ifNum, uint32_t groupId, 
                               uint32_t *NextIfNum, uint32_t *NextGroupId)
{
  open_error_t result;

  if ((result = openapiProtectedPortNextGroupGet(client_handle, ifNum, 
                                                 groupId, NextIfNum, NextGroupId))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next group and interface for"
           " a given group and interface. (result = %d)\n", result);
  }
  else
  {
    printf("Next group ID is %u.\n", *NextGroupId);
    printf("Next interface is %u.\n", *NextIfNum);
  }
  return;
}

/*********************************************************************
* @purpose  Add an interface to a protected port group.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    groupId       @b{(input)}  Group identifier
*
* @returns  none
*
* @end
*********************************************************************/
void protectedPortGroupIntfAdd(openapiClientHandle_t *client_handle, 
                               uint32_t ifNum, uint32_t groupId)
{
  open_error_t result;

  if ((result = openapiProtectedPortGroupIntfAdd(client_handle, ifNum, groupId))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to add an interface to a protected port"
           " group. (result = %d)\n", result);
  }
  else
  {
    printf("Group added to interface successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Delete an interface from a protected port group.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    groupId       @b{(input)}  Group identifier
*
* @returns  none
*
* @end
*********************************************************************/
void protectedPortGroupIntfDelete(openapiClientHandle_t *client_handle, 
                                  uint32_t ifNum, uint32_t groupId)
{
  open_error_t result;

  if ((result = openapiProtectedPortGroupIntfDelete(client_handle, ifNum, groupId))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete an interface from a protected port"
           " group. (result = %d)\n", result);
  }
  else
  {
    printf("Group removed from interface successfully.\n");
  }
  return;
}

/*********************************************************************
 * @purpose   Get the maximum number of protected port groups.
*
* @param    client_handle @b{(input)}  client handle from registration API
 * @param    maxGroupNum  @b{(output)} Group number
*
* @returns  none
*
* @end
*********************************************************************/
void protectedPortMaxGroups (openapiClientHandle_t *client_handle, 
                             uint32_t *maxGroupNum)
{
  open_error_t result;

  if ((result = openapiProtectedPortMaxGroups (client_handle, maxGroupNum))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the maximum number of protected "
           "port groups. (result = %d)\n", result);
  }
  else
  {
    printf("Maximum number of protected port groups is %u.\n", *maxGroupNum);
  }
  return;
}

/*********************************************************************
* @purpose  Get the number of protected ports which are members of a given group ID.
*
* @param    client_handle @b{(input)}  client handle from registration API
 * @param   groupId       @b{(input)}  Group identifier
 * @param   intfCount     @b{(output)} Number of interfaces
*
* @returns  none
*
* @end
*********************************************************************/
void protectedPortGroupIntfCountGet (openapiClientHandle_t *client_handle, 
                                     uint32_t groupId,
                                     uint32_t *intfCount)
{
  open_error_t result;

  if ((result = openapiProtectedPortGroupIntfCountGet (client_handle, groupId, 
                                                       intfCount))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the number of protected ports which"
           " are members of a given group ID. (result = %d)\n", result);
  }
  else
  {
    printf("Interface count is %u.\n", *intfCount);
  }
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for ProtectedPort
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
#define STR_SIZE 256
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum, val, val1;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  char bufStr[STR_SIZE+1] = {0};
  open_buffdesc buf;
  buf.pstart = bufStr;
  buf.size   = STR_SIZE;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("ProtectedPort example", &client_handle))
      != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting ProtectedPort API example "
              "application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        protectedPortGroupNameGet(&client_handle, atoi(argv[2]), &buf);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 4)
      {
        strncpy (bufStr, argv[3], STR_SIZE);  
        protectedPortGroupNameSet(&client_handle, atoi(argv[2]), &buf);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 3)
      {
        protectedPortGroupIntfListGet(&client_handle, atoi(argv[2]), &buf);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 3)
      {
        protectedPortGroupGet(&client_handle, atoi(argv[2]), &val);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 4)
      {
        protectedPortNextGroupGet(&client_handle, atoi(argv[2]), atoi(argv[3]),
                                  &val, &val1);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 4)
      {
        protectedPortGroupIntfAdd(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 4)
      {
        protectedPortGroupIntfDelete(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 2)
      {
        protectedPortMaxGroups(&client_handle, &val);  
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 3)
      {
        protectedPortGroupIntfCountGet (&client_handle, atoi(argv[2]), &val);  
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping ProtectedPort API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}


