=begin
*
* Copyright 2016-2018 Broadcom.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0*
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
=end

require "OpEN"
require "OpENUtil"

Puppet::Type.type(:netdev_vlan).provide(:icos) do
  confine :exists => "/etc/brl-release"
  @doc = "NOS netdev VLAN interface"

  mk_resource_methods

  def exists?
    @property_hash[:ensure] == :present
  end

  def create
    begin
      printf "netdev_stdlib_icos-vlan create\n"

      open = OpENUtil.new()
      open.connect("netdev_stdlib_icos-vlan")

      # Create a vlan
      ret = OpEN::openapiVlanCreate(open.client, resource[:vlan_id].to_i)
      if ret == OpEN::OPEN_E_NONE
        # Set the vlan name
        buffdesc        = OpEN::Open_buffdesc.new
        buffdesc.size   = resource[:name].length + 1
        buffdesc.pstart = open.getCharBuffer(buffdesc.size, resource[:name])
        ret = OpEN::openapiVlanNameSet(open.client, resource[:vlan_id].to_i, buffdesc)
        if ret != OpEN::OPEN_E_NONE
          printf "Failed to set name %s to VLAN %d, return value = %d\n", resource[:name], resource[:vlan_id].to_i, ret
          warning('Failed to set VLAN name.')
        end
      else
        printf "Failed to create VLAN %d, return value = %d\n", resource[:vlan_id].to_i, ret
        fail('Failed to create VLAN.')
      end

      open.terminate()

      @property_hash[:ensure] = :present
    rescue Puppet::ExecutionFailure => e
      printf "Failed to create VLAN\n"
    end
  end

  def destroy
    begin
      printf "netdev_stdlib_icos-vlan destroy\n"

      open = OpENUtil.new()
      open.connect("netdev_stdlib_icos-vlan")

      # Delete a vlan
      ret = OpEN::openapiVlanDelete(open.client, @property_hash[:vlan_id].to_i)
      if ret != OpEN::OPEN_E_NONE
        printf "Failed to delete VLAN %d, return value = %d\n", resource[:vlan_id].to_i, ret
        fail('Failed to delete VLAN.')
      end

      open.terminate()

      @property_hash.clear
    rescue Puppet::ExecutionFailure => e
      printf "Failed to destroy VLAN\n"
    end
  end

  def self.instances
    begin
      printf "netdev_stdlib_icos-vlan self.instances\n"

      open = OpENUtil.new()
      open.connect("netdev_stdlib_icos-vlan")

      vlans      = []
      vlan_num   = 0
      vlan_num_p = OpEN.new_uint32_tp()

      # Get the first vlan
      ret, vlan_num = OpEN::openapiVlanNextGet(open.client, vlan_num, vlan_num_p)
      while ret == OpEN::OPEN_E_NONE
        vlan_num = OpEN.uint32_tp_value(vlan_num_p)

        # Get the vlan name
        buffdesc   = OpEN::Open_buffdesc.new
        buffdesc.size   = 100 
        name = open.getCharBuffer(buffdesc.size, nil)
        buffdesc.pstart = name 
        OpEN::openapiVlanNameGet(open.client, vlan_num, buffdesc)

        vlans << new(:name => name.cast(), :vlan_id => vlan_num.to_s, :ensure => :present)

        ret, vlan_num = OpEN::openapiVlanNextGet(open.client, vlan_num, vlan_num_p)
      end # Loop through all the vlans

      open.terminate()

      OpEN.delete_uint32_tp(vlan_num_p)

      vlans
    end
  end

  def self.prefetch(resources)
    begin
      printf "netdev_stdlib_icos-vlan self.prefetch\n"

      vlans = instances
      resources.each do |name, params|
        if provider = vlans.find { |vlan| vlan.vlan_id == params[:vlan_id] }
          resources[name].provider = provider
        end
      end
    end
  end

end
