/*********************************************************************
*
* Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  diffserv_class_conf_example.c
*
* @purpose   DiffServ Class Configuration OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    17/05/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_diffserv_class_conf.h"

/* 
 * OpEN API set functions are processed asynchronously.  There may be some
 * delay between when the set function call returns and when the system
 * state is updated to reflect the change.  These parameters control how
 * long the test code retries the get functions to retrieve a change.
 */ 

/*******************************************************************
*
* @brief  This function prints the DiffServ Class conf Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printDiffServClassConfAppMenu()
{
  printf("Usage: diffserv_class_conf_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Create the DiffServ class : diffserv_class_conf_example 1 <class-name> <class-type> <class-L3Protocol> \n");
  printf("Test 2: Delete the DiffServ class : diffserv_class_conf_example 2 <class-index> \n");
  printf("Test 3: Get the DiffServ class index from class name : diffserv_class_conf_example 3 <class-name>\n");
  printf("Test 4: Get the DiffServ class name from class index : diffserv_class_conf_example 4 <class-index> \n");
  printf("Test 5: Get the DiffServ class info : diffserv_class_conf_example 5 \n");
  printf("Test 6: Create the DiffServ Class Rule : diffserv_class_conf_example 6 <class-index> <rule-type> <ruleVal1> <ruleVal2> \n");
  printf("Test 7: Remove the referenced class from the specified class : diffserv_class_conf_example 7 <class-index> <class-name> \n");
  printf("Test 8: Get the DiffServ class rule info : diffserv_class_conf_example 8 <class-name>\n"); 
  printf("Test 9: Run API sanity checks: diffserv_class_conf_example 9 \n");

  return;
}

/*********************************************************************
* @purpose  Get the class info.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void diffServClassInfoGet(openapiClientHandle_t *clientHandle)
{
  uint32_t classIndex = 0;
  char buf[OPEN_RULE_VAL_MAX_STRING_LENGTH] = {0}; 
  char classNameStr[OPEN_DIFFSERV_CLASS_NAME_MAX + 1] = {0};
  open_buffdesc className = {.pstart = classNameStr, .size = sizeof(classNameStr)};
  OPEN_DIFFSERV_CLASS_TYPE_t classType = 0;
  OPEN_DIFFSERV_CLASS_L3PROTOCOL_t classL3Protocol = 0;

  if (!clientHandle)
  {
    return;
  }
 
  printf("\n");
  printf("Class details: \n");
  printf("---------------------------------\n");

  while (OPEN_E_NONE == openapiDiffServClassGetNext(clientHandle, classIndex, &classIndex))
  {
    memset(buf, 0, sizeof(buf));
    className.size = sizeof(classNameStr);
    memset(classNameStr, 0, sizeof(classNameStr));
 
    printf("\n");

    /* Class Name */
    if (OPEN_E_NONE != openapiDiffServClassNameGet(clientHandle, classIndex, &className))
    {
      continue;
    }
    else
    {
      printf("Class Index: %d\n", classIndex);
      printf("Class Name: %s\n", classNameStr);
    }
  
    /* Class Type */
    if (OPEN_E_NONE != openapiDiffServClassTypeGet(clientHandle, classIndex, &classType))
    {
      continue;
    }

    memset(buf, 0, sizeof(buf));
    switch (classType)
    {
      case OPEN_DIFFSERV_CLASS_TYPE_ALL:
        sprintf(buf, "%s", "All");
        break;
      case OPEN_DIFFSERV_CLASS_TYPE_ANY:
        sprintf(buf,"%s", "Any");
        break;
      default:
        break;
    }
    printf("Class type: %s\n", buf);
	  
    /* Class L3 Protocol */
    if (OPEN_E_NONE != openapiDiffServClassL3ProtoGet(clientHandle, classIndex, &classL3Protocol))
    {
      continue;
    }

    memset(buf, 0, sizeof(buf));
    switch (classL3Protocol)
    {
      case OPEN_DIFFSERV_CLASS_L3PROTOCOL_IPV4:
        sprintf(buf, "%s", "IPV4");
        break;
      case OPEN_DIFFSERV_CLASS_L3PROTOCOL_IPV6:
        sprintf(buf, "%s", "IPV6");
        break;
      default:
        break;
    }
    printf("Class L3 Protocol: %s\n", buf);

  }
}

static open_error_t classRuleTypeToStrConvert(OPEN_DIFFSERV_CLASS_RULE_TYPE_t openRuleType,
                                              char *ruleStr)
{
  open_error_t result = OPEN_E_NONE;

  if (!ruleStr)
  {
    return OPEN_E_PARAM;
  }

  switch (openRuleType)
  {
    case OPEN_DIFFSERV_CLASS_RULE_TYPE_COS:
      strcpy(ruleStr, "Class of Service");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_COS2:
      strcpy(ruleStr, "Secondary Class of Service");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_DSTIP:
      strcpy(ruleStr, "Destination IP Address");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_SRCIP:
      strcpy(ruleStr, "Source IP Address");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_SRCL4PORT:
      strcpy(ruleStr, "Source Layer 4 Port");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_DSTL4PORT:
	  strcpy(ruleStr, "Destination Layer 4 Port");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_SRCMAC:
      strcpy(ruleStr, "Source MAC Address");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_DSTMAC:
      strcpy(ruleStr, "Destination MAC Address");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_VLANID:
      strcpy(ruleStr, "VLAN ID");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_VLANID2:
      strcpy(ruleStr, "Secondary VLAN");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_ETYPE:
      strcpy(ruleStr, "Ethertype");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_EVERY:
      strcpy(ruleStr, "Any");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_IPDSCP:
      strcpy(ruleStr, "IP DSCP");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_IPPRECEDENCE:
      strcpy(ruleStr, "IP PRECEDENCE");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_IPTOS:
      strcpy(ruleStr, "IP TOS");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_PROTOCOL:
      strcpy(ruleStr, "PROTOCOL");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_REFCLASS:
      strcpy(ruleStr, "Reference Class");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_DSTIPV6:
      strcpy(ruleStr, "Destination IPV6 Address");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_SRCIPV6:
      strcpy(ruleStr, "Source IPV6 Address");
      break;

    case OPEN_DIFFSERV_CLASS_RULE_TYPE_FLOWLBLV6:
      strcpy(ruleStr, "Flow Label V6");
      break;

    default:
      result = OPEN_E_PARAM;
      break;
  }
  return result;
} 
                                                          
/*********************************************************************
* @purpose  Get the class rule info for the given className.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void diffServClassRuleInfoGet(openapiClientHandle_t *clientHandle, open_buffdesc *className)
{
  uint32_t classIndex = 0, ruleCount = 0, ruleIndex = 0;
  uint32_t localClassIndex = 0;
  OPEN_DIFFSERV_CLASS_TYPE_t classType = 0;
  char buf[OPEN_RULE_VAL_MAX_STRING_LENGTH] = {0};
  char val1[OPEN_RULE_VAL_MAX_STRING_LENGTH] = {0};
  char val2[OPEN_RULE_VAL_MAX_STRING_LENGTH] = {0};
  open_buffdesc valBuf1;
  open_buffdesc valBuf2; 
  OPEN_DIFFSERV_CLASS_RULE_TYPE_t ruleType;
  char tempBuf[OPEN_RULE_VAL_MAX_STRING_LENGTH * 2] = {0};
  char ruleStr[OPEN_RULE_VAL_MAX_STRING_LENGTH] = {0};
  char str[OPEN_DIFFSERV_CLASS_NAME_MAX+1] = {0};
 
  if (!clientHandle)
  {
    return;
  }

  if (OPEN_E_NONE != openapiDiffServClassNameToIndex(clientHandle, className, &classIndex))
  {
    return;
  }    

  printf("\n");
  strncpy(str, className->pstart, className->size);
  printf("Class Name : %s\n", str);
  
  /* Class Type */
  if (OPEN_E_NONE != openapiDiffServClassTypeGet(clientHandle, classIndex, &classType))
  {
     return;
  }

  memset(buf, 0, sizeof(buf));
  switch (classType)
  {
    case OPEN_DIFFSERV_CLASS_TYPE_ALL:
      sprintf(buf, "%s", "All");
      break;
    case OPEN_DIFFSERV_CLASS_TYPE_ANY:
      sprintf(buf,"%s", "Any");
      break;
    default:
      break;
  } 
  printf("Class type : %s\n", buf);

  /* Display match rule count */
  openapiDiffServClassRuleCountGet(clientHandle, classIndex, &ruleCount);
  printf("Match Rule Count : %d\n", ruleCount);
    
  printf("\n");
  printf("RuleIndex   Match Criteria                     Values\n");
  printf("--------- ------------------------------ -------------------------------------------");

  localClassIndex = classIndex;
  ruleIndex = 0;

  while ((openapiDiffServClassRuleGetNext(clientHandle, localClassIndex, ruleIndex,
                                         &localClassIndex, &ruleIndex) == OPEN_E_NONE) &&
         (classIndex == localClassIndex))
  {
    memset(val1, 0, sizeof(val1));
    valBuf1.pstart = val1;
    valBuf1.size = sizeof(val1);

    memset(val2, 0, sizeof(val2));
    valBuf2.pstart = val2;
    valBuf2.size = sizeof(val2);

    if (OPEN_E_NONE != openapiDiffServClassRuleMatchEntryTypeGet(clientHandle, classIndex, ruleIndex, &ruleType))
    {
      continue;
    }

    if (OPEN_E_NONE != openapiDiffServClassRuleMatchEntryValueGet(clientHandle, classIndex, ruleIndex, ruleType, &valBuf1, &valBuf2))
    {
      continue;
    }

    memset(ruleStr, 0, sizeof(ruleStr));
    classRuleTypeToStrConvert(ruleType, ruleStr);
    sprintf(tempBuf, "(%s) (%s)", val1, val2);
    printf("\n%-9d %-30s %-43s", ruleIndex, ruleStr, tempBuf);
	
  }
  return;	
}

/*********************************************************************
* @purpose  Create the DiffServ class 
*
* @param    clientHandle   @b{(input)} Client handle from 
*                                       registration API
* @param    className       @b{(input)} class name 
* @param    classType       @b{(input)} class type
* @param    classL3Protocol @b{(input)} class L3 protocol
*
* @notes   None. 
* 
* @end
*********************************************************************/
void diffServClassCreate(openapiClientHandle_t *clientHandle,
                         open_buffdesc *className,
                         OPEN_DIFFSERV_CLASS_TYPE_t classType,
                         OPEN_DIFFSERV_CLASS_L3PROTOCOL_t classL3Protocol)
{
  open_error_t result;

  if ((result = openapiDiffServClassCreate(clientHandle, className, classType, classL3Protocol)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to create diffServ class. (result = %d)\n", result);
  }
  else
  {
    printf("DiffServ Class is created successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Delete the DiffServ class 
*
* @param    clientHandle   @b{(input)} Client handle from 
*                                       registration API
* @param    classIndex      @b{(input)} class index 
*
* @returns   None. 
* 
* @end
*********************************************************************/
void diffServClassDelete(openapiClientHandle_t *clientHandle,
                          uint32_t classIndex)
{
  open_error_t result;

  if ((result = openapiDiffServClassDelete(clientHandle, classIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete diffServ class. (result = %d)\n", result);
  }
  else
  {
    printf("DiffServ Class is deleted successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Translate a class name into its associated Class Table index 
*
* @param    clientHandle   @b{(input)} Client handle from 
*                                       registration API
* @param    className       @b{(input)} class name
* @param    classIndex      @b{(output)} class index 
*
* @returns   None. 
* 
*
* @end
*********************************************************************/
void diffServClassNameToIndex(openapiClientHandle_t *clientHandle,
                              open_buffdesc *className,
                              uint32_t *classIndex)
{
  open_error_t result;

  if ((result = openapiDiffServClassNameToIndex(clientHandle, className, classIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get diffServ class index. (result = %d)\n", result);
  }
  else
  {
    printf("DiffServ Class index : %d\n", *classIndex);
  }

  return;
}

/*********************************************************************
* @purpose Get the class name for the given class index 
*
* @param    clientHandle   @b{(input)} Client handle from 
*                                       registration API
* @param    classIndex      @b{(input)} Class index
* @param    className       @b{(output)} Class name 
*
* @returns   None. 
* 
*
* @end
*********************************************************************/
void diffServClassNameGet(openapiClientHandle_t *clientHandle,
                          uint32_t classIndex, open_buffdesc *className)
{
  open_error_t result;
  char str[OPEN_DIFFSERV_CLASS_NAME_MAX+1] = {0};

  if ((result = openapiDiffServClassNameGet(clientHandle, classIndex, className)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get diffServ class index. (result = %d)\n", result);
  }
  else
  {
    strncpy(str, className->pstart, className->size);
    printf("DiffServ Class name : %s\n", str);
  }
  return;
}

/*********************************************************************
 @purpose Create a new rule in the Class Rule Table for the specified indexes  
*
* @param    clientHandle   @b{(input)} Client handle from 
*                                       registration API
* @param    classIndex      @b{(input)} Class index
* @param    ruleType        @b{(input)} Class rule type
* @param    ruleVal1        @b{(input)} Rule Value 1
* @param    ruleVal2        @b{(input)} Rule value 2 
*
* @returns  none.
*
* @notes ruleVal - If rule type supports two parameters, 
*                  ruleVal1 and ruleVal2 both should be used.
*                  Eg:For OPEN_DIFFSERV_CLASS_RULE_TYPE_DSTIP, 
*                  ruleVal1 should be used for address and 
*                  ruleVal2 should be used for mask.
* 
*
* @end
*********************************************************************/
void diffServClassRuleCreate(openapiClientHandle_t *clientHandle,
                             uint32_t classIndex,
                             OPEN_DIFFSERV_CLASS_RULE_TYPE_t ruleType,
                             open_buffdesc *ruleVal1,
                             open_buffdesc *ruleVal2)
{
  open_error_t result;

  result = openapiDiffServClassRuleCreate(clientHandle, classIndex, ruleType,
                                           ruleVal1, ruleVal2);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to create the class rule. (result = %d)\n", result);
  }
  else
  {
    printf("DiffServ Class rule created successfully : (result = %d)\n", result);
  }
  return;

}

/*********************************************************************
* @purpose  Remove the referenced class from the specified class 
*
* @param    clientHandle   @b{(input)} Client handle from 
*                                       registration API
* @param    classIndex      @b{(input)} class index 
* @param    refClassName    @b{(input)} reference class name
*
* @returns   None. 
* 
* @end
*********************************************************************/
void diffServClassRuleMatchRefClassRemove(openapiClientHandle_t *clientHandle,
                             uint32_t classIndex, open_buffdesc *refClassName)
{
  open_error_t result;

  if ((result = openapiDiffServClassRuleMatchRefClassRemove(clientHandle, classIndex, refClassName)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete diffServ ref class. (result = %d)\n", result);
  }
  else
  {
    printf("DiffServ reference Class is deleted successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Perform DiffServ Class OpEN APIs Sanity
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void diffServClassOpENAPISanity(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  open_buffdesc className;
  OPEN_DIFFSERV_CLASS_TYPE_t classType = 1;
  OPEN_DIFFSERV_CLASS_L3PROTOCOL_t classL3Protocol = 1;
  char str[OPEN_DIFFSERV_CLASS_NAME_MAX+1];
  uint32_t classIndex, ruleCount;
  uint32_t classIndexNext;
  OPEN_DIFFSERV_CLASS_L3PROTOCOL_t classProtoType;
  OPEN_DIFFSERV_CLASS_RULE_TYPE_t ruleType = 1;
  open_buffdesc ruleVal1;
  open_buffdesc ruleVal2;
  uint32_t classRuleIndex = 1;
  uint32_t nextClassIndex;
  uint32_t nextClassRuleIndex;
  open_buffdesc refClassName;
  char str1 [OPEN_RULE_VAL_MAX_STRING_LENGTH];
  char str2 [OPEN_RULE_VAL_MAX_STRING_LENGTH];

  className.pstart = str;
  className.size = OPEN_DIFFSERV_CLASS_NAME_MAX;

  /* openapiDiffServClassCreate */
  result = openapiDiffServClassCreate((void *)NULL, &className, classType, classL3Protocol);
  
  result = openapiDiffServClassCreate(client_handle, (void *)NULL, classType, classL3Protocol);
  
  result = openapiDiffServClassCreate(client_handle, &className, 4, classL3Protocol);
  
  result = openapiDiffServClassCreate(client_handle, &className, 1, classL3Protocol);
  
  result = openapiDiffServClassCreate(client_handle, &className, classType, 4);
  
  result = openapiDiffServClassCreate(client_handle, &className, classType, 1);
  if (OPEN_E_NONE == result)
  printf("openapiDiffServClassCreate() sanity successful. \n"); 

  /* openapiDiffServClassDelete */
  classIndex = 1;
  result = openapiDiffServClassDelete((void *)NULL, classIndex);
  
  result = openapiDiffServClassDelete(client_handle, 1);
  
  result = openapiDiffServClassDelete(client_handle, 0);
  printf("openapiDiffServClassDelete() sanity successful. \n");  


  /* openapiDiffServClassNameToIndex */
  className.pstart = str;
  className.size = OPEN_DIFFSERV_CLASS_NAME_MAX;
  result = openapiDiffServClassNameToIndex((void *)NULL, &className, &classIndex);
  
  result = openapiDiffServClassNameToIndex(client_handle, (void *)NULL, &classIndex);
  
  result = openapiDiffServClassNameToIndex(client_handle, &className, (void *)NULL);
  printf("openapiDiffServClassNameToIndex() sanity successful. \n"); 

  /* openapiDiffServClassGetNext */
  classIndex = 1;
  result = openapiDiffServClassGetNext((void *)NULL, classIndex, &classIndexNext);
  
  result = openapiDiffServClassGetNext(client_handle, 1, &classIndexNext);
  
  result = openapiDiffServClassGetNext(client_handle, 0, &classIndexNext);
  
  result = openapiDiffServClassGetNext(client_handle, classIndex, (void *)NULL);
  printf("openapiDiffServClassGetNext() sanity successful. \n"); 

  /* openapiDiffServClassNameGet */
  className.pstart = str;
  className.size = OPEN_DIFFSERV_CLASS_NAME_MAX;
  classIndex = 1;  
  result = openapiDiffServClassNameGet((void *)NULL, classIndex, &className);
  
  result = openapiDiffServClassNameGet(client_handle, 1, &className);
  
  result = openapiDiffServClassNameGet(client_handle, 0, &className);
  
  result = openapiDiffServClassNameGet(client_handle, classIndex, (void *)NULL);
  printf("openapiDiffServClassNameGet() sanity successful. \n"); 

  /* openapiDiffServClassTypeGet */
  result = openapiDiffServClassTypeGet((void *)NULL, classIndex, &classType);
  
  result = openapiDiffServClassTypeGet(client_handle, 1, &classType);
  
  result = openapiDiffServClassTypeGet(client_handle, 0, &classType);
  
  result = openapiDiffServClassTypeGet(client_handle, classIndex, (void *)NULL);
  printf("openapiDiffServClassTypeGet() sanity successful. \n"); 

  /* openapiDiffServClassL3ProtoGet */
  result = openapiDiffServClassL3ProtoGet((void *)NULL, classIndex, &classProtoType);
  
  result = openapiDiffServClassL3ProtoGet(client_handle, 3, &classProtoType);
  
  result = openapiDiffServClassL3ProtoGet(client_handle, 0, &classProtoType);
  
  result = openapiDiffServClassL3ProtoGet(client_handle, classIndex, (void *)NULL);
  printf("openapiDiffServClassL3ProtoGet() sanity successful. \n"); 

  /* openapiDiffServClassRuleCreate */
  ruleVal1.pstart = str1;
  ruleVal1.size = OPEN_RULE_VAL_MAX_STRING_LENGTH -1 ;
  ruleVal2.pstart = str2;
  ruleVal2.size = OPEN_RULE_VAL_MAX_STRING_LENGTH -1;

  result = openapiDiffServClassRuleCreate((void *)NULL, classIndex, ruleType, &ruleVal1, &ruleVal2);
  
  result = openapiDiffServClassRuleCreate(client_handle, 1, ruleType, &ruleVal1, &ruleVal2);
  
  result = openapiDiffServClassRuleCreate(client_handle, 0, ruleType, &ruleVal1, &ruleVal2);
  
  result = openapiDiffServClassRuleCreate(client_handle, classIndex, 2, &ruleVal1, &ruleVal2);
  
  result = openapiDiffServClassRuleCreate(client_handle, classIndex, 1, &ruleVal1, &ruleVal2);
 
  result = openapiDiffServClassRuleCreate(client_handle, classIndex, ruleType, (void *)NULL, &ruleVal2);
  
  result = openapiDiffServClassRuleCreate(client_handle, classIndex, ruleType, &ruleVal1, (void *)NULL);
  printf("openapiDiffServClassRuleCreate() sanity successful. \n"); 

  /* openapiDiffServClassRuleMatchRefClassRemove */ 
  refClassName.pstart = str;
  refClassName.size = OPEN_DIFFSERV_CLASS_NAME_MAX;

  result = openapiDiffServClassRuleMatchRefClassRemove((void *)NULL, classIndex, &refClassName);

  result = openapiDiffServClassRuleMatchRefClassRemove(client_handle, 1, &refClassName);

  result = openapiDiffServClassRuleMatchRefClassRemove(client_handle, 0, &refClassName);

  result = openapiDiffServClassRuleMatchRefClassRemove(client_handle, classIndex, (void *)NULL);
  printf("openapiDiffServClassRuleMatchRefClassRemove() sanity successful. \n");

  /* openapiDiffServClassRuleGetNext */ 
  result = openapiDiffServClassRuleGetNext((void *)NULL, classIndex, classRuleIndex, &nextClassIndex, &nextClassRuleIndex);
  
  result = openapiDiffServClassRuleGetNext(client_handle, 1, classRuleIndex, &nextClassIndex, &nextClassRuleIndex);
  
  result = openapiDiffServClassRuleGetNext(client_handle, 0, classRuleIndex, &nextClassIndex, &nextClassRuleIndex);
  
  result = openapiDiffServClassRuleGetNext(client_handle, classIndex, 2, &nextClassIndex, &nextClassRuleIndex);
  
  result = openapiDiffServClassRuleGetNext(client_handle, classIndex, 0, &nextClassIndex, &nextClassRuleIndex);
  
  result = openapiDiffServClassRuleGetNext(client_handle, classIndex, classRuleIndex, (void *)NULL, &nextClassRuleIndex);
  
  result = openapiDiffServClassRuleGetNext(client_handle, classIndex, classRuleIndex, &nextClassIndex, (void *)NULL);
  printf("openapiDiffServClassRuleGetNext() sanity successful. \n");  

  /* openapiDiffServClassRuleCountGet */
  result = openapiDiffServClassRuleCountGet((void *)NULL, classIndex, &ruleCount);
  
  result = openapiDiffServClassRuleCountGet(client_handle, 1, &ruleCount);
  
  result = openapiDiffServClassRuleCountGet(client_handle, 0, &ruleCount);
  
  result = openapiDiffServClassRuleCountGet(client_handle, classIndex, (void *)NULL);
  printf("openapiDiffServClassRuleCountGet() sanity successful. \n"); 

  /* openapiDiffServClassRuleMatchEntryTypeGet */
  result = openapiDiffServClassRuleMatchEntryTypeGet((void *)NULL, classIndex, classRuleIndex, &ruleType);
  
  result = openapiDiffServClassRuleMatchEntryTypeGet(client_handle, 20, classRuleIndex, &ruleType);
  
  result = openapiDiffServClassRuleMatchEntryTypeGet(client_handle, 0, classRuleIndex, &ruleType);
  
  result = openapiDiffServClassRuleMatchEntryTypeGet(client_handle, classIndex, 20, &ruleType);
  
  result = openapiDiffServClassRuleMatchEntryTypeGet(client_handle, classIndex, 0, &ruleType);
  
  result = openapiDiffServClassRuleMatchEntryTypeGet(client_handle, classIndex, classRuleIndex, (void *)NULL);
  printf("openapiDiffServClassRuleMatchEntryTypeGet() sanity successful. \n"); 

  /* openapiDiffServClassRuleMatchEntryValueGet */
  result = openapiDiffServClassRuleMatchEntryValueGet((void *)NULL, classIndex, classRuleIndex, ruleType, &ruleVal1, &ruleVal2);
  
  result = openapiDiffServClassRuleMatchEntryValueGet(client_handle, 1, classRuleIndex, ruleType, &ruleVal1, &ruleVal2);
  
  result = openapiDiffServClassRuleMatchEntryValueGet(client_handle, 0, classRuleIndex, ruleType, &ruleVal1, &ruleVal2);
  
  result = openapiDiffServClassRuleMatchEntryValueGet(client_handle, classIndex, 1, ruleType, &ruleVal1, &ruleVal2);
  
  result = openapiDiffServClassRuleMatchEntryValueGet(client_handle, classIndex, 0, ruleType, &ruleVal1, &ruleVal2);
  
  result = openapiDiffServClassRuleMatchEntryValueGet(client_handle, classIndex, classRuleIndex, 1, &ruleVal1, &ruleVal2);
  
  result = openapiDiffServClassRuleMatchEntryValueGet(client_handle, classIndex, classRuleIndex, 1, &ruleVal1, &ruleVal2);
  
  result = openapiDiffServClassRuleMatchEntryValueGet(client_handle, classIndex, classRuleIndex, ruleType, (void *)NULL, &ruleVal2);
  
  result = openapiDiffServClassRuleMatchEntryValueGet(client_handle, classIndex, classRuleIndex, ruleType, &ruleVal1, (void *)NULL);
  printf("openapiDiffServClassRuleMatchEntryValueGet() sanity successful. \n"); 
 
}
/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum, arg1, arg2, classIndex;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  open_buffdesc classNameBuf;
  char str[OPEN_DIFFSERV_CLASS_NAME_MAX+1] = {0};
  open_buffdesc ruleValBuf1;
  open_buffdesc ruleValBuf2;
  char str1[OPEN_RULE_VAL_MAX_STRING_LENGTH] = {0};
  char str2[OPEN_RULE_VAL_MAX_STRING_LENGTH] = {0};

  if (argc < 2)
  {
    printDiffServClassConfAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("diffserv_class_conf_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting DiffServ Class Conf API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 5)
      {
        printDiffServClassConfAppMenu();
        exit(1);
      }

      memset(str, 0, sizeof(str));
      strncpy(str, argv[2], (sizeof(str) - 1));
      classNameBuf.pstart = str;
      classNameBuf.size = strlen(classNameBuf.pstart);

      arg1 = atoi(argv[3]);
      arg2 = atoi(argv[4]);

      diffServClassCreate(&clientHandle, &classNameBuf, arg1, arg2);
      break;
    case 2:
      if (argc != 3)
      {
        printDiffServClassConfAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      diffServClassDelete(&clientHandle, arg1);
      break;
    case 3:
      if (argc != 3)
      {
        printDiffServClassConfAppMenu();
        exit(1);
      }

      memset(str, 0, sizeof(str));
      strncpy(str, argv[2], (sizeof(str) - 1));
      classNameBuf.pstart = str;
      classNameBuf.size = strlen(classNameBuf.pstart);

      diffServClassNameToIndex(&clientHandle, &classNameBuf, &classIndex);
      break;
    case 4:
      if (argc != 3)
      {
        printDiffServClassConfAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);

      memset(str, 0, sizeof(str));
      classNameBuf.pstart = str;
      classNameBuf.size = sizeof(str);

      diffServClassNameGet(&clientHandle, arg1, &classNameBuf);
      break;
    case 5:
      if (argc != 2)
      {
        printDiffServClassConfAppMenu();
        exit(1);
      }
      diffServClassInfoGet(&clientHandle);
      break;
    case 6:
      if (argc == 4)
      {
        arg1 = atoi(argv[2]);
        if (OPEN_DIFFSERV_CLASS_RULE_TYPE_EVERY != arg1)
        {
          printDiffServClassConfAppMenu();
          exit(1);
        }
      }
      if ((argc != 5) && (argc != 6)) 
      {
        printDiffServClassConfAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
    
      memset(str1, 0, sizeof(str1));  
      strncpy(str1, argv[4], sizeof(str1)-1);
      ruleValBuf1.pstart = str1;
      ruleValBuf1.size = strlen(str1);

      if (argc == 5)
      {
        diffServClassRuleCreate(&clientHandle, arg1, arg2, &ruleValBuf1, NULL);
      }
      else if (argc == 6)
      {
        memset(str2, 0, sizeof(str2));  
        strncpy(str2, argv[5], sizeof(str2)-1);
        ruleValBuf2.pstart = str2;
        ruleValBuf2.size = strlen(str2); 
        diffServClassRuleCreate(&clientHandle, arg1, arg2, &ruleValBuf1, &ruleValBuf2);
      }
      break;
    case 7:
      if (argc != 4)
      {
        printDiffServClassConfAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);

      memset(str, 0, sizeof(str));
      strncpy(str, argv[3], (sizeof(str) - 1));
      classNameBuf.pstart = str;
      classNameBuf.size = strlen(classNameBuf.pstart);

      diffServClassRuleMatchRefClassRemove(&clientHandle, arg1, &classNameBuf);
      break;

    case 8:
      if (argc != 3)
      {
        printDiffServClassConfAppMenu();
        exit(1);
      }
      memset(str, 0, sizeof(str));
      strncpy(str, argv[2], (sizeof(str) - 1));
      classNameBuf.pstart = str;
      classNameBuf.size = strlen(classNameBuf.pstart);

      diffServClassRuleInfoGet(&clientHandle, &classNameBuf);
      break;
    case 9:
      if (argc != 2)
      {
        printDiffServClassConfAppMenu();
        exit(1);
      }
      diffServClassOpENAPISanity(&clientHandle);
      break;
    default:
      printDiffServClassConfAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping DiffServ Class Configuration API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
