/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  diffservconf_example.c
*
* @purpose   OpEN diffServConf example.
*
* @component OpEN
*
* @create    05/18/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_diffserv_conf.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the diffServConf Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Set the value of the DiffServ administrative mode: %s 1 <mode>\n", name);
  printf("Test 2: Gets admin mode of DiffServ administrative: %s 2\n", name);
  printf("Test 3: Get size of the given group Table: %s 3 <diffTableGrpObj>\n", name);
  printf("Test 4: Run API sanity checks: %s 4 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    client_handle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  OPEN_DIFFSERV_GEN_STATUS_GROUP_OBJECT_t diffTableGrpObj = 1;
  OPEN_CONTROL_t mode;
  uint32_t size;

  printf("Testing diffServConf OpEN APIs sanity:\n");
  printf("Testing openapiDiffServGenAdminModeGet():\n");
  result = openapiDiffServGenAdminModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServGenAdminModeGet(client_handle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDiffServGenTableSizeGet():\n");
  result = openapiDiffServGenTableSizeGet(NULL, diffTableGrpObj, &size);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServGenTableSizeGet(client_handle, diffTableGrpObj, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  return;
}


/*********************************************************************
* @purpose  Set the value of the DiffServ administrative mode.
*
* @param    pClientHandle  @b{(input)}  client handle from registration API
* @param    mode           @b{(input)}  DiffServ administrative mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void diffServGenAdminModeSet(openapiClientHandle_t *pClientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiDiffServGenAdminModeSet(pClientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the value of the DiffServ administrative mode. (result = %d)\n", result);
  }
  else
  {
    printf("DiffServ administrative mode is successfully %s\n",
           (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets admin mode of DiffServ administrative.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    pMode          @b{(output)}  DiffServ administrative mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void diffServGenAdminModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiDiffServGenAdminModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets admin mode of DiffServ administrative. (result = %d)\n", result);
  }
  else
  {
    printf("DiffServ administrative mode is in %s mode\n",
        (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Get size of the given group Table.
*
* @param    client_handle     @b{(input)}  client handle from registration API
* @param    diffTableGrpObj   @b{(input)}  Table group object type
* @param    pSize             @b{(output)}  table size value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServGenTableSizeGet(openapiClientHandle_t *client_handle,
                             OPEN_DIFFSERV_GEN_STATUS_GROUP_OBJECT_t diffTableGrpObj,
                             uint32_t *pSize)
{
  open_error_t result;

  if ((result = openapiDiffServGenTableSizeGet(client_handle, diffTableGrpObj, pSize)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get size of the given group Table. (result = %d)\n", result);
  }
  else
  {
    printf("Size of the given group table object:%d is %d.\n",
           diffTableGrpObj, *pSize);
  }
  return;
}



/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for diffServConf
*
* @returns   0: Success
* @returns  -1: Failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  OPEN_DIFFSERV_GEN_STATUS_GROUP_OBJECT_t diffTableGrpObj;
  uint32_t size;
  OPEN_CONTROL_t mode;
  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("diffServConf example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting diffServConf API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        mode = atoi(argv[2]);
        diffServGenAdminModeSet(&client_handle, mode);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 2)
      {
        diffServGenAdminModeGet(&client_handle, &mode);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 3)
      {
        diffTableGrpObj = atoi(argv[2]);
        diffServGenTableSizeGet(&client_handle, diffTableGrpObj, &size);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 2)
      {
        runSanity(&client_handle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping diffServConf API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}
