/*********************************************************************
*
*  Copyright 2022-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  qos_auto_vlan_example.c
*
* @purpose   OpEN QoS Auto VLAN example.
*
* @component OpEN
*
* @create    09/07/2022
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_qos_auto_vlan.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the QoS Auto VLAN Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... for all test cases user auto vlan type 0 for auto-camera, 1 for auto-wifi\n", name);
  printf("Test 1: Gets the Auto VLAN global status: %s 1 <autoVlanType>\n", name);
  printf("Test 2: Enables or disables the Auto VLAN global status: %s 2 <autoVlanType> <mode>\n", name);
  printf("Test 3: Enables or disables the Auto VLAN status on interface: %s 3 <autoVlanType> <ifNum> <mode>\n", name);
  printf("Test 4: Gets the status of Auto VLAN on interface: %s 4 <autoVlanType> <ifNum> \n", name);
  printf("Test 5: Sets the VLAN for Auto VLAN traffic on switch: %s 5 <autoVlanType> <vlanId>\n", name);
  printf("Test 6: Gets the vlan Id configured for Auto VLAN type: %s 6 <autoVlanType>\n", name);
  printf("Test 7: Sets the Priority for Auto VLAN traffic on switch: %s 7 <autoVlanType> <autoVlanPriority>\n", name);
  printf("Test 8: Gets the Priority for Auto VLAN traffic on switch: %s 8 <autoVlanType>\n", name);
  printf("Test 9: Adds Auto VLAN OUI entry: %s 9 <autoVlanType> <autoVlanOui> <autoOuiDesc>\n", name);
  printf("Test 10: Removes Auto Vlan OUI entry: %s 10 <autoVlanType> <autoVlanOui>\n", name);
  printf("Test 11: Adds Auto VLAN OUI entry at particular Index: %s 11 <autoVlanType> <ouiIndex> <autoVlanOui> <autoOuiDesc>\n", name);
  printf("Test 12: Removes Auto VLAN OUI entry at particular Index: %s 12 <ouiIndex>\n", name);
  printf("Test 13: Gets the count of OUIs already configured: %s 13 <autoVlanType>\n", name);
  printf("Test 14: Gets first Auto VLAN OUI index: %s 14 <autoVlanType>\n", name);
  printf("Test 15: Gets Auto VLAN OUI next index: %s 15 <autoVlanType> <currentIndex>\n", name);
  printf("Test 16: Gets Auto VLAN OUI entry details: %s 16 <autoVlanType> <currentIndex>\n", name);
  printf("Test 17: Gets the Auto VLAN OUI description, given OUI entry string: %s 17 <autoVlanType> <autoVlanOui>\n", name);
  printf("Test 18: Determine if the interface is valid for Auto VLAN configuration: %s 18 <ifNum>\n", name);
  printf("Test 19: Get the first interface that is valid for Auto VLAN: %s 19 \n", name);
  printf("Test 20: Get the next interface that is valid for Auto VLAN: %s 20 <prevIfNum> \n", name);
  printf("Test 21: Gets Auto VLAN operational port status: %s 21 <autoVlanType> <ifNum> \n", name);
  printf("Test 22: Gets the count of devices learned for given Auto VLAN type: %s 22 <autoVlanType>\n", name);
  printf("Test 23: Run API sanity checks: %s 23 \n", name);
  printf("Test 24: Gets the Auto VLAN learned entries: %s 24 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  uint32_t status;
  uint32_t count;
  uint32_t prevIfNum=1;
  uint32_t autoVlanPriority;
  uint32_t autoVlanOuiCount;
  uint32_t nextIfNum;
  uint32_t vlanId;
  uint32_t ifNum;
  uint32_t nextOuiIndex;
  OPEN_CONTROL_t mode;
  OPEN_AUTO_VLAN_TYPES_t autoVlanType=0;
  uint32_t ouiIndex=0;
  uint32_t firstIndex=0;
  uint32_t currentIndex=0;
  char buffer[128];
  open_buffdesc zeroLenBuf;
  open_buffdesc badBufdescPointer;
  char          oui[6], ouiDesc[32];
  open_buffdesc autoVlanOui, autoVlanDesc;
  uint32_t     intfNum = 0, index = 0, nextIndex = 0;
  open_buffdesc autoVlanMac;
  char buf[OPEN_MAC_ADDR_LEN];

  badBufdescPointer.pstart = (void *) NULL;
  badBufdescPointer.size = sizeof(buffer);
  badBufdescPointer.size = 0;
  zeroLenBuf.pstart = buffer;
  zeroLenBuf.size = 0;

  autoVlanOui.pstart = oui;
  autoVlanOui.size   = sizeof(oui);

  autoVlanDesc.pstart = ouiDesc;
  autoVlanDesc.size   = sizeof(ouiDesc);

  memset(buf, 0, sizeof(buf));
  autoVlanMac.pstart = buf;
  autoVlanMac.size = sizeof(buf);

  printf("Testing QoS Auto VLAN OpEN APIs sanity:\n");

  printf("Testing openapiQosAutoVlanGlobalModeGet():\n");
  result = openapiQosAutoVlanGlobalModeGet(NULL, autoVlanType, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosAutoVlanGlobalModeGet(clientHandle, OPEN_AUTO_VLAN_FIRST - 1, &mode);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiQosAutoVlanGlobalModeGet(clientHandle, OPEN_AUTO_VLAN_LAST + 1, &mode);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiQosAutoVlanGlobalModeGet(clientHandle, autoVlanType, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiQosAutoVlanIntfModeGet():\n");
  result = openapiQosAutoVlanIntfModeGet(NULL, OPEN_AUTO_VLAN_FIRST, ifNum, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosAutoVlanIntfModeGet(clientHandle, OPEN_AUTO_VLAN_FIRST, ifNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiQosAutoVlanIdGet():\n");
  result = openapiQosAutoVlanIdGet(NULL, autoVlanType, &vlanId);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosAutoVlanIdGet(clientHandle, OPEN_AUTO_VLAN_FIRST - 1, &vlanId);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiQosAutoVlanIdGet(clientHandle, OPEN_AUTO_VLAN_LAST + 1, &vlanId);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiQosAutoVlanIdGet(clientHandle, autoVlanType, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiQosAutoVlanPriorityGet():\n");
  result = openapiQosAutoVlanPriorityGet(NULL, autoVlanType, &autoVlanPriority);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosAutoVlanPriorityGet(clientHandle, OPEN_AUTO_VLAN_FIRST - 1, &autoVlanPriority);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiQosAutoVlanPriorityGet(clientHandle, OPEN_AUTO_VLAN_LAST + 1, &autoVlanPriority);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiQosAutoVlanPriorityGet(clientHandle, autoVlanType, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiQosAutoVlanOuiAdd():\n");
  result = openapiQosAutoVlanOuiAdd(NULL, autoVlanType, &autoVlanOui, &autoVlanDesc);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiAdd(clientHandle, OPEN_AUTO_VLAN_FIRST - 1, &autoVlanOui, &autoVlanDesc);
  printf("less than minimum value argument 4:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiAdd(clientHandle, OPEN_AUTO_VLAN_LAST + 1, &autoVlanOui, &autoVlanDesc);
  printf("greater than maximum value argument 4:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiAdd(clientHandle, autoVlanType, NULL, &autoVlanDesc);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiAdd(clientHandle, autoVlanType, &badBufdescPointer, &autoVlanDesc);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiAdd(clientHandle, autoVlanType, &zeroLenBuf, &autoVlanDesc);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiAdd(clientHandle, autoVlanType, &autoVlanOui, NULL);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiAdd(clientHandle, autoVlanType, &autoVlanOui, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiAdd(clientHandle, autoVlanType, &autoVlanOui, &zeroLenBuf);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);

  printf("Testing openapiQosAutoVlanOuiRemove():\n");
  result = openapiQosAutoVlanOuiRemove(NULL, autoVlanType, &autoVlanOui);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiRemove(clientHandle, OPEN_AUTO_VLAN_FIRST - 1, &autoVlanOui);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiRemove(clientHandle, OPEN_AUTO_VLAN_LAST + 1, &autoVlanOui);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiRemove(clientHandle, autoVlanType, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiRemove(clientHandle, autoVlanType, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiRemove(clientHandle, autoVlanType, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);

  printf("Testing openapiQosAutoVlanIndexOuiAdd():\n");
  result = openapiQosAutoVlanIndexOuiAdd(NULL, autoVlanType, ouiIndex, &autoVlanOui, &autoVlanDesc);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosAutoVlanIndexOuiAdd(clientHandle, OPEN_AUTO_VLAN_FIRST - 1, ouiIndex, &autoVlanOui, &autoVlanDesc);
  printf("less than minimum value argument 5:(result = %d)\n", result);
  result = openapiQosAutoVlanIndexOuiAdd(clientHandle, OPEN_AUTO_VLAN_LAST + 1, ouiIndex, &autoVlanOui, &autoVlanDesc);
  printf("greater than maximum value argument 5:(result = %d)\n", result);
  result = openapiQosAutoVlanIndexOuiAdd(clientHandle, autoVlanType, ouiIndex, NULL, &autoVlanDesc);
  printf("NULL argument 5:(result = %d)\n", result);
  result = openapiQosAutoVlanIndexOuiAdd(clientHandle, autoVlanType, ouiIndex, &badBufdescPointer, &autoVlanDesc);
  printf("NULL buffdesc pstart argument 5:(result = %d)\n", result);
  result = openapiQosAutoVlanIndexOuiAdd(clientHandle, autoVlanType, ouiIndex, &zeroLenBuf, &autoVlanDesc);
  printf("buffdesc 0 length size argument 5:(result = %d)\n", result);
  result = openapiQosAutoVlanIndexOuiAdd(clientHandle, autoVlanType, ouiIndex, &autoVlanOui, NULL);
  printf("NULL argument 5:(result = %d)\n", result);
  result = openapiQosAutoVlanIndexOuiAdd(clientHandle, autoVlanType, ouiIndex, &autoVlanOui, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 5:(result = %d)\n", result);
  result = openapiQosAutoVlanIndexOuiAdd(clientHandle, autoVlanType, ouiIndex, &autoVlanOui, &zeroLenBuf);
  printf("buffdesc 0 length size argument 5:(result = %d)\n", result);

  printf("Testing openapiQosAutoVlanOuiCountGet():\n");
  result = openapiQosAutoVlanOuiCountGet(NULL, autoVlanType, &autoVlanOuiCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiCountGet(clientHandle, OPEN_AUTO_VLAN_FIRST - 1, &autoVlanOuiCount);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiCountGet(clientHandle, OPEN_AUTO_VLAN_LAST + 1, &autoVlanOuiCount);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiCountGet(clientHandle, autoVlanType, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiQosAutoVlanOuiFirstEntryGet():\n");
  result = openapiQosAutoVlanOuiFirstEntryGet(NULL, autoVlanType, &firstIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiFirstEntryGet(clientHandle, OPEN_AUTO_VLAN_FIRST - 1, &firstIndex);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiFirstEntryGet(clientHandle, OPEN_AUTO_VLAN_LAST + 1, &firstIndex);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiFirstEntryGet(clientHandle, autoVlanType, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiQosAutoVlanOuiEntryNextGet():\n");
  result = openapiQosAutoVlanOuiEntryNextGet(NULL, autoVlanType, currentIndex, &nextOuiIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiEntryNextGet(clientHandle, OPEN_AUTO_VLAN_FIRST - 1, currentIndex, &nextOuiIndex);
  printf("less than minimum value argument 4:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiEntryNextGet(clientHandle, OPEN_AUTO_VLAN_LAST + 1, currentIndex, &nextOuiIndex);
  printf("greater than maximum value argument 4:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiEntryNextGet(clientHandle, autoVlanType, currentIndex, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiQosAutoVlanOuiEntryGet():\n");
  result = openapiQosAutoVlanOuiEntryGet(NULL, autoVlanType, currentIndex, &autoVlanOui, &autoVlanDesc);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiEntryGet(clientHandle, OPEN_AUTO_VLAN_FIRST - 1, currentIndex, &autoVlanOui, &autoVlanDesc);
  printf("less than minimum value argument 5:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiEntryGet(clientHandle, OPEN_AUTO_VLAN_LAST + 1, currentIndex, &autoVlanOui, &autoVlanDesc);
  printf("greater than maximum value argument 5:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiEntryGet(clientHandle, autoVlanType, currentIndex, NULL, &autoVlanDesc);
  printf("NULL argument 5:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiEntryGet(clientHandle, autoVlanType, currentIndex, &badBufdescPointer, &autoVlanDesc);
  printf("NULL buffdesc pstart argument 5:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiEntryGet(clientHandle, autoVlanType, currentIndex, &zeroLenBuf, &autoVlanDesc);
  printf("buffdesc 0 length size argument 5:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiEntryGet(clientHandle, autoVlanType, currentIndex, &autoVlanOui, NULL);
  printf("NULL argument 5:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiEntryGet(clientHandle, autoVlanType, currentIndex, &autoVlanOui, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 5:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiEntryGet(clientHandle, autoVlanType, currentIndex, &autoVlanOui, &zeroLenBuf);
  printf("buffdesc 0 length size argument 5:(result = %d)\n", result);

  printf("Testing openapiQosAutoVlanOuiDescGet():\n");
  result = openapiQosAutoVlanOuiDescGet(NULL, autoVlanType, &autoVlanOui, &autoVlanDesc);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiDescGet(clientHandle, OPEN_AUTO_VLAN_FIRST - 1, &autoVlanOui, &autoVlanDesc);
  printf("less than minimum value argument 4:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiDescGet(clientHandle, OPEN_AUTO_VLAN_LAST + 1, &autoVlanOui, &autoVlanDesc);
  printf("greater than maximum value argument 4:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiDescGet(clientHandle, autoVlanType, NULL, &autoVlanDesc);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiDescGet(clientHandle, autoVlanType, &badBufdescPointer, &autoVlanDesc);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiDescGet(clientHandle, autoVlanType, &zeroLenBuf, &autoVlanDesc);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiDescGet(clientHandle, autoVlanType, &autoVlanOui, NULL);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiDescGet(clientHandle, autoVlanType, &autoVlanOui, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiQosAutoVlanOuiDescGet(clientHandle, autoVlanType, &autoVlanOui, &zeroLenBuf);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);

  printf("Testing openapiQosAutoVlanValidIntfFirstGet():\n");
  result = openapiQosAutoVlanValidIntfFirstGet(NULL, &ifNum);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosAutoVlanValidIntfFirstGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiQosAutoVlanValidIntfNextGet():\n");
  result = openapiQosAutoVlanValidIntfNextGet(NULL, prevIfNum, &nextIfNum);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosAutoVlanValidIntfNextGet(clientHandle, prevIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiQosAutoVlanPortStatusGet():\n");
  result = openapiQosAutoVlanPortStatusGet(NULL, OPEN_AUTO_VLAN_FIRST, ifNum, &status);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosAutoVlanPortStatusGet(clientHandle, OPEN_AUTO_VLAN_FIRST, ifNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiQosAutoVlanDetectedDevicesGet():\n");
  result = openapiQosAutoVlanDetectedDevicesGet(NULL, autoVlanType, &count);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosAutoVlanDetectedDevicesGet(clientHandle, OPEN_AUTO_VLAN_FIRST - 1, &count);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiQosAutoVlanDetectedDevicesGet(clientHandle, OPEN_AUTO_VLAN_LAST + 1, &count);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiQosAutoVlanDetectedDevicesGet(clientHandle, autoVlanType, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiQosAutoVlanLearnedEntryNextGet():\n");
  result = openapiQosAutoVlanLearnedEntryNextGet(NULL, index, &autoVlanMac,
                                                 &autoVlanType, &intfNum, &nextIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiQosAutoVlanLearnedEntryNextGet(clientHandle, index, NULL,
                                                 &autoVlanType, &intfNum, &nextIndex);
  printf("NULL argument 3(MAC Address):(result = %d)\n", result);
  result = openapiQosAutoVlanLearnedEntryNextGet(clientHandle, index, &autoVlanMac,
                                                 NULL, &intfNum, &nextIndex);
  printf("NULL argument 4(Auto VLAN Type):(result = %d)\n", result);
  result = openapiQosAutoVlanLearnedEntryNextGet(clientHandle, index, &autoVlanMac,
                                                 &autoVlanType, NULL, &nextIndex);
  printf("NULL argument 5(Interface Number):(result = %d)\n", result);
  result = openapiQosAutoVlanLearnedEntryNextGet(clientHandle, index, &autoVlanMac,
                                                 &autoVlanType, &intfNum, NULL);
  printf("NULL argument 6(Next Index):(result = %d)\n", result);

  return;
}


/*********************************************************************
* @purpose  Gets the Auto VLAN global status.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    autoVlanType       @b{(input)}  Auto VLAN type
* @param    mode       @b{(output)}  Mode value ENABLE/DISABLE

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanGlobalModeGet(openapiClientHandle_t *client_handle, OPEN_AUTO_VLAN_TYPES_t autoVlanType) 
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  if ((result = openapiQosAutoVlanGlobalModeGet(client_handle, autoVlanType, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the Auto VLAN global status. (result = %d)\n", result);
  }
  else
  {
    if (OPEN_ENABLE == mode)
    {
      printf("Auto VLAN is enabled in global mode for vlantype %d\n", autoVlanType);
    }
    else
    {
      printf("Auto VLAN is disabled in global mode for vlantype %d\n", autoVlanType);
    }
  }
  return;
}


/*********************************************************************
* @purpose  Enables or disables the Auto VLAN global status.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    autoVlanType       @b{(input)}  Auto VLAN type
* @param    mode       @b{(input)}  Mode value ENABLE/DISABLE

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanGlobalModeSet(openapiClientHandle_t *client_handle, OPEN_AUTO_VLAN_TYPES_t autoVlanType, 
                              OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiQosAutoVlanGlobalModeSet(client_handle, autoVlanType, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to enables or disables the Auto VLAN global status. (result = %d)\n", result);
  }
  else
  {
    printf("Global Auto VLAN mode is set to  %d (0 for disable, 1 for enable)for auto VLAN type %d\n", mode, autoVlanType);
  }
  return;
}


/*********************************************************************
* @purpose  Enables or disables the Auto VLAN status on interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    autoVlanType       @b{(input)}  Auto VLAN type
* @param    ifNum       @b{(input)}  Interface on which auto vlan mode needs to be enabled or disabled.
* @param    mode       @b{(input)}  Mode value ENABLE/DISABLE

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanIntfModeSet(openapiClientHandle_t *client_handle, OPEN_AUTO_VLAN_TYPES_t autoVlanType, 
                            uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiQosAutoVlanIntfModeSet(client_handle, autoVlanType, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to enables or disables the Auto VLAN status on interface. (result = %d)\n", result);
  }
  else
  {
    printf("Interface %d mode %d is set for auto VLAN type %d\n",ifNum, mode, autoVlanType);
  }
  return;
}


/*********************************************************************
* @purpose  Gets the status of Auto VLAN on interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum       @b{(input)}  Interface on which vlan mode needs to be checked.
* @param    mode       @b{(output)}  Mode value ENABLE/DISABLE

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanIntfModeGet(openapiClientHandle_t *client_handle, OPEN_AUTO_VLAN_TYPES_t autoVlanType, 
                            uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t        mode;

  if ((result = openapiQosAutoVlanIntfModeGet(client_handle, autoVlanType, ifNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the status of Auto VLAN on interface. (result = %d)\n", result);
  }
  else
  {
    if (1 == mode)
    {
      printf("Auto VLAN is enabled on interface %d", ifNum);
    }
    else
    {
      printf("Auto VLAN is disabled on interface %d", ifNum);
    }
  }
  return;
}


/*********************************************************************
* @purpose  Sets the VLAN for Auto VLAN traffic on switch.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    autoVlanType       @b{(input)}  Auto VLAN type
* @param    vlanId       @b{(input)}  VLAN Id to be configured for given Auto VLAN type.

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanIdSet(openapiClientHandle_t *client_handle, OPEN_AUTO_VLAN_TYPES_t autoVlanType, uint32_t vlanId)
{
  open_error_t result;

  if ((result = openapiQosAutoVlanIdSet(client_handle, autoVlanType, vlanId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the VLAN for Auto VLAN traffic on switch. (result = %d)\n", result);
  }
  else
  {
    printf("Vlan %d is configured for auto VLAN %d\n", vlanId, autoVlanType);
  }
  return;
}


/*********************************************************************
* @purpose  Gets the vlan Id configured for Auto VLAN type.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    autoVlanType       @b{(input)}  Auto VLAN type
* @param    vlanId       @b{(output)}  VLAN id configured for given Auto VLAN type.

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanIdGet(openapiClientHandle_t *client_handle, OPEN_AUTO_VLAN_TYPES_t autoVlanType)
{
  open_error_t result;
  uint32_t     vlanId;

  if ((result = openapiQosAutoVlanIdGet(client_handle, autoVlanType, &vlanId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the vlan Id configured for Auto VLAN type. (result = %d)\n", result);
  }
  else
  {
    printf("For auto VLAN type %d, vlan %d is configured\n",autoVlanType, vlanId);
  }
  return;
}


/*********************************************************************
* @purpose  Sets the Priority for Auto VLAN traffic on switch.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    autoVlanType       @b{(input)}  Auto VLAN type
* @param    autoVlanPriority       @b{(input)}  Auto VLAN Priority to set on the switch for given Auto VLAN type.

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanPrioritySet(openapiClientHandle_t *client_handle, OPEN_AUTO_VLAN_TYPES_t autoVlanType, 
                            uint32_t autoVlanPriority)
{
  open_error_t result;

  if ((result = openapiQosAutoVlanPrioritySet(client_handle, autoVlanType, autoVlanPriority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the Priority for Auto VLAN traffic on switch. (result = %d)\n", result);
  }
  else
  {
    printf("VLAN priority %d is set for auto VLAN type %d\n", autoVlanPriority, autoVlanType);
  }
  return;
}


/*********************************************************************
* @purpose  Gets the Priority for Auto VLAN traffic on switch.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    autoVlanType       @b{(input)}  Auto VLAN type
* @param    autoVlanPriority       @b{(output)}  Auto VLAN-Priority value configured on the switch for given Auto VLAN type.

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanPriorityGet(openapiClientHandle_t *client_handle, OPEN_AUTO_VLAN_TYPES_t autoVlanType)
{
  open_error_t result;
  uint32_t     autoVlanPriority;

  if ((result = openapiQosAutoVlanPriorityGet(client_handle, autoVlanType, &autoVlanPriority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the Priority for Auto VLAN traffic on switch. (result = %d)\n", result);
  }
  else
  {
    printf("Auto VLAN priority is %d for auto VLAN type %d\n", autoVlanPriority, autoVlanType);
  }
  return;
}


/*********************************************************************
* @purpose  Adds Auto VLAN OUI entry.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    autoVlanType       @b{(input)}  Auto VLAN type
* @param    autoVlanOui       @b{(input)}  Auto VLAN OUI Entry (string format. Example 0A020C)
* @param    autoOuiDesc       @b{(input)}  Auto Vlan OUI description

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanOuiAdd(openapiClientHandle_t *client_handle, OPEN_AUTO_VLAN_TYPES_t autoVlanType, 
                       char *autoVlanOui, char *autoOuiDesc)
{
  open_error_t result;
  open_buffdesc  ouibuf, ouidescbuf;

  ouibuf.pstart = autoVlanOui;
  ouibuf.size   = strlen(autoVlanOui)+1;

  ouidescbuf.pstart = autoOuiDesc;
  ouidescbuf.size   = strlen(autoOuiDesc);

  result = openapiQosAutoVlanOuiAdd(client_handle, autoVlanType, &ouibuf, &ouidescbuf);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to adds Auto VLAN OUI entry. (result = %d)\n", result);
  }
  else
  {
    printf("Oui Entry is added successfully.\n");
  }
  return;
}


/*********************************************************************
* @purpose  Removes Auto Vlan OUI entry.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    autoVlanType       @b{(input)}  Auto VLAN type
* @param    autoVlanOui       @b{(input)}  Auto VLAN OUI Entry (string format. Example 0A020C)

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanOuiRemove(openapiClientHandle_t *client_handle, OPEN_AUTO_VLAN_TYPES_t autoVlanType, 
                         char *autoVlanOui)
{
  open_error_t result;
  open_buffdesc  ouibuf;

  ouibuf.pstart = autoVlanOui;
  ouibuf.size   = strlen(autoVlanOui)+1;

  printf("\n ->%s<-  size %d\n",autoVlanOui,ouibuf.size);
  if ((result = openapiQosAutoVlanOuiRemove(client_handle, autoVlanType, &ouibuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to removes Auto Vlan OUI entry. (result = %d)\n", result);
  }
  else
  {
    printf("Oui entry %s removed\n",autoVlanOui);
  }
  return;
}


/*********************************************************************
* @purpose  Adds Auto VLAN OUI entry at particular Index.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    autoVlanType       @b{(input)}  Auto VLAN type
* @param    ouiIndex       @b{(input)}  Index number.
* @param    autoVlanOui       @b{(input)}  Auto VLAN OUI Entry (string format. Example 0A020C)
* @param    autoOuiDesc       @b{(input)}  Auto Vlan OUI description

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanIndexOuiAdd(openapiClientHandle_t *client_handle, OPEN_AUTO_VLAN_TYPES_t autoVlanType, 
                            uint32_t ouiIndex, char *autoVlanOui, char *autoOuiDesc)
{
  open_error_t result;
  open_buffdesc  ouibuf, ouidescbuf;

  ouibuf.pstart = autoVlanOui;
  ouibuf.size   = strlen(autoVlanOui)+1;

  ouidescbuf.pstart = autoOuiDesc;
  ouidescbuf.size   = strlen(autoOuiDesc);

  if ((result = openapiQosAutoVlanIndexOuiAdd(client_handle, autoVlanType, ouiIndex, &ouibuf, &ouidescbuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to adds Auto VLAN OUI entry at particular Index. (result = %d)\n", result);
  }
  else
  {
    printf("Added Oui Entry %s at index %d for auto VLAN type %d\n", autoVlanOui, ouiIndex, autoVlanType);
  }
  return;
}


/*********************************************************************
* @purpose  Removes Auto VLAN OUI entry at particular Index.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ouiIndex       @b{(input)}  Index number.

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanIndexOuiRemove(openapiClientHandle_t *client_handle, uint32_t ouiIndex)
{
  open_error_t result;

  if ((result = openapiQosAutoVlanIndexOuiRemove(client_handle, ouiIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to removes Auto VLAN OUI entry at particular Index. (result = %d)\n", result);
  }
  else
  {
    printf("Removed Oui entry at index %d\n",ouiIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Gets the count of OUIs already configured.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    autoVlanType       @b{(input)}  Auto VLAN type
* @param    autoVlanOuiCount       @b{(output)}  Count of OUIs configured.

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanOuiCountGet(openapiClientHandle_t *client_handle, OPEN_AUTO_VLAN_TYPES_t autoVlanType)
{
  open_error_t result;
  uint32_t     autoVlanOuiCount=0;

  if ((result = openapiQosAutoVlanOuiCountGet(client_handle, autoVlanType, &autoVlanOuiCount)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the count of OUIs already configured. (result = %d)\n", result);
  }
  else
  {
    printf("Total OUI Count is %d\n", autoVlanOuiCount);
  }
  return;
}


/*********************************************************************
* @purpose  Gets first Auto VLAN OUI index.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    autoVlanType       @b{(input)}  Auto VLAN type
* @param    firstIndex       @b{(output)}  First valid OUI Index.

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanOuiFirstEntryGet(openapiClientHandle_t *client_handle, OPEN_AUTO_VLAN_TYPES_t autoVlanType)
{
  open_error_t result;
  uint32_t     firstIndex;

  if ((result = openapiQosAutoVlanOuiFirstEntryGet(client_handle, autoVlanType, &firstIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets first Auto VLAN OUI index. (result = %d)\n", result);
  }
  else
  {
    printf("Oui Entry first index is %d\n",firstIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Gets Auto VLAN OUI next index.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    autoVlanType       @b{(input)}  Auto VLAN type
* @param    currentIndex       @b{(input)}  Current OUI Index.
* @param    nextOuiIndex       @b{(output)}  Next OUI index.

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanOuiEntryNextGet(openapiClientHandle_t *client_handle, OPEN_AUTO_VLAN_TYPES_t autoVlanType, 
                                uint32_t currentIndex)
{
  open_error_t result;
  uint32_t     nextOuiIndex;

  if ((result = openapiQosAutoVlanOuiEntryNextGet(client_handle, autoVlanType, currentIndex, &nextOuiIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets Auto VLAN OUI next index. (result = %d)\n", result);
  }
  else
  {
    printf("Next OUI Index is %d\n",nextOuiIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Gets Auto VLAN OUI entry details.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    autoVlanType       @b{(input)}  Auto VLAN type
* @param    currentIndex       @b{(input)}  Current OUI Index.
* @param    autoVlanOui       @b{(output)}  Auto VLAN OUI.
* @param    autoOuiDesc       @b{(output)}  Auto Vlan OUI description

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanOuiEntryGet(openapiClientHandle_t *client_handle, OPEN_AUTO_VLAN_TYPES_t autoVlanType, 
                            uint32_t currentIndex)
			     
{
  open_error_t result;
  open_buffdesc  ouibuf, ouidescbuf;
  unsigned char           oui[6]={0}, ouidesc[33]={0};

  ouibuf.pstart = oui;
  ouibuf.size   = 6;

  ouidescbuf.pstart = ouidesc;
  ouidescbuf.size   = sizeof(ouidesc);

  if ((result = openapiQosAutoVlanOuiEntryGet(client_handle, autoVlanType, currentIndex, &ouibuf, &ouidescbuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets Auto VLAN OUI entry details. (result = %d)\n", result);
  }
  else
  {
    printf("Auto VLAN OUI Entry is %02X:%02X:%02X description %s\n", oui[0],oui[1],oui[2],ouidesc);
  }
  return;
}


/*********************************************************************
* @purpose  Gets the Auto VLAN OUI description, given OUI entry string.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    autoVlanType       @b{(input)}  Auto VLAN type
* @param    autoVlanOui       @b{(input)}  Auto VLAN OUI
* @param    autoOuiDesc       @b{(output)}  Auto Vlan OUI description

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanOuiDescGet(openapiClientHandle_t *client_handle, OPEN_AUTO_VLAN_TYPES_t autoVlanType, 
                           char *autoVlanOui)
{
  open_error_t result;
  open_buffdesc  ouibuf, ouidescbuf;
  char    description[33]={0};


  ouibuf.pstart = autoVlanOui;
  ouibuf.size   = strlen(autoVlanOui) + 1;

  ouidescbuf.pstart = description;
  ouidescbuf.size   = sizeof(description);

  if ((result = openapiQosAutoVlanOuiDescGet(client_handle, autoVlanType, &ouibuf, &ouidescbuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the Auto VLAN OUI description, given OUI entry string. (result = %d)\n", result);
  }
  else
  {
    printf("Oui Desctipion is %s for OUI %s\n",description,autoVlanOui);
  }
  return;
}


/*********************************************************************
* @purpose  Determine if the interface is valid for Auto VLAN configuration.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum       @b{(input)}  Interface number.

*
* @returns  none
*
* @end
*********************************************************************/
void qoSAutoVlanIsValidIntf(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_BOOL_t    isValid = OPEN_FALSE;

  if ((result = openapiQoSAutoVlanIsValidIntf(client_handle, ifNum, &isValid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to check Auto VLAN interface validity (result = %d)\n", result);
  }
  else
  {
    if (1 == isValid)
    {
      printf("Given interface %d is valid for Auto VLAN ", ifNum);
    }
    else
    {
      printf("Given interface %d is not valid for Auto VLAN ", ifNum);
    }
  }
  return;
}


/*********************************************************************
* @purpose  Get the first interface that is valid for Auto VLAN.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum       @b{(output)}  Interface number.

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanValidIntfFirstGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t     ifNum;

  if ((result = openapiQosAutoVlanValidIntfFirstGet(client_handle, &ifNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the first interface that is valid for Auto VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("First interface valid for auto VLAN is %d\n", ifNum);
  }
  return;
}


/*********************************************************************
* @purpose  Get the next interface that is valid for auto VLAN.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    prevIfNum       @b{(input)}  Previous interface number.
* @param    nextIfNum       @b{(output)}  Next interface number.

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanValidIntfNextGet(openapiClientHandle_t *client_handle, uint32_t prevIfNum)
{
  open_error_t result;
  uint32_t     nextIfNum;

  if ((result = openapiQosAutoVlanValidIntfNextGet(client_handle, prevIfNum, &nextIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next interface that is valid for auto VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("Present interface %d , next interface is %d\n", prevIfNum, nextIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets Auto VLAN operational port status.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum       @b{(input)}  Interface number.
* @param    status       @b{(output)}  Interface Operational status Enable/Disable.

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanPortStatusGet(openapiClientHandle_t *client_handle, OPEN_AUTO_VLAN_TYPES_t autoVlanType, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t     status;

  if ((result = openapiQosAutoVlanPortStatusGet(client_handle, autoVlanType, ifNum, &status)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets Auto VLAN operational port status. (result = %d)\n", result);
  }
  else
  {
    printf("Interface %d status(1 enable/0 disable) is %d\n", ifNum, status);
  }
  return;
}


/*********************************************************************
* @purpose  Gets the count of devices learned for given Auto VLAN type.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    autoVlanType       @b{(input)}  Auto VLAN type
* @param    count       @b{(output)}  Number of detected devices.

*
* @returns  none
*
* @end
*********************************************************************/
void qosAutoVlanDetectedDevicesGet(openapiClientHandle_t *client_handle, OPEN_AUTO_VLAN_TYPES_t autoVlanType)
{
  open_error_t result;
  uint32_t     count;

  if ((result = openapiQosAutoVlanDetectedDevicesGet(client_handle, autoVlanType, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the count of devices learned for given Auto VLAN type. (result = %d)\n", result);
  }
  else
  {
    printf("Total devices %d detected for auto VLAN type %d\n", count, autoVlanType);
  }
  return;
}

/*********************************************************************
 * @purpose  Gets the Auto VLAN learned entries.
 *
 * @param  clientHandle @b{(input)} client handle from registration API
 *
 * @returns  none
 *
 * @end
 *********************************************************************/
void qosAutoVlanLearnedEntriesGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t     intfNum = 0, index = 0, nextIndex = 0;
  OPEN_AUTO_VLAN_TYPES_t autoVlanType = 0;
  open_buffdesc autoVlanMac;
  unsigned char buf[18];
  char mac[18];
  char vlanType[10];
  open_buffdesc intfName;
  char intfNameStr[256];

  memset(buf, 0, sizeof(buf));
  memset(mac, 0, sizeof(mac));
  memset(vlanType, 0, sizeof(vlanType));
  memset(intfNameStr, 0, sizeof(intfNameStr));

  autoVlanMac.pstart = buf;
  autoVlanMac.size = sizeof(buf);

  intfName.pstart = intfNameStr;
  intfName.size = sizeof(intfNameStr);

  result = openapiQosAutoVlanLearnedEntryNextGet(clientHandle, index, &autoVlanMac,
                                                 &autoVlanType, &intfNum, &nextIndex);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get the Auto VLAN learned entry "
           "(result = %d).\n", result);
    return;
  }

  printf("\r\nMAC Address        Auto VLAN type Interface");
  printf("\r\n-----------------  -------------- ---------\r\n");

  while (OPEN_E_NONE == result)
  {
    memset(vlanType, 0, sizeof(vlanType));
    memset(intfNameStr, 0, sizeof(intfNameStr));
    memset(mac, 0, sizeof(mac));

    snprintf(mac, sizeof(mac), "%02X:%02X:%02X:%02X:%02X:%02X",
             buf[0], buf[1], buf[2], buf[3], buf[4], buf[5]);

    if (OPEN_AUTO_VLAN_CAMERA == autoVlanType)
    {
      snprintf(vlanType, sizeof(vlanType), "%s", "Camera");
    }
    else if (OPEN_AUTO_VLAN_WIFI == autoVlanType)
    {
      snprintf(vlanType, sizeof(vlanType), "%s", "Wifi");
    }

    if ((OPEN_E_NONE != openapiIfNameGet(clientHandle, intfNum, &intfName)))
    {
      printf("Bad return code trying to get interface name for %d interface. (result = %d)\n",
             intfNum, result);
      snprintf(intfNameStr, sizeof(intfNameStr), "%s", "Unknown");
    }

    printf("%-19s %-15s %-9s\r\n", mac, vlanType, intfNameStr);

    index = nextIndex;

    autoVlanType = 0;
    nextIndex = 0;
    intfNum = 0;
    memset(buf, 0, sizeof(buf));
    result = openapiQosAutoVlanLearnedEntryNextGet(clientHandle, index, &autoVlanMac,
                                                   &autoVlanType, &intfNum, &nextIndex);
  }

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for QoS Auto VLAN
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum, autoVlanType, autoVlanMode, autoVlanPriority;
  uint32_t ifNum, ifMode;
  uint32_t vlanId;
  uint32_t  ouiIndex;
  char     autoVlanOui[32]={0};
  char     autoVlanDesc[50]={0};

  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("QoS Auto VLAN example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting QoS Auto VLAN API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
	autoVlanType = atoi(argv[2]);
        qosAutoVlanGlobalModeGet(&client_handle, autoVlanType);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 4)
      {
	autoVlanType = atoi(argv[2]);
	autoVlanMode = atoi(argv[3]);
        qosAutoVlanGlobalModeSet(&client_handle, autoVlanType, autoVlanMode);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 5)
      {
	autoVlanType = atoi(argv[2]);
	ifNum = atoi(argv[3]);
	ifMode = atoi(argv[4]);
        qosAutoVlanIntfModeSet(&client_handle, autoVlanType, ifNum, ifMode);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 4)
      {
	autoVlanType = atoi(argv[2]);
	ifNum = atoi(argv[3]);
        qosAutoVlanIntfModeGet(&client_handle, autoVlanType, ifNum);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 4)
      {
	autoVlanType = atoi(argv[2]);
	vlanId = atoi(argv[3]);
        qosAutoVlanIdSet(&client_handle, autoVlanType, vlanId);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 3)
      {
	autoVlanType = atoi(argv[2]);
        qosAutoVlanIdGet(&client_handle, autoVlanType);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 4)
      {
	autoVlanType = atoi(argv[2]);
	autoVlanPriority = atoi(argv[3]);
        qosAutoVlanPrioritySet(&client_handle, autoVlanType, autoVlanPriority);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 3)
      {
	autoVlanType = atoi(argv[2]);
        qosAutoVlanPriorityGet(&client_handle, autoVlanType);
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 5)
      {
	autoVlanType = atoi(argv[2]);
	strncpy(autoVlanOui, argv[3], 32);
	strncpy(autoVlanDesc, argv[4], sizeof(autoVlanDesc));
        qosAutoVlanOuiAdd(&client_handle, autoVlanType, autoVlanOui, autoVlanDesc);
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 4)
      {
	autoVlanType = atoi(argv[2]);
	strncpy(autoVlanOui, argv[3], 32);
        qosAutoVlanOuiRemove(&client_handle, autoVlanType, autoVlanOui);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 6)
      {
	autoVlanType = atoi(argv[2]);
	ouiIndex     = atoi(argv[3]);
	strncpy(autoVlanOui, argv[4], 32);
	strncpy(autoVlanDesc, argv[5], sizeof(autoVlanDesc));
        qosAutoVlanIndexOuiAdd(&client_handle, autoVlanType, ouiIndex, autoVlanOui, autoVlanDesc);
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 3)
      {
	printf("\n Reached case 12\n");
	ouiIndex     = atoi(argv[2]);
	printf("\n ouiIndex %d\n",ouiIndex);
        qosAutoVlanIndexOuiRemove(&client_handle, ouiIndex);
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 3)
      {
	autoVlanType = atoi(argv[2]);
        qosAutoVlanOuiCountGet(&client_handle, autoVlanType);
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 3)
      {
	autoVlanType = atoi(argv[2]);
        qosAutoVlanOuiFirstEntryGet(&client_handle, autoVlanType);
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 4)
      {
	autoVlanType = atoi(argv[2]);
	ouiIndex     = atoi(argv[3]);
        qosAutoVlanOuiEntryNextGet(&client_handle, autoVlanType, ouiIndex);
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 4)
      {
	autoVlanType = atoi(argv[2]);
	ouiIndex     = atoi(argv[3]);
        qosAutoVlanOuiEntryGet(&client_handle, autoVlanType, ouiIndex);
        show_help = 0;
      }
      break;
    case 17:
      if (argc == 4)
      {
	autoVlanType = atoi(argv[2]);
	strncpy(autoVlanOui, argv[3], 32);
        qosAutoVlanOuiDescGet(&client_handle, autoVlanType, autoVlanOui);
        show_help = 0;
      }
      break;
    case 18:
      if (argc == 3)
      {
	ifNum = atoi(argv[2]);
        qoSAutoVlanIsValidIntf(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 19:
      if (argc == 2)
      {
        qosAutoVlanValidIntfFirstGet(&client_handle);
        show_help = 0;
      }
      break;
    case 20:
      if (argc == 3)
      {
	ifNum = atoi(argv[2]);
        qosAutoVlanValidIntfNextGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 21:
      if (argc == 4)
      {
      	autoVlanType = atoi(argv[2]);
	      ifNum = atoi(argv[3]);
        qosAutoVlanPortStatusGet(&client_handle, autoVlanType, ifNum);
        show_help = 0;
      }
      break;
    case 22:
      if (argc == 3)
      {
	autoVlanType = atoi(argv[2]);
        qosAutoVlanDetectedDevicesGet(&client_handle, autoVlanType);
        show_help = 0;
      }
      break;
    case 23:
      if (argc == 2)
      {
        runSanity(&client_handle);
        show_help = 0;
      }
      break;
    case 24:
      if (argc == 2)
      {
        qosAutoVlanLearnedEntriesGet(&client_handle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping QoS Auto VLAN API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}


